#
# Cookbook Name:: boa
# File:: providers/logging.rb
#
# Copyright 2016 Broadcom.
#
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

require "OpEN"
require "OpENUtil"

def whyrun_supported?
  true
end

action :create do
  if !@current_resource.exists
    converge_by("create syslog server '#{@new_resource.host}'") do
      create_syslog_server
    end
  else
    converge_by("edit syslog server '#{@new_resource.host}'") do
      edit_syslog_server
    end
  end
end

action :delete do
  if @current_resource.exists
    converge_by("remove syslog server #{@current_resource.host}") do
      delete_syslog_server
    end
  else
    Chef::Log.info("syslog server '#{current_resource.host}' doesn't exist")
  end
end

action :enable do
  converge_by("enable logging to syslog server") do
    enable_syslog
  end
end

def enable_syslog
  open = OpENUtil.new()
  mode = OpEN::OPEN_ENABLE
  conn_ret = open.connect("boa-logging")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    ret = OpEN::openapiSysLogModeSet(client,mode)
    if ret != OpEN::OPEN_E_NONE
      Chef::Log.info("Failed to set syslog mode.")
    end
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'enable_syslog'.")
  end
  open.terminate()
end

def load_current_resource
  Chef::Log.info "Loading current resource #{@new_resource.host}"
  @current_resource = Chef::Resource::BoaLogging.new(@new_resource.host)
  @current_resource.host(@new_resource.host)
  @current_resource.port(@new_resource.port)
  @current_resource.severity(@new_resource.severity)
  @current_resource.exists = false

  if resource_exists?
    @current_resource.host(@new_resource.host)
    @current_resource.exists = true
  else
    Chef::Log.info "syslog server '#{@new_resource.host}' (port: #{@new_resource.port}) doesn't exist"
  end
end

def syslog_server_exists(host)
  open = OpENUtil.new()
  conn_ret = open.connect("boa-logging")
  exists = false
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    name_len_p = OpEN.new_uint32_tp()
    OpEN::openapiLoggingHostMaxAddrLenGet(client,name_len_p)
    name_len = OpEN.uint32_tp_value(name_len_p)
    OpEN.delete_uint32_tp(name_len_p)

    # Get host 
    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = name_len
    this_host = open.getCharBuffer(name_len, "")
    buffdesc.pstart = this_host
    ret = OpEN::openapiLoggingHostIpAddrNextGet(client,buffdesc)
    while ret == OpEN::OPEN_E_NONE
      this_host_name = this_host.cast()
      if this_host_name == host
        exists = true
        break
      end
      ret = OpEN::openapiLoggingHostIpAddrNextGet(client,buffdesc)
    end
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'syslog_server_exists'.")
  end
  open.terminate()
  return exists
end

def resource_exists?
  Chef::Log.info("Looking to see if syslog server #{@new_resource.host} (port: #{@new_resource.port}) exists")
  return syslog_server_exists(@new_resource.host)
end

def logging_host_port_set(host,portnum)
  open = OpENUtil.new()
  conn_ret = open.connect("boa-logging")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = host.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size,host)

    ret = OpEN::openapiLoggingHostPortSet(client,buffdesc,portnum)
    if ret != OpEN::OPEN_E_NONE
      Chef::Log.info("Failed to set Port '#{portnum} for host '#{host}'")
    end
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'logging_host_port_set'.")
  end
  open.terminate()
end

def logging_host_severity_set(host,severity)
  open = OpENUtil.new()
  conn_ret = open.connect("boa-logging")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = host.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size,host)

    case severity
      when "emergency"
        severity = OpEN::OPEN_LOG_SEVERITY_EMERGENCY
      when "alert"
        severity = OpEN::OPEN_LOG_SEVERITY_ALERT
      when "critical"
        severity = OpEN::OPEN_LOG_SEVERITY_CRITICAL
      when "error"
        severity = OpEN::OPEN_LOG_SEVERITY_ERROR
      when "warning"
        severity = OpEN::OPEN_LOG_SEVERITY_WARNING
      when "notice"
        severity = OpEN::OPEN_LOG_SEVERITY_NOTICE
      when "info"
        severity = OpEN::OPEN_LOG_SEVERITY_INFO
      when "debug"
        severity = OpEN::OPEN_LOG_SEVERITY_DEBUG
      else 
        severity = OpEN::OPEN_LOG_SEVERITY_INFO
    end
    ret = OpEN::openapiLoggingHostSeveritySet(client,buffdesc,severity)
    if ret != OpEN::OPEN_E_NONE
      Chef::Log.info("Failed to set Severity level '#{severity} for the host '#{host}")
    end
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'logging_host_severity_set'.")
  end
  open.terminate()
end

def create_syslog_server
  open = OpENUtil.new()
  conn_ret = open.connect("boa-logging")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = new_resource.host.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size,new_resource.host)
    ret = OpEN::openapiLoggingHostAdd(client,buffdesc)

    if ret == OpEN::OPEN_E_NONE
      logging_host_port_set(new_resource.host,new_resource.port)
      logging_host_severity_set(new_resource.host,new_resource.severity)
    else
     Chef::Log.info("Failed to create syslog server '#{new_resource.host}'.")
    end
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'create_syslog_server'.")
  end
  open.terminate()
end

def delete_syslog_server
  open = OpENUtil.new()
  conn_ret = open.connect("boa-logging")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = new_resource.host.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size,new_resource.host)
    ret = OpEN::openapiLoggingHostRemove(client,buffdesc)
    if ret != OpEN::OPEN_E_NONE
      Chef::Log.info("Failed delete syslog server '#{new_resource.host}'.")
    end
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'delete_syslog_server'.")
  end
  open.terminate()
end

def edit_syslog_server
  open = OpENUtil.new()
  conn_ret = open.connect("boa-logging")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    logging_host_port_set(new_resource.host,new_resource.port)
    logging_host_severity_set(new_resource.host,new_resource.severity)
  else
    Chef::Log.info("Failed to connect to 'boa-logging' in 'edit_syslog_server'.")
  end
  open.terminate()
end

