#
# Cookbook Name:: boa
# File:: providers/vlan.rb
#
# Copyright 2016 Broadcom.
#
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

require "OpEN"
require "OpENUtil"

def whyrun_supported?
  true
end

action :create do
  if !@current_resource.exists
    converge_by("create VLAN '#{@new_resource.vlan_id}'") do
      create_vlan
    end
  else
    converge_by("edit VLAN '#{@new_resource.vlan_id}'") do
      edit_vlan
    end
  end
end

action :add_participation do
  converge_by("add ports '#{@new_resource.participate.join(', ')}' to VLAN '#{@new_resource.vlan_id}'") do
    if @current_resource.exists
      participation_vlan
    end
  end
end

action :delete do
  converge_by("remove VLAN '#{@current_resource.vlan_id}'") do
    if @current_resource.exists
      delete_vlan
    else
      Chef::Log.info "Failed to delete VLAN '#{@new_resource.vlan_id}', doesn't exist"
    end
  end
end

def load_current_resource
  Chef::Log.info "Loading current resource #{@new_resource.name}"
  @current_resource = Chef::Resource::BoaVlans.new(@new_resource.name)
  @current_resource.name(@new_resource.name)
  @current_resource.vlan_id(@new_resource.vlan_id)
  @current_resource.exists = false

  if resource_exists?
    @current_resource.vlan_id(@new_resource.vlan_id)
    @current_resource.exists = true
  else
    Chef::Log.info "VLAN '#{@new_resource.vlan_id}' doesn't exist"
  end
end

def participation_vlan
  open = OpENUtil.new()
  conn_ret = open.connect("boa-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    intIfNum = 0
    intIfNum_p = OpEN.new_uint32_tp()
    include = OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE
    exclude = OpEN::OPEN_VLAN_PARTICIPATION_MODE_EXCLUDE

    buffdesc = OpEN::Open_buffdesc.new

    ports = @new_resource.participate
    ports.each { |intf|

      buffdesc.size   = intf.length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size,intf)

      #get internal interface number for given slot/port
      ret, intIfNum = OpEN::openapiIfNumGet(client,buffdesc,intIfNum_p)
      intIfNum = OpEN.uint32_tp_value(intIfNum_p)

      if ret == OpEN::OPEN_E_NONE
        #include for given vlan id
        ret = OpEN::openapiVlanIfParticipationSet(client,new_resource.vlan_id,intIfNum,include)
        if ret != OpEN::OPEN_E_NONE
          Chef::Log.info "Failed to add port '#{intf}' to VLAN '#{new_resource.vlan_id}', returned #{ret}" 
        end
      else
        Chef::Log.info "Failed to get internal interface number for the port '#{intf}', returned #{ret}" 
      end
    }
    OpEN.delete_uint32_tp(intIfNum_p)
  else
    Chef::Log.info "Failed to connect to 'boa-vlan' in 'participation_vlan'." 
  end
  open.terminate()
end

def vlan_exists(vid)
  open = OpENUtil.new()
  conn_ret = open.connect("boa-vlan")
  exists = false
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    ret = OpEN::openapiVlanCreatedCheck(client,vid)
    if ret == OpEN::OPEN_E_NONE
      exists = true
    else
      exists = false
    end
  else
    Chef::Log.info "Failed to connect to 'boa-vlan' in 'vlan_exists'."
  end
  open.terminate()
  return exists
end

def resource_exists?
  Chef::Log.info("Looking to see if VLAN '#{@new_resource.vlan_id}' exists")
  return vlan_exists(@new_resource.vlan_id)
end

def create_vlan
  open = OpENUtil.new()
  conn_ret = open.connect("boa-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    ret = OpEN::openapiVlanCreate(client,new_resource.vlan_id)
    if ret == OpEN::OPEN_E_NONE
      count = 0
      ret = OpEN::openapiVlanCreatedCheck(client,new_resource.vlan_id)
      while ret != OpEN::OPEN_E_NONE && count < 10
        count += 1
        sleep 0.5
        ret = OpEN::openapiVlanCreatedCheck(client,new_resource.vlan_id)
      end
      #assign name to created VLAN
      buffdesc = OpEN::Open_buffdesc.new
      buffdesc.size   = new_resource.name.length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size,new_resource.name)
      ret = OpEN::openapiVlanNameSet(client,new_resource.vlan_id,buffdesc)
      if ret != OpEN::OPEN_E_NONE
        Chef::Log.info "Failed to set name '#{new_resource.vlan_id}' for VLAN '#{new_resource.vlan_id}', returned #{ret}" 
      end
    else
      Chef::Log.info "Failed to create VLAN '#{new_resource.vlan_id}', returned #{ret}" 
    end
  else
    Chef::Log.info "Failed to connect to 'boa-vlan' in 'create_vlan'." 
  end
  open.terminate()
end

def delete_vlan
  open = OpENUtil.new()
  conn_ret = open.connect("boa-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    ret = OpEN::openapiVlanDelete(client,new_resource.vlan_id)
    if ret != OpEN::OPEN_E_NONE
      Chef::Log.info "Failed to delete VLAN '#{new_resource.vlan_id}', returned #{ret}" 
    end
  else
    Chef::Log.info "Failed to connect to 'boa-vlan' in 'delete_vlan'."
  end
  open.terminate()
end

def edit_vlan
  open = OpENUtil.new()
  conn_ret = open.connect("boa-vlan")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    buffdesc = OpEN::Open_buffdesc.new
    buffdesc.size   = new_resource.name.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size,new_resource.name)
    ret = OpEN::openapiVlanNameSet(client,new_resource.vlan_id,buffdesc)
    if ret != OpEN::OPEN_E_NONE
      Chef::Log.info "Failed to set name '#{new_resource.name}' for VLAN '#{new_resource.vlan_id}', returned #{ret}" 
    end
  else
    Chef::Log.info "Failed to connect to 'boa-vlan' in 'edit_vlan'."
  end
  open.terminate()
end

