
/*********************************************************************
*
*  Copyright 2012-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  cos_example.c
*
* @purpose   class of service Example.
*
* @component OPEN
*
* @comments
*
* @create    09/05/2017
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_cos.h"

#define PRINTBADRESULT(result, msg) \
if (result==OPEN_E_UNAVAIL) { printf("Feature not supported - %s (err %d).\n", msg, result); } \
else if (result!=OPEN_E_NONE) { printf("Test Failure - %s (err %d).\n", msg, result); }

#define COS_WRED_DROP_PREC        0     /* Drop precedence 0 */
#define COS_WRED_MIN_THRESHOLD    22    /* WRED minimum threshold */
#define COS_WRED_MAX_THRESHOLD    55    /* WRED maximum threshold */
#define COS_WRED_ECN_ENABLE       1     /* Enable ECN */
#define COS_WRED_ENABLE           1     /* Enable WRED */
#define COS_WRED_DISABLE          0     /* Disable WRED */

/*********************************************************************
* @purpose  Set WRED drop parameters.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testWredDropParamsSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "openapiCosQueueWredDropParamsSet");
  }

  return (result);
}

/*********************************************************************
* @purpose  Reset WRED drop parameters.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testWredDropParamsReset(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsReset(clientHandle, OPEN_COS_ALL_INTERFACES, pri);

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "openapiCosQueueWredDropParamsReset");
  }

  return (result);
}

/*********************************************************************
* @purpose  Enable and Disable Cos Queue WRED configuration.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testCosQueueWredSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE == result)
  {
    printf("Enable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_ENABLE);
  
    if (OPEN_E_NONE != result)
    {
      PRINTBADRESULT(result, "testCosQueueWredSet");
      return (result);
    }
  }

  if (OPEN_E_NONE == result)
  {
    printf("Disable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DISABLE);
  
    if (OPEN_E_NONE != result)
    {
      PRINTBADRESULT(result, "testCosQueueWredSet");
      return (result);
    }
  }
  return (result);
}

/*********************************************************************
* @purpose  Test COS queue WRED stats get.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testCosQueueWredStatsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result              = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;
  uint64_t     wredDropCount       = 0;
  uint64_t     ecnMarked           = 0;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE == result)
  {
    printf("Enable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_ENABLE);
  
    if (OPEN_E_NONE == result)
    {
      printf("Get WRED stats from all interfaces \n");
      result = openapiCosQueueWredStatsGet(clientHandle, OPEN_COS_ALL_INTERFACES, &wredDropCount, &ecnMarked);
    }
  }

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "testCosQueueWredStatsGet");
  }

  return (result);
}

/*********************************************************************
* @purpose  Test COS queue WRED stats clear.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* 
* @returns  OPEN_E_NONE         If successful (If lag string is found)
*           OPEN_E_ERROR        On error.
* 
* @notes    none
* 
* @end
*********************************************************************/
open_error_t testCosQueueWredStatsClear(openapiClientHandle_t *clientHandle)
{
  open_error_t result              = OPEN_E_NONE;
  uint8_t      wredMinThreshold    = COS_WRED_MIN_THRESHOLD;
  uint8_t      wredMaxThreshold    = COS_WRED_MAX_THRESHOLD;
  uint8_t      wredDropProbability = 100;
  uint8_t      pri                 = 1;

  result = openapiCosQueueWredDropParamsSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_DROP_PREC,
                                            COS_WRED_ECN_ENABLE, wredMinThreshold, wredMaxThreshold, wredDropProbability);

  if (OPEN_E_NONE == result)
  {
    printf("Enable WRED on all interfaces \n");
    result = openapiCosQueueWredSet(clientHandle, OPEN_COS_ALL_INTERFACES, pri, COS_WRED_ENABLE);
  
    if (OPEN_E_NONE == result)
    {
      printf("Clear WRED stats from all interfaces \n");
      result = openapiCosQueueWredStatsClear(clientHandle, OPEN_COS_ALL_INTERFACES);
    }
  }

  if (OPEN_E_NONE != result)
  {
    PRINTBADRESULT(result, "testCosQueueWredStatsClear");
  }

  return (result);
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         COS OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  uint32_t  maxCosQueueId = 0;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("cos_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting COS API example application");

  printf ("\n");

  /* Execute sanity tests */
  printf ("Begin Sanity tests...\n");

  printf ("Verify WRED drop params set ...\n");
  result = testWredDropParamsSet(&clientHandle);
  printf ("\nComplete.\n");

  printf ("Verify WRED drop params reset ...\n");
  result = testWredDropParamsReset(&clientHandle);
  printf ("\nComplete.\n");

  printf ("Verify COS queue WRED set ...\n");
  result = testCosQueueWredSet(&clientHandle);
  printf ("\nComplete.\n");

  printf ("Verify COS queue WRED stats get ...\n");
  result = testCosQueueWredStatsGet(&clientHandle);
  printf ("\nComplete.\n");

  printf ("Verify COS queue WRED stats clear ...\n");
  result = testCosQueueWredStatsClear(&clientHandle);
  printf ("\nComplete.\n");

  printf ("Get the max COS queue ID value ...\n");
  if (OPEN_E_NONE == openapiCosQueueMaxIdGet(&clientHandle, &maxCosQueueId))
  {
    printf ("Max COS queue ID value is %u ...\n", maxCosQueueId);
  }
  else
  {
    printf ("Unable to get the max cos queue ID value ...\n");
  }
  printf ("\nComplete.\n");

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping COS API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
