/*********************************************************************
*
*  Copyright 2018-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dhcp_snooping_example.c
*
* @purpose   DHCP snooping OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    7/20/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dhcp_snooping.h"

#define DS_L2RELAY_STRING_SIZE 256
#define MAX_VLAN_ID 4094
/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DHCP snooping example application menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("Usage:  %s <test#> <arg1> <arg2> ... \n", name);
  printf("Note: in following, family is either 1 for AF_INET, or 2 for AF_INET6\n");
  printf("Test 1: Get whether DHCP snooping is enabled not: %s 1 family\n", name);
  printf("Test 2: Get whether DHCP snooping verifies source MAC addresses: %s 2 family\n", name);
  printf("Test 3: Display VLANs enabled for DHCP snooping: %s 3 family\n", name);
  printf("Test 4: Get whether DHCP snooping is enabled on a port: %s 4 family ifnum\n", name);
  printf("Test 5: Get whether DHCP snooping considers an interface trusted: %s 5 family ifnum\n", name);
  printf("Test 6: Get whether DHCP snooping logs invalid packets on an interface: %s 6 family ifnum\n", name);
  printf("Test 7: Get the remote DHCP snooping database file name: %s 7 family.\n", name);
  printf("Test 8: Get the remote DHCP snooping database IP Address: %s 8 family.\n", name);
  printf("Test 9: Get the DHCP snooping database write delay: %s 9 family.\n", name);
  printf("Test 10: Test DHCP snooping OpEN API Sanity: %s 10\n", name);
  printf("Test 11: Get DHCP L2 relay global mode: %s 11\n", name);
  printf("Test 12: Set DHCP L2 relay global mode enable (1) diable (0): %s 12 <1 | 0>\n", name);
  printf("Test 13: Get DHCP L2 relay interface mode: %s 13 <ifnum>\n", name);
  printf("Test 14: Set DHCP L2 relay interface mode enable (1) diable (0): %s 14 <ifnum> <1 | 0>\n", name);
  printf("Test 15: Get DHCP L2 relay interface trust mode: %s 15 <ifnum>\n", name);
  printf("Test 16: Set DHCP L2 relay interface trust mode true (1) false (0): %s 16 <ifnum> <1 | 0>\n", name);
  printf("Test 17: Get DHCP L2 relay mode for VLAN: %s 17 <vlan ID>\n", name);
  printf("Test 18: Set DHCP L2 relay mode for VLAN enable (1) disable (0): %s 18 <vlan ID> <1 | 0>\n", name);
  printf("Test 19: Set DHCP L2 relay mode for range of VLANs enable (1) disable (0): %s 19 <start vlan ID> <end vlan ID> <1 | 0>\n", name);
  printf("Test 20: Get DHCP L2 relay circuit ID mode for VLAN: %s 20 <vlan ID>\n", name);
  printf("Test 21: Set DHCP L2 relay circuit ID mode for VLAN enable (1) disable (0): %s 21 <vlan ID> <1 | 0>\n", name);
  printf("Test 22: Set DHCP L2 relay circuit ID mode for range of VLANs enable (1) disable (0): %s 22 <start vlan ID> <end vlan ID> <1 | 0>\n", name);
  printf("Test 23: Get DHCP L2 relay remote ID for VLAN: %s 23 <vlan ID>\n", name);
  printf("Test 24: Set DHCP L2 relay remote ID for VLAN: %s 24 <vlan ID> <remote ID>\n", name);
  printf("Test 25: Set DHCP L2 relay remote ID for range of VLANs: %s 25 <start vlan ID> <end vlan ID> <remote ID>\n", name);
  printf("Test 26: Get DHCP L2 relay mode for a given subscription on an interface: %s 26 <ifNum> <subscription name>\n", name);
  printf("Test 27: Set DHCP L2 relay mode for a given subscription on an interface."
         "enable (1) disable (0): %s 27 <ifNum> <subscription name> <1 | 0>\n", name);
  printf("Test 28: Get DHCP L2 relay circuit ID mode for a given subscription"
         " on an interface: %s 28 <ifNum> <subscription name>\n", name);
  printf("Test 29: Set DHCP L2 relay circuit ID mode for a given subscription"
         " on an interface. enable (1) disable (0):"
         " %s 29 <ifNum> <subscription name> <1 | 0>\n", name);
  printf("Test 30: Get DHCP L2 relay remote ID for a given subscription"
         " on an interface: %s 30 <ifNum> <subscription name>\n", name);
  printf("Test 31: Set DHCP L2 relay remote ID for a given subscription"
         " on an interface. :%s 31 <ifNum> <subscription name> <remote ID>\n", name);
  printf("Test 32: Get DHCP L2 relay interface statistics.: %s 32 <ifnum>\n", name);
  printf("Test 33: Clear the DHCP L2 relay interface statistics.: %s 33 <ifnum>\n", name);
  printf("Test 34: Get the configured no-option-82 action on L2 relay trusted"
         " interface.: Drop (0), keep(1) and update(2).%s 34 <ifnum>\n", name);
  printf("Test 35: Set the configured no-option-82 action on L2 relay trusted"
         " interface.: Drop (0), keep(1) and update(2).%s 35 <ifnum> <action>\n",
         name);
  printf("Test 36: Test DHCP L2 relay OpEN API Sanity: %s 36\n", name);
}

/*********************************************************************
* @purpose  Convert an IP address in a open_inet_addr_t structure to
*           a string for display
*
* @param    addr  @b{(input)}  address to be converted
* @param    buffer @b{(output)} storage to contain
*
* @returns  pointer to buffer if conversion successful
*           NULL if error
*
* @notes
*
* @end
*********************************************************************/
static const char *ipAddressFormat(open_inet_addr_t *addr, char *buffer)
{
  uint32_t  ip4_addr;
  uint8_t   addr8[INET6_ADDRSTRLEN];
  int af;

  switch (addr->family)
  {
  case OPEN_AF_INET:
    af = AF_INET;
    ip4_addr = addr->addr.ipv4;
    memcpy(addr8, &ip4_addr, sizeof(ip4_addr));
    break;
  case OPEN_AF_INET6:
    af = AF_INET6;
    memcpy(addr8, addr->addr.ipv6.u.addr8, sizeof(addr8));
    break;
  default:
    return(NULL);
  }

  return(inet_ntop(af, (void *)addr8, buffer, sizeof(addr8)));
}

/*********************************************************************
* @purpose  Get if DHCP snooping is enabled or not.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpSnoopingAdminMode(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDhcpSnoopingAdminModeGet(clientHandle, af, &mode);
  if (result != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get dhcp snooping mode. (result = %d)\n", result);
  } 
  else
  {
    if (mode == OPEN_ENABLE) 
    {
      printf("DHCP snooping is enabled\n");
    }
    else
    {
      printf("DHCP snooping is disabled\n");
    }
  }
}

/*********************************************************************
* @purpose  Get whether DHCP snooping verifies source MAC addresses.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpSnoopingVerifySourceMac(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  OPEN_CONTROL_t verify;

  result = openapiDhcpSnoopingVerifySourceMacGet(clientHandle, af, &verify);
  if (result != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get dhcp verify source mac. (result = %d)\n", result);
  } 
  else
  {
    if (verify == OPEN_ENABLE) 
    {
      printf("DHCP snooping verifies source mac addresses\n");
    }
    else
    {
      printf("DHCP snooping does not verify source mac addresses\n");
    }
  }
}

/*********************************************************************
* @purpose  Get Enabled VLAN Range
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingEnabledVlanRange(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  uint32_t vlanStart = 0;
  uint32_t vlanEnd = 0;
  int first = 1;

  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, af, &vlanStart, &vlanEnd);
  if (result != OPEN_E_NONE) 
  {
    printf("Non-success code (%d) returned by openapiDhcpSnoopingEnabledVlanRangeNextGet\n", result);
  }

  while (result == OPEN_E_NONE && vlanStart != 0)
  {
    if (!first)
    {
      putchar(',');
      first = 0;
    } 
    if (vlanEnd != vlanStart)
    {
      printf("%u - %u", vlanStart, vlanEnd);
    }
    else
    {
      printf("%u", vlanStart);
    } 
    result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, af, &vlanStart, &vlanEnd);
    if (result != OPEN_E_NONE) 
    {
      printf("Non-success code (%d) returned by openapiDhcpSnoopingEnabledVlanRangeNextGet\n", result);
    }
  }
  putchar('\n');
}

/*********************************************************************
* @purpose  Get if DHCP snooping is enabled on an interface.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* @param    ifNum            @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingPortEnabled(openapiClientHandle_t *clientHandle, OPEN_AF_t af, int ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, af, ifNum, &mode);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get dhcp snooping interface mode. (result = %d)\n", result);
  }
  else
  {
    if (mode == OPEN_ENABLE)
    {
      printf("DHCP snooping is enabled\n");
    }
    else
    {
      printf("DHCP snooping is disabled\n");
    }
  }
}

/*********************************************************************
* @purpose  Get whether DHCP snooping considers an interface as trusted.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* @param    ifNum            @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingIntfTrustedGet(openapiClientHandle_t *clientHandle, OPEN_AF_t af, int ifNum)
{
  open_error_t result;
  OPEN_BOOL_t val;

  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, af, ifNum, &val);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get whether dhcp snooping interface is trusted. (result = %d)\n", result);
  }
  else
  {
    if (val == OPEN_TRUE)
    {
      printf("DHCP snooping considers the interface trusted\n");
    }
    else
    {
      printf("DHCP snooping considers the interface not trusted\n");
    }
  }
}

/*********************************************************************
* @purpose  Get whether DHCP snooping logs invalid packets on an interface.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* @param    ifNum            @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingIntfLoggedGet(openapiClientHandle_t *clientHandle, OPEN_AF_t af, int ifNum)
{
  open_error_t result;
  OPEN_BOOL_t val;

  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, af, ifNum, &val);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get whether DHCP snooping logs invalid packets on an interface (result = %d)\n", result);
  }
  else
  {
    if (val == OPEN_TRUE)
    {
      printf("DHCP snooping logs invalid packets on the interface\n");
    }
    else
    {
      printf("DHCP snooping does not log invalid packets on the interface\n");
    }
  }
}

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingDbFileName(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  open_buffdesc fileNameBuf;
  char *buf;
  uint32_t len;

  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(clientHandle, &len);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get file name buffer size. (result = %d)\n", result);
  }
  else 
  {
    buf = (char *) malloc(len);
    if (buf) 
    {
      fileNameBuf.pstart = buf;
      fileNameBuf.size = len;
      result = openapiDhcpSnoopingDbFileNameGet(clientHandle, af, &fileNameBuf);
      if (result != OPEN_E_NONE)
      {
        printf("Bad return code trying to get dhcp snooping db filename. (result = %d)\n", result);
      }
      else
      {
        printf("DHCP snooping db filename is %s\n", buf);
      }
      free(buf);
    }
    else
    {
      printf("malloc failed trying to get snooping db filename buffer\n");
    }
  }
}

/*********************************************************************
* @purpose  Get the remote DHCP snooping database IP Address.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingDbIpAddress(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  open_inet_addr_t addr;
  char ipAddrStr[48];

  result = openapiDhcpSnoopingDbIpAddressGet(clientHandle, af, &addr);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get dhcp snooping IP address. (result = %d)\n", result);
  }
  else
  {
    addr.addr.ipv4 = htonl(addr.addr.ipv4);
    if (ipAddressFormat(&addr, ipAddrStr) != NULL)
    {
      printf("remote IP address: %s\n", ipAddrStr);
    }
    else
    {
      printf("unable to convert IP address to string\n");
    }
  }
}

/*********************************************************************
* @purpose  Get the remote DHCP snooping database write delay.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingDbWriteDelay(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  uint32_t delay;

  result = openapiDhcpSnoopingDbWriteDelayGet(clientHandle, af, &delay);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get dhcp snooping write delay. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP snooping write delay is %d seconds\n", delay);
  }
}

/*********************************************************************
* @purpose  Perform Sanity of DHCP snooping OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void OpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t vlanStart;
  uint32_t vlanEnd;
  OPEN_CONTROL_t controlVal;
  OPEN_BOOL_t boolVal;
  open_buffdesc bufdesc;
  open_buffdesc badBufdesc;
  uint32_t len;
  uint32_t val;
  open_inet_addr_t addr;

  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(clientHandle, &len);
  bufdesc.pstart = (char *) malloc(len);
  bufdesc.size = len;
  if (bufdesc.pstart == (char *) NULL)
  {
    perror("Error: failed to malloc snooping db filename buffer:");
    exit(1);
  }

  /* openapiDhcpSnoopingAdminModeGet */
  printf("\nTesting openapiDhcpSnoopingAdminModeGet(): \n");

  result = openapiDhcpSnoopingAdminModeGet(NULL, OPEN_AF_INET, &controlVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingAdminModeGet(clientHandle, 77, &controlVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingAdminModeGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL adminMode. (result = %d)\n", result);

  printf("openapiDhcpSnoopingAdminModeGet() sanity successful. \n");

  /* openapiDhcpSnoopingVerifySourceMacGet */
  printf("\nTesting openapiDhcpSnoopingVerifySourceMacGet(): \n");

  result = openapiDhcpSnoopingVerifySourceMacGet(NULL, OPEN_AF_INET, &controlVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingVerifySourceMacGet(clientHandle, 77, &controlVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingVerifySourceMacGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL adminMode. (result = %d)\n", result);

  printf("openapiDhcpSnoopingVerifySourceMacGet() sanity successful. \n");

  /* openapiDhcpSnoopingEnabledVlanRangeNextGet */
  printf("\nTesting openapiDhcpSnoopingEnabledVlanRangeNextGet(): \n");

  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(NULL, OPEN_AF_INET, &vlanStart, &vlanEnd);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, 77, &vlanStart, &vlanEnd);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, OPEN_AF_INET, NULL, &vlanEnd);
  printf("NULL vlanStart. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, OPEN_AF_INET, &vlanStart, NULL);
  printf("NULL vlanEnd. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, OPEN_AF_INET, NULL, NULL);
  printf("NULL vlanStart and vlanEnd. (result = %d)\n", result);
  printf("openapiDhcpSnoopingEnabledVlanRangeNextGet() sanity successful. \n");

  /* openapiDhcpSnoopingPortEnabledGet */
  printf("\nTesting openapiDhcpSnoopingPortEnabledGet(): \n");

  result = openapiDhcpSnoopingPortEnabledGet(NULL, OPEN_AF_INET, 1, &controlVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, 77, 1, &controlVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, OPEN_AF_INET, 9999, &controlVal);
  printf("Invalid interface. (result = %d)\n", result);
  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, OPEN_AF_INET, 9999, &controlVal);
  printf("NULL enabled. (result = %d)\n", result);

  printf("openapiDhcpSnoopingPortEnabledGet() sanity successful. \n");

  /* openapiDhcpSnoopingIntfTrustedGet */
  printf("\nTesting openapiDhcpSnoopingIntfTrustedGet(): \n");

  result = openapiDhcpSnoopingIntfTrustedGet(NULL, OPEN_AF_INET, 1, &boolVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, 77, 1, &boolVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, OPEN_AF_INET, 9999, &boolVal);
  printf("Invalid interface. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, OPEN_AF_INET, 1, NULL);
  printf("NULL trusted. (result = %d)\n", result);

  printf("openapiDhcpSnoopingIntfTrustedGet() sanity successful. \n");

  /* openapiDhcpSnoopingIntfLoggedGet */
  printf("\nTesting openapiDhcpSnoopingIntfLoggedGet(): \n");

  result = openapiDhcpSnoopingIntfLoggedGet(NULL, OPEN_AF_INET, 1, &boolVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, 77, 1, &boolVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, OPEN_AF_INET, 9999, &boolVal);
  printf("Invalid interface. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, OPEN_AF_INET, 1, NULL);
  printf("NULL logged. (result = %d)\n", result);

  printf("openapiDhcpSnoopingIntfLoggedGet() sanity successful. \n");

 /* openapiDhcpSnoopingDbFileNameMaxLengthGet */
  printf("\nTesting openapiDhcpSnoopingDbFileNameMaxLengthGet(): \n");

  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(NULL, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(clientHandle, NULL);
  printf("NULL len parameter. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbFileNameMaxLengthGet() sanity successful. \n");

  /* openapiDhcpSnoopingDbFileNameGet */
  printf("\nTesting openapiDhcpSnoopingDbFileNameGet(): \n");

  result = openapiDhcpSnoopingDbFileNameGet(NULL, OPEN_AF_INET, &bufdesc);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, 77, &bufdesc);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL open_buffdesc. (result = %d)\n", result);
  badBufdesc.pstart = NULL;
  badBufdesc.size = len;
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, OPEN_AF_INET, &badBufdesc);
  printf("bufdesc with NULL pstart. (result = %d)\n", result);

  badBufdesc.pstart = bufdesc.pstart;
  badBufdesc.size = 0;
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, OPEN_AF_INET, &badBufdesc);
  printf("bufdesc with 0 size. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbFileNameGet() sanity successful. \n");

  /* openapiDhcpSnoopingDbIpAddressGet */
  printf("\nTesting openapiDhcpSnoopingDbIpAddressGet(): \n");

  result = openapiDhcpSnoopingDbIpAddressGet(NULL, OPEN_AF_INET, &addr);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbIpAddressGet(clientHandle, 77, &addr);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbIpAddressGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL address. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbIpAddressGet() sanity successful. \n");

  /* openapiDhcpSnoopingDbWriteDelayGet */
  printf("\nTesting openapiDhcpSnoopingDbWriteDelayGet(): \n");

  result = openapiDhcpSnoopingDbWriteDelayGet(NULL, OPEN_AF_INET, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbWriteDelayGet(clientHandle, 77, &val);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbWriteDelayGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL delay. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbWriteDelayGet() sanity successful. \n");
  free(bufdesc.pstart);
}

/*********************************************************************
* @purpose  Map argument to an instance of OPEN_AF_t.
* 
* @param    prog     @b{(input)}   input to usage, if needed.
* @param    family   @b{(input)}   family command line argument.
* 
* @returns  OPEN_AF_INET  if family == 1
* @returns  OPEN_AF_INET6 if family == 1
*
* @notes function prints usage if family is neither 1 nor 2, then exits.
*   
* @end
*********************************************************************/

OPEN_AF_t toFamily(char *prog, int family)
{
  OPEN_AF_t ret = 0;

  if (family == 1)
  {
    ret = OPEN_AF_INET;
  }
  else if (family == 2)
  {
    ret = OPEN_AF_INET6;
  }
  else
  {
    printAppMenu(prog);
    exit(1);
  }
  return ret;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Global mode
*
* @param    clientHandle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayGlobalmodeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t globalMode;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayAdminModeGet (clientHandle, &globalMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay global mode. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay global mode : %u \n", globalMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Global mode
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    mode           @b{(input)}   DHCP L2 relay global mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayGlobalmodeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayAdminModeSet (clientHandle, mode)))
  {
    printf("Bad return code trying to set DHCP L2 relay global mode. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay global mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay interface mode
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayIntfModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t intfMode;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfModeGet (clientHandle, ifNum, &intfMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay interface mode. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for interface %u: %u \n", ifNum, intfMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay interface mode
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
* @param    mode           @b{(input)}   DHCP L2 relay interface mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayIntfModeSet (openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                             OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfModeSet (clientHandle, ifNum, mode)))
  {
    printf("Bad return code trying to set DHCP L2 relay interface mode. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for interface %u set successfully.\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 relay interface trust mode
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayIntfTrustGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_BOOL_t trustMode;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfTrustGet (clientHandle, ifNum, &trustMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay interface trust mode. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay trust mode for interface %u: %u \n", ifNum, trustMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay interface trust mode
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
* @param    trustMode      @b{(input)}   DHCP L2 relay interface trust mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayIntfTrustSet (openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                              OPEN_BOOL_t trustMode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfTrustSet (clientHandle, ifNum, trustMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay interface trust mode. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay trust mode for interface %u set successfully.\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 relay mode for VLAN
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayVlanModeGet(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  OPEN_CONTROL_t vlanMode;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayVlanModeGet (clientHandle, vlanId, &vlanMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay mode for VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for VLAN %u: %u \n", vlanId, vlanMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 relay mode for VLAN
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    vlanId         @b{(input)}   VLAN ID
* @param    vlanMode       @b{(input)}   DHCP L2 relay mode for VLAN
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayVlanModeSet (openapiClientHandle_t *clientHandle, uint32_t vlanId, 
                             OPEN_CONTROL_t vlanMode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayVlanModeSet (clientHandle, vlanId, vlanMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay mode for VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for vlan ID %u set successfully.\n", vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 relay mode for range of VLAN ID's
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    vlanStart      @b{(input)}   Starting VLAN number
* @param    vlanEnd        @b{(input)}   Ending VLAN number
* @param    vlanMode       @b{(input)}   DHCP L2 relay vlan mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayVlanRangeModeSet (openapiClientHandle_t *clientHandle, 
                                  uint32_t vlanStart, uint32_t vlanEnd,
                                  OPEN_CONTROL_t vlanMode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayVlanRangeModeSet (clientHandle, vlanStart, vlanEnd, vlanMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay mode for VLANs. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for vlan ID's %u - %u set successfully.\n", vlanStart, vlanEnd);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 relay circuit ID mode for VLAN
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayVlanCircuitIdModeGet(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  OPEN_CONTROL_t vlanCirMode;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayCircuitIdGet (clientHandle, vlanId, &vlanCirMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay circuit ID mode for VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay circuit ID mode for VLAN %u: %u \n", vlanId, vlanCirMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 relay circuit ID mode for VLAN
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    vlanId         @b{(input)}   VLAN ID
* @param    vlanCirMode    @b{(input)}   DHCP L2 relay circuit ID mode for VLAN
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayVlanCircuitIdModeSet (openapiClientHandle_t *clientHandle, 
                                      uint32_t vlanId, 
                                      OPEN_CONTROL_t vlanCirMode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayCircuitIdSet (clientHandle, vlanId, 
                      vlanCirMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay circuit ID mode for VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay circuit ID mode for vlan ID %u set successfully.\n", vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 relay circuit ID mode for range of VLAN ID's
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    vlanStart      @b{(input)}   Starting VLAN number
* @param    vlanEnd        @b{(input)}   Ending VLAN number
* @param    vlanMode       @b{(input)}   DHCP L2 relay vlan circuit ID mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayVlanRangeCircuitIdModeSet (openapiClientHandle_t *clientHandle, 
                                           uint32_t vlanStart, uint32_t vlanEnd,
                                           OPEN_CONTROL_t vlanCirMode)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayVlanRangeCircuitIdSet (clientHandle,
                      vlanStart, vlanEnd, vlanCirMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay circuit ID mode for VLANs. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay circuit ID mode for vlan ID's %u - %u set successfully.\n", vlanStart, vlanEnd);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 relay remote ID for VLAN
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayRemoteIdGet(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  char remoteId[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId = {.pstart = remoteId, .size = sizeof(remoteId)-1};
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayRemoteIdGet (clientHandle, vlanId, &bufId)))
  {
    printf("Bad return code trying to get DHCP L2 relay remote ID for VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay remote ID for VLAN %u: %s \n", vlanId, (char *) bufId.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 relay remote ID for VLAN
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
* @param    remoteId       @b{(input)}  Remote ID
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayRemoteIdSet(openapiClientHandle_t *clientHandle, uint32_t vlanId,
                            char *remId)
{
  open_error_t result;
  char remoteId[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
  
	strcpy(remoteId, remId);
  bufId.pstart = remoteId;
  bufId.size   = strlen (remoteId);
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayRemoteIdSet (clientHandle, vlanId, 
                      &bufId)))
  {
    printf("Bad return code trying to set DHCP L2 relay remote ID for VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay remote ID %s for vlan ID %u set successfully.\n", 
           remoteId, vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 relay remote ID for range of VLAN ID's
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    vlanStart      @b{(input)}   Starting VLAN number
* @param    vlanEnd        @b{(input)}   Ending VLAN number
* @param    remoteId       @b{(input)}  Remote ID
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayVlanRangeRemoteIdSet (openapiClientHandle_t *clientHandle, 
                                      uint32_t vlanStart, uint32_t vlanEnd,
                                      char *remId)
{
  open_error_t result;
  char remoteId[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
  
	strcpy(remoteId, remId);
  bufId.pstart = remoteId;
  bufId.size   = strlen (remoteId);
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle,
                      vlanStart, vlanEnd, &bufId)))
  {
    printf("Bad return code trying to set DHCP L2 relay remote ID for VLANs. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay remote ID %s for vlan ID's %u - %u set successfully.\n", 
           remoteId, vlanStart, vlanEnd);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay mode for a given subscription on an interface.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    subscriptionName @b((input))  subsription name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelaySubscriptionModeGet(openapiClientHandle_t *clientHandle,
                                    uint32_t ifNum, char *subscriptionName)
{
  open_error_t result;
  OPEN_CONTROL_t l2RelaySubMode;
  char subName[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
  
	strcpy(subName, subscriptionName);
  bufId.pstart = subName;
  bufId.size   = strlen (subName);
  
  if (OPEN_E_NONE != (result = openapiDsL2RelaySubscriptionModeGet (clientHandle,
                      ifNum, &bufId, &l2RelaySubMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay mode for a given "
           "subscription on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for a given subscription on an interface %u: %u \n", 
           ifNum, l2RelaySubMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a given subscription on an interface.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    subscriptionName @b((input))  subsription name
* @param    l2RelayMode      @b((input))  DHCP L2 Relay  mode
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelaySubscriptionModeSet (openapiClientHandle_t *clientHandle, 
                                     uint32_t ifNum, char *subscriptionName,
                                     OPEN_CONTROL_t l2RelayMode)
{
  open_error_t result;
  char subName[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
 
	strcpy(subName, subscriptionName);
  bufId.pstart = subName;
  bufId.size   = strlen (subName);

  if (OPEN_E_NONE != (result = openapiDsL2RelaySubscriptionModeSet (clientHandle,
                      ifNum, &bufId, l2RelayMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay mode for a given "
           "subscription on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay mode for a given subscription on an interface %u"
            " set successfully.\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay circuit ID mode for a given subscription
*           on an interface.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    subscriptionName @b((input))  subsription name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelaySubscriptionCircuitIdModeGet(openapiClientHandle_t *clientHandle,
                                             uint32_t ifNum, char *subscriptionName)
{
  open_error_t result;
  OPEN_CONTROL_t l2RelayCirMode;
  char subName[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
  
	strcpy(subName, subscriptionName);
  bufId.pstart = subName;
  bufId.size   = strlen (subName);
  
  if (OPEN_E_NONE != (result = openapiDsL2RelaySubscriptionCircuitIdModeGet (clientHandle,
                      ifNum, &bufId, &l2RelayCirMode)))
  {
    printf("Bad return code trying to get DHCP L2 relay circuit ID mode for a given "
           "subscription on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay circuit ID mode for a given subscription on an interface %u: %u \n", 
           ifNum, l2RelayCirMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay circuit ID mode for a given subscription
*           on an interface.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    subscriptionName @b((input))  subsription name
* @param    l2RelayCirMode   @b((input))  DHCP L2 Relay Circuit ID mode
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelaySubscriptionCircuitIdModeSet (openapiClientHandle_t *clientHandle, 
                                              uint32_t ifNum, char *subscriptionName,
                                              OPEN_CONTROL_t l2RelayCirMode)
{
  open_error_t result;
  char subName[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
 
	strcpy(subName, subscriptionName);
  bufId.pstart = subName;
  bufId.size   = strlen (subName);

  if (OPEN_E_NONE != (result = openapiDsL2RelaySubscriptionCircuitIdModeSet (clientHandle,
                      ifNum, &bufId, l2RelayCirMode)))
  {
    printf("Bad return code trying to set DHCP L2 relay circuit ID mode for a given "
           "subscription on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay circuit ID mode for a given subscription on an interface"
            " %u set successfully.\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay remote ID for a given subscription
*           on an interface.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    subscriptionName @b((input))  subsription name
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelaySubscriptionRemoteIdGet(openapiClientHandle_t *clientHandle,
                                        uint32_t ifNum, char *subscriptionName)
{
  open_error_t result;
  char subName[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
  char remId[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc remoteId;
  
	strcpy(subName, subscriptionName);
  bufId.pstart = subName;
  bufId.size   = strlen (subName);

  remoteId.pstart = remId;
  remoteId.size   = sizeof (remId) - 1;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelaySubscriptionRemoteIdGet (clientHandle,
                      ifNum, &bufId, &remoteId)))
  {
    printf("Bad return code trying to get DHCP L2 relay remote ID for a given "
           "subscription on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay remote ID for a given subscription on an interface %u: %s \n", 
           ifNum, (char *)remoteId.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay remote ID mode for a given subscription
*           on an interface.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    subscriptionName @b((input))  subsription name
* @param    remoteId         @b((input))  DHCP L2 Relay remote ID
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelaySubscriptionRemoteIdSet (openapiClientHandle_t *clientHandle, 
                                         uint32_t ifNum, char *subscriptionName,
                                         char *rId)
{
  open_error_t result;
  char subName[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufId;
  char remId[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc remoteId;
  
	strcpy(subName, subscriptionName);
  bufId.pstart = subName;
  bufId.size   = strlen (subName);

	strcpy(remId, rId);
  remoteId.pstart = remId;
  remoteId.size   = strlen (remId);
  
  if (OPEN_E_NONE != (result = openapiDsL2RelaySubscriptionRemoteIdSet (clientHandle,
                      ifNum, &bufId, &remoteId)))
  {
    printf("Bad return code trying to set DHCP L2 relay remote ID for a given "
           "subscription on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay remote ID for a given subscription on an interface"
            " %u set successfully.\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay interface specific statistics.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayIntfStatsGet(openapiClientHandle_t *clientHandle,
                             uint32_t ifNum)
{
  open_error_t result;
  uint32_t untrustedSrvr = 0, untrustedClnt = 0;
  uint32_t trustedSrvr = 0, trustedClnt = 0;
  

  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfStatsGet (clientHandle,
                      ifNum, &untrustedSrvr, &untrustedClnt, &trustedSrvr,
                      &trustedClnt)))
  {
    printf("Bad return code trying to get DHCP L2 relay "
           "statistics on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay statistics.\n");
    printf("Interface    UntrustedServer  UntrustedClient  TrustedServer     TrustedClient\n");
    printf("-----------  ---------------  ---------------  ----------------  ----------------\n");
    printf("%-11u  %-15u  %-15u  %-16u  %-16u\n", ifNum, untrustedSrvr, 
           untrustedClnt, trustedSrvr, trustedClnt); 
  }
  return;
}

/*********************************************************************
* @purpose  Clears the DHCP L2 Relay interface specific statistics.
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayIntfStatsClear (openapiClientHandle_t *clientHandle,
                                uint32_t ifNum)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfStatsClear (clientHandle,
                      ifNum)))
  {
    printf("Bad return code trying to clear DHCP L2 relay "
           "statistics of an interface. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP L2 relay statistics cleared successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the configured no-option-82 action on L2 relay trusted
*           interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpL2RelayIntfTrustNoOption82ActionGet(openapiClientHandle_t *clientHandle,
                                             int32_t ifNum)
{
  open_error_t result;
  uint32_t action;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfTrustNoOption82ActionGet (clientHandle, ifNum, &action)))
  {
    printf("Bad return code trying to get no option 82 action. (result = %d)\n", result);
  }
  else
  {
    printf("Action code for no option 82 for interace %u: %u \n", ifNum, action);
  }
  return;
}

/*********************************************************************
* @purpose  Configure action for packets received without option-82
*           on L2 relay trusted interfaces.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
* @param    action         @b{(input)}   drop/keep/update
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dhcpL2RelayIntfTrustNoOption82ActionSet (openapiClientHandle_t *clientHandle,
                                              uint32_t ifNum, uint32_t action)
{
  open_error_t result;
  
  if (OPEN_E_NONE != (result = openapiDsL2RelayIntfTrustNoOption82ActionSet (clientHandle, ifNum, action)))
  {
    printf("Bad return code trying to set no option 82 action. (result = %d)\n", result);
  }
  else
  {
    printf("Action code for no option 82 for interface %u set successfully.\n", ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Perform Sanity of DHCP L2 relay OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void openApiL2RelaySanity (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  OPEN_BOOL_t    val = OPEN_FALSE;
  uint32_t ifNum = 1, vlanId = MAX_VLAN_ID + 1, startVlanId = 1, endVlanId = 1;
  uint32_t statVal = 0, actionVal = 0;
  char bufStr[DS_L2RELAY_STRING_SIZE] = "";
  open_buffdesc bufDesc = {.pstart = bufStr, .size = sizeof(bufStr)-1}; 
  open_buffdesc nullBufDesc = {.pstart = NULL, .size = 0}; 

  printf("Testing DHCP L2 relay OpEN APIs sanity:\n");

  printf("\r\nTesting openapiDsL2RelayAdminModeGet():\n");
  result = openapiDsL2RelayAdminModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayAdminModeGet(clientHandle, NULL);
  printf("NULL admin mode:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayAdminModeSet():\n");
  result = openapiDsL2RelayAdminModeSet(NULL, mode);
  printf("NULL client handle:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfModeGet():\n");
  result = openapiDsL2RelayIntfModeGet(NULL, ifNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfModeGet(clientHandle, 0, &mode);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelayIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL interface mode:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfModeSet():\n");
  result = openapiDsL2RelayIntfModeSet(NULL, ifNum, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfModeSet(clientHandle, 0, mode);
  printf("NULL interface mode:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfTrustGet():\n");
  result = openapiDsL2RelayIntfTrustGet(NULL, ifNum, &val);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfTrustGet(clientHandle, 0, &val);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelayIntfTrustGet(clientHandle, ifNum, NULL);
  printf("NULL trust mode:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfTrustSet():\n");
  result = openapiDsL2RelayIntfTrustSet(NULL, ifNum, val);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfTrustSet(clientHandle, 0, val);
  printf("NULL interface value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayVlanModeGet():\n");
  result = openapiDsL2RelayVlanModeGet(NULL, vlanId, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayVlanModeGet(clientHandle, vlanId, NULL);
  printf("NULL relay mode :(result = %d)\n", result);
  result = openapiDsL2RelayVlanModeGet(clientHandle, 0, &mode);
  printf("NULL vlan ID :(result = %d)\n", result);
  result = openapiDsL2RelayVlanModeGet(clientHandle, vlanId, &mode);
  printf("Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayVlanModeSet():\n");
  result = openapiDsL2RelayVlanModeSet (NULL, vlanId, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayVlanModeSet (clientHandle, 0, mode);
  printf("NULL vlan ID :(result = %d)\n", result);
  result = openapiDsL2RelayVlanModeSet (clientHandle, vlanId, mode);
  printf("Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayVlanRangeModeSet():\n");
  result = openapiDsL2RelayVlanRangeModeSet (NULL, startVlanId, endVlanId, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeModeSet (clientHandle, 0, endVlanId, mode);
  printf("NULL value for start vlan ID:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeModeSet (clientHandle, vlanId, endVlanId, mode);
  printf("Start Vlan ID more than maximum value:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeModeSet (clientHandle, startVlanId, 0, mode);
  printf("NULL value for end vlan ID:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeModeSet (clientHandle, startVlanId, vlanId, mode);
  printf("End Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayCircuitIdGet():\n");
  result = openapiDsL2RelayCircuitIdGet (NULL, vlanId, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayCircuitIdGet (clientHandle, vlanId, NULL);
  printf("NULL circuit id mode:(result = %d)\n", result);
  result = openapiDsL2RelayCircuitIdGet (clientHandle, 0, &mode);
  printf("NULL VLAN id :(result = %d)\n", result);
  result = openapiDsL2RelayCircuitIdGet (clientHandle, vlanId, &mode);
  printf("Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayCircuitIdSet():\n");
  result = openapiDsL2RelayCircuitIdSet (NULL, vlanId, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayCircuitIdSet (clientHandle, 0, mode);
  printf("NULL VLAN id :(result = %d)\n", result);
  result = openapiDsL2RelayCircuitIdSet (clientHandle, vlanId, mode);
  printf("Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayVlanRangeCircuitIdSet():\n");
  result = openapiDsL2RelayVlanRangeCircuitIdSet (NULL, startVlanId, endVlanId, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeCircuitIdSet (clientHandle, 0, endVlanId, mode);
  printf("NULL value for start vlan ID:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeCircuitIdSet (clientHandle, vlanId, endVlanId, mode);
  printf("Start Vlan ID more than maximum value:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeCircuitIdSet (clientHandle, startVlanId, 0, mode);
  printf("NULL value for end vlan ID:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeCircuitIdSet (clientHandle, startVlanId, vlanId, mode);
  printf("End Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayRemoteIdGet():\n");
  result = openapiDsL2RelayRemoteIdGet (NULL, vlanId, &bufDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdGet (clientHandle, vlanId, NULL);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdGet (clientHandle, vlanId, &nullBufDesc);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdGet (clientHandle, 0, &bufDesc);
  printf("NULL VLAN ID:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdGet (clientHandle, vlanId, &bufDesc);
  printf("Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayRemoteIdSet():\n");
  result = openapiDsL2RelayRemoteIdSet (NULL, vlanId, &bufDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdSet (clientHandle, vlanId, NULL);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdSet (clientHandle, vlanId, &nullBufDesc);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdSet (clientHandle, 0, &bufDesc);
  printf("NULL VLAN ID:(result = %d)\n", result);
  result = openapiDsL2RelayRemoteIdSet (clientHandle, vlanId, &bufDesc);
  printf("Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayVlanRangeRemoteIdSet():\n");
  result = openapiDsL2RelayVlanRangeRemoteIdSet (NULL, startVlanId, endVlanId, &bufDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle, startVlanId, endVlanId, NULL);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle, startVlanId, endVlanId, &nullBufDesc);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle, 0, endVlanId, &bufDesc);
  printf("NULL value for start vlan ID:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle, vlanId, endVlanId, &bufDesc);
  printf("Start Vlan ID more than maximum value:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle, startVlanId, 0, &bufDesc);
  printf("NULL value for end vlan ID:(result = %d)\n", result);
  result = openapiDsL2RelayVlanRangeRemoteIdSet (clientHandle, startVlanId, vlanId, &bufDesc);
  printf("End Vlan ID more than maximum value:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelaySubscriptionModeGet():\n");
  result = openapiDsL2RelaySubscriptionModeGet (NULL, ifNum, &bufDesc, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeGet (clientHandle, 0, &bufDesc, &mode);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeGet (clientHandle, ifNum, NULL, &mode);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeGet (clientHandle, ifNum, &nullBufDesc, &mode);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeGet (clientHandle, ifNum, &bufDesc, NULL);
  printf("NULL relay subscription mode = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelaySubscriptionModeSet():\n");
  result = openapiDsL2RelaySubscriptionModeSet (NULL, ifNum, &bufDesc, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeSet (clientHandle, 0, &bufDesc, mode);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeSet (clientHandle, ifNum, NULL, mode);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionModeSet (clientHandle, ifNum, &nullBufDesc, mode);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelaySubscriptionCircuitIdModeGet():\n");
  result = openapiDsL2RelaySubscriptionCircuitIdModeGet (NULL, ifNum, &bufDesc, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeGet (clientHandle, 0, &bufDesc, &mode);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeGet (clientHandle, ifNum, NULL, &mode);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeGet (clientHandle, ifNum, &nullBufDesc, &mode);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeGet (clientHandle, ifNum, &bufDesc, NULL);
  printf("NULL relay circuit mode = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelaySubscriptionCircuitIdModeSet():\n");
  result = openapiDsL2RelaySubscriptionCircuitIdModeSet (NULL, ifNum, &bufDesc, mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeSet (clientHandle, 0, &bufDesc, mode);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeSet (clientHandle, ifNum, NULL, mode);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionCircuitIdModeSet (clientHandle, ifNum, &nullBufDesc, mode);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelaySubscriptionRemoteIdGet():\n");
  result = openapiDsL2RelaySubscriptionRemoteIdGet (NULL, ifNum, &bufDesc, &bufDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdGet (clientHandle, 0, &bufDesc, &bufDesc);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdGet (clientHandle, ifNum, NULL, &bufDesc);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdGet (clientHandle, ifNum, &nullBufDesc, &bufDesc);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdGet (clientHandle, ifNum, &bufDesc, NULL);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdGet (clientHandle, ifNum, &bufDesc, &nullBufDesc);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelaySubscriptionRemoteIdSet():\n");
  result = openapiDsL2RelaySubscriptionRemoteIdSet (NULL, ifNum, &bufDesc, &bufDesc);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdSet (clientHandle, 0, &bufDesc, &bufDesc);
  printf("NULL interface value:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdSet (clientHandle, ifNum, NULL, &bufDesc);
  printf("NULL buf descriptor:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdSet (clientHandle, ifNum, &nullBufDesc, &bufDesc);
  printf("Buf descriptor without string memory alloc:(result = %d)\n", result);
  result = openapiDsL2RelaySubscriptionRemoteIdSet (clientHandle, ifNum, &bufDesc, NULL);
  printf("NULL buf descriptor:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfStatsGet():\n");
  result = openapiDsL2RelayIntfStatsGet (NULL, ifNum, &statVal, &statVal, &statVal, &statVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfStatsGet (clientHandle, 0, &statVal, &statVal, &statVal, &statVal);
  printf("NULL interface :(result = %d)\n", result);
  result = openapiDsL2RelayIntfStatsGet (clientHandle, ifNum, NULL, &statVal, &statVal, &statVal);
  printf("NULL stats input:(result = %d)\n", result);
  result = openapiDsL2RelayIntfStatsGet (clientHandle, ifNum, &statVal, NULL, &statVal, &statVal);
  printf("NULL stats input:(result = %d)\n", result);
  result = openapiDsL2RelayIntfStatsGet (clientHandle, ifNum, &statVal, &statVal, NULL, &statVal);
  printf("NULL stats input:(result = %d)\n", result);
  result = openapiDsL2RelayIntfStatsGet (clientHandle, ifNum, &statVal, &statVal, &statVal, NULL);
  printf("NULL stats input:(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfStatsClear():\n");
  result = openapiDsL2RelayIntfStatsClear (NULL, ifNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfStatsClear (clientHandle, 0);
  printf("NULL interface :(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfTrustNoOption82ActionGet():\n");
  result = openapiDsL2RelayIntfTrustNoOption82ActionGet (NULL, ifNum, &actionVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfTrustNoOption82ActionGet (clientHandle, 0, &actionVal);
  printf("NULL interface :(result = %d)\n", result);
  result = openapiDsL2RelayIntfTrustNoOption82ActionGet (clientHandle, ifNum, NULL);
  printf("NULL action value :(result = %d)\n", result);

  printf("\r\nTesting openapiDsL2RelayIntfTrustNoOption82ActionSet():\n");
  result = openapiDsL2RelayIntfTrustNoOption82ActionSet (NULL, ifNum, actionVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDsL2RelayIntfTrustNoOption82ActionSet (clientHandle, 0, actionVal);
  printf("NULL interface :(result = %d)\n", result);
}
 
/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for DHCP server
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  OPEN_AF_t af;
  
  if (argc < 2)
  {
    printAppMenu(argv[0]);
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DHCP snooping API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingAdminMode(&clientHandle, af);
      break;
    case 2:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingVerifySourceMac(&clientHandle, af);
      break;
    case 3:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingEnabledVlanRange(&clientHandle, af);
      break;
    case 4:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingPortEnabled(&clientHandle, af, atoi(argv[3]));
      break;
    case 5:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingIntfTrustedGet(&clientHandle, af, atoi(argv[3]));
      break;
    case 6:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingIntfLoggedGet(&clientHandle, af, atoi(argv[3]));
      break;
    case 7:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingDbFileName(&clientHandle, af);
      break;
    case 8:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingDbIpAddress(&clientHandle, af);
      break;
    case 9:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingDbWriteDelay(&clientHandle, af);
      break;
    case 10:
      OpENAPISanity(&clientHandle);
      break;
    case 11:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayGlobalmodeGet(&clientHandle);
      break;
    case 12:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayGlobalmodeSet(&clientHandle, atoi(argv[2]));
      break;
    case 13:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfModeGet (&clientHandle, atoi(argv[2]));
      break;
    case 14:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfModeSet (&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
     case 15:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfTrustGet (&clientHandle, atoi(argv[2]));
      break;
    case 16:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfTrustSet (&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
    case 17:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanModeGet (&clientHandle, atoi(argv[2]));
      break;
    case 18:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanModeSet (&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
     case 19:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanRangeModeSet (&clientHandle, atoi(argv[2]), atoi(argv[3]),
                                   atoi(argv[4]));
      break;
     case 20:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanCircuitIdModeGet (&clientHandle, atoi(argv[2]));
      break;
    case 21:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanCircuitIdModeSet (&clientHandle, atoi(argv[2]), atoi(argv[3]));
      break;
     case 22:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanRangeCircuitIdModeSet (&clientHandle, atoi(argv[2]),
                                            atoi(argv[3]), atoi(argv[4]));
      break;
      case 23:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayRemoteIdGet (&clientHandle, atoi(argv[2]));
      break;
    case 24:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayRemoteIdSet (&clientHandle, atoi(argv[2]), argv[3]);
      break;
     case 25:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayVlanRangeRemoteIdSet (&clientHandle, atoi(argv[2]),
                                       atoi(argv[3]), argv[4]);
      break;
     case 26:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelaySubscriptionModeGet (&clientHandle, atoi(argv[2]), argv[3]);
      break;
     case 27:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelaySubscriptionModeSet (&clientHandle, atoi(argv[2]), argv[3],
                                      atoi(argv[4]));
      break;
     case 28:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelaySubscriptionCircuitIdModeGet (&clientHandle, atoi(argv[2]),
                                               argv[3]);
      break;
     case 29:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelaySubscriptionCircuitIdModeSet (&clientHandle, atoi(argv[2]),
                                               argv[3], atoi(argv[4]));
      break;
     case 30:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelaySubscriptionRemoteIdGet (&clientHandle, atoi(argv[2]),
                                          argv[3]);
      break;
     case 31:
      if (argc != 5)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelaySubscriptionRemoteIdSet (&clientHandle, atoi(argv[2]),
                                          argv[3], argv[4]);
      break;
    case 32:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfStatsGet (&clientHandle, atoi(argv[2]));
      break;
     case 33:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfStatsClear (&clientHandle, atoi(argv[2]));
      break;
     case 34:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfTrustNoOption82ActionGet (&clientHandle, atoi(argv[2]));
      break;
     case 35:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpL2RelayIntfTrustNoOption82ActionSet (&clientHandle, atoi(argv[2]),
                                               atoi(argv[3]));
      break;
     case 36:
      openApiL2RelaySanity(&clientHandle);
      break;
 
    default:
      printAppMenu(argv[0]);
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DHCP snooping and L2 relay API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
