/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1ad_example.c
*
* @purpose   QinQ Configuration OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    06/02/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1ad.h"
#include "openapi_if.h"
#include "openapi_vlan.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*********************************************************************
* @purpose  Set the DOT1AD mode for interface 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ifNumber         @b{(input)}   Interface number
* @param    type             @b{(input)}   Port type (nni/switchport/uni/uni-s/uni-p)
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1adModeInterfaceSet(openapiClientHandle_t *clientHandle, char *ifNumber,
                            char *type)
{
  open_error_t result;
  open_buffdesc intfNumStr;
  OPEN_DOT1AD_INTERFACE_TYPE_t portType;
  char str[256];
  uint32_t intIfNum;

  if (strcmp(type, "nni") == 0)
  {
    portType = OPEN_DOT1AD_INTERFACE_TYPE_NNI;
  }
  else  if (strcmp(type, "switchport") == 0)
  {
    portType = OPEN_DOT1AD_INTERFACE_TYPE_SWITCHPORT;
  }
  else  if (strcmp(type, "uni") == 0)
  {
    portType = OPEN_DOT1AD_INTERFACE_TYPE_UNI;
  }
  else  if (strcmp(type, "uni-s") == 0)
  {
    portType = OPEN_DOT1AD_INTERFACE_TYPE_UNI_S;
  }
  else  if (strcmp(type, "uni-p") == 0)
  {
    portType = OPEN_DOT1AD_INTERFACE_TYPE_UNI_P;
  }

  memset(str, 0, sizeof(str));
  strncpy(str, ifNumber, (sizeof(str) - 1));
  intfNumStr.pstart = str;
  intfNumStr.size = strlen(str) + 1;
  openapiIfNumGet(clientHandle, &intfNumStr, &intIfNum);


  if ((result = openapiDot1adInterfaceTypeSet(clientHandle, intIfNum, portType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NNI mode on interface %s. (result = %d)\n", str, result);
  }
  else
  {
    printf("NNI mode on interface %s is set successfully. (result = %d)\n", str, result);
  }
  if ((result = openapiDot1adInterfaceTypeGet(clientHandle, intIfNum, &portType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NNI mode on interface %s. (result = %d)\n", str, result);
  }
  else
  {
    printf("NNI mode on interface %s is get successfully. (result = %d, value = %d)\n", str, result, portType);
  }

  return;
}

/*********************************************************************
* @purpose  Example Create dot1ad service on interface
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    servName       @b{(input)}   Service name
* @param    vlanId         @b{(input)}   Service vlan id
* @param    service        @b{(input)}   Service type (e-lan/e-line/tls)
* @param    ifNumber       @b{(input)}   Interface number 
*
* @returns  none
*
* @notes
* 
* @end
*********************************************************************/
void dot1adServiceCreate(openapiClientHandle_t *clientHandle, char *servName,
                         uint32_t svid, char *service, char *ifNumber)
{
  open_error_t rc;
  open_buffdesc servNameStr;
  open_buffdesc ifNumberStr;
  OPEN_DOT1AD_SVCTYPE_t serviceType;
  uint32_t tmp;
  OPEN_BOOL_t flag;
  uint32_t nniIntfList[1];
  size_t len;

  len = strlen(servName) + 1;
  servNameStr.pstart = malloc(len);
  memset(servNameStr.pstart, 0, len);
  strncpy(servNameStr.pstart, servName, (len - 1));
  servNameStr.size = strlen(servNameStr.pstart) + 1;

  if (strcmp(service, "e-lan") == 0)
  {
    serviceType = OPEN_DOT1AD_SVCTYPE_ELAN;
  }
  else  if (strcmp(service, "e-line") == 0)
  {
    serviceType = OPEN_DOT1AD_SVCTYPE_ELINE;
  }
  else  if (strcmp(service, "tls") == 0)
  {
    serviceType = OPEN_DOT1AD_SVCTYPE_TLS;
  }

  len = strlen(ifNumber) + 1;
  ifNumberStr.pstart = malloc(len);
  memset(ifNumberStr.pstart, 0, len);
  strncpy(ifNumberStr.pstart, ifNumber, (len - 1));
  ifNumberStr.size = strlen(ifNumberStr.pstart) + 1;

  while (true)
  {
    if (openapiIfNumGet(clientHandle, &ifNumberStr, &nniIntfList[0]) != OPEN_E_EMPTY)
    {
      printf("\nERROR: Invalid interface specified\n");
      break;
    }

    /* Check if this name already exists */
    if (openapiDot1adServiceNameToSvid(clientHandle, &servNameStr, &tmp) == OPEN_E_EMPTY)
    {
      printf("\nERROR: Service with this name is already configured\n");
      break;
    }
    /* Check if this service already configured on specified service vlan id */
    openapiDot1adServiceVidIsConfigured(clientHandle, svid, &flag);
    if (flag == OPEN_TRUE)
    {
      printf("\nERROR: Service with this service VLAN ID is already configured\n");
      break;
    }
    /* Check if this service already configured on specified vlan isolate id */
    openapiDot1adServiceIsolateVidIsConfigured(clientHandle, svid, &flag);
    if(flag == OPEN_TRUE)
    {
      printf("\nERROR: Service VLAN ID is equal to Isolate VLAN ID of already configured service\n");
      break;
    }

    rc = openapiDot1adServiceCreate(clientHandle, svid);
    if (rc != OPEN_E_NONE)
    {
      printf("\nINFORMATION: Service with this name is already configured\n");
    }

    rc = openapiDot1adServiceNameAdd(clientHandle, svid, &servNameStr);
    if (rc != OPEN_E_NONE)
    {
      printf("\nINFORMATION: Adding Name Failed\n");
      openapiDot1adServiceDelete(clientHandle, svid);
      break;
    }

    rc = openapiDot1adServiceTypeSet(clientHandle, svid, serviceType);
    if (rc != OPEN_E_NONE)
    {
      printf("\nINFORMATION: Setting Type Failed\n");
      openapiDot1adServiceDelete(clientHandle, svid);
      break;
    }

    rc = openapiDot1adNniIntfListSet(clientHandle, svid, &ifNumberStr);
    if (rc != OPEN_E_NONE)
    {
      printf("\nINFORMATION: Setting NNI List Failed\n");
      openapiDot1adServiceDelete(clientHandle, svid);
      break;
    }

    printf("\nINFORMATION: Service configured successfully\n");
    break;
  }

  free(ifNumberStr.pstart);
  free(servNameStr.pstart);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t vlanId =5;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dot1ad_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DOT1AD API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS Version retrieve error\n");
  }

  printf("\n");

  if (openapiVlanCreate(&clientHandle, vlanId) == OPEN_E_NONE) /* create 5 vlan */
  {
    dot1adModeInterfaceSet(&clientHandle, "0/5", "e-lan"); /* switch 5 vlan to e-line mode */
    dot1adServiceCreate(&clientHandle, "some service name", vlanId, "e-lan", "0/5");
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DOT1AD API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

