/*********************************************************************
*
*  Copyright 2019 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1cb_example.c
*
* @purpose   OpEN Dot1cb example.
*
* @component OpEN
*
* @create    05/29/2019
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1cb.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

#define OPENAPI_TITLE         "802.1CB"
#define OPEN_IP6_ADDR_LENGTH  40
#define OPEN_MAC_ADDR_LENGTH  18

/*******************************************************************
*
* @brief  This function prints the Dot1cb Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the 802.1CB mode for a given VLAN: %s 1 <vlanID>\n", name);
  printf("Test 2: Set the 802.1CB mode for a given VLAN: %s 2 <vlanID><mode: 0-disable/1-enable>\n", name);
  printf("Test 3: Get the 802.1CB ARP/NDP forwarding mode for a given VLAN: %s 3 <vlanID>\n", name);
  printf("Test 4: Set the 802.1CB ARP/NDP forwarding mode for a given VLAN: %s 4 <vlanID><mode: 0-disable/1-enable>\n", name);
  printf("Test 5: Get the configured flow ID: %s 5 <buf><vlanID>\n", name);
  printf("Test 6: Get the configured flow type: %s 6 <buf><vlanID>\n", name);
  printf("Test 7: Set the flow type and ID of an existing macfilter: %s 7 <buf><vlanID><flowType: 1-initiator/2-transit><flowId: 1-1024>\n", name);
  printf("Test 8: Get the 802.1CB RTAG mode for the specified interface: %s 8 <ifNum>\n", name);
  printf("Test 9: Set the 802.1CB RTAG mode for the specified interface: %s 9 <ifNum><mode: 0-disable/1-enable>\n", name);
  printf("Test 10: Clear all the 802.1CB statistics: %s 10\n", name);
  printf("Test 11: Get the 802.1CB global statistics: %s 11\n", name);
  printf("Test 12: Get the 802.1CB statistics for the specified interface: %s 12 <ifNum>\n", name);
  printf("Test 13: Get the next valid 802.1CB interface: %s 13 <ifNum>\n", name);
  printf("Test 14: Add an IPv4 classifier: %s 14 <srcAddr><srcMaskLen><dstAddr><dstMaskLen><flowType><flowId>\n", name);
  printf("Test 15: Delete an IPv4 classifier: %s 15 <srcAddr><srcMaskLen><dstAddr><dstMaskLen>\n", name);
  printf("Test 16: Get an IPv4 classifier: %s 16 <srcAddr><srcMaskLen><dstAddr><dstMaskLen>\n", name);
  printf("Test 17: Get the next IPv4 classifier: %s 17 <srcAddr><srcMaskLen><dstAddr><dstMaskLen>\n", name);
  printf("Test 18: Get the number of configured IPv4 classifiers: %s 18\n", name);
  printf("Test 19: Get the status flag indicating whether IPv4 classifiers are installed in the hardware: %s 19\n", name);
  printf("Test 20: Add an IPv6 classifier: %s 20 <srcAddr><srcMaskLen><dstAddr><dstMaskLen><flowType><flowId>\n", name);
  printf("Test 21: Delete an IPv6 classifier: %s 21 <srcAddr><srcMaskLen><dstAddr><dstMaskLen>\n", name);
  printf("Test 22: Get an IPv6 classifier: %s 22 <srcAddr><srcMaskLen><dstAddr><dstMaskLen>\n", name);
  printf("Test 23: Get the next IPv6 classifier: %s 23 <srcAddr><srcMaskLen><dstAddr><dstMaskLen>\n", name);
  printf("Test 24: Get the number of configured IPv6 classifiers: %s 24\n", name);
  printf("Test 25: Get the status flag indicating whether IPv6 classifiers are installed in the hardware: %s 25\n", name);
  printf("Test 26: Get the number of transit flows installed in the hardware: %s 26\n", name);
  printf("Test 27: Get the number of initiator flows installed in the hardware: %s 27\n", name);
  printf("Test 28: Get the number of MFDB entries associated with flows: %s 28\n", name);
  printf("Test 29: Get the number of 802.1CB enabled VLANs: %s 29\n", name);
  printf("Test 30: Get the next 802.1CB enabled VLAN: %s 30 <vid>\n", name);
  printf("Test 31: Get the 802.1CB initiator flow and its status/statistics: %s 31 <flowId: 1-1024>\n", name);
  printf("Test 32: Get the next 802.1CB initiator flow and its status/statistics: %s 32 <flowId>\n", name);
  printf("Test 33: Get the 802.1CB transit flow and its status/statistics: %s 33 <flowId: 1-1024>\n", name);
  printf("Test 34: Get the next 802.1CB transit flow and its status/statistics: %s 34 <flowId>\n", name);
  printf("Test 35: Get the 802.1CB transit flow configuration: %s 35 <flowId>\n", name);
  printf("Test 36: Set the history window size for an 802.1CB transit flow: %s 36 <flowId><historyWindow>\n", name);
  printf("Test 37: Set the acceptance window size for an 802.1CB transit flow: %s 37 <flowId><acceptanceWindow>\n", name);
  printf("Test 38: Set the age time for an 802.1CB transit flow: %s 38 <flowId><ageTime>\n", name);
  printf("Test 39: Get the SA-MAC/VLAN flow association mode: %s 39\n", name);
  printf("Test 40: Set the SA-MAC/VLAN flow association mode: %s 40 <mode: 0-disable/1-enable>\n", name);
  printf("Test 41: Run API sanity checks: %s 41\n", name);

  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB mode for a given VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanID              @b{(input)}  VLAN identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbModeGet(openapiClientHandle_t *client_handle, uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiDot1cbModeGet(client_handle, vlanID, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB mode for a given VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("%s mode on VLAN %u get: %s (%u). (result = %d)\n", OPENAPI_TITLE, vlanID,
           (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the 802.1CB mode for a given VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanID              @b{(input)}  VLAN identifier
* @param    mode                @b{(input)}  802.1cb mode to be set
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbModeSet(openapiClientHandle_t *client_handle, uint32_t vlanID, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbModeSet(client_handle, vlanID, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1CB mode for a given VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("%s mode on VLAN %u set. (result = %d)\n", OPENAPI_TITLE, vlanID, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB ARP/NDP forwarding mode for a given VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanID              @b{(input)}  VLAN identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbForwardArpNdpModeGet(openapiClientHandle_t *client_handle, uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiDot1cbForwardArpNdpModeGet(client_handle, vlanID, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB ARP/NDP forwarding mode for a given VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("%s ARP/NDP forwarding mode on VLAN %u get: %s (%u). (result = %d)\n", OPENAPI_TITLE, vlanID,
           (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the 802.1CB ARP/NDP forwarding mode for a given VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanID              @b{(input)}  VLAN identifier
* @param    mode                @b{(input)}  802.1cb ARP/NDP forwarding mode to be set
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbForwardArpNdpModeSet(openapiClientHandle_t *client_handle, uint32_t vlanID, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbForwardArpNdpModeSet(client_handle, vlanID, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1CB ARP/NDP forwarding mode for a given VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("%s ARP/NDP forwarding mode set. (result = %d)\n", OPENAPI_TITLE, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the configured flow ID.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    macStr              @b{(input)}  static MAC address for the filter
* @param    vlanID              @b{(input)}  VLAN ID for the filter
*
* @returns  none
*
* @end
*********************************************************************/
void filterFlowIdGet(openapiClientHandle_t *client_handle, char *macStr, uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;
  uint32_t flowId = 0;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);

  if ((result = openapiFilterFlowIdGet(client_handle, &buffDesc, vlanID, &flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the configured flow ID. (result = %d)\n", result);
  }
  else
  {
    printf("Filter mac=%s vlan=%u flow ID get: (%u). (result = %d)\n", macStr, vlanID, flowId, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the configured flow type.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    macStr              @b{(input)}  static MAC address for the filter
* @param    vlanID              @b{(input)}  VLAN ID for the filter
*
* @returns  none
*
* @end
*********************************************************************/
void filterFlowTypeGet(openapiClientHandle_t *client_handle, char *macStr, uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;
  OPEN_DOT1CB_FLOW_TYPE_t flowType = OPEN_DOT1CB_FLOW_TYPE_NONE;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);

  if ((result = openapiFilterFlowTypeGet(client_handle, &buffDesc, vlanID, &flowType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the configured flow type. (result = %d)\n", result);
  }
  else
  {
    printf("Filter mac=%s vlan=%u flow type get: %s (%u). (result = %d)\n", macStr, vlanID,
           (flowType == OPEN_DOT1CB_FLOW_TYPE_INITIATOR)?"Initiator":"Transit", flowType, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the flow type and ID of an existing macfilter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    macStr              @b{(input)}  static MAC address for the filter
* @param    vlanID              @b{(input)}  VLAN ID for the filter
* @param    flowType            @b{(input)}  802.1cb flow type to be set
* @param    flowId              @b{(input)}  802.1cb flow ID to be set
*
* @returns  none
*
* @end
*********************************************************************/
void filterFlowTypeIdSet(openapiClientHandle_t *client_handle, char *macStr, uint32_t vlanID, OPEN_DOT1CB_FLOW_TYPE_t flowType, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);

  if ((result = openapiFilterFlowTypeIdSet(client_handle, &buffDesc, vlanID, flowType, flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the flow type and ID of an existing macfilter. (result = %d)\n", result);
  }
  else
  {
    printf("Filter mac=%s vlan=%u flow type and ID set. (result = %d)\n", macStr, vlanID, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB RTAG mode for the specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbRtagModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiDot1cbRtagModeGet(client_handle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB RTAG mode on interface. (result = %d)\n", result);
  }
  else
  {
    printf("%s RTAG mode on interface %u get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, ifNum, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the 802.1CB RTAG mode for the specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
* @param    mode                @b{(input)}  RTAG mode
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbRtagModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbRtagModeSet(client_handle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1CB RTAG mode on interface. (result = %d)\n", result);
  }
  else
  {
    printf("%s RTAG mode on interface %u set. (result = %d)\n", OPENAPI_TITLE, ifNum , result);
  }
  return;
}

/*********************************************************************
* @purpose  Clear all the 802.1CB statistics.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbStatsClear(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbStatsClear(client_handle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear all the 802.1CB statistics. (result = %d)\n", result);
  }
  else
  {
    printf("All %s statistics cleared. (result = %d)\n", OPENAPI_TITLE, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB global statistics.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbGlobalStatsGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_DOT1CB_GLOBAL_STATS_t stat;

  if ((result = openapiDot1cbGlobalStatsGet(client_handle, &stat)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB global statistics. (result = %d)\n", result);
  }
  else
  {
    printf("%s global statistics get: (%llu) (%llu) (%llu) (%llu) (result = %d)\n",
           OPENAPI_TITLE, stat.arp_to_cpu, stat.bcast_to_cpu, stat.mcast_to_cpu, stat.ucast_dropped, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB statistics for the specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIntfStatsGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_DOT1CB_INTF_STATS_t stat;

  if ((result = openapiDot1cbIntfStatsGet(client_handle, ifNum, &stat)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB statistics on interface. (result = %d)\n", result);
  }
  else
  {
    printf("%s statistics on interface %u get: (%llu) (%llu) (%llu) (%llu) (result = %d)\n",
           OPENAPI_TITLE, ifNum, stat.rtag_packets_rx, stat.rtag_packets_accepted,
           stat.rtag_packets_dups, stat.rtag_packets_tx, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next valid 802.1CB interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIntfNextGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t nextIfNum = 0;

  if ((result = openapiDot1cbIntfNextGet(client_handle, ifNum, &nextIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next valid 802.1CB interface. (result = %d)\n", result);
  }
  else
  {
    printf("Next valid %s interface get: %u. (result = %d)\n", OPENAPI_TITLE, nextIfNum, result);
  }
  return;
}

/*********************************************************************
* @purpose  Add an IPv4 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
* @param    flowType            @b{(input)}  flow type
* @param    flowId              @b{(input)}  flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv4ClassifierAdd(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen, OPEN_DOT1CB_FLOW_TYPE_t flowType, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IP address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv4ClassifierAdd(client_handle, &srcAddr, srcMaskLen,
                                               &dstAddr, dstMaskLen, flowType, flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add an IPv4 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("IPv4 classifier added: %s (src) %s (dst) (result = %d)\n", srcAddrStr, dstAddrStr, result);
  }
  return;
}

/*********************************************************************
* @purpose  Delete an IPv4 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv4ClassifierDelete(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IP address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv4ClassifierDelete(client_handle, &srcAddr, srcMaskLen,
                                                  &dstAddr, dstMaskLen)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete an IPv4 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("IPv4 classifier deleted: %s (src) %s (dst) (result = %d)\n", srcAddrStr, dstAddrStr, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get an IPv4 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv4ClassifierGet(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;
  OPEN_DOT1CB_FLOW_TYPE_t flowType = OPEN_DOT1CB_FLOW_TYPE_NONE;
  uint32_t flowId = 0;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IP address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv4ClassifierGet(client_handle, &srcAddr, srcMaskLen,
                                               &dstAddr, dstMaskLen, &flowType, &flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get an IPv4 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("IPv4 classifier flow type / ID get: %s (%u) / (%u) (result = %d)\n",
           (flowType == OPEN_DOT1CB_FLOW_TYPE_INITIATOR)?"Initiator":"Transit", flowType, flowId, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next IPv4 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv4ClassifierNextGet(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;
  OPEN_DOT1CB_FLOW_TYPE_t flowType = OPEN_DOT1CB_FLOW_TYPE_NONE;
  uint32_t flowId = 0;
  struct in_addr srcIpStr;
  struct in_addr dstIpStr;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IP address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv4ClassifierNextGet(client_handle, &srcAddr, &srcMaskLen,
                                                   &dstAddr, &dstMaskLen, &flowType, &flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next IPv4 classifier. (result = %d)\n", result);
  }
  else
  {
    srcIpStr.s_addr = htonl(srcAddr.addr.ipv4);
    dstIpStr.s_addr = htonl(dstAddr.addr.ipv4);
    printf("Next IPv4 classifier get: %s (src) %s (dst) (result = %d)\n",
           inet_ntoa(srcIpStr), inet_ntoa(dstIpStr), result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of configured IPv4 classifiers.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv4ClassifierCountGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbIpv4ClassifierCountGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of configured IPv4 classifiers. (result = %d)\n", result);
  }
  else
  {
    printf("Number of configured IPv4 classifiers get: %u (result = %d)\n", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the status flag indicating whether IPv4 classifiers are installed in the hardware.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv4ClassifierInHardwareGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbIpv4ClassifierInHardwareGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the status flag indicating whether IPv4 classifiers are installed in the hardware. (result = %d)\n", result);
  }
  else
  {
    printf("IPv4 classifiers in hardware flag get: %s (%u). (result = %d)\n",
           (value == 0)?"Not Installed":"Installed", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Add an IPv6 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
* @param    flowType            @b{(input)}  flow type
* @param    flowId              @b{(input)}  flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv6ClassifierAdd(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen, OPEN_DOT1CB_FLOW_TYPE_t flowType, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IPv6 address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IPv6 address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv6ClassifierAdd(client_handle, &(srcAddr.addr.ipv6), srcMaskLen,
                                               &(dstAddr.addr.ipv6), dstMaskLen, flowType, flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add an IPv6 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("IPv6 classifier added: %s (src) %s (dst) (result = %d)\n", srcAddrStr, dstAddrStr, result);
  }
  return;
}

/*********************************************************************
* @purpose  Delete an IPv6 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv6ClassifierDelete(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IPv6 address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IPv6 address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv6ClassifierDelete(client_handle, &(srcAddr.addr.ipv6), srcMaskLen,
                                                  &(dstAddr.addr.ipv6), dstMaskLen)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete an IPv6 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("IPv6 classifier deleted: %s (src) %s (dst) (result = %d)\n", srcAddrStr, dstAddrStr, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get an IPv6 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv6ClassifierGet(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;
  OPEN_DOT1CB_FLOW_TYPE_t flowType = OPEN_DOT1CB_FLOW_TYPE_NONE;
  uint32_t flowId = 0;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IPv6 address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IPv6 address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv6ClassifierGet(client_handle, &(srcAddr.addr.ipv6), srcMaskLen,
                                               &(dstAddr.addr.ipv6), dstMaskLen, &flowType, &flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get an IPv6 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("IPv6 classifier flow type / ID get: %s (%u) / (%u) (result = %d)\n",
           (flowType == OPEN_DOT1CB_FLOW_TYPE_INITIATOR)?"Initiator":"Transit", flowType, flowId, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next IPv6 classifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    srcAddrStr          @b{(input)}  source IP address
* @param    srcMaskLen          @b{(input)}  source IP mask length
* @param    dstAddrStr          @b{(input)}  destination IP address
* @param    dstMaskLen          @b{(input)}  destination IP mask length
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv6ClassifierNextGet(openapiClientHandle_t *client_handle, char *srcAddrStr, uint32_t srcMaskLen, char *dstAddrStr, uint32_t dstMaskLen)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t srcAddr;
  open_inet_addr_t dstAddr;
  open_buffdesc buffDesc;
  OPEN_DOT1CB_FLOW_TYPE_t flowType = OPEN_DOT1CB_FLOW_TYPE_NONE;
  uint32_t flowId = 0;

  buffDesc.pstart = srcAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &srcAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert source IPv6 address. (result = %d)\n", result);
    return;
  }

  buffDesc.pstart = dstAddrStr;
  buffDesc.size = OPEN_IP6_ADDR_LENGTH;
  if ((result = openapiInetAddrGet(client_handle, &buffDesc, &dstAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert destination IPv6 address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiDot1cbIpv6ClassifierNextGet(client_handle, &(srcAddr.addr.ipv6), &srcMaskLen,
                                                   &(dstAddr.addr.ipv6), &dstMaskLen, &flowType, &flowId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next IPv6 classifier. (result = %d)\n", result);
  }
  else
  {
    printf("Next IPv6 classifier get: %s (src) %s (dst) (result = %d)\n", srcAddrStr, dstAddrStr, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of configured IPv6 classifiers.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv6ClassifierCountGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbIpv6ClassifierCountGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of configured IPv6 classifiers. (result = %d)\n", result);
  }
  else
  {
    printf("Number of configured IPv6 classifiers get: %u (result = %d)\n", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the status flag indicating whether IPv6 classifiers are installed in the hardware.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbIpv6ClassifierInHardwareGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbIpv6ClassifierInHardwareGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the status flag indicating whether IPv6 classifiers are installed in the hardware. (result = %d)\n", result);
  }
  else
  {
    printf("IPv6 classifiers in hardware flag get: %s (%u). (result = %d)\n",
           (value == 0)?"Not Installed":"Installed", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of transit flows installed in the hardware.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbNumActiveTransitFlowsGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbNumActiveTransitFlowsGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of transit flows installed in the hardware. (result = %d)\n", result);
  }
  else
  {
    printf("Number of transit flows installed in the hardware get: %u (result = %d)\n", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of initiator flows installed in the hardware.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbNumActiveInititatorFlowsGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbNumActiveInititatorFlowsGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of initiator flows installed in the hardware. (result = %d)\n", result);
  }
  else
  {
    printf("Number of initiator flows installed in the hardware get: %u (result = %d)\n", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of MFDB entries associated with flows.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbNumMfdbClassifiersGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbNumMfdbClassifiersGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of MFDB entries associated with flows. (result = %d)\n", result);
  }
  else
  {
    printf("Number of MFDB entries associated with flows get: %u (result = %d)\n", value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of 802.1CB enabled VLANs.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiNumDot1cbVlansGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t value = 0;

  if ((result = openapiDot1cbApiNumDot1cbVlansGet(client_handle, &value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of 802.1CB enabled VLANs. (result = %d)\n", result);
  }
  else
  {
    printf("Number of %s enabled VLANs get: %u (result = %d)\n", OPENAPI_TITLE, value, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next 802.1CB enabled VLAN.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vid                 @b{(input)} VLAN identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiDot1cbEnabledVlanNextGet(openapiClientHandle_t *client_handle, uint32_t vid)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbApiDot1cbEnabledVlanNextGet(client_handle, &vid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next 802.1CB enabled VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Next %s enabled VLAN get: %u. (result = %d)\n", OPENAPI_TITLE, vid , result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB initiator flow and its status/statistics.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  initiator flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiInitFlowGet(openapiClientHandle_t *client_handle, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t classifiers = 0;
  uint32_t nextTxSequence = 0;
  uint64_t txPackets = 0;

  if ((result = openapiDot1cbApiInitFlowGet(client_handle, flowId, &classifiers, &nextTxSequence, &txPackets)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB initiator flow and its status/statistics. (result = %d)\n", result);
  }
  else
  {
    printf("%s initiator flow get: %u (flowID) %u (classifiers) %u (nextTxSequence) (result = %d)\n",
           OPENAPI_TITLE, flowId, classifiers, nextTxSequence, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next 802.1CB initiator flow and its status/statistics.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  initiator flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiInitFlowNextGet(openapiClientHandle_t *client_handle, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t classifiers = 0;
  uint32_t nextTxSequence = 0;
  uint64_t txPackets = 0;

  if ((result = openapiDot1cbApiInitFlowNextGet(client_handle, &flowId, &classifiers, &nextTxSequence, &txPackets)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next 802.1CB initiator flow and its status/statistics. (result = %d)\n", result);
  }
  else
  {
    printf("Next %s initiator flow get: %u (flowID) %u (classifiers) %u (nextTxSequence) (result = %d)\n",
           OPENAPI_TITLE, flowId, classifiers, nextTxSequence, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB transit flow and its status/statistics.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  transit flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiTransitFlowGet(openapiClientHandle_t *client_handle, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t classifiers = 0;
  uint32_t lastRxSequence = 0;
  uint64_t matchingPackets = 0;
  uint64_t acceptedPackets = 0;
  uint64_t duplicatePackets = 0;

  if ((result = openapiDot1cbApiTransitFlowGet(client_handle, flowId, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB transit flow and its status/statistics. (result = %d)\n", result);
  }
  else
  {
    printf("%s transit flow get: %u (flowID) %u (classifiers) %u (lastRxSequence) (result = %d)\n",
           OPENAPI_TITLE, flowId, classifiers, lastRxSequence, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next 802.1CB transit flow and its status/statistics.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  transit flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiTransitFlowNextGet(openapiClientHandle_t *client_handle, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t classifiers = 0;
  uint32_t lastRxSequence = 0;
  uint64_t matchingPackets = 0;
  uint64_t acceptedPackets = 0;
  uint64_t duplicatePackets = 0;

  if ((result = openapiDot1cbApiTransitFlowNextGet(client_handle, &flowId, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next 802.1CB transit flow and its status/statistics. (result = %d)\n", result);
  }
  else
  {
    printf("Next %s transit flow get: %u (flowID) %u (classifiers) %u (lastRxSequence) (result = %d)\n",
           OPENAPI_TITLE, flowId, classifiers, lastRxSequence, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the 802.1CB transit flow configuration.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  transit flow identifier
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiTransitFlowConfigGet(openapiClientHandle_t *client_handle, uint32_t flowId)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t historyWindow = 0;
  uint32_t acceptanceWindow = 0;
  uint32_t ageTime = 0;

  if ((result = openapiDot1cbApiTransitFlowConfigGet(client_handle, flowId, &historyWindow, &acceptanceWindow, &ageTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1CB transit flow configuration. (result = %d)\n", result);
  }
  else
  {
    printf("%s transit flow configuration get: %u (flowID) %u (historyWindow) %u (acceptanceWindow) %u (ageTime) (result = %d)\n",
           OPENAPI_TITLE, flowId, historyWindow, acceptanceWindow, ageTime, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the history window size for an 802.1CB transit flow.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  transit flow identifier
* @param    historyWindow       @b{(input)}  history window size in bytes
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiTransitFlowHistoryWindowSizeSet(openapiClientHandle_t *client_handle, uint32_t flowId, uint32_t historyWindow)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbApiTransitFlowHistoryWindowSizeSet(client_handle, flowId, historyWindow)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the history window size for an 802.1CB transit flow. (result = %d)\n", result);
  }
  else
  {
    printf("%s transit flow history window size set: %u (flowID) (result = %d)\n", OPENAPI_TITLE, flowId, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the acceptance window size for an 802.1CB transit flow.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  transit flow identifier
* @param    acceptanceWindow    @b{(input)}  acceptance window size in bytes
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiTransitFlowAcceptanceWindowSizeSet(openapiClientHandle_t *client_handle, uint32_t flowId, uint32_t acceptanceWindow)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbApiTransitFlowAcceptanceWindowSizeSet(client_handle, flowId, acceptanceWindow)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the acceptance window size for an 802.1CB transit flow. (result = %d)\n", result);
  }
  else
  {
    printf("%s transit flow acceptance window size set: %u (flowID) (result = %d)\n", OPENAPI_TITLE, flowId, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the age time for an 802.1CB transit flow.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flowId              @b{(input)}  transit flow identifier
* @param    ageTime             @b{(input)}  age time in 10ms ticks
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbApiTransitFlowAgeTimeSet(openapiClientHandle_t *client_handle, uint32_t flowId, uint32_t ageTime)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbApiTransitFlowAgeTimeSet(client_handle, flowId, ageTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the age time for an 802.1CB transit flow. (result = %d)\n", result);
  }
  else
  {
    printf("%s transit flow age time set: %u (flowID) (result = %d)\n", OPENAPI_TITLE, flowId, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the SA-MAC/VLAN flow association mode.

* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbSaModeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  if ((result = openapiDot1cbSaModeGet(client_handle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the SA-MAC/VLAN flow association mode. (result = %d)\n", result);
  }
  else
  {
    printf("SA-MAC/VLAN flow association mode get: %s (%u). (result = %d)\n",
           (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set the SA-MAC/VLAN flow association mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode                @b{(input)}  SA-MAC/VLAN flow association mode to be set
*
* @returns  none
*
* @end
*********************************************************************/
void dot1cbSaModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiDot1cbSaModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the SA-MAC/VLAN flow association mode. (result = %d)\n", result);
  }
  else
  {
    printf("SA-MAC/VLAN flow association mode set. (result = %d)\n", result);
  }
  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle   @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
void runSanity(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t vid;
  uint64_t txPackets;
  uint32_t vlanID = 1;
  open_inet_addr_t dstAddr;
  open_in6_addr_t dstAddr6;
  uint32_t nextIfNum;
  uint32_t dstMaskLen;
  uint32_t ageTime;
  uint64_t matchingPackets;
  uint32_t historyWindow;
  OPEN_DOT1CB_FLOW_TYPE_t flowType;
  OPEN_DOT1CB_GLOBAL_STATS_t stat;
  OPEN_DOT1CB_INTF_STATS_t intfStat;
  uint32_t srcMaskLen;
  uint64_t duplicatePackets;
  uint32_t ifNum = 1;
  uint64_t acceptedPackets;
  uint32_t classifiers;
  uint32_t value;
  open_inet_addr_t srcAddr;
  open_in6_addr_t srcAddr6;
  uint32_t flowId;
  OPEN_CONTROL_t mode;
  uint32_t nextTxSequence;
  uint32_t acceptanceWindow;
  uint32_t lastRxSequence;
  char buffer[128];
  open_buffdesc buf;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  open_buffdesc badBufdescLen;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescLen.pstart = buffer;
  badBufdescPointer.size = 0;
  buf.pstart = buffer;
  buf.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing Dot1cb OpEN APIs sanity:\n");

  printf("Testing openapiDot1cbModeGet():\n");
  result = openapiDot1cbModeGet(NULL, vlanID, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbModeGet(client_handle, vlanID, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1cbForwardArpNdpModeGet():\n");
  result = openapiDot1cbForwardArpNdpModeGet(NULL, vlanID, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbForwardArpNdpModeGet(client_handle, vlanID, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiFilterFlowIdGet():\n");
  result = openapiFilterFlowIdGet(NULL, &buf, vlanID, &flowId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiFilterFlowIdGet(client_handle, NULL, vlanID, &flowId);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiFilterFlowIdGet(client_handle, &badBufdescPointer, vlanID, &flowId);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiFilterFlowIdGet(client_handle, &zeroLenBuf, vlanID, &flowId);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);
  result = openapiFilterFlowIdGet(client_handle, &buf, vlanID, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiFilterFlowTypeGet():\n");
  result = openapiFilterFlowTypeGet(NULL, &buf, vlanID, &flowType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiFilterFlowTypeGet(client_handle, NULL, vlanID, &flowType);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiFilterFlowTypeGet(client_handle, &badBufdescPointer, vlanID, &flowType);
  printf("NULL buffdesc pstart argument 2:(result = %d)\n", result);
  result = openapiFilterFlowTypeGet(client_handle, &zeroLenBuf, vlanID, &flowType);
  printf("buffdesc 0 length size argument 2:(result = %d)\n", result);
  result = openapiFilterFlowTypeGet(client_handle, &buf, vlanID, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDot1cbRtagModeGet():\n");
  result = openapiDot1cbRtagModeGet(NULL, ifNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbRtagModeGet(client_handle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1cbStatsClear():\n");
  result = openapiDot1cbStatsClear(NULL);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1cbGlobalStatsGet():\n");
  result = openapiDot1cbGlobalStatsGet(NULL, &stat);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbGlobalStatsGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbIntfStatsGet():\n");
  result = openapiDot1cbIntfStatsGet(NULL, ifNum, &intfStat);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIntfStatsGet(client_handle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1cbIntfNextGet():\n");
  result = openapiDot1cbIntfNextGet(NULL, ifNum, &nextIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIntfNextGet(client_handle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv4ClassifierGet():\n");
  result = openapiDot1cbIpv4ClassifierGet(NULL, &srcAddr, srcMaskLen, &dstAddr, dstMaskLen, &flowType, &flowId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierGet(client_handle, NULL, srcMaskLen, &dstAddr, dstMaskLen, &flowType, &flowId);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierGet(client_handle, &srcAddr, srcMaskLen, NULL, dstMaskLen, &flowType, &flowId);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierGet(client_handle, &srcAddr, srcMaskLen, &dstAddr, dstMaskLen, NULL, &flowId);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierGet(client_handle, &srcAddr, srcMaskLen, &dstAddr, dstMaskLen, &flowType, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv4ClassifierNextGet():\n");
  result = openapiDot1cbIpv4ClassifierNextGet(NULL, &srcAddr, &srcMaskLen, &dstAddr, &dstMaskLen, &flowType, &flowId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierNextGet(client_handle, NULL, &srcMaskLen, &dstAddr, &dstMaskLen, &flowType, &flowId);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierNextGet(client_handle, &srcAddr, NULL, &dstAddr, &dstMaskLen, &flowType, &flowId);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierNextGet(client_handle, &srcAddr, &srcMaskLen, NULL, &dstMaskLen, &flowType, &flowId);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierNextGet(client_handle, &srcAddr, &srcMaskLen, &dstAddr, NULL, &flowType, &flowId);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierNextGet(client_handle, &srcAddr, &srcMaskLen, &dstAddr, &dstMaskLen, NULL, &flowId);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierNextGet(client_handle, &srcAddr, &srcMaskLen, &dstAddr, &dstMaskLen, &flowType, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv4ClassifierCountGet():\n");
  result = openapiDot1cbIpv4ClassifierCountGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierCountGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv4ClassifierInHardwareGet():\n");
  result = openapiDot1cbIpv4ClassifierInHardwareGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv4ClassifierInHardwareGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv6ClassifierGet():\n");
  result = openapiDot1cbIpv6ClassifierGet(NULL, &srcAddr6, srcMaskLen, &dstAddr6, dstMaskLen, &flowType, &flowId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierGet(client_handle, NULL, srcMaskLen, &dstAddr6, dstMaskLen, &flowType, &flowId);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierGet(client_handle, &srcAddr6, srcMaskLen, NULL, dstMaskLen, &flowType, &flowId);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierGet(client_handle, &srcAddr6, srcMaskLen, &dstAddr6, dstMaskLen, NULL, &flowId);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierGet(client_handle, &srcAddr6, srcMaskLen, &dstAddr6, dstMaskLen, &flowType, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv6ClassifierNextGet():\n");
  result = openapiDot1cbIpv6ClassifierNextGet(NULL, &srcAddr6, &srcMaskLen, &dstAddr6, &dstMaskLen, &flowType, &flowId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierNextGet(client_handle, NULL, &srcMaskLen, &dstAddr6, &dstMaskLen, &flowType, &flowId);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierNextGet(client_handle, &srcAddr6, NULL, &dstAddr6, &dstMaskLen, &flowType, &flowId);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierNextGet(client_handle, &srcAddr6, &srcMaskLen, NULL, &dstMaskLen, &flowType, &flowId);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierNextGet(client_handle, &srcAddr6, &srcMaskLen, &dstAddr6, NULL, &flowType, &flowId);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierNextGet(client_handle, &srcAddr6, &srcMaskLen, &dstAddr6, &dstMaskLen, NULL, &flowId);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierNextGet(client_handle, &srcAddr6, &srcMaskLen, &dstAddr6, &dstMaskLen, &flowType, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv6ClassifierCountGet():\n");
  result = openapiDot1cbIpv6ClassifierCountGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierCountGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbIpv6ClassifierInHardwareGet():\n");
  result = openapiDot1cbIpv6ClassifierInHardwareGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbIpv6ClassifierInHardwareGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbNumActiveTransitFlowsGet():\n");
  result = openapiDot1cbNumActiveTransitFlowsGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbNumActiveTransitFlowsGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbNumActiveInititatorFlowsGet():\n");
  result = openapiDot1cbNumActiveInititatorFlowsGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbNumActiveInititatorFlowsGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbNumMfdbClassifiersGet():\n");
  result = openapiDot1cbNumMfdbClassifiersGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbNumMfdbClassifiersGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiNumDot1cbVlansGet():\n");
  result = openapiDot1cbApiNumDot1cbVlansGet(NULL, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiNumDot1cbVlansGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiDot1cbEnabledVlanNextGet():\n");
  result = openapiDot1cbApiDot1cbEnabledVlanNextGet(NULL, &vid);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiDot1cbEnabledVlanNextGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiInitFlowGet():\n");
  result = openapiDot1cbApiInitFlowGet(NULL, flowId, &classifiers, &nextTxSequence, &txPackets);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowGet(client_handle, flowId, NULL, &nextTxSequence, &txPackets);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowGet(client_handle, flowId, &classifiers, NULL, &txPackets);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowGet(client_handle, flowId, &classifiers, &nextTxSequence, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiInitFlowNextGet():\n");
  result = openapiDot1cbApiInitFlowNextGet(NULL, &flowId, &classifiers, &nextTxSequence, &txPackets);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowNextGet(client_handle, NULL, &classifiers, &nextTxSequence, &txPackets);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowNextGet(client_handle, &flowId, NULL, &nextTxSequence, &txPackets);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowNextGet(client_handle, &flowId, &classifiers, NULL, &txPackets);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbApiInitFlowNextGet(client_handle, &flowId, &classifiers, &nextTxSequence, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiTransitFlowGet():\n");
  result = openapiDot1cbApiTransitFlowGet(NULL, flowId, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowGet(client_handle, flowId, NULL, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowGet(client_handle, flowId, &classifiers, NULL, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowGet(client_handle, flowId, &classifiers, &lastRxSequence, NULL, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowGet(client_handle, flowId, &classifiers, &lastRxSequence, &matchingPackets, NULL, &duplicatePackets);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowGet(client_handle, flowId, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiTransitFlowNextGet():\n");
  result = openapiDot1cbApiTransitFlowNextGet(NULL, &flowId, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowNextGet(client_handle, NULL, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 2:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowNextGet(client_handle, &flowId, NULL, &lastRxSequence, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowNextGet(client_handle, &flowId, &classifiers, NULL, &matchingPackets, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowNextGet(client_handle, &flowId, &classifiers, &lastRxSequence, NULL, &acceptedPackets, &duplicatePackets);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowNextGet(client_handle, &flowId, &classifiers, &lastRxSequence, &matchingPackets, NULL, &duplicatePackets);
  printf("NULL argument 6:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowNextGet(client_handle, &flowId, &classifiers, &lastRxSequence, &matchingPackets, &acceptedPackets, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDot1cbApiTransitFlowConfigGet():\n");
  result = openapiDot1cbApiTransitFlowConfigGet(NULL, flowId, &historyWindow, &acceptanceWindow, &ageTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowConfigGet(client_handle, flowId, NULL, &acceptanceWindow, &ageTime);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowConfigGet(client_handle, flowId, &historyWindow, NULL, &ageTime);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiDot1cbApiTransitFlowConfigGet(client_handle, flowId, &historyWindow, &acceptanceWindow, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDot1cbSaModeGet():\n");
  result = openapiDot1cbSaModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDot1cbSaModeGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Dot1cb
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Dot1cb example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Dot1cb API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        dot1cbModeGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
        dot1cbModeSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        dot1cbForwardArpNdpModeGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        dot1cbForwardArpNdpModeSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        filterFlowIdGet(&clientHandle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
        filterFlowTypeGet(&clientHandle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 6)
      {
        filterFlowTypeIdSet(&clientHandle, argv[2], atoi(argv[3]), atoi(argv[4]), atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        dot1cbRtagModeGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 4)
      {
        dot1cbRtagModeSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        dot1cbStatsClear(&clientHandle);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 2)
      {
        dot1cbGlobalStatsGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        dot1cbIntfStatsGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
        dot1cbIntfNextGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 8)
      {
        dot1cbIpv4ClassifierAdd(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]), atoi(argv[6]), atoi(argv[7]));
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 6)
      {
        dot1cbIpv4ClassifierDelete(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 6)
      {
        dot1cbIpv4ClassifierGet(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 6)
      {
        dot1cbIpv4ClassifierNextGet(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 2)
      {
        dot1cbIpv4ClassifierCountGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 2)
      {
        dot1cbIpv4ClassifierInHardwareGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 8)
      {
        dot1cbIpv6ClassifierAdd(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]), atoi(argv[6]), atoi(argv[7]));
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 6)
      {
        dot1cbIpv6ClassifierDelete(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 6)
      {
        dot1cbIpv6ClassifierGet(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 6)
      {
        dot1cbIpv6ClassifierNextGet(&clientHandle, argv[2], atoi(argv[3]), argv[4], atoi(argv[5]));
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 2)
      {
        dot1cbIpv6ClassifierCountGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 2)
      {
        dot1cbIpv6ClassifierInHardwareGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 2)
      {
        dot1cbNumActiveTransitFlowsGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 2)
      {
        dot1cbNumActiveInititatorFlowsGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 2)
      {
        dot1cbNumMfdbClassifiersGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 2)
      {
        dot1cbApiNumDot1cbVlansGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 3)
      {
        dot1cbApiDot1cbEnabledVlanNextGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 3)
      {
        dot1cbApiInitFlowGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 3)
      {
        dot1cbApiInitFlowNextGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 3)
      {
        dot1cbApiTransitFlowGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 34:
      if (argc == 3)
      {
        dot1cbApiTransitFlowNextGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 35:
      if (argc == 3)
      {
        dot1cbApiTransitFlowConfigGet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 36:
      if (argc == 4)
      {
        dot1cbApiTransitFlowHistoryWindowSizeSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 37:
      if (argc == 4)
      {
        dot1cbApiTransitFlowAcceptanceWindowSizeSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 38:
      if (argc == 4)
      {
        dot1cbApiTransitFlowAgeTimeSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 39:
      if (argc == 2)
      {
        dot1cbSaModeGet(&clientHandle);
        show_help = 0;
      }
      break;
    case 40:
      if (argc == 3)
      {
        dot1cbSaModeSet(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 41:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Dot1cb API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

