/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  authmgr_example.c
*
* @purpose   OpEN authmgr example.
*
* @component OpEN
*
* @create    02/23/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_authmgr.h"
#include "openapi_dot1x.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

#define DISP_BUF_SIZE 1024

/*******************************************************************
*
* @brief  This function prints the authmgr Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Set the authentication manager Dynamic Vlan Creation Mode: %s 1 <dynamicVlanMode>\n", name);
  printf("Test 2: Set the authentication manager Vlan Assignment Mode: %s 2 <vlanAssignmentMode>\n", name);
  printf("Test 3: Set authentication manager method order: %s 3 <intIfNum><index><method>\n", name);
  printf("Test 4: Set authentication manager method priority: %s 4 <intIfNum><index><method>\n", name);
  printf("Test 5: Set the max users for the port: %s 5 <intIfNum><maxUsers>\n", name);
  printf("Test 6: Set the authentication fail action (Unauthenticated vlan) value: %s 6 <intIfNum><vlanId>\n", name);
  printf("Test 7: Display the authentication interface params: %s 7 <intIfNum>\n", name);
  printf("Test 8: Display the authenticated clients info: %s 8 <intIfNum>\n", name);
  printf("Test 9: Set authentication port configured control mode: %s 9 <intIfNum> <mode>\n", name);
  printf("Test 10: Set authentication manager admin mode: %s 10 <mode>\n", name);
  printf("Test 11: Set authentication manager host mode: %s 11 <intIfNum> <mode>\n", name);
  printf("Test 12: Set authentication manager port guest VLAN ID: %s 12 <intIfNum> <vlan>\n", name);
  printf("Test 13: Set authentication manager authentication periodic: %s 13 <intIfNum> <mode>\n", name);
  printf("Test 14: Set authentication manager reauthentication periodic: %s 14 <intIfNum> <mode>\n", name);
  printf("Test 15: Set authentication manager reauth period from server: %s 15 <intIfNum> <mode>\n", name);
  printf("Test 16: Set the max number of failed authentication attempts value on the specified port: %s 16 <intIfNum><maxAuthAttempts>\n", name);
  printf("Test 17: Run API sanity checks: %s 17 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  OPEN_AUTHMGR_STATES_t authState;
  uint32_t index = 1;
  uint32_t vlanType;
  uint32_t timeLeft;
  uint32_t clientsCount;
  uint32_t terminationAction;
  OPEN_AUTHMGR_PORT_STATUS_t authStatus;
  uint32_t vlanId;
  uint32_t maxAuthAttempts;
  uint32_t reAuthPeriod;
  uint32_t sessionTimeout;
  uint32_t sessionTime;
  OPEN_CONTROL_t dynamicVlanMode;
  OPEN_CONTROL_t reauthState;
  uint32_t maxUsers;
  OPEN_CONTROL_t vlanAssignmentMode;
  OPEN_AUTHMGR_METHOD_t method;
  uint32_t intIfNum = 100;
  uint32_t lIntIfNum;
  char buffer[128];
  open_buffdesc buf;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  buf.pstart = buffer;
  buf.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing authmgr OpEN APIs sanity:\n");

  printf("Testing openapiAuthMgrDynamicVlanCreationModeGet():\n");
  result = openapiAuthMgrDynamicVlanCreationModeGet(NULL, &dynamicVlanMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrDynamicVlanCreationModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiAuthMgrVlanAssignmentModeGet():\n");
  result = openapiAuthMgrVlanAssignmentModeGet(NULL, &vlanAssignmentMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrVlanAssignmentModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiAuthMgrPortAuthMethodOrderGet():\n");
  result = openapiAuthMgrPortAuthMethodOrderGet(NULL, intIfNum, index, &method);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrPortAuthMethodOrderGet(clientHandle, intIfNum, index, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiAuthMgrPortAuthMethodPriorityGet():\n");
  result = openapiAuthMgrPortAuthMethodPriorityGet(NULL, intIfNum, index, &method);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrPortAuthMethodPriorityGet(clientHandle, intIfNum, index, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiAuthMgrPortMaxUsersGet():\n");
  result = openapiAuthMgrPortMaxUsersGet(NULL, intIfNum, &maxUsers);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrPortMaxUsersGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrPortUnauthenticatedVlanGet():\n");
  result = openapiAuthMgrPortUnauthenticatedVlanGet(NULL, intIfNum, &vlanId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrPortUnauthenticatedVlanGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrPortMaxAuthAttemptsGet():\n");
  result = openapiAuthMgrPortMaxAuthAttemptsGet(NULL, intIfNum, &maxAuthAttempts);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrPortMaxAuthAttemptsGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrEnabledMethodGet():\n");
  result = openapiAuthMgrEnabledMethodGet(NULL, intIfNum, index, &method);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrEnabledMethodGet(clientHandle, intIfNum, index, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiAuthMgrEnabledPriorityGet():\n");
  result = openapiAuthMgrEnabledPriorityGet(NULL, intIfNum, index, &method);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrEnabledPriorityGet(clientHandle, intIfNum, index, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiAuthMgrNumClientsGet():\n");
  result = openapiAuthMgrNumClientsGet(NULL, intIfNum, &clientsCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrNumClientsGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortClientMacAddrGet():\n");
  result = openapiAuthMgrLogicalPortClientMacAddrGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientMacAddrGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientMacAddrGet(clientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientMacAddrGet(clientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortClientAuthStatusGet():\n");
  result = openapiAuthMgrLogicalPortClientAuthStatusGet(NULL, intIfNum, &authStatus);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientAuthStatusGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortClientAuthStateGet():\n");
  result = openapiAuthMgrLogicalPortClientAuthStateGet(NULL, intIfNum, &authState);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientAuthStateGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortClientReauthStateGet():\n");
  result = openapiAuthMgrLogicalPortClientReauthStateGet(NULL, intIfNum, &reauthState);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientReauthStateGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortUserNameGet():\n");
  result = openapiAuthMgrLogicalPortUserNameGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortUserNameGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortUserNameGet(clientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortUserNameGet(clientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortClientAuthenticatedMethodGet():\n");
  result = openapiAuthMgrLogicalPortClientAuthenticatedMethodGet(NULL, intIfNum, &method);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientAuthenticatedMethodGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortClientVlanTypeGet():\n");
  result = openapiAuthMgrLogicalPortClientVlanTypeGet(NULL, intIfNum, &vlanType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortClientVlanTypeGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortSessionTimeGet():\n");
  result = openapiAuthMgrLogicalPortSessionTimeGet(NULL, intIfNum, &sessionTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortSessionTimeGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortSessionTimeoutGet():\n");
  result = openapiAuthMgrLogicalPortSessionTimeoutGet(NULL, intIfNum, &sessionTimeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortSessionTimeoutGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthmgrLogicalPortTimeLeftForTerminateActionGet():\n");
  result = openapiAuthmgrLogicalPortTimeLeftForTerminateActionGet(NULL, intIfNum, &timeLeft);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthmgrLogicalPortTimeLeftForTerminateActionGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortTerminationActionGet():\n");
  result = openapiAuthMgrLogicalPortTerminationActionGet(NULL, intIfNum, &terminationAction);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortTerminationActionGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortFilterIdGet():\n");
  result = openapiAuthMgrLogicalPortFilterIdGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortFilterIdGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortFilterIdGet(clientHandle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortFilterIdGet(clientHandle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrAuthenticationReAuthPeriodFromSeverGet():\n");
  result = openapiAuthMgrAuthenticationReAuthPeriodFromSeverGet(NULL, intIfNum, &reAuthPeriod);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrAuthenticationReAuthPeriodFromSeverGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrAuthenticationReAuthPeriodGet():\n");
  result = openapiAuthMgrAuthenticationReAuthPeriodGet(NULL, intIfNum, &reAuthPeriod);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrAuthenticationReAuthPeriodGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortFirstGet():\n");
  result = openapiAuthMgrLogicalPortFirstGet(NULL, intIfNum, &lIntIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortFirstGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiAuthMgrLogicalPortNextGet():\n");
  result = openapiAuthMgrLogicalPortNextGet(NULL, lIntIfNum, &intIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAuthMgrLogicalPortNextGet(clientHandle, lIntIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  return;
}

/*********************************************************************
* @purpose  Set the authentication manager Dynamic Vlan Creation Mode
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    dynamicVlanMode       @b{(input)}  radius dynamic vlan creation mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrDynamicVlanCreationModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t dynamicVlanMode)
{
  open_error_t result;

  if ((result = openapiAuthMgrDynamicVlanCreationModeSet(clientHandle, dynamicVlanMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the authentication manager Dynamic Vlan Creation Mode (result = %d)\n", result);
  }
  else
  {
    printf("Authentication manager Dynamic Vlan Creation Mode is successfully %s\n", (OPEN_ENABLE == dynamicVlanMode) ? "enabled" : "disabled");
  }
  return;
}

/*********************************************************************
* @purpose  Set the authentication manager Vlan Assignment Mode
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    vlanAssignmentMode       @b{(input)}  radius vlan assignment mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrVlanAssignmentModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t vlanAssignmentMode)
{
  open_error_t result;

  if ((result = openapiAuthMgrVlanAssignmentModeSet(clientHandle, vlanAssignmentMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the authentication manager Vlan Assignment Mode (result = %d)\n", result);
  }
  else
  {
    printf("Authentication manager Vlan Assignment Mode is successfully %s\n", (OPEN_ENABLE == vlanAssignmentMode) ? "enabled" : "disabled");
  }
  return;
}

void authMgrPortAuthMethodOrderSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t method)
{
  open_error_t result;

  if ((result = openapiAuthMgrPortAuthMethodOrderSet(clientHandle, intIfNum, index, method)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set authentication manager method order (result = %d)\n", result);
  }
  else
  {
    printf("Authentication manager method order successfully set.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set authentication manager method priority
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    index       @b{(input)}  position of the method or order
* @param    method       @b{(input)}  authentication manager methods, i,e. dot1x/mab/cp

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrPortAuthMethodPrioritySet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t method)
{
  open_error_t result;

  if ((result = openapiAuthMgrPortAuthMethodPrioritySet(clientHandle, intIfNum, index, method)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set authentication manager method priority (result = %d)\n", result);
  }
  else
  {
    printf("Authentication manager method priority successfully set.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Set the max users for the port
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    maxUsers       @b{(input)}  max users per port

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrPortMaxUsersSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t maxUsers)
{
  open_error_t result;

  if ((result = openapiAuthMgrPortMaxUsersSet(clientHandle, intIfNum, maxUsers)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the max users for the port (result = %d)\n", result);
  }
  else
  {
    printf("Setting of max users per port is succeeded. \n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the Authentication fail action (Unauthenticated vlan) value
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    vlanId       @b{(input)}  Unauthenticated vlan

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrPortUnauthenticatedVlanSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t vlanId)
{
  open_error_t result;

  if ((result = openapiAuthMgrPortUnauthenticatedVlanSet(clientHandle, intIfNum, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Authentication fail action (Unauthenticated vlan) value (result = %d)\n", result);
  }
  else
  {
    printf("Authentication fail action (Unauthenticated vlan) value successfully set.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the max number of failed authentication attempts value on the specified port
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number
* @param    maxAuthAttempts       @b{(input)}  value for the max authentication failed attempts

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrPortMaxAuthAttemptsSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t maxAuthAttempts)
{
  open_error_t result;

  if ((result = openapiAuthMgrPortMaxAuthAttemptsSet(clientHandle, intIfNum, maxAuthAttempts)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the max number of failed authentication attempts value on the specified port (result = %d)\n", result);
  }
  else
  {
    printf("Max number of failed authentication attempts value successfully set.\n");
  }
  return;
}

/*********************************************************************
* @purpose  display the specific client information  
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    lIntIfNum       @b{(input)} Client's internal Mapping number 

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrAuthenticationClientsShow(openapiClientHandle_t *client_handle, uint32_t lIntIfNum)
{
  char buf[100];
  char strFilter[256];
  char userName[OPEN_AUTHMGR_USER_NAME_LEN];
  open_buffdesc buffDesc, filterId;
  OPEN_AUTHMGR_VLAN_ASSIGNED_MODE_t vlanReason;
  uint32_t vlanId;
  uint32_t authStatus = 0, authState = 0, sessionTime = 0;
  OPEN_CONTROL_t reAuthState;
  OPEN_AUTHMGR_METHOD_t method = 0;
  uint32_t sessionTimeout = 0, timeLeft = 0, terminationAction = 0;

  memset(buf, '\0', sizeof(buf));
  buffDesc.pstart = buf;
  buffDesc.size = sizeof(buf);

  memset(strFilter, '\0', sizeof(strFilter));
  filterId.pstart = strFilter;
  filterId.size = sizeof(strFilter);

  /* mac-address */
  if (OPEN_E_NONE != openapiAuthMgrLogicalPortClientMacAddrGet(client_handle, lIntIfNum, &buffDesc))
  {
    printf ("Failed to get mac address for logical port %d \r\n", lIntIfNum);
    return;
  }

  printf ("\r\n");
  printf ("Mac address : %02X:%02X:%02X:%02X:%02X:%02X\r\n",
      buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
  /* auth status */

  if (OPEN_E_NONE != openapiAuthMgrLogicalPortClientAuthStatusGet(client_handle, lIntIfNum, &authStatus))
  {
    printf ("Failed to get username for logical port %d \r\n", lIntIfNum);
    return;
  }

  if (authStatus == OPEN_AUTHMGR_PORT_STATUS_AUTHORIZED)
  {

    /* user-name */

    memset(userName, '\0', sizeof(userName));
    buffDesc.pstart = userName;
    buffDesc.size = sizeof(userName);

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortUserNameGet(client_handle, lIntIfNum, &buffDesc))
    {
      printf ("Failed to get username for logical port %d \r\n", lIntIfNum);
      return;
    }

    printf("Username : %s\r\n", userName);


    /* auth state */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortClientAuthStateGet(client_handle, lIntIfNum, &authState))
    {
      printf ("Failed to get auth state for logical port %d \r\n", lIntIfNum);
      return;
    }

    printf ("Auth State : %s\r\n", (OPEN_AUTHMGR_AUTHENTICATED == authState) ? "Authenticated" : "In progress");

    /* client reauth state */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortClientReauthStateGet(client_handle, lIntIfNum, &reAuthState))
    {
      printf ("Failed to get reAuthState for logical port %d \r\n", lIntIfNum);
      return;
    }

    printf("Reauth State : %s \r\n", (reAuthState == OPEN_ENABLE) ? "True" : "False");

    /* client authenticated method */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortClientAuthenticatedMethodGet(client_handle, lIntIfNum, &method))
    {
      printf ("Failed to get authenticated method for logical port %d \r\n", lIntIfNum);
      return;
    }

    printf("Authenticated Method : %s \r\n", (method == OPEN_AUTHMGR_METHOD_8021X) ? "802.1X" : ((method == OPEN_AUTHMGR_METHOD_MAB) ? "MAB" : ((method == OPEN_AUTHMGR_METHOD_CP) ? "Captive-portal" : "None")));

    /* client Vlan id and assignment reason */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortVlanAssignmentGet(client_handle, lIntIfNum, &vlanId, &vlanReason))
    {
      printf ("Failed to get authenticated vlan and assignment reason for logical port %d \r\n", lIntIfNum);
      return;
    }
    printf("Vlan Id : %d \r\n", vlanId);
    printf("Vlan assignment reason : %s \r\n", (vlanReason == OPEN_AUTHMGR_RADIUS_ASSIGNED_VLAN) ? "RADIUS" : ((vlanReason == OPEN_AUTHMGR_UNAUTHENTICATED_VLAN) ? "UNAUTH" : ((vlanReason == OPEN_AUTHMGR_GUEST_VLAN) ? "Guest" : "None")));

    /* client session time */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortSessionTimeGet(client_handle, lIntIfNum, &sessionTime))
    {
      printf ("Failed to get authenticated client's session time for logical port %d \r\n", lIntIfNum);
      return;
    }

    printf("Session Time : %d \r\n", sessionTime);

    /* client session timeout */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortSessionTimeoutGet(client_handle, lIntIfNum, &sessionTimeout))
    {
      printf ("Failed to get authenticated client's session timeout for logical port %d \r\n", lIntIfNum);
      return;
    }
    printf("Session Timeout : %d \r\n", sessionTimeout);

    /* client termination action */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortTerminationActionGet(client_handle, lIntIfNum, &terminationAction))
    {
      printf ("Failed to get authenticated client's termination action for logical port %d \r\n", lIntIfNum);
      return;
    }
    printf("Termination Action : %s \r\n", (terminationAction == OPEN_AUTHMGR_TERMINATION_ACTION_RADIUS) ? "Reauthenticate" : "Default");

    /* client time leftout for termination action */

    if (OPEN_E_NONE != openapiAuthmgrLogicalPortTimeLeftForTerminateActionGet(client_handle, lIntIfNum, &timeLeft))
    {
      printf ("Failed to get time left for termination action for logical port %d \r\n", lIntIfNum);
      return;
    }
    printf("Time left for Termination action  : %d \r\n", timeLeft);

    /* client filter-id */

    if (OPEN_E_NONE != openapiAuthMgrLogicalPortFilterIdGet(client_handle, lIntIfNum, &filterId))
    {
      printf ("Failed to Filter-id for logical port %d \r\n", lIntIfNum);
      return;
    }
    printf("Filter-Id  : %s \r\n", (char *)filterId.pstart);
    printf ("\n=========================\r\n");
  }
  else
  {
    printf("The authentication is still in progress for this client\r\n");
  }
}

/*********************************************************************
* @purpose  display the authenticated clients   
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrAuthenticationClientsDisplay(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  uint32_t lIntIfNum;
  if (OPEN_E_NONE != openapiAuthMgrLogicalPortFirstGet(client_handle, intIfNum, &lIntIfNum))
  {
    printf ("No clients are present on interface %d\r\n", intIfNum);
    return;
  }

  do 
  {
    authMgrAuthenticationClientsShow(client_handle, lIntIfNum);
  }while (OPEN_E_NONE == openapiAuthMgrLogicalPortNextGet(client_handle, lIntIfNum, &lIntIfNum));
}


/*********************************************************************
* @purpose  display the authentication interface params  
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  internal interface number

*
* @returns  none
*
* @end
*********************************************************************/
void authMgrAuthenticationInterfaceShow(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;
  uint32_t adminMode = 0, reAuthPeriod, clientsCount = 0;
  OPEN_AUTHMGR_PORT_CONTROL_t portControl = 0;
  OPEN_AUTHMGR_HOST_CONTROL_t hostMode = 0;
  OPEN_AUTHMGR_METHOD_t method = 0;
  char val[DISP_BUF_SIZE];
  char *buf;
  uint32_t len = 0, remLen = 0, idx = 0;
  OPEN_BOOL_t mode = OPEN_FALSE;

  printf("Display the global and interface parameters of the port \n\n");
  printf("===== Global Parameters ===== \n");

  /* admin mode */
  result = openapiAuthMgrAdminModeGet(client_handle, &adminMode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get the admin mode (result = %d)\n", result);
    return;
  }


  printf("\nAuthentication Enable : %s \n", (adminMode == OPEN_ENABLE)? "Enabled": "Disabled");

  /* Dynamic VLAN creation mode */
  adminMode = OPEN_DISABLE;
  result = openapiAuthMgrDynamicVlanCreationModeGet(client_handle, &adminMode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get the dynamic VLAN creation mode (result = %d)\n", result);
    return;
  }


  printf("\nDynamic VLAN Creation Mode : %s \n", (adminMode == OPEN_ENABLE)? "Enabled": "Disabled");

  /* Dynamic Vlan Port assignment Mode */
  adminMode = OPEN_DISABLE;
  result = openapiAuthMgrVlanAssignmentModeGet(client_handle, &adminMode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get the dynamic VLAN creation mode (result = %d)\n", result);
    return;
  }


  printf("\nDynamic VLAN-Port Assignment Mode : %s \n", (adminMode == OPEN_ENABLE)? "Enabled": "Disabled");



  /* port control mode */
  result = openapiAuthMgrPortControlModeGet(client_handle, intIfNum, &portControl);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get the port control mode (result = %d)\n", result);
    return;
  }

  memset(val, '\0', sizeof(val));

  switch (portControl)
  {
    case OPEN_AUTHMGR_PORT_AUTO:
      strncpy(val, "auto", strlen("auto")+1);
      break;
    case OPEN_AUTHMGR_PORT_FORCE_AUTHORIZED:
      strncpy(val, "force-authorized", strlen("force-authorized")+1);
      break;
    case OPEN_AUTHMGR_PORT_FORCE_UNAUTHORIZED:
      strncpy(val, "force-unauthorized", strlen("force-unauthorized")+1);
      break;
    default:
      strncpy(val, "none", strlen("none")+1);
      break;
  }

  printf("\nPort Control Mode : %s \r\n", val);


  /* host mode */
  result = openapiAuthMgrAuthenticationHostModeGet(client_handle, intIfNum, &hostMode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get the port control mode (result = %d)\n", result);
    return;
  }

  memset(val, '\0', sizeof(DISP_BUF_SIZE));

  switch (hostMode)
  {
    case OPEN_AUTHMGR_SINGLE_AUTH_MODE:
      strncpy(val, "single-auth", strlen("single-auth")+1);
      break;
    case OPEN_AUTHMGR_MULTI_AUTH_MODE:
      strncpy(val, "multi-auth", strlen("multi-auth")+1);
      break;
    case OPEN_AUTHMGR_MULTI_HOST_MODE:
      strncpy(val, "multi-host", strlen("multi-host")+1);
      break;
    case OPEN_AUTHMGR_MULTI_DOMAIN_HOST_MODE:
      strncpy(val, "multi-domain-multi-host", strlen("multi-domain-multi-host")+1);
      break;
    default:
      strncpy(val, "none", strlen("none")+1);
      break;
  }

  printf("\nHost Mode : %s \r\n", val);

  /* configured order  */

  memset(val, '\0', sizeof(DISP_BUF_SIZE));
  remLen = DISP_BUF_SIZE -1;

  buf = &val[0];

  len = snprintf(buf, remLen, "Configured Methods : ");
  remLen = remLen - len;
  buf = buf + len;

  for (idx = 0; idx <= 2; idx++)
  {
    method = OPEN_AUTHMGR_METHOD_NONE;
    len = 0;
    if(openapiAuthMgrPortAuthMethodOrderGet(client_handle, intIfNum, idx, &method) == OPEN_E_NONE)
    {
      switch (method)
      {
        case OPEN_AUTHMGR_METHOD_8021X:
          len = snprintf(buf, remLen, "802.1X");
          break;
        case OPEN_AUTHMGR_METHOD_MAB:
          len = snprintf(buf, remLen, "MAB");
          break;
        case OPEN_AUTHMGR_METHOD_CP:
          len = snprintf(buf, remLen, "Captive-Portal");
          break;

        default:
          len = snprintf(buf, remLen, "undefined");
          break;
      }
    }

    buf += len;
    remLen = remLen - len;
    len = 0;
    len = snprintf(buf, remLen, " ");
    buf += len;
    remLen = remLen - len;
  }

  printf("\r\n %s \r\n", val);

  /* configured priority  */

  memset(val, '\0', sizeof(DISP_BUF_SIZE));
  remLen = DISP_BUF_SIZE -1;

  buf = &val[0];

  len = snprintf(buf, remLen, "Configured Priority : ");
  remLen = remLen - len;
  buf = buf + len;

  for (idx = 0; idx <= 2; idx++)
  {
    method = OPEN_AUTHMGR_METHOD_NONE;
    len = 0;
    if(openapiAuthMgrPortAuthMethodPriorityGet(client_handle, intIfNum, idx, &method) == OPEN_E_NONE)
    {
      switch (method)
      {
        case OPEN_AUTHMGR_METHOD_8021X:
          len = snprintf(buf, remLen, "802.1X");
          break;
        case OPEN_AUTHMGR_METHOD_MAB:
          len = snprintf(buf, remLen, "MAB");
          break;
        case OPEN_AUTHMGR_METHOD_CP:
          len = snprintf(buf, remLen, "Captive-Portal");
          break;

        default:
          len = snprintf(buf, remLen, "undefined");
          break;
      }
    }

    buf += len;
    remLen = remLen - len;
    len = 0;
    len = snprintf(buf, remLen, " ");
    buf += len;
    remLen = remLen - len;
  }

  printf("\r\n %s \r\n", val);


  /* Enabled methods  */

  memset(val, '\0', sizeof(DISP_BUF_SIZE));
  remLen = DISP_BUF_SIZE -1;

  buf = &val[0];

  len = snprintf(buf, remLen, "Enabled methods : ");
  remLen = remLen - len;
  buf = buf + len;

  for (idx = 0; idx <= 2; idx++)
  {
    method = OPEN_AUTHMGR_METHOD_NONE;
    len = 0;
    if(openapiAuthMgrEnabledMethodGet(client_handle, intIfNum, idx, &method) == OPEN_E_NONE)
    {
      switch (method)
      {
        case OPEN_AUTHMGR_METHOD_8021X:
          len = snprintf(buf, remLen, "802.1X");
          break;
        case OPEN_AUTHMGR_METHOD_MAB:
          len = snprintf(buf, remLen, "MAB");
          break;
        case OPEN_AUTHMGR_METHOD_CP:
          len = snprintf(buf, remLen, "Captive-Portal");
          break;

        default:
          len = snprintf(buf, remLen, "undefined");
          break;
      }
    }

    buf += len;
    remLen = remLen - len;
    len = 0;
    len = snprintf(buf, remLen, " ");
    buf += len;
    remLen = remLen - len;
  }

  printf("\r\n %s \r\n", val);

  /* Enabled priority  */

  memset(val, '\0', sizeof(DISP_BUF_SIZE));
  remLen = DISP_BUF_SIZE -1;

  buf = &val[0];

  len = snprintf(buf, remLen, "Enabled priority : ");
  remLen = remLen - len;
  buf = buf + len;

  for (idx = 0; idx <= 2; idx++)
  {
    method = OPEN_AUTHMGR_METHOD_NONE;
    len = 0;
    if(openapiAuthMgrEnabledPriorityGet(client_handle, intIfNum, idx, &method) == OPEN_E_NONE)
    {
      switch (method)
      {
        case OPEN_AUTHMGR_METHOD_8021X:
          len = snprintf(buf, remLen, "802.1X");
          break;
        case OPEN_AUTHMGR_METHOD_MAB:
          len = snprintf(buf, remLen, "MAB");
          break;
        case OPEN_AUTHMGR_METHOD_CP:
          len = snprintf(buf, remLen, "Captive-Portal");
          break;

        default:
          len = snprintf(buf, remLen, "undefined");
          break;
      }
    }

    buf += len;
    remLen = remLen - len;
    len = 0;
    len = snprintf(buf, remLen, " ");
    buf += len;
    remLen = remLen - len;
  }

  printf("\r\n %s \r\n", val);


  /* reauth status */
  mode = OPEN_FALSE;

  if ((result = openapiAuthMgrAuthenticationPeriodicGet(client_handle, intIfNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Reauthentication Periodic status  (result = %d)\n", result);
    return;
  }
  printf("\nReauthentication Enabled : %s\r\n", (OPEN_TRUE == mode) ? "True" : "False");



  /* reauthentication period */

  if ((result = openapiAuthMgrAuthenticationReAuthPeriodGet(client_handle, intIfNum, &reAuthPeriod)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Reauthentication Periodic (result = %d)\n", result);
    return;
  }

  printf("\nReauthentication Period : %d\r\n", reAuthPeriod);

  /* session timeout from Radius server Enabled */
  mode = OPEN_FALSE;
  if ((result = openapiAuthMgrAuthenticationReAuthPeriodFromSeverGet(client_handle, intIfNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the session timeout from Radius Server (result = %d)\n", result);
    return;
  }
  printf("\nSession Timeout from Radius Server Enabled : %s\r\n", (OPEN_TRUE == mode) ? "True" : "False");

  /* Max-users */
  reAuthPeriod = 0;
  if (OPEN_E_NONE != openapiAuthMgrPortMaxUsersGet(client_handle, intIfNum, &reAuthPeriod))
  {
    printf("Bad return code trying to get the max users ");
    return;
  }

  printf("\nMax Users : %d\r\n", reAuthPeriod);

  /* Guest Vlan Id */
  reAuthPeriod = 0;
  if (OPEN_E_NONE != openapiAuthMgrPortGuestVlanGet(client_handle, intIfNum, &reAuthPeriod))
  {
    printf("Bad return code trying to get the Guest VLAN Id ");
    return;
  }

  printf("\nGuest VLAN Id : %d\r\n", reAuthPeriod);


  /* Unauth Vlan Id */
  reAuthPeriod = 0;
  if (OPEN_E_NONE != openapiAuthMgrPortUnauthenticatedVlanGet(client_handle, intIfNum, &reAuthPeriod))
  {
    printf("Bad return code trying to get the Unauth VLAN Id ");
    return;
  }

  printf("\nUnauth VLAN Id : %d\r\n", reAuthPeriod);


  /* Max auth re-try Attempts */
  reAuthPeriod = 0;
  if (OPEN_E_NONE != openapiAuthMgrPortMaxAuthAttemptsGet(client_handle, intIfNum, &reAuthPeriod))
  {
    printf("Bad return code trying to get the Max auth retry Attempts ");
    return;
  }

  printf("\nMax Auth retry attempts : %d\r\n", reAuthPeriod);

  /* number of clients authenticated */

  if (OPEN_E_NONE != openapiAuthMgrNumClientsGet(client_handle, intIfNum, &clientsCount))
  {
    printf ("Failed to get number of clients authenticated for logical port %d \r\n", intIfNum);
    return;
  }

  printf("\nNumber of Authenticated clients : %d \r\n", clientsCount);
  printf ("\n=========================\r\n");
  printf ("\r\n");

  return;
}

void portControlModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_PORT_CONTROL_t mode;

  if (argc < 4)
  {
    printf("Interface and mode parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  mode = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrPortControlModeSet(clientHandle, ifNum, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set port control mode (result = %d)\n", result);
  }
  else
  {
    printf("port control mode set. (result = %d)\n", result);
  }
  return;
}

void authMgrAdminModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  printf("0 %s, 1 %s, 2 %s \n", argv[0], argv[1], argv[2]);
  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }
  errno = 0;

  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiAuthMgrAdminModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set authentication manager admin mode (result = %d)\n", result);
  }
  else
  {
    printf("authentication manager admin mode set. (result = %d)\n", result);
  }
  return;
}

void authHostModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_HOST_CONTROL_t mode;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  mode = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationHostModeSet(clientHandle, ifNum, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set port host mode configuration (result = %d)\n", result);
  }
  else
  {
    printf("Port host mode configuration set. (result = %d)\n", result);
  }
  return;
}

void portGuestVlanSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t vlan;

  if (argc < 4)
  {
    printf("Interface and VLAN ID parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  vlan = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid VLAN ID parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrPortGuestVlanSet(clientHandle, ifNum, vlan);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set port guest VLAN ID (result = %d)\n", result);
  }
  else
  {
    printf("Port guest VLAN ID set. (result = %d)\n", result);
  }
  return;
}

void authenticationPeriodicSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t val;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  val = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationPeriodicSet(clientHandle, ifNum, val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set periodic authentication configuration (result = %d)\n", result);
  }
  else
  {
    printf("periodic authentication configuration set. (result = %d)\n", result);
  }
  return;
}

void reauthenticationPeriodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t val;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  val = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationReAuthPeriodSet(clientHandle, ifNum, val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set reauthentication period configuration (result = %d)\n", result);
  }
  else
  {
    printf("Reauthentication period configuration set. (result = %d)\n", result);
  }
  return;
}

void reauthPeriodFromServerSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t val;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  val = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationReAuthPeriodFromServerSet(clientHandle, ifNum, val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set reauthentication period from server configuration (result = %d)\n", result);
  }
  else
  {
    printf("Reauthentication period from server configuration set. (result = %d)\n", result);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for authmgr
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CONTROL_t mode;
  uint32_t recvMode;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("authmgr example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting authmgr API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          authMgrDynamicVlanCreationModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          authMgrVlanAssignmentModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
    case 3:
      if (argc == 5)
      {
        authMgrPortAuthMethodOrderSet(&clientHandle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 5)
      {
        authMgrPortAuthMethodPrioritySet(&clientHandle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        authMgrPortMaxUsersSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
        authMgrPortUnauthenticatedVlanSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        authMgrAuthenticationInterfaceShow(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        authMgrAuthenticationClientsDisplay(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 4)
      {
        portControlModeSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        authMgrAdminModeSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 4)
      {
        authHostModeSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 4)
      {
        portGuestVlanSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        authenticationPeriodicSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 4)
      {
        reauthenticationPeriodSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 4)
      {
        reauthPeriodFromServerSet(&clientHandle, argc, argv);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 4)
      {
        authMgrPortMaxAuthAttemptsSet(&clientHandle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping authmgr API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
