/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1x_example.c
*
* @purpose   IEEE 802.1x OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    11/16/2016
*
* @end
*
**********************************************************************/
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1x.h"
#include "openapi_authmgr.h"

#define OPENAPI_TITLE "802.1x"
#define EXAMPLE_APP_NAME "dot1x_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

struct
{
  OPEN_USER_MGR_AUTH_METHOD_t val;
  char *text;
} dot1xUserMgrAuthMethodTextList[] =
{
  {OPEN_AUTH_METHOD_UNDEFINED, "Undefined"},
  {OPEN_AUTH_METHOD_ENABLE,    "Enable"},
  {OPEN_AUTH_METHOD_LINE,      "Line"},
  {OPEN_AUTH_METHOD_LOCAL,     "Local"},
  {OPEN_AUTH_METHOD_NONE,      "None"},
  {OPEN_AUTH_METHOD_RADIUS,    "RADIUS"},
  {OPEN_AUTH_METHOD_TACACS,    "TACACS"},
  {OPEN_AUTH_METHOD_REJECT,    "Reject"},
  {OPEN_AUTH_METHOD_IAS,       "IAS"},
};

#define DOT1X_USER_MGR_AUTH_METHOD_TEXT_LIST_SIZE (sizeof(dot1xUserMgrAuthMethodTextList)/sizeof(dot1xUserMgrAuthMethodTextList[0]))

static char *dot1xUserMgrAuthMethodTextGet(OPEN_USER_MGR_AUTH_METHOD_t val)
{
  int i;

  for (i = 0; i < DOT1X_USER_MGR_AUTH_METHOD_TEXT_LIST_SIZE; i++)
  {
    if (dot1xUserMgrAuthMethodTextList[i].val == val)
    {
      return(dot1xUserMgrAuthMethodTextList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_PORT_STATUS_t val;
  char *text;
} dot1xPortStatusTextList[] =
{
  {OPEN_DOT1X_PORT_STATUS_AUTHORIZED,   "Authorized"},
  {OPEN_DOT1X_PORT_STATUS_UNAUTHORIZED, "Unauthorized"},
  {OPEN_DOT1X_PORT_STATUS_NA,           "N/A"},
};

struct
{
  OPEN_DOT1X_APM_STATES_t val;
  char *text;
} dot1xPortApmStateList[] =
{
  {OPEN_DOT1X_APM_INITIALIZE,     "Initialize"},
  {OPEN_DOT1X_APM_DISCONNECTED,   "Disconnected"},
  {OPEN_DOT1X_APM_CONNECTING,     "Connecting"},
  {OPEN_DOT1X_APM_AUTHENTICATING, "Authenticating"},
  {OPEN_DOT1X_APM_AUTHENTICATED,  "Authenticated"},
  {OPEN_DOT1X_APM_ABORTING,       "Aborting"},
  {OPEN_DOT1X_APM_HELD,           "Held"},
  {OPEN_DOT1X_APM_FORCE_AUTH,     "Force Authorized"},
  {OPEN_DOT1X_APM_FORCE_UNAUTH,   "Force Unauthorized"},
};

#define DOT1X_PORT_APM_STATE_TEXT_LIST_SIZE (sizeof(dot1xPortApmStateList)/sizeof(dot1xPortApmStateList[0]))

static char *dot1xPortApmStateTextGet(OPEN_DOT1X_APM_STATES_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_APM_STATE_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortApmStateList[i].val == val)
    {
      return(dot1xPortApmStateList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_BAM_STATES_t val;
  char *text;
} dot1xPortBamStateList[] =
{
  {OPEN_DOT1X_BAM_REQUEST,    "Request"},
  {OPEN_DOT1X_BAM_RESPONSE,   "Response"},
  {OPEN_DOT1X_BAM_SUCCESS,    "Success"},
  {OPEN_DOT1X_BAM_FAIL,       "Fail"},
  {OPEN_DOT1X_BAM_TIMEOUT,    "Timeout"},
  {OPEN_DOT1X_BAM_IDLE,       "Idle"},
  {OPEN_DOT1X_BAM_INITIALIZE, "Initialize"},
};

#define DOT1X_PORT_BAM_STATE_TEXT_LIST_SIZE (sizeof(dot1xPortBamStateList)/sizeof(dot1xPortBamStateList[0]))

static char *dot1xPortBamStateTextGet(OPEN_DOT1X_BAM_STATES_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_BAM_STATE_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortBamStateList[i].val == val)
    {
      return(dot1xPortBamStateList[i].text);
    }
  }
  return("UNKNOWN");
}

/* MAB Authentication type */
struct
{
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val;
  char *text;
} mabPortMABAuhTypeTextList[] =
{
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_INVALID,        "Invalid"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_EAP_MD5,        "EAP MD5"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_PAP,            "PAP"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_CHAP,           "CHAP"},
};

void systemAuthenticationControlModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDot1xSystemAuthControlModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s system authentication control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s system authentication control mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

void systemAuthenticationControlModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  errno = 0;
  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xSystemAuthControlModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s system authentication control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s system authentication control mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void defaultAuthenticationMethodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_USER_MGR_AUTH_METHOD_t method;

  result = openapiDot1xDefaultAuthenMethodGet(clientHandle, &method);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s default authentication method (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s default authentication method get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xUserMgrAuthMethodTextGet(method), method, result);
  }
  return;
}

void defaultAuthenticationMethodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t method;

  if (argc < 3)
  {
    printf("Method parameter required.\n");
    return;
  }

  errno = 0;
  method = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid method parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xDefaultAuthenMethodSet(clientHandle, method);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s default authentication method (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s default authentication method set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void authPortPaeStateGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_APM_STATES_t state;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortPaeStateGet(clientHandle, ifNum, &state);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port PAE state: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port PAE state get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortApmStateTextGet(state), state, result);
  }
  return;
}

void authPortBackendAuthStateGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_BAM_STATES_t state;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortBackendAuthStateGet(clientHandle, ifNum, &state);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port backend authorization state: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port backend authorization state get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortBamStateTextGet(state), state, result);
  }
  return;
}

void dot1xEapolFloodModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDot1xEapolFloodModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s eapol flood mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s eapol flood mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

void dot1xEapolFloodModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xEapolFloodModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s eapol flood mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s eapol flood mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void dot1xPortQuietPeriodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t quietPeriod;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortQuietPeriodGet(clientHandle, ifNum, &quietPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port quiet period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port quiet period = %u. (result = %d)\n",
           OPENAPI_TITLE, quietPeriod, result);
  }
  return;
}

void dot1xPortQuietPeriodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t quietPeriod;

  if (argc < 4)
  {
    printf("Interface and quietPeriod parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  quietPeriod = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid quietPeriod parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortQuietPeriodSet(clientHandle, ifNum, quietPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port quiet period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port quiet period set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void dot1xPortTxPeriodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t txPeriod;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortTxPeriodGet(clientHandle, ifNum, &txPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port transmit period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port transmit period = %u. (result = %d)\n",
           OPENAPI_TITLE, txPeriod, result);
  }
  return;
}

void dot1xPortTxPeriodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t txPeriod;

  if (argc < 4)
  {
    printf("Interface and txPeriod parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  txPeriod = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid txPeriod parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortTxPeriodSet(clientHandle, ifNum, txPeriod);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port txPeriod (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port txPeriod set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

functionTableEntry_t functionTable[] =
{
  {systemAuthenticationControlModeGet, "Get "OPENAPI_TITLE" configured global operating control mode.", ""},
  {systemAuthenticationControlModeSet, "Set "OPENAPI_TITLE" configured global operating control mode.", "<mode>"},
  {defaultAuthenticationMethodGet, "Get "OPENAPI_TITLE" default authentication method.", ""},
  {defaultAuthenticationMethodSet, "Set "OPENAPI_TITLE" default authentication method.", "<method>"},
  {authPortPaeStateGet,            "Get "OPENAPI_TITLE" port PAE state.", "<interface>"},
  {authPortBackendAuthStateGet,    "Get "OPENAPI_TITLE" port backend authorization state.", "<interface>"},
  {dot1xEapolFloodModeSet,         "Set "OPENAPI_TITLE" eapol flood mode.", "<mode>"},
  {dot1xEapolFloodModeGet,         "Get "OPENAPI_TITLE" eapol flood mode.", ""},
  {dot1xPortQuietPeriodSet,        "Set "OPENAPI_TITLE" quiet period on the specified port.", "<interface>" "<quietPeriod>"},
  {dot1xPortQuietPeriodGet,        "Get "OPENAPI_TITLE" quiet period on the specified port.", "<interface>"},
  {dot1xPortTxPeriodSet,           "Set "OPENAPI_TITLE" transmit period on the specified port.", "<interface>" "<txPeriod>"},
  {dot1xPortTxPeriodGet,           "Get "OPENAPI_TITLE" transmit period on the specified port.", "<interface>"},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;

  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  errno = 0;
  testNum = strtol(argv[1], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid test number: (%s)\n", argv[1]);
    printAppMenu();
    exit(1);
  }

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
  }
  else
  {
    printAppMenu();
    exit(1);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
