/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mab_example.c
*
* @purpose   MAB OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    02/28/2023
*
* @end
*
**********************************************************************/
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mab.h"
#include "openapi_dot1x.h"

#define OPENAPI_TITLE "MAB"
#define EXAMPLE_APP_NAME "mab_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

/* MAB Authentication type */
struct
{
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val;
  char *text;
} mabPortMABAuhTypeTextList[] =
{
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_INVALID,        "Invalid"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_EAP_MD5,        "EAP MD5"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_PAP,            "PAP"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_CHAP,           "CHAP"},
};

#define MAB_PORT_MAB_AUTH_TYPE_TEXT_LIST_SIZE (sizeof(mabPortMABAuhTypeTextList)/sizeof(mabPortMABAuhTypeTextList[0]))

static char *mabPortMABAuhTypeTextGet(OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val)
{
  int i;

  for (i = 0; i < MAB_PORT_MAB_AUTH_TYPE_TEXT_LIST_SIZE; i++)
  {
    if (mabPortMABAuhTypeTextList[i].val == val)
    {
      return(mabPortMABAuhTypeTextList[i].text);
    }
  }
  return("UNKNOWN");
}

void portMabEnabledGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t mabEnabled;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiMabPortMabEnabledGet(clientHandle, ifNum, &mabEnabled);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port MAB enable configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port MAB enable configuration get: %s. (result = %d)\n",
           OPENAPI_TITLE, (mabEnabled == OPEN_TRUE)?"ENABLED":"DISABLED", result);
  }
  return;
}

void portMabEnabledSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t mabEnabled;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  if (strcmp("true", argv[3]) == 0)
  {
    mabEnabled = OPEN_TRUE;
  }
  else if (strcmp("false", argv[3]) == 0)
  {
    mabEnabled = OPEN_FALSE;
  }
  else
  {
    printf("Invalid enable parameter (use 'true' or 'false'): (%s)\n", argv[3]);
    return;
  }

  result = openapiMabPortMabEnabledSet(clientHandle, ifNum, mabEnabled);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port MAB enable configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port MAB enable configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void mabPortAuthTypeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t mode;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  mode = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiMabPortMABAuthTypeSet(clientHandle, ifNum, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s mab port authentication type configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s mab port authentication type configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void mabPortAuthTypeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiMabPortMABAuthTypeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s mab authentication type (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s mab authentication type get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, mabPortMABAuhTypeTextGet(mode), mode, result);
  }
  return;
}

void authPortMABModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiMabPortOperationalMabModeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port operational MAB mode: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port operational MAB mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode ? "ENABLED" : "DISABLED"), mode, result);
  }
  return;
}

functionTableEntry_t functionTable[] =
{
  {portMabEnabledGet,              "Get "OPENAPI_TITLE" port MAB enable configuration.", "<interface>"},
  {portMabEnabledSet,              "Set "OPENAPI_TITLE" port MAB enable configuration.", "<interface> <'true'|'false'>"},
  {authPortMABModeGet,             "Get "OPENAPI_TITLE" port MAB mode.", "<interface>"},
  {mabPortAuthTypeSet,             "Set "OPENAPI_TITLE" MAB authentication type.", "<interface> <mode>"},
  {mabPortAuthTypeGet,             "Get "OPENAPI_TITLE" MAB authentication type.", "<interface>"},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;

  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  errno = 0;
  testNum = strtol(argv[1], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid test number: (%s)\n", argv[1]);
    printAppMenu();
    exit(1);
  }

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
  }
  else
  {
    printAppMenu();
    exit(1);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
