/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  aclarptable_example.c
*
* @purpose   ACL ARP Status APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    12/21/2016
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_dynamic_arp_inspection.h"
#include "openapi_routing_config.h"

#define MAC_STRING_SIZE 18
/*********************************************************************
* @purpose  Emulate cli "show arp access-list <acl-name>"
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    name             @b{(input)}   acl name or ""
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showArpAccessListByName(openapiClientHandle_t *client_handle, const char *name)
{
  open_buffdesc aclName;
  open_buffdesc aclNameIn;
  open_buffdesc aclNameOut;
  open_error_t rc;
  open_inet_addr_t ipAddr;
  char macAddrOut[MAC_STRING_SIZE] = {0};
  char macAddrAny[MAC_STRING_SIZE] = "00:00:00:00:00:00";
  char aclNameStr[OPEN_ARP_ACL_NAME_LEN_MAX + 1];
  char aclNameInStr[OPEN_ARP_ACL_NAME_LEN_MAX + 1];
  char aclNameOutStr[OPEN_ARP_ACL_NAME_LEN_MAX + 1];
  char ipAddrStr[24];
  char ipStr[24];
  char macStr[32];
  char actionStr[16];
  open_buffdesc mac_addr_out;
  uint32_t ip;
  uint32_t seqIn;
  uint32_t seqOut;
  int len;
  bool aclNameArg = false;
  OPEN_ARP_ACL_RULE_ACTION_t actionOut;

  if ((len = strlen(name)) > OPEN_ARP_ACL_NAME_LEN_MAX) 
  {
    printf("ACL name %s is too long\n", name);
    return;
  }

  if (strncmp(name, OPEN_DEFAULT_VRF_NAME, OPEN_ARP_ACL_NAME_LEN_MAX)) 
  {
    aclNameArg = true;
  }

  strncpy(aclNameStr, name, sizeof(aclNameStr));
  aclName.pstart = aclNameStr;
  aclName.size = sizeof(aclNameStr); 

  if (len > 0)
  {
    if ((rc = openapiArpAclGet(client_handle, &aclName)) == OPEN_E_FAIL) 
    {
      printf("ACL name %s does not exist\n", aclNameStr);
      return;
    } 
    else if (rc == OPEN_E_PARAM)
    {
      printf("Invalid argument to openapiArpAclGet\n");
      return;
    }
  }
  printf("\n\nshow arp access-list %s\n\n", name);

  seqIn = 0;

  memset(macAddrOut, 0, sizeof(macAddrOut));
  mac_addr_out.pstart = macAddrOut;
  mac_addr_out.size = sizeof(macAddrOut);

  ipAddr.family = OPEN_AF_INET;
  ipAddr.addr.ipv4 = 0;
  rc = OPEN_E_NONE;

  printf("ARP access list %s\n", name);
  aclNameIn.size = strlen(name) + 1;
  aclNameIn.pstart = aclNameInStr;
  strncpy(aclNameInStr, name, aclNameIn.size);
  aclNameOut.pstart = aclNameOutStr;
  while (rc == OPEN_E_NONE) {
    mac_addr_out.size = sizeof(macAddrOut);
    rc = openapiArpAclRuleNextGetV2(client_handle, &aclNameIn, seqIn,
                                  &seqOut, &ipAddr, &mac_addr_out, &actionOut);
    seqIn = seqOut;
    if (rc != OPEN_E_NONE && aclNameArg == false) 
    {
      aclNameOut.size = sizeof(aclNameOutStr);
      rc = openapiArpAclNextGet(client_handle, &aclNameIn, &aclNameOut);
      if (rc != OPEN_E_NONE)
      {
        break;
      }
      printf("ARP access list %s\n", aclNameOutStr);
      aclNameIn.size = aclNameOut.size;
      memcpy(aclNameInStr, aclNameOutStr, aclNameIn.size);
      seqIn = 0;
    }
    else if (rc == OPEN_E_NONE)
    {
      ip = htonl(ipAddr.addr.ipv4);
      if(ip == 0)
      {
        sprintf(ipStr, "any");
      }
      else
      {
        if (inet_ntop(AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)) == 0)
        {
          printf("Bad return code trying to convert IP.\n");
          return;
        }
        sprintf(ipStr, "host %s", ipAddrStr);
      }
      if (strcmp(macAddrOut, macAddrAny) == 0)
      {
        sprintf(macStr, "any");
      }
      else
      {
        sprintf(macStr, "host %s", macAddrOut); 
      }
      if(actionOut == OPEN_ARP_ACL_RULE_ACTION_PERMIT)
      {
        sprintf(actionStr, "permit");
      }
      else if(actionOut == OPEN_ARP_ACL_RULE_ACTION_DENY)
      {
        sprintf(actionStr, "deny");
      }
      else
      {
        continue;
      }
      printf("     %s ip %s mac %s\n", actionStr, ipStr, macStr);
    }
  }
}

void print_usage(const char *name)
{
  printf("%s [-a aclname]\n", name);
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  char aclName[OPEN_ARP_ACL_NAME_LEN_MAX + 1];
  int option = 0;
  int rc;

  aclName[0] = '\0';

  while ((option = getopt(argc, argv,"a:")) != -1) {
    switch (option) {
      case 'a' : 
        strncpy(aclName, optarg, sizeof(aclName));
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  /* show all access list arp entries */

  if (strlen(aclName))
  {
    /* show a specific access list arp entry (if specified with -a argument) */

    showArpAccessListByName(&clientHandle, aclName);
  }
  else
  {
    showArpAccessListByName(&clientHandle, OPEN_DEFAULT_VRF_NAME);
  }

  return 0;
}


