/*********************************************************************
*
*  Copyright 2022 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  garp_example.c
*
* @purpose   OpEN GARP example.
*
* @component OpEN
*
* @create    11/16/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_garp.h"

#define GARP_LEAVE_TIMER_MIN_VAL      20
#define GARP_LEAVE_TIMER_MAX_VAL     600

#define GARP_LEAVEALL_TIMER_MIN_VAL  200
#define GARP_LEAVEALL_TIMER_MAX_VAL 6000

#define GARP_JOIN_TIMER_MIN_VAL       10
#define GARP_JOIN_TIMER_MAX_VAL      100
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the GARP Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the GMRP global status: %s 1 \n", name);
  printf("Test 2: Set the GMRP global status. enable(1) and disable (0): %s 2 <globalGmrpStatus>\n", name);
  printf("Test 3: Get the GMRP interface status: %s 3 <ifNum>\n", name);
  printf("Test 4: Set the GMRP interface status. enable(1) and disable (0): %s 4 <ifNum><intfGmrpStatus>\n", name);
  printf("Test 5: Get the GVRP global status: %s 5 \n", name);
  printf("Test 6: Set the GVRP global status. enable(1) and disable (0): %s 6 <globalGvrpStatus>\n", name);
  printf("Test 7: Get the GVRP interface status: %s 7 <ifNum>\n", name);
  printf("Test 8: Set the GVRP interface status. enable(1) and disable (0): %s 8 <ifNum><intfGvrpStatus>\n", name);
  printf("Test 9: Get the GARP Join Time: %s 9 <ifNum>\n", name);
  printf("Test 10: Set the GARP Join Time: %s 10 <ifNum><joinTime>\n", name);
  printf("Test 11: Get the GARP Leave Time: %s 11 <ifNum>\n", name);
  printf("Test 12: Set the GARP Leave Time: %s 12 <ifNum><leaveTime>\n", name);
  printf("Test 13: Get the GARP Leave All Time: %s 13 <ifNum>\n", name);
  printf("Test 14: Set the GARP Leave All Time: %s 14 <ifNum><leaveAllTime>\n", name);
  printf("Test 15: Run API sanity checks: %s 15 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t ifNum = 2, joinTime = 25;

  printf("Testing GARP OpEN APIs sanity:\n");

  printf("Testing openapiGarpJoinTimeSet():\n");
  result = openapiGarpJoinTimeSet(NULL, ifNum, joinTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiGarpJoinTimeSet(clientHandle, ifNum, GARP_JOIN_TIMER_MIN_VAL - 1);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiGarpJoinTimeSet(clientHandle, ifNum, GARP_JOIN_TIMER_MAX_VAL + 1);
  printf("greater than maximum value argument 3:(result = %d)\n", result);

  printf("Testing openapiGarpLeaveTimeSet():\n");
  result = openapiGarpLeaveTimeSet(NULL, ifNum, joinTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiGarpLeaveTimeSet(clientHandle, ifNum, GARP_LEAVE_TIMER_MIN_VAL - 1);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiGarpLeaveTimeSet(clientHandle, ifNum, GARP_LEAVE_TIMER_MAX_VAL + 1);
  printf("greater than maximum value argument 3:(result = %d)\n", result);

  printf("Testing openapiGarpLeaveAllTimeSet():\n");
  result = openapiGarpLeaveAllTimeSet(NULL, ifNum, joinTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiGarpLeaveAllTimeSet(clientHandle, ifNum, GARP_LEAVEALL_TIMER_MIN_VAL - 1);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiGarpLeaveAllTimeSet(clientHandle, ifNum, GARP_LEAVEALL_TIMER_MAX_VAL + 1);
  printf("greater than maximum value argument 3:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Get the GMRP global status.
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void garpGmrpEnableGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_CONTROL_t globalGmrpStatus;

  if ((result = openapiGarpGmrpEnableGet(client_handle, &globalGmrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GMRP global status. (result = %d)\n", result);
  }
  else
  {
    printf("GMRP global status is %u.\n", globalGmrpStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GMRP global status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    globalGmrpStatus    @b{(input)}  GMRP global status.

*
* @returns  none
*
* @end
*********************************************************************/
void garpGmrpEnableSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t globalGmrpStatus)
{
  open_error_t result;

  if ((result = openapiGarpGmrpEnableSet(client_handle, globalGmrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GMRP global status. (result = %d)\n", result);
  }
  else
  {
    printf("GMRP global status set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the GMRP interface status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface

*
* @returns  none
*
* @end
*********************************************************************/
void garpGmrpPortEnableGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t intfGmrpStatus;

  if ((result = openapiGarpGmrpPortEnableGet(client_handle, ifNum, &intfGmrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GMRP interface status. (result = %d)\n", result);
  }
  else
  {
    printf("GMRP interface status is %u.\n", intfGmrpStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GMRP interface status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
* @param    intfGmrpStatus      @b{(input)}  GVRP interface status.

*
* @returns  none
*
* @end
*********************************************************************/
void garpGmrpPortEnableSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t intfGmrpStatus)
{
  open_error_t result;

  if ((result = openapiGarpGmrpPortEnableSet(client_handle, ifNum, intfGmrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GMRP interface status. (result = %d)\n", result);
  }
  else
  {
    printf("GMRP interface status set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the GVRP global status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void garpGvrpEnableGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_CONTROL_t globalGvrpStatus;

  if ((result = openapiGarpGvrpEnableGet(client_handle, &globalGvrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GVRP global status. (result = %d)\n", result);
  }
  else
  {
    printf("GVRP global status is %u.\n", globalGvrpStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GVRP global status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    globalGvrpStatus    @b{(input)}  GVRP global status.

*
* @returns  none
*
* @end
*********************************************************************/
void garpGvrpEnableSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t globalGvrpStatus)
{
  open_error_t result;

  if ((result = openapiGarpGvrpEnableSet(client_handle, globalGvrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GVRP global status. (result = %d)\n", result);
  }
  else
  {
    printf("GVRP global status set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the GVRP interface status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface

*
* @returns  none
*
* @end
*********************************************************************/
void garpGvrpPortEnableGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t intfGvrpStatus;

  if ((result = openapiGarpGvrpPortEnableGet(client_handle, ifNum, &intfGvrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GVRP interface status. (result = %d)\n", result);
  }
  else
  {
    printf("GVRP interface status is %u\n", intfGvrpStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GVRP interface status.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
* @param    intfGvrpStatus      @b{(input)}  GVRP interface status.

*
* @returns  none
*
* @end
*********************************************************************/
void garpGvrpPortEnableSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t intfGvrpStatus)
{
  open_error_t result;

  if ((result = openapiGarpGvrpPortEnableSet(client_handle, ifNum, intfGvrpStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GVRP interface status. (result = %d)\n", result);
  }
  else
  {
    printf("GVRP interface status set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the GARP Join Time.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface

*
* @returns  none
*
* @end
*********************************************************************/
void garpJoinTimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t joinTime;

  if ((result = openapiGarpJoinTimeGet(client_handle, ifNum, &joinTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GARP Join Time. (result = %d)\n", result);
  }
  else
  {
    printf("GARP join time is %u.\n", joinTime);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GARP Join Time.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    joinTime        @b{(input)}  GARP join time in centiseconds.

*
* @returns  none
*
* @end
*********************************************************************/
void garpJoinTimeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t joinTime)
{
  open_error_t result;

  if ((result = openapiGarpJoinTimeSet(client_handle, ifNum, joinTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GARP Join Time. (result = %d)\n", result);
  }
  else
  {
    printf("GARP join time set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the GARP Leave Time.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface

*
* @returns  none
*
* @end
*********************************************************************/
void garpLeaveTimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t leaveTime;

  if ((result = openapiGarpLeaveTimeGet(client_handle, ifNum, &leaveTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GARP Leave Time. (result = %d)\n", result);
  }
  else
  {
    printf("GARP leave time is %u.\n", leaveTime);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GARP Leave Time.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    leaveTime      @b{(input)}  GARP leave time in centiseconds.

*
* @returns  none
*
* @end
*********************************************************************/
void garpLeaveTimeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t leaveTime)
{
  open_error_t result;

  if ((result = openapiGarpLeaveTimeSet(client_handle, ifNum, leaveTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GARP Leave Time. (result = %d)\n", result);
  }
  else
  {
    printf("GARP leave time set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the GARP Leave All Time.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface

*
* @returns  none
*
* @end
*********************************************************************/
void garpLeaveAllTimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t leaveAllTime;

  if ((result = openapiGarpLeaveAllTimeGet(client_handle, ifNum, &leaveAllTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the GARP Leave All Time. (result = %d)\n", result);
  }
  else
  {
    printf("GARP leave all time is %u.\n", leaveAllTime);
  }
  return;
}


/*********************************************************************
* @purpose  Set the GARP Leave All Time.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    leaveAllTime    @b{(input)}  GARP leave All time in centiseconds.

*
* @returns  none
*
* @end
*********************************************************************/
void garpLeaveAllTimeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t leaveAllTime)
{
  open_error_t result;

  if ((result = openapiGarpLeaveAllTimeSet(client_handle, ifNum, leaveAllTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the GARP Leave All Time. (result = %d)\n", result);
  }
  else
  {
    printf("GARP leave all time set successfully.\n");
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for GARP
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("GARP example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting GARP API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        garpGmrpEnableGet(&client_handle);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        garpGmrpEnableSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        garpGmrpPortEnableGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        garpGmrpPortEnableSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 2)
      {
        garpGvrpEnableGet(&client_handle);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        garpGvrpEnableSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        garpGvrpPortEnableGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 4)
      {
        garpGvrpPortEnableSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        garpJoinTimeGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 4)
      {
        garpJoinTimeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        garpLeaveTimeGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 4)
      {
        garpLeaveTimeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
        garpLeaveAllTimeGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 4)
      {
        garpLeaveAllTimeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping GARP API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
