#!/mnt/fastpath/usr/bin/python

"""instru_ft_example.py: OpEN API Instrumentation FT example"""

import OpEN_py as OpEN

from OpENUtil import *

open_ = OpENUtil()

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License")
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

#
# Python 2.6.6
#

# Define some constants for comparison convenience
CONST_FT_FLW_REC_TEMPLATE_MAX_SET             = 1
CONST_FT_DEFAULT_FLW_TEMPLATE_NAME            = "template"
CONST_FT_IP_FIX_FLW_REC_TEMPLATE_ID           = 257
CONST_FT_IP_FIX_FLW_REC_TEMPLATE_FLD_CNT      = 7
CONST_IP_FIX_INF_SRC_IPv4_ELE_ID              = 8
CONST_IP_FIX_INF_SRC_IPv4_ELE_LEN             = 4
CONST_IP_FIX_INF_DST_IPv4_ELE_ID              = 12
CONST_IP_FIX_INF_DST_IPv4_ELE_LEN             = 4
CONST_IP_FIX_INF_SRC_PORT_ELE_ID              = 7
CONST_IP_FIX_INF_SRC_PORT_ELE_LEN             = 2
CONST_IP_FIX_INF_DST_PORT_ELE_ID              = 11
CONST_IP_FIX_INF_DST_PORT_ELE_LEN             = 2
CONST_IP_FIX_INF_IP_PROTO_ELE_ID              = 4
CONST_IP_FIX_INF_IP_PROTO_ELE_LEN             = 1
CONST_IP_FIX_INF_OCT_CNT_ELE_ID               = 85 
CONST_IP_FIX_INF_OCT_CNT_ELE_LEN              = 8
CONST_IP_FIX_INF_PKT_CNT_ELE_ID               = 86
CONST_IP_FIX_INF_PKT_CNT_ELE_LEN              = 4
CONST_FT_CL_TRANS_TYPE_IPv4_UDP               = 0
CONST_FT_TRANS_TYPE_DEFAULT                   = CONST_FT_CL_TRANS_TYPE_IPv4_UDP
CONST_FT_MAX_PKT_SIZE                         = 1500
CONST_FT_CL_PROTO_IPFIXv10                    = 10
CONST_FT_FLW_GRP_MON_TYPE_FIVE_TUPLE          = (1 << 0)
CONST_FT_PKT_TYPE_FLW_TEMPLATE                = 1
CONST_FT_FLW_OD_DEFAULT                       = 1
CONST_FT_FLW_SRC_PORT_DEFAULT                 = 8081 
CONST_FT_FLW_GRP_OD_DEFAULT                   = 2
CONST_FT_FLW_GRP_SRC_PORT_DEFAULT             = 8082 
CONST_FT_DEFAULT_FLW_TEMPLATE_PERIODICITY     = 10
CONST_FT_DEFAULT_FLW_GRP_TEMPLATE_PERIODICITY = 10
CONST_FT_DEFAULT_FLW_GRP_REC_PERIODICITY      = 5
                                               
def print_sanity_results(result, test, msg, feat):
  """Print overall comparision results"""

  if result == OpEN.OPEN_E_UNAVAIL:
    print("Sanity test skipped.")
  elif result == OpEN.OPEN_E_NONE and test == True:
    print("Sanity Success - %s - %s." % (msg, feat))
  else:
    print("Sanity Failure - %s - %s." % (msg, feat))

def print_bad_result(result, msg):
  """Print some general error messages if the result is bad"""
   
  if result == OpEN.OPEN_E_UNAVAIL:
    print("Feature not supported - %s (err %d)." % (msg, result))
  elif result != OpEN.OPEN_E_NONE:
    print("Test Failure - %s (err %d)." % (msg, result))

class InstruFtExample:
  """Simple Instru FT Examples """

  def __init__(self, client, asic) :
    self.m_client = client
    self.m_asic = asic


  def show_ft_asic_capability(self):
    ft_asic_cap_p = OpEN.new_OPEN_FT_CAPABILITIES_tp()
    result=OpEN.openapiFtCapabilitiesGet(self.m_client, self.m_asic, ft_asic_cap_p)
    if (result != OpEN.OPEN_E_NONE):
      print("Error ! failed to get flow tracker asic capability")
      OpEN.delete_OPEN_FT_CAPABILITIES_tp(ft_asic_cap_p)
      return                             

    ft_asic_cap = OpEN.OPEN_FT_CAPABILITIES_tp_value(ft_asic_cap_p)
    print("Displaying Basic ASIC Capabilities")
    print("===============================================")
    print("FT Asic Multiple collector support - %d" % (ft_asic_cap.multiple_collector_support))
    print("FT Asic Maximum collectors         - %d" % (ft_asic_cap.max_collectors))
    print("FT Asic Export protocols           - %d" % (ft_asic_cap.export_protocols))
    print("FT Asic Flow group types           - %d" % (ft_asic_cap.flw_grp_types))
    print("FT Asic Maximum flow groups        - %d" % (ft_asic_cap.max_flw_grps))
    print("FT Asic Maximum flows              - %d" % (ft_asic_cap.max_flws))
    print("FT Asic Flow sampling support      - %d" % (ft_asic_cap.flw_sampling_support))

    OpEN.delete_OPEN_FT_CAPABILITIES_tp(ft_asic_cap_p)

  def config_ft_feature(self):
    print("")
    print("Setting flow tracker feature")
    print("=========================================")
    ftCfg_p = OpEN.new_OPEN_FT_CONFIG_tp()
    ftCfg = OpEN.OPEN_FT_CONFIG_tp_value(ftCfg_p)
    ftCfg.enable                            = 1
    ftCfg.flw_exp_ipfix.periodicity         = CONST_FT_DEFAULT_FLW_TEMPLATE_PERIODICITY
    ftCfg.flw_exp_ipfix.src_port            = CONST_FT_FLW_SRC_PORT_DEFAULT
    ftCfg.flw_exp_ipfix.obs_dmn             = CONST_FT_FLW_OD_DEFAULT
    ftCfg.flw_grp_exp_ipfix.periodicity     = CONST_FT_DEFAULT_FLW_GRP_TEMPLATE_PERIODICITY
    ftCfg.flw_grp_exp_ipfix.rec_periodicity = CONST_FT_DEFAULT_FLW_GRP_REC_PERIODICITY
    ftCfg.flw_grp_exp_ipfix.src_port        = CONST_FT_FLW_GRP_SRC_PORT_DEFAULT
    ftCfg.flw_grp_exp_ipfix.obs_dmn         = CONST_FT_FLW_GRP_OD_DEFAULT
                                  
    print("Setting values flow tracker enable - %d" % (ftCfg.enable))
    OpEN.OPEN_FT_CONFIG_tp_assign(ftCfg_p, ftCfg)
    result=OpEN.openapiFtConfigSet(self.m_client, self.m_asic, ftCfg_p)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Enabled flow tracker feature")
    else:
      print("Error ! failed to set flow tracker feature")
      
    OpEN.delete_OPEN_FT_CONFIG_tp(ftCfg_p)

  def ft_collector_config_create(self):
    print("")
    print("Create flow tracker collector configuration")
    print("===========================================")
    clInfo_p = OpEN.new_OPEN_FT_CL_INFO_tp()
    clInfo = OpEN.OPEN_FT_CL_INFO_tp_value(clInfo_p)
    clInfo.configured                     = True
    clInfo.trans_type                     = CONST_FT_TRANS_TYPE_DEFAULT
    clInfo.max_pkt_len                    = CONST_FT_MAX_PKT_SIZE
    clInfo.config.name                    = "collector"
    clInfo.config.proto                   = CONST_FT_CL_PROTO_IPFIXv10
    clInfo.config.port                    = 2075
    clInfo.config.addr.family             = 1 
    clInfo.config.addr.addr.ipv4          = 0x0a0b0c0d
                                  
    OpEN.OPEN_FT_CL_INFO_tp_assign(clInfo_p, clInfo)
    result=OpEN.openapiFtCollectorConfigCreate(self.m_client, self.m_asic, clInfo_p)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Created flow tracker collector configuration")
    else:
      print("Error ! failed to create flow tracker collector configuration")
      
    OpEN.delete_OPEN_FT_CL_INFO_tp(clInfo_p)

  def ft_collector_config_remove(self):
    print("")
    print("Remove flow tracker collector configuration")
    print("===========================================")
    clName = OpEN.open_buffdesc()
    buf_set = open_.getStringBuffer(len("collector") + 1, "collector")
    clName.pstart = buf_set
    clName.size = len("collector") + 1
                                  
    result=OpEN.openapiFtCollectorConfigRemove(self.m_client, self.m_asic, clName)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Removed flow tracker collector configuration")
    else:
      print("Error ! failed to remove flow tracker collector configuration")
      
	
  def ft_flow_group_config_create(self):
    print("")
    print("Create flow tracker flow group configuration")
    print("===========================================")

    flowGroup_p = OpEN.new_OPEN_FT_FLW_GRP_CONFIG_tp()
    flowGroup = OpEN.OPEN_FT_FLW_GRP_CONFIG_tp_value(flowGroup_p)
    flowGroup.type                                      = CONST_FT_FLW_GRP_MON_TYPE_FIVE_TUPLE
    flowGroup.name                                      = "flowgrp"
    flowGroup.mon_params.five_tuple.src_ip.family       = 1
    flowGroup.mon_params.five_tuple.src_ip.host.ip.addr = 0x11223344
    flowGroup.mon_params.five_tuple.src_ip.host.ip.mask = 0xffff0000
    flowGroup.mon_params.five_tuple.dst_ip.family       = 1
    flowGroup.mon_params.five_tuple.dst_ip.host.ip.addr = 0x55667788
    flowGroup.mon_params.five_tuple.dst_ip.host.ip.mask = 0xffff0000
    flowGroup.mon_params.five_tuple.protocol            = 0x11
    flowGroup.mon_params.five_tuple.src_port            = 1234
    flowGroup.mon_params.five_tuple.dst_port            = 5678
    flowGroup.mon_params.five_tuple.tuple_mask          = 0x1f
    flowGroup.cl_num                                    = 1
    OpEN.setFtFlowGroupCollectorName(flowGroup, 0, "collector")

    OpEN.OPEN_FT_FLW_GRP_CONFIG_tp_assign(flowGroup_p, flowGroup)
    result=OpEN.openapiFtFlowGroupConfigCreate(self.m_client, self.m_asic, flowGroup_p)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Created flow tracker flow group configuration")
    else:
      print("Error ! failed to create flow tracker flow group configuration")
      
    OpEN.delete_OPEN_FT_FLW_GRP_CONFIG_tp(flowGroup_p)

  def ft_flow_group_config_remove(self):
    print("")
    print("Remove flow tracker flow group configuration")
    print("===========================================")
    grpName = OpEN.open_buffdesc()
    buf_set = open_.getStringBuffer(len("flowgrp") + 1, "flowgrp")
    grpName.pstart = buf_set
    grpName.size = len("flowgrp") + 1
                                  
    result=OpEN.openapiFtFlowGroupConfigRemove(self.m_client, self.m_asic, grpName)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Removed flow tracker flow group configuration")
    else:
      print("Error ! failed to remove flow tracker flow group configuration")
      

  def ft_template_config_create(self):
    print("")
    print("Create flow tracker template configuration")
    print("===========================================")

    flowTemplate_p            = OpEN.new_OPEN_FT_FLW_REC_TEMPLATE_tp()
    flowTemplate              = OpEN.OPEN_FT_FLW_REC_TEMPLATE_tp_value(flowTemplate_p)
    flowTemplateData_p        = OpEN.new_OPEN_FT_FLW_REC_TEMPLATE_INFO_tp()
    flowTemplateData          = OpEN.OPEN_FT_FLW_REC_TEMPLATE_INFO_tp_value(flowTemplateData_p)
    flowTemplateDataEleInfo_p = OpEN.new_OPEN_IP_FIX_FLD_SPEC_WITH_EN_tp()
    flowTemplateDataEleInfo   = OpEN.OPEN_IP_FIX_FLD_SPEC_WITH_EN_tp_value(flowTemplateDataEleInfo_p)

    flowTemplateData.name            = "template"
    flowTemplateData.template_hdr_id = CONST_FT_IP_FIX_FLW_REC_TEMPLATE_ID
    flowTemplateData.num_ele         = CONST_FT_IP_FIX_FLW_REC_TEMPLATE_FLD_CNT
    flowTemplate.num_templates       = CONST_FT_FLW_REC_TEMPLATE_MAX_SET

    for idx in range(16):
      flowTemplateDataEleInfo.ent_ele_id      = 0
      flowTemplateDataEleInfo.len             = 0
      flowTemplateDataEleInfo.ent_num_present = False
      flowTemplateDataEleInfo.ent_num         = 0

      if idx == 0 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_SRC_IPv4_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_SRC_IPv4_ELE_LEN

      if idx == 1 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_DST_IPv4_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_DST_IPv4_ELE_LEN

      if idx == 2 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_SRC_PORT_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_SRC_PORT_ELE_LEN

      if idx == 3 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_DST_PORT_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_DST_PORT_ELE_LEN

      if idx == 4 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_IP_PROTO_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_IP_PROTO_ELE_LEN

      if idx == 5 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_PKT_CNT_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_PKT_CNT_ELE_LEN

      if idx == 6 :
        flowTemplateDataEleInfo.ent_ele_id    = CONST_IP_FIX_INF_OCT_CNT_ELE_ID
        flowTemplateDataEleInfo.len           = CONST_IP_FIX_INF_OCT_CNT_ELE_LEN

      OpEN.OPEN_IP_FIX_FLD_SPEC_WITH_EN_tp_assign(flowTemplateDataEleInfo_p, flowTemplateDataEleInfo)
      OpEN.OPEN_IP_FIX_FLD_SPEC_WITH_EN_tArray_setitem(flowTemplateData.ele_info, idx, flowTemplateDataEleInfo_p)

    OpEN.OPEN_FT_FLW_REC_TEMPLATE_INFO_tp_assign(flowTemplateData_p, flowTemplateData)
    OpEN.OPEN_FT_FLW_REC_TEMPLATE_INFO_tArray_setitem(flowTemplate.data, 0, flowTemplateData_p)
                                 
    OpEN.OPEN_FT_FLW_REC_TEMPLATE_tp_assign(flowTemplate_p, flowTemplate)
    result=OpEN.openapiFtFlowTemplateConfigCreate(self.m_client, self.m_asic, flowTemplate_p)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Created flow tracker template configuration")
    else:
      print("Error ! failed to create flow tracker template configuration")
      
    OpEN.delete_OPEN_IP_FIX_FLD_SPEC_WITH_EN_tp(flowTemplateDataEleInfo_p)
    OpEN.delete_OPEN_FT_FLW_REC_TEMPLATE_INFO_tp(flowTemplateData_p)
    OpEN.delete_OPEN_FT_FLW_REC_TEMPLATE_tp(flowTemplate_p)

  def ft_template_config_remove(self):
    print("")
    print("Remove flow tracker template configuration")
    print("===========================================")
    templateName = OpEN.open_buffdesc()
    buf_set = open_.getStringBuffer(len("template") + 1, "template")
    templateName.pstart = buf_set
    templateName.size = len("template") + 1
                                  
    result=OpEN.openapiFtFlowTemplateConfigRemove(self.m_client, self.m_asic, templateName)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Removed flow tracker template configuration")
    else:
      print("Error ! failed to remove flow tracker template configuration")
      

  def ft_flow_group_stats_get(self):
    print("")
    print("Get flow tracker flow group stats")
    print("===========================================")
    grpName = OpEN.open_buffdesc()
    buf_set = open_.getStringBuffer(len("flowgrp") + 1, "flowgrp")
    grpName.pstart = buf_set
    grpName.size = len("flowgrp") + 1

    grpStats_p = OpEN.new_OPEN_FT_FLW_GRP_STATS_tp()
	
    result=OpEN.openapiFtFlowGroupStatsGet(self.m_client, self.m_asic, grpName, grpStats_p)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Retrieved flow tracker flow group stats")
    else:
      print("Error ! failed to get flow tracker flow group stats")
      
    OpEN.delete_OPEN_FT_FLW_GRP_STATS_tp(grpStats_p)

  def ft_flow_group_stats_clear(self):
    print("")
    print("Clear flow tracker flow group stats")
    print("===========================================")
    grpName = OpEN.open_buffdesc()
    buf_set = open_.getStringBuffer(len("flowgrp") + 1, "flowgrp")
    grpName.pstart = buf_set
    grpName.size = len("flowgrp") + 1
                                  
    result=OpEN.openapiFtFlowGroupStatsClear(self.m_client, self.m_asic, grpName)

    if result == OpEN.OPEN_E_NONE :
      print("Success! cleared flow tracker flow group stats")
    else:
      print("Error ! failed to clear flow tracker flow group stats")
      

  def ft_packet_send(self):
    print("")
    print("Flow tracker packet send")
    print("===========================================")

    clName = OpEN.open_buffdesc()
    buf_set = open_.getStringBuffer(len("collector") + 1, "collector")
    clName.pstart = buf_set
    clName.size = len("collector") + 1

    pkt = b'6c 41 6a f4 0c 46 00 18 23 30 ea 52 08 00 45 b8 00 50 00 00 00 00 10 11 ea 65 0a 82 54 09 0a 82 56 73 1f 91 08 07 00 3c d2 6f 00 0a 00 34 59 c8 e9 c9 00 00 00 01 00 00 00 01 00 02 00 24 01 01 00 07 00 08 00 04 00 0c 00 04 00 07 00 02 00 0b 00 02 00 04 00 01 00 56 00 04 00 55 00 08 e1 6b 92 7b'
                                  
    packetPayload = OpEN.open_buffdesc()
    pkt_buf_set = open_.getStringBuffer(len(pkt) + 1, pkt)
    packetPayload.pstart = pkt_buf_set
    packetPayload.size = len(pkt) + 1
                                  
    result=OpEN.openapiFtPktSend(self.m_client, self.m_asic, clName, CONST_FT_PKT_TYPE_FLW_TEMPLATE, packetPayload.size, packetPayload)

    if result == OpEN.OPEN_E_NONE :
      print("Success! Sent packet")
    else:
      print("Error ! failed to send packet")
      
def main():
  """Demonstrate OpEN usage for FT APIs"""

  ret = open_.connect("instru_ft_example")
  if ret == OpEN.OPEN_E_NONE :
    open_.getNetworkOSVersion()
    client = open_.get_client()

    example = InstruFtExample(client, 0)
    example.show_ft_asic_capability()
    example.config_ft_feature()
    example.ft_collector_config_create()
    example.ft_template_config_create()
    example.ft_flow_group_config_create()
    example.ft_packet_send()
    example.ft_flow_group_stats_get()
    example.ft_flow_group_stats_clear()
    example.ft_flow_group_config_remove()
    example.ft_template_config_remove()
    example.ft_collector_config_remove()

    open_.terminate()
  else :
    print("Unable to connect")

if __name__ == '__main__': main()
