/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ip4dhcpclient_example.c
*
* @purpose   OpEN IPv4 DHCP-Client example.
*
* @component OpEN
*
* @create    02/08/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ip4dhcpclient.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the IP4DHCPClient Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the DHCP Vendor Option Mode: %s 1 \n", name);
  printf("Test 2: Set the DHCP Vendor Option Mode: %s 2 <mode> <actImmediate>\n", name);
  printf("Test 3: Get the DHCP Vendor Class Option string: %s 3 \n", name);
  printf("Test 4: Set the DHCP Vendor Class Option string: %s 4 <option-string> <actImmediateFlag>\n", name);
  printf("Test 5: Get the DHCP server IP address on a management interface: %s 5 <ifNum> <mgmtPortType>\n", name);
  printf("Test 6: Get the current state of the DHCP client for specified interface or management port: %s 6 <ifNum> <mgmtPortType>\n", name);
  printf("Test 7: Get the transaction ID of the DHCP client for specified interface or management port: %s 7 <ifNum> <mgmtPortType>\n", name);
  printf("Test 8: Get the lease time information of the DHCP client for specified interface or management port: %s 8 <ifNum> <mgmtPortType>\n", name);
  printf("Test 9: Get the retry count of the DHCP client for specified interface or management port: %s 9 <ifNum> <mgmtPortType>\n", name);
  printf("Test 10: Get the generated client identifier on the interface: %s 10 <ifNum> <mgmtPortType>\n", name);
  printf("Test 11: Get the DHCP Client Statistics on an interface: %s 11 <ifNum>\n", name);
  printf("Test 12: Clear the DHCP Client Statistics on an interface: %s 12 <ifNum>\n", name);

  return;
}

/*********************************************************************
* @purpose  Convert an IP address in a open_inet_addr_t structure to
*           a string for display                  
*
* @param    addr  @b{(input)}  address to be converted
* @param    buffer @b{(output)} storage to contain
*
* @returns  pointer to buffer if conversion successful
*           NULL if error
*
* @notes
*
* @end 
*********************************************************************/
static const char *ipAddressFormat(open_inet_addr_t *addr, char *buffer)
{
  uint32_t  ip4_addr;
  uint8_t   addr8[16];
  int af;

  switch (addr->family)
  {    
  case OPEN_AF_INET:
    af = AF_INET;
    ip4_addr = htonl(addr->addr.ipv4);            
    memcpy(addr8, &ip4_addr, sizeof(ip4_addr));   
    break;
  case OPEN_AF_INET6:
    af = AF_INET6;                                
    memcpy(addr8, addr->addr.ipv6.u.addr8, sizeof(addr8));
    break;
  default:
    return(NULL);
  }    

  return(inet_ntop(af, addr8, buffer, INET6_ADDRSTRLEN));
}

/*********************************************************************
* @purpose  Get the DHCP Vendor Option Mode.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpVendorClassOptionAdminModeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* DHCP vendor option admin mode */

  result = openapiDhcpVendorClassOptionAdminModeGet(client_handle, &mode);
  switch(result)
  {    
    case OPEN_E_NONE:
         printf("\nThe DHCP Vendor Option Mode is %u (1-Enabled, 0-Disabled).\n", mode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");     
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCP Vendor Option Mode. (result = %d)\n", result);
         break;
  }    
  return;
}

/*********************************************************************
* @purpose  Set the DHCP Vendor Option Mode.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    adminMode     @b{(input)}  DHCP vendor option admin mode(1 for ENABLE or 2 for DISABLE)
* @param    flag          @b{(input)}  Immediate action flag(1 for TRUE or 0 for FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpVendorClassOptionAdminModeSet(openapiClientHandle_t *client_handle, uint32_t adminMode, uint32_t flag)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* (OPEN_ENABLE or OPEN_DISABLE) */
  OPEN_BOOL_t actImmediate = OPEN_FALSE; /* (OPEN_TRUE or OPEN_FALSE) */

  if (1 < adminMode)
  {
    printf("\nERROR: Invalid adminMode value. Expected 0(for Disable) or 1(for Enable).");
    return;
  }

  if (1 == adminMode)
  {
    mode = OPEN_ENABLE;
  }

  if (1 < flag)
  {
    printf("\nERROR: Invalid flag value. Expected 0(for False) or 1(for True).");
    return;
  }

  if (1 == flag)
  {
    actImmediate = OPEN_TRUE;
  }

  result = openapiDhcpVendorClassOptionAdminModeSet(client_handle, mode, actImmediate);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given DHCP Vendor Option Mode.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the DHCP Vendor Option Mode. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the DHCP Vendor Class Option string.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpVendorClassOptionStringGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc bufd; /* Buffer descriptor to DHCP Vendor Class Option string */
  char optStr[OPEN_DHCP_VENDOR_CLASS_STRING_MAX + 1] = "";

  memset(optStr, 0, sizeof(optStr));
  bufd.pstart = optStr;
  bufd.size = sizeof(optStr);

  result = openapiDhcpVendorClassOptionStringGet(client_handle, &bufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe DHCP Vendor Class Option string is: %s.\n",(char *)(bufd.pstart));
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCP Vendor Class Option string. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the DHCP Vendor Class Option string.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    *optString    @b{(input)}  DHCP Vendor Class Option string
* @param    actFlag       @b{(input)}  Immediate flag 1(for TRUE) or 0(for FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpVendorClassOptionStringSet(openapiClientHandle_t *client_handle, char *optString, uint32_t actFlag)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc bufd; /* Buffer descriptor to DHCP Vendor Class Option string */
  char optStr[OPEN_DHCP_VENDOR_CLASS_STRING_MAX + 1] = "";
  OPEN_BOOL_t actImmediate = OPEN_FALSE; /* (OPEN_TRUE or OPEN_FALSE) */

  if (1 < actFlag)
  {
    printf("\nERROR: Invalid flag value. Expected 0(for False) or 1(for True).");
    return;
  }

  if (1 == actFlag)
  {
    actImmediate = OPEN_TRUE;
  }

  memset(optStr, 0, sizeof(optStr));
  if (OPEN_DHCP_VENDOR_CLASS_STRING_MAX < strlen(optString))
  {
    printf("\nERROR: Invalid DHCP Vendor Class Option string.\n");
    return;
  }
  strncpy(optStr, optString, sizeof(optStr) - 1);
  bufd.pstart = optStr;
  bufd.size = strlen(optStr) + 1;

  result = openapiDhcpVendorClassOptionStringSet(client_handle, &bufd, actImmediate);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given DHCP Vendor Class Option string.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the DHCP Vendor Class Option string. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the DHCP server IP address on a management interface
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    mgmtType      @b{(input)}  Mangement Interface Type
*
* @returns  none
*
* @end
*********************************************************************/
void ipDhcpClientDhcpServerIntfIPAddressGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mgmtType)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_MGMT_PORT_TYPE_t mgmtPortType = 0;
  uint32_t serverAddr = 0; /* DHCP server IP Address on an interface */
  char ipAddrStr[80] = {0};
  open_inet_addr_t openIpAddr;

  switch(mgmtType)
  {
    case 0:
      mgmtPortType = OPEN_MGMT_SERVICEPORT;
      break;
    case 1:
      mgmtPortType = OPEN_MGMT_NETWORKPORT;
      break;
    case 2:
      mgmtPortType = OPEN_MGMT_IPPORT;
      break;
    default:
      result = OPEN_E_PARAM;
      break;
  }

  if (result == OPEN_E_PARAM)
  {
    printf("\nERROR: Invalid management port type value. Expected 0(for ServicePort) or 1(for Network) or 2(for IP-Port).\n");
    return;
  }

  result = openapiIpDhcpClientDhcpServerIntfIPAddressGet(client_handle, ifNum, mgmtPortType, &serverAddr);
  switch(result)
  {
    case OPEN_E_NONE:
         memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
         openIpAddr.family = OPEN_AF_INET;
         openIpAddr.addr.ipv4 = serverAddr;
         ipAddressFormat(&openIpAddr, ipAddrStr);
         printf("\nThe DHCP server IP address on a management interface is: %s.\n", ipAddrStr);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCP server IP address on a management interface (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the current state of the DHCP client for specified interface or management port
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    mgmtType      @b{(input)}  Mangement Interface Type
*
* @returns  none
*
* @end
*********************************************************************/
void ipDhcpClientIntfStateGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mgmtType)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_MGMT_PORT_TYPE_t mgmtPortType = 0;
  OPEN_DHCP_CLIENT_STATE_t clientState; /* DHCP client state */

  switch(mgmtType)
  {
    case 0:
      mgmtPortType = OPEN_MGMT_SERVICEPORT;
      break;
    case 1:
      mgmtPortType = OPEN_MGMT_NETWORKPORT;
      break;
    case 2:
      mgmtPortType = OPEN_MGMT_IPPORT;
      break;
    default:
      result = OPEN_E_PARAM;
      break;
  }

  if (result == OPEN_E_PARAM)
  {
    printf("\nERROR: Invalid management port type value. Expected 0(for ServicePort) or 1(for Network) or 2(for IP-Port).\n");
    return;
  }

  result = openapiIpDhcpClientIntfStateGet(client_handle, ifNum, mgmtPortType, &clientState);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe current state of the DHCP client for specified interface or management port is: %u.\n", clientState);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the current state of the DHCP client for specified interface or management port (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the transaction ID of the DHCP client for specified interface or management port
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    mgmtType      @b{(input)}  Mangement Interface Type
*
* @returns  none
*
* @end
*********************************************************************/
void ipDhcpClientTransactionIntfIdGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mgmtType)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_MGMT_PORT_TYPE_t mgmtPortType = 0;
  uint32_t transId = 0; /* DHCP client transaction ID */

  switch(mgmtType)
  {
    case 0:
      mgmtPortType = OPEN_MGMT_SERVICEPORT;
      break;
    case 1:
      mgmtPortType = OPEN_MGMT_NETWORKPORT;
      break;
    case 2:
      mgmtPortType = OPEN_MGMT_IPPORT;
      break;
    default:
      result = OPEN_E_PARAM;
      break;
  }

  if (result == OPEN_E_PARAM)
  {
    printf("\nERROR: Invalid management port type value. Expected 0(for ServicePort) or 1(for Network) or 2(for IP-Port).\n");
    return;
  }

  result = openapiIpDhcpClientTransactionIntfIdGet(client_handle, ifNum, mgmtPortType, &transId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe transaction ID of the DHCP client for specified interface or management port is: %u.\n", transId);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the transaction ID of the DHCP client for specified interface or management port (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the lease time information of the DHCP client for specified interface or management port
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    mgmtType      @b{(input)}  Mangement Interface Type
*
* @returns  none
*
* @end
*********************************************************************/
void ipDhcpClientLeaseTimeIntfInfoGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mgmtType)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_MGMT_PORT_TYPE_t mgmtPortType = 0;
  uint32_t leaseTime   = 0; /* DHCP client lease time */
  uint32_t renewalTime = 0; /* DHCP client renewal time */
  uint32_t rebindTime  = 0; /* DHCP client rebinding time */

  switch(mgmtType)
  {
    case 0:
      mgmtPortType = OPEN_MGMT_SERVICEPORT;
      break;
    case 1:
      mgmtPortType = OPEN_MGMT_NETWORKPORT;
      break;
    case 2:
      mgmtPortType = OPEN_MGMT_IPPORT;
      break;
    default:
      result = OPEN_E_PARAM;
      break;
  }

  if (result == OPEN_E_PARAM)
  {
    printf("\nERROR: Invalid management port type value. Expected 0(for ServicePort) or 1(for Network) or 2(for IP-Port).\n");
    return;
  }

  result = openapiIpDhcpClientLeaseTimeIntfInfoGet(client_handle, ifNum, mgmtPortType, &leaseTime, &renewalTime, &rebindTime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe lease time information of the DHCP client for specified interface or management port is: \n");
         printf("\tDHCP client lease time: %u \n",leaseTime);
         printf("\tDHCP client renewal time: %u \n",renewalTime);
         printf("\tDHCP client rebinding time: %u \n",rebindTime);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the lease time information of the DHCP client for specified interface or management port (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the retry count of the DHCP client for specified interface or management port
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    mgmtType      @b{(input)}  Mangement Interface Type
*
* @returns  none
*
* @end
*********************************************************************/
void ipDhcpClientIntfRetryCountGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mgmtType)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_MGMT_PORT_TYPE_t mgmtPortType = 0;
  uint32_t retryCount = 0; /* DHCP client's retry count */

  switch(mgmtType)
  {
    case 0:
      mgmtPortType = OPEN_MGMT_SERVICEPORT;
      break;
    case 1:
      mgmtPortType = OPEN_MGMT_NETWORKPORT;
      break;
    case 2:
      mgmtPortType = OPEN_MGMT_IPPORT;
      break;
    default:
      result = OPEN_E_PARAM;
      break;
  }

  if (result == OPEN_E_PARAM)
  {
    printf("\nERROR: Invalid management port type value. Expected 0(for ServicePort) or 1(for Network) or 2(for IP-Port).\n");
    return;
  }

  result = openapiIpDhcpClientIntfRetryCountGet(client_handle, ifNum, mgmtPortType, &retryCount);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe retry count of the DHCP client for specified interface or management port is: %u.\n", retryCount);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the retry count of the DHCP client for specified interface or management port (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the generated client identifier on the interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    mgmtType      @b{(input)}  Mangement Interface Type
*
* @returns  none
*
* @end
*********************************************************************/
void ipDhcpClientIdGeneratedGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mgmtType)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_MGMT_PORT_TYPE_t mgmtPortType = 0;
  open_buffdesc clientIdBufd; /* Buffer descriptor to the clientId string */
  char clientStr[OPEN_DHCP_CLIENTID_MAX + 1] = "";

  memset(clientStr, 0, sizeof(clientStr));
  clientIdBufd.pstart = clientStr;
  clientIdBufd.size = sizeof(clientStr);

  switch(mgmtType)
  {
    case 0:
      mgmtPortType = OPEN_MGMT_SERVICEPORT;
      break;
    case 1:
      mgmtPortType = OPEN_MGMT_NETWORKPORT;
      break;
    case 2:
      mgmtPortType = OPEN_MGMT_IPPORT;
      break;
    default:
      result = OPEN_E_PARAM;
      break;
  }

  if (result == OPEN_E_PARAM)
  {
    printf("\nERROR: Invalid management port type value. Expected 0(for ServicePort) or 1(for Network) or 2(for IP-Port).\n");
    return;
  }
 
  result = openapiIpDhcpClientIdGeneratedGet(client_handle, ifNum, mgmtPortType, &clientIdBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe generated client identifier on the given interface is: %s.\n",(char*)(clientIdBufd.pstart));
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the generated client identifier on the interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the DHCP Client Statistics on an interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpClientStatsGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t numDiscoverSent = 0; /* Number of Discovers sent */
  uint32_t numOfferRecvd   = 0; /* Number of Offers received */
  uint32_t numRequestSent  = 0; /* Number of Requests sent */
  uint32_t numAckRecvd     = 0; /* Number of Acks received */
  uint32_t numNackRecvd    = 0; /* Number of Nacks received */
  uint32_t numReleaseSent  = 0; /* Number of Releases sent */
  uint32_t numInformSent   = 0; /* Number of Informs sent */
  uint32_t numRebindSent   = 0; /* Number of Rebinds sent */
  uint32_t numRenewSent    = 0; /* Number of Renews sent */

  result = openapiDhcpClientStatsGet(client_handle, ifNum, &numDiscoverSent, &numOfferRecvd, &numRequestSent, &numAckRecvd, &numNackRecvd, &numReleaseSent, &numInformSent, &numRebindSent, &numRenewSent);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe DHCP Client Statistics on the given interface are: \n");
         printf("\tNumber of Discovers sent: %u.\n",numDiscoverSent);
         printf("\tNumber of Offers received: %u.\n",numOfferRecvd);
         printf("\tNumber of Requests sent: %u.\n",numRequestSent);
         printf("\tNumber of Acks received: %u.\n",numAckRecvd);
         printf("\tNumber of Nacks received: %u.\n",numNackRecvd);
         printf("\tNumber of Releases sent: %u.\n",numReleaseSent);
         printf("\tNumber of Informs sent: %u.\n",numInformSent);
         printf("\tNumber of Rebinds sent: %u.\n",numRebindSent);
         printf("\tNumber of Renews sent: %u.\n",numRenewSent);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the DHCP Client Statistics on an interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Clear the DHCP Client Statistics on an interface.
*
* @param    client_handle @b{(input)} Client handle from registration API
* @param    ifNum         @b{(input)} Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void dhcpClientStatsClear(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiDhcpClientStatsClear(client_handle, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared the DHCP Client Statistics on the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the DHCP Client Statistics on an interface. (result = %d)\n", result);
         break;
  }
  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for IP4DHCPClient
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum = 0;
  uint32_t ifNum = 0;
  uint32_t value = 0;
  uint32_t mode = 0;
  uint32_t mgmtPortType = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("IP4DHCPClient example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IP4DHCPClient API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Get the DHCP Vendor Option Mode: ip4dhcpclient_example 1 */
      if (argc == 2)
      {
        dhcpVendorClassOptionAdminModeGet(&client_handle);
        show_help = 0;
      }
      break;
    case 2: /* Test 2: Set the DHCP Vendor Option Mode: ip4dhcpclient_example 2 <mode> <actImmediate> */
      if (argc == 4)
      {
        mode = atoi(argv[2]);
        value = atoi(argv[3]);
        dhcpVendorClassOptionAdminModeSet(&client_handle, mode, value);
        show_help = 0;
      }
      break;
    case 3: /* Test 3: Get the DHCP Vendor Class Option string: ip4dhcpclient_example 3 */
      if (argc == 2)
      {
        dhcpVendorClassOptionStringGet(&client_handle);
        show_help = 0;
      }
      break;
    case 4: /* Test 4: Set the DHCP Vendor Class Option string: ip4dhcpclient_example 4 <option-string> <actImmediateFlag> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        dhcpVendorClassOptionStringSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 5: /* Test 5: Get the DHCP server IP address on a management interface: ip4dhcpclient_example 5 <ifNum> <mgmtPortType> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mgmtPortType = atoi(argv[3]);
        ipDhcpClientDhcpServerIntfIPAddressGet(&client_handle, ifNum, mgmtPortType);
        show_help = 0;
      }
      break;
    case 6: /* Test 6: Get the current state of the DHCP client for specified interface or management port: ip4dhcpclient_example 6 <ifNum> <mgmtPortType> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mgmtPortType = atoi(argv[3]);
        ipDhcpClientIntfStateGet(&client_handle, ifNum, mgmtPortType);
        show_help = 0;
      }
      break;
    case 7: /* Test 7: Get the transaction ID of the DHCP client for specified interface or management port: ip4dhcpclient_example 7 <ifNum> <mgmtPortType> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mgmtPortType = atoi(argv[3]);
        ipDhcpClientTransactionIntfIdGet(&client_handle, ifNum, mgmtPortType);
        show_help = 0;
      }
      break;
    case 8: /* Test 8: Get the lease time information of the DHCP client for specified interface or management port: ip4dhcpclient_example 8 <ifNum> <mgmtPortType> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mgmtPortType = atoi(argv[3]);
        ipDhcpClientLeaseTimeIntfInfoGet(&client_handle, ifNum, mgmtPortType);
        show_help = 0;
      }
      break;
    case 9: /* Test 9: Get the retry count of the DHCP client for specified interface or management port: ip4dhcpclient_example 9 <ifNum> <mgmtPortType> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mgmtPortType = atoi(argv[3]);
        ipDhcpClientIntfRetryCountGet(&client_handle, ifNum, mgmtPortType);
        show_help = 0;
      }
      break;
    case 10: /* Test 10: Get the generated client identifier on the management interface: ip4dhcpclient_example 10 <ifNum> <mgmtPortType> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mgmtPortType = atoi(argv[3]);
        ipDhcpClientIdGeneratedGet(&client_handle, ifNum, mgmtPortType);
        show_help = 0;
      }
      break;
    case 11: /* Test 11: Get the DHCP Client Statistics on an interface: ip4dhcpclient_example 11 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        dhcpClientStatsGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 12: /* Test 12: Clear the DHCP Client Statistics on an interface: ip4dhcpclient_example 12 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        dhcpClientStatsClear(&client_handle, ifNum);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IP4DHCPClient API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

