/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ip6map_example.c
*
* @purpose   OpEN IP6Map example.
*
* @component OpEN
*
* @create    01/10/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ip6map.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the IP6Map Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the administrative mode of the router for given VRF: %s 1 <vrfName>\n", name);
  printf("Test 2: Set the administrative mode of the router for given VRF: %s 2 <vrfName> <mode>\n", name);
  printf("Test 3: Get the dynamic renewal mode of IPv6 neighbors in a given VRF: %s 3 <vrfName>\n", name);
  printf("Test 4: Set the dynamic renewal mode of IPv6 neighbors in a given VRF: %s 4 <vrfName> <mode>\n", name);
  printf("Test 5: Get the NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF: %s 5 <vrfName>\n", name);
  printf("Test 6: Set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for a given VRF: %s 6 <vrfName> <val>\n", name);
  printf("Test 7: Get the maximum unicast neighbor solicitations sent during NUD for a given VRF: %s 7 <vrfName>\n", name);
  printf("Test 8: Set the maximum unicast neighbor solicitations sent during NUD for a given VRF: %s 8 <vrfName> <val>\n", name);
  printf("Test 9: Get the maximum multicast neighbor solicitations sent during NUD for a given VRF: %s 9 <vrfName>\n", name);
  printf("Test 10: Set the maximum multicast neighbor solicitations sent during NUD for a given VRF: %s 10 <vrfName> <val>\n", name);
  printf("Test 11: Get the rate limit value set for the unresolved IPv6 data packets coming to CPU: %s 11\n", name);
  printf("Test 12: Set the rate limit value for the unresolved IPv6 data packets coming to CPU: %s 12 <val>\n", name);
  printf("Test 13: Get the default IPv6 router TTL value for a given VRF: %s 13 <vrfName>\n", name);
  printf("Test 14: Set the default IPv6 router TTL value for a given VRF: %s 14 <vrfName> <val>\n", name);
  printf("Test 15: Get the administrative state of a given interface: %s 15 <ifNum>\n", name);
  printf("Test 16: Get the operational state of a given interface: %s 16 <ifNum>\n", name);
  printf("Test 17: Set the administrative state of a given interface: %s 17 <ifNum> <mode>\n", name);
  printf("Test 18: Get the IPv6 AutoState mode of a given interface: %s 18 <ifNum>\n", name);
  printf("Test 19: Set the IPv6 AutoState mode of a given interface: %s 19 <ifNum> <mode>\n", name);
  printf("Test 20: Determine if IPv6 interface exists: %s 20 <ifNum>\n", name);
  printf("Test 21: Determine if the interface is valid for participation in the IPv6 component: %s 21 <ifNum>\n", name);
  printf("Test 22: Determine if the interface is valid to be configured with an IPv6 neighbor: %s 22 <ifNum>\n", name);
  printf("Test 23: Get the IPv6 MTU of a given interface: %s 23 <ifNum>\n", name);
  printf("Test 24: Get the IPv6 effective MTU of a given interface: %s 24 <ifNum>\n", name);
  printf("Test 25: Set the IPv6 MTU of a given interface: %s 25 <ifNum> <val>\n", name);
  printf("Test 26: Get the interface bandwidth of a given interface: %s 26 <ifNum>\n", name);
  printf("Test 27: Get the first valid interface for participation in IPv6 component: %s 27\n", name);
  printf("Test 28: Get the next valid interface for participation in IPv6 component: %s 28 <prevIfNum>\n", name);
  printf("Test 29: Get the maximum IP MTU that may be set on an interface: %s 29 <ifNum>\n", name);

  return;
}

/*********************************************************************
* @purpose  Get the IPv6 administrative mode of the router for given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrAdminModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrRtrAdminModeGet(client_handle, &vrfNameBufd, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 administrative mode of the router for given VRF is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 administrative mode of the router for given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 administrative mode of the router for given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrAdminModeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < mode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6VrRtrAdminModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 administrative mode of the router for given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the IPv6 administrative mode of the router for given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the dynamic renewal mode of IPv6 neighbors in a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNbrsDynRenewGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNbrsDynRenewGet(client_handle, &vrfNameBufd, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe dynamic renewal mode of IPv6 neighbors for default VRF is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the dynamic renewal mode of IPv6 neighbors in a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the dynamic renewal mode of IPv6 neighbors in a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    mode          @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNbrsDynRenewSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < mode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6VrNbrsDynRenewSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the dynamic renewal mode of IPv6 neighbors in the given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the dynamic renewal mode of IPv6 neighbors in a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the NUD backoff-multiple value set for calculation of
*           timeouts for NS transmissions during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudBackoffMultipleGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The backoff multiple value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudBackoffMultipleGet(client_handle, &vrfNameBufd, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for given VRF is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the NUD backoff-multiple value for calculation of timeouts
*           for NS transmissions during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    val           @b{(input)}  The backoff multiple value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudBackoffMultipleSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudBackoffMultipleSet(client_handle, &vrfNameBufd, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the maximum unicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxUnicastSolicitsGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The maximum unicast solicits value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxUnicastSolicitsGet(client_handle, &vrfNameBufd, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum unicast neighbor solicitations sent during NUD for the given VRF is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the maximum unicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the maximum unicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    val            @b{(input)}  The maximum unicast solicits value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxUnicastSolicitsSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxUnicastSolicitsSet(client_handle, &vrfNameBufd, val);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the maximum unicast neighbor solicitations sent during NUD for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the maximum unicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the maximum multicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    val           @b{(output)} The maximum multicast solicits value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxMcastSolicitsGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The maximum multicast solicits value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxMcastSolicitsGet(client_handle, &vrfNameBufd, &val);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum multicast neighbor solicitations sent during NUD for the given VRF is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the maximum multicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the maximum multicast neighbor solicitations sent during NUD for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
* @param    val           @b{(input)}  The maximum multicast solicits value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrNudMaxMcastSolicitsSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrNudMaxMcastSolicitsSet(client_handle, &vrfNameBufd, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the maximum multicast neighbor solicitations sent during NUD for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the maximum multicast neighbor solicitations sent during NUD for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the rate limit value set for the unresolved IPv6 data packets coming to CPU.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ip6UnresolvedRateLimitGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The bandwidth value in bytes */

  result = openapiIp6UnresolvedRateLimitGet(client_handle, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe rate limit value set for the unresolved IPv6 data packets coming to CPU is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the rate limit value set for the unresolved IPv6 data packets coming to CPU. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the rate limit value for the unresolved IPv6 data packets coming to CPU.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    val           @b{(input)}  The rate-limt value in pps(packet-per-second)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6UnresolvedRateLimitSet(openapiClientHandle_t *client_handle, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6UnresolvedRateLimitSet(client_handle, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the rate limit value for the unresolved IPv6 data packets coming to CPU.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the rate limit value for the unresolved IPv6 data packets coming to CPU. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the default IPv6 router TTL value for a given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrDefaultTTLGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t val = 0; /* The TTL value */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrRtrDefaultTTLGet(client_handle, &vrfNameBufd, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe default IPv6 router TTL value for the given VRF is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the default IPv6 router TTL value for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the default IPv6 router TTL value for a given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    vrfName        @b{(input)}  VRF name
* @param    val            @b{(input)}  The TTL value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6VrRtrDefaultTTLSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIp6VrRtrDefaultTTLSet(client_handle, &vrfNameBufd, val);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the default IPv6 router TTL value for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the default IPv6 router TTL value for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 administrative state of a given interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIp6RtrIntfModeGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 administrative state of a given interface is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 administrative state of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the operational state of a given interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIp6RtrIntfOperModeGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe operational state of a given interface is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the operational state of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 administrative state of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;

  if (1 < mode)
  {
    printf("\n Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6RtrIntfModeSet(client_handle, ifNum, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 administrative state of the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv6 administrative state of the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 AutoState mode of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfAutoStateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIp6RtrIntfAutoStateModeGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 AutoState mode of a given interface is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 AutoState mode of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 AutoState mode of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfAutoStateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t autoStateMode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;

  if (1 < autoStateMode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == autoStateMode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIp6RtrIntfAutoStateModeSet(client_handle, ifNum, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 AutoState mode of the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv6 AutoState mode of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if IPv6 interface exists.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6IntfExists(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  OPEN_BOOL_t result = OPEN_FALSE;

  result = openapiIp6IntfExists(client_handle, ifNum);
  switch(result)
  {
    case OPEN_TRUE:
         printf("\nTRUE.\n");
         break;

    case OPEN_FALSE:
         printf("\nFALSE.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if IPv6 interface exists. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if the interface is valid for participation in the IPv6 component.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  OPEN_BOOL_t result = OPEN_FALSE;

  result = openapiIp6MapIntfIsValid(client_handle, ifNum);
  switch(result)
  {
    case OPEN_TRUE:
         printf("\nTRUE.\n");
         break;

    case OPEN_FALSE:
         printf("\nFALSE.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if the interface is valid for participation in the IPv6 component. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine if the interface is valid to be configured with an IPv6 neighbor.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapNbrIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  OPEN_BOOL_t result = OPEN_FALSE;

  result = openapiIp6MapNbrIntfIsValid(client_handle, ifNum);
  switch(result)
  {
    case OPEN_TRUE:
         printf("\nTRUE.\n");
         break;

    case OPEN_FALSE:
         printf("\nFALSE.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to determine if the interface is valid to be configured with an IPv6 neighbor. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 MTU of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The IPv6 MTU value in bytes */

  result = openapiIp6RtrIntfIpv6MtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 MTU of the given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 MTU of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IPv6 effective MTU of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfIpv6EffectiveMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The IPv6 effective MTU value in bytes */

  result = openapiIp6RtrIntfIpv6EffectiveMtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 effective MTU of the given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IPv6 effective MTU of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the maximum IP MTU that may be set on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfMaxIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The IPv6 maximum MTU value in bytes */

  result = openapiIp6RtrIntfMaxIpv6MtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum IPv6 MTU that may be set on the given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the maximum IPv6 MTU that may be set on the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv6 MTU of a given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    val            @b{(input)}  The IPv6 MTU value in bytes
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfIpv6MtuSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrIntfIpv6MtuSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 MTU of the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv6 MTU of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface bandwidth of a given interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrIntfBandwidthGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* The bandwidth value in bytes */

  result = openapiIp6RtrIntfBandwidthGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface bandwidth of a given interface is %u.\n", val);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the interface bandwidth of a given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the first valid interface for participation in IPv6 component.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapIntfValidFirstGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0; /* First Internal Interface Number */

  result = openapiIp6MapIntfValidFirstGet(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe first valid interface for participation in IPv6 component is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the first valid interface for participation in IPv6 component. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next valid interface for participation in IPv6 component.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    prevIfNum      @b{(input)}  Previous Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6MapIntfValidNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t nextIfNum = 0; /* Next Internal Interface Number */

  result = openapiIp6MapIntfValidNextGet(client_handle, prevIfNum, &nextIfNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next valid interface for participation in IPv6 component is %u.\n", nextIfNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the next valid interface for participation in IPv6 component. (result = %d)\n", result);
         break;
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for IP6Map
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  uint32_t value = 0;
  uint32_t ifNum = 0;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("IP6Map example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IP6Map API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Get the administrative mode of the router for given VRF: ip6map_example 1 <vrfName> */
      if (argc == 3)
      {
        ip6VrRtrAdminModeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 2: /* Test 2: Set the administrative mode of the router for given VRF: ip6map_example 2 <vrfName> <mode> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrRtrAdminModeSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 3: /* Test 3: Get the dynamic renewal mode of IPv6 neighbors in a given VRF: ip6map_example 3 <vrfName> */
      if (argc == 3)
      {
        ip6VrNbrsDynRenewGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 4: /* Test 4: Set the dynamic renewal mode of IPv6 neighbors in a given VRF: ip6map_example 4 <vrfName> <mode> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNbrsDynRenewSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 5: /* Test 5: Get the NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF: ip6map_example 5 <vrfName> */
      if (argc == 3)
      {
        ip6VrNudBackoffMultipleGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 6: /* Test 6: Set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for a given VRF: ip6map_example 6 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNudBackoffMultipleSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 7: /* Test 7: Get the maximum unicast neighbor solicitations sent during NUD for a given VRF: ip6map_example 7 <vrfName> */
      if (argc == 3)
      {
        ip6VrNudMaxUnicastSolicitsGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 8: /* Test 8: Set the maximum unicast neighbor solicitations sent during NUD for a given VRF: ip6map_example 8 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNudMaxUnicastSolicitsSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 9: /* Test 9: Get the maximum multicast neighbor solicitations sent during NUD for a given VRF: %s 9 <vrfName> */
      if (argc == 3)
      {
        ip6VrNudMaxMcastSolicitsGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 10: /* Test 10: Set the maximum multicast neighbor solicitations sent during NUD for a given VRF: ip6map_example 10 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrNudMaxMcastSolicitsSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 11: /* Test 11: Get the rate limit value set for the unresolved IPv6 data packets coming to CPU: ip6map_example 11 */
      if (argc == 2)
      {
        ip6UnresolvedRateLimitGet(&client_handle);
        show_help = 0;
      }
      break;
    case 12: /* Test 12: Set the rate limit value for the unresolved IPv6 data packets coming to CPU: ip6map_example 12 <val> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        ip6UnresolvedRateLimitSet(&client_handle, value);
        show_help = 0;
      }
      break;
    case 13: /* Test 13: Get the default IPv6 router TTL value for a given VRF: ip6map_example 13 <vrfName> */
      if (argc == 3)
      {
        ip6VrRtrDefaultTTLGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 14: /* Test 14: Set the default IPv6 router TTL value for a given VRF: ip6map_example 14 <vrfName> <val> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        ip6VrRtrDefaultTTLSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 15: /* Test 15: Get the administrative state of a given interface: ip6map_example 15 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 16: /* Test 16: Get the operational state of a given interface: ip6map_example 16 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfOperModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 17: /* Test 17: Set the administrative state of a given interface: ip6map_example 17 <ifNum> <mode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfModeSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 18: /* Test 18: Get the IPv6 AutoState mode of a given interface: ip6map_example 18 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfAutoStateModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 19: /* Test 19: Set the IPv6 AutoState mode of a given interface: ip6map_example 19 <ifNum> <mode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfAutoStateModeSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 20: /* Test 20: Determine if IPv6 interface exists: ip6map_example 20 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6IntfExists(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 21: /* Test 21: Determine if the interface is valid for participation in the IPv6 component: ip6map_example 21 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6MapIntfIsValid(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 22: /* Test 22: Determine if the interface is valid to be configured with an IPv6 neighbor: ip6map_example 22 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6MapNbrIntfIsValid(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 23: /* Test 23: Get the IPv6 MTU of a given interface: ip6map_example 23 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfIpv6MtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 24: /* Test 24: Get the IPv6 effective MTU of a given interface: ip6map_example 24 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfIpv6EffectiveMtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 25: /* Test 25: Set the IPv6 MTU of a given interface: ip6map_example 25 <ifNum> <val> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        ip6RtrIntfIpv6MtuSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 26: /* Test 26: Get the interface bandwidth of a given interface: ip6map_example 26 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfBandwidthGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 27: /* Test 27: Get the first valid interface for participation in IPv6 component: ip6map_example 27 */
      if (argc == 2)
      {
        ip6MapIntfValidFirstGet(&client_handle);
        show_help = 0;
      }
      break;
    case 28: /* Test 28: Get the next valid interface for participation in IPv6 component: ip6map_example 28 <prevIfNum> */
      if (argc == 3)
      {
        value = atoi(argv[2]);
        ip6MapIntfValidNextGet(&client_handle, value);
        show_help = 0;
      }
      break;
    case 29: /* Test 29: Get the maximum IP MTU that may be set on an interface: ip6map_example 29 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrIntfMaxIpv6MtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IP6Map API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

