/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ip6radv_example.c
*
* @purpose   OpEN IP6RADV example.
*
* @component OpEN
*
* @create    01/24/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ip6radv.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the IP6RADV Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Set IPv6 router advertise Minimum advertisement interval: %s 1 <ifNum> <val>\n", name);
  printf("Test 2: Get IPv6 router advertise Minimum advertisement interval: %s 2 <ifNum>\n", name);
  printf("Test 3: Set IPv6 router advertise Maximum advertisement interval: %s 3 <ifNum> <val>\n", name);
  printf("Test 4: Get IPv6 router advertise Maximum advertisement interval: %s 4 <ifNum>\n", name);
  printf("Test 5: Set IPv6 router advertise lifetime: %s 5 <ifNum> <val>\n", name);
  printf("Test 6: Get IPv6 router advertise lifetime: %s 6 <ifNum>\n", name);
  printf("Test 7: Set IPv6 router advertise link mtu: %s 7 <ifNum> <val>\n", name);
  printf("Test 8: Get IPv6 router advertise link mtu: %s 8 <ifNum>\n", name);
  printf("Test 9: Set IPv6 router preference: %s 9 <ifNum> <val>\n", name);
  printf("Test 10: Get IPv6 router preference: %s 10 <ifNum>\n", name);
  printf("Test 11: Set IPv6 router advertise managed flag: %s 11 <ifNum> <flag>\n", name);
  printf("Test 12: Get IPv6 router advertise managed flag: %s 12 <ifNum>\n", name);
  printf("Test 13: Set IPv6 router advertise other-config flag: %s 13 <ifNum> <flag>\n", name);
  printf("Test 14: Get IPv6 router advertise other-config flag: %s 14 <ifNum>\n", name);
  printf("Test 15: Set IPv6 hop limit advertised on the interface to unspecified(0): %s 15 <ifNum> <flag>\n", name);
  printf("Test 16: Get whether IPv6 hop limit advertised on the interface to unspecified(0): %s 16 <ifNum>\n", name);
  printf("Test 17: Set IPv6 router advertise suppression: %s 17 <ifNum> <flag>\n", name);
  printf("Test 18: Get IPv6 router advertise suppression: %s 18 <ifNum>\n", name);
  printf("Test 19: Set IPv6 router as per prefix attributes: %s 19 <ifNum><prefix><plen><prefLife><validLife><lifeDecrFlags><onLink><autonomous>\n", name);
  printf("Test 20: Get IPv6 router as per prefix attributes: %s 20 <ifNum> <prefix> <plen>\n", name);

  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise Minimum advertisement interval.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    val           @b{(input)}  Value in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvMinAdvIntSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvMinAdvIntSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 router advertise Minimum advertisement interval.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise Minimum advertisement interval. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise Minimum advertisement interval.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvMinAdvIntGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  uint32_t val = 0; /* value in seconds */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvMinAdvIntGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise Minimum advertisement interval is %u seconds.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise Minimum advertisement interval. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise Maximum advertisement interval.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    val           @b{(input)}  Value in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvMaxAdvIntSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvMaxAdvIntSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 router advertise Maximum advertisement interval.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise Maximum advertisement interval. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise Maximum advertisement interval.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvMaxAdvIntGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  uint32_t val = 0; /* Value in seconds */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvMaxAdvIntGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise Maximum advertisement interval is %u seconds.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise Maximum advertisement interval. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise lifetime.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    val           @b{(input)}  Value in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvAdvLifetimeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvAdvLifetimeSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 router advertise lifetime.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise lifetime. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise lifetime.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvAdvLifetimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  uint32_t val = 0; /* Value in seconds */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvAdvLifetimeGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise lifetime is %u seconds.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise lifetime. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise link mtu.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    val           @b{(input)}  MTU value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvAdvMtuSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvAdvMtuSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 router advertise link mtu.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise link mtu. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise link mtu.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvAdvMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  uint32_t val = 0; /* MTU value */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvAdvMtuGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise link mtu is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise link mtu. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router preference.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    val           @b{(input)}  Preference value
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvRtrPrefSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvRtrPrefSet(client_handle, ifNum, val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv6 router preference.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router preference. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router preference.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvRtrPrefGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  uint32_t val = 0; /* Preference value */
  open_error_t result = OPEN_E_FAIL;

  result = openapiIp6RtrRadvRtrPrefGet(client_handle, ifNum, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router preference is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router preference. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise managed flag.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    flag          @b{(input)}  OPEN_TRUE(1) or OPEN_FALSE(0)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvManagedFlagSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t flag)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (1 < flag)
  {
    printf("\nERROR: Invalid flag value. Expected 0(for False) or 1(for True).\n");
    return;
  }

  if (1 == flag)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIp6RtrRadvManagedFlagSet(client_handle, ifNum, openFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set IPv6 router advertise managed flag.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise managed flag. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise managed flag.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvManagedFlagGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t flag = OPEN_FALSE; /* (OPEN_TRUE/OPEN_FALSE) */

  result = openapiIp6RtrRadvManagedFlagGet(client_handle, ifNum, &flag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise managed flag of given interface is %u.\n", flag);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise managed flag. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise other-config flag.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    flag          @b{(input)}  OPEN_TRUE(1) or OPEN_FALSE(0)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvOtherCfgFlagSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t flag)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (1 < flag)
  {
    printf("\nERROR: Invalid flag value. Expected 0(for False) or 1(for True).\n");
    return;
  }

  if (1 == flag)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIp6RtrRadvOtherCfgFlagSet(client_handle, ifNum, openFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set IPv6 router advertise other-config flag.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise other-config flag. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise other-config flag.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvOtherCfgFlagGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t flag = OPEN_FALSE; /* (OPEN_TRUE/OPEN_FALSE) */

  result = openapiIp6RtrRadvOtherCfgFlagGet(client_handle, ifNum, &flag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise other-config flag of given interface is %u.\n", flag);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise other-config flag. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 hop limit advertised on the interface to unspecified(0).
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    flag          @b{(input)}  OPEN_TRUE(1) or OPEN_FALSE(0)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvHopLimitUnspec(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t flag)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (1 < flag)
  {
    printf("\nERROR: Invalid flag value. Expected 0(for False) or 1(for True).\n");
    return;
  }

  if (1 == flag)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIp6RtrRadvHopLimitUnspec(client_handle, ifNum, openFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set IPv6 hop limit advertised on the interface to unspecified(0).\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 hop limit advertised on the interface to unspecified(0). (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get whether IPv6 hop limit advertised on the interface to unspecified(0)
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvHopLimitUnspecGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t flag = OPEN_FALSE; /* (OPEN_TRUE/OPEN_FALSE) */

  result = openapiIp6RtrRadvHopLimitUnspecGet(client_handle, ifNum, &flag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise other-config flag of given interface is %u.\n", flag);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get whether IPv6 hop limit is advertised on given interface to unspecified(0) (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router advertise suppression.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    flag          @b{(input)}  OPEN_TRUE(1) or OPEN_FALSE(0)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvSuppressSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t flag)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (1 < flag)
  {
    printf("\nERROR: Invalid flag value. Expected 0(for False) or 1(for True).\n");
    return;
  }

  if (1 == flag)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIp6RtrRadvSuppressSet(client_handle, ifNum, openFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set IPv6 router advertise suppression.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router advertise suppression. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router advertise suppression
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvSuppressGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t flag = OPEN_FALSE; /* (OPEN_TRUE/OPEN_FALSE) */

  result = openapiIp6RtrRadvSuppressGet(client_handle, ifNum, &flag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv6 router advertise managed flag of given interface is %u.\n", flag);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router advertise suppression (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set IPv6 router as per prefix attributes.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    *prefix       @b{(input)}  Pointer to prefix
* @param    plen          @b{(input)}  Prefix length
* @param    prefLife      @b{(input)}  Preferred life
* @param    validLife     @b{(input)}  Valid life
* @param    lifeDecrFlags @b{(input)}  Flags for lifetime decrement mode
* @param    onLink        @b{(input)}  Onlink flag (OPEN_TRUE or OPEN_FALSE)
* @param    autonomous    @b{(input)}  Autonomous flag (OPEN_TRUE or OPEN_FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvPrefixSet(openapiClientHandle_t *client_handle, uint32_t ifNum, open_inet_addr_t *prefix,
                         uint32_t plen, uint32_t prefLife, uint32_t validLife, uint16_t lifeDecrFlags,
                         uint32_t onLink, uint32_t autonomous)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t openOnLinkFlag = OPEN_FALSE;
  OPEN_BOOL_t openAutoFlag = OPEN_FALSE;

  if (1 < onLink)
  {
    printf("\nERROR: Invalid onLink flag value. Expected 0(for False) or 1(for True).\n");
    return;
  }
  if (1 == onLink)
  {
    openOnLinkFlag = OPEN_TRUE;
  }

  if (1 < autonomous)
  {
    printf("\nERROR: Invalid autonomous flag value. Expected 0(for False) or 1(for True).\n");
    return;
  }
  if (1 == autonomous)
  {
    openAutoFlag = OPEN_TRUE;
  }

  result = openapiIp6RtrRadvPrefixSet(client_handle, ifNum, *prefix, plen, prefLife, validLife, lifeDecrFlags, openOnLinkFlag, openAutoFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set IPv6 router as per prefix attributes.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set IPv6 router as per prefix attributes. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get IPv6 router as per prefix attributes.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
* @param    *prefix       @b{(input)}  Pointer to prefix
* @param    plen          @b{(input)}  Prefix length
* @param    *prefLife     @b{(output)}  Preferred life
* @param    *validLif     @b{(output)}  Valid life
* @param    *onLink       @b{(output)}  Onlink flag (OPEN_TRUE or OPEN_FALSE)
* @param    *autonomous   @b{(output)}  Autonomous flag (OPEN_TRUE or OPEN_FALSE)
*
* @returns  none
*
* @end
*********************************************************************/
void ip6RtrRadvPrefixGet(openapiClientHandle_t *client_handle, uint32_t ifNum, char *ipAddrStr, uint32_t plen)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t onLink = OPEN_FALSE; /* (OPEN_TRUE/OPEN_FALSE) */
  OPEN_BOOL_t autonomous = OPEN_FALSE; /* (OPEN_TRUE/OPEN_FALSE) */
  uint32_t prefLife  = 0;
  uint32_t validLife = 0;
  open_inet_addr_t prefix;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  if (OPEN_E_NONE != openapiInetAddrGet(client_handle, &ipBuffdesc, &prefix))
  {
    printf("\nERROR: Invalid IP-address input.\n");
    return;
  }

  result = openapiIp6RtrRadvPrefixGet(client_handle, ifNum, &prefix, plen, &prefLife, &validLife, &onLink, &autonomous);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing are the IPv6 router info as per given prefix attributes: ");
         printf("\n  Preferred life is %u seconds.", prefLife);
         printf("\n  Valid life is %u seconds.", validLife);
         printf("\n  Onlink flag value is %u.", onLink);
         printf("\n  Autonomous flag value is %u.\n", autonomous);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get IPv6 router as per prefix attributes. (result = %d)\n", result);
         break;
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for IP6RADV
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum = 0;
  uint32_t ifNum = 0;
  uint32_t val = 0;
  uint32_t plen = 0;
  uint32_t prefLife = 0;
  uint32_t validLife = 0;
  uint32_t lifeDecrFlags = 0;
  uint32_t onLink = 0;
  uint32_t autonomous = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("IP6RADV example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IP6RADV API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvMinAdvIntSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvMinAdvIntGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvMaxAdvIntSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvMaxAdvIntGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvAdvLifetimeSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvAdvLifetimeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvAdvMtuSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
	ifNum = atoi(argv[2]);
        ip6RtrRadvAdvMtuGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvRtrPrefSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvRtrPrefGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvManagedFlagSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvManagedFlagGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvOtherCfgFlagSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvOtherCfgFlagGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvHopLimitUnspec(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvHopLimitUnspecGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        val = atoi(argv[3]);
        ip6RtrRadvSuppressSet(&client_handle, ifNum, val);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ip6RtrRadvSuppressGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 10)
      {
        ifNum = atoi(argv[2]);
        plen = atoi(argv[4]);
        prefLife = atoi(argv[5]);
        validLife = atoi(argv[6]);
        lifeDecrFlags = atoi(argv[7]);
        onLink = atoi(argv[8]);
        autonomous = atoi(argv[9]);
        ip6RtrRadvPrefixSet(&client_handle, ifNum, argv[3], plen, prefLife, validLife, lifeDecrFlags, onLink, autonomous);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 5)
      {
        ifNum = atoi(argv[2]);
        plen  = atoi(argv[4]);
        ip6RtrRadvPrefixGet(&client_handle, ifNum, argv[3], plen);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IP6RADV API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

