
/*********************************************************************
*
*  Copyright 2012-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ipv4_device_tracking_example.c
*
* @purpose   IPv4DT APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    11/26/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ipv4_device_tracking.h"

/*******************************************************************
*
* @brief  This function prints the Ipv4dt Example Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printIpv4dtAppMenu()
{
  printf("Usage: ipv4dt_example <test#> <arg1> <arg2> ... \n");
  printf("Test  1: Set IPv4DT Admin mode: ipv4dt_example 1 <mode: 0-disable/1-enable> \n");
  printf("Test  2: Get IPv4DT Admin mode: ipv4dt_example 2 \n");
  printf("Test  3: Set IPv4DT Probe mode: ipv4dt_example 3 <mode: 0-disable/1-enable> \n");
  printf("Test  4: Get IPv4DT Probe mode: ipv4dt_example 4 \n");
  printf("Test  5: Set IPv4DT Probe count: ipv4dt_example 5 <count: 1-255> \n");
  printf("Test  6: Get IPv4DT Probe count: ipv4dt_example 6 \n");
  printf("Test  7: Set IPv4DT Probe interval: ipv4dt_example 7 <interval: 30-300 secs> \n");
  printf("Test  8: Get IPv4DT Probe interval: ipv4dt_example 8 \n");
  printf("Test  9: Set IPv4DT Probe delay: ipv4dt_example 9 <delay: 1-120> \n");
  printf("Test 10: Get IPv4DT Probe delay: ipv4dt_example 10 \n");
  printf("Test 11: Set IPv4DT maximum entries for an interface: ipv4dt_example 11 <interface> <max-value: 0-10, -1 for nolimit> \n");
  printf("Test 12: Get IPv4DT maximum entries for an interface: ipv4dt_example 12 <interface> \n");
  printf("Test 13: Set IPv4DT Auto-source: ipv4dt_example 13 <host> <mask> \n");
  printf("Test 14: Get IPv4DT Auto-source: ipv4dt_example 14 \n");
  printf("Test 15: Show IPv4DT entries count: ipv4dt_example 15 \n");
  printf("Test 16: Show all IPv4DT entries: ipv4dt_example 16 \n");
  printf("Test 17: Show all IPv4DT Active entries: ipv4dt_example 17 \n");
  printf("Test 18: Show all IPv4DT InActive entries: ipv4dt_example 18 \n");
  printf("Test 19: Show all IPv4DT entries learned on a interface: ipv4dt_example 19 <interface> \n");
  printf("Test 20: Show all IPv4DT entries per ip: ipv4dt_example 20 <ip>\n");
  printf("Test 21: Show all IPv4DT entries per mac: ipv4dt_example 21 <mac> \n");
  printf("Test 22: Clear all IPv4DT entries: ipv4dt_example 22 \n");
  printf("Test 23: Clear all IPv4DT entries learned on a interface: ipv4dt_example 23 <interface> \n");
  printf("Test 24: Clear all IPv4DT entries per ip: ipv4dt_example 24 <ip>\n");
  printf("Test 25: Clear all IPv4DT entries per mac: ipv4dt_example 25 <mac> \n");
  printf("Test 26: IPv4DT OpEN APIs Sanity: ipv4dt_example 26 \n\n");
  return;
}

/*********************************************************************
* @purpose  Set IPv4DT admin mode.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   admin mode
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiIpv4dtAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set IPv4DT admin mode. (result = %d) \n", result);
  }
  else
  {
    printf("IPv4DT Admin mode %s is set successfully. (result = %d) \n", (mode == OPEN_DISABLE) ? "Disable" : "Enable", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT admin mode.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4DtAdminModeGet(openapiClientHandle_t *clientHandle)
{
  OPEN_CONTROL_t  mode = OPEN_DISABLE;
  open_error_t    result = OPEN_E_NONE;

  if ((result = openapiIpv4dtAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get IPv4DT admin mode. (result = %d) \n", result);
  }
  else
  {
    printf("IPv4DT Admin mode: %s. (result = %d) \n", (mode == OPEN_DISABLE) ? "Disable" : "Enable", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set IPv4DT probe mode.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   probe mode
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtProbeModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiIpv4dtProbeModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set IPv4DT probe mode. (result = %d) \n", result);
  }
  else
  {
    printf("IPv4DT Probe mode %s is set successfully. (result = %d) \n", (mode == OPEN_DISABLE) ? "Disable" : "Enable", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT probe mode.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4DtProbeModeGet(openapiClientHandle_t *clientHandle)
{
  OPEN_CONTROL_t  mode = OPEN_DISABLE;
  open_error_t    result = OPEN_E_NONE;

  if ((result = openapiIpv4dtProbeModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get IPv4DT probe mode. (result = %d) \n", result);
  }
  else
  {
    printf("IPv4DT Probe mode: %s. (result = %d) \n", (mode == OPEN_DISABLE) ? "Disable" : "Enable", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set IPv4DT Probe count
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    count            @b{(input)}  IPv4DT Probe Count
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtProbeCountSet(openapiClientHandle_t *clientHandle,
                         uint32_t count)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiIpv4dtProbeCountSet(clientHandle, count);
  if (result == OPEN_E_NONE)
  {
    printf("IPv4DT Probe Count %d is set successfully. (result = %d) \n",
           count, result);
  }
  else if (result == OPEN_E_PARAM)
  {
    printf("Probe Count %d is not in range limits. (result = %d) \n",
           count, result);
  }
  else
  {
    printf("Bad return code trying to set IPv4DT Probe Count %d. (result = %d) \n",
           count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT Probe Count.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4DtProbeCountGet(openapiClientHandle_t *clientHandle)
{
  uint32_t      count = 0;
  open_error_t  result = OPEN_E_NONE;

  if ((result = openapiIpv4dtProbeCountGet(clientHandle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get IPv4DT Probe count. (result = %d) \n",
           result);
  }
  else
  {
    printf("IPv4DT Probe Count : %d. (result = %d) \n", count, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set IPv4DT Probe interval
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    interval         @b{(input)}  IPv4DT Probe Interval
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtProbeIntervalSet(openapiClientHandle_t *clientHandle,
                            uint32_t interval)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiIpv4dtProbeIntervalSet(clientHandle, interval);
  if (result == OPEN_E_NONE)
  {
    printf("IPv4DT Probe Interval %d is set successfully. (result = %d) \n",
           interval, result);
  }
  else if (result == OPEN_E_PARAM)
  {
    printf("Probe Interval %d is not in range limits. (result = %d) \n",
           interval, result);
  }
  else
  {
    printf("Bad return code trying to set IPv4DT Probe Interval %d. (result = %d) \n",
           interval, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT Probe Interval.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4DtProbeIntervalGet(openapiClientHandle_t *clientHandle)
{
  uint32_t      interval = 0;
  open_error_t  result = OPEN_E_NONE;

  if ((result = openapiIpv4dtProbeIntervalGet(clientHandle, &interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get IPv4DT Probe interval. (result = %d) \n",
           result);
  }
  else
  {
    printf("IPv4DT Probe Interval : %d. (result = %d) \n", interval, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set IPv4DT Probe delay
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    delay         @b{(input)}  IPv4DT Probe Delay
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtProbeDelaySet(openapiClientHandle_t *clientHandle,
                         uint32_t delay)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiIpv4dtProbeDelaySet(clientHandle, delay);
  if (result == OPEN_E_NONE)
  {
    printf("IPv4DT Probe Delay %d is set successfully. (result = %d) \n",
           delay, result);
  }
  else if (result == OPEN_E_PARAM)
  {
    printf("Probe Delay %d is not in range limits. (result = %d) \n",
           delay, result);
  }
  else
  {
    printf("Bad return code trying to set IPv4DT Probe Delay %d. (result = %d) \n",
           delay, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT Probe Delay.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4DtProbeDelayGet(openapiClientHandle_t *clientHandle)
{
  uint32_t      delay = 0;
  open_error_t  result = OPEN_E_NONE;

  if ((result = openapiIpv4dtProbeDelayGet(clientHandle, &delay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get IPv4DT Probe delay. (result = %d) \n",
           result);
  }
  else
  {
    printf("IPv4DT Probe Delay : %d. (result = %d) \n", delay, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set IPv4DT Max Entries on a interface
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    intfNum          @b{(input)}  Interface Number
* @param    intfMax          @b{(input)}  Max Entries
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtIntfMaxSet(openapiClientHandle_t *clientHandle,
                      uint32_t intfNum, int32_t intfMax)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiIpv4dtIntfMaxSet(clientHandle, intfNum, intfMax);
  if (result == OPEN_E_NONE)
  {
    printf("IPv4DT Intf Max: %d is set successfully on an interface %d. (result = %d) \n",
           intfMax, intfNum, result);
  }
  else if (result == OPEN_E_UNAVAIL)
  {
    printf("Interface %d is not supported for IPv4DT. (result = %d) \n",
           intfNum, result);
  }
  else if (result == OPEN_E_PARAM)
  {
    printf("Interface Maximum %d is not in limits. (result = %d) \n",
           intfMax, result);
  }
  else
  {
    printf("Bad return code trying to set IPv4DT Max entries %d on an interface %d. (result = %d) \n",
           intfMax, intfNum, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT Max Entries for a interface
*
* @param    clientHandle     @b{(input)}  client handle from registration API
* @param    intfNum          @b{(input)}  Interface Number
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtIntfMaxGet(openapiClientHandle_t *clientHandle,
                      uint32_t intfNum)
{
  open_error_t  result = OPEN_E_NONE;
  uint32_t       intfMax = -1;

  result = openapiIpv4dtIntfMaxGet(clientHandle, intfNum, &intfMax);
  if (result == OPEN_E_NONE)
  {
    if (intfMax == -1)
    {
      printf("IPv4DT Interface Max is Unlimited. (result = %d) \n", result);

    }
    else
    {
      printf("IPv4DT Intf Max: %d on an interface %d. (result = %d) \n",
             intfMax, intfNum, result);
    }
  }
  else if (result == OPEN_E_UNAVAIL)
  {
    printf("Interface %d is not supported for IPv4DT. (result = %d) \n",
           intfNum, result);
  }
  else
  {
    printf("Bad return code trying to Get IPv4DT Max entries on an interface %d. (result = %d) \n",
           intfNum, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set IPv4DT Probe Auto-source parameters
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    host           @b{(input)}  IPv4DT Autosource host ip
* @param    mask           @b{(input)}  IPv4DT Autosource mask
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @end
*********************************************************************/
void ipv4DtProbeAutoSourceSet(openapiClientHandle_t *clientHandle,
                              char *host, char *mask)
{
  open_error_t  result = OPEN_E_NONE;
  uint32_t      ipv4dtHost = 0;
  uint32_t      ipv4dtMask = 0;
  char          strHost[80] = {0};
  char          strMask[80] = {0};

  inet_pton(AF_INET, host, &ipv4dtHost);
  inet_pton(AF_INET, mask, &ipv4dtMask);    

  if ((result = openapiIpv4dtProbeAutoSourceSet(clientHandle, &ipv4dtHost, &ipv4dtMask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set IPv4DT Probe Auto-source. (result = %d) \n",
           result);
  }
  else
  {
    inet_ntop(AF_INET, &ipv4dtHost, strHost, sizeof(strHost));
    inet_ntop(AF_INET, &ipv4dtMask, strMask, sizeof(strMask));
    printf("IPv4DT Probe Auto-source Host: %s, Mask: %s is set successfully. (result = %d) \n",
           strHost, strMask, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT Probe Auto-Source parameters
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4DtProbeAutoSourceGet(openapiClientHandle_t *clientHandle)
{
  uint32_t      ipv4dtHost = 0;
  uint32_t      ipv4dtMask = 0;
  char          strHost[80] = {0};
  char          strMask[80] = {0};

  open_error_t result = OPEN_E_NONE;

  if ((result = openapiIpv4dtProbeAutoSourceGet(clientHandle, &ipv4dtHost, &ipv4dtMask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get IPv4DT Probe delay. (result = %d) \n",
           result);
  }
  else
  {
    inet_ntop(AF_INET, &ipv4dtHost, strHost, sizeof(strHost));
    inet_ntop(AF_INET, &ipv4dtMask, strMask, sizeof(strMask));
    printf("IPv4DT Probe AutoSource Host: %s and Mask: %s. (result = %d) \n", strHost, strMask, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get IPv4DT Entries Count
*
* @param    clientHandle     @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipv4DtEntriesCountGet(openapiClientHandle_t *clientHandle)
{
  open_error_t                 result = OPEN_E_NONE;
  OPEN_IPV4DT_ENTRIES_COUNT_t  ipv4dtEntriesCount;

  memset(&ipv4dtEntriesCount, 0, sizeof(ipv4dtEntriesCount));

  result = openapiIpv4dtEntriesCountGet(clientHandle, &ipv4dtEntriesCount);
  if (result == OPEN_E_NONE)
  {
    printf("IP Device Tracking ARP Entries Count........... %u\n", ipv4dtEntriesCount.arpEntriesCount);
    printf("IP Device Tracking DHCP Entries Count.......... %u\n", ipv4dtEntriesCount.dhcpEntriesCount);
    printf("IP Device Tracking ACTIVE Entries Count........ %u\n", ipv4dtEntriesCount.activeEntriesCount);
    printf("IP Device Tracking INACTIVE Entries Count...... %u\n\n", ipv4dtEntriesCount.inactiveEntriesCount);
    printf("IP Device Tracking Total Entries Count......... %u\n\n", ipv4dtEntriesCount.totalEntriesCount);
  }
  else
  {
    printf("Bad return code trying to Get IPv4DT Entries Count. (result = %d) \n",
           result);
  }

  return;
}

/***************************************************************************
* @purpose  Prints the Ipv4dt Entry
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    entry           @b{(output)}  IPv4DT Entry
*
*
* @returns  none
*
* @end
*****************************************************************************/
void ipv4dtEntryPrint(openapiClientHandle_t *clientHandle, OPEN_IPV4DT_ENTRY_t *entry)
{
  char             vlanStr[5] = {0};
  char             intfStr[80] = {0};
  char             ttaStr[10] = {0};
  char             tsaStr[10] = {0};
  char             sourceStr[6] = {0};
  char             stateStr[10] = {0};
  char             ipStr[80] = {0};
  char             hostIpAddrStr[80] = {0};
  char             hostMacAddrStr[80] = {0};
  OPEN_CONTROL_t   probeMode = OPEN_DISABLE;
  open_buffdesc    ifName;
  char             ifNameStr[OPEN_INTF_NAME_MAX_LENGTH] = {0};
  uint32_t         hostAddress = 0;

  ifName.size = OPEN_INTF_NAME_MAX_LENGTH;
  ifName.pstart = ifNameStr;

  /* Host IP Address */
  hostAddress = ntohl(entry->hostIpAddr);
  inet_ntop(AF_INET, &hostAddress, ipStr, sizeof(ipStr));
  sprintf(hostIpAddrStr, "%-*s", 16, ipStr);

  /* Host MAC Address */
  sprintf(hostMacAddrStr, "%2.2X:%2.2X:%2.2X:%2.2X:%2.2X:%2.2X",
          entry->hostMacAddr[0], entry->hostMacAddr[1], entry->hostMacAddr[2],
          entry->hostMacAddr[3], entry->hostMacAddr[4], entry->hostMacAddr[5]);

  /* Interface Name */
  if (openapiIntfNameGet(clientHandle, entry->intIfNum, &ifName) == OPEN_E_NONE)
  {
    sprintf(intfStr, ifNameStr);
  }
  else
  {
    sprintf(intfStr, "None");
  }

  /* VLAN Id */
  sprintf(vlanStr, "%u", entry->vlanId);

  /* Time left to InActive state */
  (void)openapiIpv4dtAdminModeGet(clientHandle, &probeMode);

  if ((probeMode == OPEN_DISABLE) && (entry->state == OPEN_IPV4DT_STATE_ACTIVE))
  {
    sprintf(ttaStr, "Infinity");
  }
  else
  {
    sprintf(ttaStr, "%u", entry->timeToInActive);
  }
  
  /* Time since InActive state */
  sprintf(tsaStr, "%u", entry->timeSinceInActive);

  /* State */
  if (entry->state == OPEN_IPV4DT_STATE_ACTIVE)
  {
    sprintf(stateStr, "ACTIVE");
  }
  else
  {
    sprintf(stateStr, "INACTIVE");
  }

  /* Source */
  if (entry->source == OPEN_IPV4DT_SOURCE_ARP)
  {
    sprintf(sourceStr, "ARP");
  }
  else
  {
    sprintf(sourceStr, "DHCP");
  }

  printf("\r\n%-16s %18.18s %5.5s %10.10s %9.9s %11.11s %10.10s %6.6s",
         hostIpAddrStr, hostMacAddrStr, vlanStr, intfStr, stateStr, ttaStr, tsaStr, sourceStr);
}

/*********************************************************************
* @purpose  Get IPv4DT entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ipv4dtShow       @b{(input)}   ipv4dt Show request parameters
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShow(openapiClientHandle_t *clientHandle, OPEN_IPV4DT_SHOW_REQUEST_t *ipv4dtShow)
{
  open_error_t                 result = OPEN_E_NONE;
  OPEN_IPV4DT_ENTRY_t          entry;
  uint32_t                     counter = 0;

  memset(&entry, 0, sizeof(entry));

  while ((result = openapiIpv4dtTableGetNext(clientHandle, ipv4dtShow, &entry)) == OPEN_E_NONE)
  {
    if (counter == 0)
    {
      printf("\r\n%-16.16s %18.18s %5.5s %10.10s %9.9s "
             "%11.11s %10.10s %6.6s", "IP Address",
             "MAC Address","Vlan","Interface","State",
             "Time-left", "Time-since","Source");
      printf("\r\n%74.74s %10.10s","to inactive", "inactive");
      printf("\r\n%74.74s %10.10s","(in secs)", "(in secs)");
      printf("\r\n---------------- ------------------ ----- "
             "---------- --------- ----------- ---------- ------");
    }
    ipv4dtEntryPrint(clientHandle, &entry);
    counter++;
  }

  if (!counter)
  {
    printf("No Entries are found.\n");
  }
  else
  {
    printf("\n\n");
  }

  return;
}

/*********************************************************************
* @purpose  Get All IPv4DT entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShowAll(openapiClientHandle_t *clientHandle)
{
  OPEN_IPV4DT_SHOW_REQUEST_t   ipv4dtShow;

  memset(&ipv4dtShow, 0, sizeof(ipv4dtShow));
  ipv4dtShow.noFilter = OPEN_TRUE;
  ipv4dtShow.getAllType = OPEN_IPV4DT_GET_ALL;
 
  ipv4dtEntriesShow(clientHandle, &ipv4dtShow);
}

/*********************************************************************
* @purpose  Get All ACTIVE IPv4DT entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShowActive(openapiClientHandle_t *clientHandle)
{
  OPEN_IPV4DT_SHOW_REQUEST_t   ipv4dtShow;

  memset(&ipv4dtShow, 0, sizeof(ipv4dtShow));
  ipv4dtShow.noFilter = OPEN_TRUE;
  ipv4dtShow.getAllType = OPEN_IPV4DT_GET_ALL_ACTIVE;

  ipv4dtEntriesShow(clientHandle, &ipv4dtShow);
}

/*********************************************************************
* @purpose  Get All INACTIVE IPv4DT entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShowInactive(openapiClientHandle_t *clientHandle)
{
  OPEN_IPV4DT_SHOW_REQUEST_t   ipv4dtShow;

  memset(&ipv4dtShow, 0, sizeof(ipv4dtShow));
  ipv4dtShow.noFilter = OPEN_TRUE;
  ipv4dtShow.getAllType = OPEN_IPV4DT_GET_ALL_INACTIVE;

  ipv4dtEntriesShow(clientHandle, &ipv4dtShow);
}

/*********************************************************************
* @purpose  Get All IPv4DT entries learned on a interface.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    intfNum        @b{(input)}   Interface Number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShowInterface(openapiClientHandle_t *clientHandle,
                                uint32_t intfNum)
{
  OPEN_IPV4DT_SHOW_REQUEST_t   ipv4dtShow;

  memset(&ipv4dtShow, 0, sizeof(ipv4dtShow));
  ipv4dtShow.intfFilter = OPEN_TRUE;
  ipv4dtShow.intIfNum = intfNum;

  ipv4dtEntriesShow(clientHandle, &ipv4dtShow);
}

/*********************************************************************
* @purpose  Get All IPv4DT entries based on a IP Address.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    hostIp         @b{(input)}   Host ip address
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShowIp(openapiClientHandle_t *clientHandle,
                         char *hostIp)
{
  OPEN_IPV4DT_SHOW_REQUEST_t   ipv4dtShow;
  uint32_t                     hostIpAddr;

  memset(&ipv4dtShow, 0, sizeof(ipv4dtShow));

  inet_pton(AF_INET, hostIp, &hostIpAddr);
  ipv4dtShow.hostIpFilter = OPEN_TRUE;
  hostIpAddr = htonl(hostIpAddr);
  ipv4dtShow.hostIpAddr = hostIpAddr;

  ipv4dtEntriesShow(clientHandle, &ipv4dtShow);
}

/*********************************************************************
* @purpose  Get All IPv4DT entries based on a MAC Address.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    hostMac        @b{(input)}   Host mac address
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesShowMac(openapiClientHandle_t *clientHandle,
                          char *hostMac)
{
  OPEN_IPV4DT_SHOW_REQUEST_t   ipv4dtShow;
  uint32_t                     mac[OPEN_MAC_ADDR_LEN] = {0};
  uint32_t                     index = 0;
  
  memset(&ipv4dtShow, 0, sizeof(ipv4dtShow));

  sscanf(hostMac, "%x:%x:%x:%x:%x:%x", &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);
  for (index = 0; index < OPEN_MAC_ADDR_LEN; index++)
  {
    ipv4dtShow.hostMacAddr[index] = (unsigned char) mac[index];
  }
  ipv4dtShow.hostMacFilter = OPEN_TRUE;

  ipv4dtEntriesShow(clientHandle, &ipv4dtShow);
}

/*********************************************************************
* @purpose  Clear IPv4DT entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ipv4dtClear      @b{(input)}   ipv4dt Clear request parameters
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesClear(openapiClientHandle_t *clientHandle, OPEN_IPV4DT_CLEAR_REQUEST_t *ipv4dtClear)
{
  open_error_t  result = OPEN_E_NONE;

  if ((result = openapiIpv4dtClearEntries(clientHandle, ipv4dtClear)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear IPv4DT Entries. (result = %d) \n", result);
  }
  else
  {
    printf("IPv4DT Entries Clear is successfully. (result = %d) \n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Clear All IPv4DT entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesClearAll(openapiClientHandle_t *clientHandle)
{
  OPEN_IPV4DT_CLEAR_REQUEST_t   ipv4dtClear;

  memset(&ipv4dtClear, 0, sizeof(ipv4dtClear));
  ipv4dtClear.clear = OPEN_IPV4DT_CLEAR_ALL;

  ipv4dtEntriesClear(clientHandle, &ipv4dtClear);
}

/*********************************************************************
* @purpose  Clear All IPv4DT entries learned on a interface.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    intfNum        @b{(input)}   Interface Number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesClearInterface(openapiClientHandle_t *clientHandle,
                                 uint32_t intfNum)
{
  OPEN_IPV4DT_CLEAR_REQUEST_t   ipv4dtClear;

  memset(&ipv4dtClear, 0, sizeof(ipv4dtClear));
  ipv4dtClear.clear = OPEN_IPV4DT_CLEAR_INTERFACE;
  ipv4dtClear.intIfNum = intfNum;
  
  ipv4dtEntriesClear(clientHandle, &ipv4dtClear);
}

/*********************************************************************
* @purpose  Clear All IPv4DT entries based on a IP Address.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    hostIp         @b{(input)}   Host ip address
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesClearIp(openapiClientHandle_t *clientHandle,
                          char *hostIp)
{
  OPEN_IPV4DT_CLEAR_REQUEST_t  ipv4dtClear;
  uint32_t                     hostIpAddr;

  memset(&ipv4dtClear, 0, sizeof(ipv4dtClear));

  inet_pton(AF_INET, hostIp, &hostIpAddr);
  ipv4dtClear.clear = OPEN_IPV4DT_CLEAR_HOST_IP;
  hostIpAddr = htonl(hostIpAddr);
  ipv4dtClear.hostIpAddr = hostIpAddr;

  ipv4dtEntriesClear(clientHandle, &ipv4dtClear);
}

/*********************************************************************
* @purpose  Clear All IPv4DT entries based on a MAC Address.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    hostMac        @b{(input)}   Host mac address
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtEntriesClearMac(openapiClientHandle_t *clientHandle,
                           char *hostMac)
{
  OPEN_IPV4DT_CLEAR_REQUEST_t  ipv4dtClear;
  uint32_t                     mac[OPEN_MAC_ADDR_LEN] = {0};
  uint32_t                     index = 0;

  memset(&ipv4dtClear, 0, sizeof(ipv4dtClear));

  sscanf(hostMac, "%x:%x:%x:%x:%x:%x", &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);
  for (index = 0; index < OPEN_MAC_ADDR_LEN; index++)
  {
    ipv4dtClear.hostMacAddr[index] = (unsigned char) mac[index];
  }
  ipv4dtClear.clear = OPEN_IPV4DT_CLEAR_HOST_MAC;

  ipv4dtEntriesClear(clientHandle, &ipv4dtClear);
}

/*********************************************************************
* @purpose  This function sanity checks all IPv4DT OpEN APIs. All
*           IPv4DT OpEN APIs are called with possible NULL and
*           invalid parameters to check the API robustness.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ipv4dtOpENAPIsTestSanity(openapiClientHandle_t *clientHandle)
{
  uint32_t                      iValue = 0;
  uint32_t                      iValue1 = 0, iValue2 = 0;
  open_error_t                  result = OPEN_E_NONE;
  OPEN_CONTROL_t                mode = OPEN_DISABLE;
  OPEN_IPV4DT_ENTRY_t           ipv4dtEntry;
  OPEN_IPV4DT_SHOW_REQUEST_t    ipv4dtShow;
  OPEN_IPV4DT_CLEAR_REQUEST_t   ipv4dtClear;
  OPEN_IPV4DT_ENTRIES_COUNT_t   ipv4dtEntriesCount;

  printf("Testing IPv4DT OpEN APIs sanity:\n");

  /* openapiIpv4dtAdminModeGet() */
  printf("\nTestig openapiIpv4dtAdminModeGet(): \n");

  result = openapiIpv4dtAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtAdminModeGet(clientHandle, NULL);
  printf("NULL parameter to IPv4DT admin mode get. (result = %d)\n", result);

  printf("openapiIpv4dtAdminModeGet() sanity successful. \n");

  /* openapiIpv4dtAdminModeSet() */
  printf("\nTesting openapiIpv4dtAdminModeSet(): \n");

  result = openapiIpv4dtAdminModeSet(NULL, OPEN_DISABLE);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtAdminModeSet(clientHandle, 3);
  printf("Set invalid mode parameter. (result = %d)\n", result);

  printf("openapiIpv4dtAdminModeSet() sanity successful. \n");

  /* openapiIpv4dtProbeModeGet() */
  printf("\nTestig openapiIpv4dtProbeModeGet(): \n");

  result = openapiIpv4dtProbeModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeModeGet(clientHandle, NULL);
  printf("NULL parameter to IPv4DT probe mode get. (result = %d)\n", result);

  printf("openapiIpv4dtProbeModeGet() sanity successful. \n");

  /* openapiIpv4dtProbeModeSet() */
  printf("\nTesting openapiIpv4dtProbeModeSet(): \n");

  result = openapiIpv4dtProbeModeSet(NULL, OPEN_DISABLE);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeModeSet(clientHandle, 3);
  printf("Set invalid mode parameter. (result = %d)\n", result);

  printf("openapiIpv4dtProbeModeSet() sanity successful. \n");

  /* openapiIpv4dtProbeCountGet() */
  printf("\nTesting openapiIpv4dtProbeCountGet(): \n");

  result = openapiIpv4dtProbeCountGet(NULL, &iValue1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeCountGet(clientHandle, NULL);
  printf("NULL parameter to IPv4DT Probe Count get. (result = %d)\n", result);

  printf("openapiIpv4dtProbeCountGet() sanity successful. \n");

  /* openapiIpv4dtProbeCountSet() */
  printf("\nTesting openapiIpv4dtProbeCountSet(): \n");

  result = openapiIpv4dtProbeCountSet(NULL, 1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeCountSet(clientHandle, 300);
  printf("Set invalid count value. (result = %d)\n", result);

  printf("openapiIpv4dtProbeCountSet() sanity successful. \n");
  
  /* openapiIpv4dtProbeIntervalGet() */
  printf("\nTesting openapiIpv4dtProbeIntervalGet(): \n");

  result = openapiIpv4dtProbeIntervalGet(NULL, &iValue1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeIntervalGet(clientHandle, NULL);
  printf("NULL parameter to IPv4DT Probe Interval get. (result = %d)\n", result);

  printf("openapiIpv4dtProbeCountGet() sanity successful. \n");

  /* openapiIpv4dtProbeIntervalSet() */
  printf("\nTesting openapiIpv4dtProbeIntervalSet(): \n");

  result = openapiIpv4dtProbeIntervalSet(NULL, 30);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeIntervalSet(clientHandle, 20);
  printf("Set invalid interval value. (result = %d)\n", result);

  printf("openapiIpv4dtProbeIntervalSet() sanity successful. \n");

  /* openapiIpv4dtProbeDelayGet() */
  printf("\nTesting openapiIpv4dtProbeDelayGet(): \n");

  result = openapiIpv4dtProbeDelayGet(NULL, &iValue1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeDelayGet(clientHandle, NULL);
  printf("NULL parameter to IPv4DT Probe Delay get. (result = %d)\n", result);

  printf("openapiIpv4dtProbeDelayGet() sanity successful. \n");

  /* openapiIpv4dtProbeDelaySet() */
  printf("\nTesting openapiIpv4dtProbeDelaySet(): \n");

  result = openapiIpv4dtProbeDelaySet(NULL, 1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeDelaySet(clientHandle, 150);
  printf("Set invalid delay value. (result = %d)\n", result);

  printf("openapiIpv4dtProbeDelaySet() sanity successful. \n");

  /* openapiIpv4dtIntfMaxGet() */
  printf("\nTesting openapiIpv4dtIntfMaxGet(): \n");

  result = openapiIpv4dtIntfMaxGet(NULL, 1, &iValue);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtIntfMaxGet(clientHandle, 1, NULL);
  printf("NULL parameter to IPv4DT Interface Max get. (result = %d)\n", result);

  printf("openapiIpv4dtIntfMaxGet() sanity successful. \n");

  /* openapiIpv4dtIntfMaxSet() */
  printf("\nTesting openapiIpv4dtIntfMaxSet(): \n");

  result = openapiIpv4dtIntfMaxSet(NULL, 1, 1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtIntfMaxSet(clientHandle, 1, 20);
  printf("Set invalid maximum value. (result = %d)\n", result);

  printf("openapiIpv4dtIntfMaxSet() sanity successful. \n");

  /* openapiIpv4dtProbeAutoSourceGet() */
  printf("\nTesting openapiIpv4dtProbeAutoSourceGet(): \n");

  result = openapiIpv4dtProbeAutoSourceGet(NULL, &iValue1, &iValue2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeAutoSourceGet(clientHandle, NULL, &iValue2);
  printf("NULL host parameter to IPv4DT auto-source get. (result = %d)\n", result);

  result = openapiIpv4dtProbeAutoSourceGet(clientHandle, &iValue1, NULL);
  printf("NULL mask parameter to IPv4DT auto-source get. (result = %d)\n", result);

  printf("openapiIpv4dtProbeAutoSourceGet() sanity successful. \n");

  /* openapiIpv4dtProbeAutoSourceSet() */
  printf("\nTesting openapiIpv4dtProbeAutoSourceSet(): \n");

  result = openapiIpv4dtProbeAutoSourceSet(NULL, &iValue1, &iValue2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtProbeAutoSourceSet(clientHandle, NULL, &iValue2);
  printf("NULL host parameter to IPv4DT auto-source set. (result = %d)\n", result);

  result = openapiIpv4dtProbeAutoSourceSet(clientHandle, &iValue1, NULL);
  printf("NULL mask parameter to IPv4DT auto-source set. (result = %d)\n", result);

  printf("openapiIpv4dtProbeAutoSourceSet() sanity successful. \n");

  /* openapiIpv4dtEntriesCountGet() */
  printf("\nTesting openapiIpv4dtEntriesCountGet(): \n");

  result = openapiIpv4dtEntriesCountGet(NULL, &ipv4dtEntriesCount);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtEntriesCountGet(clientHandle, NULL);
  printf("NULL parameter to IPv4DT Entries count get. (result = %d)\n", result);

  printf("openapiIpv4dtEntriesCountGet() sanity successful. \n");
 
  /* openapiIpv4dtClearEntries() */
  printf("\nTesting openapiIpv4dtClearEntries(): \n");

  result = openapiIpv4dtClearEntries(NULL, &ipv4dtClear);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtClearEntries(clientHandle, NULL);
  printf("NULL parameter to IPv4DT Clear Entries. (result = %d)\n", result);

  printf("openapiIpv4dtClearEntries() sanity successful. \n");

  /* openapiIpv4dtTableGetNext() */
  printf("\nTesting openapiIpv4dtTableGetNext(): \n");

  result = openapiIpv4dtTableGetNext(NULL, &ipv4dtShow, &ipv4dtEntry);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiIpv4dtTableGetNext(clientHandle, NULL, &ipv4dtEntry);
  printf("NULL show parameter to IPv4DT Table GetNext. (result = %d)\n", result);

  result = openapiIpv4dtTableGetNext(clientHandle, &ipv4dtShow, NULL);
  printf("NULL entry parameter to IPv4DT Table GetNext. (result = %d)\n", result);

  printf("openapiIpv4dtTableGetNext() sanity successful. \n");
 
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  uint32_t                arg1, arg2, testNum;
  char                    switch_os_revision_string[100];
  open_error_t            result;
  open_buffdesc           switch_os_revision;
  openapiClientHandle_t   clientHandle;

  if (argc < 2)
  {
    printIpv4dtAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ipv4_device_tracking_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IPv4DT API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("ICOS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Set IPv4DT Admin mode: ipv4dt_example 1 <mode: 0-disable / 1-enable> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4DtAdminModeSet(&clientHandle, arg1 == 0 ? OPEN_DISABLE : OPEN_ENABLE);
      break;

    case 2: /* Test 2: Get IPv4DT Admin mode: ipv4dt_example 2 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtAdminModeGet(&clientHandle);
      break;

    case 3: /* Test 3: Set IPv4DT Probe mode: ipv4dt_example 3 <mode: 0-disable / 1-enable> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4DtProbeModeSet(&clientHandle, arg1 == 0 ? OPEN_DISABLE : OPEN_ENABLE);
      break;

    case 4: /* Test 4: Get IPv4DT Probe mode: ipv4dt_example 4 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtProbeModeGet(&clientHandle);
      break;

    case 5: /* Test 5: Set IPv4DT Probe count: ipv4dt_example 5 <count: 1-255> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4DtProbeCountSet(&clientHandle, arg1);
      break;

    case 6: /* Test 6: Get IPv4DT Probe count: ipv4dt_example 6 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtProbeCountGet(&clientHandle);
      break;


    case 7: /* Test 7: Set IPv4DT Probe interval: ipv4dt_example 7 <interval: 30-300 secs> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4DtProbeIntervalSet(&clientHandle, arg1);
      break;

    case 8: /* Test 8: Get IPv4DT Probe count: ipv4dt_example 8 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtProbeIntervalGet(&clientHandle);
      break;

    case 9: /* Test 9: Set IPv4DT Probe delay: ipv4dt_example 9 <delay: 1-120> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4DtProbeDelaySet(&clientHandle, arg1);
      break;

    case 10: /* Test 10: Get IPv4DT Probe delay: ipv4dt_example 10 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtProbeDelayGet(&clientHandle);
      break;

    case 11: /* Test 11: Set IPv4DT maximum entries for an interface: ipv4dt_example 11 <interface> <max-value: 0-10, -1 for nolimit> */
      if (argc != 4)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      ipv4DtIntfMaxSet(&clientHandle, arg1, arg2);
      break;

    case 12: /* Test 12: Get IPv4DT maximum entries for an interface: ipv4dt_example 12 <interface> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4DtIntfMaxGet(&clientHandle, arg1);
      break;

    case 13: /* Test 13: Set IPv4DT Auto-source: ipv4dt_example 13 <host> <mask> */
      if (argc != 4)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtProbeAutoSourceSet(&clientHandle, argv[2], argv[3]);
      break;

    case 14: /* Test 14: Get IPv4DT Auto-source: ipv4dt_example 14 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtProbeAutoSourceGet(&clientHandle);
      break;

    case 15: /* Test 15: Show IPv4DT entries count: ipv4dt_example 15 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4DtEntriesCountGet(&clientHandle);
      break;

    case 16: /* Test 16: Show all IPv4DT entries: ipv4dt_example 16 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesShowAll(&clientHandle);
      break;

    case 17: /* Test 17: Show all IPv4DT Active entries: ipv4dt_example 17 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesShowActive(&clientHandle);
      break;

    case 18: /* Test 18: Show all IPv4DT InActive entries: ipv4dt_example 18 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesShowInactive(&clientHandle);
      break;

    case 19: /* Test 19: Show all IPv4DT entries learned on a interface: ipv4dt_example 19 <interface> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4dtEntriesShowInterface(&clientHandle, arg1);
      break;

    case 20: /* Test 20: Show all IPv4DT entries per ip: ipv4dt_example 20 <ip> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesShowIp(&clientHandle, argv[2]);
      break;

    case 21: /* Test 21: Show all IPv4DT entries per mac: ipv4dt_example 20 <mac> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesShowMac(&clientHandle, argv[2]);
      break;

    case 22: /* Test 22: Clear all IPv4DT entries: ipv4dt_example 22 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesClearAll(&clientHandle);
      break;

    case 23: /* Test 23: Clear all IPv4DT entries learned on a interface: ipv4dt_example 23 <interface> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ipv4dtEntriesClearInterface(&clientHandle, arg1);
      break;

    case 24: /* Test 24: Clear all IPv4DT entries per ip: ipv4dt_example 24 <ip> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesClearIp(&clientHandle, argv[2]);
      break;

    case 25: /* Test 25: Clear all IPv4DT entries per mac: ipv4dt_example 25 <mac> */
      if (argc != 3)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtEntriesClearMac(&clientHandle, argv[2]);
      break;

    case 26: /* Test 26: IPv4DT OpEN APIs Sanity: ipv4dt_example 26 */
      if (argc != 2)
      {
        printIpv4dtAppMenu();
        exit(1);
      }
      ipv4dtOpENAPIsTestSanity(&clientHandle);
      break;

    default:
      printIpv4dtAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IPv4DT API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}


