/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  linuxptp_example.c
*
* @purpose   OpEN LinuxPTP example.
*
* @component OpEN
*
* @create    02/13/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_linuxptp.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the LinuxPTP Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Set the linuxPTP global admin mode: %s 1 <mode>\n", name);
  printf("Test 2: Get the linuxPTP global admin mode: %s 2 \n", name);
  printf("Test 3: Set the linuxPTP interface admin mode: %s 3 <ifNum> <mode>\n",
         name);
  printf("Test 4: Get the linuxPTP interface admin mode: %s 4 <ifNum>\n", name);
  printf("Test 5: Get the E2E mean path delay: %s 5 \n", name);
  printf("Test 6: Get the linuxPTP port state: %s 6 <ifNum>\n", name);
  printf("Test 7: Get the P2P mean path delay: %s 7 <ifNum> \n", name);
  printf("Test 8: Get the statistics for the given type on an interface:"
         " %s 8 <ifNum> <statType> \n", name);
  printf("Test 9: Get (parent or GM or local) clock identity: %s 9 <type> \n",
         name);
  printf("Test 10: Get (parent or GM or local) clock parameters: %s 10 <type>\n", 
         name);
  printf("Test 11: Set priority value of the local clock: %s 11 <prioType>"
         " <prioVal>\n", name);
  printf("Test 12: Get priority value of the local clock: %s 12 <prioType>\n",
         name);
  printf("Test 13: Set linuxPTP clock operation mode: %s 13 <operMode>\n", name);
  printf("Test 14: Get linuxPTP clock operation mode: %s 14 \n", name);
  printf("Test 15: Set source IPv4 address to be used for linuxPTP communication:"
         " %s 15 <ipAddr>\n", name);
  printf("Test 16: Get the source IPv4 address used for linuxPTP communication:"
         " %s 16\n", name);
  printf("Test 17: Set source IPv6 address to be used for linuxPTP communication:"
         " %s 17 <ipAddr>\n", name);
  printf("Test 18: Get the source IPv6 address used for linuxPTP communication:"
         " %s 18 \n", name);
  printf("Test 19: Set linuxPTP profile: %s 19 <ptpProfile>\n", name);
  printf("Test 20: Get the active linuxPTP profile: %s 20 <ptpProfile>\n", name);
  printf("Test 21: Set linuxPTP profile's transport mode: %s 21 <ptpProfile> "
         "<transMode>\n", name);
  printf("Test 22: Get the linuxPTP transport mode for a given profile:"
         " %s 22 <ptpProfile> \n", name);
  printf("Test 23: Set announce interval for given linuxPTP profile: %s 23 "
         "<ptpProfile> <announceInterval>\n", name);
  printf("Test 24: Get the configured linuxPTP announce interval for a given "
         "profile: %s 24 <ptpProfile> \n", name);
  printf("Test 25: Set announce timeout for given linuxPTP profile: %s 25 "
         "<ptpProfile> <announceTimeout>\n", name);
  printf("Test 26: Get the linuxPTP announce timeout value for a given profile:"
         " %s 26 <ptpProfile> \n", name);
  printf("Test 27: Set sync interval for given linuxPTP profile: %s 27 <ptpProfile>"
         " <syncInterval>\n", name);
  printf("Test 28: Get the linuxPTP Sync Interval for a given profile: %s 28"
         " <ptpProfile> \n", name);
  printf("Test 29: Set delay request interval for given linuxPTP profile: %s 29 "
         "<ptpProfile> <delayReqInterval>\n", name);
  printf("Test 30: Get the linuxPTP delay request Interval for a given profile:"
         " %s 30 <ptpProfile> \n", name);
  printf("Test 31: Set peer delay request interval for given linuxPTP profile: "
         "%s 31 <ptpProfile> <pDelayReqInterval>\n", name);
  printf("Test 32: Get the linuxPTP peer delay request Interval for a given "
         "profile: %s 32 <ptpProfile> \n", name);
  printf("Test 33: Set domain number for given linuxPTP profile: %s 33 <ptpProfile> "
         "<domain>\n", name);
  printf("Test 34: Get the linuxPTP Domain for a given profile: %s 34 "
         "<ptpProfile> \n", name);
  printf("Test 35: Set linuxPTP delay mechanism: %s 35 <ptpProfile> <delayMech>\n",
         name);
  printf("Test 36: Get the linuxPTP delay mechanism for a given profile: "
         "%s 36 <ptpProfile> \n", name);
  printf("Test 37: Configure linuxPTP logging severity level: %s 37 <level>\n", name);
  printf("Test 38: Fetch linuxPTP logging severity level: %s 38 <level>\n", name);
  printf("Test 39: Clear linuxPTP statistics on an interface: %s 39 <ifNum>\n", name);
  printf("Test 40: Get linuxPTP interface operational status: %s 40 <ifNum>\n", name);
  printf("Test 41: Check whether the interface is eligible for linuxptp operation:"
         " %s 41 <ifNum>\n", name);
  printf("Test 42: Check if the passed domain identifier is configured:"
         " %s 42 <domain>\n", name);
  printf("Test 43: Get time properties information: %s 43 <type>\n", name);
  printf("Test 44: Get number of boundary clocks between local clock and foreign "
         "master: %s 44 \n", name);
  printf("Test 45: Get current clock offset in time between slave and master:"
         " %s 45 \n", name);
  printf("Test 46: Status of the mode the clock is operating in master or slave:"
         " %s 46 \n", name);
  printf("Test 47: Set linuxPTP path trace TLV: %s 47 <traceMode>\n", name);
  printf("Test 48: Get linuxPTP path trace TLV status: %s 48 \n", name);
  printf("Test 49: Get next clock identity in the path: %s 49 <type>\n", name);

  return;
}

/*********************************************************************
* @purpose  Set the linuxPTP Global Admin Mode.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    mode           @b{(input)}  Global admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiLinuxptpAdminModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the linuxPTP Global Admin Mode. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP Global Admin Mode.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    mode             @b{(output)} Global admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiLinuxptpAdminModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP Global Admin Mode. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Global admin mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Set the linuxPTP interface Admin Mode.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    mode          @b{(input)}  Interface admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfAdminModeSet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiLinuxptpIntfAdminModeSet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the linuxPTP interface Admin Mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Interface admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP interface Admin Mode.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    mode           @b{(output)} interface admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfAdminModeGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiLinuxptpIntfAdminModeGet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP interface Admin Mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Interface %u admin mode is %u.\n", ifNum, *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch E2E mean path delay.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockMeanPathDelayGet(openapiClientHandle_t *client_handle) 
{
  open_error_t result;
  open_buffdesc clockMeanPathDelay;
  char delay[256] = {0};

  clockMeanPathDelay.pstart = delay;
  clockMeanPathDelay.size = 256;

  if ((result = openapiLinuxptpClockMeanPathDelayGet(client_handle, 
                                                     &clockMeanPathDelay)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch E2E mean path delay. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Mean path delay : %s.\n", (char *) clockMeanPathDelay.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch PTP port state.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    portState      @b{(output)} PTP port state
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfStateGet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, int32_t *portState)
{
  open_error_t result;

  if ((result = openapiLinuxptpIntfStateGet(client_handle, ifNum, portState)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch PTP port state. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Interface %d state is %u.\n", ifNum, *portState);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch P2P mean path delay.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfDelayGet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum)
{
  open_error_t result;
  open_buffdesc p2pPathDelay;
  char delay[256] = {0};

  p2pPathDelay.pstart = delay;
  p2pPathDelay.size = 256;

  if ((result = openapiLinuxptpIntfDelayGet(client_handle, ifNum, &p2pPathDelay))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch P2P mean path delay. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Interface %u P2P mean path delay : %s\n", ifNum, (char *) p2pPathDelay.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch statistics for the given type on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    statType       @b{(input)}  statistic type
* @param    statsCount     @b{(output)} statistics for the given type
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfStatsGet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, OPEN_LINUXPTP_STATS_TYPE_t statType,
                          int32_t *statsCount)
{
  open_error_t result;

  if ((result = openapiLinuxptpIntfStatsGet(client_handle, ifNum, statType, 
                                            statsCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch statistics for the given type "
           "on an interface. (result = %d)\n", result);
  }
  else
  {
    printf("Count : %d.\n", *statsCount);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch parent or GM or local clock identity.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    type           @b{(input)}  Clock type
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockIdGet(openapiClientHandle_t *client_handle, 
                        OPEN_LINUXPTP_DS_TYPE_t type)
{
  open_error_t result;
  open_buffdesc identity;
  char id[256] = {0};

  identity.pstart = id;
  identity.size = 256;

  if ((result = openapiLinuxptpClockIdGet(client_handle, type, &identity)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch parent or GM or local clock "
           "identity. (result = %d)\n", result);
  }
  else
  {
    printf("Clock identity : %s.\n", (char *) identity.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch parent or GM or local clock parameters.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    type           @b{(input)}  Clock type
* @param    value          @b{(output)} fetched value
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockParamGet(openapiClientHandle_t *client_handle, 
                           OPEN_LINUXPTP_DS_TYPE_t type, int32_t *value)
{
  open_error_t result;

  if ((result = openapiLinuxptpClockParamGet(client_handle, type, value)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch parent or GM or local clock"
           " parameters. (result = %d)\n", result);
  }
  else
  {
    printf("Number of linuxPTP ports %d\n", *value);
  }
  return;
}

/*********************************************************************
* @purpose  Configure priority value of the local clock.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    prioType        @b{(input)}  priority 1 or 2
* @param    prioVal         @b{(input)}  priority value
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpBridgePrioritySet(openapiClientHandle_t *client_handle, 
                               OPEN_LINUXPTP_CLOCK_PRIORITY_t prioType, 
                               uint32_t prioVal)
{
  open_error_t result;

  if ((result = openapiLinuxptpBridgePrioritySet(client_handle, prioType, prioVal)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure priority value of the local "
           "clock. (result = %d)\n", result);
  }
  else
  {
    printf("Bridge priority set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get priority value of the local clock.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    prioType        @b{(input)}  priority 1 or 2
* @param    prioVal         @b{(output)} priority value
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpBridgePriorityGet(openapiClientHandle_t *client_handle, 
                               OPEN_LINUXPTP_CLOCK_PRIORITY_t prioType, 
                               uint32_t *prioVal)
{
  open_error_t result;

  if ((result = openapiLinuxptpBridgePriorityGet(client_handle, prioType, prioVal)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get priority value of the local clock."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Bridge priority : %u.\n", *prioVal);
  }
  return;
}

/*********************************************************************
* @purpose  Configure PTP clock operation mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    operMode          @b{(input)}  one-step or two-step
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockOperModeSet(openapiClientHandle_t *client_handle, 
                              OPEN_LINUXPTP_CLOCK_OPERATION_t operMode)
{
  open_error_t result;

  if ((result = openapiLinuxptpClockOperModeSet(client_handle, operMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure PTP clock operation mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Clock operation mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the Clock Operation for the local clock.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    operMode          @b{(output)} one-step or two-step
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockOperModeGet(openapiClientHandle_t *client_handle, 
                              uint32_t *operMode)
{
  open_error_t result;

  if ((result = openapiLinuxptpClockOperModeGet(client_handle, operMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Clock Operation for the local"
           " clock. (result = %d)\n", result);
  }
  else
  {
    printf("Clock operation mode : %u.\n", *operMode);
  }
  return;
}

/*********************************************************************
* @purpose  Configure source IPv4 address to be used for linuxPTP communication.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ipAddr           @b{(input)}  source IPv4 address
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpSourceIPv4AddrSet(openapiClientHandle_t *client_handle, 
                               char * ipAddr)
{
  open_error_t result;
  uint32_t inetIpAddr = 0;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipAddr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &inetIpAddr)) != 
       OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string "
           "to a 32 bit integer. result = %d.\n", result);
    return;
  }

  if ((result = openapiLinuxptpSourceIPv4AddrSet(client_handle, inetIpAddr)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure source IPv4 address to be"
           " used for linuxPTP communication. (result = %d)\n", result);
  }
  else
  {
    printf("Source IP set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the source IPv4 address used for linuxPTP communication.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpSourceIPv4AddrGet(openapiClientHandle_t *client_handle) 
{
  open_error_t result;
  uint32_t ipAddr = 0;
  open_buffdesc ipDesc;
  char ipDescStr[32];
  open_inet_addr_t openAddr;
 
  ipDesc.pstart = ipDescStr;
  ipDesc.size = sizeof(ipDescStr)-1;
  openAddr.family = OPEN_AF_INET;

  if ((result = openapiLinuxptpSourceIPv4AddrGet(client_handle, &ipAddr)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the source IPv4 address used for linuxPTP"
            "communication. (result = %d)\n", result);
  }
  else
  {
    openAddr.addr.ipv4 = ipAddr;
    (void)openapiOpenIPtoStringGet(client_handle, openAddr, &ipDesc);
    printf ("Source IP address  %s\n", (char *)ipDesc.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Configure source IPv6 address to be used for linuxPTP communication.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ipAddr         @b{(input)}  source IPv6 address
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpSourceIPv6AddrSet(openapiClientHandle_t *client_handle, 
                               char* ipAddr)
{
  open_error_t result;
  open_inet_addr_t openAddr;
  open_buffdesc    buffDesc;
  open_in6_addr_t  ipV6Addr;

  buffDesc.pstart = ipAddr;
  buffDesc.size = strlen(ipAddr) + 1;

  if (openapiInetAddrGet(client_handle, &buffDesc, &openAddr) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string.");
    return;
  }

  memcpy (&ipV6Addr, &(openAddr.addr.ipv6), sizeof(open_in6_addr_t));
  if ((result = openapiLinuxptpSourceIPv6AddrSet(client_handle, ipV6Addr)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure source IPv6 address to be"
           " used for linuxPTP communication. (result = %d)\n", result);
  }
  else
  {
    printf("Source IPv6 address set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the source IPv6 address used for linuxPTP communication.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ipAddr          @b{(input)}  source IPv6 address
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpSourceIPv6AddrGet(openapiClientHandle_t *client_handle) 
{
  open_error_t result;
  open_in6_addr_t  ipV6Addr;
  open_buffdesc ipDesc;
  char ipDescStr[256] = {0};
  open_inet_addr_t openAddr;
 
  ipDesc.pstart = ipDescStr;
  ipDesc.size = sizeof(ipDescStr)-1;
  openAddr.family = OPEN_AF_INET6;

  if ((result = openapiLinuxptpSourceIPv6AddrGet(client_handle, &ipV6Addr)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the source IPv6 address used for "
           "linuxPTP communication. (result = %d)\n", result);
  }
  else
  {
    memcpy (&(openAddr.addr.ipv6), &ipV6Addr, sizeof(open_in6_addr_t));
    (void)openapiOpenIPtoStringGet(client_handle, openAddr, &ipDesc);
    printf ("Source IPv6 address : %s.\n", (char *)ipDesc.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Configure PTP profile.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    ptpProfile         @b{(input)}  PTP profile
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpProfileSet(openapiClientHandle_t *client_handle, 
                        uint32_t ptpProfile)
{
  open_error_t result;

  if ((result = openapiLinuxptpProfileSet(client_handle, ptpProfile)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure PTP profile. (result = %d)\n", result);
  }
  else
  {
    printf("PTP profile set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the active linuxPTP profile.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ptpProfile       @b{(output)} PTP profile
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpProfileGet(openapiClientHandle_t *client_handle, 
                        uint32_t *ptpProfile)
{
  open_error_t result;

  if ((result = openapiLinuxptpProfileGet(client_handle, ptpProfile)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the active linuxPTP profile."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PTP profile : %u \n", *ptpProfile);
  }
  return;
}

/*********************************************************************
* @purpose  Configure PTP profile's transport mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ptpProfile      @b{(input)}  PTP profile
* @param    transMode       @b{(input)}  Mode : ethernet or ipv4
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpTransportSet(openapiClientHandle_t *client_handle, 
                          uint32_t ptpProfile, 
                          OPEN_LINUXPTP_TRANSPORT_MODE_t transMode)
{
  open_error_t result;

  if ((result = openapiLinuxptpTransportSet(client_handle, ptpProfile, 
                                            transMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure PTP profile's transport mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PTP profile transport mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP transport for a given profile.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ptpProfile        @b{(input)}  PTP profile
* @param    transMode         @b{(output)} Mode : ethernet or ipv4
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpTransportGet(openapiClientHandle_t *client_handle, 
                          uint32_t ptpProfile, 
                          uint32_t *transMode)
{
  open_error_t result;

  if ((result = openapiLinuxptpTransportGet(client_handle, ptpProfile, 
                                            transMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP transport for a "
           "given profile. (result = %d)\n", result);
  }
  else
  {
    printf("PTP profile transport mode is %u.\n", *transMode);
  }
  return;
}

/*********************************************************************
* @purpose  Configure announce interval for given PTP profile.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ptpProfile        @b{(input)}  PTP profile
* @param    announceInterval  @b{(input)}  time interval b/w announce frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpAnnounceIntervalSet(openapiClientHandle_t *client_handle, 
                                 uint32_t ptpProfile, 
                                 int32_t announceInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpAnnounceIntervalSet(client_handle, ptpProfile,
                                                   announceInterval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure announce interval for "
           "given PTP profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP announce interval set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the configured linuxPTP announce interval for a given profile.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    ptpProfile         @b{(input)}  PTP profile
* @param    announceInterval   @b{(output)} Announce interval
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpAnnounceIntervalGet(openapiClientHandle_t *client_handle, 
                                 uint32_t ptpProfile, 
                                 int32_t *announceInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpAnnounceIntervalGet(client_handle, ptpProfile, 
                                                   announceInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the configured linuxPTP announce "
           "interval for a given profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP announce interval is %d.\n", *announceInterval);
  }
  return;
}

/*********************************************************************
* @purpose  Configure announce timeout for given PTP profile.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ptpProfile        @b{(input)}  PTP profile
* @param    announceTimeout   @b{(input)}  Announce timeout
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpAnnounceTimeoutSet(openapiClientHandle_t *client_handle, 
                                uint32_t ptpProfile, 
                                uint32_t announceTimeout)
{
  open_error_t result;

  if ((result = openapiLinuxptpAnnounceTimeoutSet(client_handle, ptpProfile,
                                                  announceTimeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure announce timeout for given "
           "PTP profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP announce timeout set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP announce timeout value for a given profile.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ptpProfile       @b{(input)}  PTP profile
* @param    announceTimeout  @b{(output)} Announce timeout
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpAnnounceTimeoutGet(openapiClientHandle_t *client_handle, 
                                uint32_t ptpProfile, 
                                uint32_t *announceTimeout)
{
  open_error_t result;

  if ((result = openapiLinuxptpAnnounceTimeoutGet(client_handle, ptpProfile,
                                                  announceTimeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP announce timeout value"
           " for a given profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP announce timeout is : %u.\n", *announceTimeout);
  }
  return;
}

/*********************************************************************
* @purpose  Configure sync interval for given PTP profile.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ptpProfile       @b{(input)}  PTP profile
* @param    syncInterval     @b{(input)}  Time interval b/w sync frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpSyncIntervalSet(openapiClientHandle_t *client_handle, 
                             uint32_t ptpProfile, 
                             int32_t syncInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpSyncIntervalSet(client_handle, ptpProfile, 
                                               syncInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure sync interval for given PTP "
           "profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP sync interval set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP Sync Interval for a given profile.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ptpProfile        @b{(input)}  PTP profile
* @param    syncInterval      @b{(output)} Time interval b/w sync frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpSyncIntervalGet(openapiClientHandle_t *client_handle, 
                             uint32_t ptpProfile, 
                             int32_t *syncInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpSyncIntervalGet(client_handle, ptpProfile, 
                                               syncInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP Sync Interval for a"
           " given profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP sync interval is : %d.\n", *syncInterval);
  }
  return;
}

/*********************************************************************
* @purpose  Configure delay request interval for given PTP profile.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    ptpProfile         @b{(input)}  PTP profile
* @param    delayReqInterval   @b{(input)}  Time interval b/w delay-request frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpDelayRequestIntervalSet(openapiClientHandle_t *client_handle, 
                                     uint32_t ptpProfile, 
                                     int32_t delayReqInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpDelayRequestIntervalSet(client_handle, 
                                                       ptpProfile, 
                                                       delayReqInterval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure delay request interval for"
           " given PTP profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP delay request interval set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP delay request Interval for a given profile.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    ptpProfile         @b{(input)}  PTP profile
* @param    delayReqInterval   @b{(output)} Time interval b/w delay-request frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpDelayRequestIntervalGet(openapiClientHandle_t *client_handle, 
                                     uint32_t ptpProfile, 
                                     int32_t *delayReqInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpDelayRequestIntervalGet(client_handle, 
                                                       ptpProfile, 
                                                       delayReqInterval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP delay request Interval"
           " for a given profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP delay request interval is : %d.\n", *delayReqInterval);
  }
  return;
}

/*********************************************************************
* @purpose  Configure peer delay request interval for given PTP profile.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ptpProfile        @b{(input)}  PTP profile
* @param    pDelayReqInterval @b{(input)}  Time interval b/w peer delay-request 
*                                          frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpPdelayRequestIntervalSet(openapiClientHandle_t *client_handle, 
                                      uint32_t ptpProfile, 
                                      int32_t pDelayReqInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpPdelayRequestIntervalSet(client_handle, 
                                                        ptpProfile,
                                                        pDelayReqInterval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure peer delay request interval"
           " for given PTP profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP peer delay request interval set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP peer delay request Interval for a given profile.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ptpProfile        @b{(input)}  PTP profile
* @param    pDelayReqInterval @b{(output)} Time interval b/w peer delay-request
*                                          frames
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpPdelayRequestIntervalGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ptpProfile, 
                                      int32_t *pDelayReqInterval)
{
  open_error_t result;

  if ((result = openapiLinuxptpPdelayRequestIntervalGet(client_handle, 
                                                        ptpProfile, 
                                                        pDelayReqInterval))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP peer delay request "
           "Interval for a given profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP peer delay request interval is : %d.\n", *pDelayReqInterval);
  }
  return;
}

/*********************************************************************
* @purpose  Configure domain number for given PTP profile.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ptpProfile     @b{(input)}  PTP profile
* @param    domain         @b{(input)}  PTP domain identifier
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpDomainSet(openapiClientHandle_t *client_handle, 
                       uint32_t ptpProfile, uint32_t domain)
{
  open_error_t result;

  if ((result = openapiLinuxptpDomainSet(client_handle, ptpProfile, domain)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure domain number for given "
           "PTP profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP domain set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP Domain for a given profile.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ptpProfile    @b{(input)}  PTP profile
* @param    domain        @b{(output)} PTP domain identifier
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpDomainGet(openapiClientHandle_t *client_handle, 
                       uint32_t ptpProfile, uint32_t *domain)
{
  open_error_t result;

  if ((result = openapiLinuxptpDomainGet(client_handle, ptpProfile, domain)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP Domain for a given "
           "profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP domain is %u.\n", *domain);
  }
  return;
}

/*********************************************************************
* @purpose  Configure PTP delay mechanism.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ptpProfile      @b{(input)}  PTP profile
* @param    delayMech       @b{(input)}  Either end-to-end or peer-to-peer delay calculation mechanism
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpDelayMechanismSet(openapiClientHandle_t *client_handle, 
                               uint32_t ptpProfile, uint32_t delayMech)
{
  open_error_t result;

  if ((result = openapiLinuxptpDelayMechanismSet(client_handle, ptpProfile,
                                                 delayMech)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure PTP delay mechanism. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP delay mechanism set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the linuxPTP delay mechanism for a given profile.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ptpProfile      @b{(input)}  PTP profile
* @param    delayMech       @b{(output)} Delay Mechanism, E2E or P2P
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpDelayMechanismGet(openapiClientHandle_t *client_handle, 
                               uint32_t ptpProfile, uint32_t *delayMech)
{
  open_error_t result;

  if ((result = openapiLinuxptpDelayMechanismGet(client_handle, ptpProfile, 
                                                 delayMech)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the linuxPTP delay mechanism for a"
           " given profile. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP delay mechanism is : %u.\n", *delayMech);
  }
  return;
}

/*********************************************************************
* @purpose  Configure PTP logging severity level.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    level          @b{(input)}  severity level
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpLoggingSet(openapiClientHandle_t *client_handle, 
                        uint32_t level)
{
  open_error_t result;

  if ((result = openapiLinuxptpLoggingSet(client_handle, level)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure PTP logging severity level."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP logging level set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Fetch PTP logging severity level.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    level          @b{(output)} severity level
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpLoggingGet(openapiClientHandle_t *client_handle, 
                        uint32_t *level)
{
  open_error_t result;

  if ((result = openapiLinuxptpLoggingGet(client_handle, level)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch PTP logging severity "
           "level. (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP logging level is : %u.\n", *level);
  }
  return;
}

/*********************************************************************
* @purpose  Clear PTP statistics on an interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfStatsClear(openapiClientHandle_t *client_handle, 
                            uint32_t ifNum)
{
  open_error_t result;

  if ((result = openapiLinuxptpIntfStatsClear(client_handle, ifNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear PTP statistics on an interface."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP interface statistics cleared successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get PTP interface operational status.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    status        @b{(output)} Operational status
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIntfOperStateGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiLinuxptpIntfOperStateGet(client_handle, ifNum, 
                                                status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PTP interface operational status"
           ". (result = %d)\n", result);
  }
  else
  {
    printf("Linux PTP interface operational status is : %u.\n", *status);
  }
  return;
}

/*********************************************************************
* @purpose  Check whether the interface is eligible for linuxptp operation.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    isValid       @b{(output)} Valid status
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIsValidIntf(openapiClientHandle_t *client_handle, 
                         uint32_t ifNum, OPEN_BOOL_t *isValid)
{
  open_error_t result;

  if ((result = openapiLinuxptpIsValidIntf(client_handle, ifNum, 
                                           isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check whether the interface is"
           " eligible for linuxptp operation. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %u %sa valid linux ptp port.\n", ifNum,
           (*isValid == OPEN_TRUE) ? "" :"not ");
  }
  return;
}

/*********************************************************************
* @purpose  Check if the passed domain identifier is configured.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    domain            @b{(input)}  domain identifier
* @param    isValidDomain     @b{(output)} Valid status
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpIsDomainExists(openapiClientHandle_t *client_handle, 
                            uint32_t domain, OPEN_BOOL_t *isValidDomain)
{
  open_error_t result;

  if ((result = openapiLinuxptpIsDomainExists(client_handle, domain, 
                                              isValidDomain)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the passed domain identifier"
           " is configured. (result = %d)\n", result);
  }
  else
  {
    printf("Domain ID %u is %sconfigured.\n", domain,
           (*isValidDomain == OPEN_TRUE) ? "" :"not ");
  }
  return;
}

/*********************************************************************
* @purpose  Fetch time properties information.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    type           @b{(input)}  Time properties type
* @param    value          @b{(output)} Value
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpTimePropGet(openapiClientHandle_t *client_handle, 
                         uint32_t type, int32_t *value)
{
  open_error_t result;

  if ((result = openapiLinuxptpTimePropGet(client_handle, type, value))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch time properties information."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Time property value is : %d.\n", *value);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch number of boundary clocks between local clock and foreign master.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    stepsRemoved       @b{(output)} hops to foreign master
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockStepsRemovedGet(openapiClientHandle_t *client_handle, 
                                  uint32_t *stepsRemoved)
{
  open_error_t result;

  if ((result = openapiLinuxptpClockStepsRemovedGet(client_handle, 
                                                    stepsRemoved)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch number of boundary clocks"
           " between local clock and foreign master. (result = %d)\n", result);
  }
  else
  {
    printf("Steps removed is : %u.\n", *stepsRemoved);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch current clock offset in time between slave and master.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockOffsetFromMaster(openapiClientHandle_t *client_handle) 
{
  open_error_t result;
  open_buffdesc offsetFromMaster;
  char offset[256] = {0};

  offsetFromMaster.pstart = offset;
  offsetFromMaster.size = 256;


  if ((result = openapiLinuxptpClockOffsetFromMaster(client_handle, 
                                                     &offsetFromMaster)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch current clock offset in "
           "time between slave and master. (result = %d)\n", result);
  }
  else
  {
    printf("Current clock offset is : %s.\n", (char *) offsetFromMaster.pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Status of the mode the clock is operating in master or slave.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    clockRole       @b{(output)} Clock role
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpClockRoleGet(openapiClientHandle_t *client_handle, 
                          uint32_t *clockRole)
{
  open_error_t result;

  if ((result = openapiLinuxptpClockRoleGet(client_handle, clockRole)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to status of the mode the clock is "
           "operating in master or slave. (result = %d)\n", result);
  }
  else
  {
    printf("Clock role is : %u.\n", *clockRole);
  }
  return;
}

/*********************************************************************
* @purpose  Configure PTP path trace TLV.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    traceMode       @b{(input)}  Trace mode
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpPathTraceTlvSet(openapiClientHandle_t *client_handle, 
                             OPEN_BOOL_t traceMode)
{
  open_error_t result;

  if ((result = openapiLinuxptpPathTraceTlvSet(client_handle, traceMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure PTP path trace TLV."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PTP trace mode set succesfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Fetch PTP path trace TLV status.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    traceMode       @b{(output)} Trace mode
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpPathTraceTlvGet(openapiClientHandle_t *client_handle, 
                             OPEN_BOOL_t *traceMode)
{
  open_error_t result;

  if ((result = openapiLinuxptpPathTraceTlvGet(client_handle, 
                                               traceMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch PTP path trace TLV status."
           " (result = %d)\n", result);
  }
  else
  {
    printf("PTP trace status is : %u.\n", *traceMode);
  }
  return;
}

/*********************************************************************
* @purpose  Fetch next clock identity in the path.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    type          @b{(input)}  Parent identity type
*
* @returns  none
*
* @end
*********************************************************************/
void linuxptpPathTraceClockIdGetNext(openapiClientHandle_t *client_handle, 
                                     OPEN_LINUXPTP_DS_TYPE_t type) 
{
  open_error_t result;
  open_buffdesc identity;
  char id[256] = {0};

  identity.pstart = id;
  identity.size = 256;

  if ((result = openapiLinuxptpPathTraceClockIdGetNext(client_handle, type, 
                                                       &identity)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch next clock identity in the path. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Clock identity is : %s.\n", (char *)identity.pstart);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for LinuxPTP
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, val;
  int32_t  value;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_BOOL_t flag;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("LinuxPTP example", &client_handle)) 
      != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting LinuxPTP API example "
              "application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == 
      OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        linuxptpAdminModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        linuxptpAdminModeGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        linuxptpIntfAdminModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        linuxptpIntfAdminModeGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 2)
      {
        linuxptpClockMeanPathDelayGet(&client_handle);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        linuxptpIntfStateGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        linuxptpIntfDelayGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 4)
      {
        linuxptpIntfStatsGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &value);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        linuxptpClockIdGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        linuxptpClockParamGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 4)
      {
        linuxptpBridgePrioritySet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        linuxptpBridgePriorityGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
        linuxptpClockOperModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 2)
      {
        linuxptpClockOperModeGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
        linuxptpSourceIPv4AddrSet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 2)
      {
        linuxptpSourceIPv4AddrGet(&client_handle);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        linuxptpSourceIPv6AddrSet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 2)
      {
        linuxptpSourceIPv6AddrGet(&client_handle);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 3)
      {
        linuxptpProfileSet(&client_handle,  atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 2)
      {
        linuxptpProfileGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 4)
      {
        linuxptpTransportSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
        linuxptpTransportGet(&client_handle,  atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 4)
      {
        linuxptpAnnounceIntervalSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 3)
      {
        linuxptpAnnounceIntervalGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 4)
      {
        linuxptpAnnounceTimeoutSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 3)
      {
        linuxptpAnnounceTimeoutGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 4)
      {
        linuxptpSyncIntervalSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 3)
      {
        linuxptpSyncIntervalGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 4)
      {
        linuxptpDelayRequestIntervalSet(&client_handle, atoi(argv[2]), 
                                        atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 3)
      {
        linuxptpDelayRequestIntervalGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 4)
      {
        linuxptpPdelayRequestIntervalSet(&client_handle, atoi(argv[2]), 
                                         atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 3)
      {
        linuxptpPdelayRequestIntervalGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 4)
      {
        linuxptpDomainSet(&client_handle, atoi(argv[2]),  atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 34:
      if (argc == 3)
      {
        linuxptpDomainGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 35:
      if (argc == 4)
      {
        linuxptpDelayMechanismSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 36:
      if (argc == 3)
      {
        linuxptpDelayMechanismGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 37:
      if (argc == 3)
      {
        linuxptpLoggingSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 38:
      if (argc == 2)
      {
        linuxptpLoggingGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 39:
      if (argc == 3)
      {
        linuxptpIntfStatsClear(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 40:
      if (argc == 3)
      {
        linuxptpIntfOperStateGet(&client_handle, atoi(argv[2]), &flag);
        show_help = 0;
      }
      break;
    case 41:
      if (argc == 3)
      {
        linuxptpIsValidIntf(&client_handle, atoi(argv[2]), &flag);
        show_help = 0;
      }
      break;
    case 42:
      if (argc == 3)
      {
        linuxptpIsDomainExists(&client_handle, atoi(argv[2]), &flag);
        show_help = 0;
      }
      break;
    case 43:
      if (argc == 3)
      {
        linuxptpTimePropGet(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 44:
      if (argc == 2)
      {
        linuxptpClockStepsRemovedGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 45:
      if (argc == 2)
      {
        linuxptpClockOffsetFromMaster(&client_handle);
        show_help = 0;
      }
      break;
    case 46:
      if (argc == 2)
      {
        linuxptpClockRoleGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 47:
      if (argc == 3)
      {
        linuxptpPathTraceTlvSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 48:
      if (argc == 2)
      {
        linuxptpPathTraceTlvGet(&client_handle, &flag);
        show_help = 0;
      }
      break;
    case 49:
      if (argc == 3)
      {
        linuxptpPathTraceClockIdGetNext(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping LinuxPTP API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}


