/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  loop_protect_example.c
*
* @purpose   OpEN LoopProtect example.
*
* @component OpEN
*
* @create    12/23/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_loop_protect.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the LoopProtect Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets the loop protection global mode: %s 1 <adminMode>\n", name);
  printf("Test 2: Gets the loop protection global mode: %s 2 \n", name);
  printf("Test 3: Sets the transmission timer for the loop protection PDUs: %s 3 <timerVal>\n", name);
  printf("Test 4: Gets the transmission timer for the loop protection PDUs: %s 4 \n", name);
  printf("Test 5: Sets the maximum number of PDUs to be received on a port before action is taken: %s 5 <val>\n", name);
  printf("Test 6: Gets the maximum number of PDUs to be received on a port before action is taken: %s 6 \n", name);
  printf("Test 7: Sets the transmission mode for the loop protection PDUs: %s 7 <ifNum> <mode>\n", name);
  printf("Test 8: Gets the transmission timer Mode for the loop protection PDUs: %s 8 <ifNum> \n", name);
  printf("Test 9: Sets the disable duration for the port: %s 9 <val>\n", name);
  printf("Test 10: Gets the disable duration for the port: %s 10 \n", name);
  printf("Test 11: Sets the loop protection admin mode for the interface: %s 11 <ifNum> <mode>\n", name);
  printf("Test 12: Gets the loop protection admin mode for the interface: %s 12 <ifNum> \n", name);
  printf("Test 13: Sets the port action for the the specified interface: %s 13 <ifNum> <mode>\n", name);
  printf("Test 14: Gets the port action for the the specified interface: %s 14 <ifNum> \n", name);
  printf("Test 15: Sets the VLAN ID and TPID (integer) to be used for detecting a loop on the specified interface: %s 15 <ifNum> <tpid> <vlanId>\n", name);
  printf("Test 16: Gets the VLAN ID and the TPID used for detecting a loop on the specified interface: %s 16 <ifNum> \n", name);
  printf("Test 17: Clears statistics of the specified interface: %s 17 <ifNum>\n", name);
  printf("Test 18: Clears the statistics for all interfaces: %s 18 \n", name);
  printf("Test 19: Gets the loop Count for the specified interface: %s 19 <ifNum> \n", name);
  printf("Test 20: Gets the status of loop on the specified interface: %s 20 <ifNum> \n", name);
  printf("Test 21: Gets the port status. whether port is in D-Disable state due to loop protection: %s 21 <ifNum> \n", name);
  printf("Test 22: Gets the time since last loop occured for the specified interface: %s 22 <ifNum> \n", name);
  printf("Test 23: Gets the transmit loop count for the specified interface: %s 23 <ifNum> \n", name);
  printf("Test 24: Gets the receive loop count for the specified interface: %s 24 <ifNum> \n", name);
  printf("Test 25: Gets the status of an interface whether port is valid for configuration for loop protection: %s 25 <ifNum> \n", name);
  printf("Test 26: Gets the first interface that is valid for loop protection: %s 26 \n", name);
  printf("Test 27: Gets the next interface that is valid for loop protection: %s 27 <ifNum> \n", name);
  return;
}

/*********************************************************************
* @purpose  Sets the loop protection global mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    adminMode       @b{(input)}  Loop protection global admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectAdminModeSet(openapiClientHandle_t *client_handle, 
                             OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiLoopProtectAdminModeSet(client_handle, adminMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the loop protection global mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the loop protection global mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    adminMode       @b{(output)}  Loop protection global admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectAdminModeGet(openapiClientHandle_t *client_handle, 
                             OPEN_CONTROL_t *adminMode)
{
  open_error_t result;

  if ((result = openapiLoopProtectAdminModeGet(client_handle, adminMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the loop protection global mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect admin mode is : %u.\n", *adminMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the transmission timer for the loop protection PDUs.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    timerVal       @b{(input)}  Timer value
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectTxTimerSet(openapiClientHandle_t *client_handle, 
                           uint32_t timerVal)
{
  open_error_t result;

  if ((result = openapiLoopProtectTxTimerSet(client_handle, timerVal)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the transmission timer "
           "for the loop protection PDUs. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect transmit timer set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the transmission timer for the loop protection PDUs.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    timerVal       @b{(output)}  Timer value
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectTxTimerGet(openapiClientHandle_t *client_handle, 
                           uint32_t *timerVal)
{
  open_error_t result;

  if ((result = openapiLoopProtectTxTimerGet(client_handle, timerVal)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the transmission timer for"
            " the loop protection PDUs. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect transmit timer value is : %u.\n", *timerVal);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the maximum number of PDUs to be received on a port 
*           before action is taken.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    val            @b{(input)}  Maximum number of PDUs
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectMaxPduReceiveSet(openapiClientHandle_t *client_handle, 
                                 uint32_t val)
{
  open_error_t result;

  if ((result = openapiLoopProtectMaxPduReceiveSet(client_handle, val)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the maximum number of PDUs "
            "to be received on a port before action is taken. "
            "(result = %d)\n", result);
  }
  else
  {
    printf("Loop protect maximum PDU number set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the maximum number of PDUs to be received on a port 
*           before action is taken.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    val            @b{(output)}  Maximum number of PDUs
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectMaxPduReceiveGet(openapiClientHandle_t *client_handle, 
                                 uint32_t *val)
{
  open_error_t result;

  if ((result = openapiLoopProtectMaxPduReceiveGet(client_handle, val)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the maximum number of PDUs"
           " to be received on a port before action is taken. (result = %d)\n",
           result);
  }
  else
  {
    printf("Loop protect maximum PDU number is : %u.\n", *val);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the transmission mode for the loop protection PDUs.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    mode           @b{(input)}  Transmit mode
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortTxModeSet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortTxModeSet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the transmission mode for the "
           "loop protection PDUs. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect transmit mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the transmission timer Mode for the loop protection PDUs.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    mode           @b{(output)} Transmit mode
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortTxModeGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortTxModeGet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the transmission timer Mode for the "
           "loop protection PDUs. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect transmit timer mode is : %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the disable duration for the port.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    val            @b{(input)}  Disable duration in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectDisableTimerSet(openapiClientHandle_t *client_handle, 
                                uint32_t val)
{
  open_error_t result;

  if ((result = openapiLoopProtectDisableTimerSet(client_handle, val)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the disable duration for the port."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect disable timer set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the disable duration for the port.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    val            @b{(output)}  Disable duration in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectDisableTimerGet(openapiClientHandle_t *client_handle, 
                                uint32_t *val)
{
  open_error_t result;

  if ((result = openapiLoopProtectDisableTimerGet(client_handle, val)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the disable duration for the port."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect disable timer value is : %u.\n", *val);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the loop protection admin mode for the interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    mode           @b{(input)}  Interface admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortAdminModeSet(openapiClientHandle_t *client_handle, 
                                 uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortAdminModeSet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the loop protection admin mode for"
           " the interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect port admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the loop protection admin mode for the interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    mode          @b{(output)} Interface admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortAdminModeGet(openapiClientHandle_t *client_handle, 
                                 uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortAdminModeGet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the loop protection admin mode for "
           "the interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect port admin mode is : %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the port action for the the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    mode           @b{(input)}  Port action to be configured on the port
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortActionSet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, 
                              OPEN_LOOP_PROTECT_PORT_ACTION_t mode)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortActionSet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the port action for the the "
           "specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect port action set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the port action for the the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    mode           @b{(output)} Port action configured on the port
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortActionGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, 
                              OPEN_LOOP_PROTECT_PORT_ACTION_t *mode)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortActionGet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the port action for the the"
           " specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect port action is : %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the VLAN ID and the TPID to be used for detecting a 
*           loop on the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    tpid           @b{(input)}  TPID value
* @param    vlanId         @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortVlanPropertiesSet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t tpid, 
                                      uint32_t vlanId)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortVlanPropertiesSet(client_handle, ifNum,
                                                        tpid, vlanId)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the VLAN ID and the TPID to be "
           "used for detecting a loop on the specified interface. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Loop protect port vlan propertes set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the VLAN ID and the TPID used for detecting a loop 
*           on the specified interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    tpid          @b{(output)} TPID value
* @param    vlanId        @b{(output)} VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortVlanPropertiesGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t *tpid, 
                                      uint32_t *vlanId)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortVlanPropertiesGet(client_handle, 
                                                        ifNum, tpid, vlanId)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the VLAN ID and the TPID used "
           "for detecting a loop on the specified interface. (result = %d)\n",
           result);
  }
  else
  {
    printf("Loop protect port vlan tpid is : 0x%x and vlan Id is : %u.\n",
           *tpid, *vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Clears statistics of the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortStatsClear(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortStatsClear(client_handle, ifNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to clears statistics of the specified"
           " interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect port statistics cleared successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Clears the statistics for all interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectAllStatsClear(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiLoopProtectAllStatsClear(client_handle)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to clears the statistics for "
           "all interfaces. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect statistics cleared successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the loop Count for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    loopCount       @b{(output)} Loop count value
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortloopDetectCountGet(openapiClientHandle_t *client_handle, 
                                       uint32_t ifNum, uint32_t *loopCount)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortloopDetectCountGet(client_handle, 
                                                         ifNum, loopCount)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the loop Count for the specified "
           "interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect loop detect count is : %u.\n", *loopCount);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the status of loop on the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    status         @b{(output)} Loop status
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectIsLoopDetected(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiLoopProtectIsLoopDetected(client_handle, ifNum, status)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the status of loop on the specified"
           " interface. (result = %d)\n", result);
  }
  else
  {
    if (*status == OPEN_TRUE)
    {
      printf("Loop detected on interface %u.\n", ifNum);
    }
    else
    {
      printf("Loop is not detected on interface %u.\n", ifNum);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Gets the port status. whether port is in D-Disable state 
*           due to loop protection.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    status         @b{(output)} Port status
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectIsPortDisabled(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiLoopProtectIsPortDisabled(client_handle, ifNum, status)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the port status. whether port"
           " is in D-Disable state due to loop protection. (result = %d)\n", 
           result);
  }
  else
  {
    printf("Interface %u is %s.\n", ifNum, 
           (*status == OPEN_TRUE) ? "disabled" : "enabled");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the time since last loop occured for the specified interface.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    timeVal       @b{(output)} Time value
* @param    needUtc       @b{(input)}  Flag L7_TRUE for UTC time and L7_FALSE 
*                                      for raw time
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortTimeSinceLastLoopGet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, uint32_t *timeVal, 
                                         OPEN_BOOL_t needUtc)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortTimeSinceLastLoopGet(client_handle, 
                                                           ifNum, timeVal,
                                                           needUtc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the time since last loop occured for the "
           "specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("Time since last loop is %u (raw time).\n", *timeVal);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the transmit loop count for the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface number
* @param    count          @b{(output)} Transmit loop count
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortPacketTxCountGet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, uint32_t *count)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortPacketTxCountGet(client_handle, ifNum, count)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the transmit loop count for the "
           "specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect packet transmit count is : %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the receive loop count for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    count           @b{(output)} Receive loop count
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectPortPacketRxCountGet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, uint32_t *count)
{
  open_error_t result;

  if ((result = openapiLoopProtectPortPacketRxCountGet(client_handle, ifNum, count)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the transmit loop count for the "
           "specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect packet receive count is : %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the status of an interface whether port is valid for 
*           configuration for loop protection.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    status          @b{(output)} Port status
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectIsValidIntf(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                            OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiLoopProtectIsValidIntf(client_handle, ifNum, status)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the status of an interface "
           "whether port is valid for configuration for loop protection."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Interface %u is %s a valid port.\n", ifNum, 
           (*status == OPEN_TRUE) ? "" : "not");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the first interface that is valid for loop protection.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(output)} Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectValidIntfFirstGet(openapiClientHandle_t *client_handle, 
                                  uint32_t *ifNum)
{
  open_error_t result;

  if ((result = openapiLoopProtectValidIntfFirstGet(client_handle, ifNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first interface that is "
           "valid for loop protection. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect valid first interface is : %u.\n", *ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the next interface that is valid for loop protection.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
* @param    nextIfNum       @b{(output)} Next valid interface number

*
* @returns  none
*
* @end
*********************************************************************/
void loopProtectValidIntfNextGet(openapiClientHandle_t *client_handle, 
                                 uint32_t ifNum, uint32_t *nextIfNum)
{
  open_error_t result;

  if ((result = openapiLoopProtectValidIntfNextGet(client_handle, ifNum,
                                                   nextIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next interface that is "
            "valid for loop protection. (result = %d)\n", result);
  }
  else
  {
    printf("Loop protect next interface is : %u.\n", *nextIfNum);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for LoopProtect
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, val, tpid, vlanId, ifNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CONTROL_t mode;
  OPEN_LOOP_PROTECT_PORT_ACTION_t actionMode;
  OPEN_BOOL_t status;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  if (argc > 2)
  {
    ifNum = atoi(argv[2]);  
  }
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("LoopProtect example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting LoopProtect API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        loopProtectAdminModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        loopProtectAdminModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        loopProtectTxTimerSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        loopProtectTxTimerGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        loopProtectMaxPduReceiveSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 2)
      {
        loopProtectMaxPduReceiveGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
        loopProtectPortTxModeSet(&client_handle, ifNum, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        loopProtectPortTxModeGet(&client_handle, ifNum, &mode);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        loopProtectDisableTimerSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        loopProtectDisableTimerGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 4)
      {
        loopProtectPortAdminModeSet(&client_handle, ifNum, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        loopProtectPortAdminModeGet(&client_handle, ifNum, &mode);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        loopProtectPortActionSet(&client_handle, ifNum, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        loopProtectPortActionGet(&client_handle, ifNum, &actionMode);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 5)
      {
        loopProtectPortVlanPropertiesSet(&client_handle, ifNum, atoi(argv[3]),
                                         atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
        loopProtectPortVlanPropertiesGet(&client_handle, ifNum, &tpid, &vlanId);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        loopProtectPortStatsClear(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 2)
      {
        loopProtectAllStatsClear(&client_handle);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 3)
      {
        loopProtectPortloopDetectCountGet(&client_handle, ifNum, &val);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 3)
      {
        loopProtectIsLoopDetected(&client_handle, ifNum, &status);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 3)
      {
        loopProtectIsPortDisabled(&client_handle, ifNum, &status);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
        loopProtectPortTimeSinceLastLoopGet(&client_handle, ifNum, &val, OPEN_FALSE);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 3)
      {
        loopProtectPortPacketTxCountGet(&client_handle, ifNum, &val);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 3)
      {
        loopProtectPortPacketRxCountGet(&client_handle, ifNum, &val);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 3)
      {
        loopProtectIsValidIntf(&client_handle, ifNum, &status);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 2)
      {
        loopProtectValidIntfFirstGet(&client_handle, &ifNum);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 3)
      {
        loopProtectValidIntfNextGet(&client_handle, ifNum, &val);
        show_help = 0;
      }
      break;
    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping LoopProtect API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

