/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  macal_example.c
*
* @purpose   Management Access Control and Administration List (MACAL) example.
*
* @component OpEN
*
* @create    01/26/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_macal.h"

#define MACAL_NAME_MAX_LEN  256
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Macal Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: To create a new management ACAL: %s 1 <macalName>\n", name);
  printf("Test 2: To delete an existing management ACAL: %s 2 <macalName>\n", name);
  printf("Test 3: Sets VRF of a management ACAL: %s 3 <vrfName>\n", name);
  printf("Test 4: Gets VRF of a management ACAL: %s 4 \n", name);
  printf("Test 5: Activate a management ACAL: %s 5 <macalName>\n", name);
  printf("Test 6: Deactivate a management ACAL: %s 6 <macalName>\n", name);
  printf("Test 7: To add a management ACAL rule: %s 7 <macalName> <macalRulePrio>\n",
         name);
  printf("Test 8: To remove a rule from an existing ACAL: %s 8 <macalName>"
         " <macalRulePrio>\n", name);
  printf("Test 9: To add an action to management ACAL rule: %s 9 <macalName>"
         " <macalRulePrio> <action>\n", name);
  printf("Test 10: To add the source ip address and net mask to a management"
         " ACAL rule: %s 10 <macalName> <macalRulePrio> <ipAddr> <mask>\n", name);
  printf("Test 11: To add an interface (ethernet/LAG) to an existing management"
         " ACAL rule: %s 11 <macalName> <macalRulePrio> <ifNum>\n", name);
  printf("Test 12: To add a vlan to an existing management ACAL rule: %s 12 "
         "<macalName> <macalRulePrio> <vlanId>\n", name);
  printf("Test 13: To add a serviceport to an existing management ACAL rule:"
         " %s 13 <macalName> <macalRulePrio> <isServicePort>\n", name);
  printf("Test 14: Check if this management ACAL rule is for the service port:"
         " %s 14 <macalName> <macalRulePrio> \n", name);
  printf("Test 15: To add a service to management ACAL rule: %s 15 <macalName>"
         " <macalRulePrio> <service>\n", name);
  printf("Test 16: Get the name of the management ACAL already existing: %s 16\n",
          name);
  printf("Test 17: Get the active management ACAL: %s 17 \n", name);
  printf("Test 18: Get the first management ACAL rule index: %s 18 <macalName>\n", 
         name);
  printf("Test 19: Get the next management ACAL rule index: %s 19 <macalName>"
         " <macalRulePrio> \n", name);
  printf("Test 20: Check if this management ACAL rule with given priority exists:"
         " %s 20 <macalName> <macalRulePrio>\n", name);
  printf("Test 21: Get the action for a management ACAL rule: %s 21 <macalName>"
         " <macalRulePrio>\n", name);
  printf("Test 22: Get the source ip address and net mask of a management ACAL"
         " rule: %s 22 <macalName> <macalRulePrio>\n", name);
  printf("Test 23: Get the interface (ethernet/LAG) of an existing management "
         "ACAL rule: %s 23 <macalName> <macalRulePrio>\n", name);
  printf("Test 24: Get the vlan ID of an existing management ACAL rule: %s 24"
         " <macalName> <macalRulePrio>\n", name);
  printf("Test 25: Get the service of a management ACAL rule: %s 25 <macalName> "
         "<macalRulePrio>\n", name);
  printf("Test 26: Get the service port statistics for a management ACAL rule: "
         "%s 26 <macalName> <macalRulePrio>\n", name);
  printf("Test 27: Get the TCP/UDP port number of a management ACAL rule: %s 27 "
         "<macalName> <macalRulePrio>\n", name);
  printf("Test 28: Add the TCP/UDP port number to a management ACAL rule: %s 28"
         " <macalName> <macalRulePrio> <portNum>\n", name);
  printf("Test 29: Checks to see if a user has configured a field in a management"
         " ACAL rule: %s 29 <macalName> <macalRulePrio> <ruleField> \n", name);
  printf("Test 30: Check if a MACAL list is created: %s 30 \n", name);
  printf("Test 31: Check if a MACAL list is activated: %s 31 \n", name);
  printf("Test 32: Get the number of packets filtered: %s 32 \n", name);
  printf("Test 33: Get the macal deny counter interval: %s 33 \n", name);
  printf("Test 34: Set the macal deny counter interval: %s 34 <interval>\n", name);
  printf("Test 35: Get the first available priority (index) for the specified "
         "management ACAL: %s 35 <macalName>\n", name);
  printf("Test 36: Checks the specified VRF name is valid or not: %s 36"
         " <vrfName> \n", name);
  printf("Test 37: Clears  macal service port statistics: %s 37 \n", name);
  printf("Test 38: Check if the Net Filter table support is present on this"
         " platform: %s 38 <isSupported>\n", name);

  return;
}

/*********************************************************************
* @purpose  To create a new management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
*
* @returns  none
*
* @end
*********************************************************************/
void macalListCreate(openapiClientHandle_t *client_handle, 
                     char *macalName)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalListCreate(client_handle, &buf)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to create a new management ACAL."
           " (result = %d)\n", result);
  }
  else
  {
    printf("MACAL list created successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To delete an existing management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
*
* @returns  none
*
* @end
*********************************************************************/
void macalListDelete(openapiClientHandle_t *client_handle, 
                     char *macalName)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalListDelete(client_handle, &buf)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to delete an existing management"
           " ACAL. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL list deleted successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Sets VRF of a management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    vrfNam  e       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void macalListVrfSet(openapiClientHandle_t *client_handle, 
                     char *vrfName)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, vrfName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalListVrfSet(client_handle, &buf)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets VRF of a management ACAL."
           " (result = %d)\n", result);
  }
  else
  {
    printf("VRF for MACAL set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets VRF of a management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void macalListVrfGet(openapiClientHandle_t *client_handle) 
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  if ((result = openapiMacalListVrfGet(client_handle, &buf)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets VRF of a management ACAL."
           " (result = %d)\n", result);
  }
  else
  {
    printf("VRF of MACAL is : %s.\n", name);
  }
  return;
}

/*********************************************************************
* @purpose  Activate a management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
*
* @returns  none
*
* @end
*********************************************************************/
void macalActivate(openapiClientHandle_t *client_handle, 
                   char *macalName)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalActivate(client_handle, &buf)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to activate a management ACAL."
           " (result = %d)\n", result);
  }
  else
  {
    printf("MACAL list activated.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Deactivate a management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
*
* @returns  none
*
* @end
*********************************************************************/
void macalDeactivate(openapiClientHandle_t *client_handle, 
                     char *macalName)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalDeactivate(client_handle, &buf)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to deactivate a management ACAL."
           " (result = %d)\n", result);
  }
  else
  {
    printf("MACAL list deactivated.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To add a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleEntryAdd(openapiClientHandle_t *client_handle, 
                       char *macalName, uint32_t macalRulePrio)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};
  uint32_t      priority = macalRulePrio;

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleEntryAdd(client_handle, &buf, 
                                         &priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add a management ACAL rule. "
           "(result = %d)\n", result);
  }
  else
  {
    if (macalRulePrio == OPEN_MACAL_REQUEST_PRIORITY)
    {
      printf("MACAL rule added successfully and priority of the created "
             "rule is %u\n", priority);
    }
    else
    {
      printf("MACAL rule added successfully.\n");
    }
  }
  return;
}

/*********************************************************************
* @purpose  To remove a management rule from an existing ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleRemove(openapiClientHandle_t *client_handle, 
                     char *macalName, uint32_t macalRulePrio)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleRemove(client_handle, &buf, 
                                       macalRulePrio)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to to remove a management rule from"
           " an existing ACAL. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL rule removed successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To add an action to management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    action          @b{(input)}  Action (permit/deny)
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleActionAdd(openapiClientHandle_t *client_handle, 
                        char *macalName, uint32_t macalRulePrio,
                        OPEN_MACAL_ACTION_t action)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleActionAdd(client_handle, &buf, 
                                          macalRulePrio, action)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add an action to management"
           " ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL rule action added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To add the source ip address and net mask to a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    ipAddr          @b{(input)}  IP address
* @param    mask            @b{(input)}  Net Mask
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleSrcIpMaskAdd(openapiClientHandle_t *client_handle, 
                           char *macalName, uint32_t macalRulePrio,
                           char *ipAddr, char *mask)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};
  uint32_t      inetIpAddr = 0;
  char          str[40] = "";
  open_buffdesc ipBuffdesc = {.pstart = str, .size = 40};;
  uint32_t      inetMaskAddr = 0;
  char          maskStr[40] = "";
  open_buffdesc maskBuffdesc = {.pstart = maskStr, .size = 40};;

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);
  strncpy(str, ipAddr, sizeof(str) - 1);
  strncpy(maskStr, mask, sizeof(maskStr) - 1);

  if ((openapiInetAton(client_handle, &ipBuffdesc, &inetIpAddr) != OPEN_E_NONE) ||
      (openapiInetAton(client_handle, &maskBuffdesc, &inetMaskAddr) != OPEN_E_NONE))
  {
    printf("ERROR: Bad return code trying to convert internet address string "
           "to a 32 bit integer.\n");
    return;
  }

  if ((result = openapiMacalRuleSrcIpMaskAdd(client_handle, &buf, 
                                             macalRulePrio, inetIpAddr, 
                                             inetMaskAddr)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add the source ip address and "
           "net mask to a management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL rule IP and mask added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To add an interface (ethernet/LAG) to an existing management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    ifNum           @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleInterfaceAdd(openapiClientHandle_t *client_handle, 
                           char *macalName, uint32_t macalRulePrio,
                           uint32_t ifNum)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleInterfaceAdd(client_handle, &buf, 
                                             macalRulePrio, ifNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add an interface (ethernet/LAG) "
           "to an existing management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL rule interface added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To add a vlan to an existing management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    vlanId          @b{(input)}  Vlan ID to be added
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleVlanAdd(openapiClientHandle_t *client_handle, 
                      char *macalName, uint32_t macalRulePrio,
                      uint32_t vlanId)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleVlanAdd(client_handle, &buf, 
                                        macalRulePrio, vlanId)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add a vlan to an existing "
           "management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL rule vlan added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To add a serviceport to an existing management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    isServicePort   @b{(input)}  TRUE if the rule is for the service port
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleServicePortSet(openapiClientHandle_t *client_handle, 
                             char *macalName, uint32_t macalRulePrio,
                             OPEN_BOOL_t isServicePort)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleServicePortSet(client_handle, &buf, 
                                               macalRulePrio, isServicePort)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add a serviceport to an existing"
           " management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL rule interface for service port added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Check if this management ACAL rule is for the service port.
*
* @param    client_handle        @b{(input)}  Client handle from registration API
* @param    macalName            @b{(input)}  MACAL name
* @param    macalRulePrio        @b{(input)}  MACAL rule priority. 
*                                             This is used as the rule index
* @param    isRuleForServicePort @b{(output)} TRUE if the rule is for the service 
*                                             port
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleIsServicePortSet(openapiClientHandle_t *client_handle, 
                               char *macalName, 
                               uint32_t macalRulePrio, 
                               OPEN_BOOL_t *isRuleForServicePort)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleIsServicePortSet(client_handle, &buf,
                                                 macalRulePrio, 
                                                 isRuleForServicePort))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if this management ACAL rule is "
           "for the service port. (result = %d)\n", result);
  }
  else
  {
    printf("Given MACAL rule index is %sfor serviceport.\n",
           *isRuleForServicePort == OPEN_TRUE ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  To add a service to management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    service         @b{(input)}  The service to be added
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleServiceAdd(openapiClientHandle_t *client_handle, 
                         char *macalName, uint32_t macalRulePrio,
                         OPEN_MACAL_SERVICE_t service)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleServiceAdd(client_handle, &buf, 
                                           macalRulePrio, service))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to to add a service to management "
           "ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Service for MACAL rule added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the name of the management ACAL already existing.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void macalNameGet(openapiClientHandle_t *client_handle) 
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  if ((result = openapiMacalNameGet(client_handle, &buf))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the name of the management"
           " ACAL already existing. (result = %d)\n", result);
  }
  else
  {
    printf("Name of the MACAL is : %s.\n", name);
  }
  return;
}

/*********************************************************************
* @purpose  Get the active management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void macalActiveListGet(openapiClientHandle_t *client_handle) 
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  if ((result = openapiMacalActiveListGet(client_handle, &buf))
      != OPEN_E_NONE)
  {
    printf("Management ACAL is not activated.\n");
  }
  else
  {
    printf("Name of the active MACAL is : %s.\n", name);
  }
  return;
}

/*********************************************************************
* @purpose  Get the first management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleFirstGet(openapiClientHandle_t *client_handle, char *macalName) 
{
  open_error_t  result;
  uint32_t      macalRulePrio;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);
  if ((result = openapiMacalRuleFirstGet(client_handle, &buf, &macalRulePrio))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first management ACAL rule."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Index of the first MACAL rule is : %d\n", macalRulePrio);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next management ACAL rule.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    macalName         @b{(input)}  MACAL name
* @param    macalRulePrio     @b{(input)}  MACAL rule priority. 
*                                          This is used as index
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleNextGet(openapiClientHandle_t *client_handle, 
                      char *macalName, uint32_t macalRulePrio,
                      uint32_t *nextmacalRulePrio)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleNextGet(client_handle, &buf, 
                                        macalRulePrio, nextmacalRulePrio))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next management ACAL rule."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Index of the next MACAL rule is : %d\n", *nextmacalRulePrio);
  }
  return;
}

/*********************************************************************
* @purpose  Check if this management ACAL rule with given priority exists.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    isIndexExists   @b{(output)} TRUE if the rule exists
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleExists(openapiClientHandle_t *client_handle, 
                     char *macalName, uint32_t macalRulePrio,
                     OPEN_BOOL_t *isIndexExists)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleExists(client_handle, &buf, 
                                       macalRulePrio, isIndexExists))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if this management ACAL "
           "rule with given priority exists. (result = %d)\n", result);
  }
  else
  {
    printf("Rule with the given index %sexists.\n", 
           *isIndexExists == OPEN_TRUE ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Get the action for a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    action          @b{(output)} Action (permit/deny)
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleActionGet(openapiClientHandle_t *client_handle, 
                        char *macalName, uint32_t macalRulePrio,
                        OPEN_MACAL_ACTION_t *action)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleActionGet(client_handle, &buf, 
                                          macalRulePrio, action)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the action for a management "
            "ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Rule action is : %u.\n", *action);
  }
  return;
}

/*********************************************************************
* @purpose  Get the source ip address and net mask of a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleSrcIpMaskGet(openapiClientHandle_t *client_handle, 
                           char *macalName, uint32_t macalRulePrio)
{
  open_error_t     result;
  char             name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc    buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};
  open_inet_addr_t netIpAddr;
  char             str[40] = "";
  open_buffdesc    ipBuffdesc = {.pstart = str, .size = 40};;
  open_inet_addr_t netMaskAddr;
  char             maskStr[40] = "";
  open_buffdesc    maskBuffdesc = {.pstart = maskStr, .size = 40};;

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleSrcIpMaskGet(client_handle, &buf, 
                                             macalRulePrio, &(netIpAddr.addr.ipv4), 
                                             &(netMaskAddr.addr.ipv4))) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the source ip address and net "
           "mask of a management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    netIpAddr.family   = OPEN_AF_INET;  
    netMaskAddr.family = OPEN_AF_INET;  
    (void)openapiOpenIPtoStringGet(client_handle, netIpAddr, &ipBuffdesc);
    (void)openapiOpenIPtoStringGet(client_handle, netMaskAddr, &maskBuffdesc);
    printf("IP address of a MACAl rule is : %s\n", str);
    printf("Net mask of a MACAl rule is : %s\n", maskStr);
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface (ethernet/LAG) of an existing management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    ifNum           @b{(output)} Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleInterfaceGet(openapiClientHandle_t *client_handle, 
                           char *macalName, uint32_t macalRulePrio,
                           uint32_t *ifNum)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleInterfaceGet(client_handle, &buf, 
                                             macalRulePrio, ifNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the interface (ethernet/LAG) "
           "of an existing management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Interface of a MACAL rule is : %u.\n", *ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the vlan ID of an existing management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    vlanId          @b{(output)} Vlan ID
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleVlanGet(openapiClientHandle_t *client_handle, 
                      char *macalName, uint32_t macalRulePrio,
                      uint32_t *vlanId)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleVlanGet(client_handle, &buf, 
                                        macalRulePrio, vlanId)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to geta the vlan ID of an existing "
           "management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("VLAN ID of a MACAL rule is : %u.\n", *vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Get the service of a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    service         @b{(output)} Service like telnet, ssh ...etc
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleServiceGet(openapiClientHandle_t *client_handle, 
                         char *macalName, uint32_t macalRulePrio, 
                         OPEN_MACAL_SERVICE_t *service)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleServiceGet(client_handle, &buf, 
                                           macalRulePrio, service)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the service of a management"
           " ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Service of a MACAL rule is : %u.\n", *service);
  }
  return;
}

/*********************************************************************
* @purpose  Get the service port statistics for a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    packets         @b{(output)} Number of packets
* @param    bytes           @b{(output)} Number of bytes
*
* @returns  none
*
* @end
*********************************************************************/
void macalServicePortRuleStatisticsGet(openapiClientHandle_t *client_handle, 
                                       char *macalName, 
                                       uint32_t macalRulePrio, 
                                       uint32_t *packets, uint32_t *bytes)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalServicePortRuleStatisticsGet(client_handle, 
                                                         &buf, macalRulePrio,
                                                         packets, bytes)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the service port statistics for "
           "a management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Number of packets : %u.\n", *packets);
    printf("Number of bytes   : %u.\n", *bytes);
  }
  return;
}

/*********************************************************************
* @purpose  Get the TCP/UDP port number of a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    portNum         @b{(output)} TCP/UDP port number
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleTcpUdpPortNumberGet(openapiClientHandle_t *client_handle, 
                                  char *macalName, 
                                  uint32_t macalRulePrio, 
                                  uint32_t *portNum)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleTcpUdpPortNumberGet(client_handle, 
                                                    &buf, macalRulePrio,
                                                    portNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the TCP/UDP port number of a "
           "management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("MACAL port number is : %u.\n", *portNum);
  }
  return;
}

/*********************************************************************
* @purpose  Add the TCP/UDP port number to a management ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    portNum         @b{(input)}  TCP/UDP port number
*
* @returns  none
*
* @end
*********************************************************************/
void macalRuleTcpUdpPortNumberAdd(openapiClientHandle_t *client_handle, 
                                  char *macalName, 
                                  uint32_t macalRulePrio, 
                                  uint32_t portNum)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRuleTcpUdpPortNumberAdd(client_handle, &buf,
                                                    macalRulePrio, portNum))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to add the TCP/UDP port number to a"
           " management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Port number to MACAL rule added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Checks to see if a user has configured a field in a management 
*           ACAL rule.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(input)}  MACAL rule priority. 
*                                        This is used as the rule index
* @param    ruleField       @b{(input)}  Management ACAL rule field
* @param    isConfigured    @b{(output)} TRUE if the specified filed is 
*                                        configured
*
* @returns  none
*
* @end
*********************************************************************/
void macalIsFieldConfigured(openapiClientHandle_t *client_handle, 
                            char *macalName, 
                            uint32_t macalRulePrio, 
                            OPEN_MACAL_RULE_FIELD_t ruleField, 
                            OPEN_BOOL_t *isConfigured)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalIsFieldConfigured(client_handle, &buf, 
                                              macalRulePrio, ruleField,
                                              isConfigured)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to checks to see if a user has configured"
           " a field in a management ACAL rule. (result = %d)\n", result);
  }
  else
  {
    printf("Given rule field %sconfigured\n",
           *isConfigured == OPEN_TRUE ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Check if a MACAL list is created.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    isCreated       @b{(output)} TRUE if the MACAL is created
*
* @returns  none
*
* @end
*********************************************************************/
void macalIsAnyListCreated(openapiClientHandle_t *client_handle, 
                           OPEN_BOOL_t *isCreated)
{
  open_error_t result;

  if ((result = openapiMacalIsAnyListCreated(client_handle, isCreated)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if a MACAL list is created."
           " (result = %d)\n", result);
  }
  else
  {
    printf("List %screated\n", *isCreated == OPEN_TRUE ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Check if a MACAL list is activated.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    isActivated     @b{(output)} TRUE if the MACAL is activated
*
* @returns  none
*
* @end
*********************************************************************/
void macalIsAnyListActivated(openapiClientHandle_t *client_handle, 
                             OPEN_BOOL_t *isActivated)
{
  open_error_t result;

  if ((result = openapiMacalIsAnyListActivated(client_handle, isActivated))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if a MACAL list is activated. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("List %sactivated\n", *isActivated == OPEN_TRUE ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of packets filtered.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    filtPackets     @b{(output)} Number of packets filtered by MACAL
*
* @returns  none
*
* @end
*********************************************************************/
void macalStatsPacketsFilteredGet(openapiClientHandle_t *client_handle, 
                                  uint32_t *filtPackets)
{
  open_error_t result;

  if ((result = openapiMacalStatsPacketsFilteredGet(client_handle, 
                                                    filtPackets)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of packets filtered."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Number of filtered packets : %u.\n", *filtPackets);
  }
  return;
}

/*********************************************************************
* @purpose  Get the macal deny counter interval.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    interval        @b{(output)} Deny counter interval
*
* @returns  none
*
* @end
*********************************************************************/
void macalDenyCounterIntervalGet(openapiClientHandle_t *client_handle, 
                                 uint32_t *interval)
{
  open_error_t result;

  if ((result = openapiMacalDenyCounterIntervalGet(client_handle, interval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the macal deny counter interval."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Deny counter interval : %u.\n", *interval);
  }
  return;
}

/*********************************************************************
* @purpose  Set the macal deny counter interval.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    interval        @b{(input)}  Deny counter interval
*
* @returns  none
*
* @end
*********************************************************************/
void macalDenyCounterIntervalSet(openapiClientHandle_t *client_handle, 
                                 uint32_t interval)
{
  open_error_t result;

  if ((result = openapiMacalDenyCounterIntervalSet(client_handle, interval)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the macal deny counter interval. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Deny counter interval set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the first available priority (index) for the specified
*           management ACAL.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macalName       @b{(input)}  MACAL name
* @param    macalRulePrio   @b{(output)} MACAL rule priority. 
*                                        This is used as the rule index
*
* @returns  none
*
* @end
*********************************************************************/
void macalRulePriorityIndexGet(openapiClientHandle_t *client_handle, 
                               char *macalName, 
                               uint32_t *macalRulePrio)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, macalName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalRulePriorityIndexGet(client_handle, &buf,
                                                 macalRulePrio)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first available priority (index)"
           " for the specified management ACAL. (result = %d)\n", result);
  }
  else
  {
    printf("First index in the given MACAL is : %u.\n", *macalRulePrio);
  }
  return;
}

/*********************************************************************
* @purpose  Checks the specified VRF name is valid or not.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    vrfName         @b{(input)}  VRF name
* @param    isValid         @b{(output)} TRUE if the VRF is valid
*
* @returns  none
*
* @end
*********************************************************************/
void macalVrfNameValid(openapiClientHandle_t *client_handle, 
                       char *vrfName, OPEN_BOOL_t *isValid)
{
  open_error_t  result;
  char          name[MACAL_NAME_MAX_LEN+1] = "";
  open_buffdesc buf = {.pstart = name, .size = MACAL_NAME_MAX_LEN};

  strncpy (name, vrfName, MACAL_NAME_MAX_LEN);

  if ((result = openapiMacalVrfNameValid(client_handle, &buf, isValid)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to checks the specified VRF name is "
           "valid or not. (result = %d)\n", result);
  }
  else
  {
    printf("%s is %sa valid VRF.\n", vrfName, 
           (*isValid == OPEN_TRUE) ? "" :"not ");
  }
  return;
}

/*********************************************************************
* @purpose  Clears  macal service port statistics.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void macalClearServicePortStatistics(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiMacalClearServicePortStatistics(client_handle)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to clears  macal service port statistics."
           " (result = %d)\n", result);
  }
  else
  {
    printf("MACAL service port statistics cleared successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Check if the Net Filter table support is present on this platform.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    isSupported     @b{(output)} TRUE if the platform supports net 
*                                        filter table
*
* @returns  none
*
* @end
*********************************************************************/
void macalIsNetFilterTablesFeatureSupported(openapiClientHandle_t *client_handle,
                                            OPEN_BOOL_t *isSupported)
{
  open_error_t result;

  if ((result = openapiMacalIsNetFilterTablesFeatureSupported(client_handle, 
                                                              isSupported)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the Net Filter table support is"
           " present on this platform. (result = %d\n", result);
  }
  else
  {
    printf("Filter table is %ssupported on this platform.\n", 
           (*isSupported == OPEN_TRUE) ? "" :"not ");
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Macal
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, val, val1;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_BOOL_t flag;
  OPEN_MACAL_SERVICE_t service;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Macal example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Macal API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        macalListCreate(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        macalListDelete(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        macalListVrfSet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        macalListVrfGet(&client_handle);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        macalActivate(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        macalDeactivate(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
        macalRuleEntryAdd(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 4)
      {
        macalRuleRemove(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 5)
      {
        macalRuleActionAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 6)
      {
        macalRuleSrcIpMaskAdd(&client_handle, argv[2], atoi(argv[3]), 
                              argv[4], argv[5]);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 5)
      {
        macalRuleInterfaceAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 5)
      {
        macalRuleVlanAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 5)
      {
        macalRuleServicePortSet(&client_handle, argv[2], atoi(argv[3]), 
                                atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 4)
      {
        macalRuleIsServicePortSet(&client_handle, argv[2], atoi(argv[3]), 
                                  &flag);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 5)
      {
        macalRuleServiceAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 2)
      {
        macalNameGet(&client_handle);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 2)
      {
        macalActiveListGet(&client_handle);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 3)
      {
        macalRuleFirstGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 4)
      {
        macalRuleNextGet(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 4)
      {
        macalRuleExists(&client_handle, argv[2], atoi(argv[3]), &flag);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 4)
      {
        macalRuleActionGet(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 4)
      {
        macalRuleSrcIpMaskGet(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 4)
      {
        macalRuleInterfaceGet(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 4)
      {
        macalRuleVlanGet(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 4)
      {
        macalRuleServiceGet(&client_handle, argv[2], atoi(argv[3]), &service);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 4)
      {
        macalServicePortRuleStatisticsGet(&client_handle, argv[2],
                                          atoi(argv[3]), &val, &val1);  
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 4)
      {
        macalRuleTcpUdpPortNumberGet(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 5)
      {
        macalRuleTcpUdpPortNumberAdd(&client_handle, argv[2], atoi(argv[3]),
                                     atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 5)
      {
        macalIsFieldConfigured(&client_handle, argv[2], atoi(argv[3]), 
                               atoi(argv[4]), &flag);
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 2)
      {
        macalIsAnyListCreated(&client_handle, &flag);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 2)
      {
        macalIsAnyListActivated(&client_handle, &flag);
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 2)
      {
        macalStatsPacketsFilteredGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 2)
      {
        macalDenyCounterIntervalGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 34:
      if (argc == 3)
      {
        macalDenyCounterIntervalSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 35:
      if (argc == 3)
      {
        macalRulePriorityIndexGet(&client_handle, argv[2], &val);
        show_help = 0;
      }
      break;
    case 36:
      if (argc == 3)
      {
        macalVrfNameValid(&client_handle, argv[2], &flag);
        show_help = 0;
      }
      break;
    case 37:
      if (argc == 2)
      {
        macalClearServicePortStatistics(&client_handle);
        show_help = 0;
      }
      break;
    case 38:
      if (argc == 2)
      {
        macalIsNetFilterTablesFeatureSupported(&client_handle, &flag);
        show_help = 0;
      }
      break;
    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Macal API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
