/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mpls_example.c
*
* @purpose   MPLS APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    20/01/2014
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mpls.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

char *pStrInfo_mpls_na = "N/A";
char *pStrInfo_mpls_InternalError = "Internal error";
char *pStrInfo_mpls_ipv4 = "ipv4";
char *pStrInfo_mpls_ipv6 = "ipv6";
char *pStrInfo_mpls_Static = "Static";
char *pStrInfo_mpls_Dynamic = "Dynamic";
char *pStrInfo_mpls_pop = "pop";
char *pStrInfo_mpls_last_pop = "last pop";
char *pStrInfo_mpls_swap = "swap";
char *pStrInfo_mpls_bgp = "BGP";
char *pStrInfo_mpls_layer2_1 = "Layer-2";
char *pStrInfo_mpls_inserted = "Inserted";
char *pStrInfo_mpls_not_inserted = "Not Inserted";
char *pStrInfo_mpls_reason_unknown_subnet = "Unknown Subnet";
char *pStrInfo_mpls_reason_local_subnet = "Local Subnet";
char *pStrInfo_mpls_reason_out_of_resources = "Out of resources";
char *pStrInfo_mpls_reason_port_not_available = "Egress port is not available";
char *pStrInfo_mpls_reason_no_vlan = "No VLAN";
char *pStrInfo_mpls_header_1 = "Label:%u  Protocol:%s  Type:%s  Subnet:%s\r\n";
char *pStrInfo_mpls_header_2 = "     Egress Label Action:%s  Egress Label:%s\r\n";
char *pStrInfo_mpls_header_3 = "     Egress Interface Port:%s  Vlan:%s  MAC:%s\r\n";
char *pStrInfo_mpls_header_4 = "     Hardware Status:%s  Not Inserted Reason:%s\r\n";
char *pStrInfo_mpls_header_5 = "     Byte Count:%llu  Packet Count:%llu\r\n";
char *pStrInfo_mpls_header_6 = "     Duplicate insertion attempts:%u\r\n";
char *pStrInfo_mpls_header_7 = "Next Hop IP Address: %s  \r\n";
char *pStrInfo_mpls_header_8 = "     Egress Port: %u  Vlan: %u  MAC: %s\r\n";
char *pStrInfo_mpls_header_9 = "     Seconds since last update: %u Number of Routes: %u\r\n";


static void lfdbEntryPrint(OPEN_MPLS_LFDB_ENTRY_t *lfdb_entry)
{
  char  strIp[80] = {0};
  char  strMac[80] = {0};
  char  strVlan[80] = {0};
  char  strIntf[80] = {0};
  char  strReason[80] = {0};
  char  strEgressLabel[80] = {0};
  char  strTmp[80] = {0};

  if (OPEN_MPLS_TYPE_IPV4 == lfdb_entry->entryType)
  {
    inet_ntop(AF_INET, &lfdb_entry->ipv4Addr, strIp, sizeof(strIp));

    sprintf(strTmp, "/%u", lfdb_entry->ipv4PrefixLen);
    strncat(strIp, strTmp, sizeof(strIp) - strlen(strTmp));

    strncpy(strMac, pStrInfo_mpls_na, (sizeof(strMac) - 1));
    strncpy(strVlan, pStrInfo_mpls_na, (sizeof(strVlan) - 1));
    strncpy(strIntf, pStrInfo_mpls_na, (sizeof(strIntf) - 1));
  }
  else if (OPEN_MPLS_TYPE_IPV6 == lfdb_entry->entryType)
  {
    inet_ntop(AF_INET6, lfdb_entry->ipv6Subnet.u.addr8, strIp, sizeof(strIp));

    sprintf(strTmp, "/%u", lfdb_entry->ipv6PrefixLen);
    strncat(strIp, strTmp, sizeof(strIp) - strlen(strTmp));

    strncpy(strMac, pStrInfo_mpls_na, (sizeof(strMac) - 1));
    strncpy(strVlan, pStrInfo_mpls_na, (sizeof(strVlan) - 1));
    strncpy(strIntf, pStrInfo_mpls_na, (sizeof(strIntf) - 1));
  }
  else
  {
    strncpy(strIp, pStrInfo_mpls_na, (sizeof(strIp) - 1));

    snprintf(strMac, sizeof(strMac), "%02X:%02X:%02X:%02X:%02X:%02X",
             lfdb_entry->egressMac[0], lfdb_entry->egressMac[1], lfdb_entry->egressMac[2], 
             lfdb_entry->egressMac[3], lfdb_entry->egressMac[4], lfdb_entry->egressMac[5]);

    snprintf(strVlan, sizeof(strVlan), "%u", lfdb_entry->egressVlan);
    snprintf(strIntf, sizeof(strIntf), "%u", lfdb_entry->egressInterface);
  }

  if (0 != lfdb_entry->egressLabel)
  {
    snprintf(strEgressLabel, sizeof(strEgressLabel), "%u", lfdb_entry->egressLabel); 
  }
  else
  {
    strncpy(strEgressLabel, pStrInfo_mpls_na, (sizeof(strEgressLabel) -1 ));
  }

  if (OPEN_MPLS_LABEL_IN_HARDWARE == lfdb_entry->inHardware)
  {
    strncpy(strReason, pStrInfo_mpls_na, (sizeof(strReason) - 1));
  }
  else
  {
    if (OPEN_MPLS_UNKNOWN_SUBNET == lfdb_entry->notInsertedReason)
    {
      strncpy(strReason, pStrInfo_mpls_reason_unknown_subnet, (sizeof(strReason) - 1));
    }
    else if (OPEN_MPLS_LOCAL_SUBNET == lfdb_entry->notInsertedReason)
    {
      strncpy(strReason, pStrInfo_mpls_reason_local_subnet, (sizeof(strReason) - 1));
    }
    else if (OPEN_MPLS_NO_RESOURCES == lfdb_entry->notInsertedReason)
    {
      strncpy(strReason, pStrInfo_mpls_reason_out_of_resources, (sizeof(strReason) - 1));
    }
    else if (OPEN_MPLS_NO_VLAN == lfdb_entry->notInsertedReason)
    {
      strncpy(strReason, pStrInfo_mpls_reason_no_vlan, (sizeof(strReason) - 1));
    }
    else if (OPEN_MPLS_NO_PORT == lfdb_entry->notInsertedReason)
    {
      strncpy(strReason, pStrInfo_mpls_reason_port_not_available, (sizeof(strReason) - 1));
    }
    else
    {
      strncpy(strReason, pStrInfo_mpls_InternalError, (sizeof(strReason) - 1));
    }
  }

  printf(pStrInfo_mpls_header_1, 
         lfdb_entry->ingressLabel, 
         (lfdb_entry->protocol == OPEN_MPLS_PROTO_STATIC) ? pStrInfo_mpls_Static :
         (lfdb_entry->protocol == OPEN_MPLS_PROTO_DYNAMIC) ? pStrInfo_mpls_Dynamic :
         (lfdb_entry->protocol == OPEN_MPLS_PROTO_BGP) ? pStrInfo_mpls_bgp : pStrInfo_mpls_na,
         (lfdb_entry->entryType == OPEN_MPLS_TYPE_LAYER_2) ? pStrInfo_mpls_layer2_1 :
         (lfdb_entry->entryType == OPEN_MPLS_TYPE_IPV4) ? pStrInfo_mpls_ipv4 :
         (lfdb_entry->entryType == OPEN_MPLS_TYPE_IPV6) ? pStrInfo_mpls_ipv6 : pStrInfo_mpls_na,
         strIp);

  printf(pStrInfo_mpls_header_2, 
         (OPEN_MPLS_LABEL_POP      == lfdb_entry->labelAction) ? pStrInfo_mpls_pop      :
         (OPEN_MPLS_LABEL_LAST_POP == lfdb_entry->labelAction) ? pStrInfo_mpls_last_pop :
         (OPEN_MPLS_LABEL_SWAP     == lfdb_entry->labelAction) ? pStrInfo_mpls_swap     : pStrInfo_mpls_na,
         strEgressLabel);

  printf(pStrInfo_mpls_header_3, strIntf, strVlan, strMac);

  printf(pStrInfo_mpls_header_4, 
         (OPEN_MPLS_LABEL_IN_HARDWARE == lfdb_entry->inHardware) ? pStrInfo_mpls_inserted : pStrInfo_mpls_not_inserted,
         strReason);

  printf(pStrInfo_mpls_header_5, lfdb_entry->inBytes, lfdb_entry->inPackets);
  printf(pStrInfo_mpls_header_6, lfdb_entry->duplicateInsertions);

  printf("\n");
}

/*******************************************************************
*
* @brief  This function prints the MPLS Example Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printMplsAppMenu()
{
  printf("Usage: mpls_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set MPLS BGP label mode: mpls_example 1 <mode: 0-off/1-on> \n");
  printf("Test 2: Get MPLS BGP label mode: mpls_example 2 \n");
  printf("Test 3: Set MPLS BGP label for an interface: mpls_example 3 <interface> <IPv4 = 1/IPv6 = 2> <label ID> \n");
  printf("Test 4: Get MPLS BGP label for an interface: mpls_example 4 <interface> \n");
  printf("Test 5: Get MPLS global status and statistics: mpls_example 5 \n");
  printf("Test 6: Show all MPLS LFDB entries: mpls_example 6 \n");
  printf("Test 7: Create L2 MPLS LFDB entry (Action: pop): \n");
  printf("        mpls_example 7 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <interface> <vlan> <mac> \n");
  printf("Test 8: Create L2 MPLS LFDB entry (Action: last pop): \n");
  printf("        mpls_example 8 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <interface> <vlan> <mac> \n");
  printf("Test 9: Create L2 MPLS LFDB entry (Action: swap): \n");
  printf("        mpls_example 9 <ingress label ID> <new label ID> <static = 1/dynamic = 2/BGP = 3> <interface> <vlan> <mac> \n");
  printf("Test 10: Create L3(IPv4) MPLS LFDB entry: \n");
  printf("         mpls_example 10 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <pop = 1/last pop = 2/swap = 3> <subnet> <prefix> \n");
  printf("Test 11: Create L3(IPv6) MPLS LFDB entry: \n");
  printf("         mpls_example 11 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <pop = 1/last pop = 2/swap = 3> <subnet> <prefix> \n");
  printf("Test 12: Delete MPLS LFDB entry: mpls_example 12 <ingress label ID> \n");
  printf("Test 13: Clear MPLS statistics: mpls_example 13 \n");
  printf("Test 14: MPLS OpEN APIs Sanity: mpls_example 14 \n");
  printf("Test 15: Show all MPLS tunnel initiator entries: mpls_example 15 \n");
  return;
}

/*********************************************************************
* @purpose  Set MPLS BGP label mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   MPL BGP label mode
* 
* @returns  none
*   
* @notes    Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void mplsBgpModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiMplsBgpLabelModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set MPLS BGP label mode. (result = %d) \n", result);
  }
  else
  {
    printf("MPLS BGP mode %d is set successfully. (result = %d) \n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get MPLS BGP label mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsBgpModeGet(openapiClientHandle_t *clientHandle)
{
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiMplsBgpLabelModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MPLS BGP label mode. (result = %d) \n", result);
  }
  else
  {
    printf("MPLS BGP mode: %s. (result = %d) \n", (mode == OPEN_DISABLE) ? "Disable" : "Enable", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set MPLS BGP label for an interface.
* 
* @param    clientHandle     @b{(input)}  client handle from registration API 
* @param    intIfNum         @b{(input)}  the interface for which to set the label
* @param    type             @b{(input)}  type of label
* @param    label            @b{(input)}  MPLS BGP label
* 
* @returns  none
*   
* @notes    Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void mplsBgpIntfLabelSet(openapiClientHandle_t *clientHandle, 
                         uint32_t intIfNum, uint32_t type, uint32_t label)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiMplsBgpIntfLabelSet(clientHandle, intIfNum, 
                                           (type == 1) ? OPEN_AFX_IP4 : OPEN_AFX_IP6, label)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set MPLS BGP label for interface %d. (result = %d) \n", 
           result, intIfNum);
  }
  else
  {
    printf("MPLS BGP label %d is set successfully for interface %d. (result = %d) \n", 
           label, intIfNum, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get MPLS BGP label for an interface.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   the interface for which to get the label 
*
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsBgpIntfLabelGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  uint32_t label = 0;
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiMplsBgpIntfLabelGet(clientHandle, intIfNum, 
                                           OPEN_AFX_IP4, &label)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MPLS BGP label for interface %d. (result = %d) \n", 
           intIfNum, result);
  }
  else
  {
    printf("MPLS BGP IPv4 label: %d. (result = %d) \n", label, result);
  }

  if ((result = openapiMplsBgpIntfLabelGet(clientHandle, intIfNum, 
                                           OPEN_AFX_IP6, &label)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MPLS BGP label for interface %d. (result = %d) \n", 
           intIfNum, result);
  }
  else
  {
    printf("MPLS BGP IPv6 label: %d. (result = %d) \n", label, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get MPLS BGP label mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsGlobalStatusGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_MPLS_GLOBAL_STATS_t global_status;

  if ((result = openapiMplsGlobalStatusGet(clientHandle, &global_status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MPLS global status. (result = %d) \n", result);
  }
  else
  {
    printf("MPLS global status and statistics:\n");
    printf("MPLS MAC....................................... %02X:%02X:%02X:%02X:%02X:%02X\n", 
           global_status.mplsMac[0], global_status.mplsMac[1], global_status.mplsMac[2], 
           global_status.mplsMac[3], global_status.mplsMac[4], global_status.mplsMac[5]);
    printf("LFDB Size...................................... %u\n", global_status.lfdbSize);
    printf("Maximum number of MPLS tunnels................. %u\n", global_status.maxTunnelInitiators);
    printf("LFDB Label Range............................... %u - %u\n", global_status.minLfdbLabel, global_status.maxLfdbLabel);
    printf("Number of MPLS tunnels......................... %u\n", global_status.numTunnelInitiators);
    printf("Number of MPLS tunnels with 1-label............ %u\n", global_status.numTunnelsOneLabel);
    printf("Number of MPLS tunnels with 2-label............ %u\n", global_status.numTunnelsTwoLabel);
    printf("Number of MPLS tunnels with 3-label............ %u\n", global_status.numTunnelsThreeLabel);
    printf("LFDB Entries................................... %u\n", global_status.numLfdbEntries);
    printf("LFDB Entries In Hardware....................... %u\n", global_status.numLfdbInHardware);
    printf("LFDB Entries Not In hardware................... %u\n", global_status.numLfdbNotInHardware);
    printf("LFDB Static Entries............................ %u\n", global_status.numLfdbPerProto[OPEN_MPLS_PROTO_STATIC]);
    printf("LFDB Dynamic Entries........................... %u\n", global_status.numLfdbPerProto[OPEN_MPLS_PROTO_DYNAMIC]);
    printf("LFDB BGP Entries............................... %u\n", global_status.numLfdbPerProto[OPEN_MPLS_PROTO_BGP]);
    printf("LFDB Layer-2 Entries........................... %u\n", global_status.numLfdbPerType[OPEN_MPLS_TYPE_LAYER_2]);
    printf("LFDB IPv4 Entries.............................. %u\n", global_status.numLfdbPerType[OPEN_MPLS_TYPE_IPV4]);
    printf("LFDB IPv6 Entries.............................. %u\n", global_status.numLfdbPerType[OPEN_MPLS_TYPE_IPV6]);
    printf("LFDB Dynamic Insert Failure Count.............. %u\n", global_status.dynamicLfdbInsertFailures);
    printf("LFDB High Water Mark........................... %u\n", global_status.lfdbHighWaterMark);
    printf("ECMP In-Use/High/Max........................... %u/%u/%u\n", global_status.ecmpInUse, 
                                                                         global_status.ecmpHighInUse, 
                                                                         global_status.ecmpMax);
    printf("LFDB Lookup failure packets.................... %llu\n", (unsigned long long) global_status.lfdbLookupFailurePackets);
  }

  return;
}

/*********************************************************************
* @purpose  Get MPLS LFDB entries.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsLfdbEntriesShow(openapiClientHandle_t *clientHandle)
{
  uint32_t label = 0;
  uint32_t counter = 0;
  open_error_t result = OPEN_E_NONE;
  OPEN_MPLS_LFDB_ENTRY_t lfdb_entry;

  while ((result = openapiMplsLfdbNextGet(clientHandle, label, &lfdb_entry)) == OPEN_E_NONE)
  {
    label = lfdb_entry.ingressLabel;
    lfdbEntryPrint(&lfdb_entry);
    counter++;
  }

  if (!counter)
  {
    printf("No label is found.\n");
  }

  if ((result != OPEN_E_NONE) && (result != OPEN_E_NOT_FOUND))
  {
    printf("Bad return code trying to get MPLS LFDB entries. (result = %d) \n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Create L2 MPLS LFDB entry.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsL2LfdbEntryCreate(openapiClientHandle_t *clientHandle, uint32_t label, uint32_t protocol, uint32_t action,
                           uint32_t interface, uint32_t vlan, uint32_t egress_label, char *mac)
{
  uint32_t iMac[OPEN_MAC_ADDR_LEN] = {0};
  uint32_t index = 0;
  open_error_t result = OPEN_E_NONE;
  OPEN_MPLS_LFDB_ENTRY_t lfdb_entry;
  OPEN_MPLS_LFDB_ENTRY_t lfdb_entry_tmp;

  memset(&lfdb_entry, 0, sizeof(lfdb_entry));

  lfdb_entry.ingressLabel = label;
  lfdb_entry.protocol = (protocol == 1) ? OPEN_MPLS_PROTO_STATIC  : 
                        (protocol == 2) ? OPEN_MPLS_PROTO_DYNAMIC : OPEN_MPLS_PROTO_BGP;
  lfdb_entry.entryType = OPEN_MPLS_TYPE_LAYER_2;
  lfdb_entry.labelAction = action;
  lfdb_entry.egressVlan = vlan;
  lfdb_entry.egressLabel = egress_label;
  lfdb_entry.egressInterface = interface;

  sscanf(mac, "%x:%x:%x:%x:%x:%x", &iMac[0], &iMac[1], &iMac[2], &iMac[3], &iMac[4], &iMac[5]);
  for (index = 0; index < OPEN_MAC_ADDR_LEN; index++)
  {
    lfdb_entry.egressMac[index] = (unsigned char) iMac[index];
  }

  if ((result = openapiMplsLfdbGet(clientHandle, lfdb_entry.ingressLabel, &lfdb_entry_tmp)) == OPEN_E_NONE)
  {
    printf("MPLS label %d already exists. (result = %d) \n", lfdb_entry.ingressLabel, result);
    return;
  }

  if ((result = openapiMplsLfdbCreate(clientHandle, &lfdb_entry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create MPLS label. (result = %d) \n", result);
  }
  else
  {
    printf("MPLS label %d is created successfully. (result = %d) \n", label, result);
  }
}

/*********************************************************************
* @purpose  Create L2 MPLS LFDB entry.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsL3LfdbEntryCreate(openapiClientHandle_t *clientHandle, uint32_t label, uint32_t protocol, uint32_t action,
                           uint32_t type, char *subnet, uint32_t prefix_len)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_MPLS_LFDB_ENTRY_t lfdb_entry;
  OPEN_MPLS_LFDB_ENTRY_t lfdb_entry_tmp;

  memset(&lfdb_entry, 0, sizeof(lfdb_entry));

  lfdb_entry.ingressLabel = label;
  lfdb_entry.protocol = (protocol == 1) ? OPEN_MPLS_PROTO_STATIC  : 
                        (protocol == 2) ? OPEN_MPLS_PROTO_DYNAMIC : OPEN_MPLS_PROTO_BGP;
  lfdb_entry.entryType = (type == AF_INET) ? OPEN_MPLS_TYPE_IPV4 : OPEN_MPLS_TYPE_IPV6;
  lfdb_entry.labelAction = (action == 1) ? OPEN_MPLS_LABEL_POP      : 
                           (action == 2) ? OPEN_MPLS_LABEL_LAST_POP : OPEN_MPLS_LABEL_SWAP;

  if (type == AF_INET)
  {
    inet_pton(AF_INET, subnet, &lfdb_entry.ipv4Addr);
    lfdb_entry.ipv4PrefixLen = prefix_len;
  }
  else
  {
    inet_pton(AF_INET6, subnet, &lfdb_entry.ipv6Subnet.u.addr8);
    lfdb_entry.ipv6PrefixLen = prefix_len;
  }

  if ((result = openapiMplsLfdbGet(clientHandle, lfdb_entry.ingressLabel, &lfdb_entry_tmp)) == OPEN_E_NONE)
  {
    printf("MPLS label %d already exists. (result = %d) \n", lfdb_entry.ingressLabel, result);
    return;
  }

  if ((result = openapiMplsLfdbCreate(clientHandle, &lfdb_entry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create MPLS label. (result = %d) \n", result);
  }
  else
  {
    printf("MPLS label %d is created successfully. (result = %d) \n", label, result);
  }
}

/*********************************************************************
* @purpose  Delete MPLS LFDB entry.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    label            @b{(input)}   MPLS label
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsLfdbEntryDelete(openapiClientHandle_t *clientHandle, uint32_t label)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiMplsLfdbDelete(clientHandle, label)) != OPEN_E_NONE)
  {
    if (OPEN_E_NOT_FOUND == result)
    {
      printf("MPLS label %d doesn't exist and cannot be deleted. (result = %d) \n", label, result);
    }
    else
    {
      printf("Bad return code trying to delete MPLS label. (result = %d) \n", result);
    }
  }
  else
  {
    printf("MPLS label %d is deleted successfully. (result = %d) \n", label, result);
  }

  return;
}

/*********************************************************************
* @purpose  Clear MPLS Statistics.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsStatisticClear(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiMplsCountersClear(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear MPLS statistics. (result = %d) \n", result);
  }
  else
  {
    printf("MPLS statistics are cleared successfully. (result = %d) \n", result);
  }

  return;
}

/*********************************************************************
* @purpose  This function sanity checks all MPLS OpEN APIs. All 
*           MPLS OpEN APIs are called with possible NULL and 
*           invalid parameters to check the API robustness.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void mplsOpENAPIsTestSanity(openapiClientHandle_t *clientHandle)
{
  uint32_t iValue = 0;
  open_error_t result = OPEN_E_NONE;
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  OPEN_MPLS_LFDB_ENTRY_t lfdb_entry;
  OPEN_MPLS_GLOBAL_STATS_t global_status;

  printf("Testing MPLS OpEN APIs sanity:\n"); 

  /* openapiMplsBgpIntfLabelGet () */
  printf("\nTesting openapiMplsBgpIntfLabelGet(): \n");

  result = openapiMplsBgpIntfLabelGet(NULL, 0, OPEN_AFX_IP4, &iValue);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsBgpIntfLabelGet(clientHandle, 0, OPEN_AFX_IP4, NULL);
  printf("NULL parameter to MPLS BGP interface label get. (result = %d)\n", result);

  printf("openapiMplsBgpIntfLabelGet() sanity successful. \n");


  /* openapiMplsBgpIntfLabelSet () */
  printf("\nTesting openapiMplsBgpIntfLabelSet(): \n");

  result = openapiMplsBgpIntfLabelSet(NULL, 0, OPEN_AFX_IP4, 0);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiMplsBgpIntfLabelSet() sanity successful. \n");


  /* openapiMplsCountersClear () */
  printf("\nTesting openapiMplsCountersClear(): \n");

  result = openapiMplsCountersClear(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiMplsCountersClear() sanity successful. \n");


  /* openapiMplsGlobalStatusGet () */
  printf("\nTesting openapiMplsGlobalStatusGet(): \n");

  result = openapiMplsGlobalStatusGet(NULL, &global_status);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsGlobalStatusGet(clientHandle, NULL);
  printf("NULL parameter to MPLS global status get. (result = %d)\n", result);

  printf("openapiMplsGlobalStatusGet() sanity successful. \n");


  /* openapiMplsBgpLabelModeGet () */
  printf("\nTesting openapiMplsBgpLabelModeGet(): \n");

  result = openapiMplsBgpLabelModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsBgpLabelModeGet(clientHandle, NULL);
  printf("NULL parameter to MPLS BGP label mode get. (result = %d)\n", result);

  printf("openapiMplsBgpLabelModeGet() sanity successful. \n");


  /* openapiMplsBgpLabelModeSet () */
  printf("\nTesting openapiMplsBgpLabelModeSet(): \n");

  result = openapiMplsBgpLabelModeSet(NULL, OPEN_DISABLE);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsBgpLabelModeSet(clientHandle, 3);
  printf("Set invalid mode parameter. (result = %d)\n", result);

  printf("openapiMplsBgpLabelModeSet() sanity successful. \n");


  /* openapiMplsLfdbGet () */
  printf("\nTesting openapiMplsLfdbGet(): \n");

  result = openapiMplsLfdbGet(NULL, 0, &lfdb_entry);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsLfdbGet(clientHandle, 0, NULL);
  printf("NULL parameter to MPLS label entry. (result = %d)\n", result);

  printf("openapiMplsLfdbGet() sanity successful. \n");

  
  /* openapiMplsLfdbNextGet () */
  printf("\nTesting openapiMplsLfdbNextGet(): \n");

  result = openapiMplsLfdbNextGet(NULL, 0, &lfdb_entry);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsLfdbNextGet(clientHandle, 0, NULL);
  printf("NULL parameter to MPLS label entry. (result = %d)\n", result);

  printf("openapiMplsLfdbNextGet() sanity successful. \n");


  /* openapiMplsLfdbCreate () */
  printf("\nTesting openapiMplsLfdbCreate(): \n");

  result = openapiMplsLfdbCreate(NULL, &lfdb_entry);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsLfdbCreate(clientHandle, NULL);
  printf("NULL parameter to MPLS label entry. (result = %d)\n", result);

  printf("openapiMplsLfdbCreate() sanity successful. \n");


  /* openapiMplsLfdbDelete () */
  printf("\nTesting openapiMplsLfdbDelete(): \n");

  result = openapiMplsLfdbDelete(NULL, 111);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMplsLfdbDelete(clientHandle, 111);
  printf("Invalid label ID is specifeid. (result = %d)\n", result);

  printf("openapiMplsLfdbDelete() sanity successful. \n");

  return;
}

static void mplsTunnelEntryPrint(OPEN_MPLS_TUNNEL_INITIATOR_t *tunnel_entry)
{
  uint32_t i;
  char  strIp[80] = {0};
  char  strMac[80] = {0};

  if (tunnel_entry->nextHop.family == OPEN_AF_INET)
  {
    inet_ntop(AF_INET, &tunnel_entry->nextHop.addr.ipv4, strIp, sizeof(strIp));
  }
  else if (tunnel_entry->nextHop.family == OPEN_AF_INET6)
  {
    inet_ntop(AF_INET6, &tunnel_entry->nextHop.addr.ipv6.u.addr8, strIp, sizeof(strIp));
  }

  snprintf(strMac, sizeof(strMac), "%02X:%02X:%02X:%02X:%02X:%02X",
           tunnel_entry->egressMac[0], tunnel_entry->egressMac[1], tunnel_entry->egressMac[2],
           tunnel_entry->egressMac[3], tunnel_entry->egressMac[4], tunnel_entry->egressMac[5]);

  
  printf(pStrInfo_mpls_header_7, strIp);
  printf( "     MPLS Label(s): ");
  for(i=0; i < OPEN_MAX_MPLS_IMPOSE_LABELS; i++)
  {
    if (tunnel_entry->label[i] != 0)
    {
      printf("%u ", tunnel_entry->label[i]);
    }
  }
  printf("\r\n");

  printf(pStrInfo_mpls_header_8, tunnel_entry->egressIfNum, tunnel_entry->egressVlan, strMac);
  printf(pStrInfo_mpls_header_9, tunnel_entry->age, tunnel_entry->numRoutes);
}

/*********************************************************************
* @purpose  Get MPLS Tunnel entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void mplsTunnelInitiatorShow(openapiClientHandle_t *clientHandle)
{
  uint32_t counter = 0;
  open_error_t result = OPEN_E_NONE;
  OPEN_MPLS_TUNNEL_INITIATOR_t cur_tunnel_entry;
  OPEN_MPLS_TUNNEL_INITIATOR_t next_tunnel_entry;

  memset(&cur_tunnel_entry, 0x00, sizeof(cur_tunnel_entry));
  memset(&next_tunnel_entry, 0x00, sizeof(next_tunnel_entry));
  while ((result = openapiMplsTunnelInitiatorGetNext(clientHandle, &cur_tunnel_entry, 
                                                     &next_tunnel_entry)) == OPEN_E_NONE)
  {
    cur_tunnel_entry = next_tunnel_entry;
    mplsTunnelEntryPrint(&next_tunnel_entry);
    counter++;
  }

  if (!counter)
  {
    printf("No MPLS tunnel initiators found.\n");
  }

  if ((result != OPEN_E_NONE) && (result != OPEN_E_NOT_FOUND))
  {
    printf("Bad return code trying to get MPLS tunnel entries. (result = %d) \n", result);
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1, arg2, arg3, arg4, arg5;

  if (argc < 2)
  {
    printMplsAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("mpls_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting MPLS API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Set MPLS BGP label mode: mpls_example 1 <mode: 0-off / 1-on> */
      if (argc != 3)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      mplsBgpModeSet(&clientHandle, arg1 == 0 ? OPEN_DISABLE : OPEN_ENABLE);
      break;

    case 2: /* Test 2: Get MPLS BGP label mode: mpls_example 2 */
      if (argc != 2)
      {
        printMplsAppMenu();
        exit(1);
      }
      mplsBgpModeGet(&clientHandle);
      break;

    case 3: /* Test 3: Set MPLS BGP label for an interface: mpls_example 3 <interface> <IPv4 = 1/IPv6 = 2> <label ID> */
      if (argc != 5)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      mplsBgpIntfLabelSet(&clientHandle, arg1, arg2, arg3);
      break;

    case 4: /* Test 4: Get MPLS BGP label for an interface: mpls_example 4 <interface> */
      if (argc != 3)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      mplsBgpIntfLabelGet(&clientHandle, arg1);
      break;

    case 5: /* Test 5: Get MPLS global status and statistics: mpls_example 5 */
      if (argc != 2)
      {
        printMplsAppMenu();
        exit(1);
      }
      mplsGlobalStatusGet(&clientHandle);
      break;

    case 6: /* Test 6: Show all MPLS LFDB entries: mpls_example 6 */
      if (argc != 2)
      {
        printMplsAppMenu();
        exit(1);
      }
      mplsLfdbEntriesShow(&clientHandle);
      break;

    case 7: /* mpls_example 7 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <interface> <vlan> <mac> */
      if (argc != 7)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[5]);
      mplsL2LfdbEntryCreate(&clientHandle, arg1, arg2, OPEN_MPLS_LABEL_POP, arg3, arg4, 0, argv[6]);
      break;

    case 8: /* mpls_example 8 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <interface> <vlan> <mac> */
      if (argc != 7)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[5]);
      mplsL2LfdbEntryCreate(&clientHandle, arg1, arg2, OPEN_MPLS_LABEL_LAST_POP, arg3, arg4, 0, argv[6]);
      break;

    case 9: /* mpls_example 9 <ingress label ID> <new label ID> <static = 1/dynamic = 2/BGP = 3> <interface> <vlan> <mac> */
      if (argc != 8)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[5]);
      arg5 = atoi(argv[6]);
      mplsL2LfdbEntryCreate(&clientHandle, arg1, arg3, OPEN_MPLS_LABEL_SWAP, arg4, arg5, arg2, argv[7]);
      break;

    case 10: /* mpls_example 10 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <pop = 1/last pop = 2/swap = 3> <subnet> <prefix> */
      if (argc != 7)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[6]);
      mplsL3LfdbEntryCreate(&clientHandle, arg1, arg2, arg3, AF_INET, argv[5], arg4);
      break;

    case 11: /* mpls_example 11 <ingress label ID> <static = 1/dynamic = 2/BGP = 3> <pop = 1/last pop = 2/swap = 3> <subnet> <prefix> */
      if (argc != 7)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[6]);
      mplsL3LfdbEntryCreate(&clientHandle, arg1, arg2, arg3, AF_INET6, argv[5], arg4);
      break;

    case 12: /* Test 12: Delete MPLS LFDB entry: mpls_example 12 <ingress label ID> */
      if (argc != 3)
      {
        printMplsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      mplsLfdbEntryDelete(&clientHandle, arg1);
      break;

    case 13: /* Test 13: Clear MPLS statistics: mpls_example 13 */
      if (argc != 2)
      {
        printMplsAppMenu();
        exit(1);
      }
      mplsStatisticClear(&clientHandle);
      break;

    case 14: /* Test 14: MPLS OpEN APIs Sanity: mpls_example 14 */
      if (argc != 2)
      {
        printMplsAppMenu();
        exit(1);
      }
      mplsOpENAPIsTestSanity(&clientHandle);
      break;

    case 15: /* Test 15: Show all MPLS tunnel initiator entries: mpls_example 15 */
      if (argc != 2)
      {
        printMplsAppMenu();
        exit(1);
      }
      mplsTunnelInitiatorShow(&clientHandle);
      break;

    default:
      printMplsAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping MPLS API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

