/*********************************************************************
*
*  Copyright 2016-2020 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  poe_example.c
*
* @purpose   PoE Configuration OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    11/17/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_poe.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the PoE Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printPOEAppMenu()
{
  printf("Usage:  poe_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Get the PoE first interface: poe_example 1 <poe-type> \n");
  printf("Test 2: Check PoE support on this platform: poe_example 2 \n");
  printf("Test 3: Determine if the interface is capable of supporting PoE: poe_example 3 <poe-type> <interface> \n");
  printf("Test 4: Get interface port terminal classification: poe_example 4 <interface> \n");
  printf("Test 5: Get interface port sensors: poe_example 5 <interface> \n");
  printf("Test 6: Get operational status of the port detection: poe_example 6 <interface> \n");
  printf("Test 7: Get interface port error code: poe_example 7 <interface> \n");
  printf("Test 8: Get all counters of PSE state machine: poe_example 8 <interface> \n");
  printf("Test 9: Find max power limit: poe_example 9 <interface> \n");
  printf("Test 10: Get detection mode of a unit: poe_example 10 \n");
  printf("Test 11: Set detection mode of a unit: poe_example 11 <detection-mode> \n");
  printf("Test 12: Get the current watt limit the PSE can deliver to a port: poe_example 12 <interface> \n");
  printf("Test 13: Set the current watt limit the PSE can deliver to a port: poe_example 13 <interface> <power-limit> \n");
  printf("Test 14: Get port priority value used in management algorithm: poe_example 14 <interface> \n");
  printf("Test 15: Set port priority value used in management algorithm: poe_example 15 <interface> <power-priority> \n");
  printf("Test 16: Get port detection mode of a given interface: poe_example 16 <interface> \n");
  printf("Test 17: Set port detection mode of a given interface: poe_example 17 <interface> <detection-mode> \n");
  printf("Test 18: Get port ability to provide POE high power: poe_example 18 <interface> \n");
  printf("Test 19: Set port ability to provide POE high power: poe_example 19 <interface> <enable|disable> \n");
  printf("Test 20: Get port ability of power up: poe_example 20 <interface> \n");
  printf("Test 21: Set port ability of power up: poe_example 21 <interface> <af|high-inrush|pre-at|at|upoe> \n");
  printf("Test 22: Get violation type of a given interface: poe_example 22 <interface> \n");
  printf("Test 23: Set violation type of a given interface: poe_example 23 <interface> <violation-type> \n");
  printf("Test 24: Get usage threshold in percent: poe_example 24 \n");
  printf("Test 25: Set usage threshold in percent: poe_example 25 <threshold> \n");
  printf("Test 26: Get port administrative mode: poe_example 26 <interface> \n");
  printf("Test 27: Set port administrative mode: poe_example 27 <interface> <adminMode> \n");
  printf("Test 28: Get unit PoE Management mode: poe_example 28 <unit> \n");
  printf("Test 29: Set unit PoE Management mode: poe_example 29 <unit> <static|static-pri|dynamic|dynamic-pri|class>\n");
  printf("Test 30: Get unit total power consumed: poe_example 30 <unit> \n");
  printf("Test 31: Get firmware version: poe_example 31 <unit> \n");
  printf("Test 32: Get main operational status: poe_example 32 <unit> \n");
  printf("Test 33: Get unit threshold power: poe_example 33 <unit> \n");
  printf("Test 34: Get unit total available power: poe_example 34 <unit> \n");
  printf("Test 35: Display whether POE trap mode is enabled or not : poe_example 35 <unit> \n");
  printf("Test 36: Get interface timer schedule name: poe_example 36 <interface> \n");
  printf("Test 37: Reset PoE on a interface: poe_example 37 <interface> \n");
  printf("Test 38: Reset PoE globally: poe_example 38 \n");
  printf("Test 39: Set PoE administrative mode globally: poe_example 39 <enabled|disabled> \n");
  printf("Test 40: Set PoE trap mode: poe_example 40 <unit> <enabled|disabled> \n");
  printf("Test 41: Set global priority value used in management algorithm: poe_example 41 <critical|high|medium|low> \n");
  printf("Test 42: Set global current watt limit the PSE can deliver: poe_example 42 <power-limit> \n");
  printf("Test 43: Clear timer schedule ID associated with the given interface: poe_example 43 <interface> \n");
  printf("Test 44: Set interface timer schedule name: poe_example 44 <interface> <timerschedulename>\n");
  printf("Test 45: Test PoE OpEN API Sanity: poe_example 45 \n");
}

/*********************************************************************
* @purpose  Get the PoE first interface 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    poeType          @b{(input)}   PoE type
* @param    unit             @b{(input)}   Unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poeFirstInterfaceGet(openapiClientHandle_t *clientHandle, OPEN_POE_TYPE_t poeType, uint32_t unit)
{
  open_error_t result;
  uint32_t interface = 0;

  if ((result = openapiPoeFirstInterfaceGet(clientHandle, poeType, unit, &interface)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PoE first interface. (result = %d)\n", result);
  }
  else
  {
    printf("PoE first interface: %u\n", interface);
  }
}

/*********************************************************************
* @purpose  Check to see if PoE is supported on this platform. 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poeIsSupported(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_BOOL_t flag = OPEN_FALSE;

  if ((result = openapiPoeIsSupported(clientHandle, unit, &flag)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if PoE is supported on this platform. (result = %d)\n", result);
  }
  else
  {
    if (flag == OPEN_TRUE)
    {
      printf("PoE is supported on this platform.\n");
    }
    else
    {
      printf("PoE is not supported on this platform.\n");
    }
  }
}

/*********************************************************************
* @purpose  Determine if the interface is capable of supporting PoE 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    poeType          @b{(input)}   PoE type
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poeIsValidInterface(openapiClientHandle_t *clientHandle, OPEN_POE_TYPE_t poeType, uint32_t interface)
{
  open_error_t result;
  OPEN_BOOL_t flag = OPEN_FALSE;

  if((result = openapiPoeIsValidInterface(clientHandle, poeType, interface, &flag)) != OPEN_E_NONE)
  {
    if(result == OPEN_E_UNAVAIL)
    {
      printf("Interface %d is not a valid poe interface\n", interface);
    }
    else
    {
      printf("Bad return code trying to get port priority value used in management algorithm. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Interface (%u) is capable of supporting PoE.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get interface port terminal classification 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortPowerClassificationsGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  uint32_t powerClass;

  if ((result = openapiPoePsePortPowerClassificationsGet(clientHandle, unit, interface, &powerClass)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get interface port terminal classification. (result = %d)\n", result);
  }
  else
  {
    switch (powerClass)
    {
      case OPEN_POE_PORT_POWER_CLASS0:
        printf("Interface port terminal power class for interface (%u): 0\n", interface);
        break;
      case OPEN_POE_PORT_POWER_CLASS1:
        printf("Interface port terminal power class for interface (%u): 1\n", interface);
        break;
      case OPEN_POE_PORT_POWER_CLASS2:
        printf("Interface port terminal power class for interface (%u): 2\n", interface);
        break;
      case OPEN_POE_PORT_POWER_CLASS3:
        printf("Interface port terminal power class for interface (%u): 3\n", interface);
        break;
      case OPEN_POE_PORT_POWER_CLASS4:
        printf("Interface port terminal power class for interface (%u): 4\n", interface);
        break;
      case OPEN_POE_PORT_POWER_INVALID:
        printf("Interface port terminal power class for interface (%u): Unknown\n", interface);
        break;
      default:
        break;
    }
  }
}

/*********************************************************************
* @purpose  Get interface port terminal sensors
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortSensorsGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  uint32_t power, current, voltage, temperature;

  if ((result = openapiPoePsePortSensorsGet(clientHandle, unit, interface, &power, &current, &voltage, &temperature)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get interface port terminal sensors. (result = %d)\n", result);
  }
  else
  {
    printf("Interface port (%u) terminal sensors: \n"
           "\tPower Output : %u\n"
           "\tCurrent Output : %u\n"
           "\tVoltage Output : %u\n"
           "\tTemperature Output : %u\n", 
           interface, power, current, voltage, temperature);
  }
}

/*********************************************************************
* @purpose  Get operational status of the port detection
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortDetectionStatusGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  uint32_t detectionStatus;

  if ((result = openapiPoePsePortDetectionStatusGet(clientHandle, unit, interface, &detectionStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get operational status of the port detection. (result = %d)\n", result);
  }
  else
  {
    switch (detectionStatus)
    {
      case OPEN_POE_PORT_DETECTION_STATUS_REQUESTING_POWER:
        printf("Operational status of port detection for interface(%u): Requesting Power\n", interface);
        break;
      case OPEN_POE_PORT_DETECTION_STATUS_DISABLED:
        printf("Operational status of port detection for interface(%u): Disabled\n", interface);
        break;
      case OPEN_POE_PORT_DETECTION_STATUS_SEARCHING:
        printf("Operational status of port detection for interface(%u): Searching\n", interface);
        break;
      case OPEN_POE_PORT_DETECTION_STATUS_DELIVERING_POWER:
        printf("Operational status of port detection for interface(%u): Delivering Power\n", interface);
        break;
      case OPEN_POE_PORT_DETECTION_STATUS_FAULT:
        printf("Operational status of port detection for interface(%u): Fault\n", interface);
        break;
      case OPEN_POE_PORT_DETECTION_STATUS_TEST:
        printf("Operational status of port detection for interface(%u): Test\n", interface);
        break;
      case OPEN_POE_PORT_DETECTION_STATUS_OTHER_FAULT:
      default:
        printf("Operational status of port detection for interface(%u): Other Fault\n", interface);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Get interface port error code
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortErrorCodeGet(openapiClientHandle_t *clientHandle, uint32_t interface)
{
  open_error_t result;
  uint32_t errorCode;

  if ((result = openapiPoePsePortErrorCodeGet(clientHandle, interface, &errorCode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get interface port error code. (result = %d)\n", result);
  }
  else
  {
    switch (errorCode)
    {
      case OPEN_POE_PORT_FAULT_STATUS_MPS_ABSENT:
        printf("Port error for interface(%u): MPS Absent\n", interface);
        break;
      case OPEN_POE_PORT_FAULT_STATUS_SHORT:
        printf("Port error for interface(%u): Short\n", interface);
        break;
      case OPEN_POE_PORT_FAULT_STATUS_OVERLOAD:
        printf("Port error for interface(%u): Overload\n", interface);
        break;
      case OPEN_POE_PORT_FAULT_STATUS_POWER_DENIED:
        printf("Port error for interface(%u): Power Denied\n", interface);
        break;
      case OPEN_POE_PORT_FAULT_STATUS_THERMAL_SHUTDOWN:
        printf("Port error for interface(%u): Thermal Shutdown\n", interface);
        break;
      case OPEN_POE_PORT_FAULT_STATUS_STARTUP_FAILURE:
        printf("Port error for interface(%u): Startup Failure\n", interface);
        break;
      case OPEN_POE_PORT_FAULT_STATUS_NONE:
      default:
        printf("Port error for interface(%u): No Error\n", interface);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Get all counters of PSE state machine
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortCountersGet(openapiClientHandle_t *clientHandle, uint32_t interface)
{
  open_error_t result;
  uint32_t mpsAbsentCtr, invalidSignatureCtr, powerDeniedCtr, overloadCtr, shortCtr;

  if ((result = openapiPoePsePortCountersGet(clientHandle, interface, &mpsAbsentCtr, &invalidSignatureCtr, &powerDeniedCtr, &overloadCtr, &shortCtr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get interface port terminal sensors. (result = %d)\n", result);
  }
  else
  {
    printf("Counters of PSE state machine for interface (%u) : \n", interface);
    printf("\tMPS Abscent counter : %u\n", mpsAbsentCtr);
    printf("\tInvalid Signature counter : %u\n", invalidSignatureCtr);
    printf("\tPower Denied counter : %u\n", powerDeniedCtr);
    printf("\tOverload counter : %u\n", overloadCtr);
    printf("\tShort counter : %u\n", shortCtr);
  }
}

/*********************************************************************
* @purpose  Find max power limit 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poeFindMaxPowerLimit(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  uint32_t maxPowerLimit;

  if ((result = openapiPoeFindMaxPowerLimit(clientHandle, unit, interface, &maxPowerLimit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get max power limit for interface. (result = %d)\n", result);
  }
  else
  {
    printf("Max power limit for interface (%u): %u\n", interface, maxPowerLimit);
  }
}

/*********************************************************************
* @purpose  Get detection mode of a unit 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainDetectionModeGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_POE_DETECTION_t detectionMode;

  if ((result = openapiPoePseMainDetectionModeGet(clientHandle, unit, &detectionMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get detection mode of a unit. (result = %d)\n", result);
  }
  else
  {
    switch (detectionMode)
    {
      case OPEN_POE_DETECTION_INVALID:
      case OPEN_POE_DETECTION_LEGACY:
      case OPEN_POE_DETECTION_2PT_DOT3AF:
      case OPEN_POE_DETECTION_2PT_DOT3AF_LEG:
        printf("Detection mode of a unit : Invalid\n");
        break;
      case OPEN_POE_DETECTION_4PT_DOT3AF:
        printf("Detection mode of a unit : 4ptdot3af\n");
        break;
      case OPEN_POE_DETECTION_4PT_DOT3AF_LEG:
        printf("Detection mode of a unit : 4ptdot3af+legacy\n");
        break;
      case OPEN_POE_DETECTION_DOT3BT:
        printf("Detection mode of a unit : dot3bt\n");
        break;
      case OPEN_POE_DETECTION_DOT3BT_LEG:
        printf("Detection mode of a unit : dot3bt+legacy\n");
        break;
      default:
        printf("Detection mode of a unit : None\n");
        break;
    }
  }
}

/*********************************************************************
* @purpose  Set detection mode of a unit 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    detectionMode    @b{(input)}   Detection mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainDetectionModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, OPEN_POE_DETECTION_t detectionMode)
{
  open_error_t result;

  if ((result = openapiPoePseMainDetectionModeSet(clientHandle, unit, detectionMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set detection mode of a unit. (result = %d)\n", result);
  }
  else
  {
    printf("Detection mode of a unit set successfully.\n");
  }
}

/*********************************************************************
* @purpose  Get the current watt limit the PSE can deliver to a port 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortCurPowerLimitGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  uint32_t powerLimit;

  if ((result = openapiPoePsePortCurPowerLimitGet(clientHandle, unit, interface, &powerLimit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current watt limit the PSE can deliver to a port. (result = %d)\n", result);
  }
  else
  {
    printf("Current watt limit the PSE can deliver to a port (%u): %u\n", interface, powerLimit);
  }
}

/*********************************************************************
* @purpose  Set the current watt limit the PSE can deliver to a port 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* @param    powerLimit       @b{(input)}   Power limit
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortCurPowerLimitSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface, uint32_t powerLimit)
{
  open_error_t result;

  if ((result = openapiPoePsePortCurPowerLimitSet(clientHandle, unit, interface, powerLimit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the current watt limit the PSE can deliver to a port. (result = %d)\n", result);
  }
  else
  {
    printf("Current watt limit the PSE can deliver to a port (%u) set successfully.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get port priority value used in management algorithm 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortCurPowerPriorityGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  OPEN_POE_PRIORITY_t powerPriority;

  if ((result = openapiPoePsePortCurPowerPriorityGet(clientHandle, unit, interface, &powerPriority)) != OPEN_E_NONE)
  {
    if(result == OPEN_E_UNAVAIL)
    {
      printf("Interface %d is not a valid poe interface\n", interface);
    }
    else
    {
      printf("Bad return code trying to get port priority value used in management algorithm. (result = %d)\n", result);
    }
  }
  else
  {
    switch (powerPriority)
    {
      case OPEN_POE_PRIORITY_CRITICAL:
        printf("Port priority : Critical\n");
        break;
      case OPEN_POE_PRIORITY_HIGH:
        printf("Port priority : High\n");
        break;
      case OPEN_POE_PRIORITY_MEDIUM:
        printf("Port priority : Medium\n");
        break;
      case OPEN_POE_PRIORITY_LOW:
        printf("Port priority : Low\n");
        break;
      default:
        printf("Port priority : None\n");
        break;
    }
  }
}

/*********************************************************************
* @purpose  Set port priority value used in management algorithm 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* @param    powerPriority    @b{(input)}   Power priority
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortCurPowerPrioritySet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface, uint32_t powerPriority)
{
  open_error_t result;

  if ((result = openapiPoePsePortCurPowerPrioritySet(clientHandle, unit, interface, (OPEN_POE_PRIORITY_t)powerPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set port priority value used in management algorithm. (result = %d)\n", result);
  }
  else
  {
    printf("Port priority value used in management algorithm for interface (%u) set successfully.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get port detection mode of a given interface 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortDetectionModeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  OPEN_POE_DETECTION_t detectionMode;

  if ((result = openapiPoePsePortDetectionModeGet(clientHandle, unit, interface, &detectionMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get port detection mode of a given interface. (result = %d)\n", result);
    }
  }
  else
  {
    switch (detectionMode)
    {
      case OPEN_POE_DETECTION_LEGACY:
        printf("Detection mode of a interface (%u) : Legacy\n", interface);
        break;
      case OPEN_POE_DETECTION_2PT_DOT3AF:
        printf("Detection mode of a interface (%u) : 2Pt-Dot3af\n", interface);
        break;
      case OPEN_POE_DETECTION_2PT_DOT3AF_LEG:
        printf("Detection mode of a interface (%u) : 2Pt-Dot3af + Legacy\n", interface);
        break;
      case OPEN_POE_DETECTION_4PT_DOT3AF:
        printf("Detection mode of a interface (%u) : 4Pt-Dot3af\n", interface);
        break;
      case OPEN_POE_DETECTION_4PT_DOT3AF_LEG:
        printf("Detection mode of a interface (%u) : 4ptdot3af+legacy\n", interface);
        break;
      case OPEN_POE_DETECTION_DOT3BT:
        printf("Detection mode of a interface (%u) : dot3bt\n", interface);
        break;
      case OPEN_POE_DETECTION_DOT3BT_LEG:
        printf("Detection mode of a interface (%u) : dot3bt+legacy\n", interface);
        break;
      case OPEN_POE_DETECTION_INVALID:
      default:
        printf("Detection mode of a interface (%u) : None\n", interface);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Set port detection mode of a given interface 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* @param    detectionMode    @b{(input)}   Detection mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortDetectionModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface, uint32_t detectionMode)
{
  open_error_t result;

  if ((result = openapiPoePsePortDetectionModeSet(clientHandle, unit, interface, (OPEN_POE_DETECTION_t)detectionMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to set port detection mode of a given interface. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Detection mode for interface (%u) set successfully.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get port ability to provide POE high power 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortHighPowerModeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  OPEN_CONTROL_t highPowerMode;

  if ((result = openapiPoePsePortHighPowerModeGet(clientHandle, unit, interface, &highPowerMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get port ability to provide PoE high power. (result = %d)\n", result);
    }
  }
  else
  {
    switch (highPowerMode)
    {
      case OPEN_DISABLE:
        printf("High power mode for interface (%u) : disable\n", interface);
        break;
      case OPEN_ENABLE:
        printf("High power mode for interface (%u) : enable\n", interface);
        break;
      default:
        printf("High power mode for interface (%u) : none\n", interface);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Get POE main operational status
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainOperStatusGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_POE_SYSTEM_OPER_STATUS_t status;

  if ((result = openapiPoePseMainOperStatusGet(clientHandle, unit, &status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get main operational status. (result = %d)\n", result);
  }
  else
  {
    switch (status)
    {
      case OPEN_POE_SYSTEM_OPER_STATUS_ON:
        printf("Main operational status is on\n");
        break;
      case OPEN_POE_SYSTEM_OPER_STATUS_OFF:
        printf("Main operational status is off\n");
        break;
      case OPEN_POE_SYSTEM_OPER_STATUS_FAULTY:
        printf("Main operational status is faulty\n");
        break;
      default:
        printf("Unrecognized main operational status %d\n", status);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Get POE firmware image version.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poeFWImageVersionGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  char buf[OPEN_MAX_POE_CARD_INFO_BUF_SIZE];
  open_buffdesc version;

  version.pstart = buf;
  version.size = sizeof(buf);

  if ((result = openapiPoeFWImageVersionGet(clientHandle, unit, &version)) != OPEN_E_NONE)
  {
     printf("Bad return code trying to get POE firmware version. (result = %d)\n", result);
  }
  else
  {
    printf("POE Firmware version is '%s'\n", buf);
  }
}

/*********************************************************************
* @purpose  Set port ability to provide POE high power 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* @param    highPowerMode    @b{(input)}   High power mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortHighPowerModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface, uint32_t highPowerMode)
{
  open_error_t result;

  if ((result = openapiPoePsePortHighPowerModeSet(clientHandle, unit, interface, (OPEN_CONTROL_t)highPowerMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to set port ability to provide PoE high power. (result = %d)\n", result);
    }
  }
  else
  {
    printf("High power mode for interface (%u) set successfully.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get port ability of power up 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortPowerUpModeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  OPEN_POE_POWER_UP_MODE_t powerUpMode;

  if ((result = openapiPoePsePortPowerUpModeGet(clientHandle, unit, interface, &powerUpMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get port ability of power up. (result = %d)\n", result);
    }
  }
  else
  {
    switch (powerUpMode)
    {
      case OPEN_POE_POWER_UP_MODE_8023AF:
        printf("Power up mode for interface (%u) : af\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_HIGH_INRUSH:
        printf("Power up mode for interface (%u) : high-inrush\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_PRE_8023AT:
        printf("Power up mode for interface (%u) : pre-at\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_8023AT:
        printf("Power up mode for interface (%u) : at\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_UPOE:
        printf("Power up mode for interface (%u) : upoe\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_PRE_8023BT:
        printf("Power up mode for interface (%u) : pre-bt\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_TYPE3:
        printf("Power up mode for interface (%u) : type3\n", interface);
        break;
      case OPEN_POE_POWER_UP_MODE_8023BT:
        printf("Power up mode for interface (%u) : bt\n", interface);
        break;
      default:
        printf("Power up mode for interface (%u) : none\n", interface);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Set port ability of power up 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* @param    powerUpMode      @b{(input)}   Power up mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortPowerUpModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface, uint32_t powerUpMode)
{
  open_error_t result;

  if ((result = openapiPoePsePortPowerUpModeSet(clientHandle, unit, interface, (OPEN_POE_POWER_UP_MODE_t)powerUpMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to set port ability of power up. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Power up mode for interface (%u) set successfully.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get violation type of a given interface 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortViolationTypeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  OPEN_POE_LIMIT_TYPE_t violationType;

  if ((result = openapiPoePsePortViolationTypeGet(clientHandle, unit, interface, &violationType)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get violation type of a given interface. (result = %d)\n", result);
    }
  }
  else
  {
    switch (violationType)
    {
      case OPEN_POE_LIMIT_TYPE_DOT3AF:
        printf("Violation type for interface (%u) : Class Based\n", interface);
        break;
      case OPEN_POE_LIMIT_TYPE_USER:
        printf("Violation type for interface (%u) : User Defined\n", interface);
        break;
      case OPEN_POE_LIMIT_TYPE_INVALID:
      case OPEN_POE_LIMIT_TYPE_NONE:
      default:
        printf("Violation type for interface (%u) : None\n", interface);
        break;
    }
  }
}

/*********************************************************************
* @purpose  Set violation type of a given interface 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
* @param    violationType    @b{(input)}   Violation type
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePsePortViolationTypeSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface, uint32_t violationType)
{
  open_error_t result;

  if ((result = openapiPoePsePortViolationTypeSet(clientHandle, unit, interface, (OPEN_POE_LIMIT_TYPE_t)violationType)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to set violation type of a given interface. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Violation type for interface (%u) set successfully.\n", interface);
  }
}

/*********************************************************************
* @purpose  Get usage threshold in percent
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainUsageThresholdGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t threshold;

  if ((result = openapiPoePseMainUsageThresholdGet(clientHandle, unit, &threshold)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get usage threshold in percent. (result = %d)\n", result);
  }
  else
  {
    printf("Usage threshold : %u\n", threshold);
  }
}

/*********************************************************************
* @purpose  Set usage threshold in percent 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    threshold        @b{(input)}   Threshold
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainUsageThresholdSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t threshold)
{
  open_error_t result;

  if ((result = openapiPoePseMainUsageThresholdSet(clientHandle, unit, threshold)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set usage threshold in percent. (result = %d)\n", result);
  }
  else
  {
    printf("Usage threshold set successfully.\n");
  }
}

void poePsePortAdminEnableGet(openapiClientHandle_t *clientHandle, uint32_t interface)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  result = openapiPoePsePortAdminEnableGet(clientHandle, interface, &adminMode);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get port administrative mode. (result = %d)\n", result);
  }
  else
  {
    switch (adminMode)
    {
      case OPEN_ENABLE:
        printf("Administrative mode for interface (%u): Enabled\n", interface);
        break;
      case OPEN_DISABLE:
        printf("Administrative mode for interface (%u): Disabled\n", interface);
        break;
      default:
        printf("Administrative mode for interface (%u): Undecoded Value = %u\n", interface, adminMode);
        break;
    }
  }
}

void poePsePortAdminEnableSet(openapiClientHandle_t *clientHandle, uint32_t interface, uint32_t adminMode)
{
  open_error_t result;

  result = openapiPoePsePortAdminEnableSet(clientHandle, interface, adminMode);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get port administrative mode. (result = %d)\n", result);
  }
  else
  {
    printf("Mode set successfully.\n");
  }
}

/*********************************************************************
* @purpose  Get the management mode of the unit
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMgmtModeGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_POE_POWER_MGMT_t mode;

  if ((result = openapiPoePseMainPowerMgmtModeGet(clientHandle, unit, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PoE mgmt mode for unit %u. (result = %d)\n", unit, result);
  }
  else
  {
    switch (mode)
    {
      case OPEN_POE_POWER_MGMT_STATIC:
        printf("  PoE mgmt mode is static\n");
        break;
      case OPEN_POE_POWER_MGMT_STATIC_PRI:
        printf("  PoE mgmt mode is static priority\n");
        break;
      case OPEN_POE_POWER_MGMT_DYNAMIC:
        printf("  PoE mgmt mode is dynamic\n");
        break;
      case OPEN_POE_POWER_MGMT_DYNAMIC_PRI:
        printf("  PoE mgmt mode is dynamic priority\n");
        break;
      case OPEN_POE_POWER_MGMT_CLASS:
        printf("  PoE mgmt mode is class\n");
        break;
      default:
        printf("  PoE mgmt mode is invalid\n");
        break;
    }
  }
}

/*********************************************************************
* @purpose  Set the management mode of a unit
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   unit number
* @param    mode             @b{(input)}   management mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMgmtModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, 
                       OPEN_POE_POWER_MGMT_t mode)
{
  open_error_t result;

  if ((result = openapiPoePseMainPowerMgmtModeSet(clientHandle, unit, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set PoE mgmt mode for unit %u. (result = %d)\n", unit, result);
  }
  else
  {
    printf("Management Mode set successfully.\n");
  }
}

/*********************************************************************
* @purpose  Get the power consumed of the unit
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   unit number
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainConsumptionPowerGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t powerconsumed;

  if ((result = openapiPoePseMainConsumptionPowerGet(clientHandle, unit, &powerconsumed)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PoE power consumed for unit %u. (result = %d)\n", unit, result);
  }
  else
  {
    printf("Power Consumed for unit %u = %u milliwatts\n", unit, powerconsumed);
  }
}

/*********************************************************************
* @purpose  Get the threshold power of the unit
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poePseThresholdPowerGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t power;

  if ((result = openapiPoePseThresholdPowerGet(clientHandle, unit, &power)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PoE threshold power for unit %u. (result = %d)\n", unit, result);
  }
  else
  {
    printf("Threshold power for unit %u = %u milliwatts\n", unit, power);
  }
}

/*********************************************************************
* @purpose  Get the total available power of the unit
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poePseTotalAvailablePowerGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t power;

  if ((result = openapiPoePseTotalAvailablePowerGet(clientHandle, unit, &power)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PoE total available power for unit %u. (result = %d)\n", unit, result);
  }
  else
  {
    printf("Total available power for unit %u = %u milliwatts\n", unit, power);
  }
}

/*********************************************************************
* @purpose  Get whether POE trap mode is enabled or not
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void trapPoeGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_CONTROL_t enable;

  if ((result = openapiPoeTrapModeGet(clientHandle, unit, &enable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PoE trap enabled (result = %d)\n", result);
  }
  else
  {
    switch (enable)
    {
      case OPEN_DISABLE:
        printf("Trap mode : disabled\n");
        break;
      case OPEN_ENABLE:
        printf("Trap mode : enabled\n");
        break;
    }
  }
}

/*********************************************************************
* @purpose  Get POE timer schedule name on a interface.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poePortTimerScheduleNameGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;
  char buf[OPEN_MAX_POE_TIMER_NAME_BUF_LEN];
  open_buffdesc timerScheduleName;

  timerScheduleName.pstart = buf;
  timerScheduleName.size = sizeof(buf);

  if ((result = openapiPoePsePortTimerScheduleNameGet(clientHandle, unit, interface, &timerScheduleName)) != OPEN_E_NONE)
  {
     printf("Bad return code trying to get POE port timer schedule name. (result = %d)\n", result);
  }
  else
  {
    printf("POE port timer schedule name is '%s'\n", buf);
  }
}

/*********************************************************************
* @purpose  Reset POE on a interface 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    interface        @b{(input)}   Interface number
*
* @returns  none
*
* @notes
*
* @end 
*********************************************************************/
void poePortReset(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t interface)
{
  open_error_t result;

  if ((result = openapiPoePsePortReset(clientHandle, unit, interface)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to reset PoE on interface (%d) , result = %d\n", interface, result);
  }
  else
  {
    printf("PoE reset on interface was successfully \n");
  }
}

/*********************************************************************
* @purpose  Reset POE globally 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poeReset(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;

  if ((result = openapiPoePseMainReset(clientHandle, unit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to reset PoE, result = %d\n", result);
  }
  else
  {
    printf("Reset PoE successfully \n");
  }
}

/*********************************************************************
* @purpose  Set POE admin mode globally
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    adminMode        @b{(input)}   Mode setting
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poePseMainAdminEnableSet(openapiClientHandle_t *clientHandle, uint32_t unit, OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  result = openapiPoePseMainAdminEnableSet(clientHandle, unit, adminMode);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to set administrative mode(%d). (result = %d)\n", (uint32_t)adminMode, result);
  }
  else
  {
    printf("Mode set successfully.\n");
  }
}

/*********************************************************************
* @purpose  Set POE trap mode
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    trapMode         @b{(input)}   Trap mode setting
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void trapPoeSet(openapiClientHandle_t *clientHandle, uint32_t unit, OPEN_CONTROL_t trapMode)
{
  open_error_t result;

  if ((result = openapiPoeTrapModeSet(clientHandle, unit, trapMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set PoE trap (%d), result = %d\n", (uint32_t)trapMode, result);
  }
  else
  {
    printf("PoE trap mode set successfully \n");
  }
}

/*********************************************************************
* @purpose  Set port priority value used in management algorithm 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    powerPriority    @b{(input)}   Power priority
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainCurPowerPrioritySet(openapiClientHandle_t *clientHandle, uint32_t unit, OPEN_POE_PRIORITY_t powerPriority)
{
  open_error_t result;

  if ((result = openapiPoePseMainCurPowerPrioritySet(clientHandle, unit, powerPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set priority value (%d) used in management algorithm. (result = %d)\n", (uint32_t)powerPriority, result);
  }
  else
  {
    printf("Port priority value (%d) used in management algorithm for all interfaces set successfully.\n", (uint32_t)powerPriority);
  }
}

/*********************************************************************
* @purpose  Set the current watt limit the PSE can deliver on all ports
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    unit             @b{(input)}   Unit number
* @param    powerLimit       @b{(input)}   Power limit
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void poePseMainCurPowerLimitSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t powerLimit)
{
  open_error_t result;

  if ((result = openapiPoePseMainCurPowerLimitSet(clientHandle, unit, powerLimit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the current watt limit the PSE can deliver on all ports. (result = %d)\n", result);
  }
  else
  {
    printf("Current watt limit the PSE can deliver on all ports set successfully.\n");
  }
}

/*********************************************************************
* @purpose  Associate the port with the Timer Schedule. (Extended Version) 
*
* @param    clientHandle    @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch 
*
* @end 
*********************************************************************/
void poePethPsePortTimerCtrlClear(openapiClientHandle_t *clientHandle,
                                  uint32_t ifNum) 
{
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiPoePethPsePortTimerCtrlClear(clientHandle, 
                                                    ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear timer scheduler associated with given interface. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully cleared timer scheduler associated with given interface %u. \n", ifNum);
  }

  return;
}

/*********************************************************************
* @purpose  Set PoE timer schedule name on a interface.
*
* @param    clientHandle      @b{(input)}   client handle from registration API
* @param    unit              @b{(input)}   Unit number
* @param    ifNum             @b{(input)}   Interface number
* @param    timerScheduleName @b{(input)}   Timer Schedule Name
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void poePortTimerScheduleNameSet(openapiClientHandle_t *clientHandle, 
                                 uint32_t unit, 
                                 uint32_t ifNum,
                                 char *timerScheduleName)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, timerScheduleName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiPoePsePortTimerScheduleNameSet(clientHandle, unit, ifNum, &buffDesc)) != OPEN_E_NONE)
  {
     printf("Bad return code trying to set PoE port timer schedule name. (result = %d)\n", result);
  }
  else
  {
    printf("PoE port timer schedule name '%s' is SET on interface %d\n", str, ifNum);
  }
}

/*********************************************************************
* @purpose  Perform Sanity of PoE OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void poeOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t unit = 1;
  uint32_t interface = 1;
  uint32_t outputVal = 0;
  OPEN_POE_POWER_MGMT_t mode;
  uint32_t powerconsumed;
  uint32_t power;
  OPEN_POE_TYPE_t poeType = OPEN_POE_TYPE_POE;
  OPEN_BOOL_t flag = OPEN_FALSE; 
  char buf[OPEN_MAX_POE_CARD_INFO_BUF_SIZE];
  open_buffdesc version;
  open_buffdesc buffdesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  OPEN_POE_SYSTEM_OPER_STATUS_t status;
  OPEN_CONTROL_t enabled;
  char timerScheduleBuf[OPEN_MAX_POE_TIMER_NAME_BUF_LEN];
  open_buffdesc timerSchedule;

  /* openapiPoeFirstInterfaceGet */
  printf("\nTesting openapiPoeFirstInterfaceGet(): \n");

  result = openapiPoeFirstInterfaceGet(NULL, poeType, unit, &interface);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoeFirstInterfaceGet(clientHandle, (OPEN_POE_TYPE_t)8, unit, &interface);
  printf("Invalid PoE type to get PoE first interface. (result = %d)\n", result);

  result = openapiPoeFirstInterfaceGet(clientHandle, poeType, 0, &interface);
  printf("Invalid unit number to get PoE first interface. (result = %d)\n", result);

  result = openapiPoeFirstInterfaceGet(clientHandle, poeType, unit, NULL);
  printf("NULL return address to get PoE first interface. (result = %d)\n", result);

  printf("openapiPoeFirstInterfaceGet() sanity successful. \n");

  /* openapiPoeIsSupported */
  printf("\nTesting openapiPoeIsSupported(): \n");

  result = openapiPoeIsSupported(NULL, unit, &flag);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoeIsSupported(clientHandle, 0, &flag);
  printf("Invalid unit number to determine PoE support. (result = %d)\n", result);

  result = openapiPoeIsSupported(clientHandle, unit, NULL);
  printf("NULL PoE support flag to determine PoE support. (result = %d)\n", result);

  printf("openapiPoeIsSupported() sanity successful. \n");

  /* openapiPoeIsValidInterface */
  printf("\nTesting openapiPoeIsValidInterface(): \n");

  result = openapiPoeIsValidInterface(NULL, poeType, interface, &flag);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoeIsValidInterface(clientHandle, (OPEN_POE_TYPE_t)8, interface, &flag);
  printf("Invalid PoE type to determine if the interface is capable of supporting PoE. (result = %d)\n", result);

  result = openapiPoeIsValidInterface(clientHandle, poeType, -1, &flag);
  printf("Invalid interface to determine if the interface is capable of supporting PoE. (result = %d)\n", result);

  result = openapiPoeIsValidInterface(clientHandle, poeType, interface, NULL);
  printf("NULL validation flag to determine if the interface is capable of supporting PoE. (result = %d)\n", result);

  printf("openapiPoeIsValidInterface() sanity successful. \n");

  /* openapiPoePsePortPowerClassificationsGet */
  printf("\nTesting openapiPoePsePortPowerClassificationsGet(): \n");

  result = openapiPoePsePortPowerClassificationsGet(NULL, unit, interface, &outputVal);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortPowerClassificationsGet(clientHandle, 0, interface, &outputVal);
  printf("Invalid unit number to get interface port terminal classification. (result = %d)\n", result);

  result = openapiPoePsePortPowerClassificationsGet(clientHandle, unit, -1, &outputVal);
  printf("Invalid interface to get interface port terminal classification. (result = %d)\n", result);

  result = openapiPoePsePortPowerClassificationsGet(clientHandle, unit, interface, NULL);
  printf("NULL output parameter to get interface port terminal classification. (result = %d)\n", result);

  printf("openapiPoePsePortPowerClassificationsGet() sanity successful. \n");

  /* openapiPoePsePortSensorsGet */
  printf("\nTesting openapiPoePsePortSensorsGet(): \n");

  result = openapiPoePsePortSensorsGet(NULL, unit, interface, &outputVal, &outputVal, &outputVal, &outputVal);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortSensorsGet(clientHandle, 0, interface, &outputVal, &outputVal, &outputVal, &outputVal);
  printf("Invalid unit number to get interface port terminal sensors. (result = %d)\n", result);

  result = openapiPoePsePortSensorsGet(clientHandle, unit, -1, &outputVal, &outputVal, &outputVal, &outputVal);
  printf("Invalid interface to get interface port terminal sensors. (result = %d)\n", result);

  result = openapiPoePsePortSensorsGet(clientHandle, unit, interface, NULL, NULL, NULL, NULL);
  printf("NULL output parameters to get interface port terminal sensors. (result = %d)\n", result);

  printf("openapiPoePsePortSensorsGet() sanity successful. \n");

  /* openapiPoePsePortDetectionStatusGet */
  printf("\nTesting openapiPoePsePortDetectionStatusGet(): \n");

  result = openapiPoePsePortDetectionStatusGet(NULL, unit, interface, &outputVal);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortDetectionStatusGet(clientHandle, 0, interface, &outputVal);
  printf("Invalid unit number to get operational status of the port detection. (result = %d)\n", result);

  result = openapiPoePsePortDetectionStatusGet(clientHandle, unit, -1, &outputVal);
  printf("Invalid interface to get operational status of the port detection. (result = %d)\n", result);

  result = openapiPoePsePortDetectionStatusGet(clientHandle, unit, interface, NULL);
  printf("NULL output parameter to get operational status of the port detection. (result = %d)\n", result);

  printf("openapiPoePsePortDetectionStatusGet() sanity successful. \n");

  /* openapiPoePsePortErrorCodeGet */
  printf("\nTesting openapiPoePsePortErrorCodeGet(): \n");

  result = openapiPoePsePortErrorCodeGet(NULL, interface, &outputVal);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortErrorCodeGet(clientHandle, -1, &outputVal);
  printf("Invalid interface to get interface error code. (result = %d)\n", result);

  result = openapiPoePsePortErrorCodeGet(clientHandle, interface, NULL);
  printf("NULL output parameter to get interface error code. (result = %d)\n", result);

  printf("openapiPoePsePortErrorCodeGet() sanity successful. \n");

  /* openapiPoePsePortCountersGet */
  printf("\nTesting openapiPoePsePortCountersGet(): \n");

  result = openapiPoePsePortCountersGet(NULL, interface, &outputVal, &outputVal, &outputVal, &outputVal, &outputVal);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortCountersGet(clientHandle, -1, &outputVal, &outputVal, &outputVal, &outputVal, &outputVal);
  printf("Invalid interface to get all counters of PSE state machine. (result = %d)\n", result);

  result = openapiPoePsePortCountersGet(clientHandle, interface, NULL, NULL, NULL, NULL, NULL);
  printf("NULL output parameters to get all counters of PSE state machine. (result = %d)\n", result);
  printf("openapiPoePsePortCountersGet() sanity successful. \n");

  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  /* openapiPoePethPsePortTimerCtrlClear */
  printf("\n Testing openapiPoePethPsePortTimerCtrlClear(): \n");

  result = openapiPoePethPsePortTimerCtrlClear(NULL, interface);
  printf("NULL Client Handle. (result = %d) \n", result);

  result = openapiPoePethPsePortTimerCtrlClear(clientHandle, 0);
  printf("NULL interface. (result = %d) \n", result);
  
  printf("openapiPoePethPsePortTimerCtrlClear() sanity successful. \n");

  /* openapiPoePseMainPowerMgmtModeGet() */
  printf("\nTesting openapiPoePseMainPowerMgmtModeGet(): \n");
  result = openapiPoePseMainPowerMgmtModeGet(NULL, unit, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPoePseMainPowerMgmtModeGet(clientHandle, 0xFF, &mode);
  printf("Invalid unit number. (result = %d)\n", result);
  result = openapiPoePseMainPowerMgmtModeGet(clientHandle, unit, NULL);
  printf("NULL pointer to mode. (result = %d)\n", result);
  printf("openapiPoePseMainPowerMgmtModeGet() sanity successful. \n");

  /* openapiPoePseMainPowerMgmtModeSet() */
  printf("\nTesting openapiPoePseMainPowerMgmtModeSet(): \n");
  mode = OPEN_POE_POWER_MGMT_DYNAMIC;
  result = openapiPoePseMainPowerMgmtModeSet(NULL, unit, mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPoePseMainPowerMgmtModeSet(clientHandle, 0xFF, mode);
  printf("Invalid unit number. (result = %d)\n", result);
  result = openapiPoePseMainPowerMgmtModeSet(clientHandle, unit, 0xFFFFFFFF);
  printf("Invalid mode. (result = %d)\n", result);
  printf("openapiPoePseMainPowerMgmtModeSet() sanity successful. \n");

  /* openapiPoePseMainConsumptionPowerGet() */
  printf("\nTesting openapiPoePseMainConsumptionPowerGet(): \n");
  result = openapiPoePseMainConsumptionPowerGet(NULL, unit, &powerconsumed);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPoePseMainConsumptionPowerGet(clientHandle, 0xFF, &powerconsumed);
  printf("Invalid unit number. (result = %d)\n", result);
  result = openapiPoePseMainConsumptionPowerGet(clientHandle, unit, NULL);
  printf("NULL pointer to power. (result = %d)\n", result);
  printf("openapiPoePseMainConsumptionPowerGet() sanity successful. \n");

  /* openapiPoeFWImageVersionGet */
  printf("\nTesting openapiPoeFWImageVersionGet(): \n");

  version.pstart = buf;
  version.size = sizeof(buf);

  result = openapiPoeFWImageVersionGet(NULL, unit, &version);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoeFWImageVersionGet(clientHandle, 0, &version);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoeFWImageVersionGet(clientHandle, unit, NULL);
  printf("Invalid NULL buffdesc. (result = %d)\n", result);

  version.size = sizeof(buf) + 10;
  result = openapiPoeFWImageVersionGet(clientHandle, unit, &version);
  printf("Invalid buffdesc size. (result = %d)\n", result);

  version.size = sizeof(buf);
  version.pstart = NULL;
  result = openapiPoeFWImageVersionGet(clientHandle, unit, &version);
  printf("Invalid buffdesc pstart. (result = %d)\n", result);

  printf("openapiPoeFWImageVersionGet() sanity successful. \n");

  /* openapiPoePseMainOperStatusGet */
  printf("\nTesting openapiPoePseMainOperStatusGet(): \n");

  result = openapiPoePseMainOperStatusGet(NULL, unit, &status);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseMainOperStatusGet(clientHandle, 0, &status);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoePseMainOperStatusGet(clientHandle, unit, NULL);
  printf("NULL status pointer. (result = %d)\n", result);

  printf("openapiPoePseMainOperStatusGet() sanity successful. \n");

  /* openapiPoePseThresholdPowerGet */
  printf("\nTesting openapiPoePseThresholdPowerGet(): \n");

  result = openapiPoePseThresholdPowerGet(NULL, unit, &power);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseThresholdPowerGet(clientHandle, 0, &power);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoePseThresholdPowerGet(clientHandle, unit, NULL);
  printf("NULL value pointer. (result = %d)\n", result);

  printf("openapiPoePseThresholdPowerGet() sanity successful. \n");

  /* openapiPoePseTotalAvailablePowerGet */
  printf("\nTesting openapiPoePseTotalAvailablePowerGet(): \n");

  result = openapiPoePseTotalAvailablePowerGet(NULL, unit, &power);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseTotalAvailablePowerGet(clientHandle, 0, &power);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoePseTotalAvailablePowerGet(clientHandle, unit, NULL);
  printf("NULL value pointer. (result = %d)\n", result);

  printf("openapiPoePseTotalAvailablePowerGet() sanity successful. \n");

  /* openapiPoeTrapModeGet */
  printf("\nTesting openapiPoeTrapModeGet(): \n");

  result = openapiPoeTrapModeGet(NULL, unit, &enabled);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoeTrapModeGet(clientHandle, 0, &enabled);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoeTrapModeGet(clientHandle, unit, NULL);
  printf("NULL status pointer. (result = %d)\n", result);

  printf("openapiPoeTrapModeGet() sanity successful. \n");

  /* openapiPoePsePortTimerScheduleNameSet */
  printf("\nTesting openapiPoePsePortTimerScheduleNameSet(): \n");

  timerSchedule.pstart = timerScheduleBuf;
  timerSchedule.size = sizeof(timerScheduleBuf);

  result = openapiPoePsePortTimerScheduleNameSet(NULL, unit, interface, &timerSchedule);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortTimerScheduleNameSet(clientHandle, 0, interface, &timerSchedule);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoePsePortTimerScheduleNameSet(clientHandle, 0, -1, &timerSchedule);
  printf("Invalid interface. (result = %d)\n", result);

  result = openapiPoePsePortTimerScheduleNameSet(clientHandle, unit, interface, NULL);
  printf("Invalid NULL buffdesc. (result = %d)\n", result);

  timerSchedule.size = sizeof(timerScheduleBuf) + 10;
  result = openapiPoePsePortTimerScheduleNameSet(clientHandle, unit, interface, &timerSchedule);
  printf("Invalid buffdesc size. (result = %d)\n", result);

  timerSchedule.size = sizeof(timerScheduleBuf);
  timerSchedule.pstart = NULL;
  result = openapiPoePsePortTimerScheduleNameSet(clientHandle, unit, interface, &timerSchedule);
  printf("Invalid buffdesc pstart. (result = %d)\n", result);

  /* openapiPoePsePortTimerScheduleNameGet */
  printf("\nTesting openapiPoePsePortTimerScheduleNameGet(): \n");

  timerSchedule.pstart = timerScheduleBuf;
  timerSchedule.size = sizeof(timerScheduleBuf);

  result = openapiPoePsePortTimerScheduleNameGet(NULL, unit, interface, &timerSchedule);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortTimerScheduleNameGet(clientHandle, 0, interface, &timerSchedule);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoePsePortTimerScheduleNameGet(clientHandle, 0, -1, &timerSchedule);
  printf("Invalid interface. (result = %d)\n", result);

  result = openapiPoePsePortTimerScheduleNameGet(clientHandle, unit, interface, NULL);
  printf("Invalid NULL buffdesc. (result = %d)\n", result);

  timerSchedule.size = sizeof(timerScheduleBuf) + 10;
  result = openapiPoePsePortTimerScheduleNameGet(clientHandle, unit, interface, &timerSchedule);
  printf("Invalid buffdesc size. (result = %d)\n", result);

  timerSchedule.size = sizeof(timerScheduleBuf);
  timerSchedule.pstart = NULL;
  result = openapiPoePsePortTimerScheduleNameGet(clientHandle, unit, interface, &timerSchedule);
  printf("Invalid buffdesc pstart. (result = %d)\n", result);

  /* openapiPoePsePortReset */
  printf("\nTesting openapiPoePsePortReset(): \n");

  result = openapiPoePsePortReset(NULL, unit, interface);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePsePortReset(clientHandle, 0, interface);
  printf("Invalid unit number to reset PoE on interface. (result = %d)\n", result);

  result = openapiPoePsePortReset(clientHandle, unit, -1);
  printf("Invalid interface to reset PoE on interface. (result = %d)\n", result);

  printf("openapiPoePsePortReset() sanity successful. \n");

  /* openapiPoePseMainReset */
  printf("\nTesting openapiPoePseMainReset(): \n");

  result = openapiPoePseMainReset(NULL, unit);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseMainReset(clientHandle, 0);
  printf("Invalid unit number to reset PoE globally. (result = %d)\n", result);

  printf("openapiPoePseMainReset() sanity successful. \n");

  /* openapiPoePseMainPortAdminEnableSet */
  printf("\nTesting openapiPoePseMainPortAdminEnableSet (): \n");

  result = openapiPoePseMainAdminEnableSet (NULL, unit, OPEN_ENABLE);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseMainAdminEnableSet (clientHandle, 0, OPEN_ENABLE);
  printf("Invalid unit. (result = %d)\n", result);

  printf("openapiPoePseMainPortAdminEnableSet() sanity successful. \n");

  /* openapiPoeTrapModeSet */
  printf("\nTesting openapiPoeTrapModeSet(): \n");

  result = openapiPoeTrapModeSet(NULL, unit, OPEN_ENABLE);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoeTrapModeSet(clientHandle, 0, OPEN_ENABLE);
  printf("Invalid unit. (result = %d)\n", result);

  result = openapiPoeTrapModeSet(clientHandle, unit, -1);
  printf("NULL status pointer. (result = %d)\n", result);

  printf("openapiPoeTrapModeSet() sanity successful. \n");

  /* openapiPoePseMainCurPowerPrioritySet */
  printf("\nTesting openapiPoePseMainCurPowerPrioritySet(): \n");

  result = openapiPoePseMainCurPowerPrioritySet(NULL, unit, OPEN_POE_PRIORITY_MEDIUM);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseMainCurPowerPrioritySet(clientHandle, 0, OPEN_POE_PRIORITY_MEDIUM);
  printf("Invalid unit. (result = %d)\n", result);

  printf("openapiPoePseMainCurPowerPrioritySet() sanity successful. \n");

  /* openapiPoePseMainCurPowerLimitSet */
  printf("\nTesting openapiPoePseMainCurPowerLimitSet(): \n");

  result = openapiPoePseMainCurPowerLimitSet(NULL, unit, 100);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPoePseMainCurPowerLimitSet(clientHandle, 0, 100);
  printf("Invalid unit. (result = %d)\n", result);

  printf("openapiPoePseMainCurPowerLimitSet() sanity successful. \n");
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1, arg2;
  uint32_t unit = 0;
  
  if (argc < 2)
  {
    printPOEAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("poe_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting PoE API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  /* Determine/use local unit number */
  openapiSystemLocalUnitGet(&clientHandle, &unit);

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poeFirstInterfaceGet(&clientHandle, arg1, unit);
      break;
    case 2:
      if (argc != 2)
      {
        printPOEAppMenu();
        exit(1);
      }
      poeIsSupported(&clientHandle, unit);
      break;
    case 3:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      poeIsValidInterface(&clientHandle, arg1, arg2);
      break;
    case 4:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortPowerClassificationsGet(&clientHandle, unit, arg1);
      break;       
    case 5:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortSensorsGet(&clientHandle, unit, arg1);
      break;       
    case 6:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortDetectionStatusGet(&clientHandle, unit, arg1);
      break;       
    case 7:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortErrorCodeGet(&clientHandle, arg1);
      break;       
    case 8:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortCountersGet(&clientHandle, arg1);
      break;       
    case 9:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poeFindMaxPowerLimit(&clientHandle, unit, arg1);
      break;       
    case 10:
      if (argc != 2)
      {
        printPOEAppMenu();
        exit(1);
      }
      poePseMainDetectionModeGet(&clientHandle, unit);
      break;       
    case 11:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseMainDetectionModeSet(&clientHandle, unit, arg1);
      break;       
    case 12:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortCurPowerLimitGet(&clientHandle, unit, arg1);
      break;       
    case 13:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      poePsePortCurPowerLimitSet(&clientHandle, unit, arg1, arg2);
      break;       
    case 14:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortCurPowerPriorityGet(&clientHandle, unit, arg1);
      break;       
    case 15:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      poePsePortCurPowerPrioritySet(&clientHandle, unit, arg1, arg2);
      break;       
    case 16:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortDetectionModeGet(&clientHandle, unit, arg1);
      break;       
    case 17:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      poePsePortDetectionModeSet(&clientHandle, unit, arg1, arg2);
      break;       
    case 18:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortHighPowerModeGet(&clientHandle, unit, arg1);
      break;       
    case 19:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);

      if (strcmp("enable", argv[3]) == 0) 
      {
        arg2 = 1;
      }
      else if(strcmp("disable", argv[3]) == 0) 
      {
        arg2 = 0;
      }
      else
      {
        printf("\nInvalid high-power mode\n\n");
        printPOEAppMenu();
        exit(1);
      }
      poePsePortHighPowerModeSet(&clientHandle, unit, arg1, arg2);
      break;       
    case 20:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortPowerUpModeGet(&clientHandle, unit, arg1);
      break;       
    case 21:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);

      if (strcmp("af", argv[3]) == 0) 
      {
        arg2 = OPEN_POE_POWER_UP_MODE_8023AF;
      }
      else if (strcmp("high-inrush", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_HIGH_INRUSH;
      }
      else if (strcmp("pre-at", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_PRE_8023AT;
      }
      else if (strcmp("at", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_8023AT;
      }
      else if (strcmp("upoe", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_UPOE;
      }
      else if (strcmp("pre-bt", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_PRE_8023BT;
      }
      else if (strcmp("type3", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_TYPE3;
      }
      else if (strcmp("bt", argv[3]) == 0)
      {
        arg2 = OPEN_POE_POWER_UP_MODE_8023BT;
      }
      else
      {
        printf("\nInvalid power-up mode\n\n");
        printPOEAppMenu();
        exit(1);
      }
      poePsePortPowerUpModeSet(&clientHandle, unit, arg1, arg2);
      break;       
    case 22:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortViolationTypeGet(&clientHandle, unit, arg1);
      break;       
    case 23:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      poePsePortViolationTypeSet(&clientHandle, unit, arg1, arg2);
      break;       
    case 24:
      if (argc != 2)
      {
        printPOEAppMenu();
        exit(1);
      }
      poePseMainUsageThresholdGet(&clientHandle, unit);
      break;       
    case 25:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseMainUsageThresholdSet(&clientHandle, unit, arg1);
      break;       
    case 26:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePsePortAdminEnableGet(&clientHandle, arg1);
      break;
    case 27:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      poePsePortAdminEnableSet(&clientHandle, arg1, arg2);
      break;
    case 28:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseMgmtModeGet(&clientHandle, arg1);
      break;       
    case 29:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      if (0 == strcmp(argv[3], "static"))
      {
        arg2 = OPEN_POE_POWER_MGMT_STATIC;
      }
      else if (0 == strcmp(argv[3], "static-pri"))
      {
        arg2 = OPEN_POE_POWER_MGMT_STATIC_PRI;
      }
      else if (0 == strcmp(argv[3], "dynamic"))
      {
        arg2 = OPEN_POE_POWER_MGMT_DYNAMIC;
      }
      else if (0 == strcmp(argv[3], "dynamic-pri"))
      {
        arg2 = OPEN_POE_POWER_MGMT_DYNAMIC_PRI;
      }
      else if (0 == strcmp(argv[3], "class"))
      {
        arg2 = OPEN_POE_POWER_MGMT_CLASS;
      }
      else
      {
        printPOEAppMenu();
        exit(1);
      }
      poePseMgmtModeSet(&clientHandle, arg1, arg2);
      break;       
    case 30:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseMainConsumptionPowerGet(&clientHandle, arg1);
      break;
    case 31:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poeFWImageVersionGet(&clientHandle, arg1);
      break;
    case 32:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseMainOperStatusGet(&clientHandle, arg1);
      break;
    case 33:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseThresholdPowerGet(&clientHandle, arg1);
      break;
    case 34:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseTotalAvailablePowerGet(&clientHandle, arg1);
      break;
    case 35:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      trapPoeGet(&clientHandle, arg1);
      break;
    case 36:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePortTimerScheduleNameGet(&clientHandle, unit, arg1);
      break;
    case 37:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePortReset(&clientHandle, unit, arg1);
      break;
    case 38:
      if (argc != 2)
      {
        printPOEAppMenu();
        exit(1);
      }
      poeReset(&clientHandle, unit);
      break;
    case 39:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      if (0 == strcmp(argv[2], "enabled"))
      {
        arg1 = OPEN_ENABLE;
      }
      else if (0 == strcmp(argv[2], "disabled"))
      {
        arg1 = OPEN_DISABLE;
      }
      else
      {
        printPOEAppMenu();
        exit(1);
      }
      poePseMainAdminEnableSet(&clientHandle, unit, arg1);
      break;       
    case 40:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      if (0 == strcmp(argv[2], "enabled"))
      {
        arg1 = OPEN_ENABLE;
      }
      else if (0 == strcmp(argv[2], "disabled"))
      {
        arg1 = OPEN_DISABLE;
      }
      else
      {
        printPOEAppMenu();
        exit(1);
      }
      trapPoeSet(&clientHandle, unit, arg1);
      break;       
    case 41:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      if (0 == strcmp(argv[2], "critical"))
      {
        arg1 = OPEN_POE_PRIORITY_CRITICAL;
      }
      else if (0 == strcmp(argv[2], "high"))
      {
        arg1 = OPEN_POE_PRIORITY_HIGH;
      }
      else if (0 == strcmp(argv[2], "medium"))
      {
        arg1 = OPEN_POE_PRIORITY_MEDIUM;
      }
      else if (0 == strcmp(argv[2], "low"))
      {
        arg1 = OPEN_POE_PRIORITY_LOW;
      }
      else
      {
        printPOEAppMenu();
        exit(1);
      }
      poePseMainCurPowerPrioritySet(&clientHandle, unit, arg1);
      break;       
    case 42:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePseMainCurPowerLimitSet(&clientHandle, unit, arg1);
      break;       
    case 43:
      if (argc != 3)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePethPsePortTimerCtrlClear(&clientHandle, arg1);
      break;
    case 44:
      if (argc != 4)
      {
        printPOEAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      poePortTimerScheduleNameSet(&clientHandle, unit, arg1, argv[3]);
      break;	
    case 45:
      if (argc != 2)
      {
        printPOEAppMenu();
        exit(1);
      }
      poeOpENAPIsSanity(&clientHandle);
      break;
    default:
      printPOEAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping PoE API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
