/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  radius_example.c
*
* @purpose   RADIUS OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    12/12/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_radius.h"

#define OPENAPI_TITLE "RADIUS"
#define EXAMPLE_APP_NAME "radius_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

OPEN_IP_ADDRESS_TYPE_t addrTypeParmCheck(char *argv)
{
  if (strcmp(argv, "ip") == 0)
  {
    return OPEN_IP_ADDRESS_IPV4;
  }
  else if (strcmp(argv, "dns") == 0)
  {
    return OPEN_IP_ADDRESS_DNS;
  }

  printf("Invalid keyword for server address type. Valid choices are 'ip' or 'dns'.\n");
  return OPEN_IP_ADDRESS_UNKNOWN;
}

void authServerAdd(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  open_buffdesc serverName;

  if (argc != 5)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  serverName.pstart = argv[4];
  serverName.size = strlen(argv[4]) + 1;

  result = openapiRadiusAuthServerAdd(clientHandle, &serverAddr, addrType, &serverName);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to add server: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server added.\n");
  }
}

void authServerRemove(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  result = openapiRadiusAuthServerRemove(clientHandle, &serverAddr, addrType);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to remove server: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server removed.\n");
  }
}

void authServerPortNumberSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  uint32_t portNumber;

  if (argc != 5)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  errno = 0;
  portNumber = strtol(argv[4], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid port number parameter: (%s)\n", argv[4]);
    return;
  }

  result = openapiRadiusAuthServerPortNumSet(clientHandle, &serverAddr, addrType, portNumber);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set port number: (result = %d)\n",
           result);
  }
  else
  {
    printf("Port number set.\n");
  }
}

void authServerPortNumberGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  uint32_t portNumber;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  result = openapiRadiusAuthServerPortNumGet(clientHandle, &serverAddr, addrType, &portNumber);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get port number: (result = %d)\n",
           result);
  }
  else
  {
    printf("Port number retrieved. portNumber = %d\n", portNumber);
  }
}

void authServerTypeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  OPEN_RADIUS_SERVER_ENTRY_TYPE_t type;

  if (argc != 5)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  if (strcmp(argv[4], "primary") == 0)
  {
    type = OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY;
  }
  else if (strcmp(argv[4], "secondary") == 0)
  {
    type = OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY;
  }
  else
  {
    printf("Invalid keyword for server type. Valid choices are 'primary' or 'secondary'.\n");
    return;
  }

  result = openapiRadiusAuthServerTypeSet(clientHandle, &serverAddr, addrType, type);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set server type: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server type set.\n");
  }
}

void authServerTypeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  OPEN_RADIUS_SERVER_ENTRY_TYPE_t type;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  result = openapiRadiusAuthServerTypeGet(clientHandle, &serverAddr, addrType, &type);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get server type: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server type retrieved: %s\n",
           (type == OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY)?"primary":
           (type == OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY)?"secondary":"UNKNOWN");
  }
}

void authAttribute4Set(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  struct in_addr inAddr;
  open_inet_addr_t ipAddr;
  open_error_t result;

  if (argc != 3)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  if (inet_aton(argv[2], &inAddr) == 0)
  {
    printf("Error parsing IPv4 address parameter. %s", argv[2]);
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* IPv4 addresses stored in open_inet_addr_t are in host byte order */
  ipAddr.addr.ipv4 = ntohl(inAddr.s_addr);

  result = openapiRadiusAuthServerAttribute4Set(clientHandle, OPEN_TRUE, &ipAddr);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set attribute 4: (result = %d)\n",
           result);
  }
  else
  {
    printf("Attribute 4 set.\n");
  }
}

void authAttribute4Clear(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_inet_addr_t ipAddr;
  open_error_t result;

  if (argc != 2)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;

  result = openapiRadiusAuthServerAttribute4Set(clientHandle, OPEN_FALSE, &ipAddr);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to clear attribute 4: (result = %d)\n",
           result);
  }
  else
  {
    printf("Attribute 4 cleared.\n");
  }
}

void authAttribute4Get(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  OPEN_BOOL_t mode;
  open_inet_addr_t ipAddr;
  open_error_t result;
  struct in_addr inAddr;

  if (argc != 2)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));

  result = openapiRadiusAuthServerAttribute4Get(clientHandle, &mode, &ipAddr);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get attribute 4: (result = %d)\n",
           result);
  }
  else
  {
    if (mode == OPEN_FALSE)
    {
      printf("Attribute 4 not set.\n");
    }
    else
    {
      /* IPv4 addresses stored in open_inet_addr_t are in host byte order */
      inAddr.s_addr = htonl(ipAddr.addr.ipv4);
      printf("Attribute 4: %s.\n", inet_ntoa(inAddr));
    }
  }
}

void authServerSecretEncSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  open_buffdesc encryptedSecret;
  uint32_t expectedEncryptedSecretSize;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  encryptedSecret.pstart = argv[3];
  encryptedSecret.size = strlen(argv[3]) + 1;

  expectedEncryptedSecretSize = openapiRadiusEncryptedSharedSecretSizeGet(clientHandle);
  if (encryptedSecret.size != expectedEncryptedSecretSize)
  {
    printf("WARNING: Encrypted secret string is required to be %d hexadecimal characters. String with %d characters provided.\n",
           (expectedEncryptedSecretSize - 1), (encryptedSecret.size - 1));
    printf("         Proceeding to invoke OpEN call with provided data.\n");
  }

  result = openapiRadiusAuthServerSharedSecretEncryptedSet(clientHandle, &serverAddr, &encryptedSecret);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set secret: (result = %d)\n",
           result);
  }
  else
  {
    printf("Secret set.\n");
  }
}

functionTableEntry_t functionTable[] =
{
  {authServerAdd,               "Add a "OPENAPI_TITLE" authentication server", "<ipaddr|hostname> ip|dns <serverName>"},
  {authServerRemove,            "Remove a "OPENAPI_TITLE" authentication server", "<ipaddr|hostname> ip|dns"},
  {authServerPortNumberSet,     "Set a "OPENAPI_TITLE" authentication server's port number", "<ipaddr|hostname> ip|dns <port>"},
  {authServerPortNumberGet,     "Get a "OPENAPI_TITLE" authentication server's port number", "<ipaddr|hostname> ip|dns"},
  {authServerTypeSet,           "Set a "OPENAPI_TITLE" authentication server's type", "<ipaddr|hostname> ip|dns primary|secondary"},
  {authServerTypeGet,           "Get a "OPENAPI_TITLE" authentication server's type", "<ipaddr|hostname> ip|dns"},
  {authAttribute4Set,           "Set the "OPENAPI_TITLE" attribute 4", "<ipAddr>"},
  {authAttribute4Clear,         "Clear the "OPENAPI_TITLE" attribute 4", ""},
  {authAttribute4Get,           "Get the "OPENAPI_TITLE" attribute 4", ""},
  {authServerSecretEncSet,      "Set the "OPENAPI_TITLE" shared secret specified in encrypted format (128 hexadecimal digits)", "<ipaddr|hostname> <secret>"},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  printf("Usage:  In below, <serverEntryType> should be set to 1 for OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY and 2 for OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY.\n");
  printf("Usage:  In below, <serverType> should be set to 1 for OPEN_RADIUS_SERVER_TYPE_AUTH and 2 for OPEN_RADIUS_SERVER_TYPE_ACCT.\n");
  printf("Usage:  In below, <serverAddrType> should be set to 1 for OPEN_IP_ADDRESS_IPV4, 2 for OPEN_IP_ADDRESS_IPV6 and 3 for OPEN_IP_ADDRESS_DNS.\n");
  printf("Usage:  In below, <serverConfig> should be set to 0 for OPEN_RADIUSGLOBAL and 1 for OPEN_RADIUSLOCAL.\n");
  printf("Usage:  In below, <maxRetransmit> should be set in the range of 1 - 15.\n");
  printf("Usage:  In below, <timeOut> should be set in the range of 1 - 30.\n");
  printf("Usage:  In below, <mode> should be set to 0 for OPEN_DISABLE and 1 for OPEN_ENABLE.\n");
  printf("Usage:  In below, <keyType> should be set to 0 for OPEN_FALSE and 1 for OPEN_TRUE.\n");
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\tTest 10: Run API sanity checks: \n"
         "\t%s 10 \n", EXAMPLE_APP_NAME);
  printf("\tTest 11: Gets the number of configured servers of specified type (Auth or Acct): \n"
         "\t%s 11 <serverType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 12: Gets the maximum number of retransmissions parameter: \n"
         "\t%s 12 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 13: Sets the maximum number of retransmissions parameter: \n"
         "\t%s 13 <pHostAddr><maxRetransmit><serverConfig>\n", EXAMPLE_APP_NAME);
  printf("\tTest 14: Gets the value of time-out duration parameter: \n"
         "\t%s 14 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 15: Sets the time-out duration parameter: \n"
         "\t%s 15 <pHostAddr><timeOut><serverConfig>\n", EXAMPLE_APP_NAME);
  printf("\tTest 16: Gets radius-accounting mode parameter: \n"
         "\t%s 16 \n", EXAMPLE_APP_NAME);
  printf("\tTest 17: Sets radius-accounting mode parameter: \n"
         "\t%s 17 <mode>\n", EXAMPLE_APP_NAME);
  printf("\tTest 18: Gets first configured server: \n"
         "\t%s 18 \n", EXAMPLE_APP_NAME);
  printf("\tTest 19: Get the configured server next in the list after the specified server: \n"
         "\t%s 19 <pCurServerAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 20: Get the server Name of specified radius server: \n"
         "\t%s 20 <pServerAddr><serverAddrType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 21: Gets global selected radius server address and address type: \n"
         "\t%s 21\n", EXAMPLE_APP_NAME);
  printf("\tTest 22: Sets server name attribute to the Auth server: \n"
         "\t%s 22 <pServerAddr><serverAddrType><pServerName>\n", EXAMPLE_APP_NAME);
  printf("\tTest 23: Set the radius authentication server type: \n"
         "\t%s 23 <pServerAddr><serverAddrType><serverEntryType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 24: Gets server Name of specified radius server: \n"
         "\t%s 24 <pServerName>\n", EXAMPLE_APP_NAME);
  printf("\tTest 25: Get the round trip time: \n"
         "\t%s 25 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 26: Retrieve the number of RADIUS access-request packets sent to requested server: \n"
         "\t%s 26 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 27: Retrieve the number of RADIUS access-request packets retransmitted to requested server: \n"
         "\t%s 27 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 28: Retrieve the number of RADIUS access-request packets received from requested server: \n"
         "\t%s 28 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 29: Retrieve the number of RADIUS access-reject packets received from requested server: \n"
         "\t%s 29 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 30: Retrieve the number of RADIUS access-challenge packets received from requested server: \n"
         "\t%s 30 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 31: Retrieve the number of malformed RADIUS access-response packets received from requested server: \n"
         "\t%s 31 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 32: Retrieve the number of RADIUS access-response bad auth packets received from requested server: \n"
         "\t%s 32 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 33: Retrieve the number of RADIUS access-request packets destined for this server that have not\n"
         "\t\tyet timed out or received a response from this server received from requested server: \n"
          "\t%s 33 <pHostAddr><pPacketCount>\n", EXAMPLE_APP_NAME);
  printf("\tTest 34: Retrieve the  number of authentication time-outs to requested server: \n"
         "\t%s 34 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 35: Retrieve the  number of RADIUS packets of unknown type received from requested server on the\n"
         "\t\t authentication port: \n"
         "\t%s 35 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 36: Retrieve the number of RADIUS packets received from server on the authentication port that\n"
         "\t\twere dropped for some other reason: \n"
         "\t%s 36 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 37: Gets server details of specified accounting server EXAMPLE_APP_NAME: \n"
         "\t%s 37 <pServerName>\n", EXAMPLE_APP_NAME);
  printf("\tTest 38: Verify a RADIUS accounting server is a configured server: \n"
         "\t%s 38 <pServerAddr><serverAddrType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 39: Gets server EXAMPLE_APP_NAME of specified accounting server: \n"
         "\t%s 39 <pServerAddr><serverAddrType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 40: Get the round trip time: \n"
         "\t%s 40 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 41: Retrieve the number of RADIUS accounting-request packets sent to requested server: \n"
         "\t%s 41 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 42: Retrieve the number of RADIUS accounting-request packets retransmitted to requested server: \n"
         "\t%s 42 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 43: Retrieve the number of RADIUS accounting-response packets received from requested server: \n"
         "\t%s 43 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 44: Retrieve the number of malformed RADIUS accounting-response packets received from requested server: \n"
         "\t%s 44 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 45: Retrieve the number of RADIUS accounting-response bad auth packets received from requested server: \n"
         "\t%s 45 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 46: Retrieve the number of RADIUS accounting-request packets destined for this server that have not yet\n"
         "\t\ttimed out or received a response from this server received from requested server: \n"
         "\t%s 46 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 47: Retrieve the  number of accounting time-outs to requested server: \n"
         "\t%s 47 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 48: Retrieve the  number of RADIUS packets of unknown type received from requested server on the\n"
         "\t\taccounting port: \n"
         "\t%s 48 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 49: Retrieve the number of RADIUS packets received from server on the accounting port that were dropped\n"
         "\t\tfor some other reason: \n"
         "\t%s 49 <pHostAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 50: Gets first accounting configured server: \n"
         "\t%s 50 \n", EXAMPLE_APP_NAME);
  printf("\tTest 51: Get the configured accounting server next in the list after the specified server: \n"
         "\t%s 51 <pCurServerAddr>\n", EXAMPLE_APP_NAME);
  printf("\tTest 52: Gets global selected Radius server address, server type and address type: \n"
         "\t%s 52 <servType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 53: Get the port number of the accounting server being used: \n"
         "\t%s 53 <pServerAddr><serverAddrType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 54: Sets the port number of the accounting server being used: \n"
         "\t%s 54 <pServerAddr><serverAddrType><port>\n", EXAMPLE_APP_NAME);
  printf("\tTest 55: Verify a RADIUS authentication server is a configured server: \n"
         "\t%s 55 <pServerAddr><serverAddrType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 56: Set the shared secret being used between the radius client and the radius server: \n"
         "\t%s 56 <pServerAddr><serverConfig><keyType>\n", EXAMPLE_APP_NAME);
  printf("\tTest 57: Set the shared secret being used between the radius client and the accounting server: \n"
         "\t%s 57 <pServerAddr><serverAddrType><pSharedSecret>\n", EXAMPLE_APP_NAME);

  printf("\n");

  return;
}
/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t roundTripTime;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  uint32_t maxRetransmit;
  uint32_t timeout;
  uint32_t port;
  uint32_t count;
  uint32_t packetCount;
  OPEN_RADIUS_SERVER_TYPE_t servType=1;
  OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus;
  OPEN_CONTROL_t mode;
  char buffer[20];
  char strServAddr[100];
  open_buffdesc serAddr;
  char strServName[30];
  open_buffdesc serName;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  char strSharedSecret[30];
  open_buffdesc sharedSecret;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;
  serAddr.pstart = strServAddr;
  serAddr.size = sizeof(strServAddr);
  serName.pstart = strServName;
  serName.size = sizeof(strServName);
  sharedSecret.pstart = strSharedSecret;
  sharedSecret.size = sizeof(strSharedSecret);

  printf("Testing radius OpEN APIs sanity:\n");

  printf("Testing openapiRadiusConfiguredServersCountGet():\n");
  result = openapiRadiusConfiguredServersCountGet(NULL, servType, &count);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusConfiguredServersCountGet(clientHandle, servType, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerRetransGet():\n");
  result = openapiRadiusServerRetransGet(NULL, &serAddr, &maxRetransmit, &paramStatus);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerRetransGet(clientHandle, NULL, &maxRetransmit, &paramStatus);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerRetransGet(clientHandle, &badBufdescPointer, &maxRetransmit, &paramStatus);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerRetransGet(clientHandle, &zeroLenBuf, &maxRetransmit, &paramStatus);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerRetransGet(clientHandle, &serAddr, NULL, &paramStatus);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerRetransGet(clientHandle, &serAddr, &maxRetransmit, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusServerTimeOutGet():\n");
  result = openapiRadiusServerTimeOutGet(NULL, &serAddr, &timeout, &paramStatus);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerTimeOutGet(clientHandle, NULL, &timeout, &paramStatus);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerTimeOutGet(clientHandle, &badBufdescPointer, &timeout, &paramStatus);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerTimeOutGet(clientHandle, &zeroLenBuf, &timeout, &paramStatus);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerTimeOutGet(clientHandle, &serAddr, NULL, &paramStatus);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerTimeOutGet(clientHandle, &serAddr, &timeout, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusAccountingModeGet():\n");
  result = openapiRadiusAccountingModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAccountingModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiRadiusServerFirstIPHostNameGet():\n");
  result = openapiRadiusServerFirstIPHostNameGet(NULL, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerFirstIPHostNameGet(clientHandle, NULL, &addrType);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerFirstIPHostNameGet(clientHandle, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerFirstIPHostNameGet(clientHandle, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerFirstIPHostNameGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerNextIPHostNameGet():\n");
  result = openapiRadiusServerNextIPHostNameGet(NULL, &serAddr, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, NULL, &serAddr, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, &badBufdescPointer, &serAddr, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, &zeroLenBuf, &serAddr, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, &serAddr, NULL, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, &serAddr, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, &serAddr, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNextIPHostNameGet(clientHandle, &serAddr, &serAddr, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusServerHostNameGet():\n");
  result = openapiRadiusServerHostNameGet(NULL, &serAddr, addrType, &serName);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerHostNameGet(clientHandle, NULL, addrType, &serName);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerHostNameGet(clientHandle, &badBufdescPointer, addrType, &serName);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerHostNameGet(clientHandle, &zeroLenBuf, addrType, &serName);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerHostNameGet(clientHandle, &serAddr, addrType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerHostNameGet(clientHandle, &serAddr, addrType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerHostNameGet(clientHandle, &serAddr, addrType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusServerGlobalSelectedGet():\n");
  result = openapiRadiusServerGlobalSelectedGet(NULL, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerGlobalSelectedGet(clientHandle, NULL, &addrType);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerGlobalSelectedGet(clientHandle, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerGlobalSelectedGet(clientHandle, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerGlobalSelectedGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerNameSet():\n");
  result = openapiRadiusServerNameSet(NULL, &serAddr, addrType, &serName);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerNameSet(clientHandle, NULL, addrType, &serName);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNameSet(clientHandle, &badBufdescPointer, addrType, &serName);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNameSet(clientHandle, &zeroLenBuf, addrType, &serName);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNameSet(clientHandle, &serAddr, addrType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNameSet(clientHandle, &serAddr, addrType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerNameSet(clientHandle, &serAddr, addrType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusNamedServerAddrGet():\n");
  result = openapiRadiusNamedServerAddrGet(NULL, &serName, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, NULL, &serAddr, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, &badBufdescPointer, &serAddr, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, &zeroLenBuf, &serAddr, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, &serName, NULL, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, &serName, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, &serName, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedServerAddrGet(clientHandle, &serName, &serAddr, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatRTTGet():\n");
  result = openapiRadiusServerStatRTTGet(NULL, &serAddr, &roundTripTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatRTTGet(clientHandle, NULL, &roundTripTime);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatRTTGet(clientHandle, &badBufdescPointer, &roundTripTime);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatRTTGet(clientHandle, &zeroLenBuf, &roundTripTime);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatRTTGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatAccessReqGet():\n");
  result = openapiRadiusServerStatAccessReqGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessReqGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessReqGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessReqGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessReqGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatAccessRetransGet():\n");
  result = openapiRadiusServerStatAccessRetransGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRetransGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRetransGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRetransGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRetransGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatAccessAcceptGet():\n");
  result = openapiRadiusServerStatAccessAcceptGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessAcceptGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessAcceptGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessAcceptGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessAcceptGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatAccessRejectGet():\n");
  result = openapiRadiusServerStatAccessRejectGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRejectGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRejectGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRejectGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessRejectGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatAccessChallengeGet():\n");
  result = openapiRadiusServerStatAccessChallengeGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessChallengeGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessChallengeGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessChallengeGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatAccessChallengeGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatMalformedAccessResponseGet():\n");
  result = openapiRadiusServerStatMalformedAccessResponseGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatMalformedAccessResponseGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatMalformedAccessResponseGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatMalformedAccessResponseGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatMalformedAccessResponseGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatBadAuthGet():\n");
  result = openapiRadiusServerStatBadAuthGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatBadAuthGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatBadAuthGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatBadAuthGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatBadAuthGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatPendingReqGet():\n");
  result = openapiRadiusServerStatPendingReqGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatPendingReqGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatPendingReqGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatPendingReqGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatPendingReqGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatTimeoutsGet():\n");
  result = openapiRadiusServerStatTimeoutsGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatTimeoutsGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatTimeoutsGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatTimeoutsGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatTimeoutsGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatUnknownTypeGet():\n");
  result = openapiRadiusServerStatUnknownTypeGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatUnknownTypeGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatUnknownTypeGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatUnknownTypeGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatUnknownTypeGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerStatPktsDroppedGet():\n");
  result = openapiRadiusServerStatPktsDroppedGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerStatPktsDroppedGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatPktsDroppedGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatPktsDroppedGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusServerStatPktsDroppedGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusNamedAcctServerAddrGet():\n");
  result = openapiRadiusNamedAcctServerAddrGet(NULL, &serName, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, NULL, &serAddr, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, &badBufdescPointer, &serAddr, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, &zeroLenBuf, &serAddr, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, &serName, NULL, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, &serName, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, &serName, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusNamedAcctServerAddrGet(clientHandle, &serName, &serAddr, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusServerAcctHostNameGet():\n");
  result = openapiRadiusServerAcctHostNameGet(NULL, &serAddr, addrType, &serName);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerAcctHostNameGet(clientHandle, NULL, addrType, &serName);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerAcctHostNameGet(clientHandle, &badBufdescPointer, addrType, &serName);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerAcctHostNameGet(clientHandle, &zeroLenBuf, addrType, &serName);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerAcctHostNameGet(clientHandle, &serAddr, addrType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerAcctHostNameGet(clientHandle, &serAddr, addrType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerAcctHostNameGet(clientHandle, &serAddr, addrType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatRTTGet():\n");
  result = openapiRadiusAcctServerStatRTTGet(NULL, &serAddr, &roundTripTime);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRTTGet(clientHandle, NULL, &roundTripTime);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRTTGet(clientHandle, &badBufdescPointer, &roundTripTime);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRTTGet(clientHandle, &zeroLenBuf, &roundTripTime);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRTTGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatReqGet():\n");
  result = openapiRadiusAcctServerStatReqGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatReqGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatReqGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatReqGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatReqGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatRetransGet():\n");
  result = openapiRadiusAcctServerStatRetransGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRetransGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRetransGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRetransGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatRetransGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatResponseGet():\n");
  result = openapiRadiusAcctServerStatResponseGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatResponseGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatResponseGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatResponseGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatResponseGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatMalformedResponseGet():\n");
  result = openapiRadiusAcctServerStatMalformedResponseGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatMalformedResponseGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatMalformedResponseGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatMalformedResponseGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatMalformedResponseGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatBadAuthGet():\n");
  result = openapiRadiusAcctServerStatBadAuthGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatBadAuthGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatBadAuthGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatBadAuthGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatBadAuthGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatPendingReqGet():\n");
  result = openapiRadiusAcctServerStatPendingReqGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPendingReqGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPendingReqGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPendingReqGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPendingReqGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatTimeoutsGet():\n");
  result = openapiRadiusAcctServerStatTimeoutsGet(NULL, &serAddr, &timeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatTimeoutsGet(clientHandle, NULL, &timeout);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatTimeoutsGet(clientHandle, &badBufdescPointer, &timeout);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatTimeoutsGet(clientHandle, &zeroLenBuf, &timeout);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatTimeoutsGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatUnknownTypeGet():\n");
  result = openapiRadiusAcctServerStatUnknownTypeGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatUnknownTypeGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatUnknownTypeGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatUnknownTypeGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatUnknownTypeGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerStatPktsDroppedGet():\n");
  result = openapiRadiusAcctServerStatPktsDroppedGet(NULL, &serAddr, &packetCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPktsDroppedGet(clientHandle, NULL, &packetCount);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPktsDroppedGet(clientHandle, &badBufdescPointer, &packetCount);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPktsDroppedGet(clientHandle, &zeroLenBuf, &packetCount);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerStatPktsDroppedGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerFirstIPHostNameGet():\n");
  result = openapiRadiusAcctServerFirstIPHostNameGet(NULL, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerFirstIPHostNameGet(clientHandle, NULL, &addrType);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerFirstIPHostNameGet(clientHandle, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerFirstIPHostNameGet(clientHandle, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiRadiusAcctServerFirstIPHostNameGet(clientHandle, &serAddr, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiRadiusServerGlobalCurentGet():\n");
  result = openapiRadiusServerGlobalCurentGet(NULL, servType, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusServerGlobalCurentGet(clientHandle, servType, NULL, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusServerGlobalCurentGet(clientHandle, servType, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusServerGlobalCurentGet(clientHandle, servType, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusServerGlobalCurentGet(clientHandle, servType, &serAddr, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusAcctServerNextIPHostNameGet():\n");
  result = openapiRadiusAcctServerNextIPHostNameGet(NULL, &serAddr, &serAddr, &addrType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, NULL, &serAddr, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, &badBufdescPointer, &serAddr, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, &zeroLenBuf, &serAddr, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, &serAddr, NULL, &addrType);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, &serAddr, &badBufdescPointer, &addrType);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, &serAddr, &zeroLenBuf, &addrType);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusAcctServerNextIPHostNameGet(clientHandle, &serAddr, &serAddr, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusAccountingHostNameServerPortNumGet():\n");
  result = openapiRadiusAccountingHostNameServerPortNumGet(NULL, &serAddr, addrType, &port);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerPortNumGet(clientHandle, NULL, addrType, &port);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerPortNumGet(clientHandle, &badBufdescPointer, addrType, &port);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerPortNumGet(clientHandle, &zeroLenBuf, addrType, &port);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerPortNumGet(clientHandle, &serAddr, addrType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiRadiusAccountingHostNameServerSharedSecretSet():\n");
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(NULL, &serAddr, addrType, &sharedSecret);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(clientHandle, NULL, addrType, &sharedSecret);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(clientHandle, &badBufdescPointer, addrType, &sharedSecret);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(clientHandle, &zeroLenBuf, addrType, &sharedSecret);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(clientHandle, &serAddr, addrType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(clientHandle, &serAddr, addrType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiRadiusAccountingHostNameServerSharedSecretSet(clientHandle, &serAddr, addrType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Gets the number of configured servers of specified type (Auth or Acct).
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    type            @b{(input)}  Radius server type.
* @param    pCount          @b{(output)}  Server count.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusConfiguredServersCountGet(openapiClientHandle_t *pClientHandle,
                                OPEN_RADIUS_SERVER_TYPE_t type,
                                uint32_t *pCount)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiRadiusConfiguredServersCountGet(pClientHandle,
                                                       type, pCount)))
  {
    printf("Bad return code trying to gets the number of configured servers"
        " of specified type (Auth or Acct). (result = %d)\n", result);
  }
  else
  {
    printf("Number of configured %s servers are %d\n",
        (OPEN_RADIUS_SERVER_TYPE_AUTH == type) ? "Auth" :
        (OPEN_RADIUS_SERVER_TYPE_ACCT == type) ? "Acct" : "invalid type" ,
         *pCount);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the maximum number of retransmissions parameter.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Server address(ALL_RADIUS_SERVERS => global).
* @param    pMaxRetransmit  @b{(output)}  Max retransmit value.
* @param    pParamStatus    @b{(output)}  Parameter for a specific server is global or local.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerRetransGet(openapiClientHandle_t *pClientHandle,
                       open_buffdesc *pHostAddr, 
                       uint32_t *pMaxRetransmit, 
                       OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t *pParamStatus)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiRadiusServerRetransGet(pClientHandle, pHostAddr, 
                                              pMaxRetransmit, pParamStatus)))
  {
    printf("Bad return code trying to gets the maximum number of"
        " retransmissions parameter. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum retransmissions information of server:%s is maxRetransmit:%d"
        " and radius server config is %d\n", (char*)pHostAddr->pstart, *pMaxRetransmit,
        *pParamStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the maximum number of retransmissions parameter.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Server address(ALL_RADIUS_SERVERS => global).
* @param    maxRetransmit   @b{(input)}  Max retransmit value.
* @param    paramStatus     @b{(input)}  Parameter for a specific server is global or local.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerRetransSet(openapiClientHandle_t *pClientHandle,
                       open_buffdesc *pHostAddr,
                       uint32_t maxRetransmit,
                       OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiRadiusServerRetransSet(pClientHandle,
                                              pHostAddr, maxRetransmit,
                                              paramStatus)))
  {
    printf("Bad return code trying to sets the maximum number of"
        " retransmissions parameter. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of retransmissions is set to %d and server config "
        "is set to %s for server:%s\n", maxRetransmit,
        (OPEN_RADIUSGLOBAL == paramStatus) ? "GLOBAL" :
        (OPEN_RADIUSLOCAL == paramStatus) ? "LOCAL" : "Invalid" ,
         (char *)pHostAddr->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the value of time-out duration parameter.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Server address(ALL_RADIUS_SERVERS => global).
* @param    pTimeOut        @b{(output)}  Timeout value.
* @param    pParamStatus     @b{(output)}  Parameter for a specific server is global or local.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerTimeOutGet(openapiClientHandle_t *pClientHandle,
                       open_buffdesc *pHostAddr,
                       uint32_t *pTimeOut,
                       OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t *pParamStatus)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerTimeOutGet(pClientHandle, pHostAddr,
                                              pTimeOut, pParamStatus)))
  {
    printf("Bad return code trying to gets the value of time-out duration"
        " parameter. (result = %d)\n", result);
  }
  else
  {
    printf("server:%s timeout value is %d and server config is %s\n",
        (char *)pHostAddr->pstart, *pTimeOut,
        (OPEN_RADIUSGLOBAL == *pParamStatus) ? "GLOBAL" :
        (OPEN_RADIUSLOCAL == *pParamStatus) ? "LOCAL" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Sets the time-out duration parameter.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Server address (ALL_RADIUS_SERVERS => global).
* @param    timeOut         @b{(input)}  TimeOut duration.
* @param    paramStatus     @b{(input)}  Parameter for a specific server is global or local.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerTimeOutSet(openapiClientHandle_t *pClientHandle,
                       open_buffdesc *pHostAddr,
                       uint32_t timeOut,
                       OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerTimeOutSet(pClientHandle, pHostAddr,
                                              timeOut, paramStatus)))
  {
    printf("Bad return code trying to sets the time-out duration parameter."
        " (result = %d)\n", result);
  }
  else
  {
    printf("server:%s timeout value is set to %d and server config is set to %s\n",
        (char *)pHostAddr->pstart, timeOut,
        (OPEN_RADIUSGLOBAL == paramStatus) ? "GLOBAL" :
        (OPEN_RADIUSLOCAL == paramStatus) ? "LOCAL" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Gets radius-accounting mode parameter.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pAccMode       @b{(output)}  Accounting mode.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAccountingModeGet(openapiClientHandle_t *pClientHandle,
                        OPEN_CONTROL_t *pAccMode)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAccountingModeGet(pClientHandle, pAccMode)))
  {
    printf("Bad return code trying to gets radius-accounting mode parameter."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Radius accounting is in %s mode\n",
        (OPEN_ENABLE == *pAccMode) ? "enabled" : "disabled");

  }
  return;
}


/*********************************************************************
* @purpose  Sets radius-accounting mode parameter.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    accntMode       @b{(input)}  Accounting mode.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAccountingModeSet(openapiClientHandle_t *pClientHandle,
                        OPEN_CONTROL_t accntMode)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiRadiusAccountingModeSet(pClientHandle, accntMode)))
  {
    printf("Bad return code trying to sets radius-accounting mode parameter."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Radius accounting mode is successfully %s\n",
        (OPEN_ENABLE == accntMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets first configured server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerAddr    @b{(output)}  First configured radius server.
* @param    pType          @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerFirstIPHostNameGet(openapiClientHandle_t *pClientHandle,
                               open_buffdesc *pServerAddr,
                               OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerFirstIPHostNameGet(pClientHandle, 
                                                      pServerAddr, pType)))
  {
    printf("Bad return code trying to gets first configured server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Details of first configured server details:- server:%s addrType:%s\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Get the configured server next in the list after the specified server.
*
* @param    pClientHandle    @b{(input)}  Client handle from registration API.
* @param    pCurServerAddr   @b{(input)}  Current server information.
* @param    pNextServerAddr  @b{(output)}  Next configured server information.
* @param    pType            @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerNextIPHostNameGet(openapiClientHandle_t *pClientHandle,
                              open_buffdesc *pCurServerAddr,
                              open_buffdesc *pNextServerAddr,
                              OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerNextIPHostNameGet(pClientHandle, pCurServerAddr,
                                                     pNextServerAddr, pType)))
  {
    printf("Bad return code trying to get the configured server next in the"
        " list after the specified server. (result = %d)\n", result);
  }
  else
  {
    printf("Details of  next configured server details after given server:%s :-"
        " server:%s addrType:%s\n", (char *)pCurServerAddr->pstart,
        (char *)pNextServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Get the server Name of specified radius server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pServerAddr     @b{(input)}  Host name or IP address server.
* @param    type            @b{(input)}  Address type[DNS/IP address].
* @param    pServerName     @b{(output)}  Name of the server.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerHostNameGet(openapiClientHandle_t *pClientHandle,
                        open_buffdesc *pServerAddr,
                        OPEN_IP_ADDRESS_TYPE_t type,
                        open_buffdesc *pServerName)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerHostNameGet(pClientHandle, pServerAddr,
                                               type, pServerName)))
  {
    printf("Bad return code trying to get the server Name of specified radius"
        " server. (result = %d)\n", result);
  }
  else
  {
    printf("Server name of server:%s addrType:%s is %s\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid",
        (char *)pServerName->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Gets global selected radius server address and address type.
*
* @param    pClientHandle     @b{(input)}  Client handle from registration API.
* @param    pServerAddr       @b{(output)}  Radius server address.
* @param    pServerAddrType   @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerGlobalSelectedGet(openapiClientHandle_t *pClientHandle,
                              open_buffdesc *pServerAddr,
                              OPEN_IP_ADDRESS_TYPE_t *pServerAddrType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerGlobalSelectedGet(pClientHandle,
                                                     pServerAddr, pServerAddrType)))
  {
    printf("Bad return code trying to gets global selected radius server address"
        " and address type. (result = %d)\n", result);
  }
  else
  {
    printf("Global selected details server:%s addrType:%s \n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pServerAddrType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pServerAddrType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pServerAddrType) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Sets server name attribute to the Auth server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API
* @param    pServerAddr     @b{(input)}  Server host name or IP address.
* @param    type            @b{(input)}  Address type[DNS/IP address].
* @param    pServerName     @b{(input)}  Name of the server.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerNameSet(openapiClientHandle_t *pClientHandle,
                    open_buffdesc *pServerAddr,
                    OPEN_IP_ADDRESS_TYPE_t type,
                    open_buffdesc *pServerName)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerNameSet(pClientHandle, pServerAddr,
                                           type, pServerName)))
  {
    printf("Bad return code trying to sets server name attribute to the Auth"
        " server. (result = %d)\n", result);
  }
  else
  {
    printf("Server name of server:%s addrType:%s is successfully set to %s\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid",
        (char *)pServerName->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Set the radius authentication server type.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pServerAddr     @b{(input)}  Server host name or IP address.
* @param    addrType        @b{(input)}  Address type[DNS/IP address].
* @param    type            @b{(input)}  Server type.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAuthServerTypeSet(openapiClientHandle_t *pClientHandle,
                        open_buffdesc *pServerAddr,
                        OPEN_IP_ADDRESS_TYPE_t addrType,
                        uint32_t type)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAuthServerTypeSet(pClientHandle, pServerAddr,
                                               addrType, type)))
  {
    printf("Bad return code trying to set the radius authentication server type."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Server type of server:%s addrType:%s is successfully set to %s\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == addrType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == addrType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == addrType) ? "DNS" : "Invalid",
        (OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY == type) ? "Primary" :
        (OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY == type) ? "Secondary" : "Invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Gets server Name of specified radius server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerName    @b{(input)}  Name of the server.
* @param    pServerAddr    @b{(output)}  Radius server address[Host name/IP address].
* @param    pType          @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusNamedServerAddrGet(openapiClientHandle_t *pClientHandle,
                         open_buffdesc *pServerName,
                         open_buffdesc *pServerAddr,
                         OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusNamedServerAddrGet(pClientHandle, pServerName,
                                                pServerAddr, pType)))
  {
    printf("Bad return code trying to gets server Name of specified"
        " radius server. (result = %d)\n", result);
  }
  else
  { 
    printf("Server address details with serverName:%s is server:%s addrType:%s\n",
        (char *)pServerName->pstart, (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "Invalid");
  }
  return;
}

/*********************************************************************
* @purpose  Get the round trip time.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pRoundTripTime  @b{(output)}  Round trip time.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatRTTGet(openapiClientHandle_t *pClientHandle,
                       open_buffdesc *pHostAddr,
                       uint32_t *pRoundTripTime)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatRTTGet(pClientHandle, pHostAddr,
                                              pRoundTripTime)))
  {
    printf("Bad return code trying to get the round trip time."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Round trip time to radius authentication server:%s is %d \n",
        (char *)pHostAddr->pstart, *pRoundTripTime);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-request packets
*           sent to requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatAccessReqGet(openapiClientHandle_t *pClientHandle,
                             open_buffdesc *pHostAddr,
                             uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatAccessReqGet(pClientHandle,
                                                    pHostAddr,
                                                    pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        "access-request packets sent to requested server. (result = %d)\n",
        result);
  }
  else
  {
    printf("Access-request packet count sent to radius authentication"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-request packets
*           retransmitted to requested server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pHostAddr      @b{(input)}  Host Address of the radius server.
* @param    pPacketCount   @b{(output)}  Number of packets

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatAccessRetransGet(openapiClientHandle_t *pClientHandle,
                                 open_buffdesc *pHostAddr,
                                 uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiRadiusServerStatAccessRetransGet(pClientHandle,
                                                        pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " access-request packets retransmitted to requested server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Access-request packet count retransmitted to radius authentication"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-request packets received
*           from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatAccessAcceptGet(openapiClientHandle_t *pClientHandle,
                                open_buffdesc *pHostAddr,
                                uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatAccessAcceptGet(pClientHandle,
                                                       pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " access-request packets received from requested server. (result = %d)\n",
        result);
  }
  else
  {
    printf("Access-request packet count received from radius authentication"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-reject packets received
*           from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatAccessRejectGet(openapiClientHandle_t *pClientHandle,
                                open_buffdesc *pHostAddr,
                                uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatAccessRejectGet(pClientHandle,
                                                       pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " access-reject packets received from requested server. (result = %d)\n",
        result);
  }
  else
  {
    printf("Access-reject packet count received from radius authentication"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-challenge packets
*           received from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)} Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatAccessChallengeGet(openapiClientHandle_t *pClientHandle,
                                   open_buffdesc *pHostAddr,
                                   uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatAccessChallengeGet(pClientHandle,
                                                          pHostAddr,
                                                          pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " access-challenge packets received from requested server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Access-challenge packet count received from radius authentication"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of malformed RADIUS access-response
*           packets received from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatMalformedAccessResponseGet(openapiClientHandle_t *pClientHandle,
                                           open_buffdesc *pHostAddr,
                                           uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatMalformedAccessResponseGet(pClientHandle,
                                                                  pHostAddr,
                                                                  pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of malformed RADIUS"
        " access-response packets received from requested server. (result = %d)\n",
        result);
  }
  else
  {
    printf("Malformed access-response packet count received from radius"
        " authentication server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-response bad auth packets
*           received from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatBadAuthGet(openapiClientHandle_t *pClientHandle,
                           open_buffdesc *pHostAddr,
                           uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatBadAuthGet(pClientHandle,
                                                  pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " access-response bad auth packets received from requested server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Access-response bad auth packet count received from radius"
        " authentication server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS access-request packets destined
*           for requested server that have not yet timed out or received a response
*           from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatPendingReqGet(openapiClientHandle_t *pClientHandle,
                              open_buffdesc *pHostAddr,
                              uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatPendingReqGet(pClientHandle, pHostAddr,
                                                     pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " access-request packets destined for this server that have not yet"
        " timed out or received a response from requested server. (result = %d)\n", result);
  }
  else
  {
    printf("Non timed out access-request packet count received from radius"
        " authentication server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the  number of authentication time-outs to requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatTimeoutsGet(openapiClientHandle_t *pClientHandle,
                            open_buffdesc *pHostAddr,
                            uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatTimeoutsGet(pClientHandle,
                                                   pHostAddr,
                                                   pPacketCount)))
  {
    printf("Bad return code trying to retrieve the  number of authentication"
        " time-outs to requested server. (result = %d)\n", result);
  }
  else
  {
    printf("Authentication time-out count to radius authentication server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the  number of RADIUS packets of unknown type received
*           from requested server on the authentication port.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatUnknownTypeGet(openapiClientHandle_t *pClientHandle,
                               open_buffdesc *pHostAddr,
                               uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatUnknownTypeGet(pClientHandle, pHostAddr,
                                                      pPacketCount)))
  {
    printf("Bad return code trying to retrieve the  number of RADIUS packets"
        " of unknown type received from requested server on the authentication port."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Unknown type packet received count from radius authentication server:%s"
        " on authentication port is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS packets received from server
*           on the authentication port that were dropped for some other reason.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerStatPktsDroppedGet(openapiClientHandle_t *pClientHandle,
                               open_buffdesc *pHostAddr,
                               uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerStatPktsDroppedGet(pClientHandle,
                                                      pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS packets"
        " received from server on the authentication port that were dropped"
        "for some other reason. (result = %d)\n", result);
  }
  else
  {
    printf("Dropped packet count received from radius authentication server:%s"
        " on authentication port is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Gets server name of specified accounting server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerName    @b{(input)}  Name of the server.
* @param    pServerAddr    @b{(output)}  Radius server address[Host name/IP address].
* @param    pType          @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusNamedAcctServerAddrGet(openapiClientHandle_t *pClientHandle,
                             open_buffdesc *pServerName,
                             open_buffdesc *pServerAddr,
                             OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusNamedAcctServerAddrGet(pClientHandle, pServerName,
                                                    pServerAddr, pType)))
  {
    printf("Bad return code trying to gets server name of specified accounting"
        " server. (result = %d)\n", result);
  }
  else
  {
    printf("Acct server address details with serverName:%s is server:%s addrType:%s\n",
        (char *)pServerName->pstart, (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "Invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Verify a RADIUS accounting server is a configured server.
*
* @param    pClientHandle @b{(input)}  Client handle from registration API.
* @param    pServerAddr   @b{(input)}  Server address.
* @param    type          @b{(input)}  Address type[DNS/IP address].
* @param    pVerified     @b{(output)}  Returns TRUE if configured, else FALSE.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAccountingServerIPHostNameVerify(openapiClientHandle_t *pClientHandle,
                                       open_buffdesc *pServerAddr,
                                       OPEN_IP_ADDRESS_TYPE_t type,
                                       OPEN_BOOL_t  *pVerified)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAccountingServerIPHostNameVerify(pClientHandle,
                                                              pServerAddr,
                                                              type, pVerified)))
  {
    printf("Bad return code trying to verify a RADIUS accounting server"
        " is a configured server. (result = %d)\n", result);
  }
  else
  {
    printf("Radius accounting server[ip:%s addrType:%s] %s \n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid",
        (OPEN_TRUE == *pVerified) ? "Exist" : "Doesn't exist");
  }
  return;
}


/*********************************************************************
* @purpose  Gets server name of specified accounting server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pServerAddr     @b{(input)}  Configuredserver address[Host name/IP address].
* @param    type            @b{(input)}  Address type[DNS/IP address].
* @param    pServerName     @b{(output)} Name of the server.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerAcctHostNameGet(openapiClientHandle_t *pClientHandle,
                            open_buffdesc *pServerAddr,
                            OPEN_IP_ADDRESS_TYPE_t type,
                            open_buffdesc *pServerName)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerAcctHostNameGet(pClientHandle, pServerAddr,
                                                   type, pServerName)))
  {
    printf("Bad return code trying to gets server name of specified accounting"
        " server. (result = %d)\n", result);
  }
  else
  {
    printf("Server name of server:%s addrType:%s is %s\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid",
        (char *)pServerName->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the round trip time.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pRoundTripTime  @b{(output)}  Round trip time.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatRTTGet(openapiClientHandle_t *pClientHandle,
                           open_buffdesc *pHostAddr,
                           uint32_t *pRoundTripTime)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatRTTGet(pClientHandle, pHostAddr,
                                                  pRoundTripTime)))
  {
    printf("Bad return code trying to get the round trip time."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Round trip time to radius acct server:%s is %d \n",
        (char *)pHostAddr->pstart, *pRoundTripTime);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS accounting-request packets
*            sent to requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount     @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatReqGet(openapiClientHandle_t *pClientHandle,
                           open_buffdesc *pHostAddr,
                           uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatReqGet(pClientHandle, pHostAddr,
                                                  pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " accounting-request packets sent to requested server. (result = %d)\n", result);
  }
  else
  {
    printf("Accounting-request packet count sent to radius server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS accounting-request packets
*           retransmitted to requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatRetransGet(openapiClientHandle_t *pClientHandle,
                               open_buffdesc *pHostAddr,
                               uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatRetransGet(pClientHandle,
                                                      pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " accounting-request packets retransmitted to requested server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Accounting-request packet count retransmitted to radius server:%s"
        " is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS accounting-response packets
*           received from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatResponseGet(openapiClientHandle_t *pClientHandle,
                                open_buffdesc *pHostAddr,
                                uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatResponseGet(pClientHandle, pHostAddr,
                                                       pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " accounting-response packets received from requested server."
        " (result = %d)\n", result);

  }
  else
  {
    printf("Accounting-response packet count received from server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of malformed RADIUS accounting-response
*           packets received from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)}  Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatMalformedResponseGet(openapiClientHandle_t *pClientHandle,
                                         open_buffdesc *pHostAddr,
                                         uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatMalformedResponseGet(pClientHandle,
                                                                pHostAddr,
                                                                pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of malformed RADIUS"
        " accounting-response packets received from requested server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Malformed accounting-response packet count received from radius"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS accounting-response bad
*           auth packets received from requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)} Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatBadAuthGet(openapiClientHandle_t *pClientHandle,
                               open_buffdesc *pHostAddr,
                               uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatBadAuthGet(pClientHandle,
                                                      pHostAddr, pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " accounting-response bad auth packets received from requested server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Accounting-response bad auth packet count received from radius"
        " server:%s is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS accounting-request packets
*           destined for this server that have not yet timed out or 
*           received a response from this server received from requested 
*           server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)} Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatPendingReqGet(openapiClientHandle_t *pClientHandle,
                                  open_buffdesc *pHostAddr,
                                  uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatPendingReqGet(pClientHandle,
                                                         pHostAddr,
                                                         pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS"
        " accounting-request packets destined for this server that have"
        " not yet timed out or received a response from this server received"
        "from requested server. (result = %d)\n", result);
  }
  else
  {
 printf("Non timed out accounting-request packet count received from radius"
        " server:%s is %d \n", (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the  number of accounting time-outs to requested server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pTimeout        @b{(output)} Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatTimeoutsGet(openapiClientHandle_t *pClientHandle,
                                open_buffdesc *pHostAddr,
                                uint32_t *pTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatTimeoutsGet(pClientHandle,
                                                       pHostAddr,
                                                       pTimeout)))
  {
    printf("Bad return code trying to retrieve the  number of accounting"
        " time-outs to requested server. (result = %d)\n", result);
  }
  else
  {
    printf("Accounting time-out count to server:%s is %d \n",
        (char *)pHostAddr->pstart, *pTimeout);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the  number of RADIUS packets of unknown type
*           received from requested server on the accounting port.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)} Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatUnknownTypeGet(openapiClientHandle_t *pClientHandle,
                                   open_buffdesc *pHostAddr,
                                   uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatUnknownTypeGet(pClientHandle,
                                                          pHostAddr,
                                                          pPacketCount)))
  {
    printf("Bad return code trying to retrieve the  number of RADIUS packets"
        " of unknown type received from requested server on the accounting port."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Unknown type packet received count from server:%s"
        " on accounting port is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the number of RADIUS packets received from server
*           on the accounting port that were dropped for some other reason.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pHostAddr       @b{(input)}  Host Address of the radius server.
* @param    pPacketCount    @b{(output)} Number of packets.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerStatPktsDroppedGet(openapiClientHandle_t *pClientHandle,
                                   open_buffdesc *pHostAddr,
                                   uint32_t *pPacketCount)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerStatPktsDroppedGet(pClientHandle,
                                                          pHostAddr,
                                                          pPacketCount)))
  {
    printf("Bad return code trying to retrieve the number of RADIUS packets"
        " received from server on the accounting port that were dropped for"
        " some other reason. (result = %d)\n", result);
  }
  else
  {
    printf("Dropped packet count received from radius authentication server:%s"
        " on accounting port is %d \n",
        (char *)pHostAddr->pstart, *pPacketCount);
  }
  return;
}


/*********************************************************************
* @purpose  Gets first accounting configured server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerAddr    @b{(output)}  First configured radius server.
* @param    pType          @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerFirstIPHostNameGet(openapiClientHandle_t *pClientHandle,
                                   open_buffdesc *pServerAddr,
                                   OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerFirstIPHostNameGet(pClientHandle,
                                                          pServerAddr,
                                                          pType)))
  {
    printf("Bad return code trying to gets first accounting configured server."
        " (result = %d)\n", result);
  }
  else
  {
    printf("Details of first accounting configured server details:- server:%s addrType:%s\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Gets global selected Radius server address, server type
*           and address type.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    servType       @b{(input)}  Radius server type[Auth/Acct].
* @param    pServerAddr    @b{(output)}  Radius server address.
* @param    pType          @b{(output)}  Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusServerGlobalCurentGet(openapiClientHandle_t *pClientHandle,
                            OPEN_RADIUS_SERVER_TYPE_t servType,
                            open_buffdesc *pServerAddr,
                            OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusServerGlobalCurentGet(pClientHandle, servType,
                                                   pServerAddr, pType)))
  {
    printf("Bad return code trying to gets global selected Radius server"
        " address, server type and address type. (result = %d)\n", result);
  }
  else
  {
    printf("Global selected %s server:[%s addrType:%s] \n",
        (OPEN_RADIUS_SERVER_TYPE_AUTH == servType) ? "Auth" :
        (OPEN_RADIUS_SERVER_TYPE_ACCT == servType) ? "Acct" : "invalid type" ,
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Get the configured accounting server next in the list
*           after the specified server.
*
* @param    pClientHandle      @b{(input)}  Client handle from registration API.
* @param    pCurServerAddr     @b{(input)}  current server information.
* @param    pNextServerAddr    @b{(output)} next configured server information.
* @param    pType              @b{(output)} Address type[DNS/IP address].

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAcctServerNextIPHostNameGet(openapiClientHandle_t *pClientHandle,
                                  open_buffdesc *pCurServerAddr,
                                  open_buffdesc *pNextServerAddr,
                                  OPEN_IP_ADDRESS_TYPE_t *pType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAcctServerNextIPHostNameGet(pClientHandle,
                                                         pCurServerAddr,
                                                         pNextServerAddr,
                                                         pType)))
  {
    printf("Bad return code trying to get the configured accounting server next"
        " in the list after the specified server. (result = %d)\n", result);
  }
  else
  {
    printf("Details of  next configured acct server details after given server:%s :-"
        " server:%s addrType:%s\n", (char *)pCurServerAddr->pstart,
        (char *)pNextServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == *pType) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == *pType) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == *pType) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Get the port number of the accounting server being used.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pServerAddr     @b{(input)}  Radius server address.
* @param    type            @b{(input)}  Address type[DNS/IP address].
* @param    pPort           @b{(output)} port number.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAccountingHostNameServerPortNumGet(openapiClientHandle_t *pClientHandle,
                                         open_buffdesc *pServerAddr,
                                         OPEN_IP_ADDRESS_TYPE_t type,
                                         uint32_t *pPort)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAccountingHostNameServerPortNumGet(pClientHandle,
                                                                pServerAddr,
                                                                type, pPort)))
  {
    printf("Bad return code trying to get the port number of the accounting"
        " server being used. (result = %d)\n", result);
  }
  else
  {
    printf("Port number of acct server[ip:%s addrType:%s] is %d\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid",
        *pPort);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the port number of the accounting server being used.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerAddr    @b{(input)}  Radius server address.
* @param    type           @b{(input)}  Address type[DNS/IP address].
* @param    port           @b{(input)}  Port number.

*
* @returns  none
*
* @end
*********************************************************************/
void 
radiusAccountingHostNameServerPortNumSet(openapiClientHandle_t *pClientHandle,
                                         open_buffdesc *pServerAddr,
                                         OPEN_IP_ADDRESS_TYPE_t type,
                                         uint32_t port)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAccountingHostNameServerPortNumSet(pClientHandle,
                                                                pServerAddr,
                                                                type, port)))
  {
    printf("Bad return code trying to sets the port number of the accounting"
        " server being used. (result = %d)\n", result);
  }
  else
  {
    printf("Port number %d is successfully set for acct server[ip:%s addrType:%s]\n",
        port,
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid");
  }
  return;
}


/*********************************************************************
* @purpose  Verify a RADIUS authentication server is a configured server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerAddr    @b{(input)}  Server address.
* @param    type           @b{(input)}  Address type[DNS/IP address].
* @param    pVerified      @b{(output)}  Returns TRUE if configured, else FALSE.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusHostNameServerAddrVerify(openapiClientHandle_t *pClientHandle,
                               open_buffdesc *pServerAddr,
                               OPEN_IP_ADDRESS_TYPE_t type,
                               OPEN_BOOL_t  *pVerified)
{
  open_error_t result;

  if (OPEN_E_NONE !=
      (result = openapiRadiusHostNameServerAddrVerify(pClientHandle,
                                                      pServerAddr,
                                                      type, pVerified)))
  {
    printf("Bad return code trying to verify a RADIUS authentication server"
        " is a configured server. (result = %d)\n", result);
  }
  else
  {
    printf("Radius authentication server[ip:%s addrType:%s] %s \n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid",
        (OPEN_TRUE == *pVerified) ? "Exist" : "Doesn't exist");
  }
  return;
}


/*********************************************************************
* @purpose  Set the shared secret being used between the radius client
*           and the radius server.
*
* @param    pClientHandle  @b{(input)}  Client handle from registration API.
* @param    pServerAddr    @b{(input)}  Server address.
* @param    paramStatus    @b{(input)}  Radius server config param[Global/local].
* @param    keyType        @b{(input)}  Key type.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusHostNameServerSharedSecretKeyTypeSet(openapiClientHandle_t *pClientHandle,
                                           open_buffdesc *pServerAddr,
                                           OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus,
                                           OPEN_BOOL_t  keyType)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusHostNameServerSharedSecretKeyTypeSet(pClientHandle,
                                                                  pServerAddr,
                                                                  paramStatus,
                                                                  keyType)))
  {
    printf("Bad return code trying to set the shared secret being used between"
        " the radius client and the radius server. (result = %d)\n", result);
  }
  else
  {
    printf("secret for an accounting/server[ip:%s serverConfigParam:%s]"
        " is sucessfully %s \n",
        (char *)pServerAddr->pstart,
        (OPEN_RADIUSGLOBAL == paramStatus) ? "GLOBAL" :
        (OPEN_RADIUSLOCAL == paramStatus) ? "LOCAL" : "invalid",
        (OPEN_TRUE == keyType) ? "configured" : "cleared");
  }
  return;
}


/*********************************************************************
* @purpose  Set the shared secret being used between the radius client
*           and the accounting server.
*
* @param    pClientHandle   @b{(input)}  Client handle from registration API.
* @param    pServerAddr     @b{(input)}  Server address.
* @param    type            @b{(input)}  Address type[DNS/IP address].
* @param    pSharedSecret   @b{(input)}  New shared secret.

*
* @returns  none
*
* @end
*********************************************************************/
void
radiusAccountingHostNameServerSharedSecretSet(openapiClientHandle_t *pClientHandle,
                                              open_buffdesc *pServerAddr,
                                              OPEN_IP_ADDRESS_TYPE_t type,
                                              open_buffdesc *pSharedSecret)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiRadiusAccountingHostNameServerSharedSecretSet(pClientHandle,
                                                                     pServerAddr,
                                                                     type,
                                                                     pSharedSecret)))
  {
    printf("Bad return code trying to set the shared secret being used between"
        " the radius client and the accounting server. (result = %d)\n", result);
  }
  else
  {
    printf("Secret for accounting server[ip:%s addrType:%s] is sucessfully set\n",
        (char *)pServerAddr->pstart,
        (OPEN_IP_ADDRESS_IPV4 == type) ? "IPV4" :
        (OPEN_IP_ADDRESS_IPV6 == type) ? "IPV6" :
        (OPEN_IP_ADDRESS_DNS == type) ? "DNS" : "invalid");
  }
  return;
}
/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;
  int  show_help = 1;
  uint32_t port;
  OPEN_BOOL_t keyType;
  OPEN_IP_ADDRESS_TYPE_t serverAddrType;
  OPEN_RADIUS_SERVER_TYPE_t servType;
  OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t servConfig;
  char strAddr[100]; 
  open_buffdesc servAddr = {.pstart = strAddr, .size = sizeof(strAddr)};
  char strNxtAddr[100]; 
  open_buffdesc servNxtAddr = {.pstart = strNxtAddr, .size = sizeof(strNxtAddr)};
  char strName[32]; 
  open_buffdesc servName = {.pstart = strName, .size = sizeof(strName)};
  char strsharedSecret[100]; 
  open_buffdesc sharedSecret = {.pstart = strsharedSecret, .size = sizeof(strsharedSecret)};
  uint32_t maxRetransmit;
  uint32_t count;
  uint32_t packetCount;
  uint32_t timeOut;
  uint32_t roundTripTime;
  OPEN_CONTROL_t accntMode;
  OPEN_BOOL_t verified;
  OPEN_RADIUS_SERVER_ENTRY_TYPE_t serverEntryType;

  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  errno = 0;
  testNum = strtol(argv[1], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid test number: (%s)\n", argv[1]);
    printAppMenu();
    exit(1);
  }

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
    show_help = 0;
  }
  else
  {
    switch (testNum - (NUMBER_OF_FUNCTIONS))
    {
      case 0:
        if (argc == 2)
        {
          runSanity(&clientHandle);
          show_help = 0;
        }
        break;
      case 1:
        if (argc == 3)
        {
          servType = atoi(argv[2]);
          radiusConfiguredServersCountGet(&clientHandle, servType, &count);
          show_help = 0;
        }
        break;
      case 2:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerRetransGet(&clientHandle, &servAddr, &maxRetransmit, &servConfig);
          show_help = 0;
        }
        break;
      case 3:
        if (5 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          maxRetransmit = atoi(argv[3]);
          servConfig = atoi(argv[4]);
          radiusServerRetransSet(&clientHandle, &servAddr, maxRetransmit, servConfig);
          show_help = 0;
        }
        break;
      case 4:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerTimeOutGet(&clientHandle, &servAddr, &timeOut, &servConfig);
          show_help = 0;
        }
        break;
      case 5:
        if (5 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          timeOut = atoi(argv[3]);
          servConfig = atoi(argv[4]);
          radiusServerTimeOutSet(&clientHandle, &servAddr, timeOut, servConfig);
          show_help = 0;
        }
        break;
      case 6:
        if (argc == 2)
        {
          radiusAccountingModeGet(&clientHandle, &accntMode);
          show_help = 0;
        }
        break;
      case 7:
        if (argc == 3)
        {
          accntMode = atoi(argv[2]);
          radiusAccountingModeSet(&clientHandle, accntMode);
          show_help = 0;
        }
        break;
      case 8:
        if (2 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          servAddr.size = sizeof(strAddr);
          radiusServerFirstIPHostNameGet(&clientHandle, &servAddr, &serverAddrType);
          show_help = 0;
        }
        break;
      case 9:
        if (argc == 3)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          memset(strNxtAddr, 0, sizeof(strNxtAddr));
          servNxtAddr.size = sizeof(strNxtAddr);
          radiusServerNextIPHostNameGet(&clientHandle, &servAddr,
                                        &servNxtAddr, &serverAddrType);
          show_help = 0;
        }
        break;
      case 10:
        if (4 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          memset(strName, 0, sizeof(strName));
          servName.size = sizeof(strName);
          radiusServerHostNameGet(&clientHandle, &servAddr, serverAddrType,
                                  &servName);
          show_help = 0;
        }
        break;
      case 11:
        if (argc == 2)
        {
          memset(strAddr, 0, sizeof(strAddr));
          servAddr.size = sizeof(strAddr);
          radiusServerGlobalSelectedGet(&clientHandle, &servAddr,
                                        &serverAddrType);
          show_help = 0;
        }
        break;
      case 12:
        if (argc == 5)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          memset(strName, 0, sizeof(strName));
          strncpy(strName, argv[4], sizeof(strName)-1);
          servName.size = strlen(strName);
          radiusServerNameSet(&clientHandle, &servAddr, serverAddrType,
                              &servName);
          show_help = 0;
        }
        break;
      case 13:
        if (argc == 5)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          serverEntryType = atoi(argv[4]);
          radiusAuthServerTypeSet(&clientHandle, &servAddr, serverAddrType,
                                  serverEntryType);
          show_help = 0;
        }
        break;
      case 14:
        if (argc == 3)
        {
          memset(strName, 0, sizeof(strName));
          strncpy(strName, argv[2], sizeof(strName)-1);
          servName.size = strlen(strName);
          memset(strAddr, 0, sizeof(strAddr));
          servAddr.size = sizeof(strAddr);
          radiusNamedServerAddrGet(&clientHandle, &servName, &servAddr,
                                   &serverAddrType);
          show_help = 0;
        }
        break;
      case 15:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatRTTGet(&clientHandle, &servAddr, &roundTripTime);
          show_help = 0;
        }
        break;
      case 16:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatAccessReqGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 17:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatAccessRetransGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 18:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatAccessAcceptGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 19:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatAccessRejectGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 20:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatAccessChallengeGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 21:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatMalformedAccessResponseGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 22:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatBadAuthGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 23:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatPendingReqGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 24:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatTimeoutsGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 25:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatUnknownTypeGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 26:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusServerStatPktsDroppedGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 27:
        if (3 == argc)
        {
          memset(strName, 0, sizeof(strName));
          strncpy(strName, argv[2], sizeof(strName)-1);
          servName.size = strlen(strName);
          memset(strAddr, 0, sizeof(strAddr));
          servAddr.size = sizeof(strAddr);
          radiusNamedAcctServerAddrGet(&clientHandle, &servName, &servAddr,
                                       &serverAddrType);
          show_help = 0;
        }
        break;
      case 28:
        if (4 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          radiusAccountingServerIPHostNameVerify(&clientHandle, &servAddr,
                                                 serverAddrType, &verified);
          show_help = 0;
        }
        break;
      case 29:
        if (4 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          memset(strName, 0, sizeof(strName));
          servName.size = sizeof(strName);
          radiusServerAcctHostNameGet(&clientHandle, &servAddr, serverAddrType,
                                      &servName);
          show_help = 0;
        }
        break;
      case 30:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatRTTGet(&clientHandle, &servAddr, &roundTripTime);
          show_help = 0;
        }
        break;
      case 31:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatReqGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 32:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatRetransGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 33:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatResponseGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 34:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatMalformedResponseGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 35:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatBadAuthGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 36:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatPendingReqGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 37:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatTimeoutsGet(&clientHandle, &servAddr, &timeOut);
          show_help = 0;
        }
        break;
      case 38:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatUnknownTypeGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 39:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          radiusAcctServerStatPktsDroppedGet(&clientHandle, &servAddr, &packetCount);
          show_help = 0;
        }
        break;
      case 40:
        if (2 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          servAddr.size = sizeof(strAddr);
          radiusAcctServerFirstIPHostNameGet(&clientHandle, &servAddr, &serverAddrType);
          show_help = 0;
        }
        break;
      case 41:
        if (3 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          memset(strNxtAddr, 0, sizeof(strNxtAddr));
          servNxtAddr.size = sizeof(strNxtAddr);
          radiusAcctServerNextIPHostNameGet(&clientHandle, &servAddr,
                                            &servNxtAddr, &serverAddrType);
          show_help = 0;
        }
      break;
      case 42:
        if (3 == argc)
        {
          servType = atoi(argv[2]);
          radiusServerGlobalCurentGet(&clientHandle, servType, &servAddr, &serverAddrType);
          show_help = 0;
        }
        break;
      case 43:
        if (4 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          radiusAccountingHostNameServerPortNumGet(&clientHandle, &servAddr,
                                                   serverAddrType, &port);
          show_help = 0;
        }
        break;
      case 44:
        if (5 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          port = atoi(argv[4]);
          radiusAccountingHostNameServerPortNumSet(&clientHandle, &servAddr,
                                                    serverAddrType, port);
          show_help = 0;
        }
        break;
      case 45:
        if (4 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          radiusHostNameServerAddrVerify(&clientHandle, &servAddr,
                                          serverAddrType, &verified);
          show_help = 0;
        }
        break;
      case 46:
        if (5 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          servConfig = atoi(argv[3]);
          keyType = atoi(argv[4]);
          radiusHostNameServerSharedSecretKeyTypeSet(&clientHandle, &servAddr, servConfig, keyType);
          show_help = 0;
        }
        break;
      case 47:
        if (5 == argc)
        {
          memset(strAddr, 0, sizeof(strAddr));
          strncpy(strAddr, argv[2], sizeof(strAddr)-1);
          servAddr.size = strlen(strAddr);
          serverAddrType = atoi(argv[3]);
          memset(strsharedSecret, 0, sizeof(strsharedSecret));
          strncpy(strsharedSecret, argv[4], sizeof(strsharedSecret)-1);
          sharedSecret.size = sizeof(strsharedSecret);
          radiusAccountingHostNameServerSharedSecretSet(&clientHandle, &servAddr,
                                                serverAddrType, &sharedSecret);
          show_help = 0;
        }
        break;
      default:
        break;
    }
  }
  if (show_help == 1)
  {
    printAppMenu();
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
