/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  routing_example.c
*
* @purpose   Routing Configuration and Status APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    07/23/2012
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_routing.h"


/*********************************************************************
* @purpose  Convert an IP address in a open_inet_addr_t structure to
*           a string for display
*
* @param    addr  @b{(input)}  address to be converted
* @param    buffer @b{(output)} storage to contain
*
* @returns  pointer to buffer if conversion successful
*           NULL if error
*
* @notes
*
* @end
*********************************************************************/
static const char *ipAddressFormat(open_inet_addr_t *addr, char *buffer)
{
  uint32_t  ip4_addr;
  uint8_t   addr8[16];
  int af;

  switch (addr->family)
  {
  case OPEN_AF_INET:
    af = AF_INET;
    ip4_addr = htonl(addr->addr.ipv4);
    memcpy(addr8, &ip4_addr, sizeof(ip4_addr));
    break;
  case OPEN_AF_INET6:
    af = AF_INET6;
    memcpy(addr8, addr->addr.ipv6.u.addr8, sizeof(addr8));
    break;
  default:
    return(NULL);
  }

  return(inet_ntop(af, addr8, buffer, INET6_ADDRSTRLEN));
}

/*********************************************************************
 * @purpose  used for converting MAC in ascii string format to a char array of hex data
 *
 * @param    buf         contains ascii string "xx:xx:xx:xx:xx:xx"
 * @param    mac         return hex data
 *
 * @returns void
 *
 * @notes
 *
 * @end
 *********************************************************************/
static OPEN_BOOL_t rtrUtilConvertMac( unsigned char * buf, unsigned char * mac)
{
  uint32_t i, j, digit_count = 0;
  unsigned char mac_address[256];


  if (strlen ((const char *)buf) != 17)
  {                             /* test string length */
    return OPEN_FALSE;
  }

  for (i = 0, j = 0; i < 17; i++, j++)
  {
    digit_count++;
    switch (buf[i])
    {
      case '0':
        mac_address[j] = 0x0;
        break;
      case '1':
        mac_address[j] = 0x1;
        break;
      case '2':
        mac_address[j] = 0x2;
        break;
      case '3':
        mac_address[j] = 0x3;
        break;
      case '4':
        mac_address[j] = 0x4;
        break;
      case '5':
        mac_address[j] = 0x5;
        break;
      case '6':
        mac_address[j] = 0x6;
        break;
      case '7':
        mac_address[j] = 0x7;
        break;
      case '8':
        mac_address[j] = 0x8;
        break;
      case '9':
        mac_address[j] = 0x9;
        break;
      case 'a':
      case 'A':
        mac_address[j] = 0xA;
        break;
      case 'b':
      case 'B':
        mac_address[j] = 0xB;
        break;
      case 'c':
      case 'C':
        mac_address[j] = 0xC;
        break;
      case 'd':
      case 'D':
        mac_address[j] = 0xD;
        break;
      case 'e':
      case 'E':
        mac_address[j] = 0xE;
        break;
      case 'f':
      case 'F':
        mac_address[j] = 0xF;
        break;
      case ':':
        if (digit_count != 3)
        {                         /* if more or less than 2 digits return false */
          return OPEN_FALSE;
        }
        j--;
        digit_count = 0;
        break;
      default:
        return OPEN_FALSE;
        break;
    }
  }

  for (i = 0; i < 6; i++)
  {
    mac[i] = ((mac_address[(i * 2)] << 4) + mac_address[(i * 2) + 1]);
  }
  return OPEN_TRUE;
}

/*******************************************************************
*
* @brief  This function prints the Routing Configuration Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
* @end
*********************************************************************/
void printRoutingAppMenu()
{
  printf("Usage: routing_example API <test number> [<arg1>][<arg2>] ... \n");
  printf("Test API   1: Add IP address on an interface: routing_example API 1 <interface> "
         "<address family> <address type> <IP Address> <Pfx length> <arg1>\n");
  printf("Test API   2: Delete IP address on an interface: routing_example API 2 <interface> "
         "<address family> <address type> <IP Address> <Pfx length> <arg1>\n");
  printf("Test API   3: Enable/Disable IP Routing Globally: routing_example API 3 <address family> <mode>\n");
  printf("Test API   4: Add a test static ARP entry {192.168.1.1 00:11:22:33:44:55} : routing_example API 4\n");
  printf("Test API   5: Delete the test static ARP entry {192.168.1.1 00:11:22:33:44:55}: routing_example API 5\n");
  printf("Test API   6: Display all ARP entries: routing_example API 6\n");
  printf("Test API   7: Enable/Disable IP Redirects mode: routing_example API 7 <mode>\n");
  printf("Test API   8: Enable/Disable IP ICMP Echo Reply mode: routing_example API 8 <mode>\n");
  printf("Test API   9: Enable/Disable IP Helper mode: routing_example API 9 <mode>\n");
  printf("Test API  10: Add IP Helper address: routing_example API 10 <IP Address> <UDP Port>\n");
  printf("Test API  11: Delete IP Helper address: routing_example API 11 <IP Address> <UDP Port>\n");
  printf("Test API  12: Configure Interface MTU: routing_example API 12 <interface> <address family> <MTU>\n");
  printf("Test API  13: Enable/Disable IP Net Directed BCast mode: routing_example API 13 <interface> <mode>\n");
  printf("Test API  14: Enable/Disable IP Proxy ARP mode: routing_example API 14 <interface> <mode>\n");
  printf("Test API  15: Enable/Disable IP Redirects mode: routing_example API 15 <interface> <mode>\n");
  printf("Test API  16: Enable/Disable IP Destination Unreachable mode: routing_example API 16 <interface> <mode>\n");
  printf("Test API  17: Enable/Disable IP Interface Routing mode: routing_example API 17 <interface> <mode>\n");
  printf("Test API  18: Add Interface IP Helper address: routing_example API 18 <interface> <IP Address> <UDP Port>\n");
  printf("Test API  19: Delete Interface IP Helper address: routing_example API 19 <interface> <IP Address> <UDP Port>\n");
  printf("Test API  20: Add Interface IP Helper Discard entry: routing_example API 20 <interface> <UDP Port>\n");
  printf("Test API  21: Delete Interface IP Helper Discard entry: routing_example API 21 <interface> <UDP Port>\n");
  printf("Test API  22: Add an IP route: routing_example API 22 <Address Family> <IP Address> <Pfx length> "
         "<Next Hop> <rtePref> <interface> <mpls-label1> <mpls-label2> <mpls-label3>\n");
  printf("Test API  23: Delete an IP route: routing_example API 23 <Address Family> <IP Address> <Pfx length> "
         "<Next Hop> <interface> <mpls-label1> <mpls-label2> <mpls-label3>\n");
  printf("Test API  24: Create a VLAN Routing Interface: routing_example API 24 <VLAN Id> \n");
  printf("Test API  25: Delete a VLAN Routing Interface: routing_example API 25 <VLAN Id> \n");
  printf("Test API  26: Test Routing Configuration OpEN APIs sanity: routing_example API 26 \n");
  printf("Test API  27: Create a BFD session: routing_example API 27 <Address family> <Intf> <Dest IP> <Src IP> \n");
  printf("Test API  28: Delete a BFD session: routing_example API 28 <Address family> <Intf> <Dest IP> <Src IP> \n");
  printf("Test API  29: Show BFD neighbors: routing_example API 29\n");
  printf("Test API  30: Find a BFD session: routing_example API 30 <Address family> <Dest IP>\n");
  printf("Test API  31: Iterate the configured VRFs\n");
  printf("Test API  32: Show the ifnum for a VLAN Routing Interface: routing_example API 32 <VLAN Id>\n");
  printf("Test API  34: Get the router discovery mode on given interface: routing_example API 34 <interface>\n");
  printf("Test API  35: Enable/Disable the router discovery mode on given interface: routing_example API 35 <interface> <mode>\n");
  printf("Test API  36: Get advertisement packet's destination address: routing_example API 36 <interface>\n");
  printf("Test API  37: Set advertisement packet's destination address: routing_example API 37 <interface> <address>\n");
  printf("Test API  38: Get max time allowed between router advertisements from interface: routing_example API 38 <interface>\n");
  printf("Test API  39: Set max time allowed between router advertisements from interface: routing_example API 39 <interface> <time>\n");
  printf("Test API  40: Get min time allowed between router advertisements from interface: routing_example API 40 <interface>\n");
  printf("Test API  41: Set min time allowed between router advertisements from interface: routing_example API 41 <interface> <time>\n");
  printf("Test API  42: Get the default minimum advertisement interval on interface: routing_example API 42 <interface>\n");
  printf("Test API  43: Revert MinAdvertisementInterval to its default value on interface: routing_example API 43 <interface>\n");
  printf("Test API  44: Get lifetime field value of router advertisement sent from given interface: routing_example API 44 <interface>\n");
  printf("Test API  45: Set lifetime field value of router advertisement sent from given interface: routing_example API 45 <interface> <time>\n");
  printf("Test API  46: Get the default value of lifetime field of router advertisement sent from interface: routing_example API 46 <interface>\n");
  printf("Test API  47: Revert AdvertisementLifetime to its default value on interface: routing_example API 47 <interface>\n");
  printf("Test API  48: Get the preferability of the address as a default router address: routing_example API 48 <interface>\n");
  printf("Test API  49: Set the preferability of the address as a default router address: routing_example API 49 <interface> <pref-level>\n");
  printf("Test API  50: Get the Grat ARP mode on given unnumbered interface: routing_example API 50 <interface>\n");
  printf("Test API  51: Set the Grat ARP mode on given unnumbered interface: routing_example API 51 <interface> <grat-arp-mode>\n");
  printf("Test API  52: Enable or disable gratitious ARP on an interface: routing_example API 52 <interface> <grat-arp-mode>\n");
  printf("Test API  53: Enable or disable the ARP dynamic entry renew mode: routing_example API 53 <mode>\n");
  printf("Test API  54: Displays the ARP dynamic entry renew mode: routing_example API 54\n");
  printf("Test API  55: Sets the maximum number of entries in the ARP cache: routing_example API 55 <cache-size>\n");
  printf("Test API  56: Sets the ARP request max retries count: routing_example API 56 <count>\n");
  printf("Test API  57: Displays whether local proxy ARP is enabled on an interface: routing_example API 57 <interface>\n");
  printf("Test API  58: Enable or disable the local proxy ARP on an interface: routing_example API 58 <interface> <mode>\n");
  printf("Test API  59: Displays whether proxy ARP is enabled on an interface: routing_example API 59 <interface>\n");
  printf("Test API  60: Sets the ARP request response timeout: routing_example API 60 <timeout>\n");
  printf("Test API  61: Sets the ARP entry ageout time: routing_example API 61 <timeout>\n");
  printf("Test API  62: Retrieve various ARP cache statistics: routing_example API 62 <vrfName>>\n");
  printf("Test API  63: Purges a specific dynamic/gateway entry from the ARP cache: routing_example API 63 <vrfName> <ipAddress> <interface>\n");
  printf("Test API  64: Clears the ARP cache of all dynamic/gateway entries: routing_example API 64 <vrfName> <gateway-flag>\n");
  printf("Test API  65: Delete a static ARP entry of a given VRF instance: routing_example API 65 <vrfName> <ipAddress> <interface>\n");
  printf("Test API  66: Add a static ARP entry of a given VRF instance: routing_example API 66 <vrfName> <ipAddress> <interface> <mac-addr-str>\n");
  printf("Test API  67: Display all static ARP entries currently configured:routing_example API 67 <vrfName>>\n");
  printf("Test API  68: Get the configured IP MTU value on the given interface: routing_example API 68 <ifNum>\n");
  printf("Test API  69: Get the maximum IP MTU that may be set on a given interface: routing_example API 69 <ifNum>\n");
  printf("Test API  70: Get the IP MTU value being enforced on a given interface: routing_example API 70 <ifNum>\n");
  printf("Test API  71: Set the current IP ECMP global load balancing mode: routing_example API 71 <lbMode>\n");
  printf("Test API  72: Determines the current IP ECMP global load balancing mode: routing_example API 72 \n");
  printf("Test API  73: Set the current IP ECMP IPSEC SPI Hashing mode: routing_example API 73 <spiMode>\n");
  printf("Test API  74: Determines the current IP ECMP IPSEC SPI Mode: routing_example API 74 \n");
  printf("Test API  75: Get the admin mode for IP forwarding of net-directed broadcasts: routing_example API 75 <ifNum>\n");
  printf("Test API  76: Set the unnumbered status of an interface: routing_example API 76 <ifNum> <isUnnumbered> <numberedIfc>\n");
  printf("Test API  77: Determine whether given interface is unnumbered and if so, determine the interface whose address it borrows: routing_example API 77 <ifNum>\n");
  printf("Test API  78: Determine whether an interface is configured to forward multicast packets: routing_example API 78 <ifNum>\n");
  printf("Test API  79: Get the router interface mode: routing_example API 79 <ifNum>\n");
  printf("Test API  80: Get the mode of AutoState feature of an interface: routing_example API 80 <ifNum>\n");
  printf("Test API  81: Enable or disable the mode of AutoState feature of an interface: routing_example API 81 <ifNum><mode>\n");
  printf("Test API  82: Determine whether a given IP interface is up for IPv4: routing_example API 82 <ifNum>\n");
  printf("Test API  83: Get the next vlan after this vlan on which routing is enabled: routing_example API 83 <vlanId>\n");
  printf("Test API  84: Get the VLAN ID corresponding to the given internal interface number: routing_example API 84 <ifNum>\n");
  printf("Test API  85: Get the interface ID for a given VLAN: routing_example API 85 <vlanId>\n");
  printf("Test API  86: Get the internal interface number associated with the port-based routing interface corresponding to this internal VLAN: routing_example API 86 <vlanId>\n");
  printf("Test API  87: Get the first valid interface: routing_example API 87 \n");
  printf("Test API  88: Get the next valid interface: routing_example API 88 <prevIfNum>\n");
  printf("Test API  89: Get the administrative mode of sending ICMP Unreachables: routing_example API 89 <ifNum>\n");
  printf("Test API  90: Get the interface mode of sending ICMP Redirects: routing_example API 90 <ifNum>\n");
  printf("Test API  91: Get the bandwidth of the specified interface: routing_example API 91 <ifNum>\n");
  printf("Test API  92: Set the bandwidth of the specified interface: routing_example API 92 <ifNum>\n");
  printf("Test API  93: Get the bandwidth of the specified interface without making any modification to the SET bandwidth: routing_example API 93 <ifNum>\n");
  printf("Test API  94: Get the routing max equal cost entries: routing_example API 94 \n");
  printf("Test API  95: Get the routing max routes entries: routing_example API 95 \n");
  printf("Test API  96: Invokes the IPMAP API funciton to get the dampening operational values: routing_example API 96 <ifNum>\n");
  printf("Test API  97: Get the next protocol ID by iterating through all registered protocols: routing_example API 97 <protoId>\n");
  printf("Test API  98: Get the protocol ID for a given route type: routing_example API 98 <routeType>\n");
  printf("Test API  99: Get the global resilient hashing mode for ECMP trunks: routing_example API 99 \n");
  printf("Test API 100: Set the global resilient hashing mode for ECMP trunks: routing_example API 100 <resHashMode>\n");
  printf("Test API 101: Get the packets forwarded by CPU: routing_example API 101 \n");
  printf("Test API 102: Get the IP packets count received by IP-Stack: routing_example API 102 \n");
  printf("Test API 103: Get source IP interface from Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps: routing_example API 103 \n");
  printf("Test API 104: Set source IP interface for Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps: routing_example API 104 <ifNum> \n");
  printf("Test API 105: Get the configured IPv4 Management interface: routing_example API 105 \n");
  printf("Test API 106: Set the IPv4 Management interface: routing_example API 106 <ifNum> \n");
  printf("Test API 107: Set the IPv4 Management interface and enable dhcp or configure static IP on the management interface: routing_example API 107 <ifNum> <dhcpEnabled> <ipAddr> <subnetMask>\n");
  printf("Test API 108: Get the System mode for uRPF: routing_example API 108 \n");
  printf("Test API 109: Set the System mode for uRPF: routing_example API 109 <urpfEnabled> \n");
  printf("Test API 110: Get the interface uRPF mode and allowDefault option: routing_example API 110 <ifNum> \n");
  printf("Test API 111: Set the interface uRPF mode and allowDefault option: routing_example API 111 <ifNum> <mode> <allowDefault> \n");
  printf("Test API 112: Get the administrative mode of sending ICMP Echo Replies: routing_example API 112 <vrfName> \n");
  printf("Test API 113: Get the ICMP Rate Limiting parameters like burst size and interval: routing_example API 113 <vrfName> \n");
  printf("Test API 114: Get the internal interface number of the given VRF and IPv4 address: routing_example API 114 <vrfName> <ipAddr> \n");
  printf("Test API 115: Get the Router preference for the given protocol and VRF name: routing_example API 115 <vrfName> <origin> \n");
  printf("Test API 116: Set the Router preference for the given protocol and VRF name: routing_example API 116 <vrfName> <origin> <pref> \n");
  printf("Test API 117: Get the administrative mode of sending ICMP Redirects: routing_example API 117 <vrfName> \n");
  printf("Test API 118: Set the administrative mode of sending ICMP Redirects: routing_example API 118 <vrfName> <mode> \n");
  printf("Test API 119: Set ignore mode for ICMP ECHO Requests: routing_example API 119 <vrfName> <mode> \n");
  printf("Test API 120: Set the ICMP Rate Limiting parameters like burst size and interval: routing_example API 120 <vrfName> <burstSize> <interval> \n");
  printf("Test API 121: Clear the Address Conflict Detection Status of a given VRF instance: routing_example API 121 <vrfName> \n");
  printf("Test API 122: Get the Address Conflict Detection Status of a VRF instance: routing_example API 122 <vrfName> \n");
  printf("Test API 123: Get the MAC Address of the last detected address conflict in a VRF instance: routing_example API 123 <vrfName> \n");
  printf("Test API 124: Get the time in seconds since the last address conflict was detected in a VRF instance: routing_example API 124 <vrfName> \n");
  printf("Test API 125: Delete all net-prototype route entries: routing_example API 125 <vrfName> <protoId> \n");
  printf("Test API 126: Unset the VRF participation on an interface: routing_example API 126 <ifNum> \n");
  printf("Test API 127: Set the VRF participation of an interface: routing_example API 127 <ifNum> <vrfName> \n");
  printf("Test API 128: Clear the switch/management ARP entries: routing_example API 128 \n");
  printf("Test API 129: Set or reset the IPv6 address mode on a given router interface: routing_example API 129 <ifNum> <addrMode> <setFlag> \n");

  return;
}

/*********************************************************************
* @purpose  Gets the packets forwarded by CPU.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapForwardingStatsGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* Number of packets forwarded by CPU */ 

  result = openapiIpMapForwardingStatsGet(client_handle, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe number of packets forwarded by CPU is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the number of packets forwarded by CPU. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IP packets count received by IP-Stack.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapIpPacketsReceivedGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t val = 0; /* Number of IP packets received by IP-Stack */ 

  result = openapiIpMapIpPacketsReceivedGet(client_handle, &val);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe number of IP packets received by IP-Stack is %u.\n", val);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the number of IP packets received by IP-Stack. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get source IP interface from Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void sourceInterfacesGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0; /* Internal interface number of the source interface */ 

  result = openapiSourceInterfacesGet(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe source IP interface from Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the source IP interface from Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set source IP interface for Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number for setting as source interface
*
* @returns  none
*
* @end
*********************************************************************/
void sourceInterfacesSet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiSourceInterfacesSet(client_handle, ifNum);
  switch(result)
  { 
    case OPEN_E_NONE:
         printf("\nSuccessfully set the source IP interface for Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps.\n");
         break;
    
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
  
    default:
         printf("\nERROR: Bad return code trying to set source IP interface for Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps. (result = %d).\n", result);
         break;
  } 
      
  return;
}

/*********************************************************************
* @purpose  Get the configured IPv4 Management interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipManagementInterfaceGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ifNum = 0; /* Internal interface number */ 

  result = openapiIpManagementInterfaceGet(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe configured IPv4 Management interface is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the configured IPv4 Management interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv4 Management interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipManagementInterfaceSet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpManagementInterfaceSet(client_handle, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv4 Management interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the configured IPv4 Management interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the IPv4 Management interface and enable dhcp or configure static IP on the management interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
* @param    isDhcpEnabled @b{(input)}  1 (for TRUE) or 2 (for FALSE)
* @param    ipAddrStr     @b{(input)}  IP address in string format
* @param    maskStr       @b{(input)}  SubnetMask in string format
*
* @returns  none
*
* @end
*********************************************************************/
void ipManagementInterfaceParamsSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t isDhcpEnabled, char* ipAddrStr, char* maskStr)
{
  open_error_t result  = OPEN_E_FAIL;
  OPEN_BOOL_t dhcpFlag = OPEN_FALSE;
  uint32_t ipAddr      = 0;
  uint32_t subnetMask  = 0;
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string to a 32 bit integer. result = %d.\n", result);
    return;
  }

  memset(str, 0, sizeof(str));
  strncpy(str, maskStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &subnetMask)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string to a 32 bit integer. result = %d.\n", result);
    return;
  }

  if (1 < isDhcpEnabled)
  {
    printf("\n Invalid isDhcpEnabled value. Expected 0(for Disable) or 1(for Enable).");
    return;
  }

  if (1 == isDhcpEnabled)
  {
    dhcpFlag = OPEN_TRUE;
  }

  result = openapiIpManagementInterfaceParamsSet(client_handle, ifNum, dhcpFlag, ipAddr, subnetMask);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the IPv4 Management interface and set the given enable dhcp or configure static IP on the management interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the IPv4 Management interface and enable dhcp or configure static IP on the management interface. (result = %d)\n", result);
         break;
  } 
  return;
}

/*********************************************************************
* @purpose  Get the System mode for uRPF.
*
* @param    client_handle  @b{(input)} Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapSystemUrpfModeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t urpfMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */

  result = openapiIpMapSystemUrpfModeGet(client_handle, &urpfMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe system mode for uRPF is %u (1-Enabled, 0-Disabled).\n", urpfMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the System mode for uRPF. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the System mode for uRPF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    urpfMode       @b{(input)}  1-Enable or 0-Disable
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapSystemUrpfModeSet(openapiClientHandle_t *client_handle, uint32_t urpfMode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;

  if (1 < urpfMode)
  {
    printf("\n Invalid mode value. Expected 0(for Disable) or 1(for Enable).");
    return;
  }

  if (1 == urpfMode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIpMapSystemUrpfModeSet(client_handle, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the system mode for uRPF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the system mode for uRPF. (result = %d).\n", result);
         break;
  } 
  return;
}

/*********************************************************************
* @purpose  Get the interface uRPF mode and allowDefault option.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number of interface for which to get the urpf mode
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapIntfUrpfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_INTF_URPF_MODE_t mode = OPEN_IPMAP_INTF_URPF_MODE_DISABLE; /* Strict or loose or disabled */
  OPEN_BOOL_t allowDefault = OPEN_FALSE; /* Should take default route into consideration or not */

  result = openapiIpMapIntfUrpfModeGet(client_handle, ifNum, &mode, &allowDefault);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe given interface's uRPF mode is %u (0-Disabled,1-Loose,2-Strict) and allowDefault option is %u (1-TRUE,0-FALSE).\n", mode, allowDefault);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the interface uRPF mode and allowDefault option. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the interface uRPF mode and allowDefault option.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number of interface for which to set the urpf mode
* @param    mode           @b{(input)}  Strict or loose or disabled as defined by OPEN_INTF_URPF_MODE_t
* @param    allowDefault   @b{(input)}  Should take default route into consideration or not
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapIntfUrpfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mode, uint32_t allowDefault)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_INTF_URPF_MODE_t openMode = OPEN_IPMAP_INTF_URPF_MODE_DISABLE;
  OPEN_BOOL_t openFlag = OPEN_FALSE;

  if (mode > 2)
  {
    printf("\n Invalid mode value. Expected 0(for no uRPF) or 1(for Loose Mode uRPF) or 2(Strict Mode uRPF).");
    return;
  }

  if (allowDefault > 1)
  {
    printf("\n Invalid allowDefault value. Expected 0 (for FALSE) or 1 (for TRUE).");
    return;
  }

  switch (mode)
  {
    case OPEN_IPMAP_INTF_URPF_MODE_DISABLE:
         openMode = OPEN_IPMAP_INTF_URPF_MODE_DISABLE;
         break;
    case OPEN_IPMAP_INTF_URPF_MODE_LOOSE:
         openMode = OPEN_IPMAP_INTF_URPF_MODE_LOOSE;
         break;
    default:
         openMode = OPEN_IPMAP_INTF_URPF_MODE_STRICT;
         break;
  }

  if (allowDefault == 1)
  {
    openFlag = OPEN_TRUE;
  }

  result = openapiIpMapIntfUrpfModeSet(client_handle, ifNum, openMode, openFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the interface uRPF mode and allowDefault option.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the interface uRPF mode and allowDefault option. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the administrative mode of sending ICMP Echo Replies.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    *vrfName      @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrICMPEchoReplyModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpMapRtrICMPEchoReplyModeGet(client_handle, &vrfNameBufd, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe administrative mode of sending ICMP Echo Replies for the given VRF instance is %u.\n", mode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the administrative mode of sending ICMP Echo Replies for the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the ICMP Rate Limiting parameters like burst size and interval.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrICMPRatelimitGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t burstSize = 0; /* Number of ICMP messages the IPMAP is allowed per interval */
  uint32_t interval  = 0; /* The time interval between tokens being placed in the bucket */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpMapRtrICMPRatelimitGet(client_handle, &vrfNameBufd, &burstSize, &interval);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe ICMP Rate Limiting parameters for given VRF are: burst size %u and interval %u.\n", burstSize, interval);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the ICMP Rate Limiting parameters like burst size and interval for the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the internal interface number of the given VRF and IPv4 address.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    *vrfName      @b{(input)}  VRF name
* @param    ipAddrStr     @b{(input)}  IP address of interest in string format
*
* @returns  none
*
* @end
*********************************************************************/
void iPAddrToIntIf(openapiClientHandle_t *client_handle, char *vrfName, char *ipAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t ipAddr = 0;
  uint32_t ifNum  = 0; /* Internal interface number for the specified IP address */
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string to a 32 bit integer. result = %d.\n", result);
    return;
  }

  result = openapiIPAddrToIntIf(client_handle, &vrfNameBufd, ipAddr, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe internal interface number of the given VRF and IPv4 address is %u. \n", ifNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the internal interface number of the given VRF and IPv4 address. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the Router preference for the given protocol and VRF name.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    origin         @b{(input)}  Router protocol type
* @param    *pref          @b{(output)} Preference associated with the protocol
*
* @returns  none
*
* @end
*********************************************************************/
void ipVrRouterPreferenceGet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t origin)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t pref = 0;

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiVrIpRouterPreferenceGet(client_handle, &vrfNameBufd, origin, &pref);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Router preference for the given protocol and VRF name is %u.\n", pref);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the Router preference for the given protocol and VRF name. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Set the Router preference for the given protocol and VRF name.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    origin         @b{(input)}  Router protocol type
* @param    pref           @b{(input)}  Preference associated with the protocol
*
* @returns  none
*
* @end
*********************************************************************/
void ipVrRouterPreferenceSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t origin, uint32_t pref)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiVrIpRouterPreferenceSet(client_handle, &vrfNameBufd, origin, pref);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Router preference for the given protocol and VRF name.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Router preference for the given protocol and VRF name. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the administrative mode of sending ICMP Redirects.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrICMPRedirectsModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_CONTROL_t mode = OPEN_DISABLE;

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpMapRtrICMPRedirectsModeGet(client_handle, &vrfNameBufd, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe administrative mode of sending ICMP Redirects for the given VRF instance is %u.\n", mode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the administrative mode of sending ICMP Redirects for the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Set the administrative mode of sending ICMP Redirects.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrICMPRedirectsModeSet(openapiClientHandle_t *client_handle, char *vrfName, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;
 
  if (1 < mode)
  {
    printf("\n Invalid mode value. Expected 0(for Disable) or 1(for Enable).");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIpMapRtrICMPRedirectsModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the the administrative mode of sending ICMP Redirects for given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the administrative mode of sending ICMP Redirects. (result = %d)\n", result);
         break;
  } 
  return;
}

/*********************************************************************
* @purpose  Set ignore mode for ICMP ECHO Requests.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrICMPEchoReplyModeSet(openapiClientHandle_t *client_handle, char *vrfName, OPEN_CONTROL_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;
 
  if (1 < mode)
  {
    printf("\n Invalid mode value. Expected 0(for Disable) or 1(for Enable).");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIpMapRtrICMPEchoReplyModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the ignore mode for ICMP ECHO Requests for given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set ignore mode for ICMP ECHO Requests. (result = %d)\n", result);
         break;
  } 
  return;
}

/*********************************************************************
* @purpose  Set the ICMP Rate Limiting parameters like burst size and interval.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    burstSize      @b{(input)}  Number of ICMP messages the IPMAP is allowed per interval
* @param    interval       @b{(input)}  The time interval between tokens being placed in the bucket
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrICMPRatelimitSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t burstSize, uint32_t interval)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpMapRtrICMPRatelimitSet(client_handle, &vrfNameBufd, burstSize, interval);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the ICMP Rate Limiting parameters like burst size and interval for given VRF instance.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the ICMP Rate Limiting parameters like burst size and interval. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Clear the Address Conflict Detection Status of a given VRF instance.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void vrAddrConflictDetectStatusClear(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiVrAddrConflictDetectStatusClear(client_handle, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared the Address Conflict Detection Status of given VRF instance.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to clear the Address Conflict Detection Status of the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the Address Conflict Detection Status of a VRF instance.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void vrAddrConflictDetectStatusGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_BOOL_t conflictDetectStatus = OPEN_FALSE;

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiVrAddrConflictDetectStatusGet(client_handle, &vrfNameBufd, &conflictDetectStatus);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Address Conflict Detection Status of given VRF instance is %u.\n", conflictDetectStatus);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the Address Conflict Detection Status of the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the MAC Address of the last detected address conflict in a VRF instance.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void vrAddrConflictLastDetectMACGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_buffdesc conflictMAC; /* MAC of last detected conflicting host */
  char str[OPEN_MAC_ADDR_LEN+1] = "";

  memset(str, 0, sizeof(str));
  conflictMAC.pstart = str;
  conflictMAC.size = OPEN_MAC_ADDR_LEN+1;
  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiVrAddrConflictLastDetectMACGet(client_handle, &vrfNameBufd, &conflictMAC);
  switch(result)
  {
    case OPEN_E_NONE:
         if (conflictMAC.size < OPEN_MAC_ADDR_LEN)
         {
           printf("\nERROR: Invalid MAC address string length received %u.",conflictMAC.size);
         }
         else
         {
           printf("\nThe MAC Address of the last detected address conflict in given VRF is 0x%02x:%02x:%02x:%02x:%02x:%02x .\n",
                  str[0], str[1], str[2], str[3], str[4], str[5]);
         }
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the Address Conflict Detection Status of the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the time in seconds since the last address conflict was detected in a VRF instance.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    *vrfName            @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void vrAddrConflictLastDetectTimeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t conflictDetectTime = 0; /* Time since the last detected conflict */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiVrAddrConflictLastDetectTimeGet(client_handle, &vrfNameBufd, &conflictDetectTime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe time since the last address conflict was detected in given VRF is %u seconds.\n", conflictDetectTime);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the time in seconds since the last address conflict was detected for the given VRF instance. (result = %d)\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Delete all net-prototype route entries.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    *vrfName      @b{(input)}  VRF name
* @param    protoId       @b{(input)}  Protocol Id
*
* @returns  none
*
* @end
*********************************************************************/
void ipNetPrototypeRoutesDelete(openapiClientHandle_t *client_handle, char *vrfName, uint32_t protoId)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;
 
  result = openapiIpNetPrototypeRoutesDelete(client_handle, &vrfNameBufd, protoId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully deleted all net-prototype route entries.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to delete all net-prototype route entries. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Unset the VRF participation on an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrIntfVrfReset(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpMapRtrIntfVrfReset(client_handle, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully reset the VRF participation on the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to reset the VRF participation on the given interface. (result = %d).\n", result);
         break;
  } 
  return;
}

/*********************************************************************
* @purpose  Set the VRF participation of an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal Interface Number
* @param    *vrfName       @b{(input)}  VRF name
* @param    *vrfSetError   @b{(output)} VRF Error message
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRtrIntfVrfSet(openapiClientHandle_t *client_handle, uint32_t ifNum, char *vrfName)
{
  OPEN_VRF_SET_ERROR_t vrfSetError = 0;
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpMapRtrIntfVrfSet(client_handle, ifNum, &vrfNameBufd, &vrfSetError);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the VRF participation on the given interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    case OPEN_E_ERROR:
         printf("\nERROR: Subnet conflict between interface IP address and IP address in the virtual router.\n");
         break;
    case OPEN_E_FULL:
         printf("\nERROR: Unable to add connected route as RTO is FULL.\n");
         break;
    case OPEN_E_INTERNAL:
         printf("\nERROR: Earlier operation on the interface is in progress. Try again later.\n");
         break;
    case OPEN_E_FAIL:
       switch(vrfSetError)
       {
         case OPEN_LOOPBACK_ERROR:
           printf("\nERROR: Loopback interface is used by unnumbered interface and so cannot be part of non-default VRF.\n");
           break;
         case OPEN_UNNUMBERED_ERROR:
           printf("\nERROR: Unnumbered interface cannot be part of a non-default VRF.\n");
           break;
         case OPEN_SYSLOG_ERROR:
           printf("\nERROR: This routing interface is being used as a syslog source interface, cannot be part of non-default VRF.\n");
           break;
         case OPEN_IPV6_ERROR:
           printf("\nERROR: This routing interface has IPv6 address config, cannot be part of non-default VRF.\n");
           break;
         default:
           printf("\nERROR: Unable to set vrf forwarding.\n");
           break;
       }
       break;
    default:
      printf("\nERROR: Bad return code trying to set the VRF participation on the given interface. (result = %d).\n", result);
      break;
  }

  return;
}

/*********************************************************************
* @purpose  Gets the configured IPv4 MTU value on the given interface.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    ifNum          @b{(input)}   Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void intfIPv4MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t ipMtu = 0; /* The configured IP MTU value */

  result = openapiIntfIPv4MtuGet(client_handle, ifNum, &ipMtu);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe configured IP MTU value on the given interface(%u) is %u.\n", ifNum, ipMtu);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the configured IP MTU value on the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the maximum IP MTU that may be set on a given interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void intfMaxIpMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t maxIpMtu = 0; /* The maximum IP MTU value */

  result = openapiIntfMaxIpMtuGet(client_handle, ifNum, &maxIpMtu);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe maximum IP MTU that can be sent on the given interface(%u) is %u.\n", ifNum, maxIpMtu);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the maximum IP MTU that can be sent on the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Gets the IP MTU value being enforced on a given interface.
*
* @param    client_handle @b{(input)}   Client handle from registration API
* @param    ifNum         @b{(input)}   Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void intfEffectiveIpMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t ipMtu = 0; /* The effective IP MTU value */

  result = openapiIntfEffectiveIpMtuGet(client_handle, ifNum, &ipMtu);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IP MTU value being enforced on the given interface(%u) is %u.\n", ifNum, ipMtu);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the IP MTU value being enforced on a given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the current IP ECMP global load balancing mode.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    lbMode         @b{(input)}  Load balancing mode in the range
*                                       [(0)OPEN_IP_ECMP_LB_MIN..(12)OPEN_IP_ECMP_LB_HASH_OUT_SIP_DIP_SPORT_DPORT]
*
* @returns  none
*
* @end
*********************************************************************/
void ipEcmpLbSet(openapiClientHandle_t *client_handle, uint32_t lbMode)
{
  open_error_t result;
  OPEN_IP_ECMP_LB_MODE_t mode = 0;

  if (mode >= OPEN_IP_ECMP_LB_MAX)
  {
    printf("\nERROR: Invalid argument passed. lbMode input valid range is [0..10..\n");
    return;
  }

  mode = lbMode;

  result = openapiIpEcmpLbSet(client_handle, mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the current IP ECMP global load balancing mode.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the current IP ECMP global load balancing mode. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Determines the current IP ECMP global load balancing mode.
*
* @param    client_handle @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipEcmpLbGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_IP_ECMP_LB_MODE_t lbMode = OPEN_IP_ECMP_LB_MIN; /* Load balancing mode */

  result = openapiIpEcmpLbGet(client_handle, &lbMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe current IP ECMP global load balancing mode is %u.\n", lbMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to determine the current IP ECMP global load balancing mode. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the current IP ECMP IPSEC SPI Hashing mode.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mode          @b{(input)}  IPSEC SPI Mode (0 for DISABLE, 1-or-greater for ENABLE)
*
* @returns  none
*
* @end
*********************************************************************/
void ipEcmpIpsecSpiSet(openapiClientHandle_t *client_handle, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t spiMode = OPEN_DISABLE;

  if (mode)
  {
    spiMode = OPEN_ENABLE;
  }

  result = openapiIpEcmpIpsecSpiSet(client_handle, spiMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the current IP ECMP IPSEC SPI Hashing mode.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the current IP ECMP IPSEC SPI Hashing mode. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Determines the current IP ECMP IPSEC SPI Mode.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipEcmpIpsecSpiGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_CONTROL_t spiMode = OPEN_DISABLE; /* Admin mode */

  result = openapiIpEcmpIpsecSpiGet(client_handle, &spiMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe current IP ECMP IPSEC SPI mode is %u.\n", spiMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to determine the current IP ECMP IPSEC SPI mode. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Get the admin mode for IP forwarding of net-directed broadcasts.
*
* @param    client_handle    @b{(input)}   Client handle from registration API
* @param    ifNum            @b{(input)}   Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipNetDirectBcastsGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* Forwarding admin mode */

  result = openapiIpNetDirectBcastsGet(client_handle, ifNum, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe admin mode for IP forwarding of net-directed broadcasts for the given interface is %u.\n", adminMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the admin mode for IP forwarding of net-directed broadcasts for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Set the unnumbered status of an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
* @param    isUnnumbered   @b{(input)}  (0 for FALSE, 1-or-greater for TRUE)
* @param    numberedIfc    @b{(input)}  Internal interface number of the numbered
*                                       interface whose IP address the unnumbered
*                                       interface is to borrow
*
* @returns  none
*
* @end
*********************************************************************/
void ipUnnumberedSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t isUnnumbered, uint32_t numberedIfc)
{
  open_error_t result;
  OPEN_BOOL_t flag = OPEN_FALSE;

  if (isUnnumbered)
  {
    flag = OPEN_TRUE;
  }

  result = openapiIpUnnumberedSet(client_handle, ifNum, flag, numberedIfc);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the unnumbered status of an interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    case OPEN_E_EXISTS:
         printf("\nERROR: The interface has IP address is already configured.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the unnumbered status of an interface. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine whether a given interface is unnumbered and if it is,
*           determine the interface whose address the unnumbered interface borrows.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
* @param    *isUnnumbered  @b{(output)} OPEN_TRUE or OPEN_FALSE
* @param    *numberedIfc   @b{(output)} Internal interface number of the numbered interface
*
* @returns  none
*
* @end
*********************************************************************/
void ipUnnumberedGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_BOOL_t isUnnumbered = OPEN_FALSE; /* OPEN_TRUE or OPEN_FALSE */
  uint32_t numberedIfc = 0; /* Internal interface number of the numbered interface */

  result = openapiIpUnnumberedGet(client_handle, ifNum, &isUnnumbered, &numberedIfc);
  switch(result)
  {
    case OPEN_E_NONE:
         if (OPEN_TRUE == isUnnumbered)
         {
           printf("\nThe given interface %u is an unnumbered and the interface whose address it borrows is %u.\n", ifNum, numberedIfc);
         }
         else
         {
           printf("\nThe given interface %u is not an unnumbered interface.\n", ifNum);
         }
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to determine whether given interface is unnumbered. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine whether an interface is configured to forward multicast packets.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipMcastsFwdModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* Forwarding admin mode */

  result = openapiIpMcastsFwdModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Multicast packets forwarding admin mode for the given interface is %u.\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to determine the multicast packets forwarding mode for given interface. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the router interface mode.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipRtrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* Admin mode */

  result = openapiIpRtrIntfModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe router interface mode is %u.\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the router interface mode for the given interface. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the mode of AutoState feature of an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipRtrIntfAutoStateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* Admin mode */

  result = openapiIpRtrIntfAutoStateModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface's autostate mode is %u.\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the autostate mode for the given interface. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Enable or disable the mode of AutoState feature of an interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
* @param    mode          @b{(input)}  Admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void ipRtrIntfAutoStateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE;

  if (mode)
  {
    adminMode = OPEN_ENABLE;
  }

  result = openapiIpRtrIntfAutoStateModeSet(client_handle, ifNum, adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the mode of AutoState feature of the given interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to enable or disable the mode of AutoState feature of an interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Determine whether a given IP interface is up for IPv4.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipRtrIntfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* Admin mode */

  result = openapiIpRtrIntfOperModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe IPv4 mode of given interface is %u.\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the IPv4 mode for the given interface. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next vlan after this vlan on which routing is enabled.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    currVlanId        @b{(input)}  The id of the VLAN
*
* @returns  none
*
* @end
*********************************************************************/
void ipVlanRtrVlanIdGetNext(openapiClientHandle_t *client_handle, uint32_t currVlanId)
{
  open_error_t result;
  uint32_t vlanId = 0;

  vlanId = currVlanId;
  result = openapiIpVlanRtrVlanIdGetNext(client_handle, &vlanId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next vlan after the given vlan on which routing is enabled is %u.\n", vlanId);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the next vlan after the given vlan on which routing is enabled. (result = %d).\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the VLAN ID corresponding to the given internal interface number.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipVlanRtrIntIfNumToVlanId(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t vlanId = 0; /* VLAN ID */

  result = openapiIpVlanRtrIntIfNumToVlanId(client_handle, ifNum, &vlanId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe VLAN ID corresponding to the given internal interface number is %u.\n", vlanId);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the VLAN ID corresponding to the given internal interface number. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface ID (not internal interface number) for a given VLAN.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void vlanIntfIdGet(openapiClientHandle_t *client_handle, uint32_t vlanId)
{
  open_error_t result;
  uint32_t ifNum = 0; /* Interface ID (not internal interface number) */

  result = openapiVlanIntfIdGet(client_handle, vlanId, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface ID for the given VLAN ID is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the interface ID for a given VLAN. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the internal interface number associated with the port-based routing interface corresponding to this internal VLAN.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  Id of the VLAN
*
* @returns  none
*
* @end
*********************************************************************/
void ipRtrInternalVlanIdToIntIfNum(openapiClientHandle_t *client_handle, uint32_t vlanId)
{
  open_error_t result;
  uint32_t ifNum = 0; /* Internal interface number of port-based routing interface */

  result = openapiIpRtrInternalVlanIdToIntIfNum(client_handle, vlanId, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe internal interface number associated with port-based routing interface corresponding to given internal VLAN is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get internal interface number associated with port-based routing interface corresponding to this internal VLAN. (result = %d)\n",result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the first valid interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapValidIntfFirstGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t ifNum = 0; /* Internal interface number */

  result = openapiIpMapValidIntfFirstGet(client_handle, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe first valid interface is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the first valid interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next valid interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    prevIfNum     @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapValidIntfNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum)
{
  open_error_t result;
  uint32_t ifNum = 0; /* Internal interface number of next valid interface */

  result = openapiIpMapValidIntfNextGet(client_handle, prevIfNum, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next valid interface of the given interface is %u.\n", ifNum);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the next valid interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the administrative mode of sending ICMP Unreachables.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    ifNum       @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapICMPUnreachablesModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* Admin mode */

  result = openapiIpMapICMPUnreachablesModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface's autostate mode is %u.\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the administrative mode of sending ICMP Unreachables. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface mode of sending ICMP Redirects.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapIfICMPRedirectsModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_DISABLE; /* Admin mode */

  result = openapiIpMapIfICMPRedirectsModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface mode of sending ICMP Redirects is %u.\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the interface mode of sending ICMP Redirects. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the bandwidth of the specified interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ifBandwidthGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t bandwidth = 0; /* bandwidth value of the specified interface. */

  result = openapiIfBandwidthGet(client_handle, ifNum, &bandwidth);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe bandwidth of the specified interface is %u.\n", bandwidth);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the bandwidth of the specified interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the bandwidth of the specified interface.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Internal interface number
* @param    bandwidth     @b{(input)}  bandwidth value
*
* @returns  none
*
* @end
*********************************************************************/
void ifBandwidthSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t bandwidth)
{
  open_error_t result;

  result = openapiIfBandwidthSet(client_handle, ifNum, bandwidth);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the bandwidth of the specified interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the bandwidth of the specified interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the bandwidth of the specified interface without making
*           any modification to the SET bandwidth.
*
* @param    client_handle @b{(input)}   Client handle from registration API
* @param    ifNum         @b{(input)}   Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ifBWGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t bandwidth = 0; /* bandwidth value of the specified interface */

  result = openapiIfBWGet(client_handle, ifNum, &bandwidth);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe bandwidth of the specified interface is %u.\n", bandwidth);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the bandwidth of the specified interface without making any modification to the SET bandwidth. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the routing max equal cost entries.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void rtrRouteMaxEqualCostEntriesGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t maxhops = 0; /* maxhops value */

  result = openapiRtrRouteMaxEqualCostEntriesGet(client_handle, &maxhops);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe routing max equal cost entries is %u.\n", maxhops);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the routing max equal cost entries. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the routing max routes entries.
*
* @param    client_handle @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void rtrRouteMaxRouteEntriesGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t maxroutes = 0; /* routing max routes entries */

  result = openapiRtrRouteMaxRouteEntriesGet(client_handle, &maxroutes);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe routing max routes entries is %u.\n", maxroutes);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the routing max routes entries. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Invokes the IPMAP API funciton to get the dampening operational values.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    ifNum          @b{(input)}   Internal Interface Number
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapIpEvtDampGetDampParams(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t flaps        = 0; /* Number of flaps after last reset */
  uint32_t penalty      = 0; /* Maximum penalty of specified interface */
  uint32_t isSuppressed = 0; /* Is interface suppressed or not */
  uint32_t reuseTime    = 0; /* Reuse time of a suppressed interface */
  uint32_t maxPenalty   = 0; /* Maximum penalty of specified interface */

  result = openapiIpMapIpEvtDampGetDampParams(client_handle, ifNum, &flaps, &penalty, &isSuppressed, &reuseTime, &maxPenalty);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe dampening operational values are: flaps[%u], penalty[%u], isSuppressed[%u], reuseTime[%u], maxPenalty[%u].\n",flaps,penalty,isSuppressed,reuseTime,maxPenalty);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the dampening operational values. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next protocol ID by iterating through all registered protocols.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    protoId        @b{(input)}  Protocol ID. Set to 0 on input to get the first value.
*                                       Set to previous value on input to get the next value.
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapProtoNextGet(openapiClientHandle_t *client_handle, uint32_t protoId)
{
  open_error_t result;

  result = openapiIpMapProtoNextGet(client_handle, &protoId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe next protocol ID is %u.\n", protoId);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the next protocol ID. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the protocol ID for a given route type.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    routeType      @b{(input)}  Route type
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapRouteTypeToProtoId(openapiClientHandle_t *client_handle, uint32_t routeType)
{
  open_error_t result;
  uint32_t protoId = 0; /* protocol ID for protocol that uses route type */

  result = openapiIpMapRouteTypeToProtoId(client_handle, routeType, &protoId);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe protocol ID is %u for the given route-type %u.\n", protoId, routeType);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the protocol ID for a given route type. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the global resilient hashing mode for ECMP trunks.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapResilientHashingModeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_CONTROL_t resHashMode = OPEN_DISABLE;

  result = openapiIpMapResilientHashingModeGet(client_handle, &resHashMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe global resilient hashing mode value for ECMP trunks is %u.\n", resHashMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the global resilient hashing mode for ECMP trunks. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Sets the global resilient hashing mode for ECMP trunks.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    mode              @b{(input)}  Admin mode (0 for DISABLE, 1-or-greater for ENABLE)
*
* @returns  none
*
* @end
*********************************************************************/
void ipMapResilientHashingModeSet(openapiClientHandle_t *client_handle, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t resHashMode = OPEN_DISABLE;

  if (mode)
  {
    resHashMode = OPEN_ENABLE;
  }

  result = openapiIpMapResilientHashingModeSet(client_handle, resHashMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe global resilient hashing mode value for ECMP trunks is %u.\n", resHashMode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the global resilient hashing mode for ECMP trunks. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Display the router discovery mode on given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscIntfAdvertiseModeShow(openapiClientHandle_t *client_handle,
                                  uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t flag = OPEN_FALSE;


  result = openapiRtrDiscAdvertiseGet(client_handle, intIfNum, &flag);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe router discovery mode on the given interface is %u. \n", flag);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get router discovery mode for given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the router discovery mode on the interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    flag             @b{(input)}   advertisement flag
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscIntfAdvertiseModeSet(openapiClientHandle_t *client_handle,
                                 uint32_t intIfNum, OPEN_BOOL_t flag)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscAdvertiseSet(client_handle, intIfNum, flag);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given router discovery mode on given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set router discovery mode for given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the advertisement address that is used as the destination
*           ip address in the advertisement packet for the given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscIntfAdvAddrShow(openapiClientHandle_t *client_handle,
                            uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  open_inet_addr_t openAddr;
  open_buffdesc ipDesc;
  char ipDescStr[32];

  memset(&openAddr, 0, sizeof(openAddr));
  result = openapiRtrDiscAdvAddrGet(client_handle, intIfNum, &openAddr);

  switch(result)
  {
    case OPEN_E_NONE:
         if (openAddr.family != OPEN_AF_INET)
         {
           printf("\nERROR: Family type value not OPEN_AF_INET. Family type value returned is %u.\n", openAddr.family);
           return;
         }
         memset(ipDescStr, 0, sizeof(ipDescStr));
         ipDesc.pstart = ipDescStr;
         ipDesc.size = sizeof(ipDescStr)-1;
         (void)openapiOpenIPtoStringGet(client_handle, openAddr, &ipDesc);
         printf("\nThe advertisement packet's destination address for the given interface is 0x%x (%s). \n", openAddr.addr.ipv4, (char *)ipDesc.pstart);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get advertisement packet's destination address for given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the advertisement address that is used as the destination
*           ip address in the advertisement packet for the given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    ipAddrStr        @b{(input)}   advertisement address in string format
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscIntfAdvAddrSet(openapiClientHandle_t *client_handle,
                           uint32_t intIfNum, char *ipAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t intIpAddr = 0;
  open_inet_addr_t openAddr;
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &intIpAddr)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string to a 32 bit integer. result = %d.\n", result);
    return;
  }

  memset(&openAddr, 0, sizeof(openAddr));
  openAddr.family = OPEN_AF_INET;
  openAddr.addr.ipv4 = intIpAddr;
  result = openapiRtrDiscAdvAddrSet(client_handle, intIfNum, openAddr);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the given advertisement packet's destination address for given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set advertisement packet's destination address for given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the maximum time allowed between sending router
*           advertisements from the given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscMaxAdvIntervalShow(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t time = 0;


  result = openapiRtrDiscMaxAdvIntervalGet(client_handle, intIfNum, &time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe max time allowed between router advertisements from the given interface is %u seconds. \n", time);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get max time allowed between router advertisements from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the maximum time allowed between sending router
*           advertisements from the given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    time             @b{(input)}   time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscMaxAdvIntervalSet(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum, uint32_t time)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscMaxAdvIntervalSet(client_handle, intIfNum, time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the max time allowed between router advertisements from given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    case OPEN_E_ERROR:
         printf("\nERROR: The maximum advertisement interval given is out of range.\n");
         break;

    case OPEN_E_FAIL:
         printf("\nERROR: The given maximum advertisement interval is less than configured minimum advertisement interval.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set max time allowed between router advertisements from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the minimum time allowed between sending router
*           advertisements from the given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscMinAdvIntervalShow(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t time = 0;


  result = openapiRtrDiscMinAdvIntervalGet(client_handle, intIfNum, &time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe min time allowed between router advertisements from the given interface is %u seconds. \n", time);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get min time allowed between router advertisements from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the minimum time allowed between sending router
*           advertisements from the given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    time             @b{(input)}   time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscMinAdvIntervalSet(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum, uint32_t time)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscMinAdvIntervalSet(client_handle, intIfNum, time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the min time allowed between router advertisements from given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    case OPEN_E_ERROR:
         printf("\nERROR: The minimum advertisement interval given is out of range.\n");
         break;

    case OPEN_E_FAIL:
         printf("\nERROR: The given minimum advertisement interval is greater than configured maximum advertisement interval.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set min time allowed between router advertisements from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the default value of the minimum time allowed between
*           sending router advertisements from the given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscMinAdvIntervalDefaultShow(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t time = 0;


  result = openapiRtrDiscMinAdvIntervalGet(client_handle, intIfNum, &time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe default value of the min time allowed between router advertisements from the given interface is %u seconds. \n", time);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the default value of min time allowed between router advertisements from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Revert the minimum time allowed between sending router
*           advertisements from the given interface to its default value
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscMinAdvIntervalRevert(openapiClientHandle_t *client_handle,
                                 uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscMinAdvIntervalRevert(client_handle, intIfNum);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully reset the min time allowed between router advertisements to its default value.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to reset min time allowed between router advertisements to its default value. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the value of lifetime field of router advertisement sent
*           from the given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscAdvLifetimeShow(openapiClientHandle_t *client_handle,
                            uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t time = 0;


  result = openapiRtrDiscAdvLifetimeGet(client_handle, intIfNum, &time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe value of lifetime field of router advertisement sent from the given interface is %u seconds. \n", time);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the value of lifetime field of router advertisement sent from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the value of lifetime field of router advertisement sent
*           from the given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    time             @b{(input)}   time in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscAdvLifetimeSet(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum, uint32_t time)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscAdvLifetimeSet(client_handle, intIfNum, time);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the value of lifetime field of router advertisement sent from given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    case OPEN_E_ERROR:
         printf("\nERROR: The given lifetime value is greater than the max lifetime value.\n");
         break;

    case OPEN_E_FAIL:
         printf("\nERROR: The given lifetime value is less than the maximum advertisement interval.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the value of lifetime field of router advertisement sent from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the default value of the lifetime field of router
*           advertisements semt from the given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscAdvLifetimeDefaultShow(openapiClientHandle_t *client_handle,
                                   uint32_t intIfNum)
{
  open_error_t result  = OPEN_E_FAIL;
  uint32_t defLifetime = 0;


  result = openapiRtrDiscAdvLifetimeDefaultGet(client_handle, intIfNum, &defLifetime);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe default value of the lifetime field of router advertisement sent from the given interface is %u seconds. \n", defLifetime);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the default value of the lifetime field of router advertisement sent from given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Revert the advertisement lifetime to its default value on
*           the given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscAdvLifetimeRevert(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscAdvLifetimeRevert(client_handle, intIfNum);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully reset the advertisement lifetime to its default value on to the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to reset the advertisement lifetime to its default value on the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the preferability of the address as a default router
*           address, relative to other router addresses on the same subnet.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscPreferenceLevelShow(openapiClientHandle_t *client_handle,
                                uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  int32_t prefLevel   = 0;


  result = openapiRtrDiscPreferenceLevelGet(client_handle, intIfNum, &prefLevel);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe value of the preferability of the address as a default router address for the given interface is %d. \n", prefLevel);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the value of the preferability of the address as a default router address for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the preferability of the address as a default router
*           address, relative to other router addresses on the same subnet.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    prefLevel        @b{(input)}   preference level
*
* @returns  none
*
* @end
*********************************************************************/
void rtrDiscPreferenceLevelSet(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum, int32_t prefLevel)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrDiscPreferenceLevelSet(client_handle, intIfNum, prefLevel);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the preferability value of the address as a default router address for the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the preferability value of the address as a default router address for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the Grat ARP mode for the given unnumbered interface
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpUnnumberedGratArpModeShow(openapiClientHandle_t *client_handle,
                                    uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t isEnabled = OPEN_FALSE;


  result = openapiRtrIpUnnumberedGratArpGet(client_handle, intIfNum, &isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:

         printf("\nThe value of the Grat ARP mode for the given interface is %u. \n", isEnabled);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the value of the Grat ARP mode for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Set the behavior of gratuitous ARP received on unnumbered interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    mode             @b{(input)}   Grat ARP mode (0 for FALSE, 1 for TRUE)
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpUnnumberedGratArpSet(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t isEnabled = OPEN_FALSE;


  if (mode)
  {
    isEnabled = OPEN_TRUE;
  }

  result = openapiRtrIpUnnumberedGratArpSet(client_handle, intIfNum, isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Grat ARP mode for the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the Grat ARP mode for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Set the behavior of gratuitous ARP received on given interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    mode             @b{(input)}   Grat ARP mode (0 for FALSE, 1-or-greater for TRUE)
*
* @returns  none
*
* @end
*********************************************************************/
void rtrGratArpSet(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t isEnabled = OPEN_DISABLE;


  if (mode)
  {
    isEnabled = OPEN_ENABLE;
  }

  result = openapiRtrGratArpSet(client_handle, intIfNum, isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Grat ARP mode for the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the Grat ARP mode for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Enable or disable the ARP dynamic entry renew mode
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Grat ARP mode (0 for FALSE, 1-or-greater for TRUE)
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpDynamicRenewSet(openapiClientHandle_t *client_handle,
                             uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t isEnabled = OPEN_DISABLE;


  if (mode)
  {
    isEnabled = OPEN_ENABLE;
  }

  result = openapiRtrIpArpDynamicRenewSet(client_handle, isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the ARP dynamic entry renew mode.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the ARP dynamic entry renew mode. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the maximum number of entries in the ARP cache.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    cacheSize        @b{(input)}   Number of entries in the ARP cache
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpCacheSizeSet(openapiClientHandle_t *client_handle,
                          uint32_t cacheSize)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrIpArpCacheSizeSet(client_handle, cacheSize);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the maximum number of entries in the ARP cache.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the maximum number of entries in the ARP cache. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the ARP request max retries count.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    count            @b{(input)}   ARP request max retries count
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpRetriesSet(openapiClientHandle_t *client_handle,
                        uint32_t count)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrIpArpRetriesSet(client_handle, count);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the ARP request max retries count.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the ARP request max retries count. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the ARP dynamic entry renew mode
*
* @param    client_handle  @b{(input)} client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpDynamicRenewGet(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t isEnabled = OPEN_DISABLE;


  result = openapiRtrIpArpDynamicRenewGet(client_handle, &isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe value of the ARP dynamic entry renew mode is %u. \n", isEnabled);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the ARP dynamic entry renew mode value. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Enable or disable the local proxy ARP on an interface
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   internal interface number
* @param    mode             @b{(input)}   local proxy ARP mode (0 for FALSE, 1 for TRUE)
*
* @returns  none
*
* @end
*********************************************************************/
void rtrLocalProxyArpModeSet(openapiClientHandle_t *client_handle,
                             uint32_t intIfNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t isEnabled = OPEN_DISABLE;


  if (mode)
  {
    isEnabled = OPEN_ENABLE;
  }

  result = openapiRtrLocalProxyArpModeSet(client_handle, intIfNum, isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the local proxy ARP on the given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the local proxy ARP on the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays whether local proxy ARP is enabled on an interface
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrLocalProxyArpModeShow(openapiClientHandle_t *client_handle,
                              uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t isEnabled = OPEN_DISABLE;


  result = openapiRtrLocalProxyArpModeGet(client_handle, intIfNum, &isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:

         printf("\nThe value of the local proxy ARP mode for the given interface is %u. \n", isEnabled);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the value of local proxy ARP mode for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays whether proxy ARP is enabled on an interface
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIntfIpProxyArpModeShow(openapiClientHandle_t *client_handle,
                               uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t isEnabled = OPEN_DISABLE;


  result = openapiRtrIntfIpProxyArpModeGet(client_handle, intIfNum, &isEnabled);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe value of the proxy ARP mode for the given interface is %u. \n", isEnabled);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the value of proxy ARP mode for the given interface. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the ARP request response timeout
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    timeout          @b{(input)}   timeout value
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpRespTimeSet(openapiClientHandle_t *client_handle,
                         uint32_t timeout)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrIpArpRespTimeSet(client_handle, timeout);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the ARP request responce timeout.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the ARP request responce timeout. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Sets the ARP entry ageout time
*
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    timeout          @b{(input)}   timeout value
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpAgeTimeSet(openapiClientHandle_t *client_handle,
                        uint32_t timeout)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrIpArpAgeTimeSet(client_handle, timeout);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the ARP entry ageout time.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the ARP entry ageout time. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Displays the ARP cache statistics for the given VRF instance
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vrfName        @b{(input)} VRF instance name
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpCacheStatsShow(openapiClientHandle_t *client_handle,
                            char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_arpCacheStats_t arpStats;


  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(&arpStats, 0, sizeof(arpStats));
  result = openapiIpArpCacheStatsGet(client_handle, &vrfNameBufd, &arpStats);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing the ARP cache statistics for the given VRF instance - %s:",vrfNameStr);
         printf("\nCurrent overall count               %u",arpStats.cacheCurrent);
         printf("\nPeak overall count                  %u",arpStats.cachePeak);
         printf("\nMaximum (configured) overall count  %u",arpStats.cacheMax);
         printf("\nCurrent static entry count          %u",arpStats.staticCurrent);
         printf("\nMaximum allowed static entry count  %u",arpStats.staticMax);
         printf("\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the ARP stats for the given VRF instance. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Purges a specific dynamic/gateway entry from the ARP cache
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vrfName        @b{(input)} VRF instance name
* @param    ipAddrStr      @b{(input)} IP address in string format
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpEntryPurge(openapiClientHandle_t *client_handle,
                        char *vrfName, char *ipAddrStr, uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiRtrIpArpEntryPurge(client_handle, &vrfNameBufd, ipAddr, intIfNum);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully purged the specific dynamic/gateway entry for the given VRF instance.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to purge a specific dynamic/gateway entry for the given VRF instance. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Clears the ARP cache of all dynamic/gateway entries for given VRF instance
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vrfName        @b{(input)} VRF instance name
* @param    flag           @b{(input)} 0 for FALSE else TRUE
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpArpCacheClear(openapiClientHandle_t *client_handle,
                        char *vrfName, uint32_t flag)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_BOOL_t gatewayFlag = OPEN_FALSE;


  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (flag)
  {
    gatewayFlag = OPEN_TRUE;
  }

  result = openapiRtrIpArpCacheClear(client_handle, &vrfNameBufd, gatewayFlag);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared the ARP cache of all dynamic/gateway entries for the given VRF instance.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the ARP cache of all dynamic/gateway entries for the given VRF instance. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Clear the switch/management ARP entries
*
* @param    client_handle  @b{(input)} client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void rtrArpSwitchClear(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_FAIL;


  result = openapiRtrArpSwitchClear(client_handle);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully cleared the switch/management ARP entries.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the switch/management ARP entries. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Delete a static ARP entry of a given VRF instance
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vrfName        @b{(input)} VRF instance name
* @param    ipAddrStr      @b{(input)} IP address in string format
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpMapStaticArpDelete(openapiClientHandle_t *client_handle,
                             char *vrfName, char *ipAddrStr, uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];


  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpMapStaticArpDelete(client_handle, &vrfNameBufd, ipAddr, intIfNum);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully deleted a static ARP entry of the given VRF instance.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    case OPEN_E_NOT_FOUND:
         printf("\nERROR: Entry not found.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to delete a static ARP entry for the given VRF instance. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose   Add a static ARP entry of a given VRF instance
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vrfName        @b{(input)} VRF instance name
* @param    ipAddrStr      @b{(input)} IP address in string format
* @param    intIfNum       @b{(input)} internal interface number
* @param    macAddrStr     @b{(input)} MAC address in ascii string format 'xx:xx:xx:xx:xx:xx'
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpMapStaticArpAdd(openapiClientHandle_t *client_handle,
                          char *vrfName, char *ipAddrStr, uint32_t intIfNum, char *macAddrStr)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];
  open_buffdesc macAddrBufd;
  char macStr[OPEN_MAC_ADDR_LEN+1];


  memset(str, 0, sizeof(str));
  strncpy(str, ipAddrStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert IP address. (result = %d).\n", result);
    return;
  }

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(macStr, 0, sizeof(macStr));
  if (OPEN_FALSE == rtrUtilConvertMac((unsigned char *)macAddrStr, (unsigned char *)&macStr))
  {
    printf("\nERROR: Invalid MAC address string. Expected in ascii string format xx:xx:xx:xx:xx:xx.\n");
    return;
  }

  macAddrBufd.pstart = macStr;
  macAddrBufd.size = strlen(macStr) + 1;

  result = openapiIpMapStaticArpAdd(client_handle, &vrfNameBufd, ipAddr, intIfNum, &macAddrBufd);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully added given static ARP entry for the given VRF instance.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    case OPEN_E_FULL:
         printf("\nERROR: Maximum static ARP entries already configured.\n");
         break;

    case OPEN_E_EXISTS:
         printf("\nERROR: Address is assigned to a local interface.\n");
         break;

    case OPEN_E_INTERNAL:
         printf("\nERROR: If there is not yet a router interface in the same subnet as the target. Config is applied in this case.\n");
         break;

    case OPEN_E_UNAVAIL:
         printf("\nERROR: Address can neither be network directed broadcast nor subnet address.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to add a static ARP entry for the given VRF instance. (result = %d).\n", result);
         break;
  }

  return;
}

/*********************************************************************
* @purpose   Get a list of all static ARP entries currently configured.
*
* @param    client_handle  @b{(input)} client handle from registration API
* @param    vrfName        @b{(input)} VRF instance name
* @param    ipAddrStr      @b{(input)} IP address in string format
* @param    intIfNum       @b{(input)} internal interface number
* @param    macAddrStr     @b{(input)} MAC address in string format
*
* @returns  none
*
* @end
*********************************************************************/
void rtrIpMapStaticArpGetAll(openapiClientHandle_t *client_handle,
                             char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_ARP_STATIC_ALL_t openArpEntries;
  char ipAddrStr[80] = {0};
  uint32_t i=0;


  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(&openArpEntries, 0, sizeof(openArpEntries));
  result = openapiRtrIpMapStaticArpGetAll(client_handle, &vrfNameBufd, &openArpEntries);

  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing is the list of all static ARP entries currently configured for given VRF %s:\n",vrfNameStr);
         printf("\nCount of entries received is %u.\n",openArpEntries.count);
         for(i=0; i<OPEN_FD_RTR_MAX_STATIC_ARP_ENTRIES; i++)
         {
           if ((openArpEntries.arpEntries[i].ipAddr.family != OPEN_AF_INET) ||
                (openArpEntries.arpEntries[i].ipAddr.addr.ipv4 == 0))
           {
             break;
           }
           memset(ipAddrStr, 0x0, sizeof(ipAddrStr));
           ipAddressFormat(&(openArpEntries.arpEntries[i].ipAddr), ipAddrStr);
           printf("\nARP entry index %u:",i+1);
           printf("\n\tARP entry type bitmask.......... 0x%x\n",openArpEntries.arpEntries[i].type);
           printf("\t\tLOCAL: %s, RESOLVED: %s, GATEWAY: %s, NET_DIR_BCAST: %s\n",
            (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_LOCAL)?   "y" : "n",  (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_RESOLVED)?      "y" : "n",
            (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_GATEWAY)? "y" : "n",  (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_NET_DIR_BCAST)? "y" : "n");
           printf("\t\tSTATIC: %s, DISCARD_WHEN_SOURCE: %s, KERNEL: %s, UNNUMBERED: %s\n",
            (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_STATIC)?  "y" : "n",  (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_DISCARD_WHEN_SOURCE)? "y" : "n",
            (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_KERNEL)?  "y" : "n",  (openArpEntries.arpEntries[i].type & OPEN_ARP_TYPE_UNNUMBERED)?          "y" : "n");
           printf("\n\tARP entry age in seconds........ %u",    openArpEntries.arpEntries[i].age);
           printf("\n\tIP Address of ARP entry......... %s",    ipAddrStr);
           printf("\n\tVRF ID of the ARP entry......... %u",    openArpEntries.arpEntries[i].vrfId);
           printf("\n\tMAC address..................... %02x:%02x:%02x:%02x:%02x:%02x",
                                                                openArpEntries.arpEntries[i].macAddr[0],openArpEntries.arpEntries[i].macAddr[1],
                                                                openArpEntries.arpEntries[i].macAddr[2],openArpEntries.arpEntries[i].macAddr[3],
                                                                openArpEntries.arpEntries[i].macAddr[4],openArpEntries.arpEntries[i].macAddr[5]);
           printf("\n\tVLAN ID of the ARP entry........ %u",    openArpEntries.arpEntries[i].vlanId);
           printf("\n\tOutgoing internal interface..... %u",    openArpEntries.arpEntries[i].intIfNum);
           printf("\n\tHits............................ %u",    openArpEntries.arpEntries[i].hits);
           printf("\n");
         }
         printf("\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to list of all static ARP entries currently configured for the given VRF instance. (result = %d).\n", result);
         break;
  }

  return;
}

/*****************************************************************************
* @purpose  Set or reset the IPv6 address mode on a given router interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrMode         @b{(input)}   Address mode
* @param    setFlag          @b{(input)}   Set flag (OPEN_TRUE for set or OPEN_FALSE for reset)
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*******************************************************************************/
void rtrIntfIpv6AddressModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_INTF_IP6_ADDR_MODE_t addrMode, OPEN_BOOL_t setFlag)
{
  open_error_t result;
  
  result = openapiRtrIntfIpv6AddressModeSet(client_handle, ifNum, addrMode, setFlag);
  switch(result)
  {
    case OPEN_E_NONE:
         if (setFlag == OPEN_TRUE)
         { 
           printf("\nSuccessfully set the given IPv6 address mode for the given interface.\n");
         }
         else
         {
           printf("\nSuccessfully reset the given IPv6 address mode for the given interface.\n");
         }
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    case OPEN_E_UNAVAIL:
         printf("\nERROR: IPv6 package is not supported.\n");
         break;
    default:
         if (setFlag == OPEN_TRUE)
         {
           printf("\nERROR: Bad return code trying to set the given IPv6 address mode for the given interface. (result = %d)\n", result);
         }
         else
         {
           printf("\nERROR: Bad return code trying to reset the given IPv6 address mode for the given interface. (result = %d)\n", result);
         }
         break;
  }

  return;
}

/*********************************************************************
* @purpose  Add an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrType         @b{(input)}   Address Type
* @param    IPStr            @b{(input)}   IP address and prefix length. This argument is not
*                                          applicable when addrType of DHCP is used.
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    extArg           @b{(input)}   If used for Address Family IPv4:
*                                          Enable the DHCP client to specify the unique client
*                                          Id Option 61.
*                                          If used for Address Family IPv6:
*                                          Enable use of eui-64 Interface Identifier.
*
* @returns  Void
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpAddrAdd(openapiClientHandle_t *client_handle,
                      OPEN_AF_t af,uint32_t intf,
                      OPEN_INTF_IP_ADDR_TYPE_t addrType,
                      char *IPStr, uint32_t pfxLen,
                      OPEN_CONTROL_t extArg)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((addrType == OPEN_IP_ADDR_TYPE_STATIC_PRIMARY) || (addrType == OPEN_IP_ADDR_TYPE_STATIC_SECONDARY))
  {
    if (af == OPEN_AF_INET)
    {
      if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
      {
        printf("Bad return code trying to convert IP address. (result = %d)\n", result);
        return;
      }
      printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

      ipAddr.pfxLen = pfxLen;
      ipAddr.ipAddr.family = OPEN_AF_INET;
    }
    else /* (af == OPEN_AF_INET6) */
    {
      if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
      {
        printf("Bad return code trying to convert IP.\n");
      }

      ipAddr.pfxLen = pfxLen;
      ipAddr.ipAddr.family = OPEN_AF_INET6;
    }
  }
  
  if ((result = openapiRtrIntfIpAddrAdd(client_handle, af, intf, addrType, &ipAddr, extArg)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add Interface IP. (result = %d)\n", result);
  }
  else
  {
    printf("Interface IP added successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrType         @b{(input)}   Address Type
* @param    IPStr            @b{(input)}   IP address and prefix length. This argument is not
*                                          applicable when addrType of DHCP is used.  
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    extArg           @b{(input)}   If used for Address Family IPv4:
*                                          Enable the DHCP client to specify the unique client 
*                                          Id Option 61.
*                                          If used for Address Family IPv6:
*                                          Enable use of eui-64 Interface Identifier.
*
* @returns  none
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpAddrDel(openapiClientHandle_t *client_handle,
                      OPEN_AF_t af,uint32_t intf,
                      OPEN_INTF_IP_ADDR_TYPE_t addrType,
                      char *IPStr, uint32_t pfxLen,
                      OPEN_CONTROL_t extArg)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((addrType == OPEN_IP_ADDR_TYPE_STATIC_PRIMARY) || (addrType == OPEN_IP_ADDR_TYPE_STATIC_SECONDARY))
  {
    if (af == OPEN_AF_INET)
    {
      if (strcmp(IPStr, "all") != 0)
      {
        if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
        {
          printf("Bad return code trying to convert IP address. (result = %d)\n", result);
          return;
        }
        printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

        ipAddr.pfxLen = pfxLen;
        ipAddr.ipAddr.family = OPEN_AF_INET;
      }
    }
    else /* (af == OPEN_AF_INET6) */
    {
      if (strcmp(IPStr, "all") != 0)
      {
        if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
        {
          printf("Bad return code trying to convert IP.\n");
          return;
        }

        ipAddr.pfxLen = pfxLen;
        ipAddr.ipAddr.family = OPEN_AF_INET6;
      }
    }
  }

  if (strcmp(IPStr, "all") == 0)
  {
    result = openapiRtrIntfIpAddrDel(client_handle, af, intf, addrType, 0, extArg);
  }
  else
  {
    result = openapiRtrIntfIpAddrDel(client_handle, af, intf, addrType, &ipAddr, extArg);
  }

  
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete Interface IP. (result = %d)\n", result);
  }
  else
  {
    printf("Interface IP deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Routing Admin mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    routingMode     @b{(input)}   Routing mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrAdminModeSet(openapiClientHandle_t *client_handle,
                     OPEN_AF_t af, OPEN_CONTROL_t routingMode)
{
  open_error_t result;
  
  if ((result = openapiRtrAdminModeSet(client_handle, af, routingMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure Global Routing Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global Routing Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Add a test static ARP entry {192.168.1.1 00:11:22:33:44:55}.
*
* @param    client_handle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void arpEntryAdd(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char IPStr[] = "192.168.1.1";
  char mac_address[] = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55};
  open_buffdesc mac_addr;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (inet_pton(AF_INET, IPStr, (void*)&(ipAddr.addr.ipv4)) < 0)
  {
    printf("Bad return code trying to convert IP.\n");
    return;
  }

  ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
  ipAddr.family = OPEN_AF_INET;

  mac_addr.pstart = mac_address;
  mac_addr.size = 6;

  if ((result = openapiArpEntryAdd(client_handle, ipAddr, &mac_addr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add ARP entry. (result = %d)\n", result);
  }
  else
  {
    printf("ARP entry added successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete a test static ARP entry {192.168.1.1 00:11:22:33:44:55}.
*
* @param    client_handle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void arpEntryDel(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char IPStr[] = "192.168.1.1";

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (inet_pton(AF_INET, IPStr, (void*)&(ipAddr.addr.ipv4)) < 0)
  {
    printf("Bad return code trying to convert IP.\n");
    return;
  }

  ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
  ipAddr.family = OPEN_AF_INET;

  if ((result = openapiArpEntryDel(client_handle, ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete ARP entry. (result = %d)\n", result);
  }
  else
  {
    printf("ARP entry deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Display all static ARP entries.
*
* @param    client_handle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void arpEntryShow(openapiClientHandle_t *client_handle)
{
  uint32_t intf = 0;
  open_inet_addr_t ipAddr;
  char ipAddrStr[24];
  char macStr[18];
  open_buffdesc mac_addr;
  uint32_t ipv4Addr = 0;

  memset(&ipAddr, 0, sizeof(ipAddr));

  memset(macStr, 0, 18);
  mac_addr.pstart = macStr;
  mac_addr.size = 18;

  while(openapiArpEntryNextGet(client_handle, &intf, &ipAddr, &mac_addr) == OPEN_E_NONE)
  {
    /* Note: openapiArpEntryNextGet violates OpEN convention by returning
       the ipv4 address in network byte order. Thus, we call inet_ntop
       without need for conversion from host byte order to network byte
       order using htonl. */

    ipv4Addr = htonl(ipAddr.addr.ipv4);
    if (inet_ntop(AF_INET, &ipv4Addr, ipAddrStr, sizeof(ipAddrStr)) == 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    printf("IP: %s, MAC: %s Intf:%d \n", ipAddrStr, macStr, intf);
  }

  return;
}

/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   Redirects mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ipRedirectsModeSet(openapiClientHandle_t *client_handle,
                        OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiIpRedirectsModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP Redirects Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP Redirects Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP ICMP Echo Reply mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   Redirects mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ipICMPEchoReplyModeSet(openapiClientHandle_t *client_handle, 
                            OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiIpICMPEchoReplyModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP ICMP Echo Reply Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP ICMP Echo Reply Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Helper mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   IP Helper mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ipHelperModeSet(openapiClientHandle_t *client_handle, 
                     OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiIpHelperModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP Helper Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ipHelperAddressAdd(openapiClientHandle_t *client_handle, 
                        char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  /* Validate IP address */
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }
              
  ipAddr.family = OPEN_AF_INET;
 
  if ((result = openapiIpHelperAddressAdd(client_handle, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ipHelperAddressDel(openapiClientHandle_t *client_handle,
                        char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.family = OPEN_AF_INET;
 
  if ((result = openapiIpHelperAddressDel(client_handle, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Configures Router Interface MTU.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf            @b{(input)}   Router Interface
* @param    mtu             @b{(input)}   MTU
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfMTUSet(openapiClientHandle_t *client_handle, 
                   OPEN_AF_t af,uint32_t intf, uint32_t mtu)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfMTUSet(client_handle, af, intf, mtu)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface MTU. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface MTU configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables forwarding of Network-directed broadcast on a Router Interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   IP Network-directed broadcast mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpNetDirBroadcastModeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpNetDirBroadcastModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Net Directed BCast Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Net Directed BCast Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Proxy Arp mode on a Router Interface.
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    intf                  @b{(input)}   Router Interface
* @param    mode                  @b{(input)}   IP Proxy Arp mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpProxyArpModeSet(openapiClientHandle_t *client_handle, 
                              uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpProxyArpModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Proxy ARP Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Proxy ARP Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   Redirects mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpRefirectsModeSet(openapiClientHandle_t *client_handle, 
                               uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpRedirectsModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Redirects Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Redirects Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables generation of IP Destination Unreachable messages
*           on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   Unreachable mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpDestUnreachableModeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpDestUnreachableModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Destination Unreachable Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Unreachable Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Routing mode on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   Routing mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfRtrAdminModeSet(openapiClientHandle_t *client_handle, 
                            uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfRtrAdminModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Routing Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Routing Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpHelperAddressAdd(openapiClientHandle_t *client_handle, 
                               uint32_t intf, char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.family = OPEN_AF_INET;
 
  if ((result = openapiRtrIntfIpHelperAddressAdd(client_handle, intf, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpHelperAddressDel(openapiClientHandle_t *client_handle, 
                               uint32_t intf, char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.family = OPEN_AF_INET;
 
 if ((result = openapiRtrIntfIpHelperAddressDel(client_handle, intf, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpHelperDiscardAdd(openapiClientHandle_t *client_handle, 
                               uint32_t intf, uint32_t udpPort)
{
  open_error_t result;
 
  if ((result = openapiRtrIntfIpHelperDiscardAdd(client_handle, intf, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add IP Helper Discard. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Discard configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  non
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpHelperDiscardDel(openapiClientHandle_t *client_handle, 
                               uint32_t intf, uint32_t udpPort)
{
  open_error_t result;

  if ((result = openapiRtrIntfIpHelperDiscardDel(client_handle, intf, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP Helper Discard. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Discard deleted successfully \n");
  }
  return;
}


/*********************************************************************
* @purpose  Add an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    IPStr            @b{(input)}   IP address and prefix length. 
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    NextHopStr       @b{(input)}   Next Hop
* @param    rtPref           @b{(input)}   Route preference
* @param    intf             @b{(input)}   Router Interface
* @param    mpls             @b{(input)}   List of MPLS labels
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIpRouteAdd(openapiClientHandle_t *client_handle,
                   OPEN_AF_t af, char *IPStr, uint32_t pfxLen,
                   char *NextHopStr, uint32_t rtPref,
                   uint32_t intf, OPEN_MPLS_LABELS_t *mpls)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_inet_addr_t nextHop;
  open_inet_addr_t *nHptr;

  open_buffdesc ipBuffdesc, nextHopBuffdesc;
  char str[40], str1[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(str1, 0, sizeof(str1));
  strncpy(str1, NextHopStr, sizeof(str1) - 1);
  nextHopBuffdesc.pstart = str1;
  nextHopBuffdesc.size = strlen(str1) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (af == OPEN_AF_INET)
  {
    if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
    printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
      intf = 0;
    }
    else if ((strcmp(NextHopStr, "0") == 0) || (strcmp(NextHopStr, "0.0.0.0") == 0))
    {
      nHptr = (void*)0;
    }
    else
    {
      if ((result = openapiInetAddrGet(client_handle, &nextHopBuffdesc, &nextHop)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to convert IP address. (result = %d)\n", result);
        return;
      }
      nextHop.family = OPEN_AF_INET;

      nHptr = &nextHop;
    }

    printf("Next Hop = %u\n", nextHop.addr.ipv4);
  }
  else /* (af == OPEN_AF_INET6) */
  {
    if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
    }

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET6;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
      intf = 0;
    }
    else if (strcmp(NextHopStr, "0") == 0)
    {
      nHptr = (void*)0;
    }
    else
    {
      memset(&nextHop, 0, sizeof(nextHop));
      
      if (inet_pton(AF_INET6, NextHopStr, (void*)&(nextHop.addr.ipv6)) < 0)
      {
        printf("Bad return code trying to convert Next Hop.\n");
      }
      nextHop.family = OPEN_AF_INET6;
      
      nHptr = &nextHop;
    }
  }

  result = openapiIpRouteAdd(client_handle, af, &ipAddr, nHptr, rtPref, intf, mpls);  

  if ((result == OPEN_E_NONE) || (result == OPEN_E_NOT_FOUND))
  {
    printf("IP route added successfully \n");
  }
  else
  {
    printf("Bad return code trying to add IP route. (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  Delete an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    IPStr            @b{(input)}   IP address and prefix length. 
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    NextHopStr       @b{(input)}   Next Hop
* @param    intf             @b{(input)}   Router Interface
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIpRouteDel(openapiClientHandle_t *client_handle,
                   OPEN_AF_t af, char *IPStr, uint32_t pfxLen,
                   char *NextHopStr, uint32_t intf, OPEN_MPLS_LABELS_t *mpls)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_inet_addr_t nextHop;
  open_inet_addr_t *nHptr;

  open_buffdesc ipBuffdesc, nextHopBuffdesc;
  char str[40], str1[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(str1, 0, sizeof(str1));
  strncpy(str1, NextHopStr, sizeof(str1) - 1);
  nextHopBuffdesc.pstart = str1;
  nextHopBuffdesc.size = strlen(str1) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (af == OPEN_AF_INET)
  {
    if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
    printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
      intf = 0;
    }
    else if ((strcmp(NextHopStr, "0") == 0) || (strcmp(NextHopStr, "0.0.0.0") == 0))
    {
      nHptr = (void*)0;
    }
    else
    {
      if ((result = openapiInetAddrGet(client_handle, &nextHopBuffdesc, &nextHop)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to convert IP address. (result = %d)\n", result);
        return;
      }
      nextHop.family = OPEN_AF_INET;

      nHptr = &nextHop;
    }
  
      printf("Next Hop = %u\n", nextHop.addr.ipv4);
  }
  else /* (af == OPEN_AF_INET6) */
  {
    if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
    }

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET6;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
      intf = 0;
    }
    else if (strcmp(NextHopStr, "0") == 0)
    {
      nHptr = (void*)0;
    }
    else
    {
      memset(&nextHop, 0, sizeof(nextHop));
      
      if (inet_pton(AF_INET6, NextHopStr, (void*)&(nextHop.addr.ipv6)) < 0)
      {
        printf("Bad return code trying to convert Next Hop.\n");
      }
      nextHop.family = OPEN_AF_INET6;
      
      nHptr = &nextHop;
    }
  }
  
  if ((result = openapiIpRouteDel(client_handle, af, &ipAddr, nHptr, intf, mpls)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP route. (result = %d)\n", result);
  }
  else
  {
    printf("IP route deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Create a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrVlanIntfCreate(openapiClientHandle_t *client_handle,
                       uint32_t vlanId)
{
  open_error_t result;
  
  if ((result = openapiRtrVlanIntfCreate(client_handle, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create VLAN Routing Interface %u. (result = %d)\n", vlanId, result);
  }
  else
  {
    printf("VLAN Routing Interface created successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrVlanIntfDelete(openapiClientHandle_t *client_handle,
                       uint32_t vlanId)
{
  open_error_t result;
  
  if ((result = openapiRtrVlanIntfDelete(client_handle, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete VLAN Routing Interface %u. (result = %d)\n",  vlanId, result);
  }
  else
  {
    printf("VLAN Routing Interface deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Display interface number associated with a VLAN Routing interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void rtrVlanIfNumShow(openapiClientHandle_t *client_handle,
                      uint32_t vlanId)
{
  open_error_t result;
  uint32_t ifNum;
  open_buffdesc ifName;
  char ifNameStr[OPEN_INTF_NAME_MAX_LENGTH];

  ifName.pstart = ifNameStr;
  ifName.size = sizeof(ifNameStr);
  
  if ((result = openapiRtrVlanIntfIfNumGet(client_handle, vlanId, &ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Interface Number for VLAN %u. (result = %d)\n",  vlanId, result);
  }
  else
  {
    printf("VLAN Routing Interface Number for VLAN %u is %d\n", vlanId, ifNum);

    result = openapiIntfNameGet(client_handle, ifNum, &ifName);
    if (result == OPEN_E_NONE)
    {
      printf("Logical interface name for interface %u is %s\n", ifNum, ifNameStr);
    }
    else
    {
      printf("Bad return code trying to get interface name for ifNum %u. (result = %d)\n",  ifNum, result);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Create a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4=1 or IPv6=2)
* @param    ifNum            @b{(input)}   Router Interface
* @param    dstIpStr         @b{(input)}   Dest IP address.
* @param    srcIpStr         @b{(input)}   Source IP address.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionCreate(openapiClientHandle_t *client_handle, OPEN_AF_t af,
                         uint32_t ifNum, char *dstIpStr , char *srcIpStr)
{
  uint32_t sessId;
  open_error_t rc;
  openBfdEndpoint_t ep;

  memset(&ep, 0x0, sizeof(ep));
  /* Setup the BFD session parameters */
  ep.compId = 0;
  ep.vrfId = 0;
  ep.intIfNum = ifNum;
  ep.dstIpAddr.family = af;
  ep.srcIpAddr.family = af;
  ep.type = OPEN_BFD_TUNNEL_TYPE_UDP; /* Need BFD session using IP-UDP encap */

  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    if (inet_pton(AF_INET, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    ep.dstIpAddr.addr.ipv4 = ntohl(ep.dstIpAddr.addr.ipv4);
    ep.srcIpAddr.addr.ipv4 = ntohl(ep.srcIpAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }

    if (inet_pton(AF_INET6, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
  }

  rc = openapiBfdModeSet(OPEN_BFD_ENABLE);
  if (rc != OPEN_E_NONE)
  {
    printf("Failed to enable BFD %d\n", rc);
  }

  rc = openapiBfdSessionCreate(client_handle, &ep, &sessId);
 
  if (rc == OPEN_E_NONE)
  {
    printf("Successfully created BFD session %d\n", sessId);
  }
  else
  {
    printf("Failed to create BFD session %d \n", rc);
  }
}

/*********************************************************************
* @purpose  Find a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4=1 or IPv6=2)
* @param    dstIpStr         @b{(input)}   Dest IP address.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionFind(openapiClientHandle_t *client_handle, OPEN_AF_t af, char *dstIpStr)
{
  uint32_t sessIdOut;
  open_error_t rc;
  open_inet_addr_t ipAddr;

  memset(&ipAddr, 0x0, sizeof(ipAddr));
  ipAddr.family = af;

  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ipAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ipAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
  }

  /* look up the session based on destination */

  rc = openapiBfdSessionFind(client_handle, ipAddr, &sessIdOut);
  if (rc != OPEN_E_NONE)
  {
    printf("Failed to find BFD session rc=%d\n", rc);
  }
  else
  {
    printf("Successfully found BFD session %d\n", sessIdOut);
  }
}

/*********************************************************************
* @purpose  Iterate the VRF names
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void iterateVRFs(openapiClientHandle_t *client_handle)
{
  char vrfName[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  char vrfNameNext[OPEN_VRF_MAX_NAME_LEN + 1];
  open_error_t ret;
  int count = 0;

  open_buffdesc vrfNameBuf;
  open_buffdesc vrfNameNextBuf;

  vrfNameBuf.pstart = vrfName;
  vrfNameBuf.size = 1;

  vrfNameNextBuf.pstart = vrfNameNext;
  vrfNameNextBuf.size = OPEN_VRF_MAX_NAME_LEN + 1;

  printf("List of VRFs\n");
  printf("************\n\n");
  while (1) 
  {
    ret = openapiVrfNameNextGet(client_handle, &vrfNameBuf, &vrfNameNextBuf);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_PARAM) 
      {
        printf("%s: invalid argument passed to openapiVrfNameNextGet\n",
               __FUNCTION__);
      }
      break;
    }
    printf("%d: %s\n", ++count, vrfNameNext);
    strncpy(vrfName, vrfNameNext, sizeof(vrfName));
    vrfNameBuf.size = strlen(vrfName) + 1;
    vrfNameNextBuf.size = OPEN_VRF_MAX_NAME_LEN + 1;
  }
  printf("\n%d VRFs found\n", count);
}

/*********************************************************************
* @purpose  Delete a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    ifNum            @b{(input)}   Router Interface
* @param    dstIpStr         @b{(input)}   Dest IP address.
* @param    srcIpStr         @b{(input)}   Source IP address.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionDelete(openapiClientHandle_t *client_handle, OPEN_AF_t af,
                         uint32_t ifNum, char *dstIpStr , char *srcIpStr)
{
  uint32_t sessId;
  open_error_t rc;
  openBfdEndpoint_t ep;

  memset(&ep, 0x0, sizeof(ep));
  /* Setup the BFD session parameters */
  ep.compId = 0;
  ep.vrfId = 0;
  ep.intIfNum = ifNum;
  ep.dstIpAddr.family = af;
  ep.srcIpAddr.family = af;
  ep.type = OPEN_BFD_TUNNEL_TYPE_UDP; /* Need BFD session using IP-UDP encap */

  sessId = OPEN_BFD_SESSION_ID_INVALID;
  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    if (inet_pton(AF_INET, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    ep.dstIpAddr.addr.ipv4 = ntohl(ep.dstIpAddr.addr.ipv4);
    ep.srcIpAddr.addr.ipv4 = ntohl(ep.srcIpAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }

    if (inet_pton(AF_INET6, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
  }

  rc = openapiBfdSessionDelete(client_handle, &ep, sessId);

  if (rc == OPEN_E_NONE)
  {
    printf("Successfully deleted BFD session\n");
  }
  else
  {
    printf("Failed to delete BFD session %d\n", rc);
  }
}

/*********************************************************************
* @purpose  Show list of current BFD sessions
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionShow(openapiClientHandle_t *client_handle)
{
  uint32_t sessId;
  uint32_t count = 0;
  open_error_t err;
  openBfdSessionInfo_t info;
  openBfdSessionStats_t stats;

  char *pStrInfo_bfd_header_1 = "SessionId:%u  State:%s  Local Disc:%u  Remote Disc:%u\r\n";
  char *pStrInfo_bfd_header_2 = "     Destination IP:%s \r\n";
  char *pStrInfo_bfd_header_3 = "     Source IP:%s \r\n";
  char *pStrInfo_bfd_header_4 = "     Interface:%u  VrfId:%u  Uptime:%u secs\r\n";
  char *pStrInfo_bfd_header_5 = "     Min Transmit Interval:%u milliseconds\r\n";
  char *pStrInfo_bfd_header_6 = "     Min Receive Interval:%u milliseconds\r\n";
  char *pStrInfo_bfd_header_7 = "     Detection interval multiplier:%u \r\n";
  char *pStrInfo_bfd_header_8 = "     In Packets:%u  Out Packets:%u  Dropped Packets:%u\r\n";
  char *pStrInfo_bfd_header_9 = "     Echo In Packets:%u  Echo Out Packets:%u \r\n";
  char  strSrcIp[80] = {0};
  char  strDstIp[80] = {0};
  char  strState[80] = {0};

  /* Get the BFD session info */

  for (sessId=1; sessId < 96; sessId++)
  {
    memset(&info, 0x0, sizeof(info));
    memset(&stats, 0x0, sizeof(stats));
    err = openapiBfdSessionInfoGet(client_handle, sessId, &info);
    if (err == OPEN_E_NONE)
    {
      count++;
      openapiBfdSessionStatsGet(client_handle, sessId, &stats);

      ipAddressFormat(&info.key.srcIpAddr, strSrcIp);
      ipAddressFormat(&info.key.dstIpAddr, strDstIp);

      snprintf(strState, sizeof(strState), "%s",
               (info.state == OPEN_BFD_SESSION_STATE_UP) ? "UP" : "DOWN");

      printf(pStrInfo_bfd_header_1, sessId, strState, info.localDiscr, info.remoteDiscr);
      printf(pStrInfo_bfd_header_2, strDstIp);
      printf(pStrInfo_bfd_header_3, strSrcIp);
      printf(pStrInfo_bfd_header_4, info.key.intIfNum, info.key.vrfId, info.upTime );
      printf(pStrInfo_bfd_header_5, info.localMinTx);
      printf(pStrInfo_bfd_header_6, info.localMinRx);
      printf(pStrInfo_bfd_header_7, info.localDetectMult);
      printf(pStrInfo_bfd_header_8, stats.inPkts, stats.outPkts, stats.dropPkts );
      printf(pStrInfo_bfd_header_9, stats.echoInPkts, stats.echoOutPkts);
    }
  }

  if (count == 0)
  {
    printf("No BFD sessions found.\r\n");
  }
}

/*********************************************************************
* @purpose  This function sanity checks all Routing Config OpEN APIs. All
*           Routing Config OpEN APIs are called with possible NULL and
*           invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void routingOpENAPIsTestSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t invalidControlMode = 10;
  uint32_t invalidAddressFamily = 10;
  uint32_t invalidIntfAddressType = 10;
  uint32_t intf = 1, udpPort = 10, mtu = 1500;
  open_buffdesc buffDesc;
  char str[32];
  open_inet_pfx_t ipAddrPfx;
  open_inet_addr_t ipAddr;

  printf("Testing Routing Configuration OpEN APIs sanity:\n\n");

  /* openapiRtrAdminModeSet() */
  printf("Testing openapiRtrAdminModeSet():\n");

  result = openapiRtrAdminModeSet(NULL, OPEN_AF_INET, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrAdminModeSet(clientHandle, OPEN_AF_NONE, OPEN_ENABLE);

  printf("Address family NONE :(result = %d)\n", result);

  result = openapiRtrAdminModeSet(clientHandle, invalidAddressFamily, OPEN_ENABLE);

  printf("Invalid Address family :(result = %d)\n", result);

  result = openapiRtrAdminModeSet(clientHandle, OPEN_AF_NONE, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrAdminModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpAddrAdd() */
  printf("Testing openapiRtrIntfIpAddrAdd():\n");

  memset(&ipAddr, 0, sizeof(open_inet_pfx_t));

  result = openapiRtrIntfIpAddrAdd(NULL, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_NONE, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Address family NONE :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, invalidAddressFamily, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address family :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_INET, intf,
                                   invalidIntfAddressType, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address Type:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   NULL, OPEN_ENABLE);

  printf("NULL IP Address:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, invalidControlMode);

  printf("NULL Ext Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpAddrAdd() sanity successful\n\n");


  /* openapiRtrIntfIpAddrDel() */
  printf("Testing openapiRtrIntfIpAddrDel():\n");

  memset(&ipAddr, 0, sizeof(open_inet_pfx_t));

  result = openapiRtrIntfIpAddrDel(NULL, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_NONE, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Address family NONE :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, invalidAddressFamily, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address family :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_INET, intf,
                                   invalidIntfAddressType, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address Type:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   NULL, OPEN_ENABLE);

  printf("NULL IP Address:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, invalidControlMode);

  printf("NULL Ext Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpAddrDel() sanity successful\n\n");


  /* openapiArpEntryAdd() */
  printf("Testing openapiArpEntryAdd():\n");

  result = openapiArpEntryAdd(NULL, ipAddr, &buffDesc);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiArpEntryAdd(clientHandle, ipAddr, NULL);
  printf("NULL Mac Address:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  buffDesc.size = 6;

  result = openapiArpEntryAdd(clientHandle, ipAddr, &buffDesc);
  printf("NULL Mac Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiArpEntryAdd(clientHandle, ipAddr, &buffDesc);
  printf("NULL Mac Address buffer length:(result = %d)\n", result);

  printf("openapiArpEntryAdd() sanity successful\n\n");

  /* openapiArpEntryDel() */
  printf("Testing openapiArpEntryDel():\n");

  result = openapiArpEntryDel(NULL, ipAddr);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiArpEntryDel() sanity successful\n\n");


  /* openapiArpEntryNextGet() */
  printf("Testing openapiArpEntryNextGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = 18;

  result = openapiArpEntryNextGet(NULL, &intf, &ipAddr, &buffDesc);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiArpEntryNextGet(clientHandle, NULL, &ipAddr, &buffDesc);

  printf("NULL Intf:(result = %d)\n", result);

  result = openapiArpEntryNextGet(clientHandle, &intf, NULL, &buffDesc);

  printf("NULL ipAddr:(result = %d)\n", result);

  result = openapiArpEntryNextGet(clientHandle, &intf, &ipAddr, NULL);

  printf("NULL Mac Address:(result = %d)\n", result);


  buffDesc.pstart = NULL;
  buffDesc.size = 18;

  result = openapiArpEntryNextGet(clientHandle, &intf, &ipAddr, &buffDesc);
  printf("NULL Mac Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiArpEntryNextGet(clientHandle, &intf, &ipAddr, &buffDesc);
  printf("NULL Mac Address buffer length:(result = %d)\n", result);

  printf("openapiArpEntryNextGet() sanity successful\n\n");


  /* openapiIpRedirectsModeSet() */
  printf("Testing openapiIpRedirectsModeSet():\n");

  result = openapiIpRedirectsModeSet(NULL, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpRedirectsModeSet(clientHandle, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiIpRedirectsModeSet() sanity successful\n\n");

  /* openapiIpICMPEchoReplyModeSet() */
  printf("Testing openapiIpICMPEchoReplyModeSet():\n");

  result = openapiIpICMPEchoReplyModeSet(NULL, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpICMPEchoReplyModeSet(clientHandle, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiIpICMPEchoReplyModeSet() sanity successful\n\n");

  /* openapiIpHelperModeSet() */
  printf("Testing openapiIpHelperModeSet():\n");

  result = openapiIpHelperModeSet(NULL, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpHelperModeSet(clientHandle, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiIpHelperModeSet() sanity successful\n\n");

  /* openapiIpHelperAddressAdd() */
  printf("Testing openapiIpHelperAddressAdd():\n");

  result = openapiIpHelperAddressAdd(NULL, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiIpHelperAddressAdd() sanity successful\n\n");

  /* openapiIpHelperAddressDel() */
  printf("Testing openapiIpHelperAddressDel():\n");

  result = openapiIpHelperAddressDel(NULL, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiIpHelperAddressDel() sanity successful\n\n");

  /* openapiRtrIntfIpHelperAddressAdd() */
  printf("Testing openapiRtrIntfIpHelperAddressAdd():\n");

  result = openapiRtrIntfIpHelperAddressAdd(NULL, intf, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperAddressAdd() sanity successful\n\n");

  /* openapiRtrIntfIpHelperAddressDel() */
  printf("Testing openapiRtrIntfIpHelperAddressDel():\n");

  result = openapiRtrIntfIpHelperAddressDel(NULL, intf, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperAddressDel() sanity successful\n\n");

  /* openapiRtrIntfIpHelperDiscardAdd() */
  printf("Testing openapiRtrIntfIpHelperDiscardAdd():\n");

  result = openapiRtrIntfIpHelperDiscardAdd(NULL, intf, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperDiscardAdd() sanity successful\n\n");

  /* openapiRtrIntfIpHelperDiscardDel() */
  printf("Testing openapiRtrIntfIpHelperDiscardDel():\n");

  result = openapiRtrIntfIpHelperDiscardDel(NULL, intf, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperDiscardDel() sanity successful\n\n");

  /* openapiRtrIntfMTUSet() */
  printf("Testing openapiRtrIntfMTUSet():\n");

  result = openapiRtrIntfMTUSet(NULL, OPEN_AF_INET, intf, mtu);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfMTUSet(clientHandle, invalidAddressFamily, intf, mtu);

  printf("Invalid Address family :(result = %d)\n", result);

  printf("openapiRtrIntfMTUSet() sanity successful\n\n");

  /* openapiRtrIntfIpNetDirBroadcastModeSet() */
  printf("Testing openapiRtrIntfIpNetDirBroadcastModeSet():\n");

  result = openapiRtrIntfIpNetDirBroadcastModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpNetDirBroadcastModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpNetDirBroadcastModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpProxyArpModeSet() */
  printf("Testing openapiRtrIntfIpProxyArpModeSet():\n");

  result = openapiRtrIntfIpProxyArpModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpProxyArpModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpProxyArpModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpRedirectsModeSet() */
  printf("Testing openapiRtrIntfIpRedirectsModeSet():\n");

  result = openapiRtrIntfIpRedirectsModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpRedirectsModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpRedirectsModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpDestUnreachableModeSet() */
  printf("Testing openapiRtrIntfIpDestUnreachableModeSet():\n");

  result = openapiRtrIntfIpDestUnreachableModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpDestUnreachableModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpDestUnreachableModeSet() sanity successful\n\n");

  /* openapiRtrIntfRtrAdminModeSet() */
  printf("Testing openapiRtrIntfRtrAdminModeSet():\n");

  result = openapiRtrIntfRtrAdminModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfRtrAdminModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfRtrAdminModeSet() sanity successful\n\n");

  /* openapiIpRouteAdd() */
  printf("Testing openapiIpRouteAdd():\n");

  result = openapiIpRouteAdd(NULL, OPEN_AF_INET, &ipAddrPfx, &ipAddr, 0, intf, NULL);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpRouteAdd(clientHandle, invalidAddressFamily, &ipAddrPfx, &ipAddr, 0, intf, NULL);

  printf("Invalid Address Family:(result = %d)\n", result);

  result = openapiIpRouteAdd(clientHandle, OPEN_AF_INET, NULL, &ipAddr, 0, intf, NULL);

  printf("NULL IP Address Pfx:(result = %d)\n", result);

  result = openapiIpRouteAdd(clientHandle, OPEN_AF_INET, &ipAddrPfx, NULL, 0, intf, NULL);

  printf("NULL Next Hop:(result = %d)\n", result);

  printf("openapiIpRouteAdd() sanity successful\n\n");

  /* openapiIpRouteDel() */
  printf("Testing openapiIpRouteDel():\n");

  result = openapiIpRouteDel(NULL, OPEN_AF_INET, &ipAddrPfx, &ipAddr, intf, NULL);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpRouteDel(clientHandle, invalidAddressFamily, &ipAddrPfx, &ipAddr, intf, NULL);

  printf("Invalid Address Family:(result = %d)\n", result);

  result = openapiIpRouteDel(clientHandle, OPEN_AF_INET, NULL, &ipAddr, intf, NULL);

  printf("NULL IP Address Pfx:(result = %d)\n", result);

  result = openapiIpRouteDel(clientHandle, OPEN_AF_INET, &ipAddrPfx, NULL, intf, NULL);

  printf("NULL Next Hop:(result = %d)\n", result);

  printf("openapiIpRouteDel() sanity successful\n\n");

  /* openapiRtrVlanIntfCreate */
  result = openapiRtrVlanIntfCreate(NULL, 2);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrVlanIntfCreate() sanity successful\n\n");

  /* openapiRtrVlanIntfIfNumGet */
  result = openapiRtrVlanIntfIfNumGet(NULL, 2, &intf);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrVlanIntfIfNumGet(clientHandle, 2, NULL);

  printf("NULL ifnum: (result = %d)\n", result);

  result = openapiRtrVlanIntfIfNumGet(clientHandle, 3, &intf);

  printf("Invalid VLAN ifnum: (result = %d)\n", result);

  printf("openapiRtrVlanIntfIfNumGet() sanity successful\n\n");

  /* openapiRtrVlanIntfDelete */
  result = openapiRtrVlanIntfDelete(NULL, 2);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrVlanIntfDelete() sanity successful\n\n");

  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Routing Configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  int rc;
  int32_t prefLevel;
  uint32_t ifNum, testNum, addrFamily, time;
  uint32_t addrType, pfxLen, extArg, mode;
  uint32_t udpPort, mtu, pref, vlanId, value;
  OPEN_MPLS_LABELS_t mpls;

  if (argc < 2 )
  {
    printRoutingAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  if (strcmp("API", argv[1]) == 0)
  {
    if (argc < 3 )
    {
      printRoutingAppMenu();
      exit(1);
    }

    testNum = atoi(argv[2]);
   
    switch (testNum)
    {
      case 1:
        if (argc != 9)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrFamily = atoi(argv[4]);
        addrType = atoi(argv[5]);
        pfxLen = atoi(argv[7]);
        extArg = atoi(argv[8]);

        rtrIntfIpAddrAdd(&clientHandle, addrFamily, ifNum, addrType, argv[6], pfxLen, extArg);
        break;

      case 2:
        if (argc != 9)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrFamily = atoi(argv[4]);
        addrType = atoi(argv[5]);
        pfxLen = atoi(argv[7]);
        extArg = atoi(argv[8]);
        
        rtrIntfIpAddrDel(&clientHandle, addrFamily, ifNum, addrType, argv[6], pfxLen, extArg);
        break;

      case 3:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrAdminModeSet(&clientHandle, addrFamily, mode);
        break;

      case 4:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        arpEntryAdd(&clientHandle);
        break;

      case 5:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        arpEntryDel(&clientHandle);
        break;

      case 6:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        arpEntryShow(&clientHandle);
        break;

      case 7:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);

        ipRedirectsModeSet(&clientHandle, mode);
        break;

      case 8:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);

        ipICMPEchoReplyModeSet(&clientHandle, mode);
        break;

      case 9:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);
        
        ipHelperModeSet(&clientHandle, mode);
        break;

      case 10:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        udpPort = atoi(argv[4]);
        
        ipHelperAddressAdd(&clientHandle, argv[3], udpPort);
        break;

      case 11:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        udpPort = atoi(argv[4]);
        
        ipHelperAddressDel(&clientHandle, argv[3], udpPort);
        break;

      case 12:
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrFamily = atoi(argv[4]);
        mtu = atoi(argv[5]);

        rtrIntfMTUSet(&clientHandle, addrFamily, ifNum, mtu);
        break;

      case 13:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrIntfIpNetDirBroadcastModeSet(&clientHandle, ifNum, mode);
        break;

      case 14:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrIntfIpProxyArpModeSet(&clientHandle, ifNum, mode);
        break;

      case 15:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrIntfIpRefirectsModeSet(&clientHandle, ifNum, mode);
        break;

      case 16:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrIntfIpDestUnreachableModeSet(&clientHandle, ifNum, mode);
        break;

      case 17:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrIntfRtrAdminModeSet(&clientHandle, ifNum, mode);
        break;

      case 18:
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[5]);

        rtrIntfIpHelperAddressAdd(&clientHandle, ifNum, argv[4], udpPort);
        break;

      case 19:
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[5]);

        rtrIntfIpHelperAddressDel(&clientHandle, ifNum, argv[4], udpPort);
        break;

      case 20:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[4]);

        rtrIntfIpHelperDiscardAdd(&clientHandle, ifNum, udpPort);
        break;

      case 21:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[4]);

        rtrIntfIpHelperDiscardDel(&clientHandle, ifNum, udpPort);
        break;

      case 22:
        if ((argc < 9) || (argc > 12))
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        pfxLen = atoi(argv[5]);
        pref = atoi(argv[7]);
        ifNum = atoi(argv[8]);

        memset(&mpls, 0, sizeof(mpls));  
        /* Read MPLS labels */
        if (argc >= 10)
        {
          mpls.label[0] = atoi(argv[9]);
        }
        if (argc >= 11)
        {     
          mpls.label[1] = atoi(argv[10]);
        }     
        if (argc == 12)
        {
          mpls.label[2] = atoi(argv[11]);
        }

        rtrIpRouteAdd(&clientHandle, addrFamily, argv[4], 
                      pfxLen, argv[6], pref, ifNum, &mpls);

        break;

      case 23:
        if ((argc < 8) || (argc > 11))
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        pfxLen = atoi(argv[5]);
        ifNum = atoi(argv[7]);

        memset(&mpls, 0, sizeof(mpls));  
        /* Read MPLS labels */
        if (argc >= 9)
        {
          mpls.label[0] = atoi(argv[8]);
        }
        if (argc >= 10)
        {     
          mpls.label[1] = atoi(argv[9]);
        }     
        if (argc == 11)
        {
          mpls.label[2] = atoi(argv[10]);
        }

        rtrIpRouteDel(&clientHandle, addrFamily, argv[4], 
                      pfxLen, argv[6], ifNum, &mpls);

        break;

      case 24:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);

        rtrVlanIntfCreate(&clientHandle, vlanId);
        break;

      case 25:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);

        rtrVlanIntfDelete(&clientHandle, vlanId);
        break;

      case 26:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        routingOpENAPIsTestSanity(&clientHandle);
        break;

      case 27:
      case 28:
        if (argc != 7)
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        ifNum = atoi(argv[4]);
        if (testNum == 27)
          rtrBfdSessionCreate(&clientHandle, addrFamily, ifNum, argv[5], argv[6]);
        else
          rtrBfdSessionDelete(&clientHandle, addrFamily, ifNum, argv[5], argv[6]);

        break;

      case 29:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        rtrBfdSessionShow(&clientHandle);
        break;
 
      case 30:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        rtrBfdSessionFind(&clientHandle, addrFamily, argv[4]);
 
        break;

      case 31:
        iterateVRFs(&clientHandle);
        break;

      case 32:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);

        rtrVlanIfNumShow(&clientHandle, vlanId);
        break;

      case 34: /* "Test API 34: Get the router discovery mode on given interface: routing_example API 34 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscIntfAdvertiseModeShow(&clientHandle, ifNum);
        break;

      case 35: /* "Test API 35: Enable/Disable the router discovery mode on given interface: routing_example API 35 <interface> <mode>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode  = atoi(argv[4]);

        rtrDiscIntfAdvertiseModeSet(&clientHandle, ifNum, mode);
        break;

      case 36: /* "Test API 36: Get advertisement packet's destination address: routing_example API 36 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscIntfAdvAddrShow(&clientHandle, ifNum);
        break;

      case 37: /* "Test API 37: Set advertisement packet's destination address: routing_example API 37 <interface> <address>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscIntfAdvAddrSet(&clientHandle, ifNum, argv[4]);
        break;

      case 38: /* "Test API 38: Get max time allowed between router advertisements from interface: routing_example API 38 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscMaxAdvIntervalShow(&clientHandle, ifNum);
        break;

      case 39: /* "Test API 39: Set max time allowed between router advertisements from interface: routing_example API 39 <interface> <time>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        time  = atoi(argv[4]);

        rtrDiscMaxAdvIntervalSet(&clientHandle, ifNum, time);
        break;

      case 40: /* "Test API 40: Get min time allowed between router advertisements from interface: routing_example API 40 <interface>\n"); */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscMinAdvIntervalShow(&clientHandle, ifNum);
        break;

      case 41: /* "Test API 41: Set min time allowed between router advertisements from interface: routing_example API 41 <interface> <time>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        time  = atoi(argv[4]);

        rtrDiscMinAdvIntervalSet(&clientHandle, ifNum, time);
        break;

      case 42: /* "Test API 42: Get the default minimum advertisement interval on interface: routing_example API 42 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscMinAdvIntervalDefaultShow(&clientHandle, ifNum);
        break;

      case 43: /* "Test API 43: Revert MinAdvertisementInterval to its default value on interface: routing_example API 43 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscMinAdvIntervalRevert(&clientHandle, ifNum);
        break;

      case 44: /* "Test API 44: Get lifetime field value of router advertisement sent from given interface: routing_example API 44 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscAdvLifetimeShow(&clientHandle, ifNum);
        break;

      case 45: /* "Test API 45: Set lifetime field value of router advertisement sent from given interface: routing_example API 45 <interface> <time>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        time  = atoi(argv[4]);

        rtrDiscAdvLifetimeSet(&clientHandle, ifNum, time);
        break;

      case 46: /* "Test API 46: Get the default value of lifetime field of router advertisement sent from interface: routing_example API 46 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscAdvLifetimeDefaultShow(&clientHandle, ifNum);
        break;

      case 47: /* "Test API 47: Revert AdvertisementLifetime to its default value on interface: routing_example API 47 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscAdvLifetimeRevert(&clientHandle, ifNum);
        break;

      case 48: /* "Test API 48: Get the preferability of the address as a default router address: routing_example API 48 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrDiscPreferenceLevelShow(&clientHandle, ifNum);
        break;

      case 49: /* "Test API 49: Set the preferability of the address as a default router address: routing_example API 49 <interface> <pref-level>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        prefLevel = atoi(argv[4]);

        rtrDiscPreferenceLevelSet(&clientHandle, ifNum, prefLevel);
        break;

      case 50: /* "Test API 50: Get the Grat ARP mode on given unnumbered interface: routing_example API 50 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrIpUnnumberedGratArpModeShow(&clientHandle, ifNum);
        break;

      case 51: /* "Test API 51: Set the Grat ARP mode on given unnumbered interface: routing_example API 51 <interface> <grat-arp-mode>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrIpUnnumberedGratArpSet(&clientHandle, ifNum, mode);
        break;

      case 52: /* "Test API 52: Enable or disable gratitious ARP on an interface: routing_example API 52 <interface> <grat-arp-mode>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrGratArpSet(&clientHandle, ifNum, mode);
        break;

      case 53: /* "Test API 53: Enable or disable the ARP dynamic entry renew mode: routing_example API 53 <mode>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);

        rtrIpArpDynamicRenewSet(&clientHandle, mode);
        break;

      case 54: /* "Test API 54: Displays the ARP dynamic entry renew mode: routing_example API 54\n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        rtrIpArpDynamicRenewGet(&clientHandle);
        break;

      case 55: /* "Test API 55: Sets the maximum number of entries in the ARP cache: routing_example API 55 <cache-size>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);

        rtrIpArpCacheSizeSet(&clientHandle, value);
        break;

      case 56: /* "Test API 56: Sets the ARP request max retries count: routing_example API 56 <count>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);

        rtrIpArpRetriesSet(&clientHandle, value);
        break;

      case 57: /* "Test API 57: Displays whether local proxy ARP is enabled on an interface: routing_example API 57 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrLocalProxyArpModeShow(&clientHandle, ifNum);
        break;

      case 58: /* "Test API 58: Enable or disable the local proxy ARP on an interface: routing_example API 58 <interface> <mode>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);

        rtrLocalProxyArpModeSet(&clientHandle, ifNum, mode);
        break;

      case 59: /* "Test API 59: Displays whether proxy ARP is enabled on an interface: routing_example API 59 <interface>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        rtrIntfIpProxyArpModeShow(&clientHandle, ifNum);
        break;

      case 60: /* "Test API 60: Sets the ARP request response timeout: routing_example API 60 <timeout>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);

        rtrIpArpRespTimeSet(&clientHandle, value);
        break;

      case 61: /* "Test API 61: Sets the ARP entry ageout time: routing_example API 61 <timeout>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);

        rtrIpArpAgeTimeSet(&clientHandle, value);
        break;

      case 62: /* "Test API 62: Retrieve various ARP cache statistics: routing_example API 62 <vrfName>>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }

        rtrIpArpCacheStatsShow(&clientHandle, argv[3]);
        break;

      case 63: /* "Test API 63: Purges a specific dynamic/gateway entry from the ARP cache: routing_example API 63 <vrfName> <ipAddress> <interface>\n" */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[5]);

        rtrIpArpEntryPurge(&clientHandle, argv[3], argv[4], ifNum);
        break;

      case 64: /* "Test API 64: Clears the ARP cache of all dynamic/gateway entries: routing_example API 64 <vrfName> <gateway-flag>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[4]);

        rtrIpArpCacheClear(&clientHandle, argv[3], mode);
        break;

      case 65: /* "Test API 65: Delete a static ARP entry of a given VRF instance: routing_example API 65 <vrfName> <ipAddress> <interface>\n" */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[5]);

        rtrIpMapStaticArpDelete(&clientHandle, argv[3], argv[4], ifNum);
        break;

      case 66: /* "Test API 66: Add a static ARP entry of a given VRF instance: routing_example API 66 <vrfName> <ipAddress> <interface> <mac-addr-str>\n" */
        if (argc != 7)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[5]);

        rtrIpMapStaticArpAdd(&clientHandle, argv[3], argv[4], ifNum, argv[6]);
        break;

      case 67: /* "Test API 67: Display all static ARP entries currently configured:routing_example API 67 <vrfName>>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }

        rtrIpMapStaticArpGetAll(&clientHandle, argv[3]);
        break;

     case 68: /* "Test API  68: Gets the configured IP MTU value on the given interface: routing_example API 68 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        intfIPv4MtuGet(&clientHandle, ifNum);
        break;

      case 69: /* Test API  69: Get the maximum IP MTU that may be set on a given interface: routing_example API 69 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        intfMaxIpMtuGet(&clientHandle, ifNum);
        break;

      case 70: /* "Test API  70: Gets the IP MTU value being enforced on a given interface: routing_example API 70 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        intfEffectiveIpMtuGet(&clientHandle, ifNum);
        break;

      case 71: /* "Test API  71: Sets the current IP ECMP global load balancing mode: routing_example API 71 <lbMode>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);

        ipEcmpLbSet(&clientHandle, mode);
        break;

      case 72: /* "Test API  72: Determines the current IP ECMP global load balancing mode: routing_example API 72 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        ipEcmpLbGet(&clientHandle);
        break;

      case 73: /* "Test API  73: Sets the current IP ECMP IPSEC SPI Hashing mode: routing_example API 73 <spiMode>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);

        ipEcmpIpsecSpiSet(&clientHandle, mode);
        break;

      case 74: /* "Test API  74: Determines the current IP ECMP IPSEC SPI Mode: routing_example API 74 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        ipEcmpIpsecSpiGet(&clientHandle);
        break;

      case 75: /* "Test API  75: Gets the admin mode for IP forwarding of net-directed broadcasts: routing_example API 75 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipNetDirectBcastsGet(&clientHandle, ifNum);
        break;

      case 76: /* "Test API  76: Set the unnumbered status of an interface: routing_example API 76 <ifNum> <isUnnumbered> <numberedIfc>\n" */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode  = atoi(argv[4]);
        value = atoi(argv[5]);

        ipUnnumberedSet(&clientHandle, ifNum, mode, value);
        break;

      case 77: /* "Test API  77: Determine whether given interface is unnumbered and if so, determine the interface whose address it borrows: routing_example API 77 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
      
        ipUnnumberedGet(&clientHandle, ifNum);
        break;

      case 78: /* "Test API  78: Determine whether an interface is configured to forward multicast packets: routing_example API 78 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipMcastsFwdModeGet(&clientHandle, ifNum);
        break;

      case 79: /* "Test API  79: Gets the router interface mode: routing_example API 79 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipRtrIntfModeGet(&clientHandle, ifNum);
        break;

      case 80: /* "Test API  80: Get the mode of AutoState feature of an interface: routing_example API 80 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipRtrIntfAutoStateModeGet(&clientHandle, ifNum);
        break;

      case 81: /* "Test API  81: Enable or disable the mode of AutoState feature of an interface: routing_example API 81 <ifNum> <mode>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode  = atoi(argv[4]);

        ipRtrIntfAutoStateModeSet(&clientHandle, ifNum, mode);
        break;

      case 82: /* "Test API  82: Determine whether a given IP interface is up for IPv4: routing_example API 82 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipRtrIntfOperModeGet(&clientHandle, ifNum);
        break;

      case 83: /* "Test API  83: Get the next vlan after this vlan on which routing is enabled: routing_example API 83 <vlanId>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);

        ipVlanRtrVlanIdGetNext(&clientHandle, vlanId);
        break;

      case 84: /* "Test API  84: Get the VLAN ID corresponding to the given internal interface number: routing_example API 84 <ifNum> \n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipVlanRtrIntIfNumToVlanId(&clientHandle, ifNum);
        break;

      case 85: /* "Test API  85: Get the interface ID for a given VLAN: routing_example API 85 <vlanId> \n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);

        vlanIntfIdGet(&clientHandle, value);
        break;

      case 86: /* "Test API  86: Get the internal interface number associated with port-based routing interface corresponding to this internal VLAN: routing_example API 86 <vlanId>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);

        ipRtrInternalVlanIdToIntIfNum(&clientHandle, value);
        break;

      case 87: /* "Test API  87: Get the first valid interface: routing_example API 87 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipMapValidIntfFirstGet(&clientHandle);
        break;

      case 88: /* "Test API  88: Get the next valid interface: routing_example API 88 <prevIfNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipMapValidIntfNextGet(&clientHandle, ifNum);
        break;

      case 89: /* "Test API  89: Get the administrative mode of sending ICMP Unreachables: routing_example API 89 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipMapICMPUnreachablesModeGet(&clientHandle, ifNum);
        break;

      case 90: /* "Test API  90: Get the interface mode of sending ICMP Redirects: routing_example API 90 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipMapIfICMPRedirectsModeGet(&clientHandle, ifNum);
        break;

      case 91: /* "Test API  91: Get the bandwidth of the specified interface: routing_example API 91 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ifBandwidthGet(&clientHandle, ifNum);
        break;

      case 92: /* "Test API  92: Set the bandwidth of the specified interface: routing_example API 92 <ifNum><bandwidth>\n" */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        value = atoi(argv[4]);
        ifBandwidthSet(&clientHandle, ifNum, value);
        break;

      case 93: /* "Test API  93: Get the bandwidth of the specified interface without making any modification to the SET bandwidth: routing_example API 93 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ifBWGet(&clientHandle, ifNum);
        break;

      case 94: /* "Test API  94: Get the routing max equal cost entries: routing_example API 94 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        rtrRouteMaxEqualCostEntriesGet(&clientHandle);
        break;

      case 95: /* "Test API  95: Get the routing max routes entries: routing_example API 95 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        rtrRouteMaxRouteEntriesGet(&clientHandle);
        break;

      case 96: /* "Test API  96: Invokes the IPMAP API funciton to get the dampening operational values: routing_example API 96 <ifNum>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);

        ipMapIpEvtDampGetDampParams(&clientHandle, ifNum);
        break;

      case 97: /* "Test API  97: Get the next protocol ID by iterating through all registered protocols: routing_example API 97 <protoId>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);
        ipMapProtoNextGet(&clientHandle, value);
        break;

      case 98: /* "Test API  98: Get the protocol ID for a given route type: routing_example API 98 <routeType>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[3]);
        ipMapRouteTypeToProtoId(&clientHandle, value);
        break;

      case 99: /* "Test API  99: Gets the global resilient hashing mode for ECMP trunks: routing_example API 99 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        ipMapResilientHashingModeGet(&clientHandle);
        break;

    case 100: /* "Test API 100: Sets the global resilient hashing mode for ECMP trunks: routing_example API 100 <resHashMode>\n" */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);
        ipMapResilientHashingModeSet(&clientHandle, mode);
        break;

      case 101: /* Test 101: Get the packets forwarded by CPU: routing_example API 101 */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        ipMapForwardingStatsGet(&clientHandle);
        break;

      case 102: /* Test 102: Get the IP packets count received by IP-Stack: routing_example API 102 */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipMapIpPacketsReceivedGet(&clientHandle);
        break;

      case 103: /* Test 103: Get source IP interface from Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps: routing_example API 103 */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        sourceInterfacesGet(&clientHandle);
        break;

      case 104: /* Test 104: Set source IP interface for Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps: routing_example API 104 <ifNum> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        sourceInterfacesSet(&clientHandle, ifNum);
        break;

      case 105: /* "Test 105: Get the configured IPv4 Management interface: routing_example API 105 \n" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipManagementInterfaceGet(&clientHandle);
        break;

      case 106: /* Test 106: Set the IPv4 Management interface: routing_example API 106 <ifNum> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipManagementInterfaceSet(&clientHandle, ifNum);
        break;

      case 107: /* Test 107: Set the IPv4 Management interface and enable dhcp or configure static IP on the management interface: routing_example API 107 <ifNum> <dhcpEnabled> <ipAddr> <subnetMask> */
        if (argc != 7)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode  = atoi(argv[4]);
        ipManagementInterfaceParamsSet(&clientHandle, ifNum, mode, argv[5], argv[6]);
        break;

      case 108: /* Test 108: Get the System mode for uRPF: routing_example API 108 */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipMapSystemUrpfModeGet(&clientHandle);
        break;

      case 109: /* Test 109: Set the System mode for uRPF: routing_example API 109 <urpfEnabled> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode  = atoi(argv[3]);
        ipMapSystemUrpfModeSet(&clientHandle, mode);
        break;

      case 110: /* Test 110: Get the interface uRPF mode and allowDefault option: routing_example API 110 <ifNum> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipMapIntfUrpfModeGet(&clientHandle, ifNum);
        break;

      case 111: /* Test 111: Set the interface uRPF mode and allowDefault option: routing_example API 111 <ifNum> <mode> <allowDefault> */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode  = atoi(argv[4]);
        value = atoi(argv[5]);
        ipMapIntfUrpfModeSet(&clientHandle, ifNum, mode, value);
        break;

      case 112: /* Test 112: Get the administrative mode of sending ICMP Echo Replies: routing_example API 112 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipMapRtrICMPEchoReplyModeGet(&clientHandle, argv[3]);
        break;

      case 113: /* Test 113: Get the ICMP Rate Limiting parameters like burst size and interval: routing_example API 113 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipMapRtrICMPRatelimitGet(&clientHandle, argv[3]);
        break;

      case 114: /* Test 114: Get the internal interface number of the given VRF and IPv4 address: routing_example API 114 <*vrfName> <ipAddr> */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        iPAddrToIntIf(&clientHandle, argv[3], argv[4]);
        break;

      case 115: /* Test 115: Get the Router preference for the given protocol and VRF name: routing_example API 115 <vrfName> <origin> */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode  = atoi(argv[4]);
        ipVrRouterPreferenceGet(&clientHandle, argv[3], mode);
        break;

      case 116: /* Test 116: Set the Router preference for the given protocol and VRF name: routing_example API 116 <vrfName> <origin> <pref> */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode  = atoi(argv[4]);
        value = atoi(argv[5]);
        ipVrRouterPreferenceSet(&clientHandle, argv[3], mode, value);
        break;

      case 117: /* Test 117: Get the administrative mode of sending ICMP Redirects: routing_example API 117 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ipMapRtrICMPRedirectsModeGet(&clientHandle, argv[3]);
        break;

      case 118: /* Test 118: Set the administrative mode of sending ICMP Redirects: routing_example API 118 <vrfName> <mode> */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode  = atoi(argv[4]);
        ipMapRtrICMPRedirectsModeSet(&clientHandle, argv[3], mode);
        break;

      case 119: /* Test 119: Set ignore mode for ICMP ECHO Requests: routing_example API 119 <vrfName> <mode> */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode  = atoi(argv[4]);
        ipMapRtrICMPEchoReplyModeSet(&clientHandle, argv[3], mode);
        break;

      case 120: /* Test 120: Set the ICMP Rate Limiting parameters like burst size and interval: routing_example API 120 <vrfName> <burstSize> <interval> */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode  = atoi(argv[4]);
        value = atoi(argv[5]);
        ipMapRtrICMPRatelimitSet(&clientHandle, argv[3], mode, value);
        break;

      case 121: /* Test 121: Clear the Address Conflict Detection Status of a given VRF instance: routing_example API 121 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vrAddrConflictDetectStatusClear(&clientHandle, argv[3]);
        break;

      case 122: /* Test 122: Get the Address Conflict Detection Status of a VRF instance: routing_example API 122 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vrAddrConflictDetectStatusGet(&clientHandle, argv[3]);
        break;

      case 123: /* Test 123: Get the MAC Address of the last detected address conflict in a VRF instance: routing_example API 123 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vrAddrConflictLastDetectMACGet(&clientHandle, argv[3]);
        break;

      case 124: /* Test 124: Get the time in seconds since the last address conflict was detected in a VRF instance: routing_example API 124 <vrfName> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vrAddrConflictLastDetectTimeGet(&clientHandle, argv[3]);
        break;

      case 125: /* Test 125: Delete all net-prototype route entries: routing_example API 125 <vrfName> <protoId> */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        value = atoi(argv[4]);
        ipNetPrototypeRoutesDelete(&clientHandle, argv[3], value);
        break;

      case 126: /* Test 126: Unset the VRF participation on an interface: routing_example API 126 <ifNum> */
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipMapRtrIntfVrfReset(&clientHandle, ifNum);
        break;

      case 127: /* Test 127: Set the VRF participation of an interface: routing_example API 127 <ifNum> <vrfName> */
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        ipMapRtrIntfVrfSet(&clientHandle, ifNum, argv[4]);
        break;

      case 128: /* "Test API 128: Clear the switch/management ARP entries: routing_example API 128" */
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }

        rtrArpSwitchClear(&clientHandle);
        break;

      case 129: /* Test API 129: Set or reset the IPv6 address mode on a given router interface: routing_example API 129 <ifNum> <addrMode> <setFlag> */
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrType = atoi(argv[4]);
        extArg = atoi(argv[5]);
        rtrIntfIpv6AddressModeSet(&clientHandle, ifNum, addrType, extArg);
        break;

      default:
        printRoutingAppMenu();
        break;
    }
  }
  else
  {
    printRoutingAppMenu();
  }

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}


