/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_snooping_example_ext.c
*
* @purpose   OpEN openapi_snooping example extension API.
*
* @component OpEN
*
* @create    1/25/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_snooping.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the openapi_snooping Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the mode for the Fast Leave Auto Assignment Mode: %s 1 <family>\n", name);
  printf("Test 2: Set the mode for the Fast Leave Auto Assignment: %s 2 <family> <mode>\n", name);
  printf("Test 3: Get the IGMP plus mode: %s 3 <family>\n", name);
  printf("Test 4: Set the IGMP plus mode: %s 4 <family> <mode>\n", name);
  printf("Test 5: Get the VLAN list on which static mrouter is enabled: %s 5 <family> <ifNum>\n", name);
  printf("Test 6: Set the VLAN list on which static mrouter is enabled: %s 6 <family> <ifNum> <status>\n", name);
  printf("Test 7: Get the mode based for the Fast Leave Auto Assignment Mode: %s 7 <family> <ifNum>\n", name);
  printf("Test 8: Set the mode based for the Fast Leave Auto Assignment Mode: %s 8 <family> <ifNum> <mode>\n", name);
  printf("Test 9: Get the multicast router expiry time value: %s 9 <family> <ifNum>\n", name);
  printf("Test 10: Set the multicast router expiry time value: %s 10 <family> <ifNum> <expiryTime>\n", name);
  printf("Test 11: Get the Multicast Router Detected mode: %s 11 <family> <ifNum> <vlanId>\n", name);
  printf("Test 12: Set the specified interface as a multicast router interface: %s 12 <family> <ifNum> <mode>\n", name);
  printf("Test 13: Get the Multicast Router Detected status for the specified interface: %s 13 <family> <ifNum>\n", name);
  printf("Test 14: To check if Auto-Video feature is supported and enabled: %s 14 <vlanId>\n", name);
  printf("Test 15: To check if mrouter ports are excluded from the forwarding list: %s 15 <family>\n", name);
  printf("Test 16: Configure to exclude mrouter ports from the forwarding list: %s 16 <family> <mode>\n", name);
  printf("Test 17: Get the next supported protocol family given the family: %s 17 <family>\n", name);
  printf("Test 18: Get the last detected Querier MAC Address: %s 18 <family> <vlan>\n", name);
  printf("Test 19: Get the global mode based on which Reports will be flooded: %s 19 <family>\n", name);
  printf("Test 20: Set the global mode based on which Reports will be flooded: %s 20 <family> <mode>\n", name);
  printf("Test 21: Get whether router alert checking in IGMP frames is enabled: %s 21 <family>\n", name);
  printf("Test 22: Set if router alert checking in IGMP frames to be enabled: %s 22 <family> <checkRtrAlert>\n", name);
  printf("Test 23: Get the igmp plus vlan mode: %s 23 <family> <vlan>\n", name);
  printf("Test 24: Set the igmp plus vlan mode: %s 24 <family> <vlan> <mode>\n", name);
  printf("Test 25: Get the configured IGMP Snooping Mcast Router Expiry Time for the VLAN: %s 25 <family> <vlan>\n", name);
  printf("Test 26: Set the configured IGMP Snooping Mcast Router Expiry Time for the VLAN: %s 26 <family> <vlan> <expiryTime>\n", name);
  printf("Test 27: Get the static and dynamic Mrouter interface list: %s 27 <family> <vlan>\n", name);
  printf("Test 28: Get the VLAN mode for excluding mrouter ports: %s 28 <family> <vlan>\n", name);
  printf("Test 29: Set the VLAN mode for excluding mrouter ports: %s 29 <family> <vlan> <mode>\n", name);
  printf("Test 30: Get the report flood mode in the VLAN: %s 30 <family> <vlan>\n", name);
  printf("Test 31: Set the report flood mode in the VLAN: %s 31 <family> <vlan> <mode>\n", name);
  printf("Test 32: Get the IGMP static mcast router attached status for the specified interface/VLAN: %s 32 <family> <vlan> <ifNum>\n", name);

  return;
}


/*********************************************************************
* @purpose  Get the mode for the Fast Leave Auto Assignment Mode.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    family              @b{(input)}   address family L7_AF_INET/L7_AF_INET6
*
* @returns  none
*
* @end
*********************************************************************/
void snoopFastLeaveAutoAssignmentModeGet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopFastLeaveAutoAssignmentModeGet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the mode based for the Fast Leave Auto Assignment Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Fast Leave Auto Assignment Mode is %u. \n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the mode for the Fast Leave Auto Assignment.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    mode                @b{(input)}  Fast Leave Auto Assignment Mode ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void snoopFastLeaveAutoAssignmentModeSet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopFastLeaveAutoAssignmentModeSet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the mode for the Fast Leave Auto Assignment. (result = %d)\n", result);
  }
  else
  {
    printf("Fast Leave Auto Assignment mode set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the IGMP plus mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIgmpPlusModeGet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopIgmpPlusModeGet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the IGMP plus mode. (result = %d)\n", result);
  }
  else
  {
    printf("IGMP Plus Mode is %u. \n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the IGMP plus mode.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    mode           @b{(input)}  IGMP Plus Mode ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void snoopIgmpPlusModeSet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopIgmpPlusModeSet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the IGMP plus mode. (result = %d)\n", result);
  }
  else
  {
    printf("IGMP Plus Mode is set successfully.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the VLAN list on which static mrouter is enabled.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum               @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfStaticMcastRtrVlanMaskGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t ifNum)
{
  open_error_t result;
  OPEN_VLAN_LIST_t vidList;
  memset(&vidList, 0, sizeof(vidList));
  uint8_t i;

  if ((result = openapiSnoopIntfStaticMcastRtrVlanMaskGet(client_handle, family, ifNum, &vidList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the VLAN list on which static mrouter is enabled. (result = %d)\n", result);
  }
  else
  {
    printf("VLAN list retrieved: ");
    for (i = 0; i < vidList.numEntries; i++)
    {
      printf("%d", vidList.ids[i]);
      if (i < vidList.numEntries - 1)
      {
        printf(", ");
      }
    }
    printf("\n"); 
  }
  return;
}

/*********************************************************************
* @purpose  Set the VLAN list on which static mrouter is enabled.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum               @b{(input)}  interface
* @param    status              @b{(input)}  status as enable/disable
* @param    vlanStaticMcastRtr  @b{(input)}  VLAN list on which static mrouter to be set.
*                                            This list needs the vlans created and hence
*                                            example hardcodes this to 20,30 and creates
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfStaticMcastRtrVlanMaskSet(openapiClientHandle_t *client_handle, uint8_t family, 
                                        uint32_t ifNum, OPEN_VLAN_LIST_t *vidList, 
                                        OPEN_CONTROL_t status)
{
  open_error_t result;
  OPEN_VLAN_LIST_t vList;
  memset(&vList, 0, sizeof(vList));

  /* create a VLAN first and add to the list else the API will fail */
  vList.ids[0] = 20;
  vList.ids[1] = 30;
  vList.numEntries = 2;

  result = openapiVlanCreate(client_handle, 20);
  if ( OPEN_E_NONE == result)
  {
    /* create one more */
    result = openapiVlanCreate(client_handle, 30);
  }
  if ( OPEN_E_NONE != result )
  {
    printf("Bad return code trying to create VLANs 20,30. (result = %d)\n", result);
    return;
  }

  if ((result = openapiSnoopIntfStaticMcastRtrVlanMaskSet(client_handle, family, ifNum, &vList, status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the VLAN list on which static mrouter is enabled. (result = %d)\n", result);
  }
  else
  {
    printf("Static mrouter is %s for intf %d for VLANs 20 and 30 is set successfully.\n", 
            (status == OPEN_DISABLE) ? "disabled" : "enabled", ifNum);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the mode based for the Fast Leave Auto Assignment Mode.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum          @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfFastLeaveOperModeGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopIntfFastLeaveOperModeGet(client_handle, family, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the mode based for the Fast Leave Auto Assignment Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Fast leave operation mode for intf:%d is %u. \n", ifNum, *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the mode based for the Fast Leave Auto Assignment Mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum               @b{(input)}  interface
* @param    mode                @b{(input)}  Fast Leave Operational Mode ENABLE/DISABLE

*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfFastLeaveOperModeSet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopIntfFastLeaveOperModeSet(client_handle, family, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the mode based for the Fast Leave Auto Assignment Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Fast leave operation mode for intf:%d is set successfully. \n", ifNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the multicast router expiry time value.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum          @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfMcastRtrExpiryTimeGet(openapiClientHandle_t *client_handle, uint8_t family, 
                                    uint32_t ifNum, uint32_t *expiryTime)
{
  open_error_t result;

  if ((result = openapiSnoopIntfMcastRtrExpiryTimeGet(client_handle, family, ifNum, expiryTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the multicast router expiry time value. (result = %d)\n", result);
  }
  else
  {
    printf("The multicast router expiry time for interface:%u is %u.\n", ifNum, *expiryTime);
  }
  return;
}


/*********************************************************************
* @purpose  Set the multicast router expiry time value.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum          @b{(input)}  interface
* @param    expiryTime     @b{(input)}  multicast router expiry time value

*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfMcastRtrExpiryTimeSet(openapiClientHandle_t *client_handle, 
                                    uint8_t family, uint32_t ifNum, uint32_t expiryTime)
{
  open_error_t result;

  if ((result = openapiSnoopIntfMcastRtrExpiryTimeSet(client_handle, family, ifNum, expiryTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the multicast router expiry time value. (result = %d)\n", result);
  }
  else
  {
    printf("The multicast router expiry time for interface:%u is set successfully.\n", ifNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Multicast Router Detected mode
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum          @b{(input)}  interface
* @param    vlanId         @b{(input)}  VLAN ID

*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfMrouterGet(openapiClientHandle_t *client_handle, uint8_t family, 
                         uint32_t ifNum, uint32_t vlanId, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopIntfMrouterGet(client_handle, family, ifNum, vlanId, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the Multicast Router Detected mode (result = %d)\n", result);
  }
  else
  {
    printf("Multicast router detected mode for VLAN: %d intf:%d is %u. \n", vlanId, ifNum, *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the specified interface as a multicast router interface
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum          @b{(input)}  interface
* @param    mode           @b{(input)}  multicast router interface admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfMrouterSet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopIntfMrouterSet(client_handle, family, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the specified interface as a multicast router interface (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d is set as multicast router successfully. \n", ifNum );
  }
  return;
}


/*********************************************************************
* @purpose  Gets the Multicast Router Detected status for the specified interface
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    ifNum          @b{(input)}  interface

*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfMrouterStatusGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopIntfMrouterStatusGet(client_handle, family, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the Multicast Router Detected status for the specified interface (result = %d)\n", result);
  }
  else
  {
    printf("Multicast router status on the interface: %d is %u.\n", ifNum, *mode);

  }
  return;
}


/*********************************************************************
* @purpose  To check if Auto-Video feature is supported and enabled
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    vlanId         @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIsAutoVideoVlanEnabled(openapiClientHandle_t *client_handle, 
                                 uint32_t vlanId, OPEN_BOOL_t *status)
{
  openapiSnoopIsAutoVideoVlanEnabled(client_handle, vlanId, status);
  if (OPEN_TRUE == *status) 
  {
    printf("Auto-Video feature is supported and enabled.\n");
  }
  else
  {
    printf("Auto-Video feature is either not supported or not enabled.\n");
  }
  return;
}


/*********************************************************************
* @purpose  To check if mrouter ports are excluded from the forwarding list
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    mode           @b{(output)} mode

*
* @returns  none
*
* @end
*********************************************************************/
void snoopMrtrExcludeModeGet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopMrtrExcludeModeGet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to to check if mrouter ports are excluded from the forwarding list (result = %d)\n", result);
  }
  else
  {
    printf("Mode if mrouter ports are excluded from the forwarding list is %u. \n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Configure to exclude mrouter ports from the forwarding list
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    mode           @b{(input)}  mode

*
* @returns  none
*
* @end
*********************************************************************/
void snoopMrtrExcludeModeSet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopMrtrExcludeModeSet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure to exclude mrouter ports from the forwarding list (result = %d)\n", result);
  }
  else
  {
    printf("Configuration to exclude mrouter ports from the forwarding list is set successfully. \n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the next supported protocol family given the family
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    nextFamily     @b{(output)}  address family L7_AF_INET/L7_AF_INET6

*
* @returns  none
*
* @end
*********************************************************************/
void snoopProtocolNextGet(openapiClientHandle_t *client_handle, uint32_t family, uint32_t *nextFamily)
{
  open_error_t result;

  if ((result = openapiSnoopProtocolNextGet(client_handle, family, nextFamily)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next supported protocol family given the family (result = %d)\n", result);
  }
  else
  {
    printf("The next supported protocol family for the given family is: %u. \n", *nextFamily);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the Snooping Querier Configured Address
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    family             @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    snoopQuerierAddr   @b{(output)} 32bit value for family L7_AF_INET string for L7_AF_INET6
*
*
* @returns  none
*
* @end
*********************************************************************/
void snoopQuerierAddressGet(openapiClientHandle_t *client_handle, uint8_t family, open_inet_addr_t *snoopQuerierAddr)
{
  open_error_t result;
  struct in_addr ipAddrStr;

  if ((result = openapiSnoopQuerierAddressGet(client_handle, snoopQuerierAddr, family)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the Snooping Querier Configured Address (result = %d)\n", result);
  }
  else
  {
    if(OPEN_AF_INET == family) /* sample written for v4 only */
    {
      ipAddrStr.s_addr = htonl(snoopQuerierAddr->addr.ipv4);
      printf("Snooping Querier Address is: %s .\n", inet_ntoa(ipAddrStr));
    }
  }
  return;
}


/*********************************************************************
* @purpose  Sets the Snooping Querier Configured Address
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    snoopQuerierAddr    @b{(input)}  32bit value for family L7_AF_INET string for L7_AF_INET6
*
* @returns  none
*
* @end
*********************************************************************/
void snoopQuerierAddressSet(openapiClientHandle_t *client_handle, uint8_t family, open_inet_addr_t *snoopQuerierAddr)
{
  open_error_t result;

  if ((result = openapiSnoopQuerierAddressSet(client_handle, snoopQuerierAddr, family)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the Snooping Querier Configured Address (result = %d)\n", result);
  }
  else
  {
    printf("Snooping Querier Address set successfully. \n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets the last detected Querier Address
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    family             @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    snoopQuerierAddr   @b{(output)} 32bit value for family L7_AF_INET string for L7_AF_INET6

*
* @returns  none
*
* @end
*********************************************************************/
void snoopQuerierLastQuerierAddressGet(openapiClientHandle_t *client_handle, uint32_t vlan, 
                                       open_inet_addr_t *snoopQuerierAddr, uint8_t family)
{
  open_error_t result;
  struct in_addr ipAddrStr;

  if ((result = openapiSnoopQuerierLastQuerierAddressGet(client_handle, vlan, snoopQuerierAddr, family)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the last detected Querier Address (result = %d)\n", result);
  }
  else
  {
    if(OPEN_AF_INET == family) /* sample written for v4 only */
    {
      ipAddrStr.s_addr = htonl(snoopQuerierAddr->addr.ipv4);
      printf("Snooping last querier address is: %s .\n", inet_ntoa(ipAddrStr));
    }
  }
  return;
}


/*********************************************************************
* @purpose  Gets the last detected Querier MAC Address
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    family              @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan                @b{(input)}  VLAN ID
* @param    snoopQuerierMac     @b{(output)} Querier MAC address

*
* @returns  none
*
* @end
*********************************************************************/
void snoopQuerierMacAddrGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, open_buffdesc *snoopQuerierMac)
{
  open_error_t result;
  char    macAddr[OPEN_MAC_ADDR_LEN];
  
  snoopQuerierMac->pstart = macAddr;
  snoopQuerierMac->size = OPEN_MAC_ADDR_LEN;

  if ((result = openapiSnoopQuerierMacAddrGet(client_handle, family, vlan, snoopQuerierMac)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the last detected Querier MAC Address (result = %d)\n", result);
  }
  else
  {
    printf("Snooping querier MAC address is :  %02X-%02X-%02X-%02X-%02X-%02X . \n",
            macAddr[0]&0xFF,macAddr[1]&0xFF,macAddr[2]&0xFF,macAddr[3]&0xFF,macAddr[4]&0xFF,macAddr[5]&0xFF);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the mode based on which Reports will be flooded.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    mode           @b{(output)} Report Flood Mode ENABLE/DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void snoopReportFloodModeGet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopReportFloodModeGet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the mode based on which Reports will be flooded. (result = %d)\n", result);
  }
  else
  {
    printf("snooping reports flood mode is %u. \n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the mode based on which Reports will be flooded.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    family          @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    mode            @b{(input)}  Report Flood Mode ENABLE/DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void snoopReportFloodModeSet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopReportFloodModeSet(client_handle, family, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the mode based on which Reports will be flooded. (result = %d)\n", result);
  }
  else
  {
    printf("snooping reports flood mode is set successfully. \n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets whether router alert checking in IGMP frames is enabled.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    family          @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    checkRtrAlert   @b{(output)}  Router Alert Checking enabled TRUE/FALSE
*
* @returns  none
*
* @end
*********************************************************************/
void snoopRouterAlertMandatoryGet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_BOOL_t *checkRtrAlert)
{
  open_error_t result;

  if ((result = openapiSnoopRouterAlertMandatoryGet(client_handle, family, checkRtrAlert)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets whether router alert checking in IGMP frames is enabled. (result = %d)\n", result);
  }
  else
  {
    printf("Mode to check router-alert in IGMP frames is %u . \n", *checkRtrAlert);
  }
  return;
}


/*********************************************************************
* @purpose  Sets if router alert checking in IGMP frames to be enabled.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    family          @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    checkRtrAlert   @b{(input)}  Router Alert Checking enabled TRUE/FALSE
*
* @returns  none
*
* @end
*********************************************************************/
void snoopRouterAlertMandatorySet(openapiClientHandle_t *client_handle, uint8_t family, OPEN_BOOL_t checkRtrAlert)
{
  open_error_t result;

  if ((result = openapiSnoopRouterAlertMandatorySet(client_handle, family, checkRtrAlert)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets if router alert checking in IGMP frames to be enabled. (result = %d)\n", result);
  }
  else
  {
    printf("Mode to check router-alert in IGMP frames is set successfully. \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the igmp plus vlan mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    family          @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan            @b{(input)}  VLAN ID
* @param    mode            @b{(output)} igmp plus vlan mode ENABLED/DISABLED
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanIgmpPlusModeGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopVlanIgmpPlusModeGet(client_handle, family, vlan, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the igmp plus vlan mode. (result = %d)\n", result);
  }
  else
  {
    printf("IGMP plus VLAN mode is %u. \n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the igmp plus vlan mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    family          @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan            @b{(input)}  VLAN ID
* @param    mode            @b{(input)}  igmp plus vlan mode ENABLED/DISABLED
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanIgmpPlusModeSet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopVlanIgmpPlusModeSet(client_handle, family, vlan, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the igmp plus vlan mode. (result = %d)\n", result);
  }
  else
  {
    printf("IGMP plus VLAN mode is set successfully. \n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets the configured IGMP Snooping Mcast Router Expiry Time for the VLAN.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan           @b{(input)}  VLAN ID
* @param    expiryTime     @b{(output)} Snooping Mcast Router Expiry Time

*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanMcastRtrExpiryTimeGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, uint32_t *expiryTime)
{
  open_error_t result;
  short unsigned int expiryInterval = 0;

  if ((result = openapiSnoopVlanMcastRtrExpiryTimeGet(client_handle, family, vlan, &expiryInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the configured IGMP Snooping Mcast Router Expiry Time for the VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("The multicast router expiry time for VLAN:%u is %u.\n", vlan, expiryInterval);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the configured IGMP Snooping Mcast Router Expiry Time for the VLAN.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan           @b{(input)}  VLAN ID
* @param    expiryTime     @b{(input)}  Snooping Mcast Router Expiry Time
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanMcastRtrExpiryTimeSet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, uint16_t expiryTime)
{
  open_error_t result;

  if ((result = openapiSnoopVlanMcastRtrExpiryTimeSet(client_handle, family, vlan, expiryTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the configured IGMP Snooping Mcast Router Expiry Time for the VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("The multicast router expiry time for VLAN:%u is set successfully.\n", vlan);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the static and dynamic Mrouter interface list.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan           @b{(input)}  VLAN ID
* @param    dynamicMask    @b{(output)} Dynamic Mrouter interface list
* @param    staticMask     @b{(output)} Static Mrouter interface list
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanMrouterListGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan) 
{
  open_error_t result;
  char mask1[256] = {0}; /* ideally this should be L7_MAX_INTERFACE_COUNT x 12 chars */
  char mask2[256] = {0};
  open_buffdesc dynamicMask, staticMask;
  dynamicMask.pstart = mask1;
  dynamicMask.size = 256;
  staticMask.pstart = mask2;
  staticMask.size = 256;

  if ((result = openapiSnoopVlanMrouterListGet(client_handle, family, vlan, &dynamicMask, &staticMask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the static and dynamic Mrouter interface list. (result = %d)\n", result);
  }
  else
  {
    if ( dynamicMask.size)
    {
       printf("Dynamic mrouter list of interfaces : %s\n", (char *)dynamicMask.pstart);
    }
    if ( staticMask.size)
    {
       printf("Static mrouter list of interfaces : %s\n", (char *)staticMask.pstart);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Gets the VLAN mode for excluding mrouter ports.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan           @b{(input)}  VLAN ID
* @param    mode           @b{(output)} Exclude mode ENABLED/DISABLED
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanMrtrExcludeModeGet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopVlanMrtrExcludeModeGet(client_handle, family, vlan, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the VLAN mode for excluding mrouter ports. (result = %d)\n", result);
  }
  else
  {
    printf("Mode to exclude mrouter ports on VLAN: %d is %u . \n", vlan, *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the VLAN mode for excluding mrouter ports.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    family          @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan            @b{(input)}  VLAN ID
* @param    mode            @b{(input)}  Exclude mode ENABLED/DISABLED
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanMrtrExcludeModeSet(openapiClientHandle_t *client_handle, uint8_t family, 
                                 uint32_t vlan, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopVlanMrtrExcludeModeSet(client_handle, family, vlan, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the VLAN mode for excluding mrouter ports. (result = %d)\n", result);
  }
  else
  {
    printf("Mode to exclude mrouter ports on VLAN: %d is set successfully.\n", vlan);
  }
  return;
}


/*********************************************************************
* @purpose  Gets the report flood mode in the VLAN.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    family           @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan             @b{(input)}  VLAN ID
* @param    mode             @b{(output)} Flood mode ENABLED/DISABLED
*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanReportFloodModeGet(openapiClientHandle_t *client_handle, uint8_t family, 
                                 uint32_t vlan, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiSnoopVlanReportFloodModeGet(client_handle, family, vlan, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the report flood mode in the VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Report flood mode in VLAN : %d is %u.\n", vlan, *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the report flood mode in the VLAN.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan           @b{(input)}  VLAN ID
* @param    mode           @b{(input)}  Flood mode ENABLED/DISABLED

*
* @returns  none
*
* @end
*********************************************************************/
void snoopVlanReportFloodModeSet(openapiClientHandle_t *client_handle, uint8_t family, uint32_t vlan, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSnoopVlanReportFloodModeSet(client_handle, family, vlan, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the report flood mode in the VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Report flood mode in VLAN : %d is set successfully.\n", vlan);
  }
  return;
}


/*********************************************************************
* @purpose  Get the IGMP static mcast router attached status for the specified interface/VLAN.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    family         @b{(input)}  address family L7_AF_INET/L7_AF_INET6
* @param    vlan           @b{(input)}  VLAN ID
* @param    ifNum          @b{(input)}  interface number
* @param    status         @b{(output)} status ENABLED/DISABLED
*
* @returns  none
*
* @end
*********************************************************************/
void snoopIntfApiVlanStaticMcastRtrGet(openapiClientHandle_t *client_handle, uint8_t family, 
                                       uint32_t vlan, uint32_t ifNum, OPEN_CONTROL_t *status)
{
  open_error_t result;

  if ((result = openapiSnoopIntfApiVlanStaticMcastRtrGet(client_handle, family, vlan, ifNum, status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the IGMP static mcast router attached status for the specified interface/VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("IGMP static mcast router attached status for the specified interface/VLAN is %u.\n", *status);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for openapi_snooping
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  open_error_t result;
  uint32_t testNum = 0;
  uint32_t intValue = 0;
  OPEN_VLAN_LIST_t vidList;
  OPEN_BOOL_t trueVal;
  open_buffdesc switch_os_revision;
  open_buffdesc macBuffdesc;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("openapi_snooping example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting openapi_snooping API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        snoopFastLeaveAutoAssignmentModeGet(&client_handle, atoi(argv[2]), &mode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
        snoopFastLeaveAutoAssignmentModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        snoopIgmpPlusModeGet(&client_handle, atoi(argv[2]), &mode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        snoopIgmpPlusModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        snoopIntfStaticMcastRtrVlanMaskGet(&client_handle, atoi(argv[2]), atoi(argv[3]) );
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 5)
      {
        snoopIntfStaticMcastRtrVlanMaskSet(&client_handle, atoi(argv[2]), atoi(argv[3]), &vidList, atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
        snoopIntfFastLeaveOperModeGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &mode);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 5)
      {
        snoopIntfFastLeaveOperModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 4)
      {
        snoopIntfMcastRtrExpiryTimeGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &intValue);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 5)
      {
        snoopIntfMcastRtrExpiryTimeSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 5)
      {
        snoopIntfMrouterGet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]), &mode);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 5)
      {
        snoopIntfMrouterSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        snoopIntfMrouterStatusGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &mode);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        snoopIsAutoVideoVlanEnabled(&client_handle, atoi(argv[4]), &trueVal);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
        snoopMrtrExcludeModeGet(&client_handle, atoi(argv[2]), &mode);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 4)
      {
        snoopMrtrExcludeModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        snoopProtocolNextGet(&client_handle, atoi(argv[2]), &intValue);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 4)
      {
        snoopQuerierMacAddrGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &macBuffdesc);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 3)
      {
        snoopReportFloodModeGet(&client_handle, atoi(argv[2]), &mode);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 4)
      {
        snoopReportFloodModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 3)
      {
        snoopRouterAlertMandatoryGet(&client_handle, atoi(argv[2]), &trueVal);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 4)
      {
        snoopRouterAlertMandatorySet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 4)
      {
        snoopVlanIgmpPlusModeGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &mode);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 5)
      {
        snoopVlanIgmpPlusModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 4)
      {
        snoopVlanMcastRtrExpiryTimeGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &intValue);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 5)
      {
        snoopVlanMcastRtrExpiryTimeSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 4)
      {
        snoopVlanMrouterListGet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 4)
      {
        snoopVlanMrtrExcludeModeGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &mode);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 5)
      {
        snoopVlanMrtrExcludeModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 4)
      {
        snoopVlanReportFloodModeGet(&client_handle, atoi(argv[2]), atoi(argv[3]), &mode);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 5)
      {
        snoopVlanReportFloodModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 5)
      {
        snoopIntfApiVlanStaticMcastRtrGet(&client_handle, atoi(argv[2]), atoi(argv[3]), atoi(argv[4]), &mode);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping openapi_snooping API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
