/*********************************************************************
*
*  Copyright 2019-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  static_filter_example.c
*
* @purpose   OpEN Static Filter example.
*
* @component OpEN
*
* @create    06/08/2019
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_static_filter.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

#define OPENAPI_TITLE  "Static Filter"
#define MAC_STRING_SIZE 18
#define INTF_LIST_STR_SIZE ((4 * 1024) +1)

/*******************************************************************
*
* @brief  This function prints the Static Filter Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets the static MAC filter admin mode: %s 1 <adminMode>\n", name);
  printf("Test 2: Gets the static MAC filter admin mode: %s 2 \n", name);
  printf("Test 3: Creates a static MAC filter: %s 3 <macAddr> <vlanId>\n", name);
  printf("Test 4: Deletes a static MAC filter: %s 4 <macAddr> <vlanId>\n", name);
  printf("Test 5: Apply a comma separated  source interface list to the static MAC filter: %s 5 "
         "<macAddr> <vlanId> <srcIntfList>\n", name);
  printf("Test 6: Adds a source interface to the static MAC filter: %s 6 "
         "<macAddr> <vlanId> <ifNum>\n", name);
  printf("Test 7: Deletes a source interface from the source interface list of "
         "the static MAC filter: %s 7 <macAddr> <vlanId> <ifNum>\n", name);
  printf("Test 8: Adds a destination interface to the static MAC filter: %s 8 "
         "<macAddr> <vlanId> <ifNum>\n", name);
  printf("Test 9: Deletes a destination interface from the destination interface"
         " list of the static MAC filter: %s 9 <macAddr> <vlanId> <ifNum>\n", name);
  printf("Test 10: Adds a comma separated destination interface list to the "
         "static MAC filter: %s 10 <macAddr> <vlanId> <ifNum> <fwdIntfList> "
         "<filterList>\n", name);
  printf("Test 11: Deletes a comma separated destination interface list from "
          "the static MAC filter: %s 11 <macAddr> <vlanId> <ifNum> <fwdIntfList>"
          " <filterList>\n", name);
  printf("Test 12: Gets the first configured MAC static filter: %s 12 \n", name);
  printf("Test 13: Gets the next configured MAC static filter: %s 13 <macAddr> "
         "<vlanId> \n", name);
  printf("Test 14: Gets list of source interfaces for a MAC static filter: %s 14 "
         "<macAddr> <vlanId> \n", name);
  printf("Test 15: Gets list of destination interfaces for a MAC static filter: %s"
         " 15 <macAddr> <vlanId> \n", name);
  printf("Test 16: Gets the number of MAC static filters configured: %s 16 \n", name);
  printf("Test 17: Gets the number of total, multicast and unicast MAC static filters"
         " configured: %s 17 \n", name);
  printf("Test 18: Checks to see if a MAC filter is configured or not: %s 18 <macAddr>"
         " <vlanId> \n", name);
  printf("Test 19: Gets the buffer size used by filtering component for the bit mask:"
         " %s 19 \n", name);
  printf("Test 20: Checks if a particular interface is present in any MAC filter: "
         "%s 20 <ifNum>\n", name);
  printf("Test 21: Check to see if the port is the appropriate type on which the"
         " param can be set: %s 21 <ifNum> \n", name);
  printf("Test 22: Checks to see the mac address type, unicast or multicast: %s "
         "22 <macAddr>\n", name);
  printf("Test 23: Set multicast address filtering configuration for all vlans: "
         "%s 23 <mode>\n", name);
  printf("Test 24: Sets multicast address filtering configuration for the "
         "specified vlan: %s 24 <vlanId> <mode>\n", name);
  printf("Test 25: Gets multicast address filtering configuration for the "
         "specified vlan: %s 25 <vlanId> \n", name);
  printf("Test 26: Checks if the interface is valid for filtering: %s 26 "
         "<ifNum> \n", name);
  printf("Test 27: Sets the stream id of an existing macfilter with given MAC"
         " address and vlan id: %s 27 <macAddr> <vlanId> <streamId>\n", name);
  printf("Test 28: Gets the configured stream id: %s 28 <macAddr> <vlanId> \n", 
         name);
  printf("Test 29: Gets list of destination filter interfaces for a MAC "
         "static filter: %s 29 <macAddr> <vlanId> \n", name);
  return;
}

/*********************************************************************
* @purpose  Sets the static MAC filter admin mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    adminMode       @b{(input)}  Static MAC filter admin mode 
*
* @returns  none
*
* @end
*********************************************************************/
void filterAdminModeSet(openapiClientHandle_t *client_handle, 
                        OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiFilterAdminModeSet(client_handle, adminMode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the static MAC filter admin "
           "mode. (result = %d)\n", result);
  }
  else
  {
    printf("Static MAC filter admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the static MAC filter admin mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    adminMode       @b{(output)} Static MAC filter admin mode 
*
* @returns  none
*
* @end
*********************************************************************/
void filterAdminModeGet(openapiClientHandle_t *client_handle, 
                        OPEN_CONTROL_t *adminMode)
{
  open_error_t result;

  if ((result = openapiFilterAdminModeGet(client_handle, adminMode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the static MAC filter "
           "admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Static MAC filter admin mode is %u.\n", *adminMode);
  }
  return;
}

/*********************************************************************
* @purpose  Create a MAC Static Filter.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macStr          @b{(input)}  static MAC address for the filter
* @param    vlanID          @b{(input)}  VLAN ID for the filter
* 
* @returns  none
*
* @end
*********************************************************************/
void filterCreate(openapiClientHandle_t *client_handle, char *macStr, 
                  uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);
  if ((result = openapiFilterCreate(client_handle, &buffDesc, vlanID)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a MAC Static Filter. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("%s mac=%s vlan=%u created. (result = %d)\n", 
           OPENAPI_TITLE, macStr, vlanID, result);
  }
  return;
}

/*********************************************************************
* @purpose  Remove a configured MAC Static Filter.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    macStr          @b{(input)}  static MAC address for the filter
* @param    vlanID          @b{(input)}  VLAN ID for the filter
*
* @returns  none
*
* @end
*********************************************************************/
void filterRemove(openapiClientHandle_t *client_handle, char *macStr, 
                  uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);
  if ((result = openapiFilterRemove(client_handle, &buffDesc, vlanID)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to remove a configured MAC "
           "Static Filter. (result = %d)\n", result);
  }
  else
  {
    printf("%s mac=%s vlan=%u removed. (result = %d)\n", 
           OPENAPI_TITLE, macStr, vlanID, result);
  }
  return;
}

/*********************************************************************
* @purpose  Apply a source interface list to the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    srcIntfList   @b{(input)}  List of interfaces specifying source ports
*
* @returns  none
*
* @end
*********************************************************************/
void filterSrcIntfListModify(openapiClientHandle_t *client_handle, 
                             char *macAddr, uint32_t vlanId, 
                             char *srcIntfList)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char intfStr[INTF_LIST_STR_SIZE] = "";
  open_buffdesc bufId, intfList;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

	strncpy(intfStr, srcIntfList, INTF_LIST_STR_SIZE-1);
  intfList.pstart = intfStr;
  intfList.size   = strlen (intfStr);

  if ((result = openapiFilterSrcIntfListModify(client_handle, &bufId, 
                                               vlanId, &intfList)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to apply a source interface List to "
           "the static MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("List of source interface(s) applied successfully to static MAC"
           " filter.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds a source interface to the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    ifNum         @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void filterSrcIntfAdd(openapiClientHandle_t *client_handle, 
                      char *macAddr, uint32_t vlanId, uint32_t ifNum)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterSrcIntfAdd(client_handle, &bufId, vlanId, ifNum))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds a source interface to the static "
           "MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("Source interface addition to static MAC filter was successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes a source interface from the source interface list 
*           of the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    ifNum         @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void filterSrcIntfDelete(openapiClientHandle_t *client_handle, 
                         char *macAddr, uint32_t vlanId, 
                         uint32_t ifNum)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterSrcIntfDelete(client_handle, &bufId, vlanId, ifNum))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to deletes a source interface from the source "
           "interface list of the static MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("Source interface deletion from static MAC filter was successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds a destination interface to the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    ifNum         @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void filterDstIntfAdd(openapiClientHandle_t *client_handle, 
                      char *macAddr, uint32_t vlanId, uint32_t ifNum)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterDstIntfAdd(client_handle, &bufId, vlanId, ifNum))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds a destination interface to the static"
           " MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("Destination interface addition to static MAC filter was successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes a destination interface from the destination interface 
*           list of the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    ifNum         @b{(input)}  Interface number
*
* @returns  none
*
* @end
*********************************************************************/
void filterDstIntfDelete(openapiClientHandle_t *client_handle, 
                         char *macAddr, uint32_t vlanId, 
                         uint32_t ifNum)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterDstIntfDelete(client_handle, &bufId, 
                                           vlanId, ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to deletes a destination interface from"
           " the destination interface list of the static MAC filter. (result = %d)\n",
           result);
  }
  else
  {
    printf("Destination interface deletion from static MAC filter was successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds destination interfaces to the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    ifNum         @b{(input)}  Interface number
* @param    fwdIntfList   @b{(input)}  List of ports to forward
* @param    filterIntfList@b{(input)}  List of ports to forbid
*
* @returns  none
*
* @end
*********************************************************************/
void filterDstIntfListAdd(openapiClientHandle_t *client_handle, 
                          char *macAddr, uint32_t vlanId, 
                          uint32_t ifNum, char *fwdIntfList, 
                          char *filterIntfList)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char intfStr[INTF_LIST_STR_SIZE] = "";
 char filtIntfStr[INTF_LIST_STR_SIZE] = "";
  open_buffdesc bufId, intfList, filIntfList;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

	strncpy(intfStr, fwdIntfList, INTF_LIST_STR_SIZE-1);
  intfList.pstart = intfStr;
  intfList.size   = strlen (intfStr);

	strncpy(filtIntfStr, filterIntfList, INTF_LIST_STR_SIZE-1);
  filIntfList.pstart = filtIntfStr;
  filIntfList.size   = strlen (filtIntfStr);

  if ((result = openapiFilterDstIntfListAdd(client_handle, &bufId, vlanId,
                                            ifNum, &intfList, &filIntfList)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds destination interfaces to the "
           "static MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("Addition of destination interface(s) to static MAC filter was "
            "successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes destination interfaces from the static MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    ifNum         @b{(input)}  Interface number
* @param    fwdIntfList   @b{(input)}  List of ports to forward
* @param    filterIntfList@b{(input)}  List of ports to forbid
*
* @returns  none
*
* @end
*********************************************************************/
void filterDstIntfListDelete(openapiClientHandle_t *client_handle, 
                             char *macAddr, uint32_t vlanId, 
                             uint32_t ifNum, char *fwdIntfList, 
                             char *filterIntfList)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char intfStr[INTF_LIST_STR_SIZE] = "";
  char filtIntfStr[INTF_LIST_STR_SIZE] = "";
  open_buffdesc bufId, intfList, filIntfList;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

	strncpy(intfStr, fwdIntfList, INTF_LIST_STR_SIZE-1);
  intfList.pstart = intfStr;
  intfList.size   = strlen (intfStr);

	strncpy(filtIntfStr, filterIntfList, INTF_LIST_STR_SIZE-1);
  filIntfList.pstart = filtIntfStr;
  filIntfList.size   = strlen (filtIntfStr);

  if ((result = openapiFilterDstIntfListDelete(client_handle, &bufId, vlanId,
                                               ifNum, &intfList, &filIntfList)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds destination interfaces to the "
           "static MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("Deletion of destination interface(s) from static MAC filter was "
            "successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the first configured MAC static filter.
*
* @param    client_handle @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void filterFirstGet(openapiClientHandle_t *client_handle) 
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;
  uint32_t vlanId;

  bufId.pstart = mac;
  bufId.size   = MAC_STRING_SIZE;

  if ((result = openapiFilterFirstGet(client_handle, &bufId, &vlanId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the first configured MAC "
           "static filter. (result = %d)\n", result);
  }
  else
  {
    printf("First filter's MAC : %s, vlanId : %u.\n", mac, vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the next configured MAC static filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void filterNextGet(openapiClientHandle_t *client_handle, 
                   char *macAddr, uint32_t vlanId) 
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char nextMacStr[MAC_STRING_SIZE] = "";
  open_buffdesc bufId, nextMac;
  uint32_t nextVlanId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  nextMac.pstart = nextMacStr;
  nextMac.size   = MAC_STRING_SIZE;

  if ((result = openapiFilterNextGet(client_handle, &bufId, vlanId, 
                                     &nextMac, &nextVlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the next configured MAC static "
           "filter. (result = %d)\n", result);
  }
  else
  {
    printf("Next filter's MAC : %s, vlanId : %u.\n", nextMacStr, nextVlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Gets list of source interfaces for a MAC static filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void filterSrcIntfListGet(openapiClientHandle_t *client_handle, 
                          char *macAddr, uint32_t vlanId)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char intfStr[INTF_LIST_STR_SIZE] = "";
  open_buffdesc bufId, intfList;
  uint32_t numOfSrcIntf;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  intfList.pstart = intfStr;
  intfList.size   = INTF_LIST_STR_SIZE;

  if ((result = openapiFilterSrcIntfListGet(client_handle, &bufId, vlanId,
                                            &numOfSrcIntf, &intfList)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets list of source interfaces for a"
           " MAC static filter. (result = %d)\n", result);
  }
  else
  {
    printf("List of source interfaces for a filter is:\n%s\n", intfStr);
  }
  return;
}

/*********************************************************************
* @purpose  Gets list of destination interfaces for a MAC static filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void filterDstIntfListGet(openapiClientHandle_t *client_handle, 
                          char *macAddr, uint32_t vlanId) 
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char intfStr[INTF_LIST_STR_SIZE] = "";
  open_buffdesc bufId, intfList;
  uint32_t numOfDstIntf;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  intfList.pstart = intfStr;
  intfList.size   = INTF_LIST_STR_SIZE;

  if ((result = openapiFilterDstIntfListGet(client_handle, &bufId, vlanId,
                                            &numOfDstIntf, &intfList)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets list of source interfaces for a"
           " MAC static filter. (result = %d)\n", result);
  }
  else
  {
    printf("List of destination interfaces for a filter is:\n%s\n", intfStr);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the number of MAC static filters configured.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    count         @b{(output)} Number of MAC static filters configured
*
* @returns  none
*
* @end
*********************************************************************/
void filterCountGet(openapiClientHandle_t *client_handle, uint32_t *count)
{
  open_error_t result;

  if ((result = openapiFilterCountGet(client_handle, count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the number of MAC static filters "
           "configured. (result = %d)\n", result);
  }
  else
  {
    printf("Number of MAC static filters : %u\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the number of total, multicast and unicast MAC static 
*           filters configured.
*
* @param    client_handle      @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void filterCountersGet(openapiClientHandle_t *client_handle) 
{
  open_error_t result;
  uint32_t totalCount, multicastFiltCount, unicastFiltCount;

  if ((result = openapiFilterCountersGet(client_handle, &totalCount, 
                                         &multicastFiltCount, &unicastFiltCount))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the number of total, multicast and "
           "unicast MAC static filters configured. (result = %d)\n", result);
  }
  else
  {
    printf("Total number of MAC static filters : %u \n", totalCount);
    printf("Number of multicast MAC static filters : %u \n", multicastFiltCount);
    printf("Number of unicast MAC static filters : %u \n", unicastFiltCount);
  }
  return;
}

/*********************************************************************
* @purpose  Checks to see if a MAC filter is configured or not.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    macAddr        @b{(input)}  MAC address
* @param    vlanId         @b{(input)}  VLAN ID
* @param    status         @b{(output)} Status of MAC filter
*
* @returns  none
*
* @end
*********************************************************************/
void filterIsConfigured(openapiClientHandle_t *client_handle, 
                        char *macAddr, uint32_t vlanId, 
                        OPEN_BOOL_t *status)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterIsConfigured(client_handle, &bufId, 
                                          vlanId, status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to checks to see if a MAC filter is "
           "configured or not. (result = %d)\n", result);
  }
  else
  {
    printf("MAC filter is %sconfigured.\n", (*status == OPEN_TRUE) ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the buffer size used by filtering component for the bit mask.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    size            @b{(output)} Buffer size for the bit mask
*
* @returns  none
*
* @end
*********************************************************************/
void filterSizeGet(openapiClientHandle_t *client_handle, uint32_t *size)
{
  open_error_t result;

  if ((result = openapiFilterSizeGet(client_handle, size)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the buffer size used by "
           "filtering component for the bit mask. (result = %d)\n", result);
  }
  else
  {
    printf("Buffer size used by filtering component : %u.\n", *size);
  }
  return;
}

/*********************************************************************
* @purpose  Checks if a particular interface is present in any MAC filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    status        @b{(output)} Presence of MAC filter in the specified interface
*
* @returns  none
*
* @end
*********************************************************************/
void filterIsIntfInAnyFilter(openapiClientHandle_t *client_handle, 
                             uint32_t ifNum, OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiFilterIsIntfInAnyFilter(client_handle, ifNum, status)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to checks if a particular interface is "
           "present in any MAC filter. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %u is %sconfigured in MAC filter.\n", ifNum,
           (*status == OPEN_TRUE) ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Check to see if the port is the appropriate type on which 
*           the parm can be set.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    status        @b{(output)} Status of the parm can be set on 
*                                      this port type
*
* @returns  none
*
* @end
*********************************************************************/
void filterPortParmCanSet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                          OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiFilterPortParmCanSet(client_handle, ifNum, status)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to check to see if the port is the "
           "appropriate type on which the parm can be set. (result = %d)\n", 
           result);
  }
  else
  {
    printf("Interface %u is %sappropriate to configure MAC filter parameters.\n",
           ifNum, (*status == OPEN_TRUE) ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Checks to see the mac address type, unicast or multicast.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    status        @b{(output)} OPEN_TRUE  MAC address is of type multicast 
*                                      OPEN_FALSE MAC address is of type unicast
*
* @returns  none
*
* @end
*********************************************************************/
void filterCheckMacAddrTypeMulticast(openapiClientHandle_t *client_handle, 
                                     char *macAddr, 
                                     OPEN_BOOL_t *status)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterCheckMacAddrTypeMulticast(client_handle, &bufId, 
                                                       status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to checks to see the mac address type, "
           "unicast or multicast. (result = %d)\n", result);
  }
  else
  {
    printf("MAC address %s is %s.\n", macAddr,
           (*status == OPEN_TRUE) ? "multicast" : "unicast");
  }
  return;
}

/*********************************************************************
* @purpose  Set multicast address filtering configuration for all vlans.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mode          @b{(input)}  Filtering mode
*
* @returns  none
*
* @end
*********************************************************************/
void filterFilteringModeSet(openapiClientHandle_t *client_handle, 
                            OPEN_FILTER_VLAN_FILTER_MODE_t mode)
{
  open_error_t result;

  if ((result = openapiFilterFilteringModeSet(client_handle, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set multicast address filtering"
           " configuration for all vlans. (result = %d)\n", result);
  }
  else
  {
    printf("Multicast address filtering configuration for all vlans is"
           " successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Sets multicast address filtering configuration for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(input)}  Filtering mode
*
* @returns  none
*
* @end
*********************************************************************/
void filterVlanFilteringModeSet(openapiClientHandle_t *client_handle, 
                                uint32_t vlanId, 
                                OPEN_FILTER_VLAN_FILTER_MODE_t mode)
{
  open_error_t result;

  if ((result = openapiFilterVlanFilteringModeSet(client_handle, vlanId, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets multicast address filtering "
           "configuration for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("Multicast vlan filtering mode is successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets multicast address filtering configuration for the specified vlan.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    vlanId        @b{(input)}  VLAN ID
* @param    mode          @b{(output)} Filtering mode
*
* @returns  none
*
* @end
*********************************************************************/
void filterVlanFilteringModeGet(openapiClientHandle_t *client_handle, 
                                uint32_t vlanId, 
                                OPEN_FILTER_VLAN_FILTER_MODE_t *mode)
{
  open_error_t result;

  if ((result = openapiFilterVlanFilteringModeGet(client_handle, vlanId, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets multicast address filtering "
           "configuration for the specified vlan. (result = %d)\n", result);
  }
  else
  {
    printf("Multicast vlan filtering mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Checks if the interface is valid for filtering.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    ifNum         @b{(input)}  Interface number
* @param    status        @b{(output)} Validity of interface for MAC filtering
*
* @returns  none
*
* @end
*********************************************************************/
void filterIsValidIntf(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                       OPEN_BOOL_t *status)
{
  open_error_t result;

  if ((result = openapiFilterIsValidIntf(client_handle, ifNum, status)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to checks if the interface is valid"
           " for filtering. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %u is %svalid MAC filter interafce.\n", ifNum,
           (*status == OPEN_TRUE) ? "" : "not ");
  }
  return;
}

/*********************************************************************
* @purpose  Sets the stream id of an existing macfilter with given MAC address and vlan id.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    streamId      @b{(input)}  802.1qci stream-id
*
* @returns  none
*
* @end
*********************************************************************/
void filterStreamSet(openapiClientHandle_t *client_handle, char *macAddr,
                     uint32_t vlanId, uint32_t streamId)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  if ((result = openapiFilterStreamSet(client_handle, &bufId, vlanId, streamId)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the stream id of an existing macfilter "
           "with given MAC address and vlan id. (result = %d)\n", result);
  }
  else
  {
    printf("Filter stream set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the configured stream id.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
* @param    streamId      @b{(output)} 802.1qci stream-id
*
* @returns  none
*
* @end
*********************************************************************/
void filterStreamGet(openapiClientHandle_t *client_handle, 
                     char *macAddr, uint32_t vlanId, 
                     uint32_t *streamId)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);


  if ((result = openapiFilterStreamGet(client_handle, &bufId, vlanId, 
                                       streamId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the configured stream id."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Filter stream ID is %u.\n", *streamId);
  }
  return;
}

/*********************************************************************
* @purpose  Gets list of filter destination interfaces for a MAC static filter.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    macAddr       @b{(input)}  MAC address
* @param    vlanId        @b{(input)}  VLAN ID
*
* @returns  none
*
* @end
*********************************************************************/
void filterDstFiltIntfListGet(openapiClientHandle_t *client_handle, 
                              char *macAddr, uint32_t vlanId)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  char intfStr[INTF_LIST_STR_SIZE] = "";
  open_buffdesc bufId, intfList;
  uint32_t numOfDstFiltIntf;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (mac);

  intfList.pstart = intfStr;
  intfList.size   = INTF_LIST_STR_SIZE;

  if ((result = openapiFilterDstFiltIntfListGet(client_handle, &bufId, vlanId,
                                                &numOfDstFiltIntf, &intfList)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets list of source interfaces for a"
           " MAC static filter. (result = %d)\n", result);
  }
  else
  {
    printf("List of destination filter interfaces for a filter is:\n%s\n", 
           intfStr);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Static Filter
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, val;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Static Filter example", &client_handle))
      != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", 
           result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Static Filter API "
              "example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) 
      == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        filterAdminModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        filterAdminModeGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        filterCreate(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        filterRemove(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 5)
      {
        filterSrcIntfListModify(&client_handle, argv[2], atoi(argv[3]), 
                                argv[4]);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 5)
      {
        filterSrcIntfAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 5)
      {
        filterSrcIntfDelete(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 5)
      {
        filterDstIntfAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 5)
      {
        filterDstIntfDelete(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 7)
      {
        filterDstIntfListAdd(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]), 
                             argv[5], argv[6]);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 7)
      {
        filterDstIntfListDelete(&client_handle, argv[2], atoi(argv[3]), 
                                atoi(argv[4]), argv[5], argv[6]);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 2)
      {
        filterFirstGet(&client_handle);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        filterNextGet(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 4)
      {
        filterSrcIntfListGet(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 4)
      {
        filterDstIntfListGet(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 2)
      {
        filterCountGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 2)
      {
        filterCountersGet(&client_handle);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 4)
      {
        filterIsConfigured(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 2)
      {
        filterSizeGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 3)
      {
        filterIsIntfInAnyFilter(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 3)
      {
        filterPortParmCanSet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
        filterCheckMacAddrTypeMulticast(&client_handle, argv[2], &val);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 3)
      {
        filterFilteringModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 4)
      {
        filterVlanFilteringModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 3)
      {
        filterVlanFilteringModeGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 3)
      {
        filterIsValidIntf(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 27:
     if (argc == 5)
      {
        filterStreamSet(&client_handle, argv[2], atoi(argv[3]), atoi(argv[4]));
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 4)
      {
        filterStreamGet(&client_handle, argv[2], atoi(argv[3]), &val);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 4)
      {
        filterDstFiltIntfListGet(&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Static Filter API example "
              "application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

