/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  storm_control_example.c
*
* @purpose   Storm Control OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    05/02/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_storm_control.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the storm_control
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
/*******************************************************************
*
* @brief  This function prints the Storm Control Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printStormControlAppMenu()
{
  printf("Usage:  storm_control_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set the global storm control mode state: storm_control_example 1 <storm-control mode> <state> \n");
  printf("Test 2: Get the global storm control mode state: storm_control_example 2 <storm-control mode> \n");
  printf("Test 3: Set the global storm control rate unit and threshold: storm_control_example 3 <storm-control mode> <rate-unit> <threshold> \n");
  printf("Test 4: Set the global storm control rate unit and threshold to default: storm_control_example 4 <storm-control mode> \n");
  printf("Test 5: Get the global storm control rate unit and threshold: storm_control_example 5 <storm-control mode> \n");
  printf("Test 6: Set the storm control mode state on an interface: storm_control_example 6 <storm-control mode> <interface> <state> \n");
  printf("Test 7: Get the storm control mode state on an interface: storm_control_example 7 <storm-control mode> <interface> \n");
  printf("Test 8: Set the storm control rate unit and threshold on an interface: storm_control_example 8 <storm-control mode> <interface> <rate-unit> <threshold> \n");
  printf("Test 9: Set the storm control rate unit and threshold to default on an interface: storm_control_example 9 <storm-control mode> <interface> \n");
  printf("Test 10: Get the storm control rate unit and threshold on an interface: storm_control_example 10 <storm-control mode> <interface> \n");
  printf("Test 11: Set the global storm control action: storm_control_example 11 <storm-control mode> <action> \n");
  printf("Test 12: Get the global storm control action: storm_control_example 12 <storm-control mode> \n");
  printf("Test 13: Storm Control OpEN APIs Sanity: storm_control_example 13 \n");

  return;
}

/*********************************************************************
* @purpose  Set the global storm control mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    state            @b{(input)}   Storm Control mode state
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlModeSet(openapiClientHandle_t  *clientHandle,
                         OPEN_STORM_CTRL_MODE_t mode,
                         OPEN_CONTROL_t         state)
{
  open_error_t result;

  if ((result = openapiStormControlModeSet(clientHandle, mode, state)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global storm control mode. (result = %d) \n", result);
  }
  else
  {
    printf("Storm control mode %d is set successfully. result = %d) \n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global storm control mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void stormControlModeGet(openapiClientHandle_t  *clientHandle,
                         OPEN_STORM_CTRL_MODE_t mode)
{
  open_error_t   result;
  OPEN_CONTROL_t state;

  if ((result = openapiStormControlModeGet(clientHandle, mode, &state)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global storm control mode. (result = %d) \n", result);
  }
  else
  {
    printf("Storm control mode %d state: %d result = %d) \n", mode, state, result);
  }

  return; 
}

/*********************************************************************
* @purpose  Set the global storm control rate unit and threshold.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    rateUnit         @b{(input)}   Storm Control rate unit
* @param    threshold        @b{(input)}   Storm Control threshold
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlThresholdSet(openapiClientHandle_t  *clientHandle,
                              OPEN_STORM_CTRL_MODE_t mode,
                              OPEN_RATE_UNIT_t       rateUnit,
                              uint32_t               threshold)
{
  open_error_t result;
  uint32_t     minVal, maxVal;

  result = openapiStormControlThresholdMinMaxGet(clientHandle, rateUnit, &minVal, &maxVal);

  if (result == OPEN_E_NONE)
  {
    if ((threshold < minVal) || (threshold > maxVal))
    {
      printf("Invalid Threshold value. The allowable range is %d - %d.\n", minVal, maxVal);
      result = OPEN_E_PARAM;
    }
  }

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiStormControlThresholdSet(clientHandle, mode, rateUnit, threshold)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to set global storm control rate unit and threshold. (result = %d) \n", result);
    }
    else
    {
      printf("Storm control rate unit and threshold for mode %d is set successfully. result = %d) \n", mode, result);
    }
  }

  return;
}
/*********************************************************************
* @purpose  Set the global storm control rate unit and threshold to default.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlThresholdDefaultSet(openapiClientHandle_t  *clientHandle,
                                     OPEN_STORM_CTRL_MODE_t mode)
{
  open_error_t result;

  if ((result = openapiStormControlThresholdDefaultSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global storm control rate unit and threshold to default. (result = %d) \n", result);
  }
  else
  {
    printf("Storm control rate unit and threshold for mode %d is set to default successfully. result = %d) \n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global storm control rate unit and threshold.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void stormControlThresholdGet(openapiClientHandle_t  *clientHandle,
                              OPEN_STORM_CTRL_MODE_t mode)
{
  open_error_t     result;
  OPEN_RATE_UNIT_t rateUnit;
  uint32_t         threshold;

  if ((result = openapiStormControlThresholdGet(clientHandle, mode, &rateUnit, &threshold)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global storm control rate unit and threshold for mode. (result = %d) \n", result);
  }
  else
  {
    printf("Storm Control Mode: ");
    if (mode == OPEN_STORM_CTRL_MODE_UNICAST)
    {
      printf("Unicast\n");
    }
    else if (mode == OPEN_STORM_CTRL_MODE_BROADCAST)
    {
      printf("Broadcast\n");
    }
    else if (mode == OPEN_STORM_CTRL_MODE_MULTICAST)
    {
      printf("Multicast\n");
    }

    printf("Threshold: %d ", threshold);
    if (rateUnit == OPEN_RATE_UNIT_PERCENT)
    {
      printf("percent\n");
    }
    else
    {
      printf("pps\n");
    }

  }

  return;
}

/*********************************************************************
* @purpose  Set the storm control mode state on an interface.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    interface        @b{(input)}   Interface ID
* @param    state            @b{(input)}   Storm Control mode state
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlIntfModeSet(openapiClientHandle_t  *clientHandle,
                             OPEN_STORM_CTRL_MODE_t mode,
                             uint32_t               interface,
                             OPEN_CONTROL_t         state)
{
  open_error_t result;

  if ((result = openapiStormControlIntfModeSet(clientHandle, mode, interface, state)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set storm control mode on an interface. (result = %d) \n", result);
  }
  else
  {
    printf("Storm Control mode %d is set successfully on interface %d. (result = %d) \n", mode, interface, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the storm control mode state on an interface.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    interface        @b{(input)}   Interface ID
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void stormControlIntfModeGet(openapiClientHandle_t *clientHandle,
                             OPEN_STORM_CTRL_MODE_t mode,
                             uint32_t               interface)
{
  open_error_t result;
  OPEN_CONTROL_t state;

  if ((result = openapiStormControlIntfModeGet(clientHandle, mode, interface, &state)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get storm control mode on an interface. (result = %d) \n", result);
  }
  else
  {
    printf("State of Storm Control mode %d on interface %d: %d. (result = %d) \n", mode, interface, state, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the storm control rate unit and threshold on an interface.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    interface        @b{(input)}   Interface ID
* @param    rateUnit         @b{(input)}   Storm Control rate unit
* @param    threshold        @b{(input)}   Storm Control threshold
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlIntfThresholdSet(openapiClientHandle_t  *clientHandle,
                                  OPEN_STORM_CTRL_MODE_t mode,
                                  uint32_t               interface,
                                  OPEN_RATE_UNIT_t       rateUnit,
                                  uint32_t               threshold)
{
  open_error_t result;
  uint32_t     minVal, maxVal;

  result = openapiStormControlThresholdMinMaxGet(clientHandle, rateUnit, &minVal, &maxVal);

  if (result == OPEN_E_NONE)
  {
    if ((threshold < minVal) || (threshold > maxVal))
    {
      printf("Invalid Threshold value. The allowable range is %d - %d.\n", minVal, maxVal);
      result = OPEN_E_PARAM;
    }
  }

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiStormControlIntfThresholdSet(clientHandle, mode, interface, rateUnit, threshold)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to set storm control rate unit and threshold on an interface. (result = %d) \n", result);
    }
    else
    {
      printf("Storm Control rate unit and threshold for mode %d on interface %d is set successfully. (result = %d) \n", mode, interface, result);
    }
  }

  return;
}

/*********************************************************************
* @purpose  Set the storm control rate unit and threshold to default on an interface.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    interface        @b{(input)}   Interface ID
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlIntfThresholdDefaultSet(openapiClientHandle_t *clientHandle,
                                         OPEN_STORM_CTRL_MODE_t mode,
                                         uint32_t interface)
{
  open_error_t result;

  if ((result = openapiStormControlIntfThresholdDefaultSet(clientHandle, mode, interface)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set storm control rate unit and threshold to default on an interface. (result = %d) \n", result);
  }
  else
  {
    printf("Storm Control rate unit and threshold for mode %d on interface %d is set to default successfully. (result = %d) \n", mode, interface, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the storm control rate unit and threshold on an interface.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    interface        @b{(input)}   Interface ID
* 
* @returns  none
*   
* @notes 
* 
* @end
*********************************************************************/
void stormControlIntfThresholdGet(openapiClientHandle_t  *clientHandle,
                                  OPEN_STORM_CTRL_MODE_t mode,
                                  uint32_t               interface)
{
  open_error_t     result;
  OPEN_RATE_UNIT_t rateUnit = OPEN_RATE_UNIT_NONE;
  uint32_t         threshold = 0;

  if ((result = openapiStormControlIntfThresholdGet(clientHandle, mode, interface, &rateUnit, &threshold)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get storm control rate unit and threshold on an interface for mode %d. (result = %d) \n", mode, result);
  }
  else
  {
    printf("Interface: %d\n", interface);

    printf("Storm Control Mode: ");
    if (mode == OPEN_STORM_CTRL_MODE_UNICAST)
    {
      printf("Unicast\n");
    }
    else if (mode == OPEN_STORM_CTRL_MODE_BROADCAST)
    {
      printf("Broadcast\n");
    }
    else if (mode == OPEN_STORM_CTRL_MODE_MULTICAST)
    {
      printf("Multicast\n");
    }

    printf("Threshold: %d ", threshold);
    if (rateUnit == OPEN_RATE_UNIT_PERCENT)
    {
      printf("percent\n");
    }
    else
    {
      printf("pps\n");
    }
  }

  return;
}

/*********************************************************************
* @purpose  Set the global storm control action.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* @param    action           @b{(input)}   Storm Control action
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void stormControlActionSet(openapiClientHandle_t *clientHandle, OPEN_STORM_CTRL_MODE_t mode, OPEN_POLICY_PORT_ACTION_t action)
{
  open_error_t result;

  if ((result = openapiStormControlActionSet(clientHandle, mode, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global storm control action. (result = %d) \n", result);
  }
  else
  {
    printf("Storm control action set successfully. result = %d) \n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global storm control action.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   Storm Control mode
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void stormControlActionGet(openapiClientHandle_t *clientHandle, OPEN_STORM_CTRL_MODE_t mode)
{
  open_error_t result;
  OPEN_POLICY_PORT_ACTION_t action;

  if ((result = openapiStormControlActionGet(clientHandle, mode, &action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global storm control action. (result = %d) \n", result);
  }
  else
  {
    switch (action)
    {
      case OPEN_POLICY_PORT_ACTION_SHUTDOWN:
        printf("Storm control action : Shutdown\n");
        break;
      case OPEN_POLICY_PORT_ACTION_RATELIMIT:
        printf("Storm control action : Rate Limit\n");
        break;
      case OPEN_POLICY_PORT_ACTION_TRAP:
        printf("Storm control action : Trap\n");
        break;
      case OPEN_POLICY_PORT_ACTION_NONE:
      default:
        printf("Storm control action : None\n");
        break;
    }
  }

  return; 
}

/*********************************************************************
* @purpose  Storm Control OpEN APIs Sanity.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes 
* 
* @end
*********************************************************************/
void stormControlOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t              result;
  OPEN_RATE_UNIT_t          rateUnit = OPEN_RATE_UNIT_PPS;
  uint32_t                  threshold = 0;
  uint32_t                  interface = 1;
  OPEN_STORM_CTRL_MODE_t    mode = OPEN_STORM_CTRL_MODE_UNICAST;
  OPEN_CONTROL_t            state = OPEN_DISABLE;
  OPEN_POLICY_PORT_ACTION_t action = OPEN_POLICY_PORT_ACTION_NONE;

  /* openapiStormControlModeSet() */ 
  printf("\nTesting openapiStormControlModeSet(): \n");

  result = openapiStormControlModeSet(NULL, mode, state);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlModeSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, state);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlModeSet(clientHandle, mode, 3);
  printf("Invalid storm control mode state. (result = %d)\n", result);

  printf("openapiStormControlModeSet() sanity successful.\n");

  /* openapiStormControlModeGet() */
  printf("\nTesting openapiStormControlModeGet(): \n");

  result = openapiStormControlModeGet(NULL, mode, &state);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlModeGet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, &state);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlModeGet(clientHandle, mode, NULL);
  printf("NULL storm control mode. (result = %d)\n", result);

  printf("openapiStormControlModeGet() sanity successful.\n");

  /* openapiStormControlThresholdSet() */
  printf("\nTesting openapiStormControlThresholdSet(): \n");

  result = openapiStormControlThresholdSet(NULL, mode, rateUnit, threshold);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlThresholdSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, rateUnit, threshold);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlThresholdSet(clientHandle, mode, OPEN_RATE_UNIT_NONE, threshold);
  printf("Invalid storm control rate. (result = %d)\n", result);

  result = openapiStormControlThresholdSet(clientHandle, mode, OPEN_RATE_UNIT_PERCENT, 101);
  printf("Invalid threshold value for rate unit OPEN_RATE_UNIT_PERCENT. (result = %d)\n", result);

  result = openapiStormControlThresholdSet(clientHandle, mode, OPEN_RATE_UNIT_PPS, 15000000);
  printf("Invalid threshold value for rate unit OPEN_RATE_UNIT_PPS. (result = %d)\n", result);

  printf("openapiStormControlThresholdSet() sanity successful.\n");

  /* openapiStormControlThresholdDefaultSet() */
  printf("\nTesting openapiStormControlThresholdDefaultSet(): \n");

  result = openapiStormControlThresholdDefaultSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlThresholdDefaultSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE);
  printf("Invalid storm control mode. (result = %d)\n", result);

  printf("openapiStormControlThresholdDefaultSet() sanity successful.\n");

  /* openapiStormControlThresholdGet() */
  printf("\nTesting openapiStormControlThresholdGet(): \n");

  result = openapiStormControlThresholdGet(NULL, mode, &rateUnit, &threshold); 
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlThresholdGet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, &rateUnit, &threshold); 
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlThresholdGet(clientHandle, mode, NULL, &threshold); 
  printf("NULL storm control rate. (result = %d)\n", result);

  result = openapiStormControlThresholdGet(clientHandle, mode, &rateUnit, NULL); 
  printf("NULL storm control threshold. (result = %d)\n", result);

  printf("openapiStormControlThresholdGet() sanity successful.\n");

  /* openapiStormControlIntfModeSet() */
  printf("\nTesting openapiStormControlIntfModeSet(): \n");

  result = openapiStormControlIntfModeSet(NULL, mode, interface, state);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlIntfModeSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, interface, state);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlIntfModeSet(clientHandle, mode, 250, state);
  printf("Invalid interface. (result = %d)\n", result);

  result = openapiStormControlIntfModeSet(clientHandle, mode, interface, 4);
  printf("Invalid storm control mode state. (result = %d)\n", result);

  printf("openapiStormControlIntfModeSet() sanity successful.\n");

  /* openapiStormControlIntfModeGet() */
  printf("\nTesting openapiStormControlIntfModeGet(): \n");

  result = openapiStormControlIntfModeGet(NULL, mode, interface, &state);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlIntfModeGet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, interface, &state);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlIntfModeGet(clientHandle, mode, 250, &state);
  printf("Invalid interface. (result = %d)\n", result);

  result = openapiStormControlIntfModeGet(clientHandle, mode, interface, NULL);
  printf("NULL storm control mode state. (result = %d)\n", result);

  printf("openapiStormControlIntfModeGet() sanity successful.\n");

  /* openapiStormControlIntfThresholdSet() */
  printf("\nTesting openapiStormControlIntfThresholdSet(): \n");

  result = openapiStormControlIntfThresholdSet(NULL, mode, interface, rateUnit, threshold);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, interface, rateUnit, threshold);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdSet(clientHandle, mode, 300, rateUnit, threshold);
  printf("Invalid interface. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdSet(clientHandle, mode, interface, OPEN_RATE_UNIT_NONE, threshold);
  printf("Invalid storm control rate unit. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdSet(clientHandle, mode, interface, OPEN_RATE_UNIT_PERCENT, 101);
  printf("Invalid storm control threshold for rate unit OPEN_RATE_UNIT_PERCENT. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdSet(clientHandle, mode, interface, OPEN_RATE_UNIT_PPS, 15000000);
  printf("Invalid storm control threshold for rate unit OPEN_RATE_UNIT_PPS. (result = %d)\n", result);

  printf("openapiStormControlIntfThresholdSet() sanity successful.\n");

  /* openapiStormControlIntfThresholdDefaultSet() */
  printf("\nTesting openapiStormControlIntfThresholdDefaultSet(): \n");

  result = openapiStormControlIntfThresholdDefaultSet(NULL, mode, interface);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdDefaultSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, interface);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdDefaultSet(clientHandle, mode, 1000);
  printf("Invalid interface. (result = %d)\n", result);

  printf("openapiStormControlIntfThresholdDefaultSet() sanity successful.\n");

  /* openapiStormControlIntfThresholdGet() */
  printf("\nTesting openapiStormControlIntfThresholdGet(): \n");

  result = openapiStormControlIntfThresholdGet(NULL, mode, interface, &rateUnit, &threshold);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdGet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, interface, &rateUnit, &threshold);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdGet(clientHandle, mode, 20000, &rateUnit, &threshold);
  printf("Invalid interface. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdGet(clientHandle, mode, interface, NULL, &threshold);
  printf("NULL storm control rate unit. (result = %d)\n", result);

  result = openapiStormControlIntfThresholdGet(clientHandle, mode, interface, &rateUnit, NULL);
  printf("NULL storm control threshold. (result = %d)\n", result);

  printf("openapiStormControlIntfThresholdGet() sanity successful.\n");

  /* openapiStormControlActionSet() */ 
  printf("\nTesting openapiStormControlActionSet(): \n");

  result = openapiStormControlActionSet(NULL, mode, action);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlActionSet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, action);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlActionSet(clientHandle, mode, 5);
  printf("Invalid storm control action. (result = %d)\n", result);

  printf("openapiStormControlActionSet() sanity successful.\n");

  /* openapiStormControlActionGet() */
  printf("\nTesting openapiStormControlActionGet(): \n");

  result = openapiStormControlActionGet(NULL, mode, &action);
  printf("NULL client handle. (result = %d)\n", result);

  result = openapiStormControlActionGet(clientHandle, OPEN_STORM_CTRL_MODE_NONE, &action);
  printf("Invalid storm control mode. (result = %d)\n", result);

  result = openapiStormControlActionGet(clientHandle, mode, NULL);
  printf("NULL storm control action. (result = %d)\n", result);

  printf("openapiStormControlActionGet() sanity successful.\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1, arg2, arg3, arg4;

  if (argc < 2)
  {
    printStormControlAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("storm_control_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Storm Control API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 4)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stormControlModeSet(&clientHandle, arg1, arg2);
      break;
    case 2:
      if (argc != 3)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stormControlModeGet(&clientHandle, arg1);
      break;
    case 3:
      if (argc != 5)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      stormControlThresholdSet(&clientHandle, arg1, arg2, arg3);
      break;
    case 4:
      if (argc != 3)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stormControlThresholdDefaultSet(&clientHandle, arg1);
      break;
    case 5:
      if (argc != 3)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stormControlThresholdGet(&clientHandle, arg1);
      break;
    case 6:
      if (argc != 5)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      stormControlIntfModeSet(&clientHandle, arg1, arg2, arg3);
      break;
    case 7:
      if (argc != 4)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stormControlIntfModeGet(&clientHandle, arg1, arg2);
      break;
    case 8:
      if (argc != 6)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[5]);
      stormControlIntfThresholdSet(&clientHandle, arg1, arg2, arg3, arg4);
      break;
    case 9:
      if (argc != 4)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stormControlIntfThresholdDefaultSet(&clientHandle, arg1, arg2);
      break;
    case 10:
      if (argc != 4)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stormControlIntfThresholdGet(&clientHandle, arg1, arg2);
      break;
    case 11:
      if (argc != 4)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stormControlActionSet(&clientHandle, arg1, arg2);
      break;
    case 12:
      if (argc != 3)
      {
        printStormControlAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stormControlActionGet(&clientHandle, arg1);
      break;
    case 13:
      if (argc != 2)
      {
        printStormControlAppMenu();
        exit(1);
      }
      stormControlOpENAPIsSanity(&clientHandle);
      break;
    default:
      printStormControlAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Storm Control API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
