/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  nsdp_example.c
*
* @purpose   OpEN NSDP example.
*
* @component OpEN
*
* @create    03/21/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_nsdp.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the NSDP Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets the NSDP Admin Mode: %s 1 <mode>\n", name);
  printf("Test 2: Get the NSDP Admin Mode: %s 2 \n", name);
  printf("Test 3: Run API sanity checks: %s 3 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t mode;

  printf("Testing NSDP OpEN APIs sanity:\n");

  printf("Testing openapiNsdpModeGet():\n");
  result = openapiNsdpModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiNsdpModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiNsdpModeSet():\n");
  result = openapiNsdpModeSet(NULL, mode);
  printf("NULL client handle:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets the NSDP Admin Mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  mode scc mode

*
* @returns  none
*
* @end
*********************************************************************/
void nsdpModeSet(openapiClientHandle_t *client_handle, uint32_t mode)
{
  open_error_t result;

  if ((result = openapiNsdpModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the NSDP Admin Mode. (result = %d)\n", result);
  }
  else
  {
    printf("NSDP mode is set to %d\n",mode);
  }
  return;
}


/*********************************************************************
* @purpose  Get the NSDP Admin Mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  mode  pointer to scc  mode

*
* @returns  none
*
* @end
*********************************************************************/
void nsdpModeGet(openapiClientHandle_t *client_handle, uint32_t *mode)
{
  open_error_t result;

  if ((result = openapiNsdpModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the NSDP Admin Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Get's the NSDP mode as %d",*mode);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for NSDP
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("nsdp example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting NSDP API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
	      uint32_t mode;
	      mode  = atoi(argv[2]);
        nsdpModeSet(&clientHandle, mode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
	      uint32_t mode = 0;
        nsdpModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping NSDP  API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

