/*********************************************************************
*
*  Copyright 2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  switch_cpu_example.c
*
* @purpose   Switch CPU OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    08/31/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include <stdio.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_switch_cpu.h"

/*******************************************************************
*
* @brief  This function prints the example application usage.
*
* @param    name     @b{(input)}   app name (argv[0])
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("Usage:  %s <test#> <arg1> <arg2> ...\n", name);
  printf("Test 1: Get max number of tracked periods: %s 1\n", name);
  printf("Test 2: Get total CPU utilization for tracked periods: %s 2\n", name);
  printf("Test 3: Run OpEN APIs Sanity: %s 3\n", name);
  printf("Test 4: Get CPU utilization monitor parameters: %s 4 <param no>\n", name);
  printf("Test 5: Set CPU utilization monitor parameters: %s 5 <param no> <value>\n", name);
  printf("Test 6: Get task CPU utilization for tracked periods: %s 6\n", name);

  return;
}

/*********************************************************************
* @purpose  Get max number of tracked utilization periods.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    max              @b{(output)}  max number of tracked utilization periods 
* 
* @returns  OPEN_E_NONE  on success
* @returns  OPEN_E_PARAM invalid argument
*   
* @end
*********************************************************************/
open_error_t getMaxPeriods(openapiClientHandle_t *clientHandle, uint32_t *max)
{
  open_error_t result;

  if ((result = openapiCpuUtilMaxPeriodsGet(clientHandle, max)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get max periods. (result = %d)\n", result);
  }
  else
  {
    printf("Max periods is %d (result = %d)\n", *max, result);
  }

  return result;
}

/*********************************************************************
* @purpose  Get Cpu util monitor value.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    param_type       @b{(input)}   parameter type
* @param    param_val        @b{(output)}  parameter value
*
* @returns  OPEN_E_NONE  on success
* @returns  OPEN_E_PARAM invalid argument
*
* @end
*********************************************************************/
open_error_t getCpuUtilMonitorParam(openapiClientHandle_t *clientHandle,
                                    OPEN_CPU_UTIL_MONITOR_PARAMS_t param_Type)
{
  open_error_t result;
  uint32_t param_val;

  if ((result = openapiCpuUtilMonitorParamGet(clientHandle, param_Type, &param_val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get CPU utilization monitor. (result = %d)\n", result);
  }
  else
  {
    printf("CPU util monitor param %d is %d\n", param_Type, param_val);
  }

  return result;
}

/*********************************************************************
* @purpose  Set Cpu util monitor value.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    param_type       @b{(input)}   parameter type
* @param    param_val        @b{(input)}   parameter value
*
* @returns  OPEN_E_NONE  on success
* @returns  OPEN_E_PARAM invalid argument
*
* @end
*********************************************************************/
open_error_t setCpuUtilMonitorParam(openapiClientHandle_t *clientHandle,
                                    OPEN_CPU_UTIL_MONITOR_PARAMS_t param_Type,
                                    uint32_t param_val)
{
  open_error_t result;

  if ((result = openapiCpuUtilMonitorParamSet(clientHandle, param_Type, param_val)) != OPEN_E_NONE)
  {
    printf("Couldn't set CPU util monitor param %d is %d\n", param_Type, param_val);
    printf("Bad return code trying to set CPU util monitor. (result = %d)\n", result);
  }

  return result;
}

/*********************************************************************
* @purpose  Get tracked utilization periods.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @end
*********************************************************************/
void getTrackedUtilizationPeriods(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t max;
  open_buffdesc buffdesc;
  OPEN_CPU_PERIOD_UTIL_INFO_t *periods = NULL;
  OPEN_CPU_PERIOD_UTIL_INFO_t *p;
  int i;

  result = getMaxPeriods(clientHandle, &max);
  if (result == OPEN_E_NONE) 
  {
    periods = (OPEN_CPU_PERIOD_UTIL_INFO_t *) malloc(max * sizeof(OPEN_CPU_PERIOD_UTIL_INFO_t));
    if (periods == (OPEN_CPU_PERIOD_UTIL_INFO_t *) NULL)
    {
      printf("Unable to allocate array\n");
    }
    else
    {
      buffdesc.pstart = (void *) periods;
      buffdesc.size = max * sizeof(OPEN_CPU_PERIOD_UTIL_INFO_t);
      result = openapiTotalCpuUtilArrayGet(clientHandle, &buffdesc);
      if (result != OPEN_E_NONE) 
      {
        printf("Bad return code trying to get CPU utilization info. (result = %d)\n", result);
      }
      else
      {
        p = periods;
        printf("Period Utilization\n");
        printf("------ -----------\n");

        for (i = 0; i < buffdesc.size / sizeof(OPEN_CPU_PERIOD_UTIL_INFO_t); i++)
        {
          printf("%6d %d.%-4d%%\n", p->timePeriod, p->utilization / OPEN_CPU_UTIL_SCALAR, p->utilization % OPEN_CPU_UTIL_SCALAR);
          p++;
        }
      }
      free(periods);
    }
  }
  else
  {
    printf("Bad return code trying to get max number of tracked utilization periods. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get task cpu utilization
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getTaskCPUUtilGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc desc;
  char *buf;

  desc.size = (OPEN_CPU_TASK_COUNT_AVERAGE + OPEN_CPU_TASK_UTIL_HEADER_ROWS) *
                                                    OPEN_CPU_TASK_UTIL_SIZE;

  if ((buf = malloc(desc.size)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  desc.pstart = buf;
  if ((result = openapiTaskUtilPrint(clientHandle, &desc)) != OPEN_E_NONE)
  {
    printf("Bad return code with task cpu utilization get. (result = %d)\n", result);
  }
  else
  {
    printf("System Task Utilization: \n");
    printf("%s\n", buf);
  }

  free(buf);
  return;
}

/*********************************************************************
* @purpose  Run sanity check on APIs.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @note
*
* @end
*********************************************************************/
void runSanityChecks(openapiClientHandle_t *clientHandle)
{
  open_buffdesc buffdesc;
  OPEN_CPU_PERIOD_UTIL_INFO_t *p;
  uint32_t max;
  open_error_t result;
  uint32_t constBufsize;       
  uint32_t val;
  char *buf;

  /* openapiCpuUtilMaxPeriodsGet */
  printf("Testing openapiCpuUtilMaxPeriodsGet(): \n");

  result = openapiCpuUtilMaxPeriodsGet(NULL, &max);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiCpuUtilMaxPeriodsGet(clientHandle, NULL);
  printf("NULL argument. (result = %d)\n", result);

  /* openapiTotalCpuUtilArrayGet */
  printf("Testing openapiTotalCpuUtilArrayGet(): \n");

  openapiCpuUtilMaxPeriodsGet(clientHandle, &max);

  constBufsize = max * sizeof(OPEN_CPU_PERIOD_UTIL_INFO_t);  /* constant for remainder of this function */

  p = (OPEN_CPU_PERIOD_UTIL_INFO_t *) malloc(constBufsize);

  buffdesc.pstart = (void *) p;
  buffdesc.size = constBufsize;

  result = openapiTotalCpuUtilArrayGet(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  buffdesc.pstart = (void *) NULL;
  buffdesc.size = constBufsize;

  result = openapiTotalCpuUtilArrayGet(clientHandle, &buffdesc);
  printf("NULL buffdesc pstart. (result = %d)\n", result);

  buffdesc.pstart = (void *) p;
  buffdesc.size = constBufsize - 1;
  
  result = openapiTotalCpuUtilArrayGet(clientHandle, &buffdesc);
  printf("NULL buffdesc invalid size. (result = %d)\n", result);
  free(p);

  /* openapiCpuUtilMonitorParamGet */
  printf("Testing openapiCpuUtilMonitorParamGet(): \n");
  result = openapiCpuUtilMonitorParamGet(NULL, OPEN_CPU_UTIL_MONITOR_RISING_THRESHOLD_PARAM, &val);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiCpuUtilMonitorParamGet(clientHandle, OPEN_CPU_UTIL_MONITOR_RISING_THRESHOLD_PARAM, NULL);
  printf("NULL value Handle. (result = %d)\n", result);

  result = openapiCpuUtilMonitorParamGet(clientHandle, OPEN_CPU_UTIL_MONITOR_LAST_PARAM, &val);
  printf("Invalid parameter type. (result = %d)\n", result);

  /* openapiCpuUtilMonitorParamSet */
  printf("Testing openapiCpuUtilMonitorParamSet(): \n");
  result = openapiCpuUtilMonitorParamSet(NULL, OPEN_CPU_UTIL_MONITOR_RISING_THRESHOLD_PARAM, 0);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiCpuUtilMonitorParamSet(clientHandle, OPEN_CPU_UTIL_MONITOR_LAST_PARAM, 0);
  printf("Invalid parameter type. (result = %d)\n", result);

  /* openapiTaskUtilPrint */
  printf("Testing openapiTaskUtilPrint(): \n");

  constBufsize = (OPEN_CPU_TASK_COUNT_AVERAGE + OPEN_CPU_TASK_UTIL_HEADER_ROWS) *
                                                    OPEN_CPU_TASK_UTIL_SIZE;
  buf = malloc(buffdesc.size);

  buffdesc.pstart = buf;
  buffdesc.size = constBufsize;

  result = openapiTaskUtilPrint(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  buffdesc.pstart = NULL;
  buffdesc.size = constBufsize;

  result = openapiTaskUtilPrint(clientHandle, &buffdesc);
  printf("NULL buffdesc pstart. (result = %d)\n", result);

  buffdesc.pstart = buf;
  buffdesc.size = 100;

  result = openapiTaskUtilPrint(clientHandle, &buffdesc);
  printf("buffdesc with invalid size. (result = %d)\n", result);
  free(buf);

}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  uint32_t max;
  open_buffdesc switch_os_revision;
  open_buffdesc switch_hw_version;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[OPENAPI_OS_REVISION_STR_SIZE];
  char switch_hw_version_string[OPENAPI_HW_VERSION_STR_SIZE];
  OPEN_CPU_UTIL_MONITOR_PARAMS_t param;
  uint32_t val;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting CPU utilization tracking example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  switch_hw_version.pstart = switch_hw_version_string;
  switch_hw_version.size = sizeof(switch_hw_version_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS Version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  if (openapiHardwareVersionGet(&clientHandle, &switch_hw_version) == OPEN_E_NONE)
    printf("Network Processing Device = %s\n", switch_hw_version_string);
  else
    printf("Network processing device retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getMaxPeriods(&clientHandle, &max);
      break;
    case 2:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getTrackedUtilizationPeriods(&clientHandle);
      break;
    case 3:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      runSanityChecks(&clientHandle);
      break;
    case 4:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      param = atoi(argv[2]);
      getCpuUtilMonitorParam(&clientHandle, param);
      break;
    case 5:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      param = atoi(argv[2]);
      val = atoi(argv[3]);
      setCpuUtilMonitorParam(&clientHandle, param, val);
      break;
    case 6:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getTaskCPUUtilGet(&clientHandle);
      break;
    default:
      printAppMenu(argv[0]);
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping CPU utilization tracking example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
