/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename tcam_usage.c
*
* @purpose TCAM Hardware Usage API example
*
* @detail  Example application that uses the OpEN API to report
*          the current TCAM utilization.
*
* @component TCAM API Example
*
* @comments none
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_tcam.h"



/*******************************************************************
*
* @brief  Display TCAM utilization.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int openTcamUtilizationShow(openapiClientHandle_t *clientHandle)
{

  open_tcamHardwareGroupUsage_t group_usage;
  uint32_t unit = 0;
  uint32_t asic_id = 0;
  uint32_t group_id = 0;
  uint32_t stage = 0;
  open_error_t rc;
  uint32_t last_unit = ~0, last_asic_id = ~0, last_stage = ~0;
  char *stage_str;

  do 
  {
    rc = openapiTcamHardwareGroupUsageGetNext(clientHandle, &unit, &asic_id, 
                                               &stage, &group_id, &group_usage);
    if (rc != OPEN_E_NONE)
    {
      break;
    }
    if (last_unit != unit)
    {
      last_unit = unit;
      printf ("Unit:%d\n", last_unit);
    }
    if (last_asic_id != asic_id)
    {
      last_asic_id = asic_id;
      printf ("  ASIC:%d\n", last_asic_id);
    }
    if (last_stage != stage)
    {
      last_stage = stage;
      switch (last_stage)
      {
        case OPEN_TCAM_STAGE_LOOKUP:
           stage_str = "LOOKUP";
        break;
        case OPEN_TCAM_STAGE_INGRESS:
           stage_str = "INGRESS";
        break;
        case OPEN_TCAM_STAGE_EGRESS:
           stage_str = "EGRESS";
        break;
        default:
           stage_str = "UNKNOWN"; 
      }
        
      printf ("    %s\n", stage_str);
    }
    printf("      Group:%d Rules:%d/%d  Counters:%d/%d  Meters:%d/%d\n",
              group_id, 
              group_usage.in_use_rules,
              group_usage.total_rules,
              group_usage.in_use_counters,
              group_usage.total_counters,
              group_usage.in_use_meters,
              group_usage.total_meters
              );
#if 0
    printf("      Group:%d In-Use-Rules:%d Total-Rules:%d\n",
              group_id, 
              group_usage.in_use_rules,
              group_usage.total_rules
              );
    printf("      In-Use-Counters:%d Total-Counters:%d In-Use-Meters:%d Total-Meters:%d\n",
              group_usage.in_use_counters,
              group_usage.total_counters,
              group_usage.in_use_meters,
              group_usage.total_meters
              );
    printf("      Slice-Width-Physical:%d Intraslice-Mode:%d Natural-Depth:%d\n",
              group_usage.slice_width_physical,
              group_usage.intraslice_mode_enable,
              group_usage.natural_depth
              );
#endif
  }
  while (1); 
  printf("\n\n");

  return 0;
}

/*******************************************************************
*
* @brief  Example TCAM API: This is the main function that will demonstrate 
*         TCAM OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("tcam_utilization", &clientHandle)) != OPEN_E_NONE)
    {
      printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
      exit (2);
    }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
    {
      sleep (1);
    }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting TCAM Utilization application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }


  printf ("\n");

  return openTcamUtilizationShow(&clientHandle);
}
