/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.                 
*
**********************************************************************
*
* @filename  timeranges_example.c
*
* @purpose   Example application to Time Ranges information using OpEN APIs
*
* @component OpEN
*
* @comments
*
* @create    09/26/2017
*
* @end
*
**********************************************************************/

#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_timeranges.h"

/*******************************************************************
*
* @brief  This function prints the Time Ranges Application Menu.
*
* @param  none
*
* @returns  none
*
* @end 
*********************************************************************/
void printTimeRangesAppMenu()
{
  printf("Usage: timeranges_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Sanity: timeranges_example 1 \n");
  printf("Test 2: Set the admin mode for time ranges: timeranges_example 2 <mode> \n");
  printf("Test 3: Get the admin mode for time ranges: timeranges_example 3 \n");
  printf("Test 4: Add a time range: timeranges_example 4 <name> \n");
  printf("Test 5: Delete a time range: timeranges_example 5 <name> \n");
  printf("Test 6: Get the configured time ranges: timeranges_example 6 \n");
  printf("Test 7: Get the current number of time ranges configured: timeranges_example 7 \n");
  printf("Test 8: Get the maximum number of time ranges that can be configured: timeranges_example 8 \n");
  printf("Test 9: Get the time range status: timeranges_example 9 <name> \n");
  printf("Test 10: Get the current number of entries configured for the given time range: timeranges_example 10 <name> \n");
  printf("Test 11: Get the absolute time range entry index for the given time range: timeranges_example 11 <name> \n");
  printf("Test 12: Set absolute start time for the given time range: timeranges_example 12 <name> \n");
  printf("Test 13: Get absolute start time for the given time range: timeranges_example 13 <name> \n");
  printf("Test 14: Set absolute end time for the given time range: timeranges_example 14 <name> \n");
  printf("Test 15: Get absolute end time for the given time range: timeranges_example 15 <name> \n");
  printf("Test 16: Delete an entry from an existing time range: timeranges_example 16 <name> <index> \n");
  printf("Test 17: Set periodic start time for the given time range: timeranges_example 17 <name> \n");
  printf("Test 18: Get periodic start time for the given time range: timeranges_example 18 <name> <index> \n");
  printf("Test 19: Set periodic end time for the given time range: timeranges_example 19 <name> <index> \n");
  printf("Test 20: Get periodic end time for the given time range: timeranges_example 20 <name> <index> \n");

  return;
}

/*********************************************************************
* @purpose  Sanity test Time Range information  
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void timeRangeSanityTest(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode = OPEN_ENABLE;
  open_buffdesc buffdesc;
  open_buffdesc tempBuf;
  char buf[256];
  char tmpBuf[256];  
  uint32_t val = 0;
  uint32_t invalidVal = 101;

  memset(buf, 0, sizeof(buf));
  buffdesc.pstart = buf;
  buffdesc.size = sizeof(buf);

  memset(tmpBuf, 0, sizeof(tmpBuf));
  tempBuf.pstart = tmpBuf;
  tempBuf.size = sizeof(tmpBuf);

  printf("Testing TimeRanges OpEN APIs sanity:\n\n"); 

  /* openapiTimeRangeAdminModeSet */
  printf("Testing openapiTimeRangeAdminModeSet() \n");

  result = openapiTimeRangeAdminModeSet(NULL, mode);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAdminModeSet(clientHandle, 5);
  printf("Invalid mode:(result = %d)\n", result); 

  printf("openapiTimeRangeAdminModeSet() sanity successful \n");

  /* openapiTimeRangeAdminModeGet */
  printf("Testing openapiTimeRangeAdminModeGet() \n");

  result = openapiTimeRangeAdminModeGet(NULL, &mode);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAdminModeGet(clientHandle, NULL);
  printf("Invalid mode:(result = %d)\n", result); 

  printf("openapiTimeRangeAdminModeGet() sanity successful \n");

  /* openapiTimeRangeAdd */
  printf("Testing openapiTimeRangeAdd() \n");

  result = openapiTimeRangeAdd(NULL, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAdd(clientHandle, NULL);
  printf("NULL name:(result = %d)\n", result); 

  printf("openapiTimeRangeAdd() sanity successful \n");

  /* openapiTimeRangeDelete */
  printf("Testing openapiTimeRangeDelete() \n");

  result = openapiTimeRangeDelete(NULL, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeDelete(clientHandle, NULL);
  printf("NULL name:(result = %d)\n", result); 

  printf("openapiTimeRangeDelete() sanity successful \n");

  /* openapiTimeRangeCurrNumGet */
  printf("Testing openapiTimeRangeCurrNumGet() \n");

  result = openapiTimeRangeCurrNumGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeCurrNumGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeCurrNumGet() sanity successful \n");

  /* openapiTimeRangeMaxNumGet */
  printf("Testing openapiTimeRangeMaxNumGet() \n");

  result = openapiTimeRangeMaxNumGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeMaxNumGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeMaxNumGet() sanity successful \n");

  /* openapiTimeRangeNameFirstGet */
  printf("Testing openapiTimeRangeNameFirstGet() \n");

  result = openapiTimeRangeNameFirstGet(NULL, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeNameFirstGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeNameFirstGet() sanity successful \n");

  /* openapiTimeRangeNameNextGet */
  printf("Testing openapiTimeRangeNameNextGet() \n");

  result = openapiTimeRangeNameNextGet(NULL, &tempBuf, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeNameNextGet(clientHandle, NULL, &buffdesc);
  printf("NULL value:(result = %d)\n", result);

  result = openapiTimeRangeNameNextGet(clientHandle, &tempBuf, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeNameNextGet() sanity successful \n");

  /* openapiTimeRangeStatusGet */
  printf("Testing openapiTimeRangeStatusGet() \n");

  result = openapiTimeRangeStatusGet(NULL, &buffdesc, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeStatusGet(clientHandle, NULL, &val);
  printf("NULL value:(result = %d)\n", result);

  result = openapiTimeRangeStatusGet(clientHandle, &buffdesc, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeStatusGet() sanity successful \n");

  /* openapiTimeRangeNumEntriesGet */
  printf("Testing openapiTimeRangeNumEntriesGet() \n");

  result = openapiTimeRangeNumEntriesGet(NULL, &buffdesc, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeNumEntriesGet(clientHandle, NULL, &val);
  printf("NULL value:(result = %d)\n", result);

  result = openapiTimeRangeNumEntriesGet(clientHandle, &buffdesc, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeNumEntriesGet() sanity successful \n");

  /* openapiTimeRangeAbsoluteEntryNumGet */
  printf("Testing openapiTimeRangeAbsoluteEntryNumGet() \n");

  result = openapiTimeRangeAbsoluteEntryNumGet(NULL, &buffdesc, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryNumGet(clientHandle, NULL, &val);
  printf("NULL value:(result = %d)\n", result);

  result = openapiTimeRangeAbsoluteEntryNumGet(clientHandle, &buffdesc, NULL);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeAbsoluteEntryNumGet() sanity successful \n");

  /* openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd */
  printf("Testing openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd() \n");

  result = openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd(NULL, &tempBuf, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd(clientHandle, NULL, &buffdesc);
  printf("NULL value:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd(clientHandle, &tempBuf, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd() sanity successful \n");

  /* openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd */
  printf("Testing openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd() \n");

  result = openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd(NULL, &tempBuf, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd(clientHandle, NULL, &buffdesc);
  printf("NULL value:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd(clientHandle, &tempBuf, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd() sanity successful \n");

  /* openapiTimeRangeAbsoluteEntryStartDateAndTimeGet */
  printf("Testing openapiTimeRangeAbsoluteEntryStartDateAndTimeGet() \n");

  result = openapiTimeRangeAbsoluteEntryStartDateAndTimeGet(NULL, &tempBuf, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryStartDateAndTimeGet(clientHandle, NULL, &buffdesc);
  printf("NULL value:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryStartDateAndTimeGet(clientHandle, &tempBuf, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiTimeRangeAbsoluteEntryStartDateAndTimeGet() sanity successful \n");

  /* openapiTimeRangeAbsoluteEntryEndDateAndTimeGet */
  printf("Testing openapiTimeRangeAbsoluteEntryEndDateAndTimeGet() \n");

  result = openapiTimeRangeAbsoluteEntryEndDateAndTimeGet(NULL, &tempBuf, &buffdesc);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryEndDateAndTimeGet(clientHandle, NULL, &buffdesc);
  printf("NULL value:(result = %d)\n", result); 

  result = openapiTimeRangeAbsoluteEntryEndDateAndTimeGet(clientHandle, &tempBuf, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiTimeRangeAbsoluteEntryEndDateAndTimeGet() sanity successful \n");

  /* openapiTimeRangeEntryDelete */
  printf("Testing openapiTimeRangeEntryDelete() \n");

  result = openapiTimeRangeEntryDelete(NULL, &buffdesc, val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiTimeRangeEntryDelete(clientHandle, NULL, val);
  printf("NULL value:(result = %d)\n", result);

  result = openapiTimeRangeEntryDelete(clientHandle, &buffdesc, invalidVal);
  printf("NULL value:(result = %d)\n", result);

  printf("openapiTimeRangeEntryDelete() sanity successful \n");

  return;      
}

/*********************************************************************
* @purpose  Enables/Disables Time Ranges Admin Mode.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    mode           @b{(input)} time ranges admin mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangeAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;
      
  if ((result = openapiTimeRangeAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to set time ranges admin mode. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Time ranges admin mode set successfully : %u \n", mode);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets Time Ranges Admin Mode.  
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void timeRangeAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiTimeRangeAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get time ranges admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Time Ranges admin mode: %u \n", mode);
  }
  return;
}

/*********************************************************************
* @purpose  Creates a Time Range.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangeAdd(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;
      
  if ((result = openapiTimeRangeAdd(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to create time range. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Time Ranges created successfully : %s \n", str);
  }    
  return;
}

/*********************************************************************
* @purpose  Deletes a Time Range.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangeDelete(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;
      
  if ((result = openapiTimeRangeDelete(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to delete time range. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Time Range deleted successfully : %s \n", str);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets time ranges configured.  
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void timeRangeNameListGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  open_buffdesc tempBuff;
  char tempStr[OPEN_TIMERANGE_NAME_LEN_MAX];
  uint32_t count = 0;

  memset(str, 0, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = sizeof(str);

  memset(tempStr, 0, sizeof(tempStr));
  tempBuff.pstart = tempStr;
  tempBuff.size = sizeof(tempStr);

  result = openapiTimeRangeNameFirstGet(clientHandle, &buffDesc);

  while (result == OPEN_E_NONE)
  {
    count++;
    if (count == 1)
    {
      printf("List of time ranges configured : \n");
    }  
    printf("%d : %s \n", count, str);
    strncpy(tempStr, str, sizeof(tempStr));
    result = openapiTimeRangeNameNextGet(clientHandle, &tempBuff, &buffDesc);
  }  

  if (count == 0)
  {
    printf("No time ranges configured. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Gets the current number of time ranges configured.  
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void timeRangeCurrNumGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t val = 0;

  if ((result = openapiTimeRangeCurrNumGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get current number of time ranges configured. (result = %d)\n", result);
  }
  else
  {
    printf("Current number of time ranges configured: %u \n", val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the maximum number of time ranges that can be configured.  
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void timeRangeMaxNumGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t val = 0;

  if ((result = openapiTimeRangeMaxNumGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum number of time ranges that can be configured. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of time ranges that can be configured: %u \n", val);
  }
  return;
}

/*********************************************************************
* @purpose  Gets status for the given time range.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangeStatusGet(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  OPEN_TIMERANGESTATUS_t val;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;
      
  if ((result = openapiTimeRangeStatusGet(clientHandle, &buffDesc, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get time range status. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Time Range status : %s %u \n", str, val);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets current number of entries configured for the specified time range.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangeNumEntriesGet(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  uint32_t val = 0;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;
      
  if ((result = openapiTimeRangeNumEntriesGet(clientHandle, &buffDesc, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get number of entries configured for the given time range. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Number of entries configured for the given Time Range : %s %u \n", str, val);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets the absolute time range entry number for the given time range.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangeAbsoluteEntryNumGet(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  uint32_t val = 0;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;
      
  if ((result = openapiTimeRangeAbsoluteEntryNumGet(clientHandle, &buffDesc, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get absolute entry configured for the given time range. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Absolute entry configured for the given Time Range : %s %u \n", str, val);
  }    
  return;
}

/*********************************************************************
* @purpose  Sets start date and time for the given absolute time range entry.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangeAbsoluteEntryStartDateAndTimeAdd(openapiClientHandle_t *clientHandle, 
                                               char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  open_buffdesc time;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  char timeStr[OPEN_TIMERANGE_DATE_TIME_BUF_LEN];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(timeStr, 0, sizeof(timeStr)); 
  strncpy(timeStr, "2016-05-05 06:07", sizeof(timeStr));
  time.pstart = timeStr;
  time.size = strlen(timeStr) + 1;

  if ((result = openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd(clientHandle, &buffDesc, &time)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add absolute entry start date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Absolute start date and time added successfully for the given time range : %s %s \n", str, timeStr);
  }    
  return;
}

/*********************************************************************
* @purpose  Sets end date and time for the given absolute time range entry.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangeAbsoluteEntryEndDateAndTimeAdd(openapiClientHandle_t *clientHandle, 
                                             char *name) 
{
  open_error_t result;
  open_buffdesc buffDesc;
  open_buffdesc time;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  char timeStr[OPEN_TIMERANGE_DATE_TIME_BUF_LEN];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(timeStr, 0, sizeof(timeStr)); 
  strncpy(timeStr, "2018-05-05 06:07", sizeof(timeStr));
  time.pstart = timeStr;
  time.size = strlen(timeStr) + 1;

  if ((result = openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd(clientHandle, &buffDesc, &time)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add absolute entry end date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Absolute end date and time added successfully for the given time range : %s %s \n", str, timeStr);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets start date and time for the given absolute time range entry.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangeAbsoluteEntryStartDateAndTimeGet(openapiClientHandle_t *clientHandle, 
                                               char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  open_buffdesc time;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  char timeStr[OPEN_TIMERANGE_DATE_TIME_BUF_LEN];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(timeStr, 0, sizeof(timeStr)); 
  time.pstart = timeStr;
  time.size = sizeof(timeStr);

  if ((result = openapiTimeRangeAbsoluteEntryStartDateAndTimeGet(clientHandle, &buffDesc, &time)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add absolute entry start date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Absolute start date and time added successfully for the given time range : %s %s \n", str, timeStr);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets end date and time for the given absolute time range entry.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangeAbsoluteEntryEndDateAndTimeGet(openapiClientHandle_t *clientHandle, 
                                             char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  open_buffdesc time;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  char timeStr[OPEN_TIMERANGE_DATE_TIME_BUF_LEN];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(timeStr, 0, sizeof(timeStr)); 
  time.pstart = timeStr;
  time.size = sizeof(timeStr);

  if ((result = openapiTimeRangeAbsoluteEntryEndDateAndTimeGet(clientHandle, &buffDesc, &time)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add absolute entry end date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Absolute end date and time added successfully for the given time range : %s %s \n", str, timeStr);
  }    
  return;
}

/*********************************************************************
* @purpose  Set periodic start date and time at which the time range
*           becomes active.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangePeriodicEntryStartDateAndTimeSet(openapiClientHandle_t *clientHandle, 
                                               char *name) 
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  OPEN_TIMERANGE_PERIODIC_ENTRY_t val;

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&val, 0, sizeof(val));
  
  /* Date and Time */
  val.openDateAndTime.year = 2017;
  val.openDateAndTime.month = 10;
  val.openDateAndTime.day = 12;
  val.openDateAndTime.hour = 1;
  val.openDateAndTime.minute = 1;

  /* Setting pattern a weekly */
  val.pattern = OPEN_TIMERANGE_PATTERN_WEEKLY;
  val.dayMask = OPEN_TIMERANGE_DAY_SUN | OPEN_TIMERANGE_DAY_WED | OPEN_TIMERANGE_DAY_FRI;

  val.frequency = 5;

  if ((result = openapiTimeRangePeriodicEntryStartDateAndTimeSet(clientHandle, &buffDesc, val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add periodic entry start date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Periodic start date and time added successfully for the given time range : %s \n", str);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets start date and time for the given periodic time range entry.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
* @param    index          @b{(input)} time range entry index
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangePeriodicEntryStartDateAndTimeGet(openapiClientHandle_t *clientHandle, 
                                               char *name,
                                               uint32_t index)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  char buf[OPEN_TIMERANGE_BUF_LEN];
  OPEN_TIMERANGE_PERIODIC_ENTRY_t val;

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&val, 0, sizeof(val));
  memset(buf, 0, sizeof(buf));

  if ((result = openapiTimeRangePeriodicEntryStartDateAndTimeGet(clientHandle, &buffDesc, index, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add periodic entry start date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Periodic start date and time added successfully for the given time range : %s %u \n", str, index);
    printf("Date: %04u-%02u-%02u %02u:%02u \n", val.openDateAndTime.year, 
            val.openDateAndTime.month, val.openDateAndTime.day,
            val.openDateAndTime.hour, val.openDateAndTime.minute);
    
    if (val.pattern == OPEN_TIMERANGE_PATTERN_WEEKLY)
    {
      snprintf(buf, sizeof(buf), "WEEKLY - 0x%02x", val.dayMask);
    } 
    else if (val.pattern == OPEN_TIMERANGE_PATTERN_DAILY) 
    {
      snprintf(buf, sizeof(buf), "DAILY - 0x%02x", val.dayMask);
    } 
    else if (val.pattern == OPEN_TIMERANGE_PATTERN_MONTHLY) 
    {
      snprintf(buf, sizeof(buf), "MONTLY - 0x%02x", val.dayMask);
    } 
    printf("Pattern: %s \n", buf);

    printf("Frequency: %02u \n", val.frequency);
  }    
  return;
}

/*********************************************************************
* @purpose  Set periodic end date and time at which the time range
*           becomes inactive.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
* @param    index          @b{(input)} time range entry index
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangePeriodicEntryEndDateAndTimeSet(openapiClientHandle_t *clientHandle, 
                                             char *name,
                                             uint32_t index) 
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  OPEN_TIMERANGE_PERIODIC_ENTRY_t val;

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&val, 0, sizeof(val));
  
  /* Date and Time */
  val.openDateAndTime.year = 2017;
  val.openDateAndTime.month = 10;
  val.openDateAndTime.day = 22;
  val.openDateAndTime.hour = 10;
  val.openDateAndTime.minute = 10;

  /* Setting pattern as weekly */
  val.pattern = OPEN_TIMERANGE_PATTERN_WEEKLY;
  val.dayMask = OPEN_TIMERANGE_DAY_SUN | OPEN_TIMERANGE_DAY_WED | OPEN_TIMERANGE_DAY_FRI;

  val.frequency = 5;

  if ((result = openapiTimeRangePeriodicEntryEndDateAndTimeSet(clientHandle, &buffDesc, index, val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add periodic entry end date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Periodic end date and time added successfully for the given time range : %s \n", str);
  }    
  return;
}

/*********************************************************************
* @purpose  Gets end date and time for the given periodic time range entry.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
* @param    index          @b{(input)} time range entry index
*
* @returns  none
* 
* @notes  
* 
* @end 
*********************************************************************/
void timeRangePeriodicEntryEndDateAndTimeGet(openapiClientHandle_t *clientHandle, 
                                             char *name,
                                             uint32_t index)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];
  char buf[OPEN_TIMERANGE_BUF_LEN];
  OPEN_TIMERANGE_PERIODIC_ENTRY_t val;

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&val, 0, sizeof(val));
  memset(buf, 0, sizeof(buf));

  if ((result = openapiTimeRangePeriodicEntryEndDateAndTimeGet(clientHandle, &buffDesc, index, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to add periodic entry end date and time for the given time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Periodic end date and time added successfully for the given time range : %s %u \n", str, index);
    printf("Date: %04u-%02u-%02u %02u:%02u \n", val.openDateAndTime.year, 
            val.openDateAndTime.month, val.openDateAndTime.day,
            val.openDateAndTime.hour, val.openDateAndTime.minute);
    
    if (val.pattern == OPEN_TIMERANGE_PATTERN_WEEKLY)
    {
      snprintf(buf, sizeof(buf), "WEEKLY - 0x%02x", val.dayMask);
    } 
    else if (val.pattern == OPEN_TIMERANGE_PATTERN_DAILY) 
    {
      snprintf(buf, sizeof(buf), "DAILY - 0x%02x", val.dayMask);
    } 
    else if (val.pattern == OPEN_TIMERANGE_PATTERN_MONTHLY) 
    {
      snprintf(buf, sizeof(buf), "MONTLY - 0x%02x", val.dayMask);
    } 
    printf("Pattern: %s \n", buf);

    printf("Frequency: %02u \n", val.frequency);
  }    
  return;
}

/*********************************************************************
* @purpose  Deletes an entry from an existing time range.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    name           @b{(input)} time ranges name
* @param    val            @b{(input)} time range entry 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void timeRangeEntryDelete(openapiClientHandle_t *clientHandle, char *name, uint32_t val)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_TIMERANGE_NAME_LEN_MAX];

  memset(str, 0, sizeof(str));
  strncpy(str, name, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;
      
  if ((result = openapiTimeRangeEntryDelete(clientHandle, &buffDesc, val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to delete time range entry. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Time range entry deleted successfully : %s %u \n", str, val);
  }    
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Time Ranges.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t val = 0;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 2)
  {   
    printTimeRangesAppMenu();
    exit(1);
  } 

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();
          
  /* Register with OpEN */
  if ((result = openapiClientRegister("timeranges_example", &clientHandle)) != OPEN_E_NONE)
  { 
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }     
              
  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {   
    sleep(1);
  }
            
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Time Ranges API example application");
                
  printf("\n");

  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("ICOS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");
  
  /* exercise various OPEN API Time Ranges functions */
  switch (testNum)
  {
    case 1:
      if (argc != 2)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeSanityTest(&clientHandle); 
      break;
    case 2:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      val = atoi(argv[2]);
      timeRangeAdminModeSet(&clientHandle, val); 
      break;
    case 3:
      if (argc != 2)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeAdminModeGet(&clientHandle); 
      break;
    case 4:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeAdd(&clientHandle, argv[2]);
      break;
    case 5:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeDelete(&clientHandle, argv[2]);
      break;
    case 6:
      if (argc != 2)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangeNameListGet(&clientHandle); 
      break;
    case 7:
      if (argc != 2)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangeCurrNumGet(&clientHandle); 
      break;
    case 8:
      if (argc != 2)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeMaxNumGet(&clientHandle); 
      break;
    case 9:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangeStatusGet(&clientHandle, argv[2]); 
      break;
    case 10:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangeNumEntriesGet(&clientHandle, argv[2]); 
      break;
    case 11:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeAbsoluteEntryNumGet(&clientHandle, argv[2]);
      break;
    case 12:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeAbsoluteEntryStartDateAndTimeAdd(&clientHandle, argv[2]);
      break;
    case 13:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      } 
      timeRangeAbsoluteEntryStartDateAndTimeGet(&clientHandle, argv[2]);
      break;
    case 14:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangeAbsoluteEntryEndDateAndTimeAdd(&clientHandle, argv[2]); 
      break;
    case 15:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangeAbsoluteEntryEndDateAndTimeGet(&clientHandle, argv[2]); 
      break;
    case 16:
      if (argc != 4)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      val = atoi(argv[3]);
      timeRangeEntryDelete(&clientHandle, argv[2], val); 
      break;
    case 17:
      if (argc != 3)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      timeRangePeriodicEntryStartDateAndTimeSet(&clientHandle, argv[2]); 
      break;
    case 18:
      if (argc != 4)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      val = atoi(argv[3]);
      timeRangePeriodicEntryStartDateAndTimeGet(&clientHandle, argv[2], val); 
      break;
    case 19:
      if (argc != 4)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      val = atoi(argv[3]);
      timeRangePeriodicEntryEndDateAndTimeSet(&clientHandle, argv[2], val); 
      break;
    case 20:
      if (argc != 4)
      {
        printTimeRangesAppMenu();
        exit(1);
      }
      val = atoi(argv[3]);
      timeRangePeriodicEntryEndDateAndTimeGet(&clientHandle, argv[2], val); 
      break;
    default:
      printTimeRangesAppMenu();
      break;  
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Time Ranges API example application");

  return 0;
}  
