/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  authentication_example.c
*
* @purpose   Authentication OpEN APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    13/03/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

#define OPEN_MAX_APL_NAME_SIZE 15
/*******************************************************************
*
* @brief  This function prints the Authentication Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printAuthenticationAppMenu()
{
  printf("Usage: authentication_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Create authentication list name: authentication_example 1 <access-level> <list-name> \n");
  printf("Test 2: Add methods to authentication list: authentication_example 2 <access-level> <list-name> <method1> <method2> <method3> .... \n");
  printf("Test 3: Get authentication list names and configured methods of an access type: authentication_example 3 <access-level>\n");
  printf("Test 4: Set authentication list to an access line: authentication_example 4 <access-line> <access-level> <list-name> \n");
  printf("Test 5: Get authentication list name applied to access line: authentication_example 5 <access-line> <access-level> \n");
  printf("Test 6: Delete authentication list applied to the access-line: authentication_example 6 <access-line> <access-level> \n");
  printf("Test 7: Delete authentication list created: authentication_example 7 <access-level> <list-name> \n");
  printf("Test 8: Add an authentication method at the specified index in the APL: authentication_example 8 <access-line> <access-level> <list-name> <auth-method-order> <auth-method>\n");
  printf("Test 9: Get the authentication method at the specified index in the APL: authentication_example 9 <access-line> <access-level> <list-name> <auth-method-order>\n");
  printf("Test 10: Authentication OpEN APIs sanity: authentication_example 10 \n");

  return;
}

/*******************************************************************
* @purpose  Create authentication list for an access level.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    accessLevel    @b{(input)}   Access level
* @param    listName       @b{(input)}  Authentication List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authenticationListCreate(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LEVEL_t accessLevel, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  result = openapiAuthenticationListNameValidate(clientHandle, &buffDesc);

  if (result == OPEN_E_NONE)
  { 
    if ((result = openapiAuthenticationListCreate(clientHandle, accessLevel, &buffDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to create authentication list name for access level %d. (result = %d) \n", accessLevel, result);
    }
    else
    {
      printf("Authentication list name created successfully. \n");
    }
  }
  else
  {
    printf("Authentication list name is too long or contains invalid characters. \n");
  }
  return;
}

/*******************************************************************
* @purpose  Add methods to authentication list for an access level.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    accessLevel    @b{(input)}   Access level
* @param    listName       @b{(input)}   Authentication List name.
* @param    methods        @b{(input)}   Buff descriptor containing list of methods.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authenticationMethodsAdd(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LEVEL_t accessLevel, 
                              char *listName, open_buffdesc *methods)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100]; 

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;


  if ((result =  openapiAuthenticationMethodsAdd(clientHandle, accessLevel, &buffDesc, methods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add methods to authentication list for access level %d. (result = %d) \n", accessLevel, result);
  }
  else
  {
    printf("Methods added to authentication list successfully. \n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the Authentication list information
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    accessLevel      @b{(input)}   Access levels: login/enable
* 
* @returns  none
*   
* @notes 
* 
* @end
*********************************************************************/
void authenticationInfoGet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LEVEL_t accessLevel)
{
  open_error_t result;
  uint32_t authListNameSize;
  uint32_t maxAuthMethods;
  uint32_t i = 0;
  char *str;
  char *methodStr;
  open_buffdesc listName;
  open_buffdesc methodList;
  

  if ((result = openapiAuthenticationListNameSizeGet(clientHandle, &authListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get authentication list name size. (result = %d)", result);
    return;
  }
  if ((result = openapiAuthenticationMethodsMaxGet(clientHandle, &maxAuthMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum authentication methods supported. (result = %d) \n", result);
    return;
  }

  if ((str = (char *)malloc(authListNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  if ((methodStr = (char *)malloc(maxAuthMethods)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(str);
    return;
  }

  memset(str, 0, authListNameSize);
  listName.pstart = str;
  listName.size = authListNameSize;

  memset(methodStr, 0, maxAuthMethods);
  methodList.pstart = methodStr;
  methodList.size = maxAuthMethods;

  printf("Getting the first authentication list name. \n");
  if ((result = openapiAuthenticationListFirstGet(clientHandle, accessLevel, &listName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first authentication list name for access level: %d (result = %d) \n", accessLevel, result);
    free(methodStr);
    free(str);
    return;
  }
  printf("Auth_list_name   Methods\n");
  do
  {
    printf("%-15s  ", str);

    result = openapiAuthenticationMethodListGet(clientHandle, accessLevel, &listName, &methodList);
    if (result == OPEN_E_NONE)
    {
      for (i = 0; i < methodList.size; i++)
      {
        switch ((methodStr[i]-'0'))
        {
          case OPEN_AUTH_METHOD_ENABLE:
            printf("Enable ");
            break;
          case OPEN_AUTH_METHOD_LINE:
            printf("Line ");
            break;
          case OPEN_AUTH_METHOD_LOCAL:
            printf("Local ");
            break;
          case OPEN_AUTH_METHOD_NONE:
            printf("None ");
            break;
          case OPEN_AUTH_METHOD_RADIUS:
            printf("Radius ");
            break;
          case OPEN_AUTH_METHOD_TACACS:
            printf("Tacacs ");
            break;
          case OPEN_AUTH_METHOD_REJECT:
            printf("Deny ");
            break;
          default:
            break;
        }
      }
    }
    else 
    {
      printf("Method list get failed. (result = %d) \n", result);
    }
    listName.size = authListNameSize;
    methodList.size = maxAuthMethods;
    printf("\n");
  }while (openapiAuthenticationListNextGet(clientHandle, accessLevel, &listName, &listName) == OPEN_E_NONE);

  free(str);
  free(methodStr);  
  return;
}

/*******************************************************************
* @purpose  Apply authentication list to access line for an access level.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    accessLevel    @b{(input)}   Access level
* @param    listName       @b{(input)}   Authentication List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authenticationListLineApply(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t  accessLine, 
                                 OPEN_ACCESS_LEVEL_t accessLevel, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  if ((result = openapiAuthenticationListLineSet(clientHandle, accessLine, accessLevel, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to apply the authentication list name to access line %d. (result = %d) \n", accessLine, result);
  }
  else
  {
    printf("Authentication list name applied to access line successfully \n");
  }

  return; 
}

/*******************************************************************
* @purpose  Get authentication list applied to access line for an access level.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    accessLine     @b{(input)}   Access line
* @param    accessLevel    @b{(input)}   Access level
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void authenticationListLineGet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t  accessLine,
                               OPEN_ACCESS_LEVEL_t accessLevel)
{
  open_error_t result;
  char *str;
  open_buffdesc listName;
  uint32_t authListNameSize;


  if ((result = openapiAuthenticationListNameSizeGet(clientHandle, &authListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get authentication list name size. (result = %d)", result);
    return;
  }

  if ((str = (char *)malloc(authListNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, authListNameSize);
  listName.pstart = str;
  listName.size = authListNameSize;

  printf(" Getting the authentication list name applied to access line %d\n", accessLine);
  if ((result = openapiAuthenticationListLineGet(clientHandle, accessLine, accessLevel, &listName)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting authentication list name applied to access line. (result = %d) \n", result);
  }
  else
  {
    printf("Authentication list name: %s \n", str);
  }
  free(str);
  return;
  
}

/*******************************************************************
* @purpose  Delete authentication list applied to access line for an access level.
*
* @param    clientHandle   @b{(input)}   client handle from registratiion API
* @param    accessLine     @b{(input)}   Access line
* @param    accessLevel    @b{(input)}   Access level
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authenticationListLineDelete(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t  accessLine,
                                  OPEN_ACCESS_LEVEL_t accessLevel)
{
  open_error_t result;

  printf("Deleting authentication list applied to access line %d. \n", accessLine); 

  if ((result = openapiAuthenticationListLineDelete(clientHandle, accessLine, accessLevel)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete authentication list name applied to access line. (result = %d) \n", result);
  }
  else
  {
    printf("Deleted the authentication list name from the access line %d. \n", accessLine);
  }
  return;
}

/*******************************************************************
* @purpose  Delete authentication list  of an access level.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    accessLevel    @b{(input)}   Access level
* @param    listName       @b{(input)}   Authentication List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authenticationListDelete(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LEVEL_t accessLevel,
                              char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
    char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  printf("Deleting the authentication list %s \n", listName);

  if ((result = openapiAuthenticationListDelete(clientHandle, accessLevel, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete authentication list. (result = %d) \n", result);
  }
  else
  {
    printf("Authentication list %s deleted successfully. \n", (char *) buffDesc.pstart);
  }
  return;
}

/*********************************************************************
 * * @purpose  Add an authentication method at the specified index in the APL.
 * *
 * * @param    client_handle       @b{(input)}  client handle from registration API
 * * @param    accessLine       @b{(input)}  APL access line
 * * @param    accessLevel       @b{(input)}  APL access level
 * * @param    authListName       @b{(input)}  pointer to APL name
 * * @param    authMethodOrder       @b{(input)}  order this method should be attempted in the APL
 * * @param    authMethod       @b{(input)}  authentication method value
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void authenticationMethodSet(openapiClientHandle_t *client_handle, OPEN_ACCESS_LINE_t accessLine, OPEN_ACCESS_LEVEL_t accessLevel, char *authListName, uint32_t authMethodOrder, OPEN_USER_MGR_AUTH_METHOD_t authMethod)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[OPEN_MAX_APL_NAME_SIZE + 1];

  memset(str, 0, sizeof(str));
  buffDesc.pstart = str;
  strncpy(str, authListName, (sizeof(str) - 1));
  buffDesc.size = strlen(str);

  if ((result = openapiAuthenticationMethodSet(client_handle, accessLine, accessLevel, &buffDesc, authMethodOrder, authMethod)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to adds an authentication method to an APL. (result = %d)\n", result);
  }
  else
  {
    printf("Method added to authentication list successfully.\n");
  }
  return;
}

/*********************************************************************
 * * @purpose  Get the authentication method at the specified index in the APL.
 * *
 * * @param    client_handle       @b{(input)}  client handle from registration API
 * * @param    accessLine       @b{(input)}  APL access line
 * * @param    accessLevel       @b{(input)}  APL access level
 * * @param    authListName       @b{(input)}  pointer to APL name
 * * @param    authMethodOrder       @b{(input)}  order this method should be attempted in the APL
 * * @param    authMethod       @b{(output)}  authentication method value
 *
 * *
 * * @returns  none
 * *
 * * @end
 * *********************************************************************/
void authenticationMethodGet(openapiClientHandle_t *client_handle, OPEN_ACCESS_LINE_t accessLine, OPEN_ACCESS_LEVEL_t accessLevel, char *authListName, uint32_t authMethodOrder)
{
  open_error_t result;
  open_buffdesc buffDesc;
  OPEN_USER_MGR_AUTH_METHOD_t authMethod;
  char listName[OPEN_MAX_APL_NAME_SIZE];
  memset(listName, 0, sizeof(listName));
  strncpy(listName, authListName, (sizeof(listName) - 1));

  buffDesc.pstart = listName;
  buffDesc.size = OPEN_MAX_APL_NAME_SIZE + 1;

  if ((result = openapiAuthenticationMethodGet(client_handle, accessLine, accessLevel, &buffDesc, authMethodOrder, &authMethod)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the authentication method at the specified index in the APL. (result = %d)\n", result);
  }
  else
  {
    printf("Authentication method value: %u \n", authMethod);
  }
  return;
}

/*******************************************************************
* @purpose  Authentication OpEN APIs Sanity.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void authenticationOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t authListNameSize = 0, maxAuthMethods =0;
  char *str, *str1; 
  open_buffdesc buffDesc1;
  open_buffdesc buffDesc2;
  OPEN_ACCESS_LEVEL_t accessLevel = OPEN_ACCESS_LEVEL_UNKNOWN;
  OPEN_ACCESS_LINE_t accessLine = OPEN_ACCESS_LINE_UNKNOWN;

  printf("Testing authentication OpEN APIs sanity.\n");

  if ((result = openapiAuthenticationListNameSizeGet(clientHandle, &authListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get authentication name size. (result = %d) \n", result);
    return;
  }

  if ((result = openapiAuthenticationMethodsMaxGet(clientHandle, &maxAuthMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum authentication methods supported. (result = %d) \n", result);
    return; 
  }

  if ((str = (char *)malloc(authListNameSize)) == NULL)
  {
    printf("Could not allocate memory\n");
    return;
  }
  memset(str, 0, authListNameSize);
  buffDesc1.pstart = str;
  buffDesc1.size = authListNameSize;

  if ((str1 = (char *)malloc(maxAuthMethods)) == NULL)
  {
    printf("Could not allocate memory\n");
    free(str);
    return;
  }

  memset(str1, 0, maxAuthMethods);
  buffDesc2.pstart = str1;
  buffDesc2.size = maxAuthMethods;

  /* openapiAuthenticationListNameSizeGet() */
  printf("\nTesting openapiAuthenticationListNameSizeGet(): \n");
  
  result = openapiAuthenticationListNameSizeGet(NULL, &authListNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListNameSizeGet(clientHandle, NULL);
  printf("NULL string length. (result = %d)\n", result);

  printf("openapiAuthenticationListNameSizeGet() sanity successful.\n");

  /* openapiAuthenticationMethodsMaxGet() */
  printf("\nTesting openapiAuthenticationMethodsMaxGet(): \n");

  result = openapiAuthenticationMethodsMaxGet(NULL, &maxAuthMethods);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationMethodsMaxGet(clientHandle, NULL);
  printf("Maximum methods parameter NULL. (result = %d)\n", result);
  
  printf("openapiAuthenticationMethodsMaxGet() sanity successful.\n");

  /* openapiAuthenticationListCreate() */
  printf("\nTesting openapiAuthenticationListCreate() \n");

  result = openapiAuthenticationListCreate(NULL, accessLevel, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListCreate(clientHandle, 5, &buffDesc1);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationListCreate(clientHandle, accessLevel, NULL);
  printf("NULL authentication list name buff descriptor. (result = %d)\n", result);

  printf("openapiAuthenticationListCreate() sanity successful.\n");

  /*openapiAuthenticationMethodsAdd()*/
  printf("\nTesting openapiAuthenticationMethodsAdd() \n");

  result =  openapiAuthenticationMethodsAdd(NULL, accessLevel, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result =  openapiAuthenticationMethodsAdd(clientHandle, 5, &buffDesc1, &buffDesc2);
  printf("Invalid access level. (result = %d)\n", result);

  result =  openapiAuthenticationMethodsAdd(clientHandle, accessLevel, NULL,&buffDesc2);
  printf("NULL authentication list name buff descriptor. (result = %d)\n", result);

  result =  openapiAuthenticationMethodsAdd(clientHandle, accessLevel, &buffDesc1, NULL);
  printf("NULL methods buff descriptor. (result = %d)\n", result);

  printf("openapiAuthenticationMethodsAdd() sanity successful.\n");

  /* openapiAuthenticationMethodListGet() */
  printf("\nTesting openapiAuthenticationMethodListGet() \n");

  result = openapiAuthenticationMethodListGet(NULL, accessLevel, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationMethodListGet(clientHandle, 5, &buffDesc1, &buffDesc2);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationMethodListGet(clientHandle, accessLevel, NULL, &buffDesc2);
  printf("NULL buff descriptor to authentication list name. (result = %d)\n", result);

  result = openapiAuthenticationMethodListGet(clientHandle, accessLevel, &buffDesc1, NULL);
  printf("NULL buff descriptor to method list. (result = %d)\n", result);

  printf("openapiAuthenticationMethodListGet() sanity successful.\n");

  /* openapiAuthenticationListFirstGet() */
  printf("\nTesting openapiAuthenticationListFirstGet() \n");

  result = openapiAuthenticationListFirstGet(NULL, accessLevel, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListFirstGet(clientHandle, 5, &buffDesc1);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationListFirstGet(clientHandle, accessLevel, NULL);
  printf("NULL buff descriptor to authentication list name. (result = %d)\n", result);

  printf("openapiAuthenticationListFirstGet() sanity successful.\n");

  /* openapiAuthenticationListNextGet() */
  printf("\nTesting openapiAuthenticationListNextGet() \n");

  result = openapiAuthenticationListNextGet(NULL, accessLevel, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListNextGet(clientHandle, 5, &buffDesc1, &buffDesc2);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationListNextGet(clientHandle, accessLevel, NULL, &buffDesc2);
  printf("NULL buff descriptor to previous authentication list name. (result = %d)\n", result);

  result = openapiAuthenticationListNextGet(clientHandle, accessLevel, &buffDesc1, NULL);
  printf("NULL buff descriptor to next authentication list name. (result = %d)\n", result);

  printf("openapiAuthenticationListNextGet() sanity successful.\n");

  /* openapiAuthenticationListDelete() */
  printf("\nTesting openapiAuthenticationListDelete() \n");

  result = openapiAuthenticationListDelete(NULL, accessLevel, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListDelete(clientHandle, 5, &buffDesc1);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationListDelete(clientHandle, accessLevel, NULL);
  printf("NULL buff descriptor to authentication list name. (result = %d)\n", result);

  printf("openapiAuthenticationListDelete() sanity successful.\n");

  /* openapiAuthenticationListLineSet() */
  printf("\nTesting openapiAuthenticationListLineSet() \n");

  result = openapiAuthenticationListLineSet(NULL, accessLine, accessLevel, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListLineSet(clientHandle, 7, accessLevel, &buffDesc1);
  printf("Invalid access line (result = %d)\n", result);

  result = openapiAuthenticationListLineSet(clientHandle, accessLine, 5, &buffDesc1);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationListLineSet(clientHandle, accessLine, accessLevel, NULL);
  printf("NULL buff descriptor to authentication list name. (result = %d)\n", result);

  printf("openapiAuthenticationListLineSet() sanity successful.\n");

  /* openapiAuthenticationListLineGet() */
  printf("\nTesting openapiAuthenticationListLineGet() \n");

  result = openapiAuthenticationListLineGet(NULL, accessLine, accessLevel, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListLineGet(clientHandle, 8, accessLevel, &buffDesc1);
  printf("Invalid access line (result = %d)\n", result);

  result = openapiAuthenticationListLineGet(clientHandle, accessLine, 0, &buffDesc1);
  printf("Invalid access level. (result = %d)\n", result);

  result = openapiAuthenticationListLineGet(clientHandle, accessLine, accessLevel, NULL);
  printf("NULL buff descriptor to authentication list name. (result = %d)\n", result);

  printf("openapiAuthenticationListLineGet() sanity successful.\n");

  /* openapiAuthenticationListLineDelete() */
  printf("\nTesting openapiAuthenticationListLineDelete() \n");

  result = openapiAuthenticationListLineDelete(NULL, accessLine, accessLevel);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticationListLineDelete(clientHandle, 8, accessLevel);
  printf("Invalid access line (result = %d)\n", result);

  result = openapiAuthenticationListLineDelete(clientHandle, accessLine, 6);
  printf("Invalid access level. (result = %d)\n", result);

  printf("openapiAuthenticationListLineDelete() sanity successful.\n");

  free(str);
  free(str1);
   
  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Authentication Configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2, arg4, arg5;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc authMethods; 
  char str[100];
  uint32_t i = 0;

  if (argc < 2)
  {
    printAuthenticationAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("authentication_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Authentication API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API Spanning Tree functions */

  switch (testNum)
  {
    case 1:
      if (argc != 4)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      authenticationListCreate(&clientHandle, arg1, argv[3]);
      break;
    case 2:
      if (argc < 5)
      {
         printAuthenticationAppMenu();
         exit(1);
      }
      arg1 = atoi(argv[2]);
      memset(str, 0, sizeof(str));
      for (i = 4; i < argc; i++)
      {
        strncat(str, argv[i], strlen(argv[i]));
      }
      authMethods.pstart = str;
      authMethods.size = strlen(str)+1;


      authenticationMethodsAdd(&clientHandle, arg1, argv[3], &authMethods);
      break;
    case 3:
      if (argc != 3)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      authenticationInfoGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 5)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      authenticationListLineApply(&clientHandle, arg1, arg2, argv[4]);
      break;
    case 5:
      if (argc != 4)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      authenticationListLineGet(&clientHandle, arg1, arg2);
      break;
    case 6:
     if (argc != 4)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      authenticationListLineDelete(&clientHandle, arg1, arg2); 
      break;
    case 7:
      if (argc != 4)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      authenticationListDelete(&clientHandle, arg1, argv[3]);
      break;
    case 8:
      if (argc != 7)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg4 = atoi(argv[5]);
      arg5 = atoi(argv[6]);
      authenticationMethodSet(&clientHandle, arg1, arg2, argv[4], arg4, arg5);
      break;
    case 9:
      if (argc != 6)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg4 = atoi(argv[5]);
      authenticationMethodGet(&clientHandle, arg1, arg2, argv[4], arg4);
      break;
    case 10:
      if (argc != 2)
      {
        printAuthenticationAppMenu();
        exit(1);
      }
      authenticationOpENAPIsSanity(&clientHandle);
      break;
    default:
      printAuthenticationAppMenu();
      break;
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Authentication API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;

}
