/*********************************************************************
*
* Copyright 2016-2022 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
* @filename  proc_util.h
*
* @purpose   Multi-Process Utility Functions
*
* @component procmgr
*
* @comments  none
*
* @create    2/1/2012
*
* @author    atsigler
*
* @end
*             
**********************************************************************/
#ifndef INCLUDE_PROC_UTIL_H
#define INCLUDE_PROC_UTIL_H

#include <signal.h>
#include <sys/types.h>

/* Check GCC */
#if __GNUC__
  #if __x86_64__ || __ppc64__ || __aarch64__ || __powerpc64__
    #define ENV64BIT
  #else
    #define ENV32BIT
  #endif
#endif

/* L7_PRODUCT_ATLAS
** These three compiler constants only apply to the Automotive product.
** The constants control how much FLASH is reserved for the configuration 
** sector, the Crash Log sector, and the SYSLOG. 
*/
#define PROCMGR_ATLAS_CONFIG_SIZE (512*1024)
#define PROCMGR_ATLAS_CRASHLOG_SIZE (448*1024)
#define PROCMGR_ATLAS_SYSLOG_SIZE (64*1024)

/* This signal is used to trigger memory debugging for
** application processes. If an application already uses this signal then 
** change the value to some other signal that is not already in use by another 
** application. 
*/
#define PROCMGR_MEMTRACK_SIG (SIGRTMIN + 10)

/* This signal triggers the stack backtrace dump.
*/
#define PROCMGR_BACKTRACE_SIG (SIGRTMIN + 11)

/* This signal triggers the devshell command.
*/
#define PROCMGR_DEVSHELL_SIG (SIGRTMIN + 12)

/* This signal triggers the backtrace for osapiDebugStackTrace() on ARM CPU.
** Although this signal is not used for multi-process utilities, 
** the signal is defined here in order to avoid conflict with the other 
** multi-process signals. 
*/
#define INTERNAL_BACKTRACE_SIG (SIGRTMIN + 13)

/* This signal triggers the backtrace for osapiDebugStackTrace() for 
 * multi-process utilities.
*/
#define PROC_INTERNAL_BACKTRACE_SIG (SIGRTMIN + 14)

/* Application Process Memory Dump Command.
*/
#define L7PROC_MEM_DUMP_CMD_FILE "/tmp/debug-flags/mem-dump-"

/* Application Process Devshell Command.
*/
#define L7PROC_DEVSHELL_CMD_FILE "/tmp/debug-flags/devshell-"

/* Application Process Crash Log file.
*/
#define L7PROC_CRASH_LOG_FILE "/mnt/fastpath/crashlogs/proc-crash.log"

/* Maximum size of the backtrace buffer.
*/
#define L7PROC_BACKTRACE_BUF_SIZE 2048

/* Memory Analysis Utility maximum message size.
*/
#define L7PROC_MEM_BUF_SIZE  128

/* Devshell Utility maximum message size.
*/
#define L7PROC_DEVSHELL_SIZE  4096

/* Maximum SYSLOG buffer size for application processes.
*/
#define L7PROC_SYSLOG_BUF_SIZE 512

/* SYSLOG Server Socket Name.
*/
#define L7PROC_SYSLOG_SOCK "/tmp/syslog-sock"
/* LOG SEVERITY LEVELS */

/* Log Severity Types
*/
typedef enum
{
  L7PROC_LOG_SEVERITY_EMERGENCY = 0,
  L7PROC_LOG_SEVERITY_ALERT = 1,
  L7PROC_LOG_SEVERITY_CRITICAL = 2,
  L7PROC_LOG_SEVERITY_ERROR = 3,
  L7PROC_LOG_SEVERITY_WARNING = 4,
  L7PROC_LOG_SEVERITY_NOTICE = 5,
  L7PROC_LOG_SEVERITY_INFO = 6,
  L7PROC_LOG_SEVERITY_DEBUG = 7,
} L7PROC_LOG_SEVERITY_t;

/* SYSLOG Macro
*/
#define L7PROC_LOGF(__sev__, __comp__, __fmt__,__args__...)                         \
       l7proc_logf(__sev__, __comp__, (char *)__FILE__, __LINE__, (char *)__fmt__, ## __args__)

#ifndef L7_NO_PRINTF_FMT
#define L7_PRINTF_FMT(x,y) __attribute__((format (printf,x,y)))
#else
#define L7_PRINTF_FMT(x,y)
#endif

void l7proc_logf(int severity, int component,
             char * fileName, unsigned int lineNum, char * format, ...)L7_PRINTF_FMT(5,6);

/* SYSLOG message
*/
typedef struct {
    int severity;
    int component;
    char file_name[32];
    unsigned int line_num;

    char buf[L7PROC_SYSLOG_BUF_SIZE];
} l7proc_syslog_msg_t;


void l7proc_memtrack_init(void);

void l7proc_devshell_init(void);

void l7proc_crashlog_register (void);

void l7proc_backtrace_register (void);

void proc_backtrace_sig_register (void);
void l7proc_watchdog_start (int num_seconds, char * name);

void l7proc_watchdog_stop (void);

int l7proc_extagentevent_dbsockcreate(unsigned int clientId, int *fd);
int l7proc_extagentevent_dbsockclose(unsigned int clientId, int fd);

int l7proc_extagentevent_logsockcreate(unsigned int clientId, int *fd);
int l7proc_extagentevent_logsockclose(unsigned int clientId, int fd);

int l7proc_extagentevent_trapsockcreate(unsigned int clientId, int *fd);
int l7proc_extagentevent_trapsockclose(unsigned int clientId, int fd);

int l7proc_extagentpkt_rxsockcreate(unsigned int agentId, int *fd);
int l7proc_extagentpkt_rxsockclose(unsigned int agentId, int fd);

int l7proc_rpc_clientid_assign(pid_t pid, int serverId, int *clientId);
int l7proc_rpc_clientid_free(int clientId);

/* Function to get process identifier from rpc client identifier */
int l7proc_rpc_client_pid_get(int clientId, pid_t *pid);

int base64Encode(const unsigned char *data,
                 unsigned int inputLength,
                 char *encodedData, unsigned int maxLength);

#endif /* INCLUDE_PROC_UTIL_H */

