/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dcbx.h
*
* @purpose   Data center Bridging Capability Exchange Protocol APIs.
*
* @component OPEN
*
* @create    01/23/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DCBX_H_INCLUDED
#define OPENAPI_DCBX_H_INCLUDED

#include "openapi_common.h"

typedef enum 
{
  OPEN_DCBX_CAP_NONE            = 0,         /*<< No capability bits set */
  OPEN_DCBX_CAP_PFC_BITMASK     = (1 << 0),  /*<< PFC capability */
  OPEN_DCBX_CAP_ETS_CFG_BITMASK = (1 << 1),  /*<< ETS config capability  */
  OPEN_DCBX_CAP_ETS_REC_BITMASK = (1 << 2),  /*<< ETS recommendation TLV */
  OPEN_DCBX_CAP_APP_PRI_BITMASK = (1 << 3),  /*<< Application priority */
  OPEN_DCBX_CAP_CONTROL_BITMASK = (1 << 4)   /*<< Control capability  */
} OPEN_DCBX_CAPABILITIES_t;

#define OPEN_ETS_TCG_NUM_MIN      0         /*<< Minimum ETS TCG number */
#define OPEN_ETS_TCG_NUM_MAX      7         /*<< Maximum ETS TCG number */

typedef enum
{
  OPEN_DCBX_AUTO = 1,             /*<< Auto */
  OPEN_DCBX_IEEE = 2,             /*<< IEEE */
  OPEN_DCBX_CIN = 3,              /*<< CIN */
  OPEN_DCBX_CEE = 4,              /*<< CEE */
} OPEN_DCBX_VERSION_t;

typedef enum
{
  OPEN_DCBX_OP_STATUS_ENABLE = 1,      /*<< Enabled */
  OPEN_DCBX_OP_STATUS_DISABLE = 2,     /*<< Disabled */
  OPEN_DCBX_OP_STATUS_MULTI_PEER = 3,  /*<< Multi-peer */
} OPEN_DCBX_OP_STATUS;

typedef enum
{
  OPEN_DCBX_PORT_ROLE_MANUAL = 1,           /*<< Port role manual */
  OPEN_DCBX_PORT_ROLE_AUTO_UPSTREAM = 2,    /*<< Port role auto upstream */
  OPEN_DCBX_PORT_ROLE_AUTO_DOWNSTREAM = 3,  /*<< Port role auto downstream */
  OPEN_DCBX_PORT_ROLE_AUTO_CFG_SRC = 4      /*<< Port role auto config src */
} OPEN_DCBX_PORT_ROLE_t;

#define OPEN_LLDP_MGMT_STRING_SIZE_MAX       (255+1)

/*********************************************************************
* @purpose  Set the DCBX version. 
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    version         @b{(input)}   version.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxVersionSet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   OPEN_DCBX_VERSION_t version);

/*********************************************************************
* @purpose  Get the configuration port number.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(output)}  interface number.
* @param    selected        @b{(output)}  if OPEN_TRUE, config source is selected. 
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxCfgSrcGet(openapiClientHandle_t *client_handle,
                                  uint32_t *ifNum, OPEN_BOOL_t *selected);

/*********************************************************************
* @purpose  Get the operational status of DCBX.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    status          @b{(output)}  status.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    The operational state of DCBX is enabled if:
*           1) LLDP TX and RX is enabled.
*           2) DCBX TLV are enabled.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxOperStatusGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum,
                                      OPEN_DCBX_OP_STATUS *status);

/*********************************************************************
* @purpose  Get the port role configuration.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    role            @b{(output)}  role.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxAutoCfgPortRoleGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           OPEN_DCBX_PORT_ROLE_t *role);

/*********************************************************************
* @purpose  Get the DCBX version.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    version         @b{(output)}  DCBX version.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxVersionGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   OPEN_DCBX_VERSION_t *version);

/*********************************************************************
* @purpose  Get the peer DCBX running version.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    mode            @b{(output)}  peer DCBX running version.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPeerVersionGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       OPEN_DCBX_VERSION_t *mode);

/*********************************************************************
* @purpose  Get the DCBX counters.
*
* @param    client_handle @b{(input)}     client handle.
* @param    ifNum         @b{(input)}     Interface Number
* @param    txCnt         @b{(output)}    Transmits 
* @param    rxCnt         @b{(output)}    Receives
* @param    errorFrames   @b{(output)}    error frames
* @param    unknownTlv    @b{(output)}    unknown TLV
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxCounterGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   uint32_t *txCnt,
                                   uint32_t *rxCnt,
                                   uint32_t *errorFrames,
                                   uint32_t *unknownTlv);

/*********************************************************************
* @purpose  Get the MAC address of remote DCBX device.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    macAddr         @b{(output)}  mac addr.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPeerMacGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   open_buffdesc *macAddr);

/*********************************************************************
* @purpose  Get the system description of the peer device.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    descr           @b{(output)}  description string.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   descr must point to at least OPEN_LLDP_MGMT_STRING_SIZE_MAX bytes. 
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPeerDescGet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum,
                                    open_buffdesc *descr);

/*********************************************************************
* @purpose  Determine if the interface is valid for DCBX.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    valid           @b{(output)}  OPEN_TRUE if valid, OPEN_FALSE otherwise. 
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxIsValidIntf(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum,
                                    OPEN_BOOL_t *valid);

/*********************************************************************
* @purpose  Check the peer is config source for the auto configuration. 
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    isCfgSrc        @b{(output)}  OPEN_TRUE if peer is config src, else OPEN_FALSE.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxIntfIsCfgSrcGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum,
                                        OPEN_BOOL_t *isCfgSrc);

/*********************************************************************
* @purpose  Get the timed-out and multi-peer counts for an interface.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    timedOutCount   @b{(output)}  timed-out count.
* @param    multiPeerCount  @b{(output)}  multi-peer count.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxTimedOutAndMultiPeerCountGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     uint32_t *timedOutCount,
                                                     uint32_t *multiPeerCount);

/*********************************************************************
* @purpose  Gets the DCBX TLVs that need to be transmitted in the LLDP PDU.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    version         @b{(output)}  version.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortOperationalModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               OPEN_DCBX_VERSION_t *version);
                                              

/*********************************************************************
* @purpose  Get peer TLV supported mask.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    peerTlvMask     @b{(output)}  mask of TLV supported.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_NOT_FOUND    No peer detected.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPeerTlvMaskGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       OPEN_DCBX_CAPABILITIES_t *peerTlvMask);

/*********************************************************************
* @purpose  Get the Oper version supported by remote legacy DCBX device.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    operVer         @b{(output)}  operVer.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortLegacyRemOperVerGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum,
                                                uint32_t *operVer);

/*********************************************************************
* @purpose  Get the Max version supported by the remote legacy DCBX device.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    maxVer          @b{(output)}  max version number.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortLegacyRemMaxVerGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t *maxVer);

/*********************************************************************
* @purpose  Get the current sequence number that is sent in 
*           DCBX control TLVs in Legacy Mode
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    seqNum          @b{(output)}  sequence number.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortLegacyRemSeqNumGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t *seqNum);

/*********************************************************************
* @purpose  Get the current ACK number that is sent by peer in DCBX 
*           control TLVs in Legacy mode.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    rcvdAckNum      @b{(output)}  received ack number.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortLegacyRemRcvdAckNumGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum,
                                                   uint32_t *rcvdAckNum);

/*********************************************************************
* @purpose  Get the port local willing status.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    willing         @b{(output)}  port local willing status.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortWillingGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       OPEN_BOOL_t *willing);

/*********************************************************************
* @purpose  Get the local scheduler type.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    schedList       @b{(output)}  scheduler list.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxLocalEtsSchedGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         open_buffdesc *schedList);

/*********************************************************************
* @purpose  Get local ETS bandwidth. 
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    bwList          @b{(output)}  bandwidth list.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxLocalEtsBwGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum,
                                      open_buffdesc *bwList);

/*********************************************************************
* @purpose  Get the application table.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    index           @b{(output)}  Index to the application priority
* @param    protocolId      @b{(output)}  protocol Id(ethertype or socket num)
* @param    protoSel        @b{(output)}  protocol sel
* @param    priority        @b{(output)}  dot1p priority
* @param    appStatus       @b{(output)}  application status
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxAppPriTableGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       uint32_t index,
                                       uint16_t *protocolId,
                                       uint32_t *protoSel,
                                       uint8_t *priority,
                                       OPEN_BOOL_t *appStatus);

/*********************************************************************
* @purpose  Get Peer application table.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    index           @b{(output)}  Index to the application priority
* @param    protocolId      @b{(output)}  protocol Id(ethertype or socket num)
* @param    protoSel        @b{(output)}  protocol sel
* @param    priority        @b{(output)}  dot1p priority
* @param    appStatus       @b{(output)}  application status

*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_NOT_FOUND    Peer is not present
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPeerAppPriTableGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           uint32_t index,
                                           uint16_t *protocolId,
                                           uint32_t *protoSel,
                                           uint8_t *priority,
                                           OPEN_BOOL_t *appStatus);

/*********************************************************************
* @purpose  Get bit mask of DCBX TLVs enabled. 
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    dcbxCap         @b{(output)}  mask of enabled DCBX TLVs.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxPortConfigTLVsEnabledGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 OPEN_DCBX_CAPABILITIES_t *dcbxCap);

/*********************************************************************
* @purpose  Set bit mask of enabled DCBX TLVs.
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    dcbxCap         @b{(output)}  mask of enabled DCBX TLVs.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/

open_error_t openapiDcbxPortConfigTLVsEnabledSet(openapiClientHandle_t *client_handle, 
                                                 uint32_t ifNum, 
                                                 OPEN_DCBX_CAPABILITIES_t dcbxCap);

/*********************************************************************
* @purpose  Set the port role for auto configuration. 
*
* @param    client_handle   @b{(input)}   client handle.
* @param    ifNum           @b{(input)}   interface number.
* @param    role            @b{(input)}   port role for auto configuration.
*
* @returns  OPEN_E_NONE         On success.
* @returns  OPEN_E_UNAVAIL      Interface not supported.
* @returns  OPEN_E_PARAM        Bad parameter.
* @returns  OPEN_E_FAIL         Some failure occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDcbxAutoCfgPortRoleSet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           OPEN_DCBX_PORT_ROLE_t role);

#endif /* OPENAPI_DCBX_H_INCLUDED */
