/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dhcp_client.h
*
* @purpose   DHCP Client configuration and Status
*
* @component OpEN
*
* @create    10/30/2015
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_DHCP_CLIENT_H_INCLUDED
#define OPENAPI_DHCP_CLIENT_H_INCLUDED

#include "openapi_common.h"
#include "openapi_dns.h"
#include "openapi_sntp.h"

#define OPEN_DHCP_VENDOR_SPECIFIC_SUBOPTIONS_NUM    3
#define OPEN_DHCP_BOOTP_VEND_OPTIONS_LEN            64
#define OPEN_DHCP_OPT125_VENDOR_NUMBER_MAX          3

#define OPEN_HOSTNAME_MAX_LEN                       64
#define OPEN_BOOTP_FILE_LEN                         128
#define OPEN_BOOTP_SNAME_LEN                        64
#define OPEN_DNS_NAME_SERVER_ENTRIES                8
#define OPEN_SNTP_MAX_SERVER_ENTRIES                12
#define OPENAPI_DNS_HOST_NAME_LEN_MAX               255

typedef struct
{
  unsigned char type;
  unsigned char len;
  unsigned char option[OPEN_DHCP_BOOTP_VEND_OPTIONS_LEN];
} open_tlv_option_t;

typedef struct
{
  unsigned char     numSubOptions;
  open_tlv_option_t subOptions[OPEN_DHCP_VENDOR_SPECIFIC_SUBOPTIONS_NUM];
} open_dhcp_vendor_specific_option_t;

typedef struct
{
  uint32_t          vendorID;
  unsigned char     numSubOptions;
  open_tlv_option_t subOptions[OPEN_DHCP_VENDOR_SPECIFIC_SUBOPTIONS_NUM];
} open_dhcp_vendor_sub_opt_t;

typedef struct
{
  open_dhcp_vendor_sub_opt_t vendor[OPEN_DHCP_OPT125_VENDOR_NUMBER_MAX];
} open_dhcp_option125_t;

typedef enum
{
  OPEN_BOOTP_DHCP_IP_ADDR  = 1,
  OPEN_BOOTP_DHCP_NETMASK  = 2,
  OPEN_BOOTP_DHCP_GATEWAY  = 4,
  OPEN_BOOTP_DHCP_SIADDR   = 8,
  OPEN_BOOTP_DHCP_BOOTFILE = 16,
  OPEN_BOOTP_DHCP_SNAME    = 32,
  OPEN_BOOTP_DHCP_OPTION_TFTP_SVR_ADDR  = 64,
  OPEN_BOOTP_DHCP_OPTION_DNS_SVR_ADDR   = 128,
  OPEN_BOOTP_DHCP_OPTION_BOOTFILE       = 256,
  OPEN_BOOTP_DHCP_OPTION_TFTP_SVR_NAME  = 512,
  OPEN_BOOTP_DHCP_OPTION_VENDOR_SPECIFIC = 1024,
  OPEN_BOOTP_DHCP_OPTION_VENDOR_IDENT    = 2048,
  OPEN_BOOTP_DHCP_OPTION_HOSTNAME        = 4096,
  OPEN_BOOTP_DHCP_OPTION_DEF_DOMAIN_NAME = 8192,
  OPEN_BOOTP_DHCP_OPTION_NTP_SVR_ADDR    = 16384,
  OPEN_BOOTP_DHCP_OPTION_SYSLOG_SVR_ADDR = 32768
} open_bootp_dhcp_offered_options_e;

typedef struct
{
  uint32_t ip;                                    /* Host ip address */
  uint32_t netMask;                               /* Host netmask */
  uint32_t gateway;                               /* Gateway */
  uint32_t siaddr;                                /* Next server address */
  uint32_t optTftpSvrIpAddr;                      /* IP addres of TFTP server, got from option 150 field */
  uint32_t optDnsServerIpAddr[OPEN_DNS_NAME_SERVER_ENTRIES];    /* IP addres of DNS server, got from option 6 field */
  unsigned char optDefDomainName[OPENAPI_DNS_HOST_NAME_LEN_MAX + 1]; /* Domain name, got from option 15 field */
  unsigned char optHostName[OPEN_HOSTNAME_MAX_LEN + 1];                   /* hostname name, got from option 12 field */
  uint32_t optNtpServerIpAddr[OPEN_SNTP_MAX_SERVER_ENTRIES];  /* IP addres of NTP server, got from option 42 field */
  unsigned char bootFile[OPEN_BOOTP_FILE_LEN + 1];            /* Bootfile name */
  unsigned char optBootFile[OPEN_BOOTP_FILE_LEN + 1];         /* Bootfile name, got from option 67 field */
  unsigned char sname[OPEN_BOOTP_SNAME_LEN + 1];              /* Server name */
  unsigned char optTftpSvrName[OPEN_BOOTP_SNAME_LEN + 1];     /* TFTP server name, got from option 66 field */
  open_dhcp_vendor_specific_option_t *vendSpecificOptions;   /* DHCP Option-43.*/
  uint32_t offeredOptionsMask;                    /* Mask holds offered options IDs. Set mask using OR operation
                                                     with flags defined in open_bootp_dhcp_offered_options_e */
  open_dhcp_option125_t opt125;                   /* Opt 125. Vendor Identified option.*/
  OPEN_BOOL_t dhcpAddressFlushFlag;
} open_bootp_dhcp_t;

/*********************************************************************
* @purpose  Get the DHCP Client IP Address on an interface
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intIfNum         @b{(input)}  Internal interface number of the
*                                         entry
* @param    clientAddr       @b{(output)} DHCP client IPv4 address
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
**********************************************************************/
open_error_t openapiDhcpClientIPAddrGet(openapiClientHandle_t *client_handle,
                                        uint32_t intIfNum,
                                        open_inet_addr_t *clientAddr);

/*********************************************************************
* @purpose  Get the DHCP Client subnet mask on an interface
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intIfNum         @b{(input)}  Internal interface number of the
*                                         entry
* @param    netmask          @b{(output)} Network mask
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
**********************************************************************/
open_error_t openapiDhcpClientNetMaskGet(openapiClientHandle_t *client_handle,
                                         uint32_t intIfNum,
                                         open_inet_addr_t *netmask);

/*********************************************************************
* @purpose  Get the DHCP Server IP Address on an interface
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    intIfNum         @b{(input)}  Internal interface number of the
*                                         entry
* @param    serverAddr       @b{(output)} DHCP server IPv4 address
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
**********************************************************************/
open_error_t openapiDhcpClientDhcpServerIPAddrGet(openapiClientHandle_t *client_handle,
                                                  uint32_t intIfNum,
                                                  open_inet_addr_t *serverAddr);

/*********************************************************************
* @purpose  Get the DHCP Options returned by the DHCP Server
*
* @param    networkParams @b((output)) DHCP Parameters
*
* @returns  OPEN_E_NONE   if found
* @returns  OPEN_E_FAIL   if not found
* @returns  OPEN_E_PARAM  if invalid parameter is supplied
*
* @notes    This API returns the DHCP Options in the following priority
*           1. Service Port (if included in the build)
*           2. Routing/Host Interface (if Routing package is included
*                                      in the build)
*           3. Network Port (if included in the build)
* 
* @supported-in-version OpEN API Version: 1.8
* 
* @end
*********************************************************************/
open_error_t openapiDhcpClientOptionsGet(openapiClientHandle_t *client_handle,
                                         open_bootp_dhcp_t *networkParams);
#endif /* OPENAPI_DHCP_CLIENT_H_INCLUDED */
