/*********************************************************************
*
* Copyright 2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dhcp_snooping.h
*
* @purpose   DHCP Snooping API
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    7/9/2018
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_DHCP_SNOOPING_H_INCLUDED
#define OPENAPI_DHCP_SNOOPING_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
* @purpose  Gets the DHCP Snooping Admin mode
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    adminMode         @b{(output)} DHCP snooping admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments Use this function to determine if DHCP snooping is enabled.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingAdminModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_AF_t family,
                                             OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Gets whether DHCP snooping verifies source MAC addresses.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    sourceMacVerify   @b{(output)} If OPEN_ENABLE, source MAC addresses
*                                          verified, otherwise OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments When MAC verify is enabled (and port security is enabled
*           on the interface), DHCP snooping tells the hardware to
*           drop any incoming frames on the interface if the source MAC
*           is not the MAC address for a DHCP client who has been leased
*           an IP address.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingVerifySourceMacGet(openapiClientHandle_t *client_handle,
                                                   OPEN_AF_t family,
                                                   OPEN_CONTROL_t *sourceMacVerify);

/*********************************************************************
* @purpose  Given a VLAN ID, return the endpoints of the next range of
*           VLANs enabled for DHCP snooping.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    vlanStart         @b{(output)} Low end of VLAN range
* @param    vlanEnd           @b{(inout)}  High end of VLAN range
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_NOT_FOUND  No further VLANs beyond vlanEnd
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments Search for next range starts at vlanEnd + 1. To begin a
*           search, set vlanEnd to 0. To search the next following
*           range, vlanEnd should be set to the vlanEnd returned by
*           the previous call. See example code for more details.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingEnabledVlanRangeNextGet(openapiClientHandle_t *client_handle,
                                                        OPEN_AF_t family,
                                                        uint32_t *vlanStart,
                                                        uint32_t *vlanEnd);

/*********************************************************************
* @purpose  Determines whether DHCP snooping is enabled on a port.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    enabled        @b{(output)} OPEN_ENABLE if DHCP snooping is 
*                                       enabled on port, else OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments    DHCP snooping is considered enabled on a port if DHCP snooping
*              is enabled on one or more VLANs the port participates in.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingPortEnabledGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t *enabled);

/*********************************************************************
* @purpose  Determines whether DHCP snooping considers an interface as trusted.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    trusted        @b{(output)} OPEN_TRUE if trusted, else OPEN_FALSE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingIntfTrustedGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               uint32_t ifNum, 
                                               OPEN_BOOL_t *trusted);

/*********************************************************************
* @purpose  Determines whether DHCP snooping logs invalid packets on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    logged         @b{(output)} OPEN_TRUE if invalid packets logged for 
*                                       interface, else OPEN_FALSE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingIntfLoggedGet(openapiClientHandle_t *client_handle,
                                              OPEN_AF_t family,
                                              uint32_t ifNum, 
                                              OPEN_BOOL_t *logged);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name max length.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    len            @b{(output)}  Maximum length of filename including NULL.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbFileNameMaxLengthGet(openapiClientHandle_t *client_handle, uint32_t *len);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    fileName       @b((output)) Remote DHCP snooping DB file name
* @param    -                 Minimum allowed value of buffer length is iobtainable by calling openapiDhcpSnoopingFileNameMaxLengthGet()
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbFileNameGet(openapiClientHandle_t *client_handle,
                                              OPEN_AF_t family,
                                              open_buffdesc *fileName);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database IPv4 Address.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    addr           @b((output)) Remote DHCP snooping DB IP Address
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbIpAddressGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               open_inet_addr_t *addr);

/*********************************************************************
* @purpose  Get the DHCP snooping database write delay.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    uint32_t       @b((output)) Write delay in seconds.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbWriteDelayGet(openapiClientHandle_t *client_handle,
                                                OPEN_AF_t family,
                                                uint32_t *delay);

#endif /* OPENAPI_DHCP_SNOOPING_H_INCLUDED */
