/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dot1s.h
*
* @purpose   Spanning Tree Configuration and Status 
*
* @component OPEN
*
* @create    10/25/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DOT1S_H_INCLUDED
#define OPENAPI_DOT1S_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_DOT1S_FORCE_VERSION_DOT1D    = 0, /**< IEEE 802.1d Spanning Tree */
  OPEN_DOT1S_FORCE_VERSION_UNUSED   = 1, /**< unused */
  OPEN_DOT1S_FORCE_VERSION_DOT1W    = 2, /**< IEEE 802.1w Rapid Spanning Tree */
  OPEN_DOT1S_FORCE_VERSION_DOT1S    = 3, /**< IEEE 802.1s Multiple Spanning Tree */
  OPEN_DOT1S_FORCE_VERSION_LAST          /**< must be last */
}OPEN_DOT1S_FORCE_VERSION_t;

typedef enum
{
  OPEN_DOT1S_PORT_GUARD_LOOP    = 0, /**< Port Guard Loop */
  OPEN_DOT1S_PORT_GUARD_ROOT    = 1, /**< Port Guard Root */
  OPEN_DOT1S_PORT_GUARD_NONE    = 2  /**< Port Guard None */
}OPEN_DOT1S_PORT_GUARD_t;

typedef enum
{
  OPEN_DOT1S_MGMT_STR_CONFIG_NAME = 0 /**< String type Configuration name */
}OPEN_DOT1S_MGMT_STR_t;

typedef enum
{
  OPEN_DOT1S_PORT_DISCARDING        = 1, /**< Port state Discarding */
  OPEN_DOT1S_PORT_LEARNING          = 2, /**< Port state Learning */
  OPEN_DOT1S_PORT_FORWARDING        = 3, /**< Port state Forwarding */
  OPEN_DOT1S_PORT_DISABLED          = 4, /**< Port state Disabled */
  OPEN_DOT1S_PORT_MANUAL_FWD        = 5, /**< Port state Manual Forward */
  OPEN_DOT1S_PORT_NOT_PARTICIPATE   = 6  /**< Port state Not Participating */
}OPEN_DOT1S_PORT_FWD_STATE_t;

typedef enum
{
  OPEN_DOT1S_ROLE_DISABLED          = 0, /**< DisabledPort */
  OPEN_DOT1S_ROLE_ROOT              = 1, /**< RootPort */
  OPEN_DOT1S_ROLE_DESIGNATED        = 2, /**< DesignatedPort */
  OPEN_DOT1S_ROLE_ALTERNATE         = 3, /**< AlternatePort */
  OPEN_DOT1S_ROLE_BACKUP            = 4, /**< BackupPort */
  OPEN_DOT1S_ROLE_MASTER            = 5  /**< MasterPort */

}OPEN_DOT1S_PORT_ROLE_t;

#define OPENAPI_DOT1S_CST_ID 0
#define OPENAPI_DOT1S_MSTID_MAX 4094

#define OPEN_DOT1S_VLAN_ID_LEN        2
#define OPEN_DOT1S_MAC_ID_LEN         OPEN_MAC_ADDR_LEN
#define OPEN_DOT1S_VID_MAC_ADDR_LEN  (OPEN_DOT1S_VLAN_ID_LEN + OPEN_DOT1S_MAC_ID_LEN)

/** DOT1S general bridge identifier formed by combining the
 *  bridge priority, instance identifier, and MAC address into
 *  an 8-octet hex value. */

typedef struct
{
  uint8_t octets[OPEN_DOT1S_VID_MAC_ADDR_LEN];
} OPEN_DOT1S_BRIDGEID_t;


/*********************************************************************
* @purpose  Gets the Spanning Tree Management string length. This length is used to
*           allocate and initialize the open_buffdesc type argument buffers
*           used by various DOT1S OpEN APIs.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    strType         @b{(input)}   string type
* @param    strLen          @b{(output)}  string length
*
* @returns  OPEN_E_NONE        String length Get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed. 
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1sMgmtStrLenGet(openapiClientHandle_t *client_handle, 
                                       OPEN_DOT1S_MGMT_STR_t strType,
                                       uint32_t *strLen);

/*********************************************************************
* @purpose  Get the maximum number of Spanning Tree VLANs.
* 
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    vlanMax          @b{(output)} Maximum VLAN
*
* @returns  OPEN_E_NONE      if SNTP priority is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    none
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1VlanMaxGet(openapiClientHandle_t *client_handle,
                                   uint32_t *vlanMax);

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Operational Mode
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    dot1sMode       @b{(input)}   spanning tree operational mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SModeSet(openapiClientHandle_t *client_handle,
                                 OPEN_CONTROL_t dot1sMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Operational Mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    dot1sMode       @b{(output)}  spanning tree operational mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SModeGet(openapiClientHandle_t *client_handle,
                                 OPEN_CONTROL_t *dot1sMode);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Filter mode for an interface
*
* @param    client_handle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
* @param    dot1sBPDUFilterMode     @b{(input)}   spanning tree BPDU filter mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFilterModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t dot1sBPDUFilterMode);

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Filter mode for an interface
*
* @param    client_handle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
* @param    dot1sBPDUFilterMode     @b{(output)}  spanning tree BPDU filter mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFilterModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t *dot1sBPDUFilterMode);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Flood mode for an interface
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sBPDUFloodMode    @b{(input)}   spanning tree BPDU Flood mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFloodModeSet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_CONTROL_t dot1sBPDUFloodMode);

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Flood mode for an interface
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sBPDUFloodMode     @b{(output)}  spanning tree BPDU Flood mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFloodModeGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_CONTROL_t *dot1sBPDUFloodMode);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Guard Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUGuard     @b{(input)}   spanning tree BPDU guard
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUGuardModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t dot1sBPDUGuard);

/*********************************************************************
* @purpose  Gets the Spanning Tree BPDU Guard Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUGuard     @b{(output)}  spanning tree BPDU guard
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUGuardModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t *dot1sBPDUGuard);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Filter Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUFilter    @b{(input)}   spanning tree BPDU filter
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUFilterModeSet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t dot1sBPDUFilter);

/*********************************************************************
* @purpose  Gets the Spanning Tree BPDU Filter Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUFilter    @b{(output)}  spanning tree BPDU filter
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUFilterModeGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t *dot1sBPDUFilter);

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Name
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    dot1sConfigName       @b{(input)}   spanning tree configuration name
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* @notes  Maximum size for dot1sConfigName is determined by calling
*         openapiDot1sMgmtStrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigNameSet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *dot1sConfigName);

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Name
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    dot1sConfigName       @b{(output)}  spanning tree configuration name
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Maximum size for dot1sConfigName is determined by calling
*         openapiDot1sMgmtStrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigNameGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *dot1sConfigName);

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Revision
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    dot1sConfigRev      @b{(input)}   spanning tree configuration revision
* @param    -                   Minimum value @ref OPENAPI_DOT1S_MIN_REVISION_VALUE
* @param    -                   Maximum value @ref OPENAPI_DOT1S_MAX_REVISION_VALUE 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigRevisionSet(openapiClientHandle_t *client_handle,
                                           uint32_t dot1sConfigRev);

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Revision
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sConfigRev     @b{(output)}  spanning tree configuration revision
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigRevisionGet(openapiClientHandle_t *client_handle,
                                           uint32_t *dot1sConfigRev);

/*********************************************************************
* @purpose  Configures Spanning Tree Force Revision
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sForceVersion  @b{(input)}   spanning tree force version
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SForceVersionSet(openapiClientHandle_t *client_handle,
                                         OPEN_DOT1S_FORCE_VERSION_t dot1sForceVersion);

/*********************************************************************
* @purpose  Gets the Spanning Tree Force Revision
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sForceVersion  @b{(output)}  spanning tree force version
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SForceVersionGet(openapiClientHandle_t *client_handle,
                                         OPEN_DOT1S_FORCE_VERSION_t *dot1sForceVersion);

/*********************************************************************
* @purpose  Configures Spanning Tree Guard mode
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sGuardMode        @b{(input)}   spanning tree Guard mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfGuardModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                                          OPEN_DOT1S_PORT_GUARD_t dot1sGuardMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Guard mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
* @param    dot1sGuardMode     @b{(output)}  spanning tree Guard mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfGuardModeGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_DOT1S_PORT_GUARD_t *dot1sGuardMode);

/*********************************************************************
* @purpose  Creates/Deletes Spanning Tree MST Instance
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId @b{(input)}   spanning tree MST instance Id
* @param    -                  Minimum value @ref OPENAPI_DOT1S_MSTID_MIN
* @param    -                  Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
* @param    action             @b{(input)}   create/delete instance
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceSet(openapiClientHandle_t *client_handle,
                                        uint32_t dot1sMSTInstanceId,
                                        OPEN_ACTION_t action);

/*********************************************************************
* @purpose  Gets First Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    instanceId                @b{(output)}  first spanning tree MST instance Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        No next instance found.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceFirstGet(openapiClientHandle_t *client_handle,
                                             uint32_t *instanceId);

/*********************************************************************
* @purpose  Gets Next Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    currInstance              @b{(input)}   current spanning tree MST instance Id
* @param    nextInstance              @b{(output)}  next spanning tree MST instance Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        No next instance found.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceNextGet(openapiClientHandle_t *client_handle,
                                            uint32_t currInstance,
                                            uint32_t *nextInstance);

/*********************************************************************
* @purpose  Configures Spanning Tree MST Instance priority
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance    @b{(input)}   spanning tree MST instance
* @param    -                   Minimum value @ref OPENAPI_DOT1S_MSTID_MIN
* @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
* @param    priority            @b{(input)}   instance priority
* @param    -                   Minimum value @ref OPENAPI_DOT1S_PRIORITY_MIN  
* @param    -                   Maximum value @ref OPENAPI_DOT1S_PRIORITY_MAX 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstancePrioritySet(openapiClientHandle_t *client_handle, 
                                                uint32_t dot1sMSTInstance,
                                                uint32_t priority);

/*********************************************************************
* @purpose  Gets Spanning Tree MST Instance priority
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance          @b{(input)}   spanning tree MST instance
* @param    priority                  @b{(output)}  instance priority
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstancePriorityGet(openapiClientHandle_t *client_handle, 
                                                uint32_t dot1sMSTInstance,
                                                uint16_t *priority);

/*********************************************************************
* @purpose  Adds/Deletes Spanning Tree MST Instance-VLAN association
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance    @b{(input)}   spanning tree MST instance
* @param    -                   Minimum value @ref OPENAPI_DOT1S_MSTID_MIN
* @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
* @param    vlan                @b{(input)}   VLAN
* @param    -                   Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID  
* @param    -                   Maximum value openapiDot1VlanMaxGet()
* @param    action              @b{(input)}   create/delete association
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceVLANSet(openapiClientHandle_t *client_handle, 
                                            uint32_t dot1sMSTInstance,
                                            uint32_t vlan,
                                            OPEN_ACTION_t action);
/*********************************************************************
* @purpose  Gets First VLAN Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    VLANId                    @b{(output)}  VLAN Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceFirstVLANGet(openapiClientHandle_t *client_handle,
                                                 uint32_t dot1sInstance,
                                                 uint32_t *VLANId);

/*********************************************************************
* @purpose  Gets Next VLAN Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    currVLANId                @b{(input)}   current VLAN Id
* @param    nextVLANId                @b{(output)}  next VLAN Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceNextVLANGet(openapiClientHandle_t *client_handle,
                                                uint32_t dot1sInstance,
                                                uint32_t currVLANId, uint32_t *nextVLANId);

/*********************************************************************
* @purpose  Gets First Forwarding Database Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    FId                       @b{(output)}  Forwarding Database Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceFirstFIdGet(openapiClientHandle_t *client_handle,
                                                uint32_t dot1sInstance,
                                                uint32_t *FId);

/*********************************************************************
* @purpose  Gets Next Forwarding Database Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    currFId                   @b{(input)}   current Forwarding Database Id
* @param    nextFId                   @b{(output)}  next Forwarding Database Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceNextFIdGet(openapiClientHandle_t *client_handle,
                                               uint32_t dot1sInstance,
                                               uint32_t currFId, uint32_t *nextFId);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Bridge Identifier for a specific instance.
 *
 * @param    client_handle    @b{(input)}   client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}   spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
 * @param    id               @b{(output)}   bridge identifier 
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceBridgeIdentifierGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t dot1sMSTInstance, 
                                                        OPEN_DOT1S_BRIDGEID_t *id);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Time since the last Topology Change for a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
 * @param    timeVal          @b{(output)}   time value, in seconds 
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(openapiClientHandle_t *client_handle, 
                                                               uint32_t dot1sMSTInstance, 
                                                               uint32_t *timeVal);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST count of Topology Changes in a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    count            @b{(output)}   number of TCNs received
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceTopologyChangeCountGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t mstId, 
                                                           uint32_t *count);

/*****************************************************************************
 * @purpose  Get if a toplogy change is in progress for any port assigned to the MST instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    change           @b{(output)}   OPEN_TRUE, a change in progress
 *                                           OPEN_FALSE, no change in progress.
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceTopologyChangeStateGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t dot1sMSTInstance, 
                                                           OPEN_BOOL_t *change);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Designated Root Identifier in a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    rootId           @b{(output)}   designated root identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceDesignatedRootIDGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t dot1sMSTInstance,
                                                        OPEN_DOT1S_BRIDGEID_t *rootId);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Designated Root Path Cost for a specific instance
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    cost             @b{(output)}   designated root path cost
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceRootPathCostGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t dot1sMSTInstance, 
                                                    uint32_t *cost);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Root Port ID in a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    port             @b{(output)}   port ID of the root port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceRootPortIDGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t dot1sMSTInstance, 
                                                  uint32_t *port);

/*****************************************************************************
 * @purpose   Get the max age for the CIST.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     age            @b{(output)}   the current CIST max age
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTMaxAgeGet(openapiClientHandle_t *client_handle, 
                                       uint32_t *age);

/*****************************************************************************
 * @purpose   Get the Max Hop count for the device.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     count          @b{(output)}   max hop count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sBridgeMaxHopGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *count);

/*****************************************************************************
 * @purpose   Get the Tx Hold count for the device.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     count          @b{(output)}   Tx hold count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sBridgeTxHoldCountGet(openapiClientHandle_t *client_handle, 
                                              uint32_t *count);

/*****************************************************************************
 * @purpose   Get the CIST Bridge Forward Delay.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     delay          @b{(output)}   forward delay for the CIST
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTRootFwdDelayGet(openapiClientHandle_t *client_handle, 
                                             uint32_t *delay);

/*****************************************************************************
 * @purpose   Get the CIST Hello Time of a specific port.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     ifNum          @b{(input)}    interface number
 * @param     helloTime      @b{(output)}   hello time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTHelloTimeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, 
                                          uint32_t *helloTime);

/*****************************************************************************
 * @purpose   Get the CIST Bridge Hold Time.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     holdTime       @b{(output)}   hold time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTBridgeHoldTimeGet(openapiClientHandle_t *client_handle, 
                                               uint32_t *holdTime);

/*****************************************************************************
 * @purpose   Get the CIST Regional Root Identifier.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     rootId         @b{(output)}   regional root identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTRegionalRootIDGet(openapiClientHandle_t *client_handle, 
                                               OPEN_DOT1S_BRIDGEID_t *rootId);

/*****************************************************************************
 * @purpose   Get the CIST Regional Root Path Cost.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     cost           @b{(output)}   regional root path cost
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTRegionalRootPathCostGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t *cost);

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Admin Mode on an interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    dot1sMode       @b{(input)}   spanning tree mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum, 
                                     OPEN_CONTROL_t dot1sMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Admin Mode on an interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    dot1sMode       @b{(output)}  spanning tree admin mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum, 
                                     OPEN_CONTROL_t *dot1sMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Port Forwarding State
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
* @param    dot1sPortFwdState      @b{(output)}  spanning tree port forwarding state
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfPortFwdStateGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             uint32_t dot1sInstance,
                                             OPEN_DOT1S_PORT_FWD_STATE_t *dot1sPortFwdState);

/*********************************************************************
* @purpose  Gets Spanning Tree Port Role
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
* @param    dot1sPortRole          @b{(output)}  spanning tree port role
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfPortRoleGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         uint32_t dot1sInstance,
                                         OPEN_DOT1S_PORT_ROLE_t *dot1sPortRole);

/*********************************************************************
* @purpose  Configures the Spanning Tree MST Port Administrative Edge Port mode of an interface for the CIST
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
* @param    loopInconsistentState  @b{(output)}  loop inconsistent state (a boolean value)
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfPortLoopInconsistentStateGet(openapiClientHandle_t *client_handle,
                                                          uint32_t ifNum, 
                                                          uint32_t dot1sInstance,
                                                          OPEN_BOOL_t *loopInconsistentState);

/*********************************************************************
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sEdgePortMode     @b{(input)}   spanning tree edge port mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfEdgePortModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             OPEN_CONTROL_t dot1sEdgePortMode);

/*********************************************************************
* @purpose  Gets the Spanning Tree MST Port Administrative Edge Port mode of an
*           interface for the CIST
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sEdgePortMode     @b{(output)}  spanning tree edge port mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfEdgePortModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             uint32_t *dot1sEdgePortMode);

#endif /* OPENAPI_DOT1S_H_INCLUDED */
