/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_if.h
*
* @purpose   Interface Configuration and Status
*
* @component OPEN
*
* @create    10/23/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_IF_H_INCLUDED
#define OPENAPI_IF_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_PHY_CAP_PORTSPEED_HALF_10    = 0x000000001,
  OPEN_PHY_CAP_PORTSPEED_FULL_10    = 0x000000002,
  OPEN_PHY_CAP_PORTSPEED_HALF_100   = 0x000000004,
  OPEN_PHY_CAP_PORTSPEED_FULL_100   = 0x000000008,
  OPEN_PHY_CAP_PORTSPEED_HALF_1000  = 0x000000010,
  OPEN_PHY_CAP_PORTSPEED_FULL_1000  = 0x000000020,
  OPEN_PHY_CAP_PORTSPEED_FULL_2P5G  = 0x000000040,
  OPEN_PHY_CAP_PORTSPEED_FULL_5G    = 0x000000080,
  OPEN_PHY_CAP_PORTSPEED_FULL_10G   = 0x000000100,
  OPEN_PHY_CAP_PORTSPEED_AUTO_NEG   = 0x000000400,
  OPEN_PHY_CAP_PORTSPEED_SFP        = 0x000000800,
  OPEN_PHY_CAP_PORTSPEED_SFP_DETECT = 0x000001000,
  OPEN_PHY_CAP_POE_PSE              = 0x000002000,
  OPEN_PHY_CAP_POE_PD               = 0x000004000,
  OPEN_PHY_CAP_INTERNAL             = 0x000008000,  /* Indicates a blade switch internal port */
  OPEN_PHY_CAP_NATIVE_EEE           = 0x000010000,
  OPEN_PHY_CAP_AUTO_EEE             = 0x000020000,
  OPEN_PHY_CAP_DUAL_MODE_SUPPORT    = 0x000040000,   /* Used represent PHYs where phy/port capabilities change
                                                  dynamically(ex: SFP support in 10G fiber port, Combo port where two phys
                                                  are used one for fiber, one for copper)*/
  OPEN_PHY_CAP_POE_PSE_PLUS         = 0x000080000,
  OPEN_PHY_CAP_POE_PD_PLUS          = 0x000100000,
  OPEN_PHY_CAP_ENERGY_DETECT        = 0x000200000,
  OPEN_PHY_CAP_COMBO_PREFERENCE     = 0x000400000,  /* If the port is a combo port *and* the preferred media can be configured. */
  OPEN_PHY_CAP_EXPANDABLE_PORT      = 0x000800000,
  OPEN_PHY_CAP_AUTO_CONFIG          = 0x001000000,
  OPEN_PHY_CAP_SHORT_REACH          = 0x002000000,
  OPEN_PHY_CAP_UPOE_PSE             = 0x004000000,  /* UPOE capability bits */
  OPEN_PHY_CAP_UPOE_PD              = 0x008000000,
  OPEN_PHY_CAP_DYNAMIC_PORT         = 0x010000000,  /* This should be used instead of DUAL_MODE */
  OPEN_PHY_CAP_FIXED_STACK_PORT     = 0x020000000,

  OPEN_PHY_CAP_FEC_SUPPORT          = 0x040000000,
  OPEN_PHY_CAP_PORTSPEED_FULL_20G   = 0x080000000,
  OPEN_PHY_CAP_PORTSPEED_FULL_25G   = 0x081000000,
  OPEN_PHY_CAP_PORTSPEED_FULL_40G   = 0x082000000,
  OPEN_PHY_CAP_PORTSPEED_FULL_50G   = 0x084000000,
  OPEN_PHY_CAP_PORTSPEED_FULL_100G  = 0x088000000,
/* Add support for 64-Bit. Commenting for now, to be fixed by UI team. overlapping values need to removed
   pls check commdefs.h
  OPEN_PHY_CAP_PORTSPEED_FULL_25G   = 0x100000000ULL,
  OPEN_PHY_CAP_PORTSPEED_FULL_40G   = 0x200000000ULL,
  OPEN_PHY_CAP_PORTSPEED_FULL_50G   = 0x400000000ULL,
  OPEN_PHY_CAP_PORTSPEED_FULL_100G  = 0x800000000ULL,
  OPEN_PHY_CAP_PORTSPEED_FULL_200G  = 0x1000000000ULL,
  OPEN_PHY_CAP_PORTSPEED_FULL_400G  = 0x2000000000ULL,
*/
  /* Add new caps below */

} OPEN_PHY_CAPABILITIES_t;

typedef enum
{
  OPEN_PORTCTRL_PORTSPEED_AUTO_NEG      = 1,
  OPEN_PORTCTRL_PORTSPEED_HALF_100TX    = 2,
  OPEN_PORTCTRL_PORTSPEED_FULL_100TX    = 3,
  OPEN_PORTCTRL_PORTSPEED_HALF_10T      = 4,
  OPEN_PORTCTRL_PORTSPEED_FULL_10T      = 5,
  OPEN_PORTCTRL_PORTSPEED_FULL_100FX    = 6,
  OPEN_PORTCTRL_PORTSPEED_FULL_1000SX   = 7,
  OPEN_PORTCTRL_PORTSPEED_FULL_10GSX    = 8,
  OPEN_PORTCTRL_PORTSPEED_FULL_20GSX    = 9,
  OPEN_PORTCTRL_PORTSPEED_FULL_40GSX    = 10,
  OPEN_PORTCTRL_PORTSPEED_FULL_25GSX    = 11,
  OPEN_PORTCTRL_PORTSPEED_FULL_50GSX    = 12,
  OPEN_PORTCTRL_PORTSPEED_FULL_100GSX   = 13,
  OPEN_PORTCTRL_PORTSPEED_AAL5_155      = 14,
  OPEN_PORTCTRL_PORTSPEED_FULL_5FX      = 15,
  OPEN_PORTCTRL_PORTSPEED_FULL_200GSX   = 16,
  OPEN_PORTCTRL_PORTSPEED_FULL_400GSX   = 17,
  OPEN_PORTCTRL_PORTSPEED_FULL_2P5FX    = 128,	/* Speed 2.5G FD, Bit 7th set */
  OPEN_PORTCTRL_PORTSPEED_LAG           = 129,
  OPEN_PORTCTRL_PORTSPEED_UNKNOWN       = 130
} OPEN_PORT_SPEEDS_t;

/** OPEN uses these enumerators to indicate the reason a port is
 *  diagnostically disabled. */
typedef enum
{
  OPEN_DISABLE_REASON_NONE                  = 1,
  OPEN_DISABLE_REASON_UDLD                  = 2,
  OPEN_DISABLE_REASON_STP                   = 3,
  OPEN_DISABLE_REASON_BPDUGUARD             = 4,
  OPEN_DISABLE_REASON_BPDUSTORM             = 5,
  OPEN_DISABLE_REASON_DHCPSNOOPING          = 6,
  OPEN_DISABLE_REASON_ARPINSPECTION         = 7,
  OPEN_DISABLE_REASON_STORMCONTROL_BCAST    = 8,
  OPEN_DISABLE_REASON_STORMCONTROL_MCAST    = 9,
  OPEN_DISABLE_REASON_STORMCONTROL_UCAST    = 10,
  OPEN_DISABLE_REASON_XCEIVER               = 11,
  OPEN_DISABLE_REASON_XCEIVERPLUS           = 12,
  OPEN_DISABLE_REASON_LOOPPROTECT           = 13,
  OPEN_DISABLE_REASON_PML                   = 14,
  OPEN_DISABLE_REASON_DOS                   = 15,
  OPEN_DISABLE_REASON_LINK_FLAP             = 16,
  OPEN_DISABLE_REASON_AUTHMGR_AUTH_VIOLATION  = 17
} OPEN_DISABLE_REASON_t;

/** OPEN uses these enumerators to indicate the interface
 *  active media type. */
typedef enum
{
  OPEN_PORT_CABLE_MEDIA_TYPE_NONE       = 0,
  OPEN_PORT_CABLE_MEDIA_TYPE_COPPER     = 1,
  OPEN_PORT_CABLE_MEDIA_TYPE_FIBER      = 2,
} OPEN_PORT_CABLE_MEDIA_TYPE_t;

/** OPEN uses these enumerators to indicate a port that is
 *  currently being used for a special purpose, such as as a
 *  trunk member, probe port, etc. */
typedef enum
{
  OPEN_SPECIAL_PORT_TYPE_NORMAL         = 1,
  OPEN_SPECIAL_PORT_TYPE_TRUNK_MEMBER   = 2,
  OPEN_SPECIAL_PORT_TYPE_MIRRORED_PORT  = 3,
  OPEN_SPECIAL_PORT_TYPE_PROBE_PORT     = 4,
  OPEN_SPECIAL_PORT_TYPE_L2TUNNEL_PORT  = 5,
  OPEN_SPECIAL_PORT_TYPE_SERVICE_PORT   = 6
} OPEN_SPECIAL_PORT_TYPE_t;

typedef enum
{
  OPEN_PORT_DUPLEX_HALF     = 0,
  OPEN_PORT_DUPLEX_FULL     = 1,
  OPEN_PORT_DUPLEX_UNKNOWN  = 0xffff

} OPEN_PORT_DUPLEX_t;

typedef enum
{
  OPEN_PORT_AUTO_NEG_ENABLE  = 0,
  OPEN_PORT_AUTO_NEG_DISABLE = 1,
  OPEN_PORT_AUTO_NEG_UNKNOWN = 0xffff

} OPEN_PORT_AUTO_NEG_t;

#define OPENAPI_INTERFACE_NAME_LENGTH 60

/**********************************************************************************
* @purpose  Gets the speed status of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    speed          @b{(output)}  speed of the interface.
*                          returns OPEN_PORTCTRL_PORTSPEED_UNKNOWN when link is down
*
* @returns  OPEN_E_NONE        speed of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the speed of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
***************************************************************************************/
open_error_t openapiIfSpeedGet (openapiClientHandle_t * client_handle,
				uint32_t ifNum, OPEN_PORT_SPEEDS_t * speed);

/*********************************************************************
* @purpose  Sets the speed of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    speed          @b{(input)}   speed of the interface.
*
* @returns  OPEN_E_NONE         Speed of the interface set successfully.
* @returns  OPEN_E_FAIL         Failed to set the speed of the interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfSpeedSet (openapiClientHandle_t * client_handle,
				uint32_t ifNum, OPEN_PORT_SPEEDS_t speed);


/*********************************************************************
* @purpose  Gets the duplex mode of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1  
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    duplex_mode    @b{(output)}  duplex mode of the interface.
*                          returns OPEN_PORT_DUPLEX_UNKNOWN when duplex is auto
*
* @returns  OPEN_E_NONE        duplex mode of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the duplex mode of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfDuplexModeGet (openapiClientHandle_t * client_handle,
				     uint32_t ifNum, OPEN_PORT_DUPLEX_t * duplex_mode);

/*********************************************************************
* @purpose  Gets the auto-negotiation mode of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    autoneg        @b{(output)}  auto-negotiation mode of the interface.
*
* @returns  OPEN_E_NONE   auto-negotiation mode of the interface returned.
* @returns  OPEN_E_FAIL   Failed to get the auto-negotiation mode of the interface.
* @returns  OPEN_E_PARAM  Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfAutoNegGet (openapiClientHandle_t * client_handle,
				  uint32_t ifNum, OPEN_PORT_AUTO_NEG_t * autoneg);

/*********************************************************************
* @purpose  Sets the auto-negotiation mode of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    autoneg        @b{(input)}   auto-negotiation mode of the interface.
*
* @returns  OPEN_E_NONE    auto-negotiation mode of the interface set successfully.
* @returns  OPEN_E_FAIL    Failed to set the auto-negotiation mode of the interface.
* @returns  OPEN_E_PARAM   Error in parameter passed.
* @returns  OPEN_E_UNAVAIL Requested mode is not available for this interface. 
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfAutoNegSet (openapiClientHandle_t * client_handle,
				  uint32_t ifNum, uint32_t autoneg);

/*********************************************************************
* @purpose  Gets the ethernet header size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    headerSize     @b{(output)}  Ethernet header size
*
* @returns  OPEN_E_NONE        On success.
* @returns  OPEN_E_FAIL        On failure.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfEthernetHeaderSizeGet(openapiClientHandle_t *client_handle,
          uint32_t ifNum, uint32_t *headerSize);

/*********************************************************************
* @purpose  Gets the MTU size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    mtu_size       @b{(output)}  MTU size of the interface.
*
* @returns  OPEN_E_NONE        MTU size of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the MTU size of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfMtuSizeGet (openapiClientHandle_t * client_handle,
				  uint32_t ifNum, uint32_t * mtu_size);

/*********************************************************************
* @purpose  Sets the MTU size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    mtu_size       @b{(input)}   MTU size of the interface.
*
* @returns  OPEN_E_NONE        MTU size of the interface set successfully.
* @returns  OPEN_E_FAIL        Failed to set the MTU size of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
* @notes  This function, although named openapiIfMtuSizeSet, actually is
*         setting the maximum frame size (i.e., ethernet packet size)
*
* @open-status-deprecated @replaced-by openapiIfMtuSizeSetV2()
* @deprecated-in-version OpEN API Version: 1.18
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfMtuSizeSet (openapiClientHandle_t * client_handle,
				  uint32_t ifNum, uint32_t mtu_size);

/*********************************************************************
* @purpose  Sets the MTU size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    mtu_size       @b{(input)}   MTU size of the interface.
*
* @returns  OPEN_E_NONE        MTU size of the interface set successfully.
* @returns  OPEN_E_FAIL        Failed to set the MTU size of the interface.
* @returns  OPEN_E_ERROR       FIP snooping is enabled and resulting MTU is 
*                              less than the minimum FCoE frame size.  
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
* @notes  The MTU describes the maximum layer 3 packet size excluding headers.
* @notes  The maximum frame size will be set to mtu_size + frame header size as 
*         a result of calling this function. 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfMtuSizeSetV2(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum, 
                                   uint32_t mtu_size);

/*********************************************************************
* @purpose  Sets the Maximum frame size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    maxFrameSize   @b{(input)}   Maximum frame size of the interface.
*
* @returns  OPEN_E_NONE        On success
* @returns  OPEN_E_FAIL        Failed to set the Maximum frame size of the interface.
* @returns  OPEN_E_ERROR       FIP snooping is enabled and resulting MTU is 
*                              less than the minimum FCoE frame size.  
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
* @notes  Calling this function will also set the MTU, which will be 
*         set to maxFrameSize - the size of the ethernet headers.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfMaxFrameSizeSet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum, uint32_t maxFrameSize);

/*********************************************************************
* @purpose  Gets the Maximum frame size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    maxFrameSize   @b{(input)}   Maximum frame size of the interface.
*
* @returns  OPEN_E_NONE        On success
* @returns  OPEN_E_FAIL        Failed to get the Maximum frame size of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfMaxFrameSizeGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum, uint32_t *maxFrameSize);

/*********************************************************************************************
* @purpose  Gets the Soft-shutdown status of  a given interface.
*           When the soft shutdown is set to OPEN_ENABLE, no traffic other than
*           LLDPDU and ISDPDU packets will be sent or received on the interface
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    soft_shut      @b{(output)}  soft-shutdown status of the interface.
*
* @returns  OPEN_E_NONE        The soft-shutdown status of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the soft-shutdown status of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Feature not supported.
*
* @Note  Vlan configurations and ingress filtering configuration will not effect 
*        on the interface until soft-shutdown is disabled.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
**********************************************************************************************/
open_error_t openapiIfSoftShutdownStateGet (openapiClientHandle_t *
					    client_handle, uint32_t ifNum,
					    OPEN_CONTROL_t * soft_shut);

/****************************************************************************************
* @purpose  Sets the Soft-shutdown status of  a given interface.
*           When the soft shutdown is set to OPEN_ENABLE, no traffic other than
*           LLDPDU and ISDPDU packets will be sent or received on the interface
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    soft_shut      @b{(input)}   soft-shutdown status of the interface.
*
* @returns  OPEN_E_NONE        The soft-shutdown status of the interface set successfully.
* @returns  OPEN_E_FAIL        Failed to set the soft-shutdown status of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Feature not supported.
*
* @Note  Vlan configurations and ingress filtering configuration will not effect 
*        on the interface until soft-shutdown is disabled. 
*
* @notes  Calling this API will change the running configuration of the switch
* @supported-in-version OpEN API Version: 1.1
*
* @end
*****************************************************************************************/
open_error_t openapiIfSoftShutdownStateSet (openapiClientHandle_t *
				            client_handle, uint32_t ifNum,
					    OPEN_CONTROL_t soft_shut);

/*********************************************************************
* @purpose  Gets the MAC address size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    size           @b{(output)}  MAC address size of the interface.
*
* @returns  OPEN_E_NONE        The MAC address size of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the mac address size of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfMacAddrSizeGet (openapiClientHandle_t * client_handle,
				  uint32_t ifNum, uint32_t * size);


/*********************************************************************
* @purpose  Gets the MAC address of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    mac_addr       @b{(output)}  MAC address of the interface.
*                          returns array of MAC address.
*                          use openapiIfMacAddrSizeGet to get the size of MAC
*
* @returns  OPEN_E_NONE        The MAC address of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the mac address of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfMacAddrGet (openapiClientHandle_t * client_handle,
				  uint32_t ifNum, open_buffdesc * mac_addr);

/*********************************************************************
* @purpose  Gets the Description size of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    size           @b{(output)}  Description size of the interface.
*
* @returns  OPEN_E_NONE        Description size of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the description size of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfDescrSizeGet (openapiClientHandle_t * client_handle,
				uint32_t ifNum, uint32_t* size);


/*********************************************************************
* @purpose  Gets the Description of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    descrip        @b{(output)}  Description of the interface.
*                          use openapiIfDescrSizeGet to get the size of Description
*
* @returns  OPEN_E_NONE        Description the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the description of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfDescrGet (openapiClientHandle_t * client_handle,
				uint32_t ifNum, open_buffdesc * descrip);

/*********************************************************************
* @purpose  Sets the Description of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    descrip        @b{(input)}   Description of the interface.
*                                        upto OPEN_INTF_DESCR_SIZE
*
* @returns  OPEN_E_NONE        Description of the interface successfully set
* @returns  OPEN_E_FAIL        Failed to set the description of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfDescrSet (openapiClientHandle_t * client_handle,
				uint32_t ifNum, open_buffdesc * descrip);

/*********************************************************************
* @purpose  Sets the administrative state for the specified interface.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ifNum           @b{(input)}  ID of the interface.
* @param    adminState      @b{(input)}  Administrative state to set.
*
* @returns  OPEN_E_NONE        Administrative state set.
* @returns  OPEN_E_FAIL        Administrative state set failed.
* @returns  OPEN_E_NOT_FOUND   Interface not found.
* @returns  OPEN_E_UNAVAIL     Interface not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfAdminStateSet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    OPEN_CONTROL_t adminState);

/*********************************************************************
* @purpose  Gets the interface identifier for the first interface of
*           the specified interface type.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifType       @b{(input)}   Type of interface.
* @param    firstIfNum   @b{(output)}  Storage for the returned interface ID
*
* @returns  OPEN_E_NONE        The first interface of the specified type returned.
* @returns  OPEN_E_FAIL        No interface of the specified type was found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfFirstGet(openapiClientHandle_t *client_handle,
                               OPEN_INTF_TYPE_t ifType, 
                               uint32_t *firstIfNum);

/*********************************************************************
* @purpose  Gets the interface identifier for the first interface of
*           the specified interface type that has the next
*           identifier.  Used to iterate over interfaces of a type in
*           the system.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifType       @b{(input)}   Type of interface.To get all interface
*                                      types pass OPEN_INTF_TYPE_ANY
* @param    ifNum        @b{(input)}   ID of the current interface.
* @param    nextIfNum    @b{(output)}  Storage for the returned interface ID
*
* @returns  OPEN_E_NONE        The next interface of the specified type returned.
* @returns  OPEN_E_FAIL        No next interface of the specified type was found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfNextGet(openapiClientHandle_t *client_handle,
                              OPEN_INTF_TYPE_t ifType, 
                              uint32_t ifNum, 
                              uint32_t *nextIfNum);

/*********************************************************************
* @purpose  Gets the link state for the specified interface.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum        @b{(input)}   ID of the interface.
* @param    linkState    @b{(output)}  Storage for the returned link state.
*
* @returns  OPEN_E_NONE        Link state returned.
* @returns  OPEN_E_FAIL        Link state retrieval failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfLinkStateGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum, 
                                   OPEN_LINK_STATE_t *linkState);

/*********************************************************************
* @purpose  Gets the administrative state for the specified interface.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum        @b{(input)}   ID of the interface.
* @param    adminState   @b{(output)}  Storage for the returned link state.
*
* @returns  OPEN_E_NONE        Administrative state returned.
* @returns  OPEN_E_FAIL        Administrative state retrieval failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfAdminStateGet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    OPEN_CONTROL_t *adminState);

/*********************************************************************
* @purpose  Gets the displayable name for the specified interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intf             @b{(input)}   ID of the interface.
* @param    intfName         @b{(output)}  Storage for the interface name.
*                            Buffer length value @ref OPENAPI_INTERFACE_NAME_LENGTH
*
* @returns  OPEN_E_NONE        Interface name returned.
* @returns  OPEN_E_FAIL        Interface name retrieval failed.
* @returns  OPEN_E_NOT_FOUND   Interface does not exist.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfNameGet(openapiClientHandle_t *client_handle,
                              uint32_t intf, 
                              open_buffdesc *intfName);

/*********************************************************************
* @purpose  Gets the interface number given the displayable name for the interface.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    intfName     @b{(input)}  The display string for the interface.
* @param    ifNum        @b{(output)}   ID of the interface.
*
* @returns  OPEN_E_NONE        Interface name returned.
* @returns  OPEN_E_FAIL        Interface name retrieval failed.
* @returns  OPEN_E_NOT_FOUND   Interface does not exist.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfNumGet (openapiClientHandle_t * client_handle,
		                      open_buffdesc *intfName, uint32_t *ifNum );

/*********************************************************************
* @purpose  Gets the next interface which has a pending change or a
*           pending deletion.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum        @b{(input)}   ID of the current interface.
* @param    nextIfNum    @b{(output)}  Storage for the returned interface ID
* @param    delete_pending    @b{(output)}  Flag Indicating that the interface
*                                   is pending for deletion.
*
* @returns  OPEN_E_NONE        The next interface of the specified type returned.
* @returns  OPEN_E_FAIL        No next interface of the specified type was found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIfNextChangedGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum,
                                     uint32_t *nextIfNum,
                                     uint32_t *delete_pending);

/*********************************************************************
* @purpose  Gets the interface type associated with the internal
*           interface number.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  ID of the interface.
* @param    ifType          @b{(output)} Interface type.
*
* @returns  OPEN_E_NONE        Interface type returned.
* @returns  OPEN_E_FAIL        Interface type retrieval failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiIfTypeGet(openapiClientHandle_t * client_handle,
                              uint32_t ifNum,
                              OPEN_INTF_TYPE_t *ifType);

/*********************************************************************
* @purpose  Gets the time of last link state change in seconds.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIfLastChangeGet(openapiClientHandle_t *client_handle,
                                    uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  Gets the total number of octets received on the interface
*           including framing characters.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIfInOctetsGet(openapiClientHandle_t *client_handle,
                                    uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  The number of packets, delivered by this sub-layer to a
*           higher sub-layer.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @comments More specifically which were not addressed to a multicast or
*           broadcast address at this sub-layer.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfInUcastPktsGet(openapiClientHandle_t *client_handle,
                                     uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  The number of inbound packets which were chosen to be discarded.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @comments More specifically even though no errors had been detected to prevent
*           their being deliverable to a higher-layer protocol.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfInDiscardsGet(openapiClientHandle_t *client_handle,
                                    uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  Returns the current value of the errors counter.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfInErrorsGet(openapiClientHandle_t *client_handle,
                                  uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  The number of packets received via the interface which were
*           discarded because of an unknown or unsupported protocol
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfInUnknownProtosGet(openapiClientHandle_t *client_handle,
                                         uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  The total number of octets transmitted out of the interface,
*           including framing characters.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfOutOctetsGet(openapiClientHandle_t *client_handle,
                                   uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  The total number of packets that higher-level protocols
*           requested be transmitted.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* 
* @comments And more specifically which were not addressed to a multicast or
*           broadcast address at this sub-layer, including
*           those that were discarded or not sent
* 
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfOutUcastPktsGet(openapiClientHandle_t *client_handle,
                                      uint32_t interface, uint32_t *value);
 
/*********************************************************************
* @purpose  The number of outbound packets which were chosen to be discarded.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @comments    Specifically those which were chosen to be discarded
*              even though no errors had been detected to prevent
*              their being transmitted
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfOutDiscardsGet(openapiClientHandle_t *client_handle,
                                     uint32_t interface, uint32_t *value);

/*********************************************************************
* @purpose  For packet-oriented interfaces, the number of outbound
*           packets that could not be transmitted because of errors.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    interface       @b{(input)}  internal interface number.
* @param    value           @b{(output)} returned value of counter.
*
* @returns  OPEN_E_NONE     If the counter requested exist.
* @returns  OPEN_E_FAIL     If the counter requested does not exist.
* @returns  OPEN_E_PARAM    Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiIfOutErrorsGet(openapiClientHandle_t *client_handle,
                                   uint32_t interface, uint32_t *value);

/**********************************************************************************
* @purpose  Gets the speed mode of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1  
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    speed          @b{(output)}  speed of the interface.
*
* @returns  OPEN_E_NONE        Speed of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the speed of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiIfSpeedModeGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum, OPEN_PORT_SPEEDS_t *speed);


/*********************************************************************
* @purpose  Gets the duplex status of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1  
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    duplex_mode    @b{(output)}  duplex mode of the interface.
*                          returns OPEN_PORT_DUPLEX_UNKNOWN when link is down
* 
* @returns  OPEN_E_NONE        Duplex mode of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the duplex mode of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIfDuplexGet(openapiClientHandle_t *client_handle,
                                uint32_t ifNum, OPEN_PORT_DUPLEX_t *duplex_mode);

/*********************************************************************
* @purpose  Gets the capabilities of a given interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1  
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    portCapability @b{(output)}  capabilities of the interface.
* 
* @returns  OPEN_E_NONE        Capabilities of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the capabilities of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_ERROR       If interface is LAG member.
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiIfCapabilitiesGet(openapiClientHandle_t * client_handle,
                                      uint32_t ifNum, uint64_t *portCapability);

/*********************************************************************
* @purpose  Gets the disable reason code for an interface.
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1  
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    reason         @b{(output)}  The disable reason code for the interface
*
* @returns  OPEN_E_NONE        Disable reason code of the interface returned.
* @returns  OPEN_E_FAIL        Failed to get the disable reason code of the interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIfDisableReasonGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       OPEN_DISABLE_REASON_t *reason);

/*********************************************************************
* @purpose  Translate internal interface number into root slot and port.
*           Primarily used for translating an expanded port interface
*           into its root port.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    rootSlot       @b{(output)}  root slot number.
* @param    rootPort       @b{(output)}  root port number.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to translate interface number.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIfExpPortRootPortGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         uint32_t *rootSlot,
                                         uint32_t *rootPort);

/*********************************************************************
* @purpose Determine the number of lanes running under this port based
*          on the operating mode of this port.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    laneCount      @b{(output)}  Number of lanes for this port.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to determine number of lanes.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIfExpPortOperModeLaneCountGet(openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum,
                                                  uint32_t *laneCount);

/*********************************************************************
* @purpose Get the interface number of the port's first and presumably
*          the smallest peer interface.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    peerIfNum      @b{(output)}  Internal interface number of the first peer
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to determine number of lanes.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIfExpPortFirstPeerGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          uint32_t *peerIfNum);

/*********************************************************************
* @purpose Determine the number of lanes running under this port based
*          on the running speed of this port.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    laneCount      @b{(output)}  Number of lanes for this port.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to determine number of lanes.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiIfPortSpeedLaneCountGet(openapiClientHandle_t *client_handle,
                                            uint32_t ifNum,
                                            uint32_t *laneCount);

/*********************************************************************
* @purpose Get the active media type for a given interface.
*
* @param    client_handle  @b{(input)}   Client handle from registration API.
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    mediaType      @b{(output)}  Active media type.
* @param    -              See @ref OPEN_PORT_CABLE_MEDIA_TYPE_t for possible values.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to get value.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_ERROR       Interface not in attached state. 
* @returns  OPEN_E_UNAVAIL     Not supported on this interface.
* @returns  OPEN_E_INTERNAL    Other internal error occurred.
* 
* @comments This operation is valid only for physical interfaces currently
*           in an attached state.
* 
* @comments If the mediaType is reported as @ref OPEN_PORT_CABLE_MEDIA_TYPE_FIBER,
*           the @ref openapiBoxsSfpTypeGet() API can provide information on
*           the type of fiber module in use.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfActiveMediaTypeGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         OPEN_PORT_CABLE_MEDIA_TYPE_t *mediaType);

/*********************************************************************
* @purpose Get the negotiated Rx and Tx pause status of a given interface.
*
* @param    client_handle  @b{(input)}   Client handle from registration API.
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    rxPauseAllowed @b{(output)}  Receive pause allowed by autonegotiation
* @param    txPauseAllowed @b{(output)}  Transmit pause allowed by autonegotiation
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to get value.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_ERROR       Interface not in attached state. 
* @returns  OPEN_E_UNAVAIL     Not supported on this interface.
* @returns  OPEN_E_INTERNAL    Other internal error occurred.
* 
* @comments This operation is valid only for physical interfaces currently
*           in an attached state.
* 
* @comments If @ref OPEN_TRUE is indicated for either rxPauseAllowed and/or txPauseAllowed,
*           then the interface flow control operational state is considered to be active.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfPauseStatusGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum,
                                     OPEN_BOOL_t *rxPauseAllowed,
                                     OPEN_BOOL_t *txPauseAllowed);

/*********************************************************************
* @purpose Get the special port type designation of a given interface.
*
* @param    client_handle  @b{(input)}   Client handle from registration API.
* @param    ifNum          @b{(input)}   Interface number.
* @param    -              Minimum value 1
* @param    -              Maximum value openapiMaxInterfaceCountGet()
* @param    specialType    @b{(output)}  Special port type.
* @param    -              See @ref OPEN_SPECIAL_PORT_TYPE_t for possible values.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Failed to get value.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal error occurred.
* 
* @comments A port is designated as having a "special type" when, for example,
*           it becomes a trunk (LAG) member; is a mirrored, probe, or tunnel port;
*           or is a service port.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiIfSpecialPortTypeGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         OPEN_SPECIAL_PORT_TYPE_t *specialType);

#endif /* OPENAPI_IF_H_INCLUDED */
