/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_lag.h
*
* @purpose   LAG Configuration and Status 
*
* @component OPEN
*
* @create    10/10/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_LAG_H_INCLUDED
#define OPENAPI_LAG_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

/** Hash Mode names */
typedef enum
{
  OPEN_LAG_HASHING_MODE_SA_VLAN     = 0, /**< Src MAC, VLAN, EType, incoming port */
  OPEN_LAG_HASHING_MODE_DA_VLAN     = 1, /**< Dest MAC, VLAN, EType, incoming port */
  OPEN_LAG_HASHING_MODE_SDA_VLAN    = 2, /**< Src/Dest MAC, VLAN, EType, incoming port */
  OPEN_LAG_HASHING_MODE_SIP_SPORT   = 3, /**< Src IP and Src TCP/UDP Port fields */
  OPEN_LAG_HASHING_MODE_DIP_DPORT   = 4, /**< Dest IP and Dest TCP/UDP Port fields */
  OPEN_LAG_HASHING_MODE_SDIP_DPORT  = 5, /**< Src/Dest IP and TCP/UDP Port fields */
  OPEN_LAG_HASHING_MODE_ENHANCED    = 6  /**< Enhanced hashing mode */
} OPEN_LAG_HASH_MODE_t;

/** OpEN uses these enumerations to indicate DOT3AD state types. */
typedef enum
{
  OPEN_DOT3AD_STATE_AGGREGATION     = 0, /**< aggregation. */
  OPEN_DOT3AD_STATE_LACP_TIMEOUT    = 1, /**< short timeout. */
  OPEN_DOT3AD_STATE_LACP_ACTIVITY   = 2  /**< active. */
} OPEN_DOT3AD_STATE_TYPE_t;

#define OPEN_MAX_MEMBERS_PER_LAG 32
#define OPEN_LAG_MEMBER_BUFFER_SIZE (sizeof(uint32_t) * OPEN_MAX_MEMBERS_PER_LAG)

/*********************************************************************
* @purpose  Adds a physical interface to a LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    phyIfNum       @b{(input)}  physical interface.
*
* @returns  OPEN_E_NONE        physical interface added to LAG.
* @returns  OPEN_E_FAIL        Failed to add physical interface to LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLagPortAdd (openapiClientHandle_t *client_handle,
                                uint32_t lagIfNum,
                                uint32_t phyIfNum);

/*********************************************************************
* @purpose  Delete a physical interface from a LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    phyIfNum       @b{(input)} physical interface.
*
* @returns  OPEN_E_NONE        physical interface deleted from LAG.
* @returns  OPEN_E_FAIL        Failed to delete physical interface to LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLagPortDelete (openapiClientHandle_t *client_handle,
                                   uint32_t lagIfNum,
                                   uint32_t phyIfNum);

/*********************************************************************
* @purpose  Gets static mode of the LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    staticMode     @b{(output)} OPEN_ENABLE if static
*                                       OPEN_DISABLE if dynamic.
*
* @returns  OPEN_E_NONE        LAG static mode returned.
* @returns  OPEN_E_FAIL        Failed to get the static mode of the LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLagStaticModeGet (openapiClientHandle_t *client_handle,
                                      uint32_t lagIfNum,
                                      OPEN_CONTROL_t *staticMode);

/*********************************************************************
* @purpose  Sets static mode of the LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    staticMode     @b{(input)}  OPEN_ENABLE if static
*                                       OPEN_DISABLE if dynamic.
*
* @returns  OPEN_E_NONE        LAG static mode successfully set.
* @returns  OPEN_E_FAIL        Failed to set the static mode of the LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_FULL        Table is full.
*                              This may occur when a LAG's mode is set to dynamic and maximum 
*                              number of dynamic lags are already present in the system
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLagStaticModeSet (openapiClientHandle_t *client_handle,
                                      uint32_t lagIfNum,
                                      OPEN_CONTROL_t staticMode);

/*********************************************************************
* @purpose  Get the maximum LAG name length.
*
* @param    client_handle          @b{(input)}  Client handle from registration API
* @param    maxLen                 @b{(output)} Maximum length for a LAG name.
*
* @returns  OPEN_E_NONE     if LAD name maximum length is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLagNameMaxLengthGet(openapiClientHandle_t *client_handle,
                                        uint32_t *maxLen);

/***************************************************************************
* @purpose  Gets name of the LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    name           @b{(output)} Name of the LAG interface. dmx
*
* @returns  OPEN_E_NONE        LAG name returned.
* @returns  OPEN_E_FAIL        Failed to get the name of the LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @notes  Maximum size for LAG name is determined by calling
*         openapiLagNameMaxLengthGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
***************************************************************************/
open_error_t openapiLagNameGet (openapiClientHandle_t *client_handle,
                                uint32_t lagIfNum, 
                                open_buffdesc *name);

/***************************************************************************
* @purpose  Sets name of the LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    name           @b{(input)}  Name of the LAG interface.
*
* @returns  OPEN_E_NONE        LAG name successfully set.
* @returns  OPEN_E_FAIL        Failed to set the name of the LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
* @notes  Maximum size for LAG name is determined by calling
*         openapiLagNameMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
***************************************************************************/
open_error_t openapiLagNameSet (openapiClientHandle_t *client_handle,
                                uint32_t lagIfNum,
                                open_buffdesc *name);

/********************************************************************************
* @purpose  Gets load balancing mode of the LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    lbMode         @b{(output)} load balance mode of the LAG interface.
*
* @returns  OPEN_E_NONE        load balancing mode returned.
* @returns  OPEN_E_FAIL        Failed to get the load balance mode of the LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagLoadBalanceModeGet (openapiClientHandle_t *client_handle,
                                           uint32_t lagIfNum,
                                           OPEN_LAG_HASH_MODE_t *lbMode);

/********************************************************************************
* @purpose  Sets load balancing mode of the LAG interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    lbMode         @b{(input)} load balance mode of the LAG interface.
*
* @returns  OPEN_E_NONE        load balance mode set.
* @returns  OPEN_E_FAIL        Failed to set the load balance mode of the LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagLoadBalanceModeSet (openapiClientHandle_t *client_handle,
                                           uint32_t lagIfNum,
                                           OPEN_LAG_HASH_MODE_t lbMode);

/*********************************************************************
* @purpose Get the LAG Threshold minimum and maximum values.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    minThreshold    @b{(output)} Minimum threshold
* @param    maxThreshold    @b{(output)} Maximum threshold
*
* @returns  OPEN_E_NONE     if min and max values are returned.
* @returns  OPEN_E_FAIL     if failed to get the range values.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiLagThresholdMinMaxGet(openapiClientHandle_t *client_handle,
                                          uint32_t *minThreshold,
                                          uint32_t *maxThreshold);

/********************************************************************************
* @purpose  Gets minimum uplinks for the LAG interface.  
*           These are the minimum number of physical interfaces that needs to
*           to be UP for the LAG interface to UP
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    minUpLinks     @b{(output)} minimum uplinks of the LAG interface.
*
* @returns  OPEN_E_NONE        minimum uplinks returned.
* @returns  OPEN_E_FAIL        Failed to get the minimum uplinks for a LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagMinUplinksGet (openapiClientHandle_t *client_handle,
                                      uint32_t lagIfNum,
                                      uint32_t *minUpLinks);

/********************************************************************************
* @purpose  Sets minimum uplinks for the LAG interface.
*           These are the minimum number of physical interfaces that needs to
*           to be UP for the LAG interface to UP
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    minUpLinks     @b{(input)}  minimum uplinks of the LAG interface.
*           -              The uplinks ranges are available via openapiLagThresholdMinMaxGet()
* 
* @returns  OPEN_E_NONE        minimum uplinks successfully set.
* @returns  OPEN_E_FAIL        Failed to set the minimum uplinks for a LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagMinUplinksSet (openapiClientHandle_t *client_handle,
                                      uint32_t lagIfNum,
                                      uint32_t minUpLinks);

/*********************************************************************
* @purpose  Get the maximum number of members allowed for a LAG.
* 
* @param    client_handle     @b{(input)}  Client handle from 
*                                          registration API
* @param    memberMax         @b{(output)} Maximum members
*
* @returns  OPEN_E_NONE      if member maximum is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    none
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLagMaxMemberCountGet(openapiClientHandle_t *client_handle,
                                         uint32_t *memberMax);

/********************************************************************************
* @purpose  Gets the list of member ports for the LAG interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    count          @b{(inout)}  size of LAG member list
*                          Input:  maximum number of LAG members to output
*                          Output: actual number of LAG member ports in list
* @param    bufd           @b{(output)} buffer for outputting list of LAG members.
*                          returns array of Physical interfaces that are part of LAG
* 
* @notes   Maximum number of LAG members is determined by calling
*          openapiLagMaxMemberCountGet().
*
* @returns  OPEN_E_NONE        members list returned successfully.
* @returns  OPEN_E_FAIL        Failed to set the minimum uplinks for a LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagMembersGet (openapiClientHandle_t *client_handle,
                                   uint32_t lagIfNum,
                                   uint32_t *count,
                                   open_buffdesc *bufd);

/********************************************************************************
* @purpose  Gets a comma-delimited list of member ports for the LAG interface.
* 
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    count          @b{(inout)}  size of LAG member list
*                          Input:  maximum number of LAG members to output
*                          Output: actual number of LAG member ports in list
* @param bufd              @b{(output)} buffer for outputting comma separated list of LAG members.
*                          Comma separated list of LAG member internal interface numbers
* 
* @notes   Maximum number of LAG members is determined by calling
*          openapiLagMaxMemberCountGet().
*
* @returns  OPEN_E_NONE        members list returned successfully.
* @returns  OPEN_E_FAIL        Failed to get members list for the given LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagMembersListGet (openapiClientHandle_t *client_handle,
                                       uint32_t lagIfNum,
                                       uint32_t *count,
                                       open_buffdesc *bufd);

/********************************************************************************
* @purpose  Gets the list of active member ports for the LAG interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    count          @b{(inout)} size of LAG member list
*                          Input:  maximum number of LAG members to output
*                          Output: actual number of LAG member ports in list
* @param    bufd           @b{(output)} buffer for outputting list of LAG members.
*                          returns array of Physical interfaces that are part of LAG
* 
* @notes   Maximum number of LAG members is determined by calling
*          openapiLagMaxMemberCountGet().
* 
* @returns  OPEN_E_NONE        members list returned successfully.
* @returns  OPEN_E_FAIL        Failed to set the minimum uplinks for a LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiLagActiveMembersGet (openapiClientHandle_t *client_handle,
                                         uint32_t lagIfNum,
                                         uint32_t *count,
                                         open_buffdesc *bufd);

/********************************************************************************
* @purpose  Gets a comma-delimited list of active member ports for the LAG interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    lagIfNum       @b{(input)}  LAG interface number.
* @param    count          @b{(inout)}  size of LAG member list
*                          Input:  maximum number of LAG members to output
*                          Output: actual number of LAG member ports in list
* @param bufd              @b{(output)} buffer for outputting comma separated list of LAG members.
*                          Comma separated list of LAG member internal interface numbers
*
* @notes   Maximum number of LAG members is determined by calling
*          openapiLagMaxMemberCountGet().
*
* @returns  OPEN_E_NONE        active members list returned successfully.
* @returns  OPEN_E_FAIL        Failed to get members list for the given LAG.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiLagActiveMembersListGet (openapiClientHandle_t *client_handle,
                                             uint32_t lagIfNum,
                                             uint32_t *count,
                                             open_buffdesc *bufd);

/********************************************************************************
* @purpose  Gets the LACP mode of the physical interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  physical interface number.
* @param    lacpMode       @b{(output)} LACP mode of the physical interface.
*
* @returns  OPEN_E_NONE        LACP mode of the physical interface returned.
* @returns  OPEN_E_FAIL        Failed to get LACP mode of the physical interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiDot3adAggPortLacpModeGet (openapiClientHandle_t *client_handle,
                                              uint32_t ifNum,
                                              OPEN_CONTROL_t *lacpMode);

/********************************************************************************
* @purpose  Sets the LACP mode of the physical interface.  
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  physical interface number.
* @param    lacpMode       @b{(intput)} LACP mode of the physical interface.
*
* @returns  OPEN_E_NONE        LACP mode of the physical interface set successfully.
* @returns  OPEN_E_FAIL        Failed to set LACP mode of the physical interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiDot3adAggPortLacpModeSet (openapiClientHandle_t *client_handle,
                                              uint32_t ifNum,
                                              OPEN_CONTROL_t lacpMode);

/********************************************************************************
* @purpose  Gets the LACP actor admin state.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  physical interface number.
* @param    type           @b{(input)}  type of the LACP actor admin state.
* @param    state          @b{(output)} LACP actor admin state.
*
* @returns  OPEN_E_NONE        LACP actor admin state get successfully.
* @returns  OPEN_E_FAIL        Failed to get LACP actor admin state.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiDot3adAggPortActorAdminStateGet (openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_DOT3AD_STATE_TYPE_t type,
                                                     OPEN_CONTROL_t *state);

/********************************************************************************
* @purpose  Sets the LACP actor admin state.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  physical interface number.
* @param    type           @b{(input)}  type of the LACP actor admin state.
* @param    mode           @b{(input)}  LACP actor admin state.
*
* @returns  OPEN_E_NONE        LACP actor admin state set successfully.
* @returns  OPEN_E_FAIL        Failed to set LACP actor admin state.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************************/
open_error_t openapiDot3adAggPortActorAdminStateSet (openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_DOT3AD_STATE_TYPE_t type,
                                                     OPEN_CONTROL_t mode);

#endif
