/********************************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.?0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
********************************************************************************
*
* @filename  openapi_snooping.h
*
* @purpose   Provide interface for L2 Multicast Snooping API's
*
* @component OpEN
*
* @create    06/18/2015
*
* @end
*
********************************************************************************/
#ifndef OPENAPI_SNOOPING_H_INCLUDED
#define OPENAPI_SNOOPING_H_INCLUDED

#include "openapi_common.h"

#define OPEN_INTF_LEN 4

typedef enum 
{
  OPEN_SNOOP_SSM_FILTER_MODE_NONE    = 0,  /** Non-SSM Filter Mode State */
  OPEN_SNOOP_SSM_FILTER_MODE_INCLUDE = 1,  /** SSM Filter Mode Include */
  OPEN_SNOOP_SSM_FILTER_MODE_EXCLUDE = 2   /** SSM Filter Mode Exclude */
} OPEN_SNOOP_SSM_FILTER_MODE_t;

typedef enum
{
  OPEN_SNOOP_QUERIER_DISABLED    = 0,      /** Querier Disabled */
  OPEN_SNOOP_QUERIER_QUERIER     = 1,      /** Querier State */
  OPEN_SNOOP_QUERIER_NON_QUERIER = 2       /** Non-Querier State */

} OPEN_SNOOP_QUERIER_STATE_t;

/*********************************************************************************
* @purpose  Checks if the passed family type snooping is supported or not.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    family              @b{(input)}   address family.
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* @param    valid               @b{(output)}  validation status
* 
* @returns  OPEN_E_NONE         Validity of family checked.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopProtocolGet(openapiClientHandle_t * client_handle,
                                     uint8_t family,
                                     OPEN_BOOL_t * status);

/*********************************************************************************
* @purpose  Returns the IGMP/MLD Snooping Admin mode.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    adminMode           @b{(output)}  admin mode.
* @param    family              @b{(input)}   address family.
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD Snooping Admin mode returned.
* @returns  OPEN_E_FAIL         Failed to get IGMP/MLD Snooping Admin mode.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopAdminModeGet(openapiClientHandle_t * client_handle,
                                      OPEN_CONTROL_t * adminMode,
                                      uint8_t family);

/*********************************************************************************
* @purpose  Sets the IGMP/MLD Snooping Admin mode.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    adminMode           @b{(input)}   admin mode.
* @param    family              @b{(input)}   address family.
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD static mcast router attached status set successful.
* @returns  OPEN_E_FAIL         Failed to set IGMP/MLD Snooping Admin mode.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopAdminModeSet(openapiClientHandle_t * client_handle,
                                      OPEN_CONTROL_t adminMode,
                                      uint8_t family);

/*********************************************************************************
* @purpose  Returns the number of multicast control frames processed
*           by the CPU.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    processedFrames     @b{(output)}  number of control frames processed.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Number of multicast control (i.e. IGMP/MLD) frames
*                               processed by the CPU returned.
* @returns  OPEN_E_FAIL         Failed to get number of multicast control
*                               (i.e. IGMP/MLD) frames processed by the CPU.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopControlFramesProcessedGet(openapiClientHandle_t * client_handle,
                                                   uint32_t * processedFrames,
                                                   uint8_t family);

/*********************************************************************************
* @purpose  Returns the number of multicast data frames forwarded by the CPU.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    forwardedFrames     @b{(output)}  number of data frames forwarded.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Number of multicast data frames forwarded
*                               by the CPU returned.
* @returns  OPEN_E_FAIL         Failed to get Inumber of multicast data frames
*                               forwarded by the CPU.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopDataFramesForwardedGet(openapiClientHandle_t * client_handle,
                                                uint32_t * forwardedFrames,
                                                uint8_t family);

/*********************************************************************************
* @purpose  Returns the IGMP/MLD Proxy Querier Admin mode for the specified snooping
*           instance.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    adminMode           @b{(output)}  admin mode.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD Proxy Querier Admin mode for the specified
*                               snooping instance returned.
* @returns  OPEN_E_FAIL         Failed to get IGMP/MLD Proxy Querier Admin mode for
*                               the specified snooping instance.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopProxyQuerierAdminModeGet(openapiClientHandle_t * client_handle,
                                                  OPEN_CONTROL_t * adminMode,
                                                  uint8_t family);

/*********************************************************************************
* @purpose  Sets the IGMP/MLD Proxy Querier Admin mode for the specified snooping
*           instance.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    adminMode           @b{(input)}   admin mode.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD Proxy Querier Admin mode for the specified
*                               snooping instance  set successful.
* @returns  OPEN_E_FAIL         Failed to set IGMP/MLD Proxy Querier Admin mode for
*                               the specified snooping instance.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopProxyQuerierAdminModeSet(openapiClientHandle_t * client_handle,
                                                  OPEN_CONTROL_t adminMode,
                                                  uint8_t family);

/*********************************************************************************
* @purpose  Returns the configured mode for flooding unknown multicast traffic
*           to all ports.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    floodAll            @b{(output)}  OPEN_TRUE if flooding to all ports.
* 
* @returns  OPEN_E_NONE         Configured mode for flooding unknown multicast traffic
*                               to all ports returned.
* @returns  OPEN_E_FAIL         Failed to get configured mode for flooding unknown
*                               multicast traffic to all ports.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopFloodUnknownAllPortsGet(openapiClientHandle_t * client_handle,
                                                 OPEN_BOOL_t * floodAll);

/*********************************************************************************
* @purpose  Sets the configured mode for flooding unknown multicast traffic
*           to all ports.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    floodAll            @b{(input)}   OPEN_TRUE if flooding to all ports.
* 
* @returns  OPEN_E_NONE         Configured mode for flooding unknown multicast traffic
*                               to all ports set successful.
* @returns  OPEN_E_FAIL         Failed to set configured mode for flooding unknown
*                               multicast traffic to all ports.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopFloodUnknownAllPortsSet(openapiClientHandle_t * client_handle,
                                                 OPEN_BOOL_t floodAll);

/*********************************************************************************
* @purpose  Flushes all Snooping entries and multicast router entries.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         All entries flushed successful.
* @returns  OPEN_E_FAIL         Failed to flush all entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopEntriesFlush(openapiClientHandle_t * client_handle,
                                      uint8_t family);

/*********************************************************************************
* @purpose  Returns the interface mask of interfaces enabled for IGMP/MLD Snooping.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    mask                @b{(output)}  comma separated string of the interfaces.
*           -                                 Buffer allocated size must be at least
*           -                                 @ref L7_IGMP_SNOOPING_MAX_INTF
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Interface mask of interfaces enabled for IGMP/MLD
*                               Snooping returned.
* @returns  OPEN_E_FAIL         Failed to get interface mask of interfaces enabled
*                               for IGMP/MLD Snooping.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfEnabledMaskGet(openapiClientHandle_t * client_handle,
                                            open_buffdesc * mask,
                                            uint8_t family);

/********************************************************************************
* @purpose  Sets IGMP/MLD Snooping interface mode via a comma separated string.
*
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    mask                @b{(input)}   comma separated string of the interfaces.
*           -                                 Buffer allocated size must be at least
*           -                                 @ref L7_IGMP_SNOOPING_MAX_INTF
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE        IGMP/MLD Snooping interface mode via a bit mask set successful.
* @returns  OPEN_E_FAIL        Failed to set IGMP/MLD Snooping interface mode via a bit mask.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfEnabledMaskSet(openapiClientHandle_t * client_handle,
                                            open_buffdesc * mask,
                                            uint8_t family);

/*********************************************************************************
* @purpose  Returns the IGMP/MLD Snooping mode for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    mode                @b{(output)}  snooping mode
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD Snooping mode for the specified interface returned.
* @returns  OPEN_E_FAIL         Failed to get IGMP/MLD Snooping mode for the specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfModeGet(openapiClientHandle_t * client_handle,
                                     uint32_t intIfNum,
                                     OPEN_CONTROL_t * mode,
                                     uint8_t family);

/*********************************************************************************
* @purpose  Sets the IGMP/MLD Snooping mode for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    mode                @b{(input)}   snooping mode
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD Snooping mode for the specified interface set successful.
* @returns  OPEN_E_FAIL         Failed to set IGMP/MLD Snooping mode for the specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfModeSet(openapiClientHandle_t * client_handle,
                                     uint32_t intIfNum,
                                     OPEN_CONTROL_t mode,
                                     uint8_t family);

/*********************************************************************************
* @purpose  Returns the IGMP/MLD group membership interval for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    queryTime           @b{(output)}  query time.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD group membership interval for the specified
*                               interface returned.
* @returns  OPEN_E_FAIL         Failed to get IGMP/MLD group membership interval for
*                               the specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfGroupMembershipIntervalGet(openapiClientHandle_t * client_handle,
                                                        uint32_t intIfNum,
                                                        uint32_t * queryTime,
                                                        uint8_t family);

/*********************************************************************************
* @purpose  Sets the IGMP/MLD group membership interval for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    queryTime           @b{(input)}   query time.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD group membership interval for the specified
*                               interface set successful.
* @returns  OPEN_E_FAIL         Failed to set IGMP/MLD group membership interval for
*                               the specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfGroupMembershipIntervalSet(openapiClientHandle_t * client_handle,
                                                        uint32_t intIfNum,
                                                        uint32_t queryTime,
                                                        uint8_t family);

/*********************************************************************************
* @purpose  Returns the IGMP/MLD query response time for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    responseTime        @b{(output)}  response time.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD query response time for the specified interface returned.
* @returns  OPEN_E_FAIL         Failed to get IGMP/MLD query response time for the specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfResponseTimeGet(openapiClientHandle_t * client_handle,
                                             uint32_t intIfNum,
                                             uint32_t * responseTime,
                                             uint8_t family);

/*********************************************************************************
* @purpose  Sets the IGMP/MLD query response time for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    responseTime        @b{(input)}   response time.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         IGMP/MLD query response time for the specified interface
*                               set successful.
* @returns  OPEN_E_FAIL         Failed to set IGMP/MLD query response time for the
*                               specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfResponseTimeSet(openapiClientHandle_t * client_handle,
                                             uint32_t intIfNum,
                                             uint32_t responseTime,
                                             uint8_t family);

/*********************************************************************************
* @purpose  Returns the Fast-Leave Admin mode for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    mode                @b{(output)}  Fast-Leave interface admin mode.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Fast-Leave Admin mode for the specified interface returned.
* @returns  OPEN_E_FAIL         Failed to get Fast-Leave Admin mode for the specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfFastLeaveAdminModeGet(openapiClientHandle_t * client_handle,
                                                   uint32_t intIfNum,
                                                   OPEN_CONTROL_t * mode,
                                                   uint8_t family);

/*********************************************************************************
* @purpose  Sets the Fast-Leave Admin mode for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    mode                @b{(input)}   Fast-Leave interface admin mode.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Fast-Leave Admin mode for the specified interface
*                               set succesful.
* @returns  OPEN_E_FAIL         Failed to set Fast-Leave Admin mode for the
*                               specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfFastLeaveAdminModeSet(openapiClientHandle_t * client_handle,
                                                   uint32_t intIfNum,
                                                   OPEN_CONTROL_t mode,
                                                   uint8_t family);

/*********************************************************************************
* @purpose  Returns the Proxy Querier Admin mode for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    mode                @b{(output)}  Proxy Querier Admin mode.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Proxy Querier Admin mode for the specified interface returned.
* @returns  OPEN_E_FAIL         Failed to get Proxy Querier Admin mode for the
*                               specified interface.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfProxyQuerierModeGet(openapiClientHandle_t * client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_CONTROL_t * mode,
                                                 uint8_t family);

/*********************************************************************************
* @purpose  Sets the Proxy Querier Admin mode for the specified interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    mode                @b{(input)}   Proxy Querier Admin mode.
* @param    family              @b{(input)}   address family
*                                             OPEN_AF_INET  => IGMP Snooping
*                                             OPEN_AF_INET6 => MLD Snooping
* 
* @returns  OPEN_E_NONE         Proxy Querier Admin mode for the specified interface
*                               set successful.
* @returns  OPEN_E_FAIL         Failed to set Proxy Querier Admin mode for the
*                               specified interface.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIntfProxyQuerierModeSet(openapiClientHandle_t * client_handle,
                                                 uint32_t intIfNum,
                                                 OPEN_CONTROL_t mode,
                                                 uint8_t family);

/*********************************************************************************
* @purpose  Checks if interface is a valid IGMP/MLD snooping Interface.
* 
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    intIfNum            @b{(input)}   internal interface number.
* @param    valid               @b{(output)}  validation status.
* 
* @returns  OPEN_E_NONE         Validity of interface checked.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************************/
open_error_t openapiSnoopIsValidIntf(openapiClientHandle_t * client_handle,
                                     uint32_t intIfNum,
                                     OPEN_BOOL_t * status);

/*****************************************************************************
* @purpose  Gets the source filter mode (Include or Exclude) for the specified
*           group on the specified interface and VLAN.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(input)}  pointer to the group address.
* @param    pIntIfNum        @b{(input)}  pointer to the internal interface number.
* @param    pVlanId          @b{(input)}  VLAN ID for the group entry.
* @param    pfilterMode      @b{(output)} pointer to the filter mode. 

* @returns  OPEN_E_NONE      SSM group filter mode returned.
* @returns  OPEN_E_FAIL      SSM group filter mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t 
openapiSnoopSSMGroupFilterModeGet(openapiClientHandle_t *client_handle,
                                  uint8_t family, open_inet_addr_t *pGroup,
                                  uint32_t *pIntIfNum, uint32_t *pVlanId,
                                  OPEN_SNOOP_SSM_FILTER_MODE_t *pfilterMode);

/*****************************************************************************
* @purpose  Gets the IP address of the source of last membership report
*           received for the specified group address on the specified interface and VLAN.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(input)}  pointer to the group address.
* @param    pIntIfNum        @b{(input)}  pointer to the internal interface number.
* @param    pVlanId          @b{(input)}  VLAN ID for the group entry.
* @param    pSrcIp           @b{(output)} pointer to the last reporter IP address.

* @returns  OPEN_E_NONE      SSM group last reporter returned.
* @returns  OPEN_E_FAIL      SSM group last reporter retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t 
openapiSnoopSSMGroupLastReporterGet(openapiClientHandle_t *client_handle,
                                    uint8_t family, open_inet_addr_t *pGroup,
                                    uint32_t *pIntIfNum, uint32_t *pVlanId,
                                    open_inet_addr_t *pSrcIp);

/*****************************************************************************
* @purpose   Gets the first learnt entry from SSM group database.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(input)}  pointer to the group address.
* @param    pIntIfNum        @b{(input)}  pointer to the internal interface number.
* @param    pVlanId          @b{(input)}  VLAN ID for the group entry.
* 
* @returns  OPEN_E_NONE      SSM group record first returned.
* @returns  OPEN_E_FAIL      SSM group record first retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    pGroup is returned as NULL if there are no entries.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t 
openapiSnoopSSMGroupRecordFirstGet(openapiClientHandle_t *client_handle,
                                   uint8_t family, open_inet_addr_t *pGroup,
                                   uint32_t *pIntIfNum, uint32_t *pVlanId);

/*****************************************************************************
* @purpose   Gets the next learnt entry from SSM group database.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(inout)}  pointer to the group address.
* @param    pIntIfNum        @b{(inout)}  pointer to the internal interface number.
* @param    pVlanId          @b{(inout)}  VLAN ID for the group entry.
* 
* @returns  OPEN_E_NONE      SSM group record next returned.
* @returns  OPEN_E_FAIL      SSM group record next retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    pGroup is returned as NULL if there are no entries.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t 
openapiSnoopSSMGroupRecordNextGet(openapiClientHandle_t *client_handle,
                                   uint8_t family, open_inet_addr_t *pGroup,
                                   uint32_t *pIntIfNum, uint32_t *pVlanId);

/*****************************************************************************
* @purpose  To get the current number of entries in the MFDB table.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    currEntries      @b{(output)} number of current entries.
* 
* @returns  OPEN_E_NONE      current number of entries returned.
* @returns  OPEN_E_FAIL      current number of entries retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiSnoopSSMMfdbCurrEntriesGet(openapiClientHandle_t *client_handle,
                                               uint8_t family, uint32_t *currEntries);

/******************************************************************************
* @purpose  Get the exclude port list for a given SSM MFDB entry.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(input)}  pointer to the group address.
* @param    pIntIfNum        @b{(input)}  pointer to the internal interface number.
* @param    pVlanId          @b{(input)}  VLAN ID for the group entry.
* @param    excPortMask      @b{(output)} pointer to interface mask.
*           -                             Buffer allocated size must be at least
*           -                             @ref L7_NIM_IFNAME_SIZE
*
* @returns  OPEN_E_NONE      port list returned.
* @returns  OPEN_E_FAIL      port list retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t 
openapiSnoopSSMMfdbEntryExcPortMaskGet(openapiClientHandle_t *client_handle,
                                       uint8_t family, open_inet_addr_t *pGroup,
                                       open_inet_addr_t *pSource, uint32_t *pVlanId,
                                       open_buffdesc *excPortMaskName);

/*****************************************************************************
* @purpose   Gets the first learnt entry from SSM MFDB.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(output)}  pointer to the group address.
* @param    pSource          @b{(output)}  pointer to the internal interface number.
* @param    pVlanId          @b{(output)}  VLAN ID for the group entry.
* 
* @returns  OPEN_E_NONE      first learnt entry returned.
* @returns  OPEN_E_FAIL      first learnt entry retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    pGroup is returned as NULL if there are no entries.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiSnoopSSMMfdbEntryFirstGet(openapiClientHandle_t *client_handle,
                                              uint8_t family, open_inet_addr_t *pGroup,
                                              open_inet_addr_t *pSource, uint32_t *pVlanId);

/******************************************************************************
* @purpose  Get the include port list for a given SSM MFDB entry.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(input)}  pointer to the group address.
* @param    pIntIfNum        @b{(input)}  pointer to the internal interface number.
* @param    pVlanId          @b{(input)}  VLAN ID for the group entry.
* @param    incPortMask      @b{(output)} pointer to interface mask.
*           -                             Buffer allocated size must be at least
*           -                             @ref L7_NIM_IFNAME_SIZE
*
* @returns  OPEN_E_NONE      port list returned.
* @returns  OPEN_E_FAIL      port list retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t 
openapiSnoopSSMMfdbEntryIncPortMaskGet(openapiClientHandle_t *client_handle,
                                       uint8_t family, open_inet_addr_t *pGroup,
                                       open_inet_addr_t *pSource, uint32_t *pVlanId,
                                       open_buffdesc *incPortMaskName);

/******************************************************************************
* @purpose  Gets next learnt entry from SSM MFDB.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(inout)}  pointer to the group address.
* @param    pIntIfNum        @b{(inout)}  pointer to the internal interface number.
* @param    pVlanId          @b{(inout)}  VLAN ID for the group entry.
*
* @returns  OPEN_E_NONE      next learnt entry returned.
* @returns  OPEN_E_FAIL      next learnt entry retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    pGroup is returned as NULL if there are no entries.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopSSMMfdbEntryNextGet(openapiClientHandle_t *client_handle,
                                             uint8_t family,
                                             open_inet_addr_t *pGroup,
                                             open_inet_addr_t *pSource, 
                                             uint32_t *pVlanId);

/*****************************************************************************
* @purpose  To get the maximum number of entries that the SSM MFDB table
*           can hold for a given family.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    maxEntries       @b{(output)} total number of entries.
* 
* @returns  OPEN_E_NONE      maximum number of entries returned.
* @returns  OPEN_E_FAIL      maximum number of entries retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end                                          
******************************************************************************/
open_error_t openapiSnoopSSMMfdbMaxTableEntriesGet(openapiClientHandle_t *client_handle,
                                                   uint8_t family, 
                                                   uint32_t *maxEntries);

/*****************************************************************************
* @purpose  To get the most number of entries ever in the SSM MFDB table
*           for a given family.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    mostEntries      @b{(output)} most number of entries ever.
* 
* @returns  OPEN_E_NONE      number of entries returned.
* @returns  OPEN_E_FAIL      number of entries retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end                                          
******************************************************************************/
open_error_t openapiSnoopSSMMfdbMostEntriesGet(openapiClientHandle_t *client_handle,
                                               uint8_t family, 
                                               uint32_t *mostEntries);

/******************************************************************************
* @purpose  To validate if the specified source address exists for the
*           specified group address and interface or not.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(input)}  pointer to the group address.
* @param    pIntIfNum        @b{(input)}  pointer to the internal interface number.
* @param    pVlanId          @b{(input)}  VLAN ID for the group entry.
* @param    pSrcIp           @b{(input)}  pointer to the Source IP address.
*
* @returns  OPEN_E_NONE      entry exists.
* @returns  OPEN_E_FAIL      if an entry is not exist.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    pGroup is returned as NULL if there are no entries.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopSSMSrcListEntryGet(openapiClientHandle_t *client_handle,
                                            uint8_t family,
                                            open_inet_addr_t *pGroup,
                                            uint32_t *pIntIfNum,
                                            uint32_t *pVlanId,
                                            open_inet_addr_t *pSrcIp);

/******************************************************************************
* @purpose  Get the next source list entry for the specified group address
*           interface and source address.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    pGroup           @b{(inout)}  pointer to the group address.
* @param    pIntIfNum        @b{(inout)}  pointer to the internal interface number.
* @param    pVlanId          @b{(inout)}  VLAN ID for the group entry.
* @param    pSrcIp           @b{(inout)}  pointer to the Source IP address.
*
* @returns  OPEN_E_NONE      source list entry exists.
* @returns  OPEN_E_FAIL      if an source list entry is not exist.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @comments if pIntIfNum = 0, pGroup = 0 & pSrcIp = 0, then return the first
*           valid source list entry.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopSSMSrcListEntryNextGet(openapiClientHandle_t *client_handle,
                                                uint8_t family, 
                                                open_inet_addr_t *pGroup,
                                                uint32_t *pIntIfNum, 
                                                uint32_t *pVlanId,
                                                open_inet_addr_t *pSrcIp);

/******************************************************************************
* @purpose  Check if VLAN is eligible for snooping configuration.
*
* @param    client_handle       @b((input))  client handle from registration API.
* @param    vlanId              @b{(input)}  VLAN ID.
* 
* @returns  OPEN_E_NONE      valid VLAN ID.
* @returns  OPEN_E_FAIL      invalid VLAN for configuration.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanCheckValid(openapiClientHandle_t *client_handle, uint32_t vlanId);

/******************************************************************************
* @purpose  Gets next VLAN ID for snooping configuration.
*
* @param    client_handle       @b((input))  client handle from registration API.
* @param    vlanId              @b{(input)}  VLAN ID.
* @param    vlanId              @b{(output)} next VLAN ID.
* 
* @returns  OPEN_E_NONE      valid VLAN ID.
* @returns  OPEN_E_FAIL      invalid VLAN for configuration.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanNextGet(openapiClientHandle_t *client_handle, uint32_t vlanId, 
                                     uint32_t *nextVlan);

/******************************************************************************
* @purpose  Gets the configured IGMP/MLD snooping fast leave mode for the specified VLAN.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    fastLeaveMode    @b{(output)} configured IGMP/MLD snooping fast leave.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      if mode was retrieved successfully.
* @returns  OPEN_E_FAIL      if VLAN ID out of range.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanFastLeaveModeGet(openapiClientHandle_t *client_handle, 
                                              uint32_t vlanId,
                                              OPEN_BOOL_t *fastLeaveMode,
                                              uint8_t family); 

/******************************************************************************
* @purpose  Sets the configured IGMP/MLD snooping fast leave mode for the specified VLAN.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    fastLeaveMode    @b{(input)}  configured IGMP/MLD snooping fast leave.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      if mode was sets successfully.
* @returns  OPEN_E_FAIL      if VLAN ID out of range or fast leave mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanFastLeaveModeSet(openapiClientHandle_t *client_handle, 
                                              uint32_t vlanId,
                                              OPEN_BOOL_t fastLeaveMode,
                                              uint8_t family);

/******************************************************************************
* @purpose  Gets the configured IGMP/MLD snooping group membership interval
*           for the specified VLAN.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    groupMembershipInterval    @b{(output)} configured IGMP/MLD snooping group membership interval.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      group membership interval was retrieved successfully.
* @returns  OPEN_E_FAIL      group membership interval retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanGroupMembershipIntervalGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t vlanId,
                                                        uint16_t *groupMembershipInterval,
                                                        uint8_t family);

/******************************************************************************
* @purpose  Sets the configured IGMP/MLD snooping group membership interval
*           for the specified VLAN.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    groupMembershipInterval    @b{(input)}  configured IGMP/MLD snooping group membership interval.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      group membership interval was sets successfully.
* @returns  OPEN_E_FAIL      group membership interval sets failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanGroupMembershipIntervalSet(openapiClientHandle_t *client_handle, 
                                                        uint32_t vlanId,
                                                        uint16_t groupMembershipInterval,
                                                        uint8_t family);

/******************************************************************************
* @purpose  Gets the configured IGMP/MLD snooping maximum response time
*           for the specified VLAN.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    responseTime               @b{(output)} configured IGMP/MLD snooping maximum response time.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      maximum response time was retrieved successfully.
* @returns  OPEN_E_FAIL      maximum response time retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanMaximumResponseTimeGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t vlanId,
                                                    uint16_t *responseTime,
                                                    uint8_t family);

/******************************************************************************
* @purpose  Set response time for the specified VLAN.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    responseTime               @b{(input)}  configured IGMP/MLD snooping maximum response time.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      response time was sets successfully.
* @returns  OPEN_E_FAIL      response time sets failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanMaximumResponseTimeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t vlanId,
                                                    uint16_t responseTime,
                                                    uint8_t family);

/******************************************************************************
* @purpose  Get the IGMP/MLD snooping mode corresponding to the VLAN ID.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    snoopVlanMode              @b{(output)} IGMP/MLD snooping mode for the VLAN ID.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping VLAN mode was retrieved successfully.
* @returns  OPEN_E_FAIL      snooping VLAN mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t vlanId,
                                     uint32_t *snoopVlanMode,
                                     uint8_t family);

/******************************************************************************
* @purpose  Set the IGMP/MLD snooping mode corresponding to the VLAN ID.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    snoopVlanMode              @b{(input)}  IGMP/MLD snooping mode for the VLAN ID.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping VLAN mode was sets successfully.
* @returns  OPEN_E_FAIL      snooping VLAN mode sets failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t vlanId,
                                     uint32_t snoopVlanMode,
                                     uint8_t family);

/******************************************************************************
* @purpose  Gets the configured snooping proxy querier mode for the
*           specified VLAN of a snoop instance.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    mode                       @b{(output)} configured snooping proxy querier mode.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping proxy querier mode was retrieved successfully.
* @returns  OPEN_E_FAIL      snooping proxy querier mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanProxyQuerierModeGet(openapiClientHandle_t *client_handle,
                                                 uint32_t vlanId,
                                                 uint32_t *mode,
                                                 uint8_t family);

/******************************************************************************
* @purpose  Sets the configured snooping proxy querier mode for the
*           specified VLAN of a snoop instance.
* 
* @param    client_handle              @b((input))  client handle from registration API.
* @param    vlanId                     @b{(input)}  VLAN ID.
* @param    mode                       @b{(input)}  configured snooping proxy querier mode.
* @param    family                     @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                                   OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping proxy querier mode was set successfully.
* @returns  OPEN_E_FAIL      snooping proxy querier mode set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanProxyQuerierModeSet(openapiClientHandle_t *client_handle,
                                                 uint32_t vlanId,
                                                 uint32_t mode,
                                                 uint8_t family);

/******************************************************************************
* @purpose  Get the configured IGMP/MLD snooping report suppression mode for the
*           specified VLAN.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    reportSuppMode   @b{(output)} reference to the IGMP/MLD snooping report suppression mode.
* 
* @returns  OPEN_E_NONE      if suppression mode was retrieved successfully.
* @returns  OPEN_E_FAIL      if VLAN ID out of range.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanReportSuppModeGet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId,
                                               uint8_t family,
                                               OPEN_BOOL_t *reportSuppMode); 

/******************************************************************************
* @purpose  Sets the configured IGMP/MLD snooping report suppression mode for the
*           specified VLAN.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* @param    reportSuppMode   @b{(input)}  reference to the IGMP/MLD snooping report suppression mode.
* 
* @returns  OPEN_E_NONE      suppression mode was set successfully.
* @returns  OPEN_E_FAIL      if VLAN ID out of range.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopVlanReportSuppModeSet(openapiClientHandle_t *client_handle, 
                                               uint32_t vlanId,
                                               uint8_t family,
                                               OPEN_BOOL_t reportSuppMode); 

/******************************************************************************
* @purpose  Gets the snooping querier configured address of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    snoopQuerierAddr @b{(output)} pointer to querier address.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier configured address was retrieved successfully.
* @returns  OPEN_E_FAIL      querier configured address retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierAddressGet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *snoopQuerierAddr,
                                           uint8_t family);

/******************************************************************************
* @purpose  Sets the snooping querier configured address of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    snoopQuerierAddr @b{(input)}  pointer to querier address.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier configured address was set successfully.
* @returns  OPEN_E_FAIL      querier configured address set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierAddressSet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *snoopQuerierAddr,
                                           uint8_t family);

/******************************************************************************
* @purpose  Gets the snooping querier admin mode of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    adminMode        @b{(output)} snooping querier admin mode.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier admin mode was retrieved successfully.
* @returns  OPEN_E_FAIL      querier admin mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierAdminModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t *adminMode,
                                             uint8_t family);

/******************************************************************************
* @purpose  Gets the Snooping Querier Admin mode of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    adminMode        @b{(input)}  snooping querier admin mode.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier admin mode address was set successfully.
* @returns  OPEN_E_FAIL      querier admin mode address set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierAdminModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t adminMode,
                                             uint8_t family);

/******************************************************************************
* @purpose  Gets the other querier expiry interval of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    expiryInterval   @b{(output)} interval for declaring last querier as expired.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier expiry interval was retrieved successfully.
* @returns  OPEN_E_FAIL      querier expiry interval retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierExpiryIntervalGet(openapiClientHandle_t *client_handle,
                                                  uint32_t *expiryInterval,
                                                  uint8_t family);

/******************************************************************************
* @purpose  Sets the other querier expiry interval of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    expiryInterval   @b{(input)}  interval for declaring last querier as expired.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier expiry interval address was set successfully.
* @returns  OPEN_E_FAIL      querier expiry interval address set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierExpiryIntervalSet(openapiClientHandle_t *client_handle,
                                                  uint32_t expiryInterval,
                                                  uint8_t family);

/******************************************************************************
* @purpose  Gets the detected last querier address for specified VLAN ID
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    querierAddr      @b{(output)} last querier VLAN address.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier address was retrieved successfully.
* @returns  OPEN_E_FAIL      querier address retrieval failed.
* @returns  OPEN_E_NOT_FOUND if querier address is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierLastQuerierAddressGet(openapiClientHandle_t *client_handle,
                                                      uint32_t vlanId,
                                                      open_inet_addr_t *querierAddr,
                                                      uint8_t family);

/******************************************************************************
* @purpose  Gets the detected last querier's version for specified VLAN ID
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    version          @b{(output)} last querier protocol version.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      last querier's version was retrieved successfully.
* @returns  OPEN_E_FAIL      last querier's version retrieval failed.
* @returns  OPEN_E_NOT_FOUND if last querier's version is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierLastQuerierVersionGet(openapiClientHandle_t *client_handle,
                                                      uint32_t vlanId,
                                                      uint32_t *version,
                                                      uint8_t family);

/******************************************************************************
* @purpose  Gets the operational value of max response time for specified
*           VLAN ID of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    maxRespTime      @b{(output)} operational max response time value.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      max response time was retrieved successfully.
* @returns  OPEN_E_FAIL      max response time retrieval failed.
* @returns  OPEN_E_NOT_FOUND if max response time is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes    When success, if maxRespTime == 0, it means that the max response
*           time is less than 1 sec
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierOperMaxRespTimeGet(openapiClientHandle_t *client_handle,
                                                   uint32_t vlanId,
                                                   uint32_t *maxRespTime,
                                                   uint8_t family);

/******************************************************************************
* @purpose  Gets the operational state of querier for specified VLAN ID
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    state            @b{(output)} operational state OPEN_SNOOP_QUERIER_STATE_t.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      state of querier was retrieved successfully.
* @returns  OPEN_E_FAIL      state of querier retrieval failed.
* @returns  OPEN_E_NOT_FOUND if state of querier is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierOperStateGet(openapiClientHandle_t *client_handle,
                                             uint32_t vlanId,
                                             uint32_t *state,
                                             uint8_t family);

/******************************************************************************
* @purpose  Gets the operational version of querier  for specified VLAN ID
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    version          @b{(output)} operational version.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      operational version was retrieved successfully.
* @returns  OPEN_E_FAIL      operational version retrieval failed.
* @returns  OPEN_E_NOT_FOUND if operational version is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierOperVersionGet(openapiClientHandle_t *client_handle,
                                               uint32_t vlanId,
                                               uint32_t *version,
                                               uint8_t family);

/******************************************************************************
* @purpose  Gets the snooping querier query interval of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    queryInterval    @b{(output)} interval for sending periodic queries.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping interval was retrieved successfully.
* @returns  OPEN_E_FAIL      snooping interval retrieval failed.
* @returns  OPEN_E_NOT_FOUND if snooping interval is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierQueryIntervalGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *queryInterval,
                                                 uint8_t family);

/******************************************************************************
* @purpose  Sets the qnooping querier query interval of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    queryInterval    @b{(input)}  interval for sending periodic queries.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping interval was set successfully.
* @returns  OPEN_E_FAIL      snooping interval set failed.
* @returns  OPEN_E_NOT_FOUND if snooping interval is not found.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierQueryIntervalSet(openapiClientHandle_t *client_handle,
                                                 uint32_t queryInterval,
                                                 uint8_t family);

/******************************************************************************
* @purpose  Gets the snooping querier configured version of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    version          @b{(output)} configured querier version.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier configured version was retrieved successfully.
* @returns  OPEN_E_FAIL      querier configured version retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVersionGet(openapiClientHandle_t *client_handle,
                                           uint32_t *version,
                                           uint8_t family);

/******************************************************************************
* @purpose  Sets the snooping querier configured version of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    version          @b{(input)}  configured querier version.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      configured querier version was set successfully.
* @returns  OPEN_E_FAIL      configured querier version set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVersionSet(openapiClientHandle_t *client_handle,
                                           uint32_t version,
                                           uint8_t family);

/******************************************************************************
* @purpose  Gets the qnooping querier configured address for specified VLAN ID
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    snoopQuerierAddr @b{(output)} querier VLAN address.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier VLAN address was retrieved successfully.
* @returns  OPEN_E_FAIL      querier VLAN address retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVlanAddressGet(openapiClientHandle_t *client_handle,
                                               uint32_t vlanId,
                                               open_inet_addr_t *snoopQuerierAddr,
                                               uint8_t family);

/******************************************************************************
* @purpose  Sets the snooping querier configured address for specified VLAN ID
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    snoopQuerierAddr @b{(input)}  querier VLAN address.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier VLAN address was set successfully.
* @returns  OPEN_E_FAIL      querier VLAN address set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVlanAddressSet(openapiClientHandle_t *client_handle,
                                               uint32_t vlanId,
                                               open_inet_addr_t *snoopQuerierAddr,
                                               uint8_t family);

/******************************************************************************
* @purpose  Gets the configured snooping querier election mode for the
*           specified vlan ID of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    vlanMode         @b{(output)} configured snooping querier VLAN election
*                                         participate mode.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier election mode was retrieved successfully.
* @returns  OPEN_E_FAIL      querier election mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVlanElectionModeGet(openapiClientHandle_t *client_handle,
                                                    uint32_t vlanId,
                                                    uint32_t *vlanMode,
                                                    uint8_t family);

/******************************************************************************
* @purpose  Sets the configured snooping querier election mode for the
*           specified VLAN ID of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    vlanMode         @b{(input)}  configured snooping querier VLAN election
*                                         participate mode.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      querier election mode was set successfully.
* @returns  OPEN_E_FAIL      querier election mode set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVlanElectionModeSet(openapiClientHandle_t *client_handle,
                                                    uint32_t vlanId,
                                                    uint32_t vlanMode,
                                                    uint8_t family);

/******************************************************************************
* @purpose  Gets the configured snooping querier mode for the specified vlanId
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    vlanMode         @b{(output)} configured snooping querier VLAN mode.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping querier VLAN mode was retrieved successfully.
* @returns  OPEN_E_FAIL      snooping querier VLAN mode retrieval failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVlanModeGet(openapiClientHandle_t *client_handle,
                                            uint32_t vlanId,
                                            OPEN_CONTROL_t *vlanMode,
                                            uint8_t family);

/******************************************************************************
* @purpose  Set the configured snooping querier mode for the specified vlanId
*           of a snoop instance.
*
* @param    client_handle    @b((input))  client handle from registration API.
* @param    vlanId           @b{(input)}  VLAN ID.
* @param    vlanMode         @b{(input)}  configured snooping querier VLAN mode.
* @param    family           @b{(input)}  OPEN_AF_INET  => IGMP Snooping.
*                                         OPEN_AF_INET6 => MLD Snooping.
* 
* @returns  OPEN_E_NONE      snooping querier VLAN mode was set successfully.
* @returns  OPEN_E_FAIL      snooping querier VLAN mode set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
******************************************************************************/
open_error_t openapiSnoopQuerierVlanModeSet(openapiClientHandle_t *client_handle,
                                            uint32_t vlanId,
                                            OPEN_CONTROL_t vlanMode,
                                            uint8_t family);

#endif /* OPENAPI_SNOOPING_H_INCLUDED */

