/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_sntp.h
*
* @purpose   SNTP Configuration and Status 
*
* @component OpEN
*
* @create    28/01/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_SNTP_H_INCLUDED
#define OPENAPI_SNTP_H_INCLUDED

#include "openapi_common.h"

/** OpEN uses this enumeration to define SNTP client modes */
typedef enum
{
  OPEN_SNTP_CLIENT_DISABLED     = 0, /**< SNTP client mode disabled */
  OPEN_SNTP_CLIENT_UNICAST      = 1, /**< SNTP client unicast mode */
  OPEN_SNTP_CLIENT_BROADCAST    = 2  /**< SNTP client broadcast mode */
} OPEN_SNTP_CLIENT_MODE_t;


/** OpEN uses this enumeration to define SNTP supported client modes */
typedef enum
{  
 OPEN_SNTP_SUPPORTED_DISABLED               = 0, /**< SNTP client mode disabled */
 OPEN_SNTP_SUPPORTED_UNICAST                = 1, /**< SNTP client unicast mode */
 OPEN_SNTP_SUPPORTED_BROADCAST              = 2, /**< SNTP client broadcast mode */
 OPEN_SNTP_SUPPORTED_UNICAST_AND_BROADCAST  = 3  /**< SNTP client unicast and broadcast modes */
}OPEN_SNTP_SUPPORTED_MODE_t;

/** OpEN uses this enumeration to define SNTP address types  */
typedef enum
{
  OPEN_SNTP_ADDRESS_UNKNOWN = 0, /**< Address type unknown */
  OPEN_SNTP_ADDRESS_IPV4    = 1, /**< Address type IPv4 */
  OPEN_SNTP_ADDRESS_IPV6    = 2, /**< Address type IPv6 */
  OPEN_SNTP_ADDRESS_DNS     = 3  /**< Address type DNS */
} OPEN_SNTP_ADDRESS_TYPE_t;

/** OpEN uses this enumeration to define SNTP server modes */
typedef enum
{
  OPEN_SNTP_SERVER_MODE_RESERVED            = 0, /**< Reserved Mode */
  OPEN_SNTP_SERVER_MODE_SYMMETRIC_ACTIVE    = 1, /**< Symmetric Active  Mode */
  OPEN_SNTP_SERVER_MODE_SYMMETRIC_PASSIVE   = 2, /**< Symmetric Passive  Mode */
  OPEN_SNTP_SERVER_MODE_CLIENT              = 3, /**< Client Mode */
  OPEN_SNTP_SERVER_MODE_SERVER              = 4, /**< Server Mode */
  OPEN_SNTP_SERVER_MODE_BROADCAST           = 5, /**< Broadcast Mode */
  OPEN_SNTP_SERVER_MODE_RESERVED_CONTROL    = 6, /**< Reserved Control Mode */
  OPEN_SNTP_SERVER_MODE_RESERVED_PRIVATE    = 7  /**< Reserved Private  Mode */
} OPEN_SNTP_SERVER_MODE_t;

/*********************************************************************
* @purpose  Add an SNTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff descriptor for
*                                         IP address of SNTP server - 
*                                         DNS, IPV4 and IPv6 addresses
*
* @returns  OPEN_E_NONE      if SNTP server is added successfully.
* @returns  OPEN_E_FAIL      if failed to add SNTP server.
* @returns  OPEN_E_EXISTS    if the server is already configured.
* @returns  OPEN_E_FULL      if no more space is available to add the server.
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Calling this API will change the running configuration of the switch. 
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet(). 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerAdd(openapiClientHandle_t *client_handle,
                                  open_buffdesc *ipAddr);

/*********************************************************************
* @purpose  Set priority to the SNTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff descriptor for
*                                         IP address of SNTP server - 
*                                         DNS, IPV4 and IPv6 addresses
* @param    priority         @b{(input)}  SNTP server priority
* @param    -                Minimum value @ref OPENAPI_SNTP_MIN_SERVER_PRIORITY  
* @param    -                Maximum value openapiSntpServerPriorityMaxGet()
* 
* @returns  OPEN_E_NONE      if SNTP server priority is set successfully.
* @returns  OPEN_E_FAIL      if failed to set SNTP server priority.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Calling this API will change the running configuration of the switch. 
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerPrioritySet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *ipAddr,
                                          uint16_t priority);

/*********************************************************************
* @purpose  Get the maximum SNTP priority.
* 
* @param    client_handle     @b{(input)}  Client handle from 
*                                          registration API
* @param    priorityMax       @b{(output)} Maximum priority
*
* @returns  OPEN_E_NONE      if SNTP priority is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    none
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerPriorityMaxGet(openapiClientHandle_t *client_handle,
                                             uint32_t *priorityMax);

/*********************************************************************
* @purpose  Set SNTP server port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff descriptor for
*                                         IP address of SNTP server - 
*                                         DNS, IPV4 and IPv6 addresses
* @param    port             @b{(input)}  SNTP server port
* @param    -                Minimum value @ref OPENAPI_SNTP_MIN_SERVER_PORT
* @param    -                Maximum value @ref OPENAPI_SNTP_MAX_SERVER_PORT
*
* @returns  OPEN_E_NONE      if SNTP server port is set successfully.
* @returns  OPEN_E_FAIL      if failed to set SNTP server port.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Calling this API will change the running configuration of the switch. 
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerPortSet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *ipAddr, uint16_t port);

/*********************************************************************
* @purpose  Set SNTP server version.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff descriptor for
*                                         IP address of SNTP server - 
*                                         DNS, IPV4 and IPv6 addresses
* @param    version          @b{(input)}  SNTP server version
* @param    -                Minimum value @ref OPENAPI_SNTP_VERSION_MIN
* @param    -                Maximum value @ref OPENAPI_SNTP_VERSION_MAX 
*
* @returns  OPEN_E_NONE      if SNTP server version is set successfully.
* @returns  OPEN_E_FAIL      if failed to set SNTP server version.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Calling this API will change the running configuration of the switch. 
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerVersionSet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *ipAddr, uint16_t version);

/*********************************************************************
* @purpose  Get the maximum number of SNTP servers supported.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    maxSntpServers   @b{(output}) Maximum number of SNTP servers 
*                                         supported
*
* @returns  OPEN_E_NONE      if the maximum number of SNTP servers is returned.
* @returns  OPEN_E_FAIL      if failed to get maximum number of SNTP servers.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServersMaxSupportedGet(openapiClientHandle_t *client_handle,
                                               uint32_t *maxSntpServers);

/*********************************************************************
* @purpose  Delete an SNTP server. 
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff Descriptor for 
*                                         IP address of SNTP server - 
*                                         DNS, IPV4 and IPv6 addresses
* 
* @returns  OPEN_E_NONE      if SNTP server is deleted successfully.
* @returns  OPEN_E_FAIL      if failed to delete SNTP server.
* @returns  OPEN_E_NOT_FOUND if SNTP server is not found.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerDelete(openapiClientHandle_t *client_handle,
                                     open_buffdesc *ipAddr);

/*********************************************************************
* @purpose  Get the maximum length of SNTP server address.
* 
* @param    client_handle     @b{(input)}  Client handle from 
*                                          registration API
* @param    sntpServerAddrLen @b{(output)} Maximum length of SNTP server address
*
* @returns  OPEN_E_NONE      if SNTP server address length is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    none
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerMaxAddrLenGet(openapiClientHandle_t *client_handle,
                                            uint32_t *sntpServerAddrLen);

/*********************************************************************
* @purpose  Get the first SNTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sntpServer       @b{(output)} SNTP server 
*
* @returns  OPEN_E_NONE      if first SNTP server is returned.
* @returns  OPEN_E_FAIL      if SNTP server does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerFirstGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *sntpServer);

/*********************************************************************
* @purpose  Get the next SNTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sntpServer       @b{(input)}  Previous SNTP server 
* @param    nextSntpServer   @b{(output)} Next SNTP server
*
* @returns  OPEN_E_NONE      if next SNTP server is returned.
* @returns  OPEN_E_NOT_FOUND if next SNTP server does not exist.
* @returns  OPEN_E_FAIL      if previous SNTP server is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerNextGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *sntpServer,
                                      open_buffdesc *nextSntpServer);

/*********************************************************************
* @purpose  Get the SNTP server address type.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    sntpServer           @b{(input)}  SNTP server 
* @param    sntpServerAddrType   @b{(output)} SNTP server address type
*
* @returns  OPEN_E_NONE      if SNTP server address type is returned.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_FAIL      if failed to get SNTP server address type.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerAddressTypeGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *sntpServer, 
                                             OPEN_SNTP_ADDRESS_TYPE_t *sntpServerAddrType);

/*********************************************************************
* @purpose  Get the SNTP server priority.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    sntpServer           @b{(input)}  SNTP server 
* @param    serverPriority       @b{(output)} SNTP server priority
*
* @returns  OPEN_E_NONE      if SNTP server priority is returned.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_FAIL      if failed to get SNTP server priority.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerPriorityGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *sntpServer,
                                          uint16_t *serverPriority);

/*********************************************************************
* @purpose  Get the SNTP server version.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    sntpServer           @b{(input)}  SNTP server 
* @param    serverVersion        @b{(output)} SNTP server version
*
* @returns  OPEN_E_NONE      if SNTP server version is returned.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_FAIL      if failed to get SNTP server version.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerVersionGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *sntpServer,
                                         uint16_t *serverVersion);

/*********************************************************************
* @purpose  Get the SNTP server port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sntpServer       @b{(input)}  SNTP server 
* @param    serverPort       @b{(output)} SNTP server port
*
* @returns  OPEN_E_NONE      if SNTP server port is returned.
* @returns  OPEN_E_NOT_FOUND if SNTP server does not exist.
* @returns  OPEN_E_FAIL      if failed to get SNTP server port.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerPortGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *sntpServer,
                                      uint16_t *serverPort);

/*********************************************************************
* @purpose  Set the SNTP client mode.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    mode             @b{(input)}  SNTP client mode
*
* @returns  OPEN_E_NONE      if SNTP client mode is set successfully.
* @returns  OPEN_E_FAIL      if failed to set SNTP client mode.
* @returns  OPEN_E_ERROR     if invalid SNTP client mode
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   To unconfigure SNTP client mode, pass the mode as OPEN_SNTP_CLIENT_DISABLED.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpClientModeSet(openapiClientHandle_t *client_handle,
                                      OPEN_SNTP_CLIENT_MODE_t mode);

/*********************************************************************
* @purpose  Get the SNTP client mode.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    sntpClientMode       @b{(output)} SNTP client mode
*
* @returns  OPEN_E_NONE      if SNTP client mode is returned.
* @returns  OPEN_E_FAIL      if failed to get SNTP client mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpClientModeGet(openapiClientHandle_t *client_handle,
                                      OPEN_SNTP_CLIENT_MODE_t *sntpClientMode);

/*********************************************************************
* @purpose  Get the SNTP client supported mode.
* 
* @param    client_handle           @b{(input)}  Client handle from 
*                                                registration API
* @param    sntpClientSupportedMode @b{(output)} SNTP client supported mode
*
* @returns  OPEN_E_NONE      if SNTP client supported mode is returned.
* @returns  OPEN_E_FAIL      if failed to get SNTP client mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpClientSupportedModeGet(openapiClientHandle_t *client_handle,
                                               OPEN_SNTP_SUPPORTED_MODE_t *sntpClientSupportedMode);


/*********************************************************************
* @purpose  Get the active SNTP server address type from which last 
*           packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    type                 @b{(output)} SNTP server address type
*
* @returns  OPEN_E_NONE      if active SNTP server address type is returned.
* @returns  OPEN_E_FAIL      if failed to get active SNTP server type.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiActiveSntpServerAddressTypeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_SNTP_ADDRESS_TYPE_t *type);

/*********************************************************************
* @purpose  Get the active SNTP server from which last packet is 
*           received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    sntpServer           @b{(output)} Active SNTP server
*
* @returns  OPEN_E_NONE      if active SNTP server address is returned.
* @returns  OPEN_E_FAIL      if there are no active SNTP servers .
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL  if internal error.
*
* @notes   Maximum size for ipAddr is determined by calling
*          openapiSntpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiActiveSntpServerIpAddressGet(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *sntpServer);

/*********************************************************************
* @purpose  Get the stratum of the SNTP server from which 
*           last packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    stratum              @b{(output)} Stratum of the SNTP server
*
* @returns  OPEN_E_NONE      if stratum of the SNTP server is returned.
* @returns  OPEN_E_FAIL      if failed to get stratum of SNTP server.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiActiveSntpServerStratumGet(openapiClientHandle_t *client_handle,
                                               uint32_t *stratum);

/*********************************************************************
* @purpose  Get the SNTP server reference identifier from which  last 
*           packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    reference            @b{(output)} Reference Identifier of SNTP server
*
* @returns  OPEN_E_NONE      if reference of the SNTP server is returned.
* @returns  OPEN_E_FAIL      if failed to get reference of SNTP server.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiActiveSntpServerReferenceIdGet(openapiClientHandle_t *client_handle,
                                                   open_buffdesc *reference);

/*********************************************************************
* @purpose  Get the maximum length of the SNTP reference ID string.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    refStrLen            @b{(output)} Reference ID string length
*
* @returns  OPEN_E_NONE      if reference ID string length id returned.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerReferenceStringLengthGet(openapiClientHandle_t *client_handle,
                                                       uint32_t *refStrLen);

/*********************************************************************
* @purpose  Get the mode of the SNTP server from which last packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    serverMode           @b{(output)} SNTP server mode
*
* @returns  OPEN_E_NONE      if SNTP server mode is returned.
* @returns  OPEN_E_FAIL      if failed to get SNTP server mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiActiveSntpServerModeGet(openapiClientHandle_t *client_handle,
                                            OPEN_SNTP_SERVER_MODE_t *serverMode);

/*********************************************************************
* @purpose  Get number of SNTP servers configured.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    currEntries          @b{(output)} Number of SNTP servers configured.
*
* @returns  OPEN_E_NONE      if current number of SNTP servers is returned.
* @returns  OPEN_E_FAIL      if failed to get current number of SNTP servers.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSntpServerCurrentEntriesGet(openapiClientHandle_t *client_handle,
                                                uint32_t *currEntries);
#endif
