/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_stacking.h
*
* @purpose   Stacking Configuration and Status 
*
* @component OpEN
*
* @create    10/22/2015
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_STACKING_H_INCLUDED
#define OPENAPI_STACKING_H_INCLUDED

#include "openapi_common.h"

/** Stacking Unit Status */
typedef enum
{
  OPEN_USMDB_UNITMGR_UNIT_OK          = 1,  /**< Unit Ok */
  OPEN_USMDB_UNITMGR_UNIT_UNSUPPORTED = 2,  /**< Unit Unsupported */
  OPEN_USMDB_UNITMGR_CODE_MISMATCH    = 3,  /**< Code Mismatch */
  OPEN_USMDB_UNITMGR_CFG_MISMATCH     = 4,  /**< CFG Mismatch */
  OPEN_USMDB_UNITMGR_SDM_MISMATCH     = 5,  /**< SDM Mismatch */
  OPEN_USMDB_UNITMGR_UNIT_NOT_PRESENT = 6,  /**< Unit Not Present */
  OPEN_USMDB_UNITMGR_CODE_UPDATE      = 7,  /**< Code Update */
  OPEN_USMDB_UNITMGR_STM_MISMATCH     = 8   /**< STM Mismatch*/
} OPEN_USMDB_UNITMGR_STATUS_t;

/** Management Flag */
typedef enum
{
  OPEN_USMDB_UNITMGR_MGMT_ENABLED     = 1,  /**< Enabled */
  OPEN_USMDB_UNITMGR_MGMT_DISABLED    = 2,  /**< Disabled */
  OPEN_USMDB_UNITMGR_MGMT_UNASSIGNED  = 3   /**< Unassigned */
} OPEN_USMDB_UNITMGR_MGMT_FLAG_t;

/** Unit's Management Preference */
typedef enum
{
  OPEN_USMDB_UNITMGR_MGMTPREF_DISABLED    = 0,  /**< Disabled */
  OPEN_USMDB_UNITMGR_MGMTPREF_1           = 1,  /**< Preference 1 */
  OPEN_USMDB_UNITMGR_MGMTPREF_2           = 2,  /**< Preference 2 */
  OPEN_USMDB_UNITMGR_MGMTPREF_3           = 3,  /**< Preference 3 */
  OPEN_USMDB_UNITMGR_MGMTPREF_4           = 4,  /**< Preference 4 */
  OPEN_USMDB_UNITMGR_MGMTPREF_5           = 5,  /**< Preference 5 */
  OPEN_USMDB_UNITMGR_MGMTPREF_6           = 6,  /**< Preference 6 */
  OPEN_USMDB_UNITMGR_MGMTPREF_7           = 7,  /**< Preference 7 */
  OPEN_USMDB_UNITMGR_MGMTPREF_8           = 8,  /**< Preference 8 */
  OPEN_USMDB_UNITMGR_MGMTPREF_9           = 9,  /**< Preference 9 */
  OPEN_USMDB_UNITMGR_MGMTPREF_10          = 10, /**< Preference 10 */
  OPEN_USMDB_UNITMGR_MGMTPREF_11          = 11, /**< Preference 11 */
  OPEN_USMDB_UNITMGR_MGMTPREF_12          = 12, /**< Preference 12 */
  OPEN_USMDB_UNITMGR_MGMTPREF_13          = 13, /**< Preference 13 */
  OPEN_USMDB_UNITMGR_MGMTPREF_14          = 14, /**< Preference 14 */
  OPEN_USMDB_UNITMGR_MGMTPREF_15          = 15, /**< Preference 15 */
  OPEN_USMDB_UNITMGR_MGMTFUNC_UNASSIGNED  = 16  /**< Preference 16 */
} OPEN_USMDB_UNITMGR_MGMTPREF_t;

/** Stack Firmware Synchronization Status */
typedef enum
{
  OPEN_SFS_SWITCH_STATUS_NO_ACTION    = 1,  /**< No Action */
  OPEN_SFS_SWITCH_STATUS_IN_PROGRESS  = 2   /**< In Progress */
} OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t;

/** Stack Firmware Synchronization Last Attempt Status */
typedef enum
{
  OPEN_SFS_NONE                             = 1,  /**< None */
  OPEN_SFS_SUCCESS                          = 2,  /**< Success */
  OPEN_SFS_FAIL                             = 3,  /**< Fail */
  OPEN_SFS_MIN_BOOTCODE_VERSION_NOT_PRESENT = 4   /**< Min Bootcode Version Not Present */
} OPEN_SFS_LAST_ATTEMPT_STATUS_t;


/*********************************************************************
* @purpose  Given a unit, gets the next member of the current stack
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    startUnit       @b{(input)}  Initial unit number
* @param    nextUnit        @b{(output)} Pointer to return next unit in stack
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackMemberGetNext(openapiClientHandle_t *client_handle,
                                              uint32_t              startUnit,
                                              uint32_t              *nextUnit);

/*********************************************************************
* @purpose  Get the manager's unit number
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(output)} Pointer to manager's unit number
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrNumberGet(openapiClientHandle_t *client_handle,
                                     uint32_t              *unit);

/*********************************************************************
* @purpose  Given a unit, gets the unit's serial number
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  unit number
* @param    serialNumber     @b{(output)} Pointer to unit's serial number
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_UNITMGR_SERIAL_NUMBER_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSerialNumberGet(openapiClientHandle_t *client_handle,
                                           uint32_t              unit,
                                           open_buffdesc         *serialNumber);

/*********************************************************************
* @purpose  Given a unit, gets the service tag
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    serviceTag       @b{(output)} Pointer to unit's service tag.
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSMGMT_SERVICE_TAG_MAX_LENGTH
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrServiceTagGet(openapiClientHandle_t *client_handle,
                                         uint32_t              unit,
                                         open_buffdesc         *serviceTag);

/*********************************************************************
* @purpose  Given a unit, gets the unit's model identifier string
*           assigned by HPC(Hardware Platform Control)
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    modelId          @b{(output)} Pointer to model id string storage
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_MODEL_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrPreconfigModelIdGet(openapiClientHandle_t *client_handle,
                                               uint32_t              unit,
                                               open_buffdesc         *modelId);

/*********************************************************************
* @purpose  Given a unit, verify that it's within the valid range.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    isValid          @b{(output)} Returns @ref OPEN_TRUE if unit number
*                                         is valid, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitIsValid(openapiClientHandle_t *client_handle,
                                       uint32_t              unit,
                                       OPEN_BOOL_t           *isValid);

/*********************************************************************
* @purpose  Gets the maximum stacking unit number (largest possible
*           number of units in the stack).
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(output)} Unit number
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes    The default unit number is returned for non-stacking builds.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitMaxGet(openapiClientHandle_t *client_handle,
                                      uint32_t              *unit);

/*********************************************************************
* @purpose  Given a unit, gets the unit's status.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    status           @b{(output)} Unit status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitStatusGet(openapiClientHandle_t       *client_handle,
                                         uint32_t                    unit,
                                         OPEN_USMDB_UNITMGR_STATUS_t *status);

/*********************************************************************
* @purpose  Given a unit, gets the unit's management status.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    mgmtStatus       @b{(output)} Management status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrMgmtUnitStatusGet(openapiClientHandle_t          *client_handle,
                                             uint32_t                       unit,
                                             OPEN_USMDB_UNITMGR_MGMT_FLAG_t *mgmtStatus);

/*********************************************************************
* @purpose  Given a unit, gets the unit's model identifier string
*           assigned by HPC and pre-configured model identifier string.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    modelId          @b{(output)} Model id
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_MODEL_SIZE
* @param    preconfigModelId @b{(output)} Pre-configured model id
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_MODEL_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrModelIdentifierGet(openapiClientHandle_t *client_handle,
                                              uint32_t              unit,
                                              open_buffdesc         *modelId,
                                              open_buffdesc         *preconfigModelId);

/*********************************************************************
* @purpose  Given a unit, gets the unit's version of code in flash.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    codeVersion      @b{(output)} Code version
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDetectCodeVerFlashGet(openapiClientHandle_t *client_handle,
                                                 uint32_t              unit,
                                                 open_revision_data_t  *codeVersion);

/*********************************************************************
* @purpose  Given a unit, gets the unit's running version of code.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    codeVersion      @b{(output)} Code version
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrDetectCodeVerRunningGet(openapiClientHandle_t *client_handle,
                                                   uint32_t              unit,
                                                   open_revision_data_t  *codeVersion);

/*********************************************************************
* @purpose  Given a unit, gets the unit's hardware and admin management preference.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    hwPref           @b{(output)} Hardware preference
* @param    adminPref        @b{(output)} Admin preference
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrMgmtPrefGet(openapiClientHandle_t         *client_handle,
                                       uint32_t                      unit,
                                       OPEN_USMDB_UNITMGR_MGMTPREF_t *hwPref,
                                       OPEN_USMDB_UNITMGR_MGMTPREF_t *adminPref);

/*********************************************************************
* @purpose  Given a unit, gets the unit's type and index.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    unitType         @b{(output)} Unit type of hardware 
*                                         used to determine stacking compatibility
* @param    unitTypeIndex    @b{(output)} Unit type index
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitTypeGet(openapiClientHandle_t *client_handle,
                                       uint32_t              unit,
                                       uint32_t              *unitType,
                                       uint32_t              *unitTypeIndex);

/*********************************************************************
* @purpose  Get unit description that corresponds to the index provided.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unitTypeIndex    @b{(input)}  Unit index
* @param    unitDescr        @b{(output)} Unit description
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_UNIT_DESCRIPTION_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedDescriptionGet(openapiClientHandle_t *client_handle,
                                                   uint32_t              unitTypeIndex,
                                                   open_buffdesc         *unitDescr);

/*********************************************************************
* @purpose  Get stack firmware synchronization status of member unit.
*
* @param    client_handle       @b{(input)}  Client handle from
*                                            registration API
* @param    unit                @b{(input)}  Unit number
* @param    firmwareSyncStatus  @b{(output)} Firmware synchronization status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncMemberStatusGet(openapiClientHandle_t             *client_handle,
                                                            uint32_t                          unit,
                                                            OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t *firmwareSyncStatus);

/*********************************************************************
* @purpose  Get stack firmware synchronization status.
*
* @param    client_handle       @b{(input)}  Client handle from
*                                            registration API
* @param    firmwareSyncStatus  @b{(output)} Firmware synchronization status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncStatusGet(openapiClientHandle_t             *client_handle,
                                                      OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t *firmwareSyncStatus);

/*********************************************************************
* @purpose  Get stack firmware synchronization last attempt status.
*
* @param    client_handle     @b{(input)}  Client handle from
*                                          registration API
* @param    unit              @b{(input)}  Unit number
* @param    lastAttemptStatus @b{(output)} Last attempt status
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStackFirmwareSyncMemberLastAttemptStatusGet(openapiClientHandle_t *client_handle,
                                                                       uint32_t              unit,
                                                                       uint32_t              *lastAttemptStatus);

/*********************************************************************
* @purpose  Gets active template id of the stack unit.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    templateId       @b{(output)} Template id
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiStmUnitActiveTemplateIdGet(openapiClientHandle_t *client_handle,
                                               uint32_t              unit,
                                               uint32_t              *templateId);

/*********************************************************************
* @purpose  Gets description of specified template id.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    templateId       @b{(input)}  Template id
* @param    templateDescr    @b{(output)} Template description
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiStmTemplateDescriptionGet(openapiClientHandle_t *client_handle,
                                              uint32_t              templateId,
                                              open_buffdesc         *templateDescr);

/*********************************************************************
* @purpose  Given a unit, gets the unit's up time in seconds.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    upTime           @b{(output)} Up time
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitUpTimeGet(openapiClientHandle_t *client_handle,
                                         uint32_t              unit,
                                         uint32_t              *upTime);

/*********************************************************************
* @purpose  Get unit type identifier that corresponds to
*           the index provided
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unitTypeIndex    @b{(input)}  Unit type index
* @param    unitType         @b{(output)} Unit type of hardware 
*                                         used to determine stacking compatibility
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedUnitIdGet(openapiClientHandle_t *client_handle,
                                              uint32_t              unitTypeIndex,
                                              uint32_t              *unitType);

/*********************************************************************
* @purpose  Get unit type identifier that corresponds to
*           the next index
*
* @param    client_handle     @b{(input)}  Client handle from
*                                          registration API
* @param    unitTypeIndex     @b{(input)}  Unit type index
* @param    nextUnitTypeIndex @b{(output)} Unit type index
* @param    unitType         @b{(output)} Unit type of hardware 
*                                         used to determine stacking compatibility
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrSupportedUnitIdNextGet(openapiClientHandle_t *client_handle,
                                                  uint32_t              unitTypeIndex,
                                                  uint32_t              *nextUnitTypeIndex,
                                                  uint32_t              *unitType);

/*********************************************************************
* @purpose  Create a new unit record for unit
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    unitTypeIndex    @b{(input)}  Unit type index
* @param    synchMode        @b{(input)}  Synchronous mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if create operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes    When it is set to 1, registrants within the switch 
*           are notified of card insertion(s) during the execution 
*           of the API call, otherwise they are notified afterwards (asynchronously).
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitCreate(openapiClientHandle_t *client_handle,
                                      uint32_t              unit,
                                      uint32_t              unitTypeIndex,
                                      OPEN_BOOL_t           synchMode);

/*********************************************************************
* @purpose  Remove a unit only if it is not an active member of
*           the stack
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if delete operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_ERROR     if unit is an active member of the stack
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitDelete(openapiClientHandle_t *client_handle,
                                      uint32_t              unit);

/*********************************************************************
* @purpose  Transfers the management function from the current unit
*           to another
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    targetUnit       @b{(input)}  Target unit
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrUnitTransfer(openapiClientHandle_t *client_handle,
                                        uint32_t              targetUnit);

/*********************************************************************
* @purpose  Get QOS configuration for all front-panel stacking ports
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    fpsQosMode       @b{(output)} Returns @ref OPEN_ENABLE if QOS mode
*                                         is enabled, else @ref OPEN_DISABLE if not
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigQosModeGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t        *fpsQosMode);

/*********************************************************************
* @purpose  Set QOS configuration for all front-panel stacking ports
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    fpsQosMode       @b{(input)}  Front panel stacking QOS mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if set operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigQosModeSet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t        fpsQosMode);

/*********************************************************************
* @purpose  Gets the configured and standby unit number and admin mode in the stack.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    configUnit       @b{(output)} Configured unit
* @param    standbyUnit      @b{(output)} Standby unit
* @param    standbyAdminMode @b{(output)} Returns @ref OPEN_TRUE if admin mode
*                                         is enabled, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStandbyGet(openapiClientHandle_t *client_handle,
                                      uint32_t              *configUnit,
                                      uint32_t              *standbyUnit,
                                      OPEN_BOOL_t           *standbyAdminMode);

/*********************************************************************
* @purpose  Sets the unit as standby in the stack
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    standbyUnit      @b{(input)}  Standby unit
* @param    standbyAdminMode @b{(input)}  Standby admin mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if set operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiUnitMgrStandbySet(openapiClientHandle_t *client_handle,
                                      uint32_t              standbyUnit,
                                      OPEN_BOOL_t           standbyAdminMode);

/*********************************************************************
* @purpose  Get stacking mode configuration for a front-panel
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    slot             @b{(input)}  Slot
* @param    port             @b{(input)}  Port
* @param    stackPortMode    @b{(output)} Returns @ref OPEN_TRUE if admin mode
*                                         is enabled, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if issued successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigStackingModeGet(openapiClientHandle_t *client_handle,
                                                uint32_t              unit,
                                                uint32_t              slot,
                                                uint32_t              port,
                                                OPEN_BOOL_t           *stackPortMode);

/*********************************************************************
* @purpose  Set stacking mode configuration for a front-panel
*           stack port
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    slot             @b{(input)}  Slot
* @param    port             @b{(input)}  Port
* @param    stackPortMode    @b{(input)}  Enable front-panel stacking mode
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if set operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_UNAVAIL   if feature is not supported 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.10 
*
* @end
*********************************************************************/
open_error_t openapiSpmFpsConfigStackingModeSet(openapiClientHandle_t *client_handle,
                                                uint32_t              unit,
                                                uint32_t              slot,
                                                uint32_t              port,
                                                OPEN_BOOL_t           stackPortMode);



/*********************************************************************
* @purpose  Given a unit and slot number, verify whether the slot exists in the unit.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit number
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isValid          @b{(output)} Returns @ref OPEN_TRUE if slot number
*                                         exists in the unit, else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotGet(openapiClientHandle_t *client_handle,
                                       uint32_t unit,
                                       uint32_t slot,
                                       OPEN_BOOL_t *isValid);

/**************************************************************************
* @purpose  Given a unit number, it retrieves all slot information from the unit descriptor database.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b((input))  unit type identifier
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    count            @b((output)) The number of slots in array
* @param    bufd             @b((output)) The slots array. The size of each slot is uint32_t.
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_L7_MAX_PHYSICAL_SLOTS_PER_UNIT
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_FAIL      if get operation has failed
* 
* @supported-in-version OpEN API Version: 1.10
*
* @end
*************************************************************************/
open_error_t openapiUnitDbEntrySlotsGet(openapiClientHandle_t *client_handle,
                                        uint32_t unit,
                                        uint32_t *count,
                                        open_buffdesc *bufd);

/*********************************************************************
* @purpose  Given a unit and slot number, verify whether the slot exists in the unit.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isFull           @b{(output)} Returns @ref OPEN_TRUE if slot is full
*                                         else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotIsFullGet(openapiClientHandle_t *client_handle,
                                  uint32_t unit,
                                  uint32_t slot,
                                  OPEN_BOOL_t *isFull);

/*********************************************************************
* @purpose  Given a unit and slot number, gets admin mode of slot.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    adminMode        @b{(output)} Configured Administrative mode
* @param    -                             Returns @ref OPEN_ENABLE if adminMode is enabled
*                                         else @ref OPEN_DISABLE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotAdminModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Given a unit,  slot number and admin mode, configures it on device.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    adminMode        @b{(input)}  Configured Administrative mode
* @param    -                             Configures enabled if adminMode is @ref OPEN_ENABLE
* @param    -                             Configures disabled if adminMode is @ref OPEN_DISABLE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotAdminModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Given a unit and slot number, gets power mode of slot.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    powerMode        @b{(output)} Configured Power mode
* @param    -                             Returns @ref OPEN_ENABLE if powerMode is enabled
*                                         else @ref OPEN_DISABLE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotPowerModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t *powerMode);

/*********************************************************************
* @purpose  Given a unit,  slot number and power mode, configures it on device.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    powerMode        @b{(input)}  Configured power mode
* @param    -                             Configures enabled if powerMode is @ref OPEN_ENABLE
* @param    -                             Configures disabled if powerMode is @ref OPEN_DISABLE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotPowerModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t unit,
                                     uint32_t slot,
                                     OPEN_CONTROL_t powerMode);

/*********************************************************************
* @purpose  Given a unit and slot number, gets configured card type and inserted card type.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    cardIdConfig     @b{(output)} Configured card type
* @param    cardIdInsert     @b{(output)} Inserted card type
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiCardTypeGet(openapiClientHandle_t *client_handle,
                                uint32_t unit,
                                uint32_t slot,
                                uint32_t *cardIdConfig,
                                uint32_t *cardIdInsert);

/*********************************************************************
* @purpose  Given card index, gets model Identifier.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    cardId           @b{(input)}  Card Index.
* @param    bufd             @b{(output)} Buffer to store Card Model Id
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_CARD_MODEL_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiCardSupportedCardModelIdGet(openapiClientHandle_t *client_handle,
                                                uint32_t cardId,
                                                open_buffdesc *bufd);

/*********************************************************************
* @purpose  Given card index, gets card description.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    cardId           @b{(input)}  Card Index.
* @param    bufd             @b{(output)} Buffer to store Card description
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_SYSAPI_HPC_CARD_DESCRIPTION_SIZE
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiCardSupportedCardDescriptionGet(openapiClientHandle_t *client_handle,
                                                    uint32_t cardId,
                                                    open_buffdesc *bufd);

/*********************************************************************
* @purpose  Given a unit and slot number, verify whether it has pluggable indicator.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isPluggable      @b{(output)} Returns @ref OPEN_TRUE if card can be plugged into slot
*                                         else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotSupportedPluggableGet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              uint32_t slot,
                                              OPEN_BOOL_t *isPluggable);

/*********************************************************************
* @purpose  Given a unit and slot number, verify whether it powerdown indicator.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit ID of the unit
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    slot             @b{(input)}  Slot ID of the slot port
* @param    -                             Maximum value @ref OPENAPI_L7_MAX_SLOTS_PER_UNIT
* @param    isPowerdown      @b{(output)} Returns @ref OPEN_TRUE if power down on slot
*                                         else @ref OPEN_FALSE if not
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_NOT_FOUND If this slot is not configured
* @returns  OPEN_E_FAIL      if get operation has failed
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiSlotSupportedPowerdownGet(openapiClientHandle_t *client_handle,
                                              uint32_t unit,
                                              uint32_t slot,
                                              OPEN_BOOL_t *isPowerdown);

#endif /* OPENAPI_STACKING_H_INCLUDED */
