/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_system.h
*
* @purpose   System Configuration and Status 
*
* @component OpEN
*
* @create    28/01/2013
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_SYSTEM_H_INCLUDED
#define OPENAPI_SYSTEM_H_INCLUDED
#define OPENAPI_HW_VERSION_STR_SIZE 128
#define OPENAPI_OS_REVISION_STR_SIZE 100

#include "openapi_common.h"

/** OpEN uses this enumeration to define network connectivity protocol modes */
typedef enum
{
  OPEN_PROTOCOL_MODE_INVALID = 0, /**< Invalid Protocol mode */
  OPEN_PROTOCOL_MODE_NONE    = 1, /**< Protocol mode none */
  OPEN_PROTOCOL_MODE_BOOTP   = 2, /**< Protocol mode BOOTP */
  OPEN_PROTOCOL_MODE_DHCP    = 3  /**< Protocol mode DHCP */
} OPEN_PROTOCOL_MODE_t;

#define OPEN_MANUFACTURER_SIZE 6 + 1
#define OPEN_OS_VERSION_SIZE   60
#define OPEN_SYSTEM_PARAM_SIZE 256

/*********************************************************************
* @purpose  Set the system prompt.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    systemPrompt     @b{(input)}  System prompt
*
* @returns  OPEN_E_NONE      if system prompt set successfully.
* @returns  OPEN_E_FAIL      if system prompt contains non printable 
*                            characters.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Maximum size for systemPrompt is determined by calling
*          openapiSystemPromptMaxSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemPromptSet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *systemPrompt);

/*********************************************************************
* @purpose  Get the maximum size of system prompt.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    promptSize       @b{(output)} Maximum size of the system prompt
*
* @returns  OPEN_E_NONE      if maximum prompt size is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemPromptMaxSizeGet(openapiClientHandle_t *client_handle,
                                           uint32_t *promptSize);

/*********************************************************************
* @purpose  Set the system hostname.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sysHostName     @b{(input)}   System hostname
*
* @returns  OPEN_E_NONE      if system hostname is set successfully.
* @returns  OPEN_E_FAIL      if system hostname contains non printable 
*                            characters.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Maximum size for systemPrompt is determined by calling
*          openapiSystemPromptMaxSizeGet().
* @notes   To set the system host name to default, pass the host 
*          name as empty string
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemHostnameSet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *sysHostName);

/*********************************************************************
* @purpose  Get the system hostname.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    sysHostName     @b{(output)}  System hostname
*
* @returns  OPEN_E_NONE      if system hostname is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Maximum size for sysHostName is determined by calling
*          openapiSystemHostnameSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemHostnameGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *sysHostName);

/*********************************************************************
* @purpose  Get the maximum size of system hostname.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    hostNameSize     @b{(output)} Maximum size of system hostname
*
* @returns  OPEN_E_NONE      if maximum hostname size is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemHostnameSizeGet(openapiClientHandle_t *client_handle,
                                          uint32_t *hostNameSize);

/*********************************************************************
* @purpose Get the current unit's ID
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(output)} Local unit number
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes    Functional for both stacking and non-stacking builds
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiSystemLocalUnitGet(openapiClientHandle_t *client_handle,
                                       uint32_t *unit);

/*********************************************************************
* @purpose  Get the administratively-assigned system name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    systemName       @b{(output)}  System name
*
* @returns  OPEN_E_NONE      if system hostname is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Maximum size for systemName is determined by calling
*          openapiSystemNameMaxSizeGet().
* 
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiSystemNameGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *systemName);

/*********************************************************************
* @purpose  Get the maximum size of the system name.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    systemNameSize   @b{(output)} Maximum size of the system name
*
* @returns  OPEN_E_NONE      if maximum prompt size is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiSystemNameMaxSizeGet(openapiClientHandle_t *client_handle,
                                         uint32_t *systemNameSize);

/*********************************************************************
* @purpose  Set the terminal length of an access line.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access Line: Console/Telnet/SSH
* @param    termLength       @b{(input)}  Terminal length of access line
* @param    -                Minimum value @ref OPENAPI_MIN_TERMINAL_LENGTH
* @param    -                Maximum value @ref OPENAPI_MAX_TERMINAL_LENGTH
* 
* @returns  OPEN_E_NONE      if terminal length for the given access line 
*                            is set successfully 
* @returns  OPEN_E_FAIL      if access line is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Setting the terminal length to 0 will disable pagination on 
*           that access line.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLineTerminalLengthSet(openapiClientHandle_t *client_handle,
                                          OPEN_ACCESS_LINE_t accessLine,
                                          uint32_t termLength);

/*********************************************************************
* @purpose  Set the terminal length of an access line to default.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access Line: Console/Telnet/SSH
*
* @returns  OPEN_E_NONE      if terminal length for the given access line 
*                            is set to default successfully. 
* @returns  OPEN_E_FAIL      if access line is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLineTerminalLengthDefaultSet(openapiClientHandle_t *client_handle,
                                                 OPEN_ACCESS_LINE_t accessLine);

/*********************************************************************
* @purpose  Get the terminal length of an access line.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    accessLine       @b{(input)}  Access Line: Console/Telnet/SSH
* @param    termLength       @b{(output)} Terminal length of the
*                                         access line
*
* @returns  OPEN_E_NONE      if terminal length of a given access line 
*                            is returned successfully.
* @returns  OPEN_E_FAIL      if access line is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiLineTerminalLengthGet(openapiClientHandle_t *client_handle,
                                          OPEN_ACCESS_LINE_t accessLine,
                                          uint32_t *termLength);

/*********************************************************************
* @purpose  Set the serial timeout.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    serialTimeout    @b{(input)}  Serial timeout (minutes)
* @param    -                Maximum value @ref OPENAPI_SERIAL_TIMEOUT_MAX
* 
* @returns  OPEN_E_NONE      if serial timeout is set successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSerialTimeoutSet(openapiClientHandle_t *client_handle,
                                     uint32_t serialTimeout);

/*********************************************************************
* @purpose  Set the serial timeout to default value.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* 
* @returns  OPEN_E_NONE      if serial timeout default value is set successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSerialTimeoutDefaultSet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the serial timeout.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    serialTimeout    @b{(output)} Serial timeout
* 
* @returns  OPEN_E_NONE      if serial timeout is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSerialTimeoutGet(openapiClientHandle_t *client_handle,
                                     uint32_t *serialTimeout);

/*********************************************************************
* @purpose  Get the network management VLAN of the system
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    mgmtVlan         @b{(output)} network management VLAN ID
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiNetworkMgmtVlanGet(openapiClientHandle_t *client_handle,
                                       uint32_t *mgmtVlan);

/*********************************************************************
* @purpose  Set the network management VLAN on the system
* 
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    mgmtVlan         @b{(input)}  network management VLAN ID
* @param    -                Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
* @param    -                Maximum value openapiDot1VlanMaxGet()
* 
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  An internal error occurred.
* @returns  OPEN_E_EXISTS    VLAN already in use 
*                            (e.g. for a port-based routing interface).
*  
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.14
* 
* @end
*********************************************************************/
open_error_t openapiNetworkMgmtVlanSet(openapiClientHandle_t *client_handle,
                                       uint32_t mgmtVlan);

/*********************************************************************
* @purpose  Get network interface link state.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    linkState        @b{(output)} Interface link state
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
* @returns  OPEN_E_ERROR     otherwise
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortLinkStateGet(openapiClientHandle_t *client_handle,
                                            OPEN_LINK_STATE_t *linkState);

/*********************************************************************
* @purpose  Set the network parameters.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Network port IPv4 address
* @param    netMask          @b{(input)}  Network mask
* @param    gateway          @b{(input)}  Gateway IP address
* 
* @returns  OPEN_E_NONE      if the network parameters are set successfully.
* @returns  OPEN_E_ERROR     if IP address conflicts occur.
* @returns  OPEN_E_EXISTS    if the same values of IP address, 
*                            network mask and gateway are already set.
* @returns  OPEN_E_FAIL      if network protocol mode is not set to none or
*                            if ip address or network mask are passed as 0.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Network protocol mode must be set to "none" before applying 
*          this configuration.
* @notes   Pass the structure variable gateway->addr.ipv4 as 0 if not required to be set.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv4ParametersSet(openapiClientHandle_t *client_handle,
                                                 open_inet_addr_t *ipAddr,
                                                 open_inet_addr_t *netMask,
                                                 open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Delete network port IPv4 parameters.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* 
* @returns  OPEN_E_NONE      if the network port IPv4 parameters are deleted successfully.
* @returns  OPEN_E_FAIL      if network protocol mode is not set to none.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Network protocol mode must be set to "none" before applying 
*          this configuration.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv4ParametersDelete(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the network port IPv4 address and subnet mask.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(output)} network port IPv4 address
* @param    netMask          @b{(output)} Network mask
* 
* @returns  OPEN_E_NONE      if the network port IPv4 address 
*                            and subnet mask are returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv4AddrGet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *ipAddr,
                                           open_inet_addr_t *netMask);

/*********************************************************************
* @purpose  Get the network port gateway IPv4 address.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    gateway          @b{(output)} Gateway IPv4 address
* 
* @returns  OPEN_E_NONE      if the network port gateway IPv4 
*                            address is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortGatewayIPv4AddrGet(openapiClientHandle_t *client_handle,
                                                  open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Set IPv6 administrative mode on network port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    adminMode        @b{(input)}  IPv6 admin mode of network port
* 
* @returns  OPEN_E_NONE      if IPv6 admin mode of network port is set successfully.
* @returns  OPEN_E_FAIL      if unable to set IPv6 admin mode on network port.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6AdminModeSet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Get IPv6 administrative mode of network port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    adminMode        @b{(output)} IPv6 admin mode of network port
* 
* @returns  OPEN_E_NONE      if IPv6 admin mode of network port is returned.
* @returns  OPEN_E_FAIL      if invalid mode is returned.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6AdminModeGet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Set IPv6 auto-config mode on Network port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    autoConfigMode   @b{(input)}  IPv6 auto config mode
* 
* @returns  OPEN_E_NONE      if IPv6 auto-config mode is set successfully.
* @returns  OPEN_E_FAIL      if unable to set auto-config mode.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch. 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6AutoConfigModeSet(openapiClientHandle_t *client_handle,
                                                     OPEN_CONTROL_t autoConfigMode);

/*********************************************************************
* @purpose  Get IPv6 auto-config mode on network port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    autoConfigMode   @b{(output)} IPv6 auto config mode
* 
* @returns  OPEN_E_NONE      if IPv6 auto-config mode of network port is returned.
* @returns  OPEN_E_FAIL      if invalid mode is returned.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6AutoConfigModeGet(openapiClientHandle_t *client_handle,
                                                     OPEN_CONTROL_t *autoConfigMode);

/*********************************************************************
* @purpose  Set IPv6 protocol mode on network port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(input)}  IPv6 protocol mode: dhcp/none
* 
* @returns  OPEN_E_NONE      if IPv6 protocol mode is set successfully.
* @returns  OPEN_E_FAIL      if unable to set IPv6 protocol mode.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6ProtocolModeSet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t protoMode);

/*********************************************************************
* @purpose  Set IPv6 protocol mode on network port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(output)} IPv6 protocol mode
* 
* @returns  OPEN_E_NONE      if IPv6 protocol mode is returned successfully.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6ProtocolModeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t *protoMode);

/*********************************************************************
* @purpose  Set the network  IPv6 prefix and prefix length.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipv6Addr         @b{(input)}  IPv6 address and prefix length
* 
* @returns  OPEN_E_NONE      if the network parameters are set successfully.
* @returns  OPEN_E_EXISTS    if IP address conflicts occur with 
*                            service port/network port prefixes.
* @returns  OPEN_E_UNAVAIL   if invalid prefix length is given.
* @returns  OPEN_E_FULL      if no more space to add ipv6 prefixes.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6PrefixSet(openapiClientHandle_t *client_handle,
                                             open_inet_pfx_t *ipv6Addr);

/*********************************************************************
* @purpose  Set the network  port IPv6 gateway.
*
* @param    client_handle    @b{(input)}  Client handle from 
*
* @param    gateway          @b{(input)}  IPv6 gateway address.
* 
* @returns  OPEN_E_NONE      if the network port IPv6 gateway is set successfully.
* @returns  OPEN_E_EXISTS    if IPv6 address conflicts occur with 
*                            service port/network port gateway
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Pass the structure variable gateway->addr.ipv4 as 0 if not required to be set.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6GatewaySet(openapiClientHandle_t *client_handle,
                                              open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Get next network port configured IPv6 address and prefix length.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipv6Addr         @b{(inout)} IPv6 address and prefix length
* 
* @returns  OPEN_E_NONE      if the next configured IPv6 network address and prefix 
*                            length are returned.
* @returns  OPEN_E_FAIL      if no more entries are found.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Set all the structure elements pointed by ipv6Addr to 0
*          to get the first address.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6AddressesNextGet(openapiClientHandle_t *client_handle,
                                                    open_inet_pfx_t *ipv6Addr);

/*********************************************************************
* @purpose  Get the network port IPv6 gateway address.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    gateway          @b{(output)} IPv6 gateway address
* 
* @returns  OPEN_E_NONE      if IPv6 gateway address is returned.
* @returns  OPEN_E_FAIL      if IPv6 gateway address is not configured on network port.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6GatewayGet(openapiClientHandle_t *client_handle,
                                              open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Delete IPv6 prefix(es) on network port.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipv6Addr         @b{(input)}  IPv6 address and prefix length.
* 
* @returns  OPEN_E_NONE      if the network parameters are deleted successfully.
* @returns  OPEN_E_FAIL      if unable remove the parameters or
*                            if invalid prefix is passed or 
*                            if parameters does not exist.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Set all the structure elements pointed by ipv6Addr to 0 to delete all
*          the IPv6 prefixes.
*                                                   
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv6PrefixDelete(openapiClientHandle_t *client_handle,
                                                open_inet_pfx_t *ipv6Addr);

/*********************************************************************
* @purpose  Set the service port IPv4 parameters.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  IPv4 address
* @param    netMask          @b{(input)}  Network mask
* @param    gateway          @b{(input)}  Gateway
* 
* @returns  OPEN_E_NONE      if the service port IPv4 parameters are set successfully.
* @returns  OPEN_E_ERROR     if IP address conflicts occur.
* @returns  OPEN_E_EXISTS    if the same values of IP address, 
*                            service port subnet mask and gateway are already set.
* @returns  OPEN_E_FAIL      if service protocol mode is not set to none.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Service port protocol mode must be set to none before applying 
*          this configuration.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv4ParametersSet(openapiClientHandle_t *client_handle,
                                                 open_inet_addr_t *ipAddr,
                                                 open_inet_addr_t *netMask,
                                                 open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Delete IPv4 service port parameters.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* 
* @returns  OPEN_E_NONE      if the service port IPv4 parameters are deleted successfully.
* @returns  OPEN_E_FAIL      if service protocol mode is not set to none.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Service port protocol mode must be set to none before applying 
*          this configuration.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv4ParametersDelete(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the service port IPv4 address and subnet mask.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(output)} Service port IPv4 address
* @param    netMask          @b{(output)} Network mask
* 
* @returns  OPEN_E_NONE      if the service port IPv4 address 
*                            and subnet mask are returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv4AddrGet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *ipAddr,
                                           open_inet_addr_t *netMask);

/*********************************************************************
* @purpose  Get the service port subnet mask.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    netMask          @b{(output)} Network mask
* 
* @returns  OPEN_E_NONE      if the service port subnet  mask is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortMaskGet(openapiClientHandle_t *client_handle,
                                       open_inet_addr_t *netMask);

/*********************************************************************
* @purpose  Get the service port gateway IPv4 address.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    gateway          @b{(output)} Gateway IPv4 address
* 
* @returns  OPEN_E_NONE      if the service port gateway IP 
*                            address is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortGatewayIPv4AddrGet(openapiClientHandle_t *client_handle,
                                                  open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Set IPv6 administrative mode of service port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    adminMode        @b{(input)}  IPv6 admin mode of service port
* 
* @returns  OPEN_E_NONE      if IPv6 admin mode of service port is set successfully.
* @returns  OPEN_E_FAIL      if unable to set IPv6 admin mode of service port.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6AdminModeSet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Get IPv6 administrative mode of service port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    adminMode        @b{(output)} IPv6 admin mode of service port
* 
* @returns  OPEN_E_NONE      if IPv6 admin mode of service port is returned.
* @returns  OPEN_E_FAIL      if invalid mode is returned.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6AdminModeGet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Set IPv6 auto-config mode on service port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    autoConfigMode   @b{(input)} IPv6 auto config mode
* 
* @returns  OPEN_E_NONE      if IPv6 auto-config mode is set successfully.
* @returns  OPEN_E_FAIL      if unable to set auto-config mode.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch. 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6AutoConfigModeSet(openapiClientHandle_t *client_handle,
                                                     OPEN_CONTROL_t autoConfigMode);

/*********************************************************************
* @purpose  Get IPv6 auto-config mode of service port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    autoConfigMode   @b{(output)} IPv6 auto config mode
* 
* @returns  OPEN_E_NONE      if IPv6 auto-config mode is returned.
* @returns  OPEN_E_FAIL      if invalid mode is returned.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6AutoConfigModeGet(openapiClientHandle_t *client_handle,
                                                     OPEN_CONTROL_t *autoConfigMode);

/*********************************************************************
* @purpose  Set IPv6 protocol mode on service port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(input)}  IPv6 protocol mode: dhcp/none
* 
* @returns  OPEN_E_NONE      if IPv6 protocol mode is set successfully.
* @returns  OPEN_E_FAIL      if unable to set IPv6 protocol mode.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6ProtocolModeSet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t protoMode);

/*********************************************************************
* @purpose  Get IPv6 protocol mode of service port.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(output)} IPv6 protocol mode
* 
* @returns  OPEN_E_NONE      if IPv6 protocol mode is returned.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6ProtocolModeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t *protoMode);

/*********************************************************************
* @purpose  Set the service port IPv6 parameters.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipv6Addr         @b{(input)}  IPv6 address and prefix length
*
* @returns  OPEN_E_NONE      if the service port prefix and prefix length 
*                            are set successfully.
* @returns  OPEN_E_EXISTS    if IP address conflicts occur with 
*                            service port/network port prefixes.
* @returns  OPEN_E_UNAVAIL   if invalid prefix length is given.
* @returns  OPEN_E_FULL      if no more space to add ipv6 prefixes.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6PrefixSet(openapiClientHandle_t *client_handle,
                                             open_inet_pfx_t *ipv6Addr);

/*********************************************************************
* @purpose  Set the service port IPv6 gateway.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    gateway          @b{(input)}  IPv6 gateway address
*
* @returns  OPEN_E_NONE      if the service port gateway is set successfully.
* @returns  OPEN_E_EXISTS    if IP address conflicts occur with 
*                            service port/network port gateway.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_INTERNAL  if internal error has occurred.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Pass the structure variable gateway->addr.ipv4 as 0 if not required to be set.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6GatewaySet(openapiClientHandle_t *client_handle,
                                              open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Delete IPv6 parameters set on a service port.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipv6Addr         @b{(input)}  IPv6 address and prefix length
* 
* @returns  OPEN_E_NONE       if the service port IPv6 prefix and prefix 
*                             length are deleted successfully.
* @returns  OPEN_E_FAIL       if unable remove the parameters or
*                             if invalid prefix is passed or 
*                             if parameters does not exist.
* @returns  OPEN_E_UNAVAIL    if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Set all the structure elements pointed by ipv6Addr to 0 to delete all
*          the IPv6 prefixes.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6PrefixDelete(openapiClientHandle_t *client_handle,
                                                open_inet_pfx_t *ipv6Addr);

/*********************************************************************
* @purpose  Get next service port IPv6 address and prefix length.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipv6Addr         @b{(inout)} IPv6 address and prefix length
* 
* @returns  OPEN_E_NONE      if the next IPv6 service port address and prefix 
*                            length are returned.
* @returns  OPEN_E_FAIL      if no more entries are found.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*                                              
* @notes   Set all the structure elements pointed by ipv6Addr to 0
*          to get the first address.    
*                                              
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6AddressesNextGet(openapiClientHandle_t *client_handle,
                                                    open_inet_pfx_t *ipv6Addr);

/*********************************************************************
* @purpose  Get the service port IPv6 gateway address.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    gateway          @b{(output)} IPv6 gateway address
* 
* @returns  OPEN_E_NONE      if IPv6 gateway address is returned.
* @returns  OPEN_E_FAIL      if unable to get IPv6 gateway address.
* @returns  OPEN_E_UNAVAIL   if IPv6 feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv6GatewayGet(openapiClientHandle_t *client_handle,
                                              open_inet_addr_t *gateway);

/*********************************************************************
* @purpose  Set the network port IPv4 protocol mode.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(input)}  Network port IPv4 protocol mode: 
*                                         none/dhcp/bootp
* 
* @returns  OPEN_E_NONE      if the network port IPv4 protocol mode is set successfully
* @returns  OPEN_E_FAIL      if service port is configured with dhcp/bootp 
*                            and trying to configure bootp on network port.
*                            Only one interface can run DHCP or Bootp.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv4ProtocolModeSet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t protoMode);

/*********************************************************************
* @purpose  Get the network port IPv4 protocol mode.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(output)} Network port IPv4 protocol mode: 
*                                         none/dhcp/bootp
* 
* @returns  OPEN_E_NONE      if the network port IPv4 protocol mode is returned.
* @returns  OPEN_E_FAIL      if invalid network port IPv4 protocol mode is returned, 
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @note
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiNetworkPortIPv4ProtocolModeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t *protoMode);

/*********************************************************************
* @purpose  Set the service port IPv4 protocol mode.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(input)}  Service port IPv4 protocol mode: 
*                                         none/dhcp/bootp
* 
* @returns  OPEN_E_NONE      if the service port IPv4 protocol mode is set successfully
* @returns  OPEN_E_FAIL      if network port is configured with dhcp/bootp 
*                            and trying to configure bootp on service port.
*                            Only one interface can run DHCP or Bootp.
* @returns  OPEN_E_UNAVAIL   if feature is unsupported.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv4ProtocolModeSet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t protoMode);

/*********************************************************************
* @purpose  Get the service port IPv4 protocol mode.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    protoMode        @b{(output)} Service protocol mode: 
*                                         none/dhcp/bootp
* 
* @returns  OPEN_E_NONE      if the service port IPv4 protocol mode is returned successfully
* @returns  OPEN_E_FAIL      if invalid service port IPv4 protocol mode is returned. 
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @note
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiServicePortIPv4ProtocolModeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_PROTOCOL_MODE_t *protoMode);

/*********************************************************************
* @purpose  Set the telnet server admin mode.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    telnetServerMode @b{(input)}  Telnet server admin mode
* 
* @returns  OPEN_E_NONE      if telnet server admin mode set successfully.
* @returns  OPEN_E_FAIL      if unable to set telnet server mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTelnetServerAdminModeSet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t telnetServerMode);

/*********************************************************************
* @purpose  Get the telnet server admin mode.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    telnetServerMode @b{(output)} Telnet server admin mode
* 
* @returns  OPEN_E_NONE      if telnet server admin mode is returned.
* @returns  OPEN_E_FAIL      is unable to get telnet server admin mode or 
*                            if invalid mode is returned.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTelnetServerAdminModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t *telnetServerMode);

/*********************************************************************
* @purpose  Set the transferred bytes.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    serialTimeout    @b{(input)}  Transferred Bytes
*
* @returns  OPEN_E_NONE      if transferred bytes is set successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTransferBytesCompletedSet(openapiClientHandle_t *client_handle,
                                              uint32_t bytes);

/*********************************************************************
* @purpose  Get the stack up time in seconds.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    sysUpTimeSec     @b{(output)} System up time (seconds)
*
* @returns  OPEN_E_NONE      if system up time is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @comments  Stack up time is the time since the stack performed a cold
*            restart. Stack up time does not reset on a warm restart.
* 
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemUpTimeGet(openapiClientHandle_t *client_handle,
                                    uint32_t *sysUpTimeSec);

/*********************************************************************
* @purpose  Get the stack up time in milliseconds.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    sysUpTimeMsec    @b{(output)} System up time (milliseconds)
*
* @returns  OPEN_E_NONE      if system up time is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSystemUpTimeMsecGet(openapiClientHandle_t *client_handle,
                                        uint32_t *sysUpTimeMsec);

/*********************************************************************
* @purpose  Get warm restart status.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    warmRestartStatus   @b{(output)} Warm restart status
* 
* @returns  OPEN_E_PARAM      If the passed params are invalid
* @returns  OPEN_E_NONE       If the OPEN call is successful
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiWarmRestartStatusGet(openapiClientHandle_t *client_handle,
                                         OPEN_BOOL_t *warmRestartStatus);

/*********************************************************************
* @purpose  Get System MAC Address
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    macAddress     @b{(output)} MAC address buffer
*
* @returns  OPEN_E_NONE      if MAC get operation is successful
* @returns  OPEN_E_PARAM     if invalid parameter or size is supplied 
*
* @notes    MAC Address buffer should be at least @ref OPEN_MAC_ADDR_LEN bytes long
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiSystemMacAddrGet(openapiClientHandle_t *client_handle, 
                                     open_buffdesc         *macAddress);

/*********************************************************************
* @purpose  Gets System Burned in MAC Address
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    mac_addresss     @b{(output)} MAC address buffer
*
* @returns  OPEN_E_NONE      if MAC get operation is successful
* @returns  OPEN_E_FAIL      if MAC get operation resulted in failure
* @returns  OPEN_E_PARAM     if invalid parameter or size is supplied 
*
* @notes    MAC Address buffer should be at least @ref OPEN_MAC_ADDR_LEN bytes long
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiSystemBurnedInMacAddrGet ( openapiClientHandle_t *client_handle,
                                               open_buffdesc         *mac_address );

/*********************************************************************
* @purpose  Get current system memory allocation information.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    memTotal         @b{(output)} Total amount of memory in the system.
* @param    memAlloc         @b{(output)} Amount of memory currently allocated.
* @param    memFree          @b{(output)} Amount of free memory available.
*
* @returns  OPEN_E_NONE      Operation succeeded.
* @returns  OPEN_E_FAIL      Unable to get data values.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @comments Provides current values of allocated and free memory
*           (system RAM) in kilobytes (KB) along with their sum total.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiSystemMemoryInfoGet(openapiClientHandle_t *client_handle,
                                        uint32_t *memTotal,
                                        uint32_t *memAlloc,
                                        uint32_t *memFree);

/*********************************************************************
* @purpose  Saves the configuration to non-volatile storage
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
*
* @returns  OPEN_E_NONE      if save operation is successful
* @returns  OPEN_E_FAIL      if save operation resulted in failure 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiSaveConfigurationSet(openapiClientHandle_t *client_handle);

/*********************************************************************
 * @purpose  Clear the switch configuration to factory defaults
 *
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
*
* @returns  OPEN_E_NONE      if clear operation is successful
* @returns  OPEN_E_FAIL      if clear operation resulted in failure 
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes  You can reset the configuration to factory default values without
* @notes  powering off the switch.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiResetConfigurationSet ( openapiClientHandle_t *client_handle );

/*********************************************************************
* @purpose  Resets the switch
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
*
* @returns  OPEN_E_NONE      if reset is issued successfully
* @returns  OPEN_E_FAIL      if reset operation has failed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiSystemReload(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the manufacturer.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    unit             @b{(input)}  Unit number
* @param    manufacturer     @b{(output)} Pointer to manufacturer
*           -                             Buffer allocated size must be at least
*           -                             @ref OPEN_MANUFACTURER_SIZE
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiManufacturerGet(openapiClientHandle_t *client_handle,
                                    uint32_t              unit,
                                    open_buffdesc         *manufacturer);

/*********************************************************************
* @purpose  Get the time zone minutes parameter
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    minutes          @b{(output)} Time zone offset in minutes
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiTimeZoneOffsetGet(openapiClientHandle_t *client_handle,
                                      uint32_t              *timeZoneOffsetMin);

/*********************************************************************
* @purpose  Get the operating system
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    operSys          @b{(output)} Pointer to operating system string storage
*           -                             Buffer allocated size must be at least
*           -                             @ref OPEN_OS_VERSION_SIZE
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiOperSysGet(openapiClientHandle_t *client_handle,
                               open_buffdesc         *operSys);

/*********************************************************************
* @purpose  Get UID of the System 
*           Uid of the system is Datapath ID of the ofdb.
*           If OFDB component is not enabled, we will return
*           service port Mac address prefixed by 2 bytes (default value 0)
*
*
* @param[in]     client_handle
* @param[out]    uid -- UID of the system 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiUidGet(openapiClientHandle_t * client_handle,open_buffdesc *uid);

/*********************************************************************
* @purpose  Get the maintenance level of the system
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    maintenanceLevel @b{(output)} Pointer to maintenance level string storage
*           -                             Buffer allocated size must be at least
*           -                             @ref OPEN_SYSTEM_PARAM_SIZE
*
* @returns  OPEN_E_NONE      if operation is successful
* @returns  OPEN_E_FAIL      if operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiMaintenanceLevelGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc         *maintenanceLevel);

/*********************************************************************
* @purpose  Sets resource tracking flag for an OpEN API client.
*
* @param    client_handle       @b{(input)} Storage to contain handle information.
* @param    trackingControl     @b{(input)} Resource tracking control mode.
* 
* @returns  OPEN_E_NONE         Successfully set resource tracking.
* @returns  OPEN_E_ERROR        Unable to set resource tracking.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    When tracking is enabled, all the HW resources allocated by
*           a given OpEN API client will be tracked and those resources
*           will be freed when OpEN API client ceases to exist.
*
* @supported-in-version OpEN API Version: 1.15
* 
* @end
*********************************************************************/
open_error_t openapiClientResourceTrackingSet(openapiClientHandle_t *client_handle,
                                              OPEN_CONTROL_t trackingControl);

/*********************************************************************
* @purpose  Get license status for component.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    featureName      @b{(input)}  Feature name
* @param    enabled          @b{(output)} Feature enabled status
* 
* @returns  OPEN_E_NONE      Successfully fetched feature enabled status.
* @returns  OPEN_E_ERROR     Unable to fetch feature enabled status.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_UNAVAIL   Fetching feature status functionality not suppported.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.18
* 
* @end
*********************************************************************/
open_error_t openapiLicenseFeatureStatusGet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *featureName,
                                            OPEN_BOOL_t *enabled);
#endif
