/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename openapi_tcam.h
*
* @purpose TCAM API: Provide an interface for managing policy in hardware.
*
* @component Policy Manager (TCAM API)
*
* @create   09/01/2012
*
* @author   
* @end
*
**********************************************************************/
#ifndef OPENAPI_TCAM_H_INCLUDED
#define OPENAPI_TCAM_H_INCLUDED
 
/**
 * The typical sequence is:<br>
 *    openapiTcamApiInit() to initialize the TCAM API<br>
 *    repeat openapiTcamPolicyCreate() to create a policy and get a handle for
 *                                     each policy<br>
 *    repeat openapiTcamPolicyIntfAdd() to add the policy to a set of
 *                                       interfaces for each policy<br>
 *    repeat openapiTcamPolicyIntfRem() to remove the policy from a set of
 *                                       interfaces for each policy<br>
 *    repeat openapiTcamPolicyDestroy() to destroy the policy for each
 *                                      policy<br>
 *    openapiTcamApiFini() to uninitialize the TCAM API<br>
 */
 
#include "openapi_common.h"

/** The length of a MAC address in bytes */
#define TCAM_MAC_ADDR_LEN 6

/** The current TCAM API major version */
#define OPEN_TCAM_API_CURR_VER_MAJOR 0
/** The current TCAM API minor version */
#define OPEN_TCAM_API_CURR_VER_MINOR 1

/** Invalid Policy ID */
#define OPEN_TCAM_API_POLICYID_INVALID 0

/** Policy Prority */
#define OPEN_TCAM_POLICY_RULE_PRIORITY_LOWEST  0
#define OPEN_TCAM_POLICY_RULE_PRIORITY_DEFAULT 1

/** Enumeration of debug flags */
typedef enum
{
 TCAM_API_DEBUG_LEVEL_NONE  = 0, /*!< No debug                */
 TCAM_API_DEBUG_LEVEL_INFO  = 1, /*!< Informational debugging */
 TCAM_API_DEBUG_LEVEL_ERROR = 2, /*!< Error messages          */
 TCAM_API_DEBUG_LEVEL_DEBUG = 3, /*!< Debug messages          */
 TCAM_API_DEBUG_LEVEL_LAST  = 4  /*!< End value               */
} TCAM_API_DEBUG_FLAG_t;
 
/** Debug macro for logging messages */
#define TCAM_API_DEBUG(dbgLvl, args...) {\
    if (tcamApiDebugFlag >= dbgLvl) \
    {\
      SYSAPI_PRINTF(SYSAPI_LOGGING_ALWAYS,args);\
    }\
} 
 
/** Transmit/Receive maximum frame size */
#define OPEN_TXRX_MAX_FRAME_SIZE         9216
/** External agent transmit */
#define OPEN_TCAM_EXT_AGENT_PKT_TRANSMIT (0x1)
/** External agent receive */
#define OPEN_TCAM_EXT_AGENT_PKT_RECEIVE  (0x2)


typedef struct
{
  uint32_t   version; /* Desc struct version number */
  uint32_t   pktLength; /* Length of the packet */
  uint32_t   descLen; /* Length of the packet meta-data */

  uint32_t   rxIfNum; /* Interface number on which packet was received */

  /* Add new meta-data at the end */
  uint32_t destIntIfNum;   /* Destination intIfNum */

  /** Parameter to specify whether the packet ingressing on physical port is 
   *  having VLAN tag or not.
   *  0 - packet is tagged, 1 - packet is untagged
   */
  uint32_t rxUntagged;
} open_sysExtAgentPktRxDesc_t;
/** Structure to define the action that should be taken.
 *  Only some fields will be relevant for the given rewrite mask.
 */
typedef struct
{
  uint32_t                redirIntfNum;            /*!< Redirect interface */
  unsigned char           srcAddr[TCAM_MAC_ADDR_LEN]; /*!< Replacement MAC SA */
  unsigned char           dstMac[TCAM_MAC_ADDR_LEN];  /*!< Replacement MAC DA */
  uint32_t                etherType;                  /*!< Replacement Etype  */
  uint16_t                vlanId;
  unsigned char           vlanPrio;
  unsigned char           dscp;

} OPEN_TCAM_ACTION_t;


/** No action */
#define OPEN_TCAM_ACTION_NONE               0    
/** Drop the frame */
#define OPEN_TCAM_ACTION_DROP               1 /* ((uint32_t)(1 << 0)) */
/** Redirect the frame to another port */
#define OPEN_TCAM_ACTION_REDIRECT           2 /* ((uint32_t)(1 << 1)) */
/** Rewrite the outer VLAN field */
#define OPEN_TCAM_ACTION_OUTERVLAN          4 /* ((uint32_t)(1 << 2)) */
/** Rewrite the source MAC address */
#define OPEN_TCAM_ACTION_SRCMAC_NEW         8 /* ((uint32_t)(1 << 3)) */
/** Rewrite the destination MAC address */
#define OPEN_TCAM_ACTION_DSTMAC_NEW         16 /* ((uint32_t)(1 << 4)) */
/** Rewrite the priority */
#define OPEN_TCAM_ACTION_USERPRIO           32 /* ((uint32_t)(1 << 5)) */
/** Rewrite the outer TPID */
#define OPEN_TCAM_ACTION_OUTERTPID_NEW      64 /* ((uint32_t)(1 << 6)) */
/** redirect packet to CPU */
#define OPEN_TCAM_ACTION_REDIRECT_CPU       128 /* ((uint32_t)(1 << 7)) */

/** Packet Trace rule number for TCAM */
#define OPEN_PT_MATCH_RULE_NUM              61

#define OPEN_CPU_RATE_LIMIT                 5   
#define OPEN_CPU_RATE_LIMIT_BURST           5 

/** Type for Policy actions */
typedef uint32_t OPEN_TCAM_POLICY_ACTION_t;
/** Type for Policy IDs */
typedef uint32_t OPEN_TCAM_API_POLICYID_t;

/** Enumeration to describe where the policy should apply */
typedef enum
{
  OPEN_TCAM_POLICY_TYPE_OPENFLOW    = 1, /*!< OpenFlow policy         */
  OPEN_TCAM_POLICY_TYPE_GEN         = 2, /*!< Gen (IPv6) policy       */
  OPEN_TCAM_POLICY_TYPE_EGRESS      = 3, /*!< Egress policy           */
  OPEN_TCAM_POLICY_TYPE_SYSTEM_PORT = 4, /*!< System Port policy      */ 
  OPEN_TCAM_POLICY_TYPE_PORT        = 5, /*!< Port policy      */ 
  OPEN_TCAM_POLICY_TYPE_LAST        = 6  /*!< End value               */
}OPEN_TCAM_POLICY_TYPE_t;

/**Version information for the TCAM API */ 
typedef struct
{
  uint32_t versionMajor;               /*!< Major version number    */
  uint32_t versionMinor;               /*!< Minor version number    */
} open_tcam_version_t;


/** Header that goes on the front of each policy */
/* User needs to populate the header witht the version of the API 
    it expects to interact with*/
typedef struct tcam_api_policyHeader_s
{
  open_tcam_version_t versionInfo;
  
  uint32_t headerLen;                  /*!< Hdr length in bytes     */
}tcam_api_policyHeader_t;

/* Metering Definitions */
typedef enum
{
  OPEN_METER_COLOR_BLIND,
  OPEN_METER_COLOR_AWARE
} OPEN_METER_COLOR_MODE_t;

typedef enum
{
  OPEN_METER_BYTES,
  OPEN_METER_PACKETS
} OPEN_METER_MODE_t;

typedef struct
{
  uint64_t in_prof;   /* green/yellow */
  uint64_t out_prof;  /* red */
} OPEN_METER_STATS_t;

typedef struct
{
  uint32_t                cir, cbs;
  uint32_t                pir, pbs;
  OPEN_METER_COLOR_MODE_t colorMode;
  OPEN_METER_MODE_t       meterMode;
} OPEN_METER_ENTRY_t;

#define  OPEN_COUNT_PACKETS  0x1   /* matching packet counter */
#define  OPEN_COUNT_BYTES    0x2   /* matching byte counter   */

typedef struct
{
  uint64_t byteCount;
  uint64_t pktCount;
} OPEN_COUNTER_STATS_t;

/** Structure that contains the full TCAM API policy: headers, match
 *  values and action.
 */
OPEN_PACKED_STRUCT tcamApiPolicy_s
{
  /* Policy attributes*/
  tcam_api_policyHeader_t      policyHeader;
      /*!< Policy header.  This must be filled in               */
  OPEN_TCAM_POLICY_TYPE_t      policyType;
      /*!< Policy type.  Where the policy should be applied     */
  uint32_t                     policyPrio;
      /*!< Priority of the policy compared to other policies    */

  /* Classification attributes */
  uint64_t                     ruleNum;
  uint32_t                     rulePrio;
  unsigned char                matchDstMac[TCAM_MAC_ADDR_LEN];
  unsigned char                matchSrcMac[TCAM_MAC_ADDR_LEN];
  uint16_t                     matchVlanVal;
  unsigned char                matchVlanPrio;
  uint16_t                     matchEthType;
  uint32_t                     matchDstIp;
  uint32_t                     matchDstIpMask;
  uint32_t                     matchSrcIp;
  uint32_t                     matchSrcIpMask;
  uint32_t                     matchIpProto;
  uint32_t                     matchIpProtoMask;
  uint32_t                     matchL4Sport;
  uint32_t                     matchL4SportMask;
  uint32_t                     matchL4Dport;
  uint32_t                     matchL4DportMask;
  uint32_t                     matchIpDscp;
  unsigned char                matchIpDscpMask;
  uint16_t                     matchSrcPortIcmpType;
  uint16_t                     matchSrcPortIcmpTypeMask;
  uint16_t                     matchDstPortIcmpCode;
  uint16_t                     matchDstPortIcmpCodeMask;
  open_in6_addr_t              matchSrcIpv6Addr;
  open_in6_addr_t              matchSrcIpv6Mask;
  open_in6_addr_t              matchDstIpv6Addr;
  open_in6_addr_t              matchDstIpv6Mask;

  /* Actions*/
  OPEN_TCAM_POLICY_ACTION_t         actionType;
      /*!< Bitmap of the actions that should be applied         */
  OPEN_TCAM_ACTION_t                actionVal;
      /*!< Values to be used by the actions                     */

  /* Policy Metering information */
  OPEN_METER_ENTRY_t           meterInfo;
  uint8_t                      meterPresent;  /* 1 meterInfo is valid, 0 not valid */
} open_tcamApiPolicy_t;

/** Hardware TCAM group usage information.
*/
typedef struct open_tcamHardwareGroupUsage_s
{
    uint32_t in_use_rules; /**< Number of rules currently used in the hardware group. */
    uint32_t total_rules; /**< Total number of rules that can be added to the hardware group. */

    uint32_t in_use_counters; /**< Number of counters currently used in the hardware group. */
    uint32_t total_counters; /**< Total counters available for the hardware group. */

    uint32_t in_use_meters; /**< Number of meters currently used in the hardware group. */
    uint32_t total_meters; /**< Total meters available for the hardware group. */

    uint32_t slice_width_physical;   /**< Slice width of group based on physical slices. */
    uint32_t intraslice_mode_enable; /**< Indicates if intraslice mode is enabled. */
    uint32_t natural_depth;          /**< Number of rules when slice depth is one. */
} open_tcamHardwareGroupUsage_t;

/** Maximum number of characters in the policy name.
*/
#define OPEN_TCAM_POLICY_NAME_SIZE 64

#define OPEN_TCAM_STAGE_LOOKUP   1  /**< Hardware VCAP */
#define OPEN_TCAM_STAGE_INGRESS  2  /**< Hardware ICAP */
#define OPEN_TCAM_STAGE_EGRESS   3  /**< Hardware ECAP */

/** Hardware TCAM Policy Usage Information.
*/ 
typedef struct open_tcamHardwarePolicyUsage_s
{
  uint8_t policy_name [OPEN_TCAM_POLICY_NAME_SIZE]; /**< Policy Name. */
  uint32_t num_policy_rules;  /**< Number of rules in this policy. */
  uint32_t hardware_stage; /**< In which TCAM the policy is installed. */
  uint32_t hardware_group; /**< The group in the TCAM to which this policy belongs. */
  uint32_t in_use_rules; /**< Number of rules currently used in the hardware group. */
  uint32_t total_rules; /**< Total number of rules that can be added to the hardware group. */
} open_tcamHardwarePolicyUsage_t;

/**************************************************************************
* @purpose  Initialize the TCAM API
*
* @param   clientHandle    {(input)}  OpEN client handle
*
* @returns  OPEN_E_NONE: API initialized successfully
* @returns  OPEN_E_FAIL: Error encountered in TCAM API initialization
*
*   
* @comments  none
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamApiInit(openapiClientHandle_t *clientHandle);

/**************************************************************************
* @purpose  Uninitialize the TCAM API and return its resources 
*           
*
* @param   clientHandle    {(input)}  OpEN client handle

* @returns  OPEN_E_NONE: Unintialize succeeded and resources returned
* @returns  OPEN_E_FAIL: Fini failed
*
* @comments none
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamApiFini(openapiClientHandle_t *clientHandle);

/**************************************************************************
* @purpose  Create a policy in the hardware directly
*
* @param   clientHandle {(input)}  OpEN client handle
* @param   policyInfo   {(input)}  Information about the policy to be created 
* @param   handle       {(output)} The handle to be used for subsequent
*                                  operations on this policy
*
* @returns  OPEN_E_NONE: Policy created successfully
* @returns  OPEN_E_FAIL: Policy creation failure
*
*   
* @comments From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyCreate(openapiClientHandle_t *clientHandle,
                                     open_tcamApiPolicy_t policyInfo,
                                     OPEN_TCAM_API_POLICYID_t *handle);

/**************************************************************************
* @purpose  Delete the policy created
*

*
* @param   clientHandle    {(input)}  OpEN client handle
* @param   policyHandle    {(input)}  Handle associated with the policy
*
* @returns  OPEN_E_NONE: Policy deleted successfully
* @returns  OPEN_E_FAIL: Policy deletion failure
*
* @comments From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyDestroy(openapiClientHandle_t *clientHandle,
                                      OPEN_TCAM_API_POLICYID_t *policyHandle);

/**************************************************************************
* @purpose  Add the TCAM policy to the list of interfaces provided.  This
*           should be done after the policy handle has been created
*           with openapiTcamPolicyCreate().
*
*
* @param   clientHandle  {(input)} OpEN client handle
* @param   policyHandle  {(input)} Handle associated with the policy
* @param   intf          {(input)} Interface to be added
*
* @returns  OPEN_E_NONE: Policy added to all the interfaces listed
* @returns  OPEN_E_FAIL: Policy could not be added to interfaces
*
* @comments The interfaces should be listed as separated by a comma (,)
*           or hyphen (-) for contiguous range 
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyIntfAdd(openapiClientHandle_t *clientHandle,
                                       OPEN_TCAM_API_POLICYID_t *policyHandle, 
                                       uint32_t intf);

/**************************************************************************
* @purpose  Remove TCAM policy from the list of interfaces provided
*
*
* @param   clientHandle {(input)}  OpEN client handle
* @param   policyHandle {(input)} Handle associated with the policy
* @param   intf         {(input)} Interface to be removed
*
* @returns  OPEN_E_NONE: Policy removed from all the interfaces listed
* @returns  OPEN_E_FAIL: Policy could not be removed from at least one
*                       of the interfaces.
*
* @comments The interfaces should be listed as separated by a comma (,)
*           or hyphen (-) for contiguous range 
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyIntfRem(openapiClientHandle_t *clientHandle,
                                       OPEN_TCAM_API_POLICYID_t *policyHandle, 
                                       uint32_t intf);

/**************************************************************************
* @purpose  Get TCAM policy information from the policy handle provided
*
*
* @param   clientHandle    {(input)}  OpEN client handle
* @param   policyHandle {(input)}  Handle associated with the policy
* @param    policyInfo   {(output)} Information about the policy associated with the handle 
*
* @returns  OPEN_E_NONE: Policy found
* @returns  OPEN_E_FAIL: Policy could not be found 
*
* @comments None.
*          
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyGet(openapiClientHandle_t *clientHandle,
                                  OPEN_TCAM_API_POLICYID_t *policyHandle, 
                                  open_tcamApiPolicy_t *policyInfo);

/**************************************************************************
* @purpose  Get the first TCAM policy information 
*
*
* @param   clientHandle    {(input)}  OpEN client handle
* @param   policyInfo       {(output)} Information about the next policy 
* @param   nextPolicyHandle {(output)}  Handle associated with the retrieved policy
*
* @returns  OPEN_E_NONE: Policy found
* @returns  OPEN_E_FAIL: Policy could not be found 
*
* @comments This API returns the first policy created by the TCAM API.
*           There might be other policies created outside the TCAM API
*           but those won't be included in these call.
*          
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyGetFirst(openapiClientHandle_t *clientHandle,
                                  open_tcamApiPolicy_t *policyInfo,
                                      OPEN_TCAM_API_POLICYID_t *nextPolicyHandle);

/**************************************************************************
* @purpose  Get the next TCAM policy information from the policy handle provided
*
*
* @param   clientHandle     {(input)}  OpEN client handle
* @param   policyHandle     {(input)}  Handle associated with the policy
* @param   policyInfo       {(output)} Information about the next policy 
* @param   nextPolicyHandle {(output)} Handle associated with the retrieved policy
*
* @returns  OPEN_E_NONE: Policy found
* @returns  OPEN_E_FAIL: Policy could not be found 
*
* @comments This API returns the first policy created by the TCAM API.
*           There might be other policies created outside the TCAM API
*           but those won't be included in these call.
*          
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyGetNext(openapiClientHandle_t *clientHandle,
                                  OPEN_TCAM_API_POLICYID_t *policyHandle,  
                                  open_tcamApiPolicy_t *policyInfo,
                                      OPEN_TCAM_API_POLICYID_t *nextPolicyHandle); 

/**************************************************************************
* @purpose  Delete all the policies created by TCAM API
*
*
* @param   clientHandle    {(input)}  OpEN client handle
*
* @returns  OPEN_E_NONE: Policy found
* @returns  OPEN_E_FAIL: Policy could not be found 
*
*
* @comments 
*          
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyDeleteAll(openapiClientHandle_t *clientHandle);

/**************************************************************************
* @purpose  Get the version supported by the current API
*
*
* @param   clientHandle    {(input)}  OpEN client handle
* @param   version         {(output)} version structure with
*                                     version information populated
*
* @returns  OPEN_E_NONE: Version returned.
* @returns  OPEN_E_PARAM: Insufficient string size provided. 
*
* @comments None.
*          
*   
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiTcamApiVersionGet(openapiClientHandle_t *clientHandle,
                                  open_tcam_version_t *version);

/**************************************************************************
* @purpose  Register an External Packet Transmit/Receive Agent 
*
* @param    agentId   {(input)} Unique Agent Id. The acceptable range of Agent Ids
*                               can be retrieved using usmDbExtAgentIdRangeGet API.
* @param    agentName {(input)} Name of the receive agent.
*
* @param    flags    {(input)}  OPEN_TCAM_EXT_AGENT_PKT_TRANSMIT: Register as external 
*                                                          transmit agent
*                               OPEN_TCAM_EXT_AGENT_PKT_RECEIVE: Register as external 
*                                                         receive agent 
*
* @returns  OPEN_E_NONE: Agent registered successfully
*           OPEN_E_EXISTS: Agent with the given id is already 
*                                  registered
*           OPEN_E_FAIL: Other failures
*
* @comments  Only OPEN_TCAM_EXT_AGENT_PKT_RECEIVE flag is supported
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiExtAgentPktRegister( uint32_t    agentId, 
                                 open_buffdesc *agentName,
                                 uint32_t    flags);

/**************************************************************************
* @purpose  Unregister an External Packet Transmit/Receive Agent 
*
* @param    agentId   {(input)}  Id to uniquely identify agent
*
* @returns  OPEN_E_NONE: Agent unregistered successfully
*           OPEN_E_FAIL: Invalid agentId passed
*
* @comments 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*************************************************************************/
open_error_t openapiExtAgentPktUnregister( uint32_t agentId);

/*********************************************************************
* @purpose  Get the policy hardware usage statistics for the next
*           policy after the specified policy.
*
* @param    client_handle    @b{(input)}  OpEN client handle
* @param    unit             @b{(input/output)}  On stackable platforms
*                               specifies the stack unit ID. Always returns 1 on
*                               non-stacking platforms.
* @param    asic_id          @b{(input/output)}  On multi-ASIC units 
*                               specifies the ASIC number on a unit. Always
*                               returns 0 on units with one ASIC.
* @param    policy_id        @b{(input/output)}  Policy ID to search for 
* @param    tcam_usage       @b{(output)} Hardware policy usage info.
*
* @returns  OPEN_E_NONE  Policy is found.
* @returns  OPEN_E_NOT_FOUND  Next policy is not found.
*
* @notes   In order to get the first policy set unit/asic_id/policyId to 0 
*
* @supported-in-version OpEN API Version: 1.13
*   
* @end
*********************************************************************/
open_error_t openapiTcamHardwarePolicyUsageGetNext(openapiClientHandle_t *client_handle,
                                                   uint32_t *unit, 
                                                   uint32_t *asic_id,
                                                   uint32_t *policy_id, 
                                                   open_tcamHardwarePolicyUsage_t *tcam_usage);

/**************************************************************************
* @purpose  Get the hardware usage statistics for the next
*           hardware group after the specified hardware group.
*
*
* @param    client_handle  @b{(input)}  OpEN client handle
* @param    unit           @b{(input/output)}  On stackable platforms
*                               specifies the stack unit ID. Always returns 1 on
*                               non-stacking platforms.
* @param    asic_id        @b{(input/output)}  On multi-ASIC units 
*                               specifies the ASIC number on a unit. Always
*                               returns 0 on units with one ASIC.
* @param    stage          @b{(input/output)}  TCAM Stage
* @param    group_id       @b{(input/output)}  Group ID to search for 
* @param    group_usage    @b{(output)} Hardware group usage info.
*
* @returns  OPEN_E_NONE: Hardware Group Found.
* @returns  OPEN_E_NOT_FOUND: Hardware Group was not found.
*
* @comments In order to get the first group set unit/asic_id/stage/group_id
*           to 0 
* 
* @supported-in-version OpEN API Version: 1.13
* 
* @end
*************************************************************************/
open_error_t openapiTcamHardwareGroupUsageGetNext(openapiClientHandle_t *client_handle,
                                                  uint32_t *unit, 
                                                  uint32_t *asic_id,
                                                  uint32_t  *stage, 
                                                  uint32_t  *group_id,
                                                  open_tcamHardwareGroupUsage_t *group_usage);

/**************************************************************************
* @purpose  Add the TCAM policy to all interfaces.
*
* @param   clientHandle  {(input)} OpEN client handle
* @param   policyHandle  {(input)} Handle associated with the policy
*
* @returns  OPEN_E_NONE: Policy added to all the interfaces
* @returns  OPEN_E_FAIL: Policy could not be added to interfaces
*
* @comments This should be done after the policy handle has been
*           created with openapiTcamPolicyCreate().
*   
* @supported-in-version OpEN API Version: 1.14
*
* @end
*************************************************************************/
open_error_t openapiTcamPolicyIntfAllAdd(openapiClientHandle_t *clientHandle,
                                         OPEN_TCAM_API_POLICYID_t *policyHandle); 

/**************************************************************************
* @purpose  Get TCAM policy statistics for the policy handle provided
*
*
* @param    clientHandle {(input)}  OpEN client handle
* @param    policyHandle {(input)}  Handle associated with the policy
* @param    inProf       {(output)} in-profile metering stats of policy
* @param    outProf      {(output)} out-profile metering stats of policy
*
* @returns  OPEN_E_NONE: Policy stats retrieved successfully
* @returns  OPEN_E_FAIL: Policy stats retrieval failed
*
* @comments None.
*          
* @supported-in-version OpEN API Version: 1.14
*   
* @end
*************************************************************************/
open_error_t openapiTcamPolicyMeteringStatsGet(openapiClientHandle_t *clientHandle,
                                               OPEN_TCAM_API_POLICYID_t *policyHandle,
                                               uint64_t *inProf, uint64_t *outProf);

/**************************************************************************
* @purpose  Clear TCAM policy statistics for the policy handle provided
*
*
* @param    clientHandle {(input)}  OpEN client handle
* @param    policyHandle {(input)}  Handle associated with the policy
*
* @returns  OPEN_E_NONE: Policy stats cleared successfully
* @returns  OPEN_E_FAIL: Policy stats clear failed
*
* @comments None.
*          
* @supported-in-version OpEN API Version: 1.14
*   
* @end
*************************************************************************/
open_error_t openapiTcamPolicyStatsClear(openapiClientHandle_t *clientHandle,
                                         OPEN_TCAM_API_POLICYID_t *policyHandle); 

#endif /* OPENAPI_TCAM_H_INCLUDED*/
