/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  boxs_example.c
*
* @purpose   Box Services APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    11/12/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_boxs.h"

void fiberPortsOpticsFaultStatusGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t localFaultStatus = 0;
  uint32_t remoteFaultStatus = 0;

  result = openapiFiberPortsOpticsFaultStatusGet(clientHandle, ifNum, &localFaultStatus, &remoteFaultStatus);

  if (result == OPEN_E_NONE)
  {
    /* localFaultStatus/remoteFaultStatus equal to 0 indicates no fault */
    printf("\n\nFiber port fault status for port %d : %s", ifNum,
           (((localFaultStatus != 0) && (remoteFaultStatus != 0)) ?
            "Local & Remote Faults\n" : ((localFaultStatus != 0) ?
             "Local Fault\n" : ((remoteFaultStatus != 0) ? "Remote Fault\n" : "No Fault\n"))));
  }
  else
  {
    printf("Bad return code trying to get fiber ports optics fault status. (result = %d)\n", result);
  }
}

void fiberPortsOpticsGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t temperature = 0;
  uint32_t voltage = 0;
  uint32_t current = 0;
  OPEN_BOOL_t txFault = 0;
  OPEN_BOOL_t los = 0;
  int32_t  tNotSupported = (int32_t) (~0);
  uint32_t vcNotSupported = (uint32_t) (~0);
  uint64_t pNotSupported = (10 * log10((uint64_t)((uint32_t)(~0))/10000));
  uint64_t powerIn;
  uint64_t powerOut;

  result = openapiFiberPortsOpticsGet(clientHandle, ifNum, &temperature, &voltage, &current,
                                      &powerIn, &powerOut, &txFault, &los);

  if (result == OPEN_E_NONE)
  {
    printf("%d    ", ifNum);

    /* Temperature */
    if (temperature == tNotSupported)
    {
      printf("    N/A");
    }
    else
    {
      printf("    %4d.%1.1d", temperature/1000, (abs(temperature)%1000)/100);
    }

    /* Voltage */
    if (voltage == vcNotSupported)
    {
      printf("    N/A");
    }
    else
    {
      printf("    %1d.%3.3d", voltage / 1000, voltage % 1000);
    }

    /* Current */
    if (current == vcNotSupported)
    {
      printf("    N/A");
    }
    else
    {
      printf("    %5d.%1.1d", current/1000, (current%1000)/100);
    }

    /* Power Out*/
    if (powerOut == pNotSupported)
    {
      printf("    N/A");
    }
    else
    {
      printf("  %lld ", powerOut);
    }

    /* Power In*/
    if (powerIn == pNotSupported)
    {
      printf("    N/A");
    }
    else
    {
      printf("  %lld ", powerIn);
    }

    /* Tx Fault */
    printf("    %s     %s\n", ((OPEN_TRUE == txFault) ? "Yes":"No"), ((OPEN_TRUE == los) ? "Yes":"No"));
  }
  else if (result == OPEN_E_NOT_FOUND)
  {
    printf("%d         N/A          N/A      N/A             N/A            N/A         N/A      N/A\n", ifNum);
  }
  else
  {
    printf("Bad return code trying to get fiber ports optics. (result = %d)\n", result);
  }
}

/******************************************************************
  @brief  Print SFP info such as voltage, temperature, current etc
*******************************************************************/
void printSFPModuleInfo(openapiClientHandle_t *clientHandle)
{
  uint32_t ifNum = 0;
  OPEN_SFP_MODULE_TYPE_t moduleType;
  OPEN_LINK_STATE_t linkState;
  open_error_t result;

  printf("\nPort      Temperature  Voltage  Current     Output Power    Input Power   TX Fault    LOS\n");
  if ((result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &ifNum)) == OPEN_E_NONE)
  {
    do
    {
      if ((result = openapiIfLinkStateGet(clientHandle, ifNum, &linkState)) != OPEN_E_NONE)
      {
        printf("Bad Error code while getting link state(result = %d)\n", result);
      }
      if (linkState == OPEN_LINK_DOWN)
      {
        continue;
      }

      if ((result = openapiBoxsSfpTypeGet(clientHandle, ifNum, &moduleType)) != OPEN_E_NONE)
      {
        printf("Failed to get module type(result = %d)\n", result);
        break;
      }

      if ((linkState == OPEN_LINK_UP) && (moduleType != OPEN_MODULE_NONE))
      {
        /* Get fiber-ports optical-transceiver data */
        fiberPortsOpticsGet(clientHandle, ifNum);

        /* Get fiber-ports optical-transreceiver fault status data */
        fiberPortsOpticsFaultStatusGet(clientHandle, ifNum);
      }
    } while (openapiIfNextGet(clientHandle, OPEN_INTF_TYPE_PHY, ifNum, &ifNum) == OPEN_E_NONE);
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         Box Services OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/

int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  uint32_t i, numFans, fanSpeed, fanDuty, numPowSuplies, numOfTempSemsors, temperature;
  uint32_t tempSensorDescSize, powSupplyDescSize, fanDescSize;
  uint32_t genTemp, genFanSpeed, genFanDuty, genTempRangeMin, genTempRangeMax;
  OPEN_BOXS_ITEM_STATE_t fanState, psState;
  OPEN_BOXS_TEMPSENSOR_STATE_t state;
  OPEN_BOXS_ITEM_TYPE_t itemType;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc cpld_version;
  char cpld_version_string[OPEN_SYSTEM_PARAM_SIZE+1];
  char buf[128];
  open_buffdesc descrip;
  open_buffdesc vendorNameDesc, serialNumberDesc, partNumberDesc, vendorRevDesc, complianceDesc;
  char *sfpBuf, *str;
  uint32_t vendorNameSize, serialNumberSize, partNumberSize, vendorRevSize, complianceSize;
  uint32_t linkLength50um, linkLength62_5um, nominalSigRate;
  uint32_t ifNum = 0;
  OPEN_SFP_MODULE_TYPE_t moduleType;
  OPEN_LINK_STATE_t linkState;
  uint32_t headerPrinted = 0;

  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("boxs_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n",
            result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0,
	       "Starting box services API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) ==
      OPEN_E_NONE)
    printf ("Network OS version = %s\n", switch_os_revision_string);
  else
    printf ("Network OS version retrieve error\n");

  printf ("\n");
  cpld_version.pstart = cpld_version_string;
  cpld_version.size = sizeof (cpld_version_string);
  result = openapiCpldVersionGet (&clientHandle, &cpld_version);
  if (OPEN_E_NONE == result)
    printf ("CPLD version = %s\n", cpld_version_string);
  else
    printf ("Unable to get CPLD version. (result = %d)\n", result);

  printf ("\n");

  /* exercise various box services OPEN API functions */
  do
  { 
    if ((result = openapiBoxsGeneralTempGet(&clientHandle, &genTemp)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get general temperature in the unit. (result = %d)\n",result);
      break;
    }
    else
    {
     printf("General temperature in the unit : %d\n", genTemp);
    }

    if ((result = openapiBoxsTempRangeGet(&clientHandle, &genTempRangeMin, &genTempRangeMax)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get allowed temperature range in the unit. (result = %d)\n",result);
      break;
    }
    else
    {
     printf("Allowed temperature range in the unit : %d-%d\n", genTempRangeMin, genTempRangeMax);
    }

    if ((result = openapiBoxsiGeneralFanSpeedGet(&clientHandle, &genFanSpeed)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get general fan speed in the unit. (result = %d)\n",result);
      break;
    }
    else
    {
     printf("General fan speed in the unit : %d\n", genFanSpeed);
    }

    if ((result = openapiBoxsiGeneralFanDutyLevelGet(&clientHandle, &genFanDuty)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get general fan duty-level in the unit. (result = %d)\n",result);
      break;
    }
    else
    {
     printf("General fan duty-level in the unit : %d\n", genFanDuty);
    }

    if ((result = openapiBoxsNumOfFansGet(&clientHandle, &numFans)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get number of FANs in the unit. (result = %d)\n",result);
      break;
    }
    else
    {
     printf("Number of FANs in the unit : %d\n",numFans);
    }

    if ((result = openapiBoxsFanDescripSizeGet(&clientHandle, &fanDescSize)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get the fan description size. (result = %d)\n",result);
      break;
    }
    else
    {
      printf("Description size of fan in the unit %d\n", fanDescSize);
    }

    str = (char *)malloc(fanDescSize);
    if (NULL == str)
    {
      printf("Error: Could not allocate memory.\n");
      break;
    }
    for (i = 0; i< numFans; i++)
    {
      memset (str, 0, fanDescSize);
      descrip.pstart = str;
      descrip.size = fanDescSize;
      if ((result = openapiBoxsFanDescripGet(&clientHandle, i, &descrip)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to description of fan %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        printf("Fan %d description : %s\n",(i + 1),str);
      }
    }
    free(str);

    for (i = 0; i< numFans; i++)
    {
      if ((result = openapiBoxsFanSpeedGet(&clientHandle, i, &fanSpeed)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get Speed of FAN %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        printf("FAN number %d speed %d\n",(i + 1),fanSpeed);
      }
    }

    for (i = 0; i< numFans; i++)
    {
      if ((result = openapiBoxsFanStateGet(&clientHandle, i, &fanState)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get state of FAN %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        switch (fanState)
        {
          case OPEN_ITEMSTATE_NONE:
            sprintf(buf,"NONE");
            break;
          case OPEN_ITEMSTATE_NOT_PRESENT:
            sprintf(buf,"Not Present");
            break;
          case OPEN_ITEMSTATE_OPERATIONAL:
            sprintf(buf,"Operational");
            break;
          case OPEN_ITEMSTATE_FAILED:
            sprintf(buf,"Failed");
            break;
          default:
            sprintf(buf,"NONE");
            break;
        }
        printf("FAN number %d FAN state : %s\n",(i + 1),buf);
      }
    }

    for (i = 0; i< numFans; i++)
    {
      if ((result = openapiBoxsFanTypeGet(&clientHandle, i, &itemType)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get type of FAN %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        switch (itemType)
        {
          case OPEN_TYPE_FIXED:
            sprintf(buf,"Fixed");
            break;
          case OPEN_TYPE_REMOVABLE:
            sprintf(buf,"Removable");
            break;
          default:
            sprintf(buf,"NONE");
            break;
        }
        printf("FAN number %d FAN Type : %s\n",(i + 1),buf);
      }
    }

    for (i = 0; i< numFans; i++)
    {
      if ((result = openapiBoxsFanDutyCycleGet(&clientHandle, i, &fanDuty)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get Duty-Cycle of FAn %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        printf("FAN number %d Duty-Cycle : %d\n",(i + 1),fanDuty);
      }
    }
    if ((result = openapiBoxsNumOfPowSuppliesGet(&clientHandle, &numPowSuplies)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get the number of power supplies in the unit. (result = %d)\n",result);
      break;
    }
    else
    {
      printf("Number of power supplies in the unit %d\n",numPowSuplies);
    }

    if ((result = openapiBoxsPowSupplyDescripSizeGet(&clientHandle, &powSupplyDescSize)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get the power supply description size. (result = %d)\n",result);
      break;
    }
    else
    {
      printf("Description size of power supply in the unit %d\n", powSupplyDescSize);
    }

    str = (char *)malloc(powSupplyDescSize);
    if (NULL == str)
    {
      printf("Error: Could not allocate memory.\n");
      break;
    }
    for (i = 0; i< numPowSuplies; i++)
    {
      memset (str, 0, powSupplyDescSize);
      descrip.pstart = str;
      descrip.size = powSupplyDescSize;
      if ((result = openapiBoxsPowSupplyDescripGet(&clientHandle, i, &descrip)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to description of power supply %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        printf("Power supply %d description : %s\n",(i + 1),str);
      }
    }
    free(str);

    for (i = 0; i<numPowSuplies; i++)
    {
      if ((result = openapiBoxsPowSupplyStateGet(&clientHandle, i, &psState)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get state of the power supply %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        switch (psState)
        {
          case OPEN_ITEMSTATE_NONE:
            sprintf(buf,"NONE");
            break;
          case OPEN_ITEMSTATE_NOT_PRESENT:
            sprintf(buf,"Not-Present");
            break;
          case OPEN_ITEMSTATE_OPERATIONAL:
            sprintf(buf,"Operational");
            break;
          case OPEN_ITEMSTATE_FAILED:
            sprintf(buf,"Failed");
            break;
          case OPEN_ITEMSTATE_POWERING:
            sprintf(buf,"Powering");
            break;
          case OPEN_ITEMSTATE_NOPOWER:
            sprintf(buf,"Not powered");
            break;
          default:
            sprintf(buf,"-------");
            break;
        }
        printf("Power supply %d state : %s\n",(i + 1),buf);
      }
    }

    for (i = 0; i<numPowSuplies; i++)
    {
      if ((result = openapiBoxsPowSupplyTypeGet(&clientHandle, i, &itemType)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get Type of the power supply %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        switch (itemType)
        {
          case OPEN_TYPE_FIXED:
            sprintf(buf,"Fixed");
            break;
          case OPEN_TYPE_REMOVABLE:
            sprintf(buf,"Removable");
            break;
          default:
            sprintf(buf,"-------");
            break;
        }
        printf("Power supply %d Type : %s\n",(i + 1),buf);
      }
    }
    if ((result = openapiBoxsNumOfTempSensorsGet(&clientHandle, &numOfTempSemsors)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get the number of thermal sensors. (result = %d)\n",result);
      break;
    }
    else
    {
      printf("Number of thermal senosrs in the unit %d\n",numOfTempSemsors);
    }

    if ((result = openapiBoxsTempSensorDescripSizeGet(&clientHandle, &tempSensorDescSize)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get the thermal sensors description size. (result = %d)\n",result);
      break;
    }
    else
    {
      printf("Description size of thermal senosrs in the unit %d\n", tempSensorDescSize);
    }

    str = (char *)malloc(tempSensorDescSize);
    if (NULL == str)
    {
      printf("Error: Could not allocate memory.\n");
      break;
    }
    for (i = 0; i<numOfTempSemsors; i++)
    {
      memset (str, 0, tempSensorDescSize);
      descrip.pstart = str;
      descrip.size = tempSensorDescSize;
      if ((result = openapiBoxsTempSensorDescripGet(&clientHandle, i, &descrip)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to description of thermal sensor %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        printf("Thermal sensor %d description : %s\n",(i + 1),str);
      }
    }
    free(str);

    for (i = 0; i<numOfTempSemsors; i++)
    {
      if ((result = openapiBoxsTempGet(&clientHandle, i, &temperature)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get temperature for thermal sensor %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        printf("Thermal sensor %d temperature : %d\n",(i + 1),temperature);
      }
    }
    for (i = 0; i<numOfTempSemsors; i++)
    {
      if ((result = openapiBoxsTempSensorStatusGet(&clientHandle, i, &state)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get the state of thermal sensor %d. (result = %d)\n",(i + 1),result);
        break;
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        switch (state)
        {
          case OPEN_TEMPSENSOR_STATE_NORMAL:
            sprintf(buf,"Normal");
            break;
          case OPEN_TEMPSENSOR_STATE_WARNING:
            sprintf(buf,"Warning");
            break;
          case OPEN_TEMPSENSOR_STATE_CRITICAL:
            sprintf(buf,"Critical");
            break;
          case OPEN_TEMPSENSOR_STATE_SHUTDOWN:
            sprintf(buf,"Shutdown");
            break;
          case OPEN_TEMPSENSOR_STATE_NOTPRESENT:
            sprintf(buf,"Not-Present");
            break;
          case OPEN_TEMPSENSOR_STATE_NOTOPERATIONAL:
            sprintf(buf,"Non-Operational");
            break;
          default:
            sprintf(buf,"--------");
            break;
        }
        printf("Thermal sensor %d state : %s\n",(i + 1),buf);
      }
    }

    /* Print SFP transceiver data */
    printf("\n");
    printf("SFP transceiver data:\n");
    if ((result = openapiIfFirstGet(&clientHandle, OPEN_INTF_TYPE_PHY, &ifNum)) == OPEN_E_NONE)
    {
      do
      {
        if ((result = openapiIfLinkStateGet(&clientHandle, ifNum, &linkState)) != OPEN_E_NONE)
        {
          printf("Bad Error code while getting link state(result = %d)\n", result);
        }
        if (linkState == OPEN_LINK_DOWN)
        {
          continue;
        }

        if ((result = openapiBoxsSfpTypeGet(&clientHandle, ifNum, &moduleType)) != OPEN_E_NONE)
        {
          printf("Failed to get module type(result = %d)\n", result);
          break;
        }

        if ((linkState == OPEN_LINK_UP) && (moduleType != OPEN_MODULE_NONE))
        {
          if (headerPrinted == 0)
          {
            printf("                                                                         Link Length    Nominal\n");
            printf("                                                                       50um    62.5um  Signal Rate\n");
            printf("Port   Vendor Name       Serial Number     Part Number       Revision   [m]      [m]     [mbps]     Compliance\n");
            printf("-----  ----------------  ----------------  ----------------  --------  ------  ------  -----------  ----------------\n");

            headerPrinted = 1; /* set the headerPrinted flag */
          }

          printf("%-5d  ", ifNum);

          if ((result = openapiBoxsSfpVendorNameSizeGet(&clientHandle, &vendorNameSize)) != OPEN_E_NONE)
          {
            printf("Bad error code getting sfp vendor name size.(result = %d)\n", result);
            break;
          }

          if ((sfpBuf = (char*)malloc(vendorNameSize)) == NULL)
          {
            printf("Could not allocate memory for vendor name.\n");
            return 0;
          }
          memset(sfpBuf, 0, vendorNameSize);
          vendorNameDesc.pstart = sfpBuf;
          vendorNameDesc.size = vendorNameSize;          
          if ((result = openapiBoxsSfpVendorNameGet(&clientHandle, ifNum, &vendorNameDesc)) == OPEN_E_NONE)
          {
            printf("%-16s  ", sfpBuf);
          }
          else
          {
            printf("Bad error code while getting vendor name on interface %d. (result = %d)\n", ifNum, result);
            break;
          }
          free(sfpBuf);

          if ((result = openapiBoxsSfpSerialNumberSizeGet(&clientHandle, &serialNumberSize)) != OPEN_E_NONE)
          {
            printf("Bad error code getting sfp serial number size. (result = %d)\n", result);
            break;
          }

          if ((sfpBuf = (char*)malloc(serialNumberSize)) == NULL)
          {
            printf("Could not allocate memory for serial number.\n");
            return 0;
          }
          memset(sfpBuf, 0, serialNumberSize);
          serialNumberDesc.pstart = sfpBuf;
          serialNumberDesc.size = serialNumberSize;
          if ((result = openapiBoxsSfpSerialNumberGet(&clientHandle, ifNum, &serialNumberDesc)) == OPEN_E_NONE)
          {
            printf("%-16s  ", sfpBuf);
          }
          else
          {
            printf("Bad error code while getting serial number on interface %d. (result = %d)\n", ifNum, result);
            break;
          }
          free(sfpBuf);


          if ((result = openapiBoxsSfpPartNumberSizeGet(&clientHandle, &partNumberSize) ) != OPEN_E_NONE)
          {
            printf("Bad error code getting sfp part number size.(result = %d)\n", result); 
            break;
          }

          if ((sfpBuf = (char*)malloc(partNumberSize)) == NULL)
          {
            printf("Could not allocate memory for part number.\n");
            return 0;
          }
          memset(sfpBuf, 0, partNumberSize);
          partNumberDesc.pstart = sfpBuf;
          partNumberDesc.size = partNumberSize;
          if ((result = openapiBoxsSfpPartNumberGet(&clientHandle, ifNum, &partNumberDesc)) == OPEN_E_NONE)
          {
            printf("%-16s  ", sfpBuf);
          }
          else
          {
            printf("Bad error code while getting part number on interface %d. (result = %d)\n", ifNum, result);
            break;
          }
          free(sfpBuf);


          if ((result = openapiBoxsSfpVendorRevSizeGet(&clientHandle, &vendorRevSize)) != OPEN_E_NONE)
          {
            printf("Bad error code getting sfp vendor revision.(result = %d)\n", result);
            break;
          }

          if ((sfpBuf = (char*)malloc(vendorRevSize)) == NULL)
          {
            printf("Could not allocate memory for vendor Revision.\n");
            return 0;
          }
          memset(sfpBuf, 0, vendorRevSize);
          vendorRevDesc.pstart = sfpBuf;
          vendorRevDesc.size = vendorRevSize;
          if ((result = openapiBoxsSfpVendorRevisionGet(&clientHandle, ifNum, &vendorRevDesc)) == OPEN_E_NONE)
          {
            printf("%-8s  ", sfpBuf);
          }
          else
          {
            printf("Bad error code while getting vendor revision on interface %d. (result = %d)\n", ifNum, result);
            break;
          }
          free(sfpBuf);

          if ((result = openapiBoxsSfpLinkLength50umGet(&clientHandle, ifNum, &linkLength50um)) == OPEN_E_NONE)
          {
            printf("%-6d  ", linkLength50um);
          }
          else
          {
            printf("Bad error code while getting link length(50um) on interface %d. (result = %d)\n", ifNum, result);
            break;
          }

          if ((result = openapiBoxsSfpLinkLength62_5umGet(&clientHandle, ifNum, &linkLength62_5um)) == OPEN_E_NONE)
          {
            printf("%-6d  ", linkLength62_5um);
          }
          else
          {
            printf("Bad error code while getting link length(62_5um) on interface %d. (result = %d)\n", ifNum, result);
            break;
          }

          if ((result = openapiBoxsSfpNominalSignalRate(&clientHandle, ifNum, &nominalSigRate)) == OPEN_E_NONE)
          {
            printf("%-11d  ", nominalSigRate);
          }
          else
          {
            printf("Bad error code while getting nominalSigRate on interface %d. (result = %d)\n", ifNum, result);
            break;
          }

          if ((result = openapiBoxsSfpComplianceSizeGet(&clientHandle, &complianceSize)) != OPEN_E_NONE)
          {
            printf("Bad error code getting sfp compliance size.(result = %d)\n", result);
            break;
          }

          if ((sfpBuf = (char*)malloc(complianceSize)) == NULL)
          {
            printf("Could not allocate memory for compliance.\n");
            return 0;
          }          
          memset(sfpBuf, 0, complianceSize);
          complianceDesc.pstart = sfpBuf;
          complianceDesc.size = complianceSize;
          if ((result = openapiBoxsSfpComplianceGet(&clientHandle, ifNum, &complianceDesc)) == OPEN_E_NONE)
          {
            printf("%-16s\n", sfpBuf); 
          }
          else
          {
            printf("Bad error code while getting compliance on interface %d. (result = %d)\n", ifNum, result);
            break;
          }
          free(sfpBuf);
        }
      }while (openapiIfNextGet(&clientHandle, OPEN_INTF_TYPE_PHY, ifNum, &ifNum) == OPEN_E_NONE);

      if (headerPrinted == 0)
      {
        printf("SFP modules not plugged in.\n");
      }
    }

    /* Print SFP module info */
    printf("\n");
    printSFPModuleInfo(&clientHandle);


    /* Port Locator  */
    printf("\n");
    printf("Port Locator:\n");

    printf("\nSetting locator on ports 1, 3 and 5");
    sleep(2);
    if ((result = openapiPortLocatorInterfaceEnable(&clientHandle, 1, OPEN_ENABLE)) != OPEN_E_NONE)
    {
      printf("Bad error code setting port 1 (result = %d)\n", result);
      break;
    }
    if ((result = openapiPortLocatorInterfaceEnable(&clientHandle, 3, OPEN_ENABLE)) != OPEN_E_NONE)
    {
      printf("Bad error code setting port 1 (result = %d)\n", result);
      break;
    }
    if ((result = openapiPortLocatorInterfaceEnable(&clientHandle, 5, OPEN_ENABLE)) != OPEN_E_NONE)
    {
      printf("Bad error code setting port 1 (result = %d)\n", result);
      break;
    }
    printf("\nDisabling port locator on port 3");
    sleep(2);
    if ((result = openapiPortLocatorInterfaceEnable(&clientHandle, 3, OPEN_DISABLE)) != OPEN_E_NONE)
    {
      printf("Bad error code disabling port 3 (result = %d)\n", result);
      break;
    }
    printf("\nGlobally disabling port locator");
    sleep(2);
    if ((result = openapiPortLocatorGlobalDisable(&clientHandle) != OPEN_E_NONE))
    {
        printf("Bad error code disabling port locator (result = %d)\n", result);
        break;
    }
  }
  while (0);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0,
	       "Stopping environment API example application");

  printf("\n");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
