
/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  bridge_example.c
*
* @purpose   Switch Bridge APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    06/02/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_bridge.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrate Bridge OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  /* More information about bridge you may find in RFC 1493 */
  openapiClientHandle_t clientHandle;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100] = {0};
  char buffer[256] = {0};
  open_buffdesc string;
  char bridgeAddress[40] = {0};
  uint32_t bridgeDigit = 0;
  uint64_t bridgeTime = 0;
  uint16_t value = 0;
  uint32_t numPorts = 0;
  open_error_t result;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("bridge_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting result = %d.\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Bridge API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s.\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS version retrieve error.\n");
  }
  printf("\n");
 
  /* exercise various OPEN API bridge functions */
    /*      
      dot1dBase ->  bridgeAddress
    */
   string.pstart = bridgeAddress;
   string.size = sizeof(bridgeAddress);
   result = openapiDot1dBaseBridgeAddressGet(&clientHandle, &string);
   if (result != OPEN_E_NONE)
   {
     printf("Bad return code when we are trying to get the unique MAC address. result = %d.\n", result);
   }
   else
   {  
      printf("Bridge MAC address is: %s. Bridge MAC address lenght is: %d.\n",
              bridgeAddress, string.size);
   }

   /* 
      dot1dBase -> Type 
   */
   result = openapiDot1dBaseTypeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     /*Print number of bridge */
     printf("Bridge base type is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_FAIL) 
   {
     printf("Bad return code when we are trying to get base type of bridging.\n Returned values are invalid. result = %d.\n", result);
   }
   else if (result == OPEN_E_ERROR) 
   {
     printf("Bad return code when we are trying to get base type of bridging.\n Counter requested does not exist. result = %d.\n", result);
   }
   else
   {
     printf("Bad return code when we are trying to get base type of bridging. Error in parameter passed. Exiting. result = %d.\n", result);
   }
    
    /*
       dot1dBase ->  numPorts                          
    */ 
    result = openapiDot1dBaseNumPortsGet(&clientHandle, &numPorts);
    if (result == OPEN_E_NONE)
    {
      /*
         dot1dBase ->  PortEntry (index: dot1dBasePort)                        
      */
      result = openapiDot1dBasePortEntryGet(&clientHandle, numPorts);
      if (result == OPEN_E_NONE)
      {
        printf("Number of ports controlled by bridging entity is: %d.\n", numPorts);
      }
      else if (result == OPEN_E_FAIL) 
      {
        printf("Bad return code when we are trying to get base type of bridging.\n Returned values are invalid. result = %d.\n", result);
      }
      else
      {
        printf("Bad return code when we are trying to get base type of bridging. Error in parameter passed. Exiting. result = %d.\n", result);        
      }
    }
    else if (result == OPEN_E_FAIL) 
    {
      printf("Bad return code when we are trying to get number of ports controlled by bridging entity.\n Returned values are invalid. result = %d.\n", result);
    }
    else if (result == OPEN_E_ERROR) 
    {
      printf("Bad return code when we are trying to get number of ports controlled by bridging entity.\n Counter requested does not exist. result = %d.\n", result);
    }
    else
    {
      printf("Bad return code when we are trying to get number of ports controlled by bridging entity. Error in parameter passed. Exiting. result = %d.\n", result);     
    }
    /*
       dot1dBase ->  BasePortEntry ->  PortCircuit
       numPorts == (index: dot1dBasePort)
    */
    string.pstart = buffer;
    string.size = sizeof(buffer);
    result = openapiDot1dBasePortCircuitGet(&clientHandle, numPorts, &string);
    if (result == OPEN_E_NONE)
    {
      printf("Name of an object instance unique to this port is: %s.\n", buffer);
    }
     else if (result == OPEN_E_FAIL) 
    {
      printf("Returned values are invalid. result = %d.\n", result);
    }
    else if (result == OPEN_E_ERROR) 
    {
      printf("Counter requested does not exist. result = %d.\n", result);
    }
    else
    {
      printf("Error in parameter passed. Exiting. result = %d.\n", result);     
    }

    /*
       dot1dBase ->  BasePortEntry ->  PortIfIndex
       numPorts == (index: dot1dBasePort)
    */ 
    result = openapiDot1dBasePortIfIndexGet(&clientHandle, numPorts, &bridgeDigit);
    if (result == OPEN_E_NONE)
    {
      printf("Interface index is: %d.\n", bridgeDigit);
    }
     else if (result == OPEN_E_FAIL) 
    {
      printf("Bad return code when we are trying to get interface index.\n Returned values are invalid. result = %d.\n", result);
    }
    else if (result == OPEN_E_ERROR) 
    {
      printf("Counter requested does not exist. result = %d.\n", result);
    }
    else
    {
      printf("Error in parameter passed. Exiting. result = %d.\n", result);      
    }
   
    /*
       dot1dBase ->  BasePort ->  DelayExceededDiscards
       numPorts == (index: dot1dBasePort)
    */ 
    result = openapiDot1dBasePortDelayExceededDiscardsGet(&clientHandle, numPorts, &bridgeDigit);
    if (result == OPEN_E_NONE)
    {
      printf("Number of frames discarded by this port due to delay through the bridge is: %d.\n", bridgeDigit);
    }
     else if (result == OPEN_E_FAIL) 
    {
      printf("Bad return code when we are trying to get number of frames. \nReturned values are invalid. result = %d.\n", result);
    }
    else if (result == OPEN_E_ERROR) 
    {
      printf("Bad return code when we are trying to get number of frames. result = %d.\n", result);
    }
    else if (result == OPEN_E_UNAVAIL) 
    {
      printf("Feature is not supported. result = %d.\n", result);
    }
    else
    {
      printf("Error in parameter passed. Exiting. result = %d.\n", result);      
    }
    /*
       dot1dBase ->  BasePort ->  DelayExceededDiscards
       numPorts == (index: dot1dBasePort)
    */ 
    result = openapiDot1dBasePortMtuExceededDiscardsGet(&clientHandle, numPorts, &bridgeDigit);
    if (result == OPEN_E_NONE)
    {
      printf("Number of frames discarded by this port due to delay through the bridge is: %d.\n", bridgeDigit);
    }
     else if (result == OPEN_E_FAIL) 
    {
      printf("Returned values are invalid. result = %d.\n", result);
    }
    else if (result == OPEN_E_ERROR) 
    {
     printf("Bad return code when we are trying to get number of frames. result = %d.\n", result);
    }
    else if (result == OPEN_E_NOT_FOUND) 
    {
      printf("Feature is not supported. result = %d.\n", result);
    }
    else
    {
      printf("Error in parameter passed. Exiting. result = %d.\n", result);     
    }

    /*
       dot1dStp -> ProtocolSpecification    
    */ 
   result = openapiDot1dStpProtocolSpecificationGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     switch (bridgeDigit)
     {
       case OPEN_STP_PROTOCOL_SPEC_IEEE8021D:
         printf("STP protocol is IEEE8021D(dot1dBridge).\n");
         break;
       case OPEN_STP_PROTOCOL_SPEC_UNKNOWN:
         printf("STP protocol is unknown.\n");
         break;
       default:
         printf("Error.\n");
         break;
     }
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Priority    
   */ 
   result = openapiDot1dStpPriorityGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Spanning tree bridge priority is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> TimeSinceTopologyChange    
   */ 
   result = openapiDot1dStpTimeSinceTopologyChangeGet(&clientHandle, &bridgeTime);
   if (result == OPEN_E_NONE)
   {
     printf("Time since last topology change is: %d.\n", bridgeDigit);
   }
    else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> TopChanges    
   */ 
   result = openapiDot1dStpTopChangesGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP change count is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);
     
   }

   /*
      dot1dStp -> DesignatedRoot    
   */ 
   string.pstart = buffer;
   string.size = sizeof(buffer);
   result = openapiDot1dStpDesignatedRootGet(&clientHandle, &string);
   if (result == OPEN_E_NONE)
   {
     printf("STP designated root is: %s .\n", buffer);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);
   }

   /*
      dot1dStp -> RootCost    
   */ 
   result = openapiDot1dStpRootCostGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Cost of the path is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);
   }

   /*
      dot1dStp -> RootPort    
   */ 
   result = openapiDot1dStpRootPortGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Port number with lowest cost path from bridge to the root bridge is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> MaxAge    
   */ 
   result = openapiDot1dStpMaxAgeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP max age is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> HelloTime    
   */ 
   result = openapiDot1dStpHelloTimeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP hello time is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> HoldTime    
   */ 
   result = openapiDot1dStpHoldTimeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP hold time is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> ForwardDelay    
   */ 
   result = openapiDot1dStpForwardDelayGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP forward delay is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> BridgeMaxAge    
   */ 
   result = openapiDot1dStpBridgeMaxAgeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP bridge max age is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);    
   }

   /*
      dot1dStp -> BridgeHelloTime    
   */
   result = openapiDot1dStpBridgeHelloTimeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP bridge hello time is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> BridgeHelloTime    
   */
   result = openapiDot1dStpBridgeForwardDelayGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP bridge forward delay is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> Priority    
   */
   result = openapiDot1dStpPortPriorityGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP port priority is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   result = openapiDot1dIsInterfaceInCurrentInstance(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     if (bridgeDigit == true)
     {
       printf("Member was removed.\n");
     }
     else
     {
       printf("Member was not removed.\n");
     }
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*Also, you may use while*/
   if ((result = openapiDot1dInstanceMemberGetNext(&clientHandle, numPorts, &bridgeDigit)) == OPEN_E_NONE)
   {
      if (openapiDot1dBasePortEntryGet(&clientHandle, bridgeDigit) == OPEN_E_NONE &&
          openapiDot1dIsValidIntf(&clientHandle, bridgeDigit) == OPEN_E_NONE)
      {
         printf("Next member is: %d.\n", bridgeDigit);
      }
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> State    
   */
   result = openapiDot1dStpPortStateGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP port state is: ");
     switch (bridgeDigit)
     {
       case OPEN_DOT1D_BLOCKING:
         printf("blocking.\n");
         break;
         case OPEN_DOT1D_LISTENING:
         printf("listening.\n");
         break;
       case OPEN_DOT1D_LEARNING:
         printf("learning.\n");
         break;
       case OPEN_DOT1D_FORWARDING:
         printf("forwarding.\n");
         break;
       case OPEN_DOT1D_DISABLED:
         printf("disabled.\n");
         break;
       case OPEN_DOT1D_MANUAL_FWD:
         printf("manual FWD.\n");
         break;
       case OPEN_DOT1D_NOT_PARTICIPATE:
         printf("participate.\n");
         break;
       default:
         printf("Error.\n");
         break;
     }
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> PathCost    
   */
   result = openapiDot1dStpPortPathCostGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP port path cost is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> DesignatedRoot    
   */
   string.pstart = buffer;
   string.size = sizeof(buffer);
   result = openapiDot1dStpPortDesignatedRootGet(&clientHandle, numPorts, &string);
   if (result == OPEN_E_NONE)
   {
     printf("STP designated root for the switch is: %s.\n", buffer);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> DesignatedCost    
   */
   result = openapiDot1dStpPortDesignatedCostGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("STP designated cost for port[%d] is: %d.\n", numPorts, bridgeDigit);
     
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> DesignatedBridge    
   */
   string.pstart = buffer;
   string.size = sizeof(buffer);
   result = openapiDot1dStpPortDesignatedBridgeGet(&clientHandle, numPorts, &string);
   if (result == OPEN_E_NONE)
   {
     printf("STP designated bridge for port[%d] is: %s .\n", numPorts, buffer);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> DesignatedPort    
   */
   result = openapiDot1dStpPortDesignatedPortGet(&clientHandle, numPorts, &value);
   if (result == OPEN_E_NONE)
   {
     printf("STP designated port id for port[%d] is: %d.\n", numPorts, value);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dStp -> Port (index: dot1dSrPort) -> ForwardTransitions    
   */
   result = openapiDot1dStpPortForwardTransitionsGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Number of times port has transitioned from Learning to Forwarding state is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);   
   }

   /*
      dot1dTp ->  LearnedEntryDiscards
   */
   result = openapiDot1dTpLearnedEntryDiscardsGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Number of discarded entries is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get number of discarded entries. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);    
   }

   /*
      dot1dTp ->  AgingTime
   */
   result = openapiDbDot1dTpAgingTimeGet(&clientHandle, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Timeout period for aging out dynamically learned forwarding information is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get timeout period. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dTp ->  AgingTime
   */
   result = openapiDbDot1dTpAgingTimeSet(&clientHandle, bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Timeout period for aging out dynamically learned forwarding information was successful set.\n");
   }
   else if (result == OPEN_E_UNAVAIL)
   {
     printf("Parameter/feature is not supported. result = %d.\n", result);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to set timeout period. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result); 
   }

   /*
      dot1dTp ->  AgingTime
   */
   string.pstart = bridgeAddress;
   string.size = sizeof(bridgeAddress);
   if ((result = openapiDot1dTpFdbEntryNextGet(&clientHandle, &string)) == OPEN_E_NONE)
   {
      /* Print MAC address as series of 12 hecadecimal digits separated  by colons.*/
     printf("Specific unicast MAC address is: %02x:%02x:%02x:%02x:%02x:%02x.\n",
            bridgeAddress[0], bridgeAddress[1], bridgeAddress[2], bridgeAddress[3], bridgeAddress[4], bridgeAddress[5]);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get specific unicast MAC address. result = %d.\n", result);
   }
   else
   {  
     printf("Bad return code when we are trying to get the specific unicast MAC address. result = %d.\n", result);    
   }

   /*
      dot1dTp ->  FdbEntry
   */
   string.pstart = bridgeAddress;
   string.size = sizeof(bridgeAddress);
   if ((result = openapiDot1dTpFdbEntryGet(&clientHandle, &string)) == OPEN_E_NONE)
   {
           /* Print MAC address as series of 12 hecadecimal digits separated  by colons.*/
     printf("Specific unicast MAC address is: %02x:%02x:%02x:%02x:%02x:%02x.\n",
            bridgeAddress[0], bridgeAddress[1], bridgeAddress[2], bridgeAddress[3], bridgeAddress[4], bridgeAddress[5]);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get specific unicast MAC address. result = %d.\n", result);
   }
   else
   {  
     printf("Error in parameter passed. Exiting. result = %d.\n", result);    
   }

   /*
      dot1dTp ->  FdbPort
   */
   string.pstart = bridgeAddress;
   string.size = sizeof(bridgeAddress);
   if ((result = openapiDot1dTpFdbPortGet(&clientHandle, &string, &bridgeDigit)) == OPEN_E_NONE)
   {
     /* Print MAC address as series of 12 hecadecimal digits separated  by colons.*/
     printf("MAC address is: %02x:%02x:%02x:%02x:%02x:%02x. Port number is: %d.\n",
            bridgeAddress[0], bridgeAddress[1], bridgeAddress[2], bridgeAddress[3], bridgeAddress[4], bridgeAddress[5], bridgeDigit);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get MAC address or port number. result = %d.\n", result);
   }
   else
   {  
     printf("Error in parameter passed. Exiting. result = %d.\n", result);
     
   }

   /*
      dot1dTp ->  FdbStatus
   */
   string.pstart = bridgeAddress;
   string.size = sizeof(bridgeAddress);
   if ((result = openapiDot1dTpFdbStatusGet(&clientHandle, &string, &bridgeDigit)) == OPEN_E_NONE)
   {
           /* Print MAC address as series of 12 hecadecimal digits separated  by colons.*/
     printf("MAC address is: %02x:%02x:%02x:%02x:%02x:%02x. Port status is: %d.\n",
            bridgeAddress[0], bridgeAddress[1], bridgeAddress[2], bridgeAddress[3], bridgeAddress[4], bridgeAddress[5], bridgeDigit);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get MAC address or port number. result = %d.\n", result);
   }
   else
   {  
     printf("Bad return code when we are trying to get the MAC address or port number. result = %d.\n", result);     
   }

   /*
      dot1dTp -> Port (index: dot1dTpPort) -> MaxInfo
   */
   result = openapiDot1dTpPortMaxInfoGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Maximum size of the INFO field this port will receive or transmit is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get maximum size. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);     
   }

   /*
      dot1dTp -> Port (index: dot1dTpPort) -> InFrames
   */
   result = openapiDot1dTpPortInFramesGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Number of frames received by this port from its segment is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get number of frames. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);    
   }

   /*
      dot1dTp -> Port (index: dot1dTpPort) -> OutFrames
   */
   result = openapiDot1dTpPortOutFramesGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Number of frames transmitted by this port to its segment is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get number of frames. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);
   }

   /*
      dot1dTp -> Port (index: dot1dTpPort) -> InDiscards
   */
   result = openapiDot1dTpPortInDiscardsGet(&clientHandle, numPorts, &bridgeDigit);
   if (result == OPEN_E_NONE)
   {
     printf("Count of valid frames received which were discarded by the Forwarding Process is: %d.\n", bridgeDigit);
   }
   else if (result == OPEN_E_FAIL)
   {
     printf("Fail to get valid frames. result = %d.\n", result);
   }
   else
   {
     printf("Error in parameter passed. Exiting. result = %d.\n", result);
   }

  (void) openapiClientTearDown(&clientHandle);   
  return 1;
}

