/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  cable_status_example.c
*
* @purpose   Cable Status OpEN API Examples
*
* @component OpEN
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_cable_status.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Cable Status Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printCableStatusAppMenu()
{
  printf("Usage: cable_status_example <test#> <interface> <arg1> <arg2> ... \n");
  printf("Test 1: Run cable test on specified interface and display results: cable_status_example 1 <interface> \n");

  return;
}

/*********************************************************************
* @purpose  Run the cable test and report status on the specified interface.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void cableStatusGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CABLE_STATUS_t cableStatus;
  uint32_t lengthKnown;
  uint32_t shortestLength;
  uint32_t longestLength;
  uint32_t cableFailureLength;
  
  result = openapiCableStatusGet(clientHandle, ifNum, &cableStatus, &lengthKnown, &shortestLength, &longestLength, &cableFailureLength);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get cable status. (result = %d)\n", result);
  }
  else
  {
    printf("Cable status for interface %u: ", ifNum);
    switch (cableStatus)
    {
      case OPEN_CABLE_STATUS_UNTESTED:
        printf("untested");
        break;
      case OPEN_CABLE_STATUS_TEST_FAIL:
        printf("test failed for unknown reason");
        break;
      case OPEN_CABLE_STATUS_NORMAL:
        printf("normal");
        break;
      case OPEN_CABLE_STATUS_OPEN:
        printf("open");
        break;
      case OPEN_CABLE_STATUS_SHORT:
        printf("shorted");
        break;
      case OPEN_CABLE_STATUS_OPENSHORT:
        printf("open and/or shorted");
        break;
      case OPEN_CABLE_STATUS_CROSSTALK:
        printf("crosstalk");
        break;
      case OPEN_CABLE_STATUS_NOCABLE:
        printf("no cable");
        break;
      default:
        printf("undecoded value for cable status");
        break;
    }
    printf(", lengthKnown = %u, shortestLength = %u, longestLength = %u, cableFailureLength = %u\n",
           lengthKnown, shortestLength, longestLength, cableFailureLength);
  }
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t ifNum, testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 3)
  {
    printCableStatusAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  ifNum = atoi(argv[2]);
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("cable_status_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Cable Status API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printCableStatusAppMenu();
        exit(1);
      }
      cableStatusGet(&clientHandle, ifNum);
      break;

    default:
      printCableStatusAppMenu();
      break;
  }
                                          
  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Cable Status API example application");

  return 0;
}
