/*********************************************************************
*
* (C) Copyright Broadcom Corporation 2003-2018
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  cloud_managed_example.c
*
* @purpose   Cloud Managed Configuration OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    12/03/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_cloud_managed.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/**************************************************************************
*
* @purpose      create ascii displayable string from IP (4/6) address
*
* @param        family          address family
*               addr            pointer to address
*               str             pointer to output string
*               len             length of output string
*
* @returns      output string or L7_NULL
*
* @end
*
*************************************************************************/
char *cmInetNtop(uint32_t family, char *addr, char *str, uint32_t len)
{
   const char *rp;
   OPEN_AF_t    fam;  

   switch (family)
   {
     case OPEN_AF_NONE:  fam = AF_INET;  break;
     case OPEN_AF_INET:  fam = AF_INET;  break;
     case OPEN_AF_INET6: fam = AF_INET6; break;
   }
   rp = inet_ntop(fam, addr, str, len);
   return (char *)rp;
}

/**************************************************************************
*
* @purpose      create ascii displayable string from encryption type
*
* @param        encryptType     encryption type enum value
*
* @returns      output string
*
* @end
*
*************************************************************************/
char *encryptTypePrintable(OPEN_PASSWORD_ENCRYPT_ALG_t encryptType)
{
   char *tmp;

   switch (encryptType)
   {
     case OPEN_PASSWORD_ENCRYPT_NONE:  tmp = "NONE";      break;
     case OPEN_PASSWORD_ENCRYPT_AES:   tmp = "AES";       break;
     case OPEN_PASSWORD_ENCRYPT_MD5:   tmp = "MD5";       break;
     default:                          tmp = "<INVALID>"; break;
   }
   return tmp;
}

/**************************************************************************
*
* @purpose      convert ascii displayable string to upper case in place
*
* @param        family          address faminly
*               addr            pointer to address
*               str             pointer to ouput string
*               len             length of output string
*
* @returns      output string or L7_NULL
*
* @end
*
*************************************************************************/
char *strupr(char *s)
{
    char* tmp = s;

    for (;*tmp;++tmp) {
        *tmp = toupper((unsigned char) *tmp);
    }

    return s;
}

/*******************************************************************
*
* @brief  This function prints the Cloud Managed Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printCloudManagedAppMenu()
{
  printf("\nUsage:  cloud_managed_example <test#> <arg1> <arg2> ... \n\n");
  printf("Test 0: Cloud Managed OpEN APIs Sanity:   cloud_managed_example 0 \n");
  printf("Test 1: Get Cloud Managed Mode:           cloud_managed_example 1 \n");
  printf("Test 2: Set Cloud Managed Mode:           cloud_managed_example 2 <mode>\n");
  printf("Test 3: Get Cloud Managed Proxy IP        cloud_managed_example 3 \n");
  printf("Test 4: Set Cloud Managed Proxy IP        cloud_managed_example 4 <IP> <port>\n");
  printf("Test 5: Get Cloud Managed Proxy\n");
  printf("        Username, Password & Encryption   cloud_managed_example 5 \n");
  printf("Test 6: Set Cloud Managed Proxy\n");
  printf("        Username, Password & Encryption   cloud_managed_example 6 <user> <pwd> [AES]\n");
  printf("Test 7: Get Cloud Managed Server ID       cloud_managed_example 7 \n");
  printf("Test 8: Set Cloud Managed Server ID       cloud_managed_example 8 <addr>\n");
  printf("\n");

  return;
}

/*********************************************************************
* @purpose  Get Cloud Managed Mode address.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void cloudManagedModeGet (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t mode;

  result = openapiCloudManagedModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get Cloud Managed Mode: (result = %d)\n", result);
  }
  else
  {
    printf("Cloud Managed Mode Get: %s (%d). (result = %d)\n",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  }
  return;
}

/*********************************************************************
* @purpose  Set Cloud Managed Mode address.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void cloudManagedModeSet (openapiClientHandle_t *clientHandle,
                          uint32_t mode)
{
  open_error_t result;

  result = openapiCloudManagedModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set Cloud Managed Mode: (result = %d)\n", result);
  }
  else
  {
    printf("Cloud Managed Mode Set: %s (%d). (result = %d)\n",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  }
  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedModeGet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t mode;

  printf("\nTesting openapiCloudManagedModeGet(): \n");

  result = openapiCloudManagedModeGet(NULL, &mode);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedModeGet(clientHandle, &mode);
  printf(" %s -- Cloud Managed Mode Get: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedModeSet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityModeSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t mode;

  printf("\nTesting openapiCloudManagedModeSet(): \n");

  result = openapiCloudManagedModeSet(NULL, mode);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  /* test disable */
  mode = OPEN_DISABLE;
  result = openapiCloudManagedModeSet(clientHandle, mode);
  printf(" %s -- Cloud Managed Mode Set: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  result = openapiCloudManagedModeGet(clientHandle, &mode);
  printf(" %s -- Cloud Managed Mode Get: %s (%d). (result = %d)\n",
         ((OPEN_E_NONE==result)&&(mode==OPEN_DISABLE))?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);

  /* test enable */
  mode = OPEN_ENABLE;
  result = openapiCloudManagedModeSet(clientHandle, mode);
  printf(" %s -- Cloud Managed Mode Set: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  result = openapiCloudManagedModeGet(clientHandle, &mode);
  printf(" %s -- Cloud Managed Mode Get: %s (%d). (result = %d)\n",
         ((OPEN_E_NONE==result)&&(mode==OPEN_ENABLE))?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);

  /* test out of range */
  mode = 4;
  result = openapiCloudManagedModeSet(clientHandle, mode);
  printf(" %s -- Cloud Managed Mode Set: %s (%d). (result = %d)\n",
         (OPEN_E_PARAM==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  result = openapiCloudManagedModeGet(clientHandle, &mode);
  printf(" %s -- Cloud Managed Mode Get: %s (%d). (result = %d)\n",
         ((OPEN_E_NONE==result)&&(mode==OPEN_ENABLE))?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedProxyIpGet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityProxyIpGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  uint32_t portNum;
  char buffer[256];

  printf("\nTesting openapiCloudManagedProxyIpGet(): \n");

  result = openapiCloudManagedProxyIpGet(NULL, &ipAddr, &portNum);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyIpGet(clientHandle, NULL, &portNum);
  printf(" %s -- NULL IP Addr. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyIpGet(clientHandle, &ipAddr, NULL);
  printf(" %s -- NULL Port Num. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyIpGet(clientHandle, &ipAddr, &portNum);
  (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
  printf(" %s -- Cloud Managed Proxy IP Address Get: %s:%d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED", buffer, portNum, result);
  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedProxyIpSet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityProxyIpSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  uint32_t portNum;
  char buffer[256];
  open_buffdesc buffDesc;

  printf("\nTesting openapiCloudManagedProxyIpSet(): \n");

  ipAddr.family = OPEN_AF_INET;
  ipAddr.addr.ipv4 = 0x0A0B0C0D;
  portNum = 50000;
  result = openapiCloudManagedProxyIpSet(NULL, &ipAddr, portNum);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyIpSet(clientHandle, NULL, portNum);
  printf(" %s -- NULL IP Addr. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  ipAddr.family = 4;
  result = openapiCloudManagedProxyIpSet(clientHandle, &ipAddr, portNum);
  printf(" %s -- Address Family out of range. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  ipAddr.family = OPEN_AF_INET;
  portNum = 50000+50000;
  result = openapiCloudManagedProxyIpSet(clientHandle, &ipAddr, portNum);
  printf(" %s -- Port Num too high. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  ipAddr.family = OPEN_AF_INET;
  ipAddr.addr.ipv4 = 0x0A0B0C0D;
  portNum = 50000;
  result = openapiCloudManagedProxyIpSet(clientHandle, &ipAddr, portNum);
  if (OPEN_E_NONE==result)
  {
    (void)openapiCloudManagedProxyIpGet(clientHandle, &ipAddr, &portNum);
  }
  (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
  printf(" %s -- Cloud Managed Proxy IP Address Set: %s:%d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         buffer, portNum, result);

  memset(buffer, 0, sizeof(buffer));
  strncpy(buffer, "10.27.65.57", (sizeof(buffer) - 1));
  buffDesc.pstart = buffer;
  buffDesc.size = strlen(buffDesc.pstart)+1;
  portNum = 50505;
  (void)openapiInetAddrGet(clientHandle, &buffDesc, &ipAddr);
  result = openapiCloudManagedProxyIpSet(clientHandle, &ipAddr, portNum);
  if (OPEN_E_NONE==result)
  {
    (void)openapiCloudManagedProxyIpGet(clientHandle, &ipAddr, &portNum);
  }
  (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
  printf(" %s -- Cloud Managed Proxy IP Address Set: %s:%d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         buffer, portNum, result);

  memset(buffer, 0, sizeof(buffer));
  strncpy(buffer, "2001:cdba:0000:0000:0000:0000:0867:5309", (sizeof(buffer) - 1));
  buffDesc.pstart = buffer;
  buffDesc.size = strlen(buffDesc.pstart)+1;
  portNum = 60005;
  (void)openapiInetAddrGet(clientHandle, &buffDesc, &ipAddr);
  result = openapiCloudManagedProxyIpSet(clientHandle, &ipAddr, portNum);
  if (OPEN_E_NONE==result)
  {
    (void)openapiCloudManagedProxyIpGet(clientHandle, &ipAddr, &portNum);
  }
  (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
  printf(" %s -- Cloud Managed Proxy IP Address Set: %s:%d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         buffer, portNum, result);

  memset(buffer, 0, sizeof(buffer));
  strncpy(buffer, "2001:0DB8:AC10:FE01::", (sizeof(buffer) - 1));
  buffDesc.pstart = buffer;
  buffDesc.size = strlen(buffDesc.pstart)+1;
  portNum = 0;
  (void)openapiInetAddrGet(clientHandle, &buffDesc, &ipAddr);
  result = openapiCloudManagedProxyIpSet(clientHandle, &ipAddr, portNum);
  if (OPEN_E_NONE==result)
  {
    (void)openapiCloudManagedProxyIpGet(clientHandle, &ipAddr, &portNum);
  }
  (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
  printf(" %s -- Cloud Managed Proxy IP Address Set: %s:%d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         buffer, portNum, result);

  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedProxyUserGet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityProxyUserGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc userName;
  open_buffdesc password;
  OPEN_PASSWORD_ENCRYPT_ALG_t encryptType = OPEN_PASSWORD_ENCRYPT_NONE;
  char buff1[256];
  char buff2[256];

  printf("\nTesting openapiCloudManagedProxyUserGet(): \n");

  userName.pstart = buff1;
  userName.size   = sizeof(buff1);
  password.pstart = buff2;
  password.size   = sizeof(buff2);

  result = openapiCloudManagedProxyUserGet(NULL, &userName, &encryptType, &password);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyUserGet(clientHandle, NULL, &encryptType, &password);
  printf(" %s -- NULL User Name. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyUserGet(clientHandle, &userName, NULL, &password);
  printf(" %s -- NULL Encryption Type. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyUserGet(clientHandle, &userName, &encryptType, NULL);
  printf(" %s -- NULL Password. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyUserGet(clientHandle, &userName, &encryptType, &password);
  printf(" %s -- Cloud Managed Proxy User Credentials Get: UID: '%s'(sz:%d) PWD: '%s'(sz:%d) "
         "ENCRYPT: %s. (result = %d)\n", (OPEN_E_NONE==result)?"PASSED":"FAILED", 
         (char *)userName.pstart, userName.size, (char *)password.pstart, password.size,
         encryptTypePrintable(encryptType),
         result);

  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedProxyUserSet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityProxyUserSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc userName;
  open_buffdesc password;
  OPEN_PASSWORD_ENCRYPT_ALG_t encryptType = OPEN_PASSWORD_ENCRYPT_NONE;
  char pwBuff[256];
  char buff1[256];
  char buff2[256];

  printf("\nTesting openapiCloudManagedProxyUserSet(): \n");

  userName.pstart = buff1;
  userName.size   = sizeof(buff1);
  password.pstart = buff2;
  password.size   = sizeof(buff2);

  result = openapiCloudManagedProxyUserSet(NULL, &userName, encryptType, &password);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyUserSet(clientHandle, NULL, encryptType, &password);
  printf(" %s -- NULL User Name. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  encryptType = OPEN_PASSWORD_ENCRYPT_MD5;
  result = openapiCloudManagedProxyUserSet(clientHandle, &userName, encryptType, &password);
  printf(" %s -- MD5 Encryption Type Not Supported. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  encryptType = OPEN_PASSWORD_ENCRYPT_MD5+3;
  result = openapiCloudManagedProxyUserSet(clientHandle, &userName, encryptType, &password);
  printf(" %s -- Invalid Encryption Type. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedProxyUserSet(clientHandle, &userName, encryptType, NULL);
  printf(" %s -- NULL Password. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  memset(buff1, 0, sizeof(buff1));
  strncpy(buff1, "administrator", (sizeof(buff1) - 1));
  memset(buff2, 0, sizeof(buff2));
  strncpy(buff2, "secret password", (sizeof(buff2) - 1));
  memset(pwBuff, 0, sizeof(pwBuff));
  strncpy(pwBuff, buff2, (sizeof(pwBuff) - 1));
  userName.size   = strlen(userName.pstart)+1;
  password.size   = strlen(password.pstart)+1;
  encryptType = OPEN_PASSWORD_ENCRYPT_NONE;
  result = openapiCloudManagedProxyUserSet(clientHandle, &userName, encryptType, &password);
  if (OPEN_E_NONE == result)
  {
    memset(buff1, 0, sizeof(buff1));
    userName.size = sizeof(buff1);
    memset(buff2, 0, sizeof(buff2));
    password.size = sizeof(buff2);
    result = openapiCloudManagedProxyUserGet(clientHandle, &userName, &encryptType, &password);
    if (OPEN_E_NONE != result)
    {
      printf("Error getting Proxy user info (result = %d)\n", result);
    }
  }
  printf(" %s -- Cloud Managed Proxy User Credentials Set (Unencrypted PWD): UID: '%s'(sz:%d) PWD: '%s'(sz:%d) "
         "ENCRYPT: %s. (result = %d)\n", (OPEN_E_NONE==result)?"PASSED":"FAILED", 
         (char *)userName.pstart, userName.size, pwBuff, (int)strlen(pwBuff)+1,
         encryptTypePrintable(encryptType),
         result);
  /* repeat setting password, this time as AES encrypted value (use encrypted password value just read) */
  encryptType = OPEN_PASSWORD_ENCRYPT_AES;
  result = openapiCloudManagedProxyUserSet(clientHandle, &userName, encryptType, &password);
  if (OPEN_E_NONE == result)
  {
    (void)openapiCloudManagedProxyUserGet(clientHandle, &userName, &encryptType, &password);
  }
  printf(" %s -- Cloud Managed Proxy User Credentials Set (Encrypted PWD): UID: '%s'(sz:%d) PWD: '%s'(sz:%d) "
         "ENCRYPT: %s. (result = %d)\n", (OPEN_E_NONE==result)?"PASSED":"FAILED", 
         (char *)userName.pstart, userName.size, (char *)password.pstart, password.size,
         encryptTypePrintable(encryptType),
         result);

  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedServerUrlGet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityServerUrlGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc url;
  char buff[256];

  printf("\nTesting openapiCloudManagedServerUrlGet(): \n");

  result = openapiCloudManagedServerUrlGet(NULL, &url);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedServerUrlGet(clientHandle, NULL);
  printf(" %s -- NULL Server URL. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  url.pstart = buff;
  url.size   = sizeof(buff);
  result = openapiCloudManagedServerUrlGet(clientHandle, &url);
  printf(" %s -- Cloud Managed Server URL Get: Server URL: %s (sz:%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         (char *)url.pstart, url.size, result);
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiCloudManagedServerUrlSet()
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanityServerUrlSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc url;
  char buff[256];

  printf("\nTesting openapiCloudManagedServerUrlSet(): \n");

  url.pstart = buff;
  url.size   = sizeof(buff);

  result = openapiCloudManagedServerUrlSet(NULL, &url);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiCloudManagedServerUrlSet(clientHandle, NULL);
  printf(" %s -- NULL Server URL. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  memset(buff, 0, sizeof(buff));
  strncpy(buff, "http://somewhere.in-the-cloud.net/abc_inc", (sizeof(buff) - 1));
  url.size   = strlen(url.pstart)+1;
  result = openapiCloudManagedServerUrlSet(clientHandle, &url);
  if (OPEN_E_NONE == result)
  {
    (void)openapiCloudManagedServerUrlGet(clientHandle, &url);
  }
  printf(" %s -- Cloud Managed Server URL Set: Server URL:%s (sz:%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         (char *)url.pstart, url.size, result);
  return;
}

/*********************************************************************
* @purpose  Sanity of cloudManaged OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void cloudManagedOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  cloudManagedOpENAPISanityModeSet(clientHandle);
  cloudManagedOpENAPISanityModeGet(clientHandle);
  cloudManagedOpENAPISanityProxyIpSet(clientHandle);
  cloudManagedOpENAPISanityProxyIpGet(clientHandle);
  cloudManagedOpENAPISanityProxyUserSet(clientHandle);
  cloudManagedOpENAPISanityProxyUserGet(clientHandle);
  cloudManagedOpENAPISanityServerUrlSet(clientHandle);
  cloudManagedOpENAPISanityServerUrlGet(clientHandle);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char buffer[256];
  open_buffdesc buffDesc2;
  char buffer2[256];
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[100];
  uint32_t mode;
  open_inet_addr_t ipAddr;
  uint32_t portNum;
  OPEN_PASSWORD_ENCRYPT_ALG_t encryptType = OPEN_PASSWORD_ENCRYPT_NONE;

  buffer[0] = 0;
  if (argc < 2)
  {
    printCloudManagedAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("cloud_managed_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Cloud Managed API example application");

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 0:
      if (argc != 2)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      cloudManagedOpENAPISanity(&clientHandle);
      break;

    case 1:
      if (argc != 2)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      result = openapiCloudManagedModeGet(&clientHandle, &mode);
      printf(" %s -- Cloud Managed Mode Get: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             mode==OPEN_ENABLE?"Enable":"Disable", result);
      break;

    case 2:
      if (argc != 3)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      mode = atoi(argv[2]);
      result = openapiCloudManagedModeSet(&clientHandle, mode);
      printf(" %s -- Cloud Managed Mode Set: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             mode==OPEN_ENABLE?"Enable":"Disable", result);
      break;

    case 3:
      if (argc != 2)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      result = openapiCloudManagedProxyIpGet(&clientHandle, &ipAddr, &portNum);
      (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
      printf(" %s -- Cloud Managed Proxy Server IP Address Get: %s:%d (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", buffer, portNum, result);
      break;

    case 4:
      if (argc != 4)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      openapiInetAddrGet(&clientHandle, &buffDesc, &ipAddr);
      portNum = atoi(argv[3]);
      result = openapiCloudManagedProxyIpSet(&clientHandle, &ipAddr, portNum);
      (void)openapiCloudManagedProxyIpGet(&clientHandle, &ipAddr, &portNum);
      (void)cmInetNtop(ipAddr.family, (char *)&ipAddr.addr, buffer, sizeof(buffer));
      printf(" %s -- Cloud Managed Proxy Server IP Address Set: %s:%d (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", buffer, portNum, result);
      break;

    case 5:
      if (argc != 2)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      buffDesc.pstart = buffer;
      buffDesc.size   = sizeof(buffer);
      buffDesc2.pstart = buffer2;
      buffDesc2.size   = sizeof(buffer2);
      result = openapiCloudManagedProxyUserGet(&clientHandle, &buffDesc, &encryptType, &buffDesc2);
      printf(" %s -- Cloud Managed Proxy User Credentials Get: UID: '%s'(sz:%d) PWD: '%s'(sz:%d) "
             "ENCRYPT: %s. (result = %d)\n\n", (OPEN_E_NONE==result)?"PASSED":"FAILED", 
             (char *)buffDesc.pstart, buffDesc.size, (char *)buffDesc2.pstart, buffDesc2.size,
             encryptTypePrintable(encryptType),
             result);
      break;
  
    case 6:
      if ((argc != 4) && (argc != 5))
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      if (argc == 5)
      {
        if (0 == strncmp(strupr(argv[4]), "AES", 3))
            encryptType = OPEN_PASSWORD_ENCRYPT_AES;
        else
        {
          printCloudManagedAppMenu();
          exit(1);
        }
      }
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      buffDesc2.pstart = argv[3];
      buffDesc2.size   = strlen(buffDesc2.pstart) + 1;
      result = openapiCloudManagedProxyUserSet(&clientHandle, &buffDesc, encryptType, &buffDesc2);
      (void)openapiCloudManagedProxyUserGet(&clientHandle, &buffDesc, &encryptType, &buffDesc2);
      printf(" %s -- Cloud Managed Proxy User Credentials Get: UID: '%s'(sz:%d) PWD: '%s'(sz:%d) "
             "ENCRYPT: %s. (result = %d)\n\n", (OPEN_E_NONE==result)?"PASSED":"FAILED", 
             (char *)buffDesc.pstart, buffDesc.size, (char *)buffDesc2.pstart, buffDesc2.size,
             encryptTypePrintable(encryptType),
             result);
      break;

    case 7:
      if (argc != 2)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      buffDesc.pstart = buffer;
      buffDesc.size   = sizeof(buffer);
      result = openapiCloudManagedServerUrlGet(&clientHandle, &buffDesc);
      printf(" %s -- Cloud Managed Server URL Set: Server URL:%s (sz:%d). (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc.pstart, buffDesc.size, result);
      break;

    case 8:
      if (argc != 3)
      {
        printCloudManagedAppMenu();
        exit(1);
      }
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      result = openapiCloudManagedServerUrlSet(&clientHandle, &buffDesc);
      printf(" %s -- Cloud Managed Server URL Set: Server URL:%s (sz:%d). (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc.pstart, buffDesc.size, result);
      break;

    default:
      printCloudManagedAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Cloud Managed API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
