
/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*************************************************************************
*
* @filename  instru_ft_example.c
*
* @purpose   Instrumentation - Flow Tracker (FT)  Example.
*
* @component OPEN
*
* @comments
*
* @create    11/09/2017
*
* @end
*
***************************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_instru_flow_tracker.h"

#define FT_FLW_REC_TEMPLATE_MAX_SETS       1
#define FT_DEFAULT_FLW_TEMPLATE_NAME       "template"
#define FT_IP_FIX_FLW_REC_TEMPLATE_ID      257
#define FT_IP_FIX_FLW_REC_TEMPLATE_FLD_CNT  7
#define IP_FIX_INF_SRC_IPv4_ELE_ID          8
#define IP_FIX_INF_SRC_IPv4_ELE_LEN         4
#define IP_FIX_INF_DST_IPv4_ELE_ID          12
#define IP_FIX_INF_DST_IPv4_ELE_LEN         4
#define IP_FIX_INF_SRC_PORT_ELE_ID          7
#define IP_FIX_INF_SRC_PORT_ELE_LEN         2
#define IP_FIX_INF_DST_PORT_ELE_ID         11
#define IP_FIX_INF_DST_PORT_ELE_LEN         2
#define IP_FIX_INF_IP_PROTO_ELE_ID          4
#define IP_FIX_INF_IP_PROTO_ELE_LEN         1
#define IP_FIX_INF_OCT_CNT_ELE_ID          85 
#define IP_FIX_INF_OCT_CNT_ELE_LEN          8
#define IP_FIX_INF_PKT_CNT_ELE_ID          86
#define IP_FIX_INF_PKT_CNT_ELE_LEN          4

#define FT_CL_TRANS_TYPE_IPv4_UDP           0
#define FT_TRANS_TYPE_DEFAULT               FT_CL_TRANS_TYPE_IPv4_UDP
#define FT_MAX_PKT_SIZE                     1500
#define FT_CL_PROTO_IPFIXv10                10
#define FT_FLW_GRP_MON_TYPE_FIVE_TUPLE      (1 << 0)
#define FT_PKT_TYPE_FLW_TEMPLATE            1
#define FT_FLW_OD_DEFAULT                       1
#define FT_FLW_SRC_PORT_DEFAULT                 8081 
#define FT_FLW_GRP_OD_DEFAULT                   2
#define FT_FLW_GRP_SRC_PORT_DEFAULT             8082 
#define FT_DEFAULT_FLW_TEMPLATE_PERIODICITY     10
#define FT_DEFAULT_FLW_GRP_TEMPLATE_PERIODICITY 10
#define FT_DEFAULT_FLW_GRP_REC_PERIODICITY      5

void           *cookie;
uint32_t       cb_cookie;
bool           trigger_event = false;

/* structure to hold ASIC capabilities, the information of this structure is
   used by all the functions and while configuring many resources. */

OPEN_ASIC_CAPABILITIES_t  asicInfo;
openapiClientHandle_t clientHandle;

/*****************************************************************************
* @purpose  Read basic ASIC capabilities and display them.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    This function gives the details of ASIC, for example max
*           collectors, export protocols, flow group types, max flow groups
*           and max flows etc.
* 
* @end
******************************************************************************/
open_error_t testAsicBasic(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t           result             = OPEN_E_NONE;
  OPEN_FT_CAPABILITIES_t ftAsicCapabilities;
  
  printf("Testing API -- openapiFtAsicCapabilitiesGet\n");
  printf("------------------------------------------\n\n");
    
  result = openapiFtCapabilitiesGet(clientHandle, asicId, &ftAsicCapabilities);
  if (OPEN_E_NONE == result)
  {
    printf("Displaying Basic ASIC Capabilities\n");
    printf("----------------------------------\n");
    printf("Multiple collector support      %d\n", ftAsicCapabilities.multiple_collector_support);
    printf("Maximum collectors              %d\n", ftAsicCapabilities.max_collectors);
    printf("Export protocols                %d\n", ftAsicCapabilities.export_protocols);
    printf("Flow group types                %d\n", ftAsicCapabilities.flw_grp_types);
    printf("Maximum flow groups             %d\n", ftAsicCapabilities.max_flw_grps);
    printf("Maximum flows                   %d\n", ftAsicCapabilities.max_flws);
    printf("Flow sampling support           %d\n", ftAsicCapabilities.flw_sampling_support);
  }
  return result;
}

/*****************************************************************************
* @purpose  Function to test if FT can be enabled/disabled and print the status. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtConfig(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  OPEN_FT_CONFIG_t ftConfig;
  OPEN_FT_CONFIG_t ftReadConfig;
  open_error_t     set_result = OPEN_E_NONE;

  memset(&ftConfig, 0, sizeof(OPEN_FT_CONFIG_t));
  memset(&ftReadConfig, 0, sizeof(OPEN_FT_CONFIG_t));

  printf("\n\n");
  printf("Testing API -- openapiFtConfigSet\n");
  printf("------------------------------------\n");

  /* We will try to enable FT */
  ftConfig.enable                            = 1;
  ftConfig.flw_exp_ipfix.periodicity         = FT_DEFAULT_FLW_TEMPLATE_PERIODICITY;
  ftConfig.flw_exp_ipfix.src_port            = FT_FLW_SRC_PORT_DEFAULT;
  ftConfig.flw_exp_ipfix.obs_dmn             = FT_FLW_OD_DEFAULT;
  ftConfig.flw_grp_exp_ipfix.periodicity     = FT_DEFAULT_FLW_GRP_TEMPLATE_PERIODICITY;
  ftConfig.flw_grp_exp_ipfix.rec_periodicity = FT_DEFAULT_FLW_GRP_REC_PERIODICITY;
  ftConfig.flw_grp_exp_ipfix.src_port        = FT_FLW_GRP_SRC_PORT_DEFAULT;
  ftConfig.flw_grp_exp_ipfix.obs_dmn         = FT_FLW_GRP_OD_DEFAULT;

  set_result = openapiFtConfigSet(clientHandle, asicId, &ftConfig);
  if (OPEN_E_NONE != set_result)
  {
    printf("Setting basic FT configuration failed, result = %d\n", set_result);
    return OPEN_E_FAIL;
  }  

  /* We will try to disable FT */
  ftConfig.enable                            = 0;

  set_result = openapiFtConfigSet(clientHandle, asicId, &ftConfig);
  if (OPEN_E_NONE != set_result)
  {
    printf("Setting basic FT configuration failed, result = %d\n", set_result);
    return OPEN_E_FAIL;
  }  

  /* Re enable FT */
  ftConfig.enable                            = 1;

  set_result = openapiFtConfigSet(clientHandle, asicId, &ftConfig);
  if (OPEN_E_NONE != set_result)
  {
    printf("Setting basic FT configuration failed, result = %d\n", set_result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE; 
}

/*****************************************************************************
* @purpose  Function to test FT collector config create. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtCollectorConfigCreate(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t      result = OPEN_E_NONE;
  OPEN_FT_CL_INFO_t cl_info;

  printf("\n\n");
  printf("Testing API -- openapiFtCollectorConfigCreate\n");
  printf("------------------------------------\n");

  memset(&cl_info, 0, sizeof(cl_info));
  cl_info.configured = true;
  cl_info.trans_type =  FT_TRANS_TYPE_DEFAULT;
  cl_info.max_pkt_len = FT_MAX_PKT_SIZE;
  strncpy(&(cl_info.config.name[0]), "collector", strlen("collector"));
  cl_info.config.proto = FT_CL_PROTO_IPFIXv10;
  cl_info.config.port = 2075;
  cl_info.config.addr.addr.ipv4 = 0x0a0b0c0d;

  result = openapiFtCollectorConfigCreate(clientHandle,
                                          asicId,
                                          &cl_info);
  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker collector configuration create failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test FT collector config remove. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtCollectorConfigRemove(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t  result = OPEN_E_NONE;
  char name[10] = "collector";
  open_buffdesc clName;

  printf("\n\n");
  printf("Testing API -- openapiFtCollectorConfigRemove\n");
  printf("------------------------------------\n");

  clName.pstart = name;
  clName.size = strlen(name) + 1;

  result = openapiFtCollectorConfigRemove(clientHandle,
                                          asicId,
                                          &clName);
  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker collector configuration remove failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test FT flow group config create. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtFlowGroupConfigCreate(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t     result = OPEN_E_NONE;
  OPEN_FT_FLW_GRP_CONFIG_t flowGroup;

  printf("\n\n");
  printf("Testing API -- openapiFtFlowGroupConfigCreate\n");
  printf("------------------------------------\n");

  memset(&flowGroup, 0, sizeof(flowGroup));

  strncpy(&(flowGroup.name[0]), "flowgrp", strlen("flowgrp"));
  flowGroup.type = FT_FLW_GRP_MON_TYPE_FIVE_TUPLE;
  flowGroup.mon_params.five_tuple.src_ip.family = AF_INET;
  flowGroup.mon_params.five_tuple.src_ip.host.ip.addr = 0x11223344;
  flowGroup.mon_params.five_tuple.src_ip.host.ip.mask = 0xffff0000;
  flowGroup.mon_params.five_tuple.dst_ip.family = AF_INET;
  flowGroup.mon_params.five_tuple.dst_ip.host.ip.addr = 0x55667788;
  flowGroup.mon_params.five_tuple.dst_ip.host.ip.mask = 0xffff0000;
  flowGroup.mon_params.five_tuple.protocol = 0x11;
  flowGroup.mon_params.five_tuple.src_port = 1234;
  flowGroup.mon_params.five_tuple.dst_port = 5678;
  flowGroup.mon_params.five_tuple.tuple_mask = 0x1f;
  strncpy (&(flowGroup.cl_name_list[0][0]), "collector", strlen("collector"));
  flowGroup.cl_num = 1;

  result = openapiFtFlowGroupConfigCreate(clientHandle,
                                          asicId,
                                          &flowGroup);
  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker flow group configuration create failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test FT flow group config remove. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtFlowGroupConfigRemove(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t  result = OPEN_E_NONE;
  char name[10] = "flowgrp";
  open_buffdesc grpName;

  printf("\n\n");
  printf("Testing API -- openapiFtFlowGroupConfigRemove\n");
  printf("------------------------------------\n");

  grpName.pstart = name;
  grpName.size = strlen(name) + 1;

  result = openapiFtFlowGroupConfigRemove(clientHandle,
                                          asicId,
                                          &grpName);
  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker flow group configuration remove failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test FT flow template config create. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtFlowTemplateConfigCreate(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t     result = OPEN_E_NONE;
  OPEN_FT_FLW_REC_TEMPLATE_t flowTemplate;

  printf("\n\n");
  printf("Testing API -- openapiFtFlowTemplateConfigCreate\n");
  printf("------------------------------------\n");

  memset(&flowTemplate, 0x0, sizeof(flowTemplate));
  strncpy(&(flowTemplate.data[0].name[0]), "template", strlen("template"));
  flowTemplate.num_templates                       = FT_FLW_REC_TEMPLATE_MAX_SETS;
  flowTemplate.data[0].template_hdr_id             = FT_IP_FIX_FLW_REC_TEMPLATE_ID;
  flowTemplate.data[0].num_ele                     = FT_IP_FIX_FLW_REC_TEMPLATE_FLD_CNT;
  
  flowTemplate.data[0].ele_info[0].ent_ele_id      = IP_FIX_INF_SRC_IPv4_ELE_ID;
  flowTemplate.data[0].ele_info[0].len             = IP_FIX_INF_SRC_IPv4_ELE_LEN;
  flowTemplate.data[0].ele_info[0].ent_num_present = false;
  flowTemplate.data[0].ele_info[0].ent_num         = 0;
  
  flowTemplate.data[0].ele_info[1].ent_ele_id      = IP_FIX_INF_DST_IPv4_ELE_ID;
  flowTemplate.data[0].ele_info[1].len             = IP_FIX_INF_DST_IPv4_ELE_LEN;
  flowTemplate.data[0].ele_info[1].ent_num_present = false;
  flowTemplate.data[0].ele_info[1].ent_num         = 0;
  
  flowTemplate.data[0].ele_info[2].ent_ele_id      = IP_FIX_INF_SRC_PORT_ELE_ID;
  flowTemplate.data[0].ele_info[2].len             = IP_FIX_INF_SRC_PORT_ELE_LEN;
  flowTemplate.data[0].ele_info[2].ent_num_present = false;
  flowTemplate.data[0].ele_info[2].ent_num         = 0;
  
  flowTemplate.data[0].ele_info[3].ent_ele_id      = IP_FIX_INF_DST_PORT_ELE_ID;
  flowTemplate.data[0].ele_info[3].len             = IP_FIX_INF_DST_PORT_ELE_LEN;
  flowTemplate.data[0].ele_info[3].ent_num_present = false;
  flowTemplate.data[0].ele_info[3].ent_num         = 0;
  
  flowTemplate.data[0].ele_info[4].ent_ele_id      = IP_FIX_INF_IP_PROTO_ELE_ID;
  flowTemplate.data[0].ele_info[4].len             = IP_FIX_INF_IP_PROTO_ELE_LEN;
  flowTemplate.data[0].ele_info[4].ent_num_present = false;
  flowTemplate.data[0].ele_info[4].ent_num         = 0;
  
  flowTemplate.data[0].ele_info[5].ent_ele_id      = IP_FIX_INF_PKT_CNT_ELE_ID;
  flowTemplate.data[0].ele_info[5].len             = IP_FIX_INF_PKT_CNT_ELE_LEN;
  flowTemplate.data[0].ele_info[5].ent_num_present = false;
  flowTemplate.data[0].ele_info[5].ent_num         = 0;
  
  flowTemplate.data[0].ele_info[6].ent_ele_id      = IP_FIX_INF_OCT_CNT_ELE_ID;
  flowTemplate.data[0].ele_info[6].len             = IP_FIX_INF_OCT_CNT_ELE_LEN;
  flowTemplate.data[0].ele_info[6].ent_num_present = false;
  flowTemplate.data[0].ele_info[6].ent_num         = 0;

  result = openapiFtFlowTemplateConfigCreate(clientHandle,
                                             asicId,
                                             &flowTemplate);
  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker flow template configuration create failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test FT flow template config remove. 
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtFlowTemplateConfigRemove(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t  result = OPEN_E_NONE;
  char name[10] = FT_DEFAULT_FLW_TEMPLATE_NAME;
  open_buffdesc templateName;

  printf("\n\n");
  printf("Testing API -- openapiFtFlowTemplateConfigRemove\n");
  printf("------------------------------------\n");

  templateName.pstart = name;
  templateName.size = strlen(name) + 1;

  result = openapiFtFlowTemplateConfigRemove(clientHandle,
                                             asicId,
                                             &templateName);
  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker flow group configuration remove failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test sending IPFIX packet to collector.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtPacketSend(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t  result = OPEN_E_NONE;
  open_buffdesc clName, pktPayload;
  char name[10] = "collector";
  unsigned int length = 98;
  unsigned char data[98] = { 0x6c, 0x41, 0x6a, 0xf4, 0x0c, 0x46, 0x00, 0x18,
                             0x23, 0x30, 0xea, 0x52, 0x08, 0x00, 0x45, 0xb8,
                             0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x10, 0x11,
                             0xea, 0x65, 0x0a, 0x82, 0x54, 0x09, 0x0a, 0x82,
                             0x56, 0x73, 0x1f, 0x91, 0x08, 0x07, 0x00, 0x3c,
                             0xd2, 0x6f, 0x00, 0x0a, 0x00, 0x34, 0x59, 0xc8,
                             0xe9, 0xc9, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
                             0x00, 0x01, 0x00, 0x02, 0x00, 0x24, 0x01, 0x01,
                             0x00, 0x07, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0c,
                             0x00, 0x04, 0x00, 0x07, 0x00, 0x02, 0x00, 0x0b,
                             0x00, 0x02, 0x00, 0x04, 0x00, 0x01, 0x00, 0x56,
                             0x00, 0x04, 0x00, 0x55, 0x00, 0x08, 0xe1, 0x6b,
                             0x92, 0x7b };

  printf("\n\n");
  printf("Testing API -- openapiFtPktSend\n");
  printf("------------------------------------\n");

  clName.pstart = name;
  clName.size = strlen(name) + 1;

  pktPayload.pstart = data;
  pktPayload.size = length;

  result = openapiFtPktSend(clientHandle,
                            asicId, &clName,
                            FT_PKT_TYPE_FLW_TEMPLATE,
                            length, &pktPayload);

  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker sending IPFIX packet to collector failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test get flow tracker flow group statistics.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtFlowGroupStatsGet(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t  result = OPEN_E_NONE;
  char name[10] = "flowgrp";
  open_buffdesc grpName;
  OPEN_FT_FLW_GRP_STATS_t stat;

  printf("\n\n");
  printf("Testing API -- openapiFtFlowGroupStatsGet\n");
  printf("------------------------------------\n");

  grpName.pstart = name;
  grpName.size = strlen(name) + 1;

  result = openapiFtFlowGroupStatsGet(clientHandle,
                                      asicId, &grpName, &stat);

  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker getting flow group statistics failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

/*****************************************************************************
* @purpose  Function to test clear flow tracker flow group statistics.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    asicId         @b{(input)}  ASIC number in the system.
* 
* @returns  none
* 
* @notes    FT (Flow tracker) needs to be enabled. 
* 
* @end
******************************************************************************/
open_error_t testFtFlowGroupStatsClear(openapiClientHandle_t *clientHandle, uint32_t asicId)
{
  open_error_t  result = OPEN_E_NONE;
  char name[10] = "flowgrp";
  open_buffdesc grpName;

  printf("\n\n");
  printf("Testing API -- openapiFtFlowGroupStatsClear\n");
  printf("------------------------------------\n");

  grpName.pstart = name;
  grpName.size = strlen(name) + 1;

  result = openapiFtFlowGroupStatsClear(clientHandle,
                                        asicId, &grpName);

  if (OPEN_E_NONE != result)
  {
    printf("Flow tracker clearing flow group statistics failed, result = %d\n", result);
    return OPEN_E_FAIL;
  }  

  return OPEN_E_NONE;
}

int main (int argc, char **argv)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t asicId = 0;
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("instru_ft_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. result = %d exit test\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting FT API example application");

  printf ("\n");

  /* Execute sanity tests */
  printf ("Begin Sanity tests...\n");

  /* Basic ASIC capabilities, here we try to read the basic ASIC capabilities
     like max collectors, export protocols, flow group types, max flow groups
     and max flows, failing to read the basic capabilities can be understood
     as fatal error and test exits */

  result = testAsicBasic(&clientHandle, asicId);
  if (result != OPEN_E_NONE)
  {
    printf("Reading of ASIC capabilities failed, exit test\n");
    exit(1);  
  }
  
  /* Test if FT can be enabled/disabled */
  result = testFtConfig(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT collector config create */
  result = testFtCollectorConfigCreate(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT flow template config create */
  result = testFtFlowTemplateConfigCreate(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT flow group config create */
  result = testFtFlowGroupConfigCreate(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT packet send */
  result = testFtPacketSend(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT flow group stats get */
  result = testFtFlowGroupStatsGet(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT flow group stats clear */
  result = testFtFlowGroupStatsClear(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT flow group config delete */
  result = testFtFlowGroupConfigRemove(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT flow template config delete */
  result = testFtFlowTemplateConfigRemove(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  /* Test FT collector config delete */
  result = testFtCollectorConfigRemove(&clientHandle, asicId);
  if (result == OPEN_E_FAIL)
  {
    exit(1);  
  }

  printf ("\nComplete.\n");
  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping FT API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

