
/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  interface_example.c
*
* @purpose   Switch Interface APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    10/15/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_if.h"
#include "openapi_boxs.h"


void getPhysicalModes(uint64_t portCapability)
{
  printf("------------------------------------------------\n");
  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_AUTO_NEG)) == (OPEN_PHY_CAP_PORTSPEED_AUTO_NEG))
  {
    printf("Auto Negotiate \n");
  }

  if (( portCapability) & (OPEN_PHY_CAP_PORTSPEED_HALF_10))
  {
    printf("10 Mbps Half Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_10))
  {
    printf("10 Mbps Full Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_HALF_100))
  {
    printf("100 Mbps Half Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_100))
  {
    printf("100 Mbps Full Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_HALF_1000))
  {
    printf("1000 Mbps Half Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_1000))
  {
    printf("1000 Mbps Full Duplex \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_2P5G))
  {
    printf("2.5 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_5G))
  {
    printf("5 Gbps Full \n");
  }

  if ((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_10G))
  {
    printf("10 Gbps Full \n");
  }

  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_20G)) == (OPEN_PHY_CAP_PORTSPEED_FULL_20G))
  {
    printf("20 Gbps Full \n");
  }

  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_25G)) == (OPEN_PHY_CAP_PORTSPEED_FULL_25G))
  {
    printf("25 Gbps Full \n");
  }

  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_40G)) == (OPEN_PHY_CAP_PORTSPEED_FULL_40G))
  {
    printf("40 Gbps Full \n");
  }

  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_50G)) == (OPEN_PHY_CAP_PORTSPEED_FULL_50G))
  {
    printf("50 Gbps Full \n");
  }

  if (((portCapability) & (OPEN_PHY_CAP_PORTSPEED_FULL_100G)) == (OPEN_PHY_CAP_PORTSPEED_FULL_100G))
  {
    printf("100 Gbps Full \n");
  }
}

static char *reasonToStr(OPEN_DISABLE_REASON_t reason)
{
  char *ret = "Unknown reason";

  switch(reason)
  {
    case OPEN_DISABLE_REASON_NONE:
      ret = "OPEN_DISABLE_REASON_NONE";
      break;
    case OPEN_DISABLE_REASON_UDLD:
      ret = "OPEN_DISABLE_REASON_UDLD";
      break;
    case OPEN_DISABLE_REASON_STP:
      ret = "OPEN_DISABLE_REASON_STP";
      break;
    case OPEN_DISABLE_REASON_BPDUGUARD:
      ret = "OPEN_DISABLE_REASON_BPDUGUARD";
      break;
    case OPEN_DISABLE_REASON_BPDUSTORM:
      ret = "OPEN_DISABLE_REASON_BPDUSTORM";
      break;
    case OPEN_DISABLE_REASON_DHCPSNOOPING:
      ret = "OPEN_DISABLE_REASON_DHCPSNOOPING";
      break;
    case OPEN_DISABLE_REASON_ARPINSPECTION:
      ret = "OPEN_DISABLE_REASON_ARPINSPECTION";
      break;
    case OPEN_DISABLE_REASON_STORMCONTROL_BCAST:
      ret = "OPEN_DISABLE_REASON_STORMCONTROL_BCAST";
      break;
    case OPEN_DISABLE_REASON_STORMCONTROL_MCAST:
      ret = "OPEN_DISABLE_REASON_STORMCONTROL_MCAST";
      break;
    case OPEN_DISABLE_REASON_STORMCONTROL_UCAST:
      ret = "OPEN_DISABLE_REASON_STORMCONTROL_UCAST";
      break;
    case OPEN_DISABLE_REASON_XCEIVER:
      ret = "OPEN_DISABLE_REASON_XCEIVER";
      break;
    case OPEN_DISABLE_REASON_XCEIVERPLUS:
      ret = "OPEN_DISABLE_REASON_XCEIVERPLUS";
      break;
    case OPEN_DISABLE_REASON_LOOPPROTECT:
      ret = "OPEN_DISABLE_REASON_LOOPPROTECT";
      break;
    case OPEN_DISABLE_REASON_PML:
      ret = "OPEN_DISABLE_REASON_PML";
      break;
    case OPEN_DISABLE_REASON_DOS:
      ret = "OPEN_DISABLE_REASON_DOS";
      break;
    case OPEN_DISABLE_REASON_LINK_FLAP:
      ret = "OPEN_DISABLE_REASON_LINK_FLAP";
      break;
    case OPEN_DISABLE_REASON_AUTHMGR_AUTH_VIOLATION:
      ret = "OPEN_DISABLE_REASON_AUTHMGR_AUTH_VIOLATION";
      break;
    default:
      break;
  }
  return ret;
}

static char *specialPortTypeToStr(OPEN_SPECIAL_PORT_TYPE_t specialPortType)
{
  char *ret = "Unknown";

  switch(specialPortType)
  {
    case OPEN_SPECIAL_PORT_TYPE_NORMAL:
      ret = "Normal port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_TRUNK_MEMBER:
      ret = "Trunk member";
      break;
    case OPEN_SPECIAL_PORT_TYPE_MIRRORED_PORT:
      ret = "Mirrored port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_PROBE_PORT:
      ret = "Probe port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_L2TUNNEL_PORT:
      ret = "L2 tunnel port";
      break;
    case OPEN_SPECIAL_PORT_TYPE_SERVICE_PORT:
      ret = "Service port";
      break;
    default:
      break;
  }
  return ret;
}

static char *mediaTypeToStr(OPEN_PORT_CABLE_MEDIA_TYPE_t mediaType, OPEN_SFP_MODULE_TYPE_t moduleType)
{
  char *ret = "Unknown";

  if (mediaType == OPEN_PORT_CABLE_MEDIA_TYPE_COPPER)
  {
    ret = "Copper";
  }
  else if (mediaType == OPEN_PORT_CABLE_MEDIA_TYPE_FIBER)
  {
    switch(moduleType)
    {
      case OPEN_MODULE_NONE:
        ret = "None";
        break;
      case OPEN_MODULE_SFP:
        ret = "SFP";
        break;
      case OPEN_MODULE_SFPPLUS:
        ret = "SFP+";
        break;
      case OPEN_MODULE_QSFP:
        ret = "QSFP";
        break;
      case OPEN_MODULE_DAC:
        ret = "Direct attach cable";
        break;
      case OPEN_MODULE_AX741:
        ret = "XFP 10G plugin module (AX741)";
        break;
      case OPEN_MODULE_AX742:
        ret = "Stacking module (AX742)";
        break;
      case OPEN_MODULE_AX743:
        ret = "SFP+ plugin module (AX743)";
        break;
      case OPEN_MODULE_AX744:
        ret = "CX4 plugin module (AX744)";
        break;
      case OPEN_MODULE_AX745:
        ret = "Copper 10G plugin module (AX745)";
        break;
      default:
        break;
    }
  }
  return ret;
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         Interface OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/

int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  uint32_t IfNum, mtu, mtu2, efs;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc intfName;
  char intfNameStr[256];
  char buf[128];
  open_buffdesc mac;
  OPEN_PORT_SPEEDS_t speed;
  uint32_t duplex;
  uint32_t autoneg;
  uint32_t value = 0;
  OPEN_CONTROL_t admin_state;
  uint64_t portCapability;
  OPEN_DISABLE_REASON_t reason;
  OPEN_SPECIAL_PORT_TYPE_t specialPortType;
  OPEN_PORT_CABLE_MEDIA_TYPE_t mediaType = OPEN_PORT_CABLE_MEDIA_TYPE_NONE;
  OPEN_SFP_MODULE_TYPE_t moduleType = OPEN_MODULE_NONE;
  OPEN_BOOL_t rxStatus, txStatus;

  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("interface_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting Interface API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }

  printf ("\n");

  /* exercise various OPEN API interface functions */
  do
  { 
    if ((result = openapiIfFirstGet (&clientHandle, OPEN_INTF_TYPE_PHY,
                                     &IfNum)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get first physical interface. (result = %d)\n",result);
      break;
    }

    intfName.pstart = intfNameStr;
    intfName.size = sizeof(intfNameStr);
    if ((result = openapiIfNameGet(&clientHandle, IfNum, &intfName)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get first physical interface name. (result = %d)\n",result);
      break;
    }

    printf("Using physical interface %d, name %s\n", IfNum, (char *)intfName.pstart);

    if ((result = openapiIfAdminStateGet(&clientHandle, IfNum, &admin_state)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get admin state physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if (admin_state == OPEN_ENABLE) sprintf(buf,"ENABLE");
      else sprintf(buf,"DISABLE");
      printf("physical interface %d get admin-state : %s\n", IfNum, buf);
    }

    admin_state = (admin_state == OPEN_ENABLE) ? OPEN_DISABLE : OPEN_ENABLE;

    if ((result = openapiIfAdminStateSet(&clientHandle, IfNum, admin_state)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set admin state physical interface. (result = %d)\n",result);
    }

    if ((result = openapiIfAdminStateGet(&clientHandle, IfNum, &admin_state)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get admin state physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if (admin_state == OPEN_ENABLE) sprintf(buf,"ENABLE");
      else sprintf(buf,"DISABLE");
      printf("physical interface %d set admin-state : %s\n", IfNum, buf);
    }

    if ((result = openapiIfSpeedGet(&clientHandle, IfNum, &speed)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get speed of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      switch (speed)
      {
        case OPEN_PORTCTRL_PORTSPEED_AUTO_NEG:
          sprintf(buf,"AUTO");
          break;
        case OPEN_PORTCTRL_PORTSPEED_HALF_100TX:
          sprintf(buf,"100 half");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_100TX:
          sprintf(buf,"100 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_HALF_10T:
          sprintf(buf,"10 half");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_10T:
          sprintf(buf,"10 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_100FX:
          sprintf(buf,"100 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_1000SX:
          sprintf(buf,"1000 full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_10GSX:
          sprintf(buf,"10G full");
          break;
        case OPEN_PORTCTRL_PORTSPEED_FULL_40GSX:
          sprintf(buf,"40G full");
          break;
        default:
          sprintf(buf,"UNKNOWN");
          break;
      }
      printf("physical interface %d speed %s\n",IfNum,buf);
    }

    if ((result = openapiIfDuplexModeGet(&clientHandle, IfNum, &duplex)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get duplex mode of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      switch (duplex)
      {
        case OPEN_PORT_DUPLEX_HALF:
          sprintf(buf,"half-duplex");
          break;
        case OPEN_PORT_DUPLEX_FULL:
          sprintf(buf,"full-duplex");
          break;
        case OPEN_PORT_DUPLEX_UNKNOWN:
          sprintf(buf,"UNKNOWN");
          break;
        default:
          sprintf(buf,"UNKNOWN");
          break;
      }
      printf("physical interface %d duplex mode %s\n",IfNum,buf);
    }

    if ((result = openapiIfAutoNegGet(&clientHandle, IfNum, &autoneg)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get auto-negotiation mode of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if (autoneg == OPEN_PORT_AUTO_NEG_ENABLE)
        sprintf(buf,"ENABLE");
      else if(autoneg == OPEN_PORT_AUTO_NEG_DISABLE)
        sprintf(buf,"DISABLE");
      else
        sprintf(buf,"UNKNOWN");

      printf("physical interface %d auto-negotiation mode : %s\n",IfNum,buf);
    }

    printf("setting auto-negotiation to DISABLE on interface %d \n",IfNum);
    autoneg = OPEN_PORT_AUTO_NEG_DISABLE;
    if ((result = openapiIfAutoNegSet(&clientHandle, IfNum, autoneg)) != OPEN_E_NONE)
    {
      if (result == OPEN_E_UNAVAIL)
      {
        printf(" Cannot set auto-negotiation on an unsupported interface. (result = %d)\n", result);   
      }
      else
      {
        printf("  Bad return code trying to set auto-negotiation mode of  physical interface. (result = %d)\n",result);
      }
    }
    else
    {
      sleep(5);
      if ((result = openapiIfAutoNegGet(&clientHandle, IfNum, &autoneg)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get auto-negotiation mode of  physical interface. (result = %d)\n",result);
      }
      else
      {
        memset (buf, 0, sizeof(buf));
        if (autoneg == OPEN_PORT_AUTO_NEG_ENABLE)
          sprintf(buf,"ENABLE");
        else if(autoneg == OPEN_PORT_AUTO_NEG_DISABLE)
          sprintf(buf,"DISABLE");
        else
          sprintf(buf,"UNKNOWN");

        printf("physical interface %d auto-negotiation mode : %s\n",IfNum,buf);
      }

      printf("setting the speed to 100M full-duplex on interface %d\n",IfNum);
      speed = OPEN_PORTCTRL_PORTSPEED_FULL_100TX;
      if ((result = openapiIfSpeedSet(&clientHandle, IfNum, speed)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to set speed of  physical interface. (result = %d)\n",result);
      }
      else
      {
        sleep(5);
        if ((result = openapiIfSpeedGet(&clientHandle, IfNum, &speed)) != OPEN_E_NONE)
        {
          printf("  Bad return code trying to get speed of  physical interface. (result = %d)\n",result);
        }
        else
        {
          memset (buf, 0, sizeof(buf));
          switch (speed)
          {
            case OPEN_PORTCTRL_PORTSPEED_AUTO_NEG:
              sprintf(buf,"AUTO");
              break;
            case OPEN_PORTCTRL_PORTSPEED_HALF_100TX:
              sprintf(buf,"100 half");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_100TX:
              sprintf(buf,"100 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_HALF_10T:
              sprintf(buf,"10 half");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_10T:
              sprintf(buf,"10 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_100FX:
              sprintf(buf,"100 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_1000SX:
              sprintf(buf,"1000 full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_10GSX:
              sprintf(buf,"10G full");
              break;
            case OPEN_PORTCTRL_PORTSPEED_FULL_40GSX:
              sprintf(buf,"40G full");
              break;
            default:
              sprintf(buf,"unknown");
              break;
          }
          printf("physical interface %d speed %s\n",IfNum,buf);
        }
      }
    }

    if ((result = openapiIfEthernetHeaderSizeGet(&clientHandle, IfNum, &efs)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get ethernet frame size of physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d ethernet header size : %d\n", IfNum, efs);
    }

    if ((result = openapiIfMtuSizeGet(&clientHandle, IfNum, &mtu)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get mtu size of physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d MTU size : %d\n", IfNum, mtu);
    }

    mtu = 9216;
    printf("setting the MTU to %d\n", mtu);
    if ((result = openapiIfMtuSizeSetV2(&clientHandle, IfNum, mtu)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set mtu size of  physical interface. (result = %d)\n",result);
    }
    else
    {
      sleep(5);
      if ((result = openapiIfMtuSizeGet(&clientHandle, IfNum, &mtu2)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get mtu size of  physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d MTU size : %d\n",IfNum, mtu2);
        printf("MTU is%sequal to the set MTU\n", (mtu == mtu2 ? " " : " not "));
      }
    }
    
    mtu = 8148;
    printf("setting the max frame size to %d\n", mtu);
    if ((result = openapiIfMaxFrameSizeSet(&clientHandle, IfNum, mtu)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set max frame size of physical interface. (result = %d)\n",result);
    }
    else
    {
      sleep(5);
      if ((result = openapiIfMtuSizeGet(&clientHandle, IfNum, &mtu2)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get mtu size of physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d MTU size : %d\n",IfNum,mtu2);
      }
      if ((result = openapiIfMaxFrameSizeGet(&clientHandle, IfNum, &mtu2)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get max frame size of physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d max frame size : %d\n", IfNum, mtu2);
        printf("Max frame size is%sequal to the set max frame size\n", (mtu == mtu2 ? " " : " not "));
      }
    }

    memset (buf, 0, sizeof(buf));
    mac.pstart = buf;
    mac.size = sizeof(buf);
    if ((result = openapiIfMacAddrGet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get mac-address of  physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d mac-address : %s\n",IfNum,buf);
    }

    memset (buf, 0, sizeof(buf));
    mac.pstart = buf;
    mac.size = sizeof(buf);
    if ((result = openapiIfDescrGet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get description of  physical interface. (result = %d)\n",result);
    }
    else
    {
      printf("physical interface %d description : %s\n",IfNum,buf);
    }

    memset (buf, 0, sizeof(buf));
    sprintf(buf,"test_port_%d_description",IfNum);
    mac.pstart = buf;
    mac.size = strlen(buf) + 1;
    printf("setting the interface %d description to %s\n",IfNum,buf);
    if ((result = openapiIfDescrSet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to set description of  physical interface. (result = %d)\n",result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      mac.pstart = buf;
      mac.size = sizeof(buf);
      if ((result = openapiIfDescrGet(&clientHandle, IfNum, &mac)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get description of  physical interface. (result = %d)\n",result);
      }
      else
      {
        printf("physical interface %d description : %s\n",IfNum,buf);
      }
    }

    if ((result = openapiIfDisableReasonGet(&clientHandle, IfNum, &reason)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get disable reason. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d disable reason: %d (%s)\n", IfNum, reason, reasonToStr(reason));
    }

    if ((result = openapiIfSpecialPortTypeGet(&clientHandle, IfNum, &specialPortType)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get special port type. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d special port type designation: %d (%s)\n", IfNum, specialPortType, specialPortTypeToStr(specialPortType));
    }

    if ((result = openapiIfActiveMediaTypeGet(&clientHandle, IfNum, &mediaType)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get active media type. result = %d\n", result);
    }
    else if ((mediaType == OPEN_PORT_CABLE_MEDIA_TYPE_FIBER) &&
             ((result = openapiBoxsSfpTypeGet(&clientHandle, IfNum, &moduleType)) != OPEN_E_NONE))
    {
      printf("Bad return code trying to get SFP type. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d active media type: %d (%s)\n", IfNum, mediaType, mediaTypeToStr(mediaType, moduleType));
    }

    if ((result = openapiIfPauseStatusGet(&clientHandle, IfNum, &rxStatus, &txStatus)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get pause status. result = %d\n", result);
    }
    else
    {
      memset (buf, 0, sizeof(buf));
      if ((rxStatus == OPEN_TRUE) || (txStatus == OPEN_TRUE))
      {
        sprintf(buf,"ACTIVE");
      }
      else
      {
        sprintf(buf,"INACTIVE");
      }
      printf("Physical interface %d flow control status: %s\n", IfNum, buf);
    }

    printf("--- interface statistic---\n");
    sleep(2);

    if ((result = openapiIfLastChangeGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get time of last link state change. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Time of last link state change: %d\n", IfNum, value);
    }
    value = 0;
    if ((result = openapiIfInOctetsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get total number of octets. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Total number of octets: %d\n", IfNum, value);
    }
    value = 0;
    if ((result = openapiIfInUcastPktsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of packets, delivered by this sub-layer to a higher sub-layer. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of packets, delivered by this sub-layer to a higher sub-layer: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfInDiscardsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of inbound packets which were chosen to be discarded. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of inbound packets which were chosen to be discarded: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfInErrorsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get In errors counter. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d In errors counter: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfInUnknownProtosGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of packets received via the interface \nwhich were discarded because of an unknown or unsupported protocol. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of packets received via the interface \nwhich were discarded because of an unknown or unsupported protocol: %d\n", IfNum, value);
    }
   
    value = 0;
    if ((result = openapiIfOutOctetsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get total number of octets transmitted out of the interface, including framing characters. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Total number of octets transmitted out of the interface, including framing characters: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutUcastPktsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get total number of packets that higher-level protocols requested be transmitted. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Total number of packets that higher-level protocols requested be transmitted: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutDiscardsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of outbound packets which were chosen to be discarded. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of outbound packets which were chosen to be discarded: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutDiscardsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get number of outbound packets which were chosen to be discarded. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Number of outbound packets which were chosen to be discarded: %d\n", IfNum, value);
    }

    value = 0;
    if ((result = openapiIfOutErrorsGet(&clientHandle, IfNum, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get out errors counter. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d Out errors counter: %d\n", IfNum, value);
    }

    if ((result = openapiIfCapabilitiesGet(&clientHandle, IfNum, &portCapability)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get port capabilities. result = %d\n", result);
    }
    else
    {
      printf("Physical interface %d is capable of following speeds: \n", IfNum);
      getPhysicalModes(portCapability);
    }

    break;
  }
  while (0);
  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0,
               "Stopping Interface API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
