
/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  lag_example.c
*
* @purpose   LAG APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    10/15/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_lag.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         Lag OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  uint32_t ifNum, minlinks, maxMem, lagIfNum;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  char portChannelMembersListStr[32];
  char buf[20];
  open_buffdesc name;
  uint32_t count = 0, mode;
  uint32_t minThreshold, maxThreshold;
  uint32_t *portChannelMembers;
  open_buffdesc portChannelMembersList;
  open_buffdesc portChannelMembersDesc;
  OPEN_LAG_HASH_MODE_t hashmode;
  OPEN_CONTROL_t  lacpMode;
  OPEN_CONTROL_t  state;

  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("lag_example", &clientHandle)) != OPEN_E_NONE)
    {
      printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
      exit (2);
    }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
    {
      sleep (1);
    }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting LAG API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }

  printf ("\n");
  /* exercise various OPEN API LAG functions */
  do
    {
      /*Testing max no of member ports */
      printf("Testing the maximum number of members per LAG supported on the platform\n");
      if (openapiLagMaxMemberCountGet (&clientHandle, &maxMem) != OPEN_E_NONE)
      {
	    printf ("Failed to get the maximum number of member ports \n");
      }
      else
      {
	    printf("Maximum of [ %u ] member ports supported on this platform\n", maxMem);
      }
      printf ("Press enter to continue...\n");
      getchar ();

      /*Testing LAG member ports ADD and DELETE */
      printf("Attempting to get the first LAG interface to ADD members to it.\n");
      if ((result = openapiIfFirstGet (&clientHandle, OPEN_INTF_TYPE_LAG,
		                       &lagIfNum)) != OPEN_E_NONE)
	  {
	    printf("  Bad return code trying to get first LAG interface. (result = %d)\n", result);
	    break;
	  }
      printf ("Using the first LAG interface %d\n", lagIfNum);
      printf ("Adding ports from 1 to %d to LAG interface %d.\n", maxMem, lagIfNum);
      for (ifNum = 1; ifNum < (maxMem + 1); ifNum++)
	  {
	    if (openapiLagPortAdd (&clientHandle, lagIfNum, ifNum) != OPEN_E_NONE)
	    {
	      printf ("Failed to add interface %u to LAG interface %d\n", ifNum, lagIfNum);
	      break;
	    }
	  }
      printf ("Getting the members list on LAG interface %d after ADD\n", lagIfNum);

      memset(portChannelMembersListStr, 0, sizeof(portChannelMembersListStr));
      portChannelMembersList.pstart = portChannelMembersListStr;
      portChannelMembersList.size = sizeof(portChannelMembersListStr);

      count = maxMem;
      if (openapiLagMembersListGet(&clientHandle, lagIfNum, &count,
                               &portChannelMembersList) != OPEN_E_NONE)
	  {
	    printf ("Failed to get the member ports for the LAG interface %d\n", lagIfNum);
	  }
      else
      {
        memcpy(portChannelMembersListStr, portChannelMembersList.pstart, portChannelMembersList.size);
	    printf ("Comma delimited list of member ports for LAG interface %d (%d) (%s):\n", lagIfNum, count, portChannelMembersListStr);
	  }

      count = maxMem;
      portChannelMembers = (uint32_t *) malloc(maxMem * (sizeof(uint32_t)));
      if (!portChannelMembers)
      {
        printf("Failed to allocate memory\n");
        break;
      }
      portChannelMembersDesc.pstart = (char *) portChannelMembers;
      portChannelMembersDesc.size = (maxMem * (sizeof(uint32_t)));

      if (openapiLagMembersGet(&clientHandle, lagIfNum, &count,
                               &portChannelMembersDesc) != OPEN_E_NONE)
      {
        printf ("Failed to get the member ports for the LAG interface %d\n", lagIfNum);
      }
      else
      {
        printf ("List of member ports for LAG interface %d :\n", lagIfNum);
        for (ifNum = 0; ifNum < count; ifNum++)
        {
          printf ("port : %d \n", portChannelMembers[ifNum]);
        }
      }

      count = maxMem;
      portChannelMembersDesc.pstart = (char *) portChannelMembers;
      portChannelMembersDesc.size = (maxMem * (sizeof(uint32_t)));
      printf ("Getting the active members list on LAG interface %d\n", lagIfNum);
      if (openapiLagActiveMembersGet (&clientHandle, lagIfNum, &count, &portChannelMembersDesc) != OPEN_E_NONE)
      {
        printf("Failed to get the active member ports for the LAG interface %d\n", lagIfNum);
      }
      else
      {
        if (count == 0)
        {
          printf ("LAG interface %d does not have any ACTIVE members...\n", lagIfNum);
        }
        else
        {
          printf ("List of active member ports for LAG interface %d : \n", lagIfNum);
          for (ifNum = 0; ifNum < count; ifNum++)
          {
            printf ("port : %d \n", portChannelMembers[ifNum]);
          }
        }
      }
      printf ("Deleting port 1 to %d from LAG interface %d\n", maxMem, lagIfNum);
      for (ifNum = 1; ifNum < (maxMem + 1); ifNum++)
      {
        if (openapiLagPortDelete (&clientHandle, lagIfNum, ifNum) != OPEN_E_NONE)
        {
        printf("Failed to delete interface %u from LAG inaterface %d\n", ifNum, lagIfNum);
        break;
        }
      }

      printf ("Getting the  members list on LAG interface %d after DELETE\n", lagIfNum);
      count = maxMem;
      portChannelMembersDesc.pstart = (char *) portChannelMembers;
      portChannelMembersDesc.size = (maxMem * (sizeof(uint32_t)));
      if (openapiLagMembersGet (&clientHandle, lagIfNum, &count, &portChannelMembersDesc) != OPEN_E_NONE)
      {
        printf("Failed to get the member ports for the LAG interface %d \n", lagIfNum);
      }
      else
      {
        if (count == 0)
        { 
          printf ("LAG interface %d  does not have any members...\n", lagIfNum);
        }
        else
        {
          printf ("List of member ports for LAG interface %d :\n", lagIfNum);
          for (ifNum = 0; ifNum < count; ifNum++)
          {
            printf ("port : %d \n", portChannelMembers[ifNum]);
          }
        }
      }
      printf ("Press enter to continue...\n");
      getchar ();
      /*Testing the LAG static mode set and get */
      printf ("Testing LAG static mode set/get APIs\n");
      printf ("Getting the current LAG interface %d static mode\n", lagIfNum);
      if (openapiLagStaticModeGet (&clientHandle, lagIfNum, &mode) != OPEN_E_NONE)
      {
        printf ("Failed to get the static mode for the LAG interface: %d\n", lagIfNum);
        break;
      }
      else
	{
	  if (mode == 1)
          {
	    sprintf (buf, "STATIC");
          }
	  else
          {
	    sprintf (buf, "DYNAMIC");
          }

	  printf ("LAG  interface %u is %s\n", lagIfNum, buf);
	}
      if (mode == 0)
	{
	  mode = 1;
	  sprintf (buf, "STATIC");
	}
      else
	{
	  mode = 0;
	  sprintf (buf, "DYNAMIC");
	}
      printf ("Now Setting the LAG interface %d static mode to %s\n", lagIfNum, buf);
      if (openapiLagStaticModeSet (&clientHandle, lagIfNum, mode) != OPEN_E_NONE)
	{
	  printf ("Failed to set the LAG interface [%u] mode to %s\n", lagIfNum, buf);
	  break;
	}
      printf ("Verifying  the LAG interface %d mode after SET\n", lagIfNum);
      if (openapiLagStaticModeGet (&clientHandle, lagIfNum, &mode) != OPEN_E_NONE)
	{
	  printf ("Failed to get the static mode for the LAG interface %d\n", lagIfNum);
	  break;
	}
      else
	{
	  if (mode == 1)
          {
	    sprintf (buf, "STATIC");
          }
	  else
          {
	    sprintf (buf, "DYNAMIC");
          }

	  printf ("LAG  interface %u is %s\n", lagIfNum, buf);
	}
      printf ("Press enter to continue...\n");
      getchar ();
      printf ("Testing LAG name set/get APIs\n");
      printf ("Getting the current LAG name config\n");
      memset (buf, 0, sizeof (buf));
      name.pstart = buf;
      name.size = sizeof (buf);
      if (openapiLagNameGet (&clientHandle, lagIfNum, &name) != OPEN_E_NONE)
	{
	  printf ("Failed to get the name for the LAG interface %d\n",
		  lagIfNum);
	  break;
	}
      else
	{
	  printf ("LAG  interface %u name %s\n", lagIfNum, (char *)name.pstart);
	}
      printf("Setting the LAG names to 'TEST123' LAG interface %d\n", lagIfNum);
      memset (buf, 0, sizeof (buf));
      sprintf (buf, "TEST123");
      name.pstart = buf;
      name.size = sizeof (buf);
      if (openapiLagNameSet (&clientHandle, lagIfNum, &name) != OPEN_E_NONE)
	{
	  printf ("Failed to set the LAG interface [%u] name to 'TEST123'\n", lagIfNum);
	  break;
	}
      printf ("Verifying  the LAG interface %d name after SET \n", lagIfNum);
      if (openapiLagNameGet (&clientHandle, lagIfNum, &name) != OPEN_E_NONE)
	{
	  printf ("Failed to get the name for the LAG interface %d\n", lagIfNum);
	  break;
	}
      else
	{
	  printf ("LAG interface %u name  %s\n", lagIfNum, (char *) name.pstart);
	}

      printf ("Press enter to continue...\n");
      getchar ();
      printf ("Testing LAG hashmode set/get APIs\n");
      printf ("Getting the current hashmode for LAg interface %d\n", lagIfNum);
      if (openapiLagLoadBalanceModeGet (&clientHandle, lagIfNum, &hashmode) != OPEN_E_NONE)
	{
	  printf ("Failed to get the hashmode for the LAG interface %d\n", lagIfNum);
	  break;
	}
      else
	{
	  printf ("LAG interface %u hashmode %d\n", lagIfNum, hashmode);
	}
      printf ("Setting the LAG hashmode to 6  for LAG interface %d\n", lagIfNum);
      hashmode = 6;
      if (openapiLagLoadBalanceModeSet (&clientHandle, lagIfNum, hashmode) != OPEN_E_NONE)
	{
	  printf ("Failed to set the LAG interface [%u] hashmode to '6'\n", lagIfNum);
	  break;
	}
      printf ("Verifying  the LAG interface %d hashmode after SET \n", lagIfNum);
      if (openapiLagLoadBalanceModeGet (&clientHandle, lagIfNum, &hashmode) != OPEN_E_NONE)
	{
	  printf ("Failed to get the hashmode for the LAG interface %d\n", lagIfNum);
	  break;
	}
      else
	{
	  printf ("LAG  interface %u hashmode  %d\n", lagIfNum, hashmode);
	}

    printf ("Press enter to continue...\n");
    getchar ();
    printf ("Testing LAG minimum uplinks  set/get APIs\n");

    if (openapiLagThresholdMinMaxGet (&clientHandle, &minThreshold, &maxThreshold) != OPEN_E_NONE)
	{
	  printf("Failed to get the Threshold minimum and maximum ranges values\n");
	  break;
	}
    else
	{
	  printf ("LAG  Threshold ranges are %d - %d\n", minThreshold, maxThreshold);
	}

    printf ("Getting the minimum uplinks for LAG interface %d\n", lagIfNum);
    if (openapiLagMinUplinksGet (&clientHandle, lagIfNum, &minlinks) != OPEN_E_NONE)
	{
	  printf("Failed to get the minimum uplinks for the LAG interface %d\n", lagIfNum);
	  break;
	}
      else
	{
	  printf ("LAG  interface %u minimum uplinks %d\n", lagIfNum, minlinks);
	}
      printf ("Setting the LAG minimum uplinks to '8' for LAG interface %d\n", lagIfNum);
      minlinks = 8;
      if (openapiLagMinUplinksSet (&clientHandle, lagIfNum, minlinks) != OPEN_E_NONE)
	{
	  printf ("Failed to set the LAG interface [%u] minimum uplinks to '8'\n", lagIfNum);
	  break;
	}
      printf("Verifying  the minimum uplinks for LAG interface %d after SET \n", lagIfNum);
      if (openapiLagMinUplinksGet (&clientHandle, lagIfNum, &minlinks) != OPEN_E_NONE)
	{
	  printf ("Failed to get the minimum uplinks for LAG interface %d\n", lagIfNum);
	  break;
	}
      else
	{
	  printf ("LAG  interface %u minimum uplinks  %d\n", lagIfNum, minlinks);
	}
      printf("Getting the port LACP mode for interface : 1\n");
      if(openapiDot3adAggPortLacpModeGet(&clientHandle, 1, &lacpMode) != OPEN_E_NONE)
      {
        printf("Failed to get the port LACP mode for interface : 1 \n");
        break;
      }
      else
      {
        printf(" interface : 1 LACP mode : %s \n",(lacpMode == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      }
      printf(" Setting port LACP mode of interface : 1 to : %s\n",(lacpMode != OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      (lacpMode != OPEN_ENABLE) ? (lacpMode = OPEN_ENABLE) : (lacpMode = OPEN_DISABLE);
      if(openapiDot3adAggPortLacpModeSet(&clientHandle, 1, lacpMode) != OPEN_E_NONE)
      {
        printf("Failed to set the port LACP mode for interface : 1 \n");
        break;
      }
      else
      {
        printf("Getting the port LACP mode after set...\n");
        if(openapiDot3adAggPortLacpModeGet(&clientHandle, 1, &lacpMode) != OPEN_E_NONE)
        {
          printf("Failed to get the port LACP mode for interface : 1 \n");
          break;
        }
        else
        {
          printf(" interface : 1 LACP mode : %s \n",(lacpMode == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
        }
      }

      printf("Getting the port LACP actor admin state for interface : 1\n");
      if(openapiDot3adAggPortActorAdminStateGet(&clientHandle, 1, OPEN_DOT3AD_STATE_AGGREGATION, &state) != OPEN_E_NONE)
      {
        printf("Failed to get the LACP actor admin state for interface : 1 \n");
        break;
      }
      else
      {
        printf(" interface : 1 LACP actor admin state 'aggregation' : %s \n",(state == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      }
      printf(" Setting LACP actor admin state 'aggregation' of interface : 1 to : %s\n",(state != OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      (state != OPEN_ENABLE) ? (state = OPEN_ENABLE) : (state = OPEN_DISABLE);
      if(openapiDot3adAggPortActorAdminStateSet(&clientHandle, 1, OPEN_DOT3AD_STATE_AGGREGATION, state) != OPEN_E_NONE)
      {
        printf("Failed to set the LACP actor admin state for interface : 1 \n");
        break;
      }
      else
      {
        printf("Getting the LACP actor admin state 'aggregation' after set...\n");
        if(openapiDot3adAggPortActorAdminStateGet(&clientHandle, 1, OPEN_DOT3AD_STATE_AGGREGATION, &state) != OPEN_E_NONE)
        {
          printf("Failed to get the LACP actor admin state for interface : 1 \n");
          break;
        }
        else
        {
          printf(" interface : 1 LACP actor admin state 'aggregation' : %s \n",(state == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
        }
      }
      if(openapiDot3adAggPortActorAdminStateGet(&clientHandle, 1, OPEN_DOT3AD_STATE_LACP_TIMEOUT, &state) != OPEN_E_NONE)
      {
        printf("Failed to get the LACP actor admin state for interface : 1 \n");
        break;
      }
      else
      {
        printf(" interface : 1 LACP actor admin state 'short timeout' : %s \n",(state == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      }
      printf(" Setting LACP actor admin state 'short timeout' of interface : 1 to : %s\n",(state != OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      (state != OPEN_ENABLE) ? (state = OPEN_ENABLE) : (state = OPEN_DISABLE);
      if(openapiDot3adAggPortActorAdminStateSet(&clientHandle, 1, OPEN_DOT3AD_STATE_LACP_TIMEOUT, state) != OPEN_E_NONE)
      {
        printf("Failed to set the LACP actor admin state for interface : 1 \n");
        break;
      }
      else
      {
        printf("Getting the LACP actor admin state 'short timeout' after set...\n");
        if(openapiDot3adAggPortActorAdminStateGet(&clientHandle, 1, OPEN_DOT3AD_STATE_LACP_TIMEOUT, &state) != OPEN_E_NONE)
        {
          printf("Failed to get the LACP actor admin state for interface : 1 \n");
          break;
        }
        else
        {
          printf(" interface : 1 LACP actor admin state 'short timeout' : %s \n",(state == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
        }
      }
      if(openapiDot3adAggPortActorAdminStateGet(&clientHandle, 1, OPEN_DOT3AD_STATE_LACP_ACTIVITY, &state) != OPEN_E_NONE)
      {
        printf("Failed to get the LACP actor admin state for interface : 1 \n");
        break;
      }
      else
      {
        printf(" interface : 1 LACP actor admin state 'active' : %s \n",(state == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      }
      printf(" Setting LACP actor admin state 'active' of interface : 1 to : %s\n",(state != OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      (state != OPEN_ENABLE) ? (state = OPEN_ENABLE) : (state = OPEN_DISABLE);
      if(openapiDot3adAggPortActorAdminStateSet(&clientHandle, 1, OPEN_DOT3AD_STATE_LACP_ACTIVITY, state) != OPEN_E_NONE)
      {
        printf("Failed to set the LACP actor admin state for interface : 1 \n");
        break;
      }
      else
      {
        printf("Getting the LACP actor admin state 'active' after set...\n");
        if(openapiDot3adAggPortActorAdminStateGet(&clientHandle, 1, OPEN_DOT3AD_STATE_LACP_ACTIVITY, &state) != OPEN_E_NONE)
        {
          printf("Failed to get the LACP actor admin state for interface : 1 \n");
          break;
        }
        else
        {
          printf(" interface : 1 LACP actor admin state 'active' : %s \n",(state == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
        }
      }
    }
  while (0);
  if (portChannelMembers)
  {
    free (portChannelMembers);
  }
  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping LAG API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
